/* StarPU --- Runtime system for heterogeneous multicore architectures.
 *
 * Copyright (C) 2009-2023  Université de Bordeaux, CNRS (LaBRI UMR 5800), Inria
 * Copyright (C) 2010       Mehdi Juhoor
 *
 * StarPU is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 *
 * StarPU is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See the GNU Lesser General Public License in COPYING.LGPL for more details.
 */

/* LU StarPU implementation using explicit tag dependencies */
#include "xlu.h"
#include "xlu_kernels.h"

#include "starpu_cusolver.h"

/*
 *	Construct the DAG
 */

static struct starpu_task *create_task(starpu_tag_t id)
{
	struct starpu_task *task = starpu_task_create();
		task->cl_arg = NULL;

	task->use_tag = 1;
	task->tag_id = id;

	return task;
}

static struct starpu_task *create_task_getrf(starpu_data_handle_t dataA, unsigned k, unsigned no_prio, int nblocks)
{
/*	printf("task GETRF k = %d TAG = %llx\n", k, (TAG_GETRF(k))); */

	struct starpu_task *task = create_task(TAG_GETRF(k));

	task->cl = &cl_getrf;
	task->color = 0xffff00;

	/* which sub-data is manipulated ? */
	task->handles[0] = starpu_data_get_sub_data(dataA, 2, k, k);

#if defined(STARPU_USE_CUDA) && defined(STARPU_HAVE_LIBCUSOLVER)
	task->handles[1] = scratch;
#endif

	/* this is an important task */
	if (!no_prio)
		task->priority = 3*nblocks - 3*k; /* Bottom-level-based prio */

	/* enforce dependencies ... */
	if (k > 0)
	{
		starpu_tag_declare_deps(TAG_GETRF(k), 1, TAG_GEMM(k-1, k, k));
	}

	return task;
}

static int create_task_trsm_ll(starpu_data_handle_t dataA, unsigned k, unsigned j, unsigned no_prio, int nblocks)
{
	int ret;

/*	printf("task TRSM_LL k,i = %d,%d TAG = %llx\n", k,i, TAG_TRSM_LL(k,i)); */

	struct starpu_task *task = create_task(TAG_TRSM_LL(k, j));

	task->cl = &cl_trsm_ll;
	task->color = 0x8080ff;

	/* which sub-data is manipulated ? */
	task->handles[0] = starpu_data_get_sub_data(dataA, 2, k, k);
	task->handles[1] = starpu_data_get_sub_data(dataA, 2, j, k);

	if (!no_prio)
		task->priority = 3*nblocks - (2*k + j); /* Bottom-level-based prio */

	/* enforce dependencies ... */
	if (k > 0)
	{
		starpu_tag_declare_deps(TAG_TRSM_LL(k, j), 2, TAG_GETRF(k), TAG_GEMM(k-1, k, j));
	}
	else
	{
		starpu_tag_declare_deps(TAG_TRSM_LL(k, j), 1, TAG_GETRF(k));
	}

	ret = starpu_task_submit(task);
	if (ret != -ENODEV) STARPU_CHECK_RETURN_VALUE(ret, "starpu_task_submit");
	return ret;
}

static int create_task_trsm_ru(starpu_data_handle_t dataA, unsigned k, unsigned i, unsigned no_prio, int nblocks)
{
	int ret;
	struct starpu_task *task = create_task(TAG_TRSM_RU(k, i));

	task->cl = &cl_trsm_ru;
	task->color = 0x8080c0;

	/* which sub-data is manipulated ? */
	task->handles[0] = starpu_data_get_sub_data(dataA, 2, k, k);
	task->handles[1] = starpu_data_get_sub_data(dataA, 2, k, i);

	if (!no_prio)
		task->priority = 3*nblocks - (2*k + i); /* Bottom-level-based prio */

	/* enforce dependencies ... */
	if (k > 0)
	{
		starpu_tag_declare_deps(TAG_TRSM_RU(k, i), 2, TAG_GETRF(k), TAG_GEMM(k-1, i, k));
	}
	else
	{
		starpu_tag_declare_deps(TAG_TRSM_RU(k, i), 1, TAG_GETRF(k));
	}

	ret = starpu_task_submit(task);
	if (ret != -ENODEV) STARPU_CHECK_RETURN_VALUE(ret, "starpu_task_submit");
	return ret;
}

static int create_task_gemm(starpu_data_handle_t dataA, unsigned k, unsigned i, unsigned j, unsigned no_prio, int nblocks)
{
	int ret;

/*	printf("task GEMM k,i,j = %d,%d,%d TAG = %llx\n", k,i,j, TAG_GEMM(k,i,j)); */

	struct starpu_task *task = create_task(TAG_GEMM(k, i, j));

	task->cl = &cl_gemm;
	task->color = 0x00ff00;

	/* which sub-data is manipulated ? */
	task->handles[0] = starpu_data_get_sub_data(dataA, 2, k, i); /* produced by TAG_TRSM_RU(k, i) */
	task->handles[1] = starpu_data_get_sub_data(dataA, 2, j, k); /* produced by TAG_TRSM_LL(k, j) */
	task->handles[2] = starpu_data_get_sub_data(dataA, 2, j, i); /* produced by TAG_GEMM(k-1, i, j) */

	if (!no_prio)
		task->priority = 3*nblocks - (k + i + j); /* Bottom-level-based prio */

	/* enforce dependencies ... */
	if (k > 0)
	{
		starpu_tag_declare_deps(TAG_GEMM(k, i, j), 3, TAG_GEMM(k-1, i, j), TAG_TRSM_LL(k, j), TAG_TRSM_RU(k, i));
	}
	else
	{
		starpu_tag_declare_deps(TAG_GEMM(k, i, j), 2, TAG_TRSM_LL(k, j), TAG_TRSM_RU(k, i));
	}

	ret = starpu_task_submit(task);
	if (ret != -ENODEV) STARPU_CHECK_RETURN_VALUE(ret, "starpu_task_submit");
	return ret;
}

/*
 *	code to bootstrap the factorization
 */

static int dw_codelet_facto_v3(starpu_data_handle_t dataA, unsigned nblocks, unsigned no_prio)
{
	int ret;
	double start;
	double end;

	struct starpu_task *entry_task = NULL;

	/* create all the DAG nodes */
	unsigned i,j,k;

	if (bound)
		starpu_bound_start(bounddeps, boundprio);

	for (k = 0; k < nblocks; k++)
	{
		starpu_iteration_push(k);
		struct starpu_task *task = create_task_getrf(dataA, k, no_prio, nblocks);

		/* we defer the launch of the first task */
		if (k == 0)
		{
			entry_task = task;
		}
		else
		{
			ret = starpu_task_submit(task);
			if (ret == -ENODEV) return ret;
			STARPU_CHECK_RETURN_VALUE(ret, "starpu_task_submit");
		}

		for (i = k+1; i<nblocks; i++)
		{
			ret = create_task_trsm_ll(dataA, k, i, no_prio, nblocks);
			if (ret == -ENODEV) return ret;
			ret = create_task_trsm_ru(dataA, k, i, no_prio, nblocks);
			if (ret == -ENODEV) return ret;
		}

		for (i = k+1; i<nblocks; i++)
		{
			for (j = k+1; j<nblocks; j++)
			{
				ret = create_task_gemm(dataA, k, i, j, no_prio, nblocks);
			     if (ret == -ENODEV) return ret;
			}
		}
		starpu_iteration_pop();
	}

	/* schedule the codelet */
	start = starpu_timing_now();
	ret = starpu_task_submit(entry_task);
	if (ret == -ENODEV) return ret;
	STARPU_CHECK_RETURN_VALUE(ret, "starpu_task_submit");

	/* stall the application until the end of computations */
	starpu_tag_wait(TAG_GETRF(nblocks-1));

	end = starpu_timing_now();

	if (bound)
		starpu_bound_stop();

	double timing = end - start;
	unsigned n = starpu_matrix_get_nx(dataA);
	double flop = (2.0f*n*n*n)/3.0f;

	PRINTF("# size\tms\tGFlop/s");
	if (bound)
		PRINTF("\tTms\tTGFlop/s");
	PRINTF("\n");
	PRINTF("%u\t%.0f\t%.1f", n, timing/1000, flop/timing/1000.0f);
	if (bound)
	{
		double min;
		starpu_bound_compute(&min, NULL, 0);
		PRINTF("\t%.0f\t%.1f", min, flop/min/1000000.0f);
	}
	PRINTF("\n");

	return 0;
}

int STARPU_LU(lu_decomposition)(TYPE *matA, unsigned size, unsigned ld, unsigned nblocks, unsigned no_prio)
{
	starpu_data_handle_t dataA;

	/* monitor and partition the A matrix into blocks :
	 * one block is now determined by 2 unsigned (i,j) */
	starpu_matrix_data_register(&dataA, STARPU_MAIN_RAM, (uintptr_t)matA, ld, size, size, sizeof(TYPE));

	/* We already enforce deps by hand */
	starpu_data_set_sequential_consistency_flag(dataA, 0);

	struct starpu_data_filter f =
	{
		.filter_func = starpu_matrix_filter_vertical_block,
		.nchildren = nblocks
	};

	struct starpu_data_filter f2 =
	{
		.filter_func = starpu_matrix_filter_block,
		.nchildren = nblocks
	};

	starpu_data_map_filters(dataA, 2, &f, &f2);

	lu_kernel_init(size / nblocks);

	int ret = dw_codelet_facto_v3(dataA, nblocks, no_prio);

	lu_kernel_fini();

	/* gather all the data */
	starpu_data_unpartition(dataA, STARPU_MAIN_RAM);
	starpu_data_unregister(dataA);

	return ret;
}
