# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Tests the scopes UI helper."""

from django.test import RequestFactory
from django.urls import reverse

from debusine.db.playground import scenarios
from debusine.test.django import TestCase
from debusine.web.icons import Icons
from debusine.web.views.ui.scopes import CustomURLPlace, ScopeUI


class ScopeUITests(TestCase):
    """Test the :py:class:`ScopeUI` class."""

    scenario = scenarios.DefaultContext(set_current=True)

    def ui(self) -> ScopeUI:
        """Instantiate the UI helper to test."""
        factory = RequestFactory()
        request = factory.get("/")
        return self.scenario.scope.ui(request)

    def test_place(self) -> None:
        place = self.ui().place
        self.assertEqual(place.title, self.scenario.scope.name)
        self.assertEqual(place.url, reverse("scopes:detail"))
        self.assertEqual(place.icon, Icons.SCOPE)
        self.assertIsNone(place.icon_title)

    def test_place_with_icon(self) -> None:
        self.scenario.scope.icon = "icon.png"
        self.scenario.scope.save()
        place = self.ui().place
        self.assertIsInstance(place, CustomURLPlace)
        self.assertEqual(place.title, self.scenario.scope.name)
        self.assertEqual(place.url, reverse("scopes:detail"))
        self.assertEqual(place.icon, "icon.png")
        self.assertIsNone(place.icon_title)


class CustomURLPlaceTests(TestCase):
    """Test the :py:class:`CustomURLPlace` class."""

    def test_as_nav(self) -> None:
        place = CustomURLPlace(title="test title", url="/test-url")
        tree = self.assertHTMLValid(place.as_nav())
        a = self.assertHasElement(tree, "//a")
        self.assertTextContentEqual(a, "test title")
        self.assertEqual(a.get("href"), "/test-url")

    def test_as_nav_icon(self) -> None:
        place = CustomURLPlace(
            title="test title",
            url="/test-url",
            icon="icon.png",
            icon_title="icon-title",
        )
        tree = self.assertHTMLValid(place.as_nav())
        a = self.assertHasElement(tree, "//a")
        self.assertTextContentEqual(a, "test title")
        self.assertEqual(a.get("href"), "/test-url")

        img = self.assertHasElement(a, "img")
        self.assertEqual(img.get("src"), "/static/icon.png")

    def test_as_page_title(self) -> None:
        place = CustomURLPlace(title="test title")
        tree = self.assertHTMLValid(place.as_page_title())
        title = self.assertHasElement(tree, "//h1")
        self.assertTextContentEqual(title, "test title")

    def test_as_page_title_icon(self) -> None:
        place = CustomURLPlace(title="test title", icon="icon.png")
        tree = self.assertHTMLValid(place.as_page_title())
        h1 = self.assertHasElement(tree, "//h1")
        self.assertTextContentEqual(h1, "test title")

        img = self.assertHasElement(h1, "img")
        self.assertEqual(img.get("src"), "/static/icon.png")

    def test_as_page_title_icon_title(self) -> None:
        place = CustomURLPlace(
            title="test title", icon="icon.png", icon_title="icon-title"
        )
        tree = self.assertHTMLValid(place.as_page_title())
        h1 = self.assertHasElement(tree, "//h1")
        self.assertTextContentEqual(h1, "test title")

        img = self.assertHasElement(h1, "img")
        self.assertEqual(img.get("src"), "/static/icon.png")
        self.assertEqual(img.get("alt"), "icon-title")
