/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/frame/platform_config.hpp
 *
 * $Id: platform_config.hpp,v 1.4 2005/06/23 09:54:20 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @brief [LEVEL: beta] Platform Configuration for the logging
 * framework: There are several switches:
 *
 * @arg @ref DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED
 * @arg @ref DIAGNOSTICS_SWITCH_MULTITHREADING_ENABLED
 * @arg @ref DIAGNOSTICS_SWITCH_MEMORY_ANALYSIS_ENABLED
 * @arg @ref DIAGNOSTICS_SWITCH_CHECK_FOR_NESTING
 *
 * Based on these swtiches and based on the available platform, this
 * file determines the which libraries to use and how to use them.
 *
 * @test none
 */

#ifndef DIAGNOSTICS__FRAME__PLATFORM_CONFIG_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__FRAME__PLATFORM_CONFIG_HPP__INCLUDE_GUARD

#include <diagnostics/frame/namespace.hpp>

#ifndef WIN32
/** 
 * @brief the following header file is constructed 
 * dynamically and is not available in win32
 */
#	include <diagnostics/features.hpp>
#endif

////////////////////////////////////////////////////////////////////////////////
//
// System Features
// 

/**
 * @brief Switches for compatibility with Windows 
 * (ignored in non-win32 environments)
 */
#ifdef WIN32
#  ifndef DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED
#    define DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED 0
#  endif
#  ifndef DIAGNOSTICS_SWITCH_MULTITHREADING_ENABLED
#    define DIAGNOSTICS_SWITCH_MULTITHREADING_ENABLED 0
#  endif
#  ifndef DIAGNOSTICS_HAVE_ACE
#    define DIAGNOSTICS_HAVE_ACE 0
#  endif
#endif
// @}

/**
 * @brief set to 1 to use ACE
 */
#ifndef DIAGNOSTICS_HAVE_ACE
#  define DIAGNOSTICS_HAVE_ACE 1
#endif
// @}


////////////////////////////////////////////////////////////////////////////////
//
// Switches and Parameters
// 

/**
 * @brief set to 1 for memory analysis, i.e., providing primitives
 * which allow to monitor the memory allocation behavior of the
 * application.
 *
 * On some systems, memory analysis does not work properly. In this
 * case, use this switch to switch off memory analysis. 
 *
 * Also, switching on memory analysis might degrade the performance of
 * @ref ::diagnostics::Logging_Facility::log to a certain degree since
 * recursive mutexing will be required (if
 * DIAGNOSTICS_SWITCH_MULTITHREADING_ENABLED ==1).
 */
#ifndef DIAGNOSTICS_SWITCH_MEMORY_ANALYSIS_ENABLED
#  define DIAGNOSTICS_SWITCH_MEMORY_ANALYSIS_ENABLED 1
#endif

/**
 * @brief set to 1 to check against nested logging, i.e., whehter
 * there is a recursive call to @ref
 * ::diagnostics::Logging_Facility::log. This feature requires a
 * recursive mutex, if @ref DIAGNOSTICS_SWITCH_MEMORY_ANALYSIS_ENABLED
 * ==1.
 */
#ifndef DIAGNOSTICS_SWITCH_CHECK_FOR_NESTING
#  define DIAGNOSTICS_SWITCH_CHECK_FOR_NESTING 1
#endif

#if DIAGNOSTICS_SWITCH_CHECK_FOR_NESTING == 0 && DIAGNOSTICS_SWITCH_MEMORY_ANALYSIS_ENABLED == 1
#  error Check for nested logging is necessary for memory analysis.
#endif


/**
 * @brief set to 1 for platform support, i.e. process, thread, timing,
 * and host information. 
 */
#ifndef DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED
#  if DIAGNOSTICS_HAVE_ACE == 1
#    define DIAGNOSTICS_INTERNAL_SWITCH_USE_ACE 1
#    define DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED 1
#  else
#    define DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED 0
#  endif
#else
#  if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
#    if DIAGNOSTICS_HAVE_ACE == 0
#      error some portability library is necessary to support system calls!
#    endif
#  endif
#endif

/**
 * @brief set to 1 to enable multithreading (require @ref
 * DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED set to 1)
 */
#ifndef DIAGNOSTICS_SWITCH_MULTITHREADING_ENABLED 
#  if DIAGNOSTICS_HAVE_ACE == 1
#    define DIAGNOSTICS_INTERNAL_SWITCH_USE_ACE 1
#    define DIAGNOSTICS_SWITCH_MULTITHREADING_ENABLED 1
#  else
#    define DIAGNOSTICS_SWITCH_MULTITHREADING_ENABLED 0
#  endif
#else
#  if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
#    if DIAGNOSTICS_HAVE_ACE == 0
#      error some portability library is necessary to support multithreading!
#    endif
#  endif
#endif



////////////////////////////////////////////////////////////////////////////////
//
// Derived Configuration (Features matching up the Switches and Parameters)
//

#ifndef DIAGNOSTICS_INTERNAL_SWITCH_USE_ACE 
#  define DIAGNOSTICS_INTERNAL_SWITCH_USE_ACE 0
#endif

#if DIAGNOSTICS_INTERNAL_SWITCH_USE_ACE == 1
#  define _REENTRANT 1 // ACE requires _REENTRANT to be defined
#endif

#if DIAGNOSTICS_SWITCH_MULTITHREADING_ENABLED == 0
#  if DIAGNOSTICS_SWITCH_CHECK_FOR_NESTING == 1
#    define DIAGNOSTICS_INTERNAL_SWITCH_MUTEX 3 // recursion check
#  else
#    define DIAGNOSTICS_INTERNAL_SWITCH_MUTEX 0 // no mutex
#  endif
#else
#  if DIAGNOSTICS_SWITCH_CHECK_FOR_NESTING == 1
#    define DIAGNOSTICS_INTERNAL_SWITCH_MUTEX 2 // recursive mutex
#  else 
#    define DIAGNOSTICS_INTERNAL_SWITCH_MUTEX 1 // non-recursive mutex
#  endif
#endif


// DIAGNOSTICS_INTERNAL_SWITCH_USE_ACE set to 1 if ACE is used

// DIAGNOSTICS_INTERNAL_SWITCH_MUTEX set to 0,1,2,3 for no mutex,
// non-recursive mutex, recursive mutex, recursion check (without a
// mutex), respectively


#endif

// vim:ts=4:sw=4
