/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/logger/stream_logger.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref diagnostics::Stream_Logger
 *
 * $Id: stream_logger.t.cpp,v 1.13 2005/06/23 09:54:21 esdentem Exp $
 *
 * @author Christian Schallhart
 */
#include <diagnostics/unittest.hpp>

#include <diagnostics/logger/stream_logger.hpp>

// used components
#include <diagnostics/frame/record.hpp>
#include <diagnostics/frame/logging_config_exception.hpp>

// test support
#include <diagnostics/util/to_string.hpp>

#include <diagnostics/util/dummy_values_for_tid_pid.ts.hpp>

#include <sstream>
#include <fstream>

#define TEST_COMPONENT_NAME Stream_Logger
#define TEST_COMPONENT_NAMESPACE diagnostics

/**
 * @brief two records to play with -- assembly_cstr
 */
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
#  define RECORD_1 \
    Record r1(LEVEL_TEST,TYPE_TESTCASE_ENTER,0,"what","base","file",1, \
              Dummy_Values_For_Pid::value2,Dummy_Values_For_Tid::value3,4,5,"host")
#  define RECORD_2 \
    Record r2(LEVEL_AUDIT,TYPE_TESTCASE_EXIT,1,"what1","base1","file1",2, \
              Dummy_Values_For_Pid::value3,Dummy_Values_For_Tid::value2,5,6,"host1")
#else 
#  define RECORD_1 \
    Record r1(LEVEL_TEST,TYPE_TESTCASE_ENTER,0,"what","base","file",1)
#  define RECORD_2 \
    Record r2(LEVEL_AUDIT,TYPE_TESTCASE_EXIT,1,"what1","base1","file1",2)
#endif


DIAGNOSTICS_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;
using namespace ::diagnostics::unittest;

void all_methods(Test_Data & data)
{
    using ::diagnostics::internal::to_string;

    RECORD_1;
    RECORD_2;

    // we assume that the operator<< works as it should.
    ::std::string const sr1(to_string(r1)+"\n");
    ::std::string const sr1r2(to_string(r1)+"\n"+to_string(r2)+"\n");

    ::std::ostringstream stream;
    Stream_Logger logger(stream);

    // log the first and check
    logger.log(r1);
    TEST_ASSERT(sr1==stream.str());

    // log the second and check
    logger.log(r2);
    TEST_ASSERT(sr1r2==stream.str());

    // close and check the stream
    logger.close();
    TEST_ASSERT(sr1r2==stream.str());

    // should generate a PANIC LOG
    ::std::ofstream bugg_stream("this_directory)does)not)exist/and_thus_this_file_cannot_be_opened");
    TEST_ASSERT(!bugg_stream.good());
    Stream_Logger buggy_logger(bugg_stream);
    buggy_logger.log(r1);
}

void log_1024_1024(Test_Data & data)
{
    RECORD_1;

    ::std::string buffer;
    buffer.resize(1024*128*256);
    for(int i=0;i<8;++i) {
	::std::ostringstream stream(buffer);
	Stream_Logger logger(stream);
	for(int j=0;j<1024*128;++j) logger.log(r1);
    }
}


TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&all_methods,LEVEL_PROD);
TEST_NONFCT_CASE(&log_1024_1024,30);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;

// vim:ts=4:sw=4
