# GSW-Python

[![Tests](https://github.com/TEOS-10/GSW-Python/actions/workflows/tests.yml/badge.svg)](https://github.com/TEOS-10/GSW-Python/actions/workflows/tests.yml)
[![Wheels](https://github.com/TEOS-10/GSW-Python/actions/workflows/cibuildwheel.yml/badge.svg)](https://github.com/TEOS-10/GSW-Python/actions/workflows/cibuildwheel.yml)
[![DOI](https://zenodo.org/badge/86503067.svg)](https://zenodo.org/badge/latestdoi/86503067)


This Python implementation of the Thermodynamic Equation of Seawater 2010 (TEOS-10) is based primarily on numpy ufunc wrappers of the GSW-C implementation.
This library replaces the original [python-gsw](https://github.com/TEOS-10/python-gsw) pure-python implementation.
The primary reasons for this change are that by building on the C implementation we reduce code duplication and we gain an immediate update to the 75-term equation.  
Additional benefits include a major increase in speed,
a reduction in memory usage,
and the inclusion of more functions.
The penalty is that a C (or MSVC C++ for Windows) compiler is required to build the package from source.

**Warning: this is for Python >=3.8 only.**

Documentation is provided at https://teos-10.github.io/GSW-Python/.

For the core functionality, we use an auto-generated C extension
module to wrap the C functions as numpy [ufuncs](https://docs.scipy.org/doc/numpy/reference/ufuncs.html),
and then use an autogenerated Python module to add docstrings and handle masked arrays.
165 scalar C functions with only double-precision arguments and return values are wrapped as ufuncs,
and 158 of these are exposed in the ``gsw`` namespace with an additional wrapper in Python.

A hand-written wrapper is used for one C function, and others are re-implemented directly in Python instead of being wrapped.
Additional functions present in GSW-Matlab but not in GSW-C may be re-implemented in Python,
but there is no expectation that all such functions will be provided.

## Installation

Pip users can install the pre-built wheels with:

```shell
pip install gsw
```

conda users will find binaries on conda-forge,

```shell
conda install gsw --channel conda-forge
```

The development version of the package can be installed from a clone of the repo using

```shell
pip install .
```

## Citation

If you use GSW-Python, please cite: McDougall, T.J. and P.M. Barker, 2011: Getting started with TEOS-10 and the Gibbs Seawater (GSW) Oceanographic Toolbox, 28pp., SCOR/IAPSO WG127, ISBN 978-0-646-55621-5

```bibtex
@book{mcdougall2011getting,
  author = {McDougall, T. J. and Barker, P. M.},
  title = {Getting started with TEOS-10 and the Gibbs Seawater (GSW) Oceanographic Toolbox},
  year = {2011},
  pages = {28},
  publisher = {SCOR/IAPSO WG127},
  isbn = {978-0-646-55621-5}
}
```

## Note for xarray users

A wrapper around gsw called [gsw-xarray](https://github.com/DocOtak/gsw-xarray) exists for xarray.
It adds CF compliant attributes when possible, units, and name.

## Note on generating the docstrings

The autogenerated docstrings are checked with codespell in the CIs.
When autogenerating them we need to run ``pre-commit run --all-files`` and fix the documentation issues found.

## Development notes

You will need a suitable compiler:
gcc or clang for unix-like systems,
or the MSVC compiler set used for Python itself on Windows.
For Windows, some of the source code has been modified to C++ because the MSVC C compiler does not support the
C99 complex data type used in original GSW-C.

The subdirectory ('tools') contains modules and scripts for
maintaining the code that is autogenerated from the upstream
GSW-Matlab and GSW-C repos.  The scripts are to be run from
this directory; they make assumptions about where they
are, and about where upstream repos are.  Specifically, it
is assumed that GSW-Matlab, GSW-C, and GSW-Python git repos
are all in the same base directory.

### Scripts
- `copy_from_GSW-C.py`: copies the relevant .c and .h files from a
    sibling GSW-C repo, if the latter are newer.
- `mat2npz.py`: generates an npz file in gsw/tests containing the
    test data and check values from the gsw_data_v3_0.mat file
    in the sibling GSW-Matlab repo.
- `make_ufuncs.py`: Generates the src/_ufuncs.c file to turn the
    scalar C functions into numpy ufuncs.  It writes ufuncs.list
    in the current directory as a record of the ufunc names.
    Functions are identified as ufunc candidates based on their
    signatures, parsed from src/c_gsw/gswteos-10.h.
- `make_wrapped_ufuncs.py`: Generates gsw/_wrapped_ufuncs.py based on
    the output of make_ufuncs.py.  It adds docstrings constructed
    from the Matlab help text.
- `fix_wrapped_ufunc_typos.py`: Fixes docstring typos that have been
    identified, but not yet fixed, in the GSW-Matlab repo.
- `codegen`: Runs the last three python scripts.

### Modules
- `c_header_parser.py`: Functions for taking apart the function
    declarations in gswteos-10.h.
- `matlab_parser.py`: Functions specifically for reading the GSW-Matlab
    function signatures and for splitting out the help text.
- `docstring_parts.py`: Blocks of text for assembling docstrings.
- `docstring_utils.py`: Functions for assembling docstrings in numpydoc
    format.
- `categories.py`: Functions for listing gsw function names by category.
    This is not used by any of the other functions or scripts, but
    was used when initially categorizing the functions for inclusion
    in submodules.

### Notes

- In addition to the generated src/_ufuncs.c, there are two C files
    that are hand-written: src/method_bodies.c and src/method_def_entries.c.
    These are imported by src/_ufuncs.c. They handle some C functions
    that are not suitable for ufuncs.
- Specialized Matlab parsing is also done in gsw/tests/check_functions.py,
    which is used by gsw/tests/test_check_functions.py; see the docstring
    of the former for more info.

### Testing

To test, after installation, run "pytest --pyargs gsw".
