/* Copyright (c) 2001-2024, The HSQL Development Group
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * Neither the name of the HSQL Development Group nor the names of its
 * contributors may be used to endorse or promote products derived from this
 * software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL HSQL DEVELOPMENT GROUP, HSQLDB.ORG,
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


package org.hsqldb.jdbc;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;

import java.math.BigDecimal;
import java.math.RoundingMode;

import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.JDBCType;
import java.sql.NClob;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.RowId;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLType;
import java.sql.SQLWarning;
import java.sql.SQLXML;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;

import java.time.Duration;
import java.time.Period;

import java.util.Calendar;
import java.util.Map;

import org.hsqldb.ColumnBase;
import org.hsqldb.HsqlException;
import org.hsqldb.SessionInterface;
import org.hsqldb.error.ErrorCode;
import org.hsqldb.lib.IntValueHashMap;
import org.hsqldb.lib.StringInputStream;
import org.hsqldb.lib.java.JavaSystem;
import org.hsqldb.navigator.RowSetNavigator;
import org.hsqldb.navigator.RowSetNavigatorClient;
import org.hsqldb.result.Result;
import org.hsqldb.result.ResultConstants;
import org.hsqldb.result.ResultMetaData;
import org.hsqldb.result.ResultProperties;
import org.hsqldb.types.BinaryData;
import org.hsqldb.types.BinaryUUIDType;
import org.hsqldb.types.BlobData;
import org.hsqldb.types.BlobDataID;
import org.hsqldb.types.ClobDataID;
import org.hsqldb.types.DateTimeType;
import org.hsqldb.types.IntervalMonthData;
import org.hsqldb.types.IntervalSecondData;
import org.hsqldb.types.IntervalType;
import org.hsqldb.types.JavaObjectData;
import org.hsqldb.types.TimeData;
import org.hsqldb.types.TimestampData;
import org.hsqldb.types.Type;
import org.hsqldb.types.Types;

// campbell-burnet@users 20051207 - patch 1.9.0 - initial JDBC 4.0 support work
// fredt@users    20060431 - patch 1.9.0 rewrite with RowSetNavigator
// campbell-burnet@users 20060522 - doc   1.9.0 - full synch up to JAVA 1.6 (Mustang) Build 84
// Revision 1.21  2006/07/12 12:27:25  boucherb
// - full synch up to JAVA 1.6 (Mustang) b90

/**
 * A table of data representing a database result set, which
 * is usually generated by executing a statement that queries the database.
 *
 * <P>A {@code ResultSet} object  maintains a cursor pointing
 * to its current row of data.  Initially the cursor is positioned
 * before the first row. The {@code next} method moves the
 * cursor to the next row, and because it returns {@code false}
 * when there are no more rows in the {@code ResultSet} object,
 * it can be used in a {@code while} loop to iterate through
 * the result set.
 * <P>
 * A default {@code ResultSet} object is not updatable and
 * has a cursor that moves forward only.  Thus, you can
 * iterate through it only once and only from the first row to the
 * last row. It is possible to
 * produce {@code ResultSet} objects that are scrollable and/or
 * updatable.  The following code fragment, in which {@code con}
 * is a valid {@code Connection} object, illustrates how to make
 * a result set that is scrollable and insensitive to updates by others, and
 * that is updatable. See {@code ResultSet} fields for other
 * options.
 * <PRE>
 *
 *       Statement stmt = con.createStatement(
 *                                      ResultSet.TYPE_SCROLL_INSENSITIVE,
 *                                      ResultSet.CONCUR_UPDATABLE);
 *       ResultSet rs = stmt.executeQuery("SELECT a, b FROM TABLE2");
 *       // rs will be scrollable, will not show changes made by others,
 *       // and will be updatable
 *
 * </PRE>
 * The {@code ResultSet} interface provides
 * <i>getter</i> methods ({@code getBoolean}, {@code getLong}, and so on)
 * for retrieving column values from the current row.
 * Values can be retrieved using either the index number of the
 * column or the name of the column.  In general, using the
 * column index will be more efficient.  Columns are numbered from 1.
 * For maximum portability, result set columns within each row should be
 * read in left-to-right order, and each column should be read only once.
 *
 * <P>For the getter methods, a JDBC driver attempts
 * to convert the underlying data to the Java type specified in the
 * getter method and returns a suitable Java value.  The JDBC specification
 * has a table showing the allowable mappings from SQL types to Java types
 * that can be used by the {@code ResultSet} getter methods.
 *
 * <P>Column names used as input to getter methods are case
 * insensitive.  When a getter method is called  with
 * a column name and several columns have the same name,
 * the value of the first matching column will be returned.
 * The column name option is
 * designed to be used when column names are used in the SQL
 * query that generated the result set.
 * For columns that are NOT explicitly named in the query, it
 * is best to use column numbers. If column names are used, the
 * programmer should take care to guarantee that they uniquely refer to
 * the intended columns, which can be assured with the SQL <i>AS</i> clause.
 * <P>
 * A set of updater methods were added to this interface
 * in the JDBC 2.0 API (Java 2 SDK,
 * Standard Edition, version 1.2). The comments regarding parameters
 * to the getter methods also apply to parameters to the
 * updater methods.
 * <P>
 * The updater methods may be used in two ways:
 * <ol>
 * <LI>to update a column value in the current row.  In a scrollable
 *     {@code ResultSet} object, the cursor can be moved backwards
 *     and forwards, to an absolute position, or to a position
 *     relative to the current row.
 *     The following code fragment updates the {@code NAME} column
 *     in the fifth row of the {@code ResultSet} object
 *     {@code rs} and then uses the method {@code updateRow}
 *     to update the data source table from which {@code rs} was derived.
 * <PRE>
 *
 *       rs.absolute(5); // moves the cursor to the fifth row of rs
 *       rs.updateString("NAME", "AINSWORTH"); // updates the
 *          // {@code NAME} column of row 5 to be {@code AINSWORTH}
 *       rs.updateRow(); // updates the row in the data source
 *
 * </PRE>
 * <LI>to insert column values into the insert row.  An updatable
 *     {@code ResultSet} object has a special row associated with
 *     it that serves as a staging area for building a row to be inserted.
 *     The following code fragment moves the cursor to the insert row, builds
 *     a three-column row, and inserts it into {@code rs} and into
 *     the data source table using the method {@code insertRow}.
 * <PRE>
 *
 *       rs.moveToInsertRow(); // moves cursor to the insert row
 *       rs.updateString(1, "AINSWORTH"); // updates the
 *          // first column of the insert row to be {@code AINSWORTH}
 *       rs.updateInt(2,35); // updates the second column to be {@code 35}
 *       rs.updateBoolean(3, true); // updates the third column to {@code true}
 *       rs.insertRow();
 *       rs.moveToCurrentRow();
 *
 * </PRE>
 * </ol>
 * <P>A {@code ResultSet} object is automatically closed when the
 * {@code Statement} object that
 * generated it is closed, re-executed, or used
 * to retrieve the next result from a sequence of multiple results.
 *
 * <P>The number, types and properties of a {@code ResultSet}
 * object's columns are provided by the {@code ResultSetMetaData}
 * object returned by the {@code ResultSet.getMetaData} method.
 *
 * <!-- start release-specific documentation -->
 * <div class="ReleaseSpecificDocumentation">
 * <p class="rshead">HSQLDB-Specific Information:</p>
 *
 * A {@code ResultSet} object generated by HSQLDB is by default of
 * {@code ResultSet.TYPE_FORWARD_ONLY} (as is standard JDBC behavior)
 * and does not allow the use of absolute and relative positioning
 * methods.  If a statement is created with:
 *
 * <pre class="JavaCodeExample">
 * Statement stmt = conn.<b>createStatement</b>(ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_READ_ONLY);
 * </pre>
 *
 * then the {@code ResultSet} objects it produces support
 * using all of  the absolute and relative positioning methods of JDBC2
 * to set the position of the current row, for example:
 *
 * <pre class="JavaCodeExample">
 * rs.<b>absolute</b>(<span class="JavaNumericLiteral">5</span>);
 * String fifthRowValue = rs.<b>getString</b>(<span class="JavaNumericLiteral">1</span>);
 * rs.<b>relative</b>(<span class="JavaNumericLiteral">4</span>);
 * String ninthRowValue = rs.<b>getString</b>(<span class="JavaNumericLiteral">1</span>);
 * </pre>
 *
 * A non-updatable HSQLDB {@code ResultSet} object stays open if it is not
 * explicitly closed, even after its connection is closed. A connection
 * opened with the property setting {@code close_result=true } closes
 * any remaining open results when the connection is closed. <p>
 *
 * From HSQLDB 2.0, there is full support for updatable result sets.
 * Supported methods
 * include all updateXXX methods for the supported types, as well as the
 * {@link #insertRow},
 * {@link #updateRow}, {@link #deleteRow}, {@link #moveToInsertRow}
 * methods.<p>
 *
 * The Statement must be created with ResultSet.CONCUR_UPDATABLE instead of
 * CONCUR_READ_ONLY.<p>
 *
 * Updatability of a result set follows the SQL standards. Some or all columns
 * of an updatable result set can be updated. The current row in such result sets
 * can be deleted using the {@link #deleteRow} method. Some updatable result set
 * can also be inserted into and support {@link #moveToInsertRow}.<p>
 *
 * A result set is updatable if the SELECT statement
 * is updatable. This includes SELECT from TABLE and updatable VIEW objects.
 * An updatable SELECT statement has a single underlying table or view.
 * HSQLDB supports both scrollable and forward-only result sets for
 * updatability.<p>
 *
 * <pre class="JavaCodeExample">
 * -- In the SELECT below, columns A and B are updatable, any row can be
 * -- deleted, but it is not insertable-into as column C is not directly from
 * -- the table.
 * SELECT A, B, A + B AS C FROM T WHERE ...
 *
 * -- The SELECT below can be insertable-into so long as other columns of the
 * -- table that do not appear in the SELECT list have a default value.
 * SELECT A, B FROM T WHERE ...
 * </pre>
 *
 * From version 2.7.4, support for updatable result sets is extended to make
 * updates and deletes visible. This feature is useful with scrollable result
 * sets. After calling {@code updateRow}, the {@code rowUpdated} method now
 * returns {@code true} on an updated row. After calling {@code deleteRow},
 * the {@code rowDeleted} method now returns {@code true} on a deleted row.
 * Calling a {@code getXXX} method returns the updated values for
 * updated rows and {@code null} for deleted rows.
 *
 * (fredt@users) <br>
 * (campbell-burnet@users)
 *
 * </div>
 * @see JDBCStatement#executeQuery
 * @see JDBCStatement#getResultSet
 * @see java.sql.ResultSetMetaData
 *
 * @author Campbell Burnet (campbell-burnet@users dot sourceforge.net)
 * @author Fred Toussi (fredt@users dot sourceforge.net)
 * @version 2.7.4
 * @since JDK 1.1 HSQLDB 1.9.0
 */
public class JDBCResultSet implements ResultSet {

    /**
     * Moves the cursor forward one row from its current position.
     * A {@code ResultSet} cursor is initially positioned
     * before the first row; the first call to the method
     * {@code next} makes the first row the current row; the
     * second call makes the second row the current row, and so on.
     * <p>
     * When a call to the {@code next} method returns {@code false},
     * the cursor is positioned after the last row. Any
     * invocation of a {@code ResultSet} method which requires a
     * current row will result in a {@code SQLException} being thrown.
     *  If the result set type is {@code TYPE_FORWARD_ONLY}, it is vendor specified
     * whether their JDBC driver implementation will return {@code false} or
     *  throw an {@code SQLException} on a
     * subsequent call to {@code next}.
     *
     * <P>If an input stream is open for the current row, a call
     * to the method {@code next} will
     * implicitly close it. A {@code ResultSet} object's
     * warning chain is cleared when a new row is read.
     *
     * @return {@code true} if the new current row is valid;
     * {@code false} if there are no more rows
     * @throws SQLException if a database access error occurs or this method is
     *         called on a closed result set
     */
    public boolean next() throws SQLException {

        checkClosed();

        if (isOnInsertRow || isRowUpdated) {
            throw JDBCUtil.sqlExceptionSQL(ErrorCode.X_24513);
        }

        rootWarning = null;

        return navigator.next();
    }

    /**
     * Releases this {@code ResultSet} object's database and
     * JDBC resources immediately instead of waiting for
     * this to happen when it is automatically closed.
     *
     * <P>The closing of a {@code ResultSet} object does <strong>not</strong> close the {@code Blob},
     * {@code Clob} or {@code NClob} objects created by the {@code ResultSet}. {@code Blob},
     * {@code Clob} or {@code NClob} objects remain valid for at least the duration of the
     * transaction in which they are created, unless their {@code free} method is invoked.
     * <p>
     * When a {@code ResultSet} is closed, any {@code ResultSetMetaData}
     * instances that were created by calling the  {@code getMetaData}
     * method remain accessible.
     *
     * <P><B>Note:</B> A {@code ResultSet} object
     * is automatically closed by the
     * {@code Statement} object that generated it when
     * that {@code Statement} object is closed,
     * re-executed, or is used to retrieve the next result from a
     * sequence of multiple results.
     * <p>
     * Calling the method {@code close} on a {@code ResultSet}
     * object that is already closed is a no-op.
     *
     *
     * @throws SQLException if a database access error occurs
     */
    public void close() throws SQLException {

        if (navigator == null) {
            return;
        }

        if (ResultProperties.isHeld(rsProperties)) {
            session.closeNavigator(navigator.getId());
        } else {
            navigator.release();
        }

        navigator = null;

        if (autoClose && statement != null) {
            statement.close();
        }
    }

    /**
     * Reports whether
     * the last column read had a value of SQL {@code NULL}.
     * Note that you must first call one of the getter methods
     * on a column to try to read its value and then call
     * the method {@code wasNull} to see if the value read was
     * SQL {@code NULL}.
     *
     * @return {@code true} if the last column value read was SQL
     *         {@code NULL} and {@code false} otherwise
     * @throws SQLException if a database access error occurs or this method is
     *         called on a closed result set
     */
    public boolean wasNull() throws SQLException {
        checkClosed();

        return wasNullValue;
    }

    //======================================================================
    // Methods for accessing results by column index
    //======================================================================

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code String} in the Java programming language.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public String getString(int columnIndex) throws SQLException {

        checkColumn(columnIndex);

        Type sourceType = resultMetaData.columnTypes[columnIndex - 1];

        if (sourceType.typeCode == Types.SQL_CLOB) {
            ClobDataID x = (ClobDataID) getColumnInType(
                columnIndex,
                sourceType);

            if (x == null) {
                return null;
            }

            long length = x.length(session);

            if (length > Integer.MAX_VALUE) {
                throw JDBCUtil.sqlException(ErrorCode.X_42561);
            }

            return x.getSubString(session, 0, (int) length);
        }

        return (String) getColumnInType(columnIndex, Type.SQL_VARCHAR);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code boolean} in the Java programming language.
     *
     * <P>If the designated column has a datatype of CHAR or VARCHAR
     * and contains a "0" or has a datatype of BIT, TINYINT, SMALLINT, INTEGER or BIGINT
     * and contains  a 0, a value of {@code false} is returned.  If the designated column has a datatype
     * of CHAR or VARCHAR
     * and contains a "1" or has a datatype of BIT, TINYINT, SMALLINT, INTEGER or BIGINT
     * and contains  a 1, a value of {@code true} is returned.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code false}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public boolean getBoolean(int columnIndex) throws SQLException {
        Object o = getColumnInType(columnIndex, Type.SQL_BOOLEAN);

        return o != null && ((Boolean) o).booleanValue();
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code byte} in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB converts the numeric value to the return type. If the value is
     * out of the range for the return type, an error is returned. For example,
     * this can happen if getByte() or getShort() is used to retrieve a value
     * of type INTEGER or BIGINT and the value is beyond the range covered by
     * the return type.
     *
     * </div>
     * <!-- end release-specific documentation -->
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code 0}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public byte getByte(int columnIndex) throws SQLException {

        Object o = getColumnInType(columnIndex, Type.TINYINT);

        return o == null
               ? 0
               : ((Number) o).byteValue();
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code short} in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB converts the numeric value to the return type. If the value is
     * out of the range for the return type, an error is returned. For example,
     * this can happen if getByte() or getShort() is used to retrieve a value
     * of type INTEGER or BIGINT and the value is beyond the range covered by
     * the return type.
     *
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code 0}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public short getShort(int columnIndex) throws SQLException {

        Object o = getColumnInType(columnIndex, Type.SQL_SMALLINT);

        return o == null
               ? 0
               : ((Number) o).shortValue();
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * an {@code int} in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB converts the numeric value to the return type. If the value is
     * out of the range for the return type, an error is returned. For example,
     * this can happen if getInt() or getLong() is used to retrieve a value
     * of type DECIMAL or NUMERIC with a large precision and the value is beyond
     * the range covered by the return type.
     *
     * </div>
     * <!-- end release-specific documentation -->
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code 0}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public int getInt(int columnIndex) throws SQLException {

        Object o = getColumnInType(columnIndex, Type.SQL_INTEGER);

        return o == null
               ? 0
               : ((Number) o).intValue();
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code long} in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB converts the numeric value to the return type. If the value is
     * out of the range for the return type, an error is returned. For example,
     * this can happen if getInt() or getLong() is used to retrieve a value
     * of type DECIMAL or NUMERIC with a large precision and the value is beyond
     * the range covered by the return type.
     *
     * </div>
     * <!-- end release-specific documentation -->
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code 0}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public long getLong(int columnIndex) throws SQLException {

        Object o = getColumnInType(columnIndex, Type.SQL_BIGINT);

        return o == null
               ? 0
               : ((Number) o).longValue();
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code float} in the Java programming language.
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB converts the numeric value to the return type. If the value is
     * out of the range for the return type, an error is returned. For example,
     * this can happen if getFloat() or getDouble() is used to retrieve a value
     * of type DECIMAL or NUMERIC with a large precision and the value is beyond
     * the range covered by the return type.
     *
     * </div>
     * <!-- end release-specific documentation -->
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code 0}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public float getFloat(int columnIndex) throws SQLException {

        Object o = getColumnInType(columnIndex, Type.SQL_DOUBLE);

        return o == null
               ? (float) 0.0
               : ((Number) o).floatValue();
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code double} in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB converts the numeric value to the return type. If the value is
     * out of the range for the return type, an error is returned. For example,
     * this can happen if getFloat() or getDouble() is used to retrieve a value
     * of type DECIMAL or NUMERIC with a large precision and the value is beyond
     * the range covered by the return type.
     *
     * </div>
     * <!-- end release-specific documentation -->
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code 0}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public double getDouble(int columnIndex) throws SQLException {

        Object o = getColumnInType(columnIndex, Type.SQL_DOUBLE);

        return o == null
               ? 0.0
               : ((Number) o).doubleValue();
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code java.sql.BigDecimal} in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB converts the result and sets the scale
     * with BigDecimal.ROUND_HALF_DOWN.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param scale the number of digits to the right of the decimal point
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @deprecated Use {@code getBigDecimal(int columnIndex)}
     *             or {@code getBigDecimal(String columnLabel)}
     */

//#ifdef DEPRECATEDJDBC
    @Deprecated
    public BigDecimal getBigDecimal(
            int columnIndex,
            int scale)
            throws SQLException {

        if (scale < 0) {
            throw JDBCUtil.outOfRangeArgument();
        }

        BigDecimal bd = getBigDecimal(columnIndex);

        if (bd != null) {
            bd = bd.setScale(scale, RoundingMode.DOWN);
        }

        return bd;
    }

//#endif DEPRECATEDJDBC

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code byte} array in the Java programming language.
     * The bytes represent the raw values returned by the driver.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB returns correct values for columns of binary types
     * {@code BINARY}, {@code BIT},  {@code BLOB}
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public byte[] getBytes(int columnIndex) throws SQLException {

        checkColumn(columnIndex);

        Type sourceType = resultMetaData.columnTypes[columnIndex - 1];

        if (sourceType.typeCode == Types.SQL_BLOB) {
            BlobData x = (BlobData) getColumnInType(columnIndex, sourceType);

            if (x == null) {
                return null;
            }

            long length = x.length(session);

            if (length > Integer.MAX_VALUE) {
                throw JDBCUtil.sqlException(ErrorCode.X_42561);
            }

            return x.getBytes(session, 0, (int) length);
        }

        Object x = getColumnInType(columnIndex, Type.SQL_VARBINARY);

        if (x == null) {
            return null;
        }

        return ((BinaryData) x).getBytes();
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code java.sql.Date} object in the Java programming language.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public Date getDate(int columnIndex) throws SQLException {

        Object t = getColumnInType(columnIndex, Type.SQL_DATE);

        if (t == null) {
            return null;
        }

        return (Date) Type.SQL_DATE.convertSQLToJava(session, t);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code java.sql.Time} object in the Java programming language.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public Time getTime(int columnIndex) throws SQLException {

        Object t = getColumnInType(columnIndex, Type.SQL_TIME);

        if (t == null) {
            return null;
        }

        return (Time) Type.SQL_TIME.convertSQLToJava(session, t);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code java.sql.Timestamp} object in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * The JDBC specification for this method is vague. HSQLDB interprets the
     * specification as follows:
     *
     * <ol>
     * <li>If the SQL type of the column is WITH TIME ZONE, then the UTC value
     * of the returned java.sql.Timestamp object is the UTC of the SQL value
     * without modification.
     * </li>
     * <li>If the SQL type of the column is WITHOUT TIME ZONE, then the
     * UTC value of the returned java.sql.Timestamp will represent the correct
     * timestamp for the time zone (including daylight saving time) of the
     * SQL session.</li>
     * </ol>
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public Timestamp getTimestamp(int columnIndex) throws SQLException {

        TimestampData tsd;
        Object        value = getColumnValue(columnIndex);
        
        if (value == null) {
            return null;
        }

        if (resultMetaData.columnTypes[columnIndex - 1].typeCode
                == Types.SQL_TIMESTAMP_WITH_TIME_ZONE) {
            tsd = (TimestampData) value;
            
            Timestamp ts = new Timestamp(tsd.getMillis());

			ts.setNanos(tsd.getNanos());

			return ts;
        }

        tsd = (TimestampData) getColumnInType(columnIndex, Type.SQL_TIMESTAMP);
        
        return (Timestamp) Type.SQL_TIMESTAMP.convertSQLToJava(session, tsd);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a stream of ASCII characters. The value can then be read in chunks from the
     * stream. This method is particularly
     * suitable for retrieving large {@code LONGVARCHAR} values.
     * The JDBC driver will
     * do any necessary conversion from the database format into ASCII.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a getter method implicitly closes the stream.  Also, a
     * stream may return {@code 0} when the method
     * {@code InputStream.available}
     * is called whether there is data available or not.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * The limitation noted above does not apply to HSQLDB.<p>
     *
     * When the column is of type CHAR and its variations, it requires no
     * conversion since it is represented internally already as a
     * Java String object. When the column is not of type CHAR and its
     * variations, the returned stream is based on a conversion to the
     * Java {@code String} representation of the value. In either case,
     * the obtained stream is always equivalent to a stream of the low order
     * bytes from the value's String representation. <p>
     *
     * HSQLDB SQL {@code CHAR} and its variations are all Unicode strings
     * internally, so the recommended alternatives to this method are
     * {@link #getString(int) getString},
     * {@link #getUnicodeStream(int) getUnicodeStream} (<b>deprecated</b>)
     * and new to 1.7.0: {@link #getCharacterStream(int) getCharacterStream}
     * (now preferred over the deprecated getUnicodeStream alternative).
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a Java input stream that delivers the database column value
     * as a stream of one-byte ASCII characters;
     * if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public java.io.InputStream getAsciiStream(
            int columnIndex)
            throws SQLException {

        Object o = getColumnValue(columnIndex);

        if (o == null) {
            return null;
        }

        if (o instanceof ClobDataID) {
            JDBCClobClient clob = new JDBCClobClient(session, (ClobDataID) o);

            return clob.getAsciiStream();
        } else if (o instanceof Clob) {
            return ((Clob) o).getAsciiStream();
        }

        String s = getString(columnIndex);

        try {
            return new ByteArrayInputStream(s.getBytes(JavaSystem.CS_US_ASCII));
        } catch (Throwable e) {
            throw JDBCUtil.sqlException(e);
        }
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object
     * as a stream of two-byte Unicode characters. The first byte is
     * the high byte; the second byte is the low byte.
     *
     * The value can then be read in chunks from the
     * stream. This method is particularly
     * suitable for retrieving large {@code LONGVARCHAR}values.  The
     * JDBC driver will do any necessary conversion from the database
     * format into Unicode.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a getter method implicitly closes the stream.
     * Also, a stream may return {@code 0} when the method
     * {@code InputStream.available}
     * is called, whether there is data available or not.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * The limitation noted above does not apply to HSQLDB.<p>
     *
     * When the column is of type CHAR and its variations, it requires no
     * conversion since it is represented internally already as
     * Java Strings. When the column is not of type CHAR and its variations,
     * the returned stream is based on a conversion to the
     * Java {@code String} representation of the value. In either case,
     * the obtained stream is always equivalent to a stream of
     * bytes from the value's String representation, with high-byte first.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a Java input stream that delivers the database column value
     *         as a stream of two-byte Unicode characters;
     *         if the value is SQL {@code NULL}, the value returned is
     *         {@code null}
     *
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @deprecated use {@code getCharacterStream} in place of
     *              {@code getUnicodeStream}
     */

//#ifdef DEPRECATEDJDBC
    @Deprecated
    public java.io.InputStream getUnicodeStream(
            int columnIndex)
            throws SQLException {

        String s = getString(columnIndex);

        if (s == null) {
            return null;
        }

        return new StringInputStream(s);
    }

//#endif DEPRECATEDJDBC

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a  stream of
     * uninterpreted bytes. The value can then be read in chunks from the
     * stream. This method is particularly
     * suitable for retrieving large {@code LONGVARBINARY} values.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a getter method implicitly closes the stream.  Also, a
     * stream may return {@code 0} when the method
     * {@code InputStream.available}
     * is called whether there is data available or not.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a Java input stream that delivers the database column value
     *         as a stream of uninterpreted bytes;
     *         if the value is SQL {@code NULL}, the value returned is
     *         {@code null}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public java.io.InputStream getBinaryStream(
            int columnIndex)
            throws SQLException {

        Object o = getColumnValue(columnIndex);

        if (o == null) {
            return null;
        }

        if (o instanceof BlobDataID) {
            return ((BlobDataID) o).getBinaryStream(session);
        } else if (o instanceof Blob) {
            return ((Blob) o).getBinaryStream();
        } else if (o instanceof BinaryData) {
            byte[] b = getBytes(columnIndex);

            return new ByteArrayInputStream(b);
        }

        throw JDBCUtil.sqlException(ErrorCode.X_42561);
    }

    //======================================================================
    // Methods for accessing results by column label
    //======================================================================

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code String} in the Java programming language.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public String getString(String columnLabel) throws SQLException {
        return getString(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code boolean} in the Java programming language.
     *
     * <P>If the designated column has a datatype of CHAR or VARCHAR
     * and contains a "0" or has a datatype of BIT, TINYINT, SMALLINT, INTEGER or BIGINT
     * and contains  a 0, a value of {@code false} is returned.  If the designated column has a datatype
     * of CHAR or VARCHAR
     * and contains a "1" or has a datatype of BIT, TINYINT, SMALLINT, INTEGER or BIGINT
     * and contains  a 1, a value of {@code true} is returned.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code false}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public boolean getBoolean(String columnLabel) throws SQLException {
        return getBoolean(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code byte} in the Java programming language.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code 0}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public byte getByte(String columnLabel) throws SQLException {
        return getByte(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code short} in the Java programming language.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code 0}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public short getShort(String columnLabel) throws SQLException {
        return getShort(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * an {@code int} in the Java programming language.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code 0}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public int getInt(String columnLabel) throws SQLException {
        return getInt(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code long} in the Java programming language.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code 0}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public long getLong(String columnLabel) throws SQLException {
        return getLong(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code float} in the Java programming language.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code 0}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public float getFloat(String columnLabel) throws SQLException {
        return getFloat(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code double} in the Java programming language.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code 0}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public double getDouble(String columnLabel) throws SQLException {
        return getDouble(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code java.math.BigDecimal} in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB converts the result and sets the scale
     * with BigDecimal.ROUND_HALF_DOWN.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param scale the number of digits to the right of the decimal point
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @deprecated Use {@code getBigDecimal(int columnIndex)}
     *             or {@code getBigDecimal(String columnLabel)}
     */

//#ifdef DEPRECATEDJDBC
    @Deprecated
    public BigDecimal getBigDecimal(
            String columnLabel,
            int scale)
            throws SQLException {
        return getBigDecimal(findColumn(columnLabel), scale);
    }

//#endif DEPRECATEDJDBC

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code byte} array in the Java programming language.
     * The bytes represent the raw values returned by the driver.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public byte[] getBytes(String columnLabel) throws SQLException {
        return getBytes(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code java.sql.Date} object in the Java programming language.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public Date getDate(String columnLabel) throws SQLException {
        return getDate(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code java.sql.Time} object in the Java programming language.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column value;
     * if the value is SQL {@code NULL},
     * the value returned is {@code null}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public Time getTime(String columnLabel) throws SQLException {
        return getTime(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code java.sql.Timestamp} object in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * The JDBC specification for this method is vague. HSQLDB interprets the
     * specification as follows:
     *
     * <ol>
     * <li>If the SQL type of the column is WITH TIME ZONE, then the UTC value
     * of the returned java.sql.Timestamp object is the UTC of the SQL value
     * without modification.
     * </li>
     * <li>If the SQL type of the column is WITHOUT TIME ZONE, then the
     * UTC value of the returned java.sql.Timestamp will represent the correct
     * timestamp for the time zone (including daylight saving time) of the
     * SQL session.</li>
     * </ol>
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public Timestamp getTimestamp(String columnLabel) throws SQLException {
        return getTimestamp(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a stream of
     * ASCII characters. The value can then be read in chunks from the
     * stream. This method is particularly
     * suitable for retrieving large {@code LONGVARCHAR} values.
     * The JDBC driver will
     * do any necessary conversion from the database format into ASCII.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a getter method implicitly closes the stream. Also, a
     * stream may return {@code 0} when the method {@code available}
     * is called whether there is data available or not.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return a Java input stream that delivers the database column value
     * as a stream of one-byte ASCII characters.
     * If the value is SQL {@code NULL},
     * the value returned is {@code null}.
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public java.io.InputStream getAsciiStream(
            String columnLabel)
            throws SQLException {
        return getAsciiStream(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a stream of two-byte
     * Unicode characters. The first byte is the high byte; the second
     * byte is the low byte.
     *
     * The value can then be read in chunks from the
     * stream. This method is particularly
     * suitable for retrieving large {@code LONGVARCHAR} values.
     * The JDBC technology-enabled driver will
     * do any necessary conversion from the database format into Unicode.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a getter method implicitly closes the stream.
     * Also, a stream may return {@code 0} when the method
     * {@code InputStream.available} is called, whether there
     * is data available or not.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return a Java input stream that delivers the database column value
     *         as a stream of two-byte Unicode characters.
     *         If the value is SQL {@code NULL}, the value returned
     *         is {@code null}.
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @deprecated use {@code getCharacterStream} instead
     */

//#ifdef DEPRECATEDJDBC
    @Deprecated
    public java.io.InputStream getUnicodeStream(
            String columnLabel)
            throws SQLException {
        return getUnicodeStream(findColumn(columnLabel));
    }

//#endif DEPRECATEDJDBC

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a stream of uninterpreted
     * {@code byte}s.
     * The value can then be read in chunks from the
     * stream. This method is particularly
     * suitable for retrieving large {@code LONGVARBINARY}
     * values.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a getter method implicitly closes the stream. Also, a
     * stream may return {@code 0} when the method {@code available}
     * is called whether there is data available or not.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return a Java input stream that delivers the database column value
     * as a stream of uninterpreted bytes;
     * if the value is SQL {@code NULL}, the result is {@code null}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public java.io.InputStream getBinaryStream(
            String columnLabel)
            throws SQLException {
        return getBinaryStream(findColumn(columnLabel));
    }

    //=====================================================================
    // Advanced features:
    //=====================================================================

    /**
     * Retrieves the first warning reported by calls on this
     * {@code ResultSet} object.
     * Subsequent warnings on this {@code ResultSet} object
     * will be chained to the {@code SQLWarning} object that
     * this method returns.
     *
     * <P>The warning chain is automatically cleared each time a new
     * row is read.  This method may not be called on a {@code ResultSet}
     * object that has been closed; doing so will cause an
     * {@code SQLException} to be thrown.
     * <P>
     * <B>Note:</B> This warning chain only covers warnings caused
     * by {@code ResultSet} methods.  Any warning caused by
     * {@code Statement} methods
     * (such as reading OUT parameters) will be chained on the
     * {@code Statement} object.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not produce {@code SQLWarning}
     * objects on any ResultSet object warning chain; this
     * method always returns {@code null}.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @return the first {@code SQLWarning} object reported or
     *         {@code null} if there are none
     * @throws SQLException if a database access error occurs or this method is
     *         called on a closed result set
     */
    public SQLWarning getWarnings() throws SQLException {
        checkClosed();

        return rootWarning;
    }

    /**
     * Clears all warnings reported on this {@code ResultSet} object.
     * After this method is called, the method {@code getWarnings}
     * returns {@code null} until a new warning is
     * reported for this {@code ResultSet} object.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not produce {@code SQLWarning}
     * objects on any ResultSet object warning chain; calls to this method
     * are ignored.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @throws SQLException if a database access error occurs or this method is
     *         called on a closed result set
     */
    public void clearWarnings() throws SQLException {
        checkClosed();

        rootWarning = null;
    }

    /**
     * Retrieves the name of the SQL cursor used by this {@code ResultSet}
     * object.
     *
     * <P>In SQL, a result table is retrieved through a cursor that is
     * named. The current row of a result set can be updated or deleted
     * using a positioned update/delete statement that references the
     * cursor name. To ensure that the cursor has the proper isolation
     * level to support update, the cursor's {@code SELECT} statement
     * should be of the form {@code SELECT FOR UPDATE}. If
     * {@code FOR UPDATE} is omitted, the positioned updates may fail.
     *
     * <P>The JDBC API supports this SQL feature by providing the name of the
     * SQL cursor used by a {@code ResultSet} object.
     * The current row of a {@code ResultSet} object
     * is also the current row of this SQL cursor.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature when the cursor has a name.
     *
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @return the SQL name for this {@code ResultSet} object's cursor
     * @throws SQLException if a database access error occurs or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     */
    public String getCursorName() throws SQLException {

        checkClosed();

        if (result == null) {
            return "";
        }

        return result.getMainString();
    }

    /**
     * Retrieves the  number, types and properties of
     * this {@code ResultSet} object's columns.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * <B>Example:</B> <p>
     *
     * The following code fragment creates a {@code ResultSet} object rs,
     * creates a {@code ResultSetMetaData} object rsmd, and uses rsmd
     * to find out how many columns rs has and whether the first column
     * in rs can be used in a {@code WHERE} clause.
     *
     * <pre class="JavaCodeExample">
     * ResultSet rs   = stmt.<b>executeQuery</b>(<span class="JavaStringLiteral">"SELECT a, b, c FROM TABLE2"</span>);
     * ResultSetMetaData rsmd = rs.<b>getMetaData</b>();<br>
     * int numberOfColumns = rsmd.<b>getColumnCount</b>();<br>
     * boolean b = rsmd.<b>isSearchable</b>(1);<br>
     * </pre>
     *
     * <hr>
     *
     * <B>Changes:</B> <p>
     *
     * With version 2.0, the engine's SQL implementation has been
     * completely rewritten. Changes to this class and the implementation of
     * ResultSetMetaData reflect the engine's new capabilities and provide
     * more accurate information. <p>
     *
     * changes to consider:
     *
     * <ol>
     * <li>isAutoIncrement(int) <i>always</i> returned {@code false}</li>
     * <li>isNullable(int) returns the nullability of a real table or view
     *     column in the ResultSet and returns
     *     {@code columnNoNulls} for non-base-column ResultSet columns
     *     (columns of the ResultSet that are based on expressions or
     *     aggregates).</li>
     * <li>getColumnDisplaySize(int) returns correct results even for expression
     *     columns.</li>
     * <li>getPrecision(int) returns the correct precision even for expression
     *     columns.</li>
     * <li>getScale(int) returns the correct precision even for expression
     *     columns.</li>
     * <li>getCatalogName(int) returns the catalog name of the database.</li>
     * </ol>
     *
     * <hr>
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @return the description of this {@code ResultSet} object's columns
     * @throws SQLException if a database access error occurs or this method is
     *         called on a closed result set
     */
    public ResultSetMetaData getMetaData() throws SQLException {

        checkClosed();

        if (resultSetMetaData == null) {
            resultSetMetaData = new JDBCResultSetMetaData(
                resultMetaData,
                isUpdatable,
                isInsertable,
                connection);
        }

        return resultSetMetaData;
    }

    /**
     * <p>Gets the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * an {@code Object} in the Java programming language.
     *
     * <p>This method will return the value of the given column as a
     * Java object.  The type of the Java object will be the default
     * Java object type corresponding to the column's SQL type,
     * following the mapping for built-in types specified in the JDBC
     * specification. If the value is an SQL {@code NULL},
     * the driver returns a Java {@code null}.
     *
     * <p>This method may also be used to read database-specific
     * abstract data types.
     *
     * In the JDBC 2.0 API, the behavior of method
     * {@code getObject} is extended to materialize
     * data of SQL user-defined types.
     * <p>
     * If {@code Connection.getTypeMap} does not throw a
     * {@code SQLFeatureNotSupportedException},
     * then when a column contains a structured or distinct value,
     * the behavior of this method is as
     * if it were a call to: {@code getObject(columnIndex,
     * this.getStatement().getConnection().getTypeMap())}.
     *
     * If {@code Connection.getTypeMap} does throw a
     * {@code SQLFeatureNotSupportedException},
     * then structured values are not supported, and distinct values
     * are mapped to the default Java class as determined by the
     * underlying SQL type of the DISTINCT type.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a {@code java.lang.Object} holding the column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public Object getObject(int columnIndex) throws SQLException {

        checkColumn(columnIndex);

        Type sourceType = resultMetaData.columnTypes[columnIndex - 1];

        switch (sourceType.typeCode) {

            case Types.SQL_ARRAY :
                return getArray(columnIndex);

            case Types.SQL_DATE :
                return getDate(columnIndex);

            case Types.SQL_TIME :
                return getTime(columnIndex);

            case Types.SQL_TIME_WITH_TIME_ZONE :
                return getTimeWithZone(columnIndex);

            case Types.SQL_TIMESTAMP :
                return getTimestamp(columnIndex);

            case Types.SQL_TIMESTAMP_WITH_TIME_ZONE :
                return getTimestampWithZone(columnIndex);

            case Types.SQL_BINARY :
            case Types.SQL_VARBINARY :
                return getBytes(columnIndex);

            case Types.SQL_GUID : {
                BinaryData bd = (BinaryData) getColumnValue(columnIndex);

                return BinaryUUIDType.getJavaUUID(bd);
            }

            case Types.SQL_BIT : {
                BinaryData bd = (BinaryData) getColumnValue(columnIndex);

                if (wasNull()) {
                    return null;
                }

                byte[] bytes = bd.getBytes();

                if (bytes.length == 0) {
                    return Boolean.FALSE;
                }

                return bytes[0] == 0
                       ? Boolean.FALSE
                       : Boolean.TRUE;
            }

            case Types.SQL_CLOB :
                return getClob(columnIndex);

            case Types.SQL_BLOB :
                return getBlob(columnIndex);

            case Types.OTHER :
            case Types.JAVA_OBJECT : {
                Object o = getColumnInType(columnIndex, sourceType);

                if (o == null) {
                    return null;
                }

                try {
                    return ((JavaObjectData) o).getObject();
                } catch (HsqlException e) {
                    throw JDBCUtil.sqlException(e);
                }
            }

            default :
                return getColumnInType(columnIndex, sourceType);
        }
    }

    /**
     * <p>Gets the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * an {@code Object} in the Java programming language.
     *
     * <p>This method will return the value of the given column as a
     * Java object.  The type of the Java object will be the default
     * Java object type corresponding to the column's SQL type,
     * following the mapping for built-in types specified in the JDBC
     * specification. If the value is an SQL {@code NULL},
     * the driver returns a Java {@code null}.
     * <P>
     * This method may also be used to read database-specific
     * abstract data types.
     * <P>
     * In the JDBC 2.0 API, the behavior of the method
     * {@code getObject} is extended to materialize
     * data of SQL user-defined types.  When a column contains
     * a structured or distinct value, the behavior of this method is as
     * if it were a call to: {@code getObject(columnIndex,
     * this.getStatement().getConnection().getTypeMap())}.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return a {@code java.lang.Object} holding the column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     */
    public Object getObject(String columnLabel) throws SQLException {
        return getObject(findColumn(columnLabel));
    }

    //----------------------------------------------------------------

    /**
     * Maps the given {@code ResultSet} column label to its
     * {@code ResultSet} column index.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * Starting with 1.9.x, HSQLDB does an exhaustive search, backed by
     * a cache lookup (to improve performance for subsequent invocations with
     * a given input). <p>
     *
     * Upon careful investigation of the JDBC specification and the behaviour
     * of existing JDBC drivers, there is actually nothing preventing the
     * findColumn method from doing an exhaustive search, as long as it conforms
     * to the following rules (which describe the new implementation):
     *
     * <ol>
     * <li> the entire search is case insensitive
     * <li> each search iteration occurs from leftmost to rightmost column,
     *      returning the first match encountered
     * <li> the first pass matches only bare column labels
     * <li> the second pass matches only simple column names
     * <li> further passes conform to the identifier qualification
     *      and identifier quoting rules of the engine
     * </ol>
     *
     * In this implementation, the SQL tokenizer is not employed, both because
     * it does not yet correctly handle greater than two part qualification
     * and also because is is not immediately considered important to do a
     * truly exhaustive search, handling the full range of possibly mixed quoted
     * and unquoted identifier components. <p>
     *
     * Instead:
     * <ul>
     * <li> a third pass matches simple table-dot-column qualified names
     * <li> a fourth pass matches simple schema-dot-table-dot-column qualified column names
     * </ul>
     * </div>
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column index of the given column name
     * @throws SQLException if the {@code ResultSet} object
     * does not contain a column labeled {@code columnLabel}, a database access error occurs
     *  or this method is called on a closed result set
     */
    public int findColumn(final String columnLabel) throws SQLException {

        checkClosed();

        if (columnLabel == null) {
            throw JDBCUtil.nullArgument();
        }

        int columnIndex;

        // faster lookup for subsequent access
        if (columnMap != null) {
            columnIndex = columnMap.get(columnLabel, -1);

            if (columnIndex != -1) {
                return columnIndex;
            }
        }

        final String[] colLabels = resultMetaData.columnLabels;

        columnIndex = -1;

        // column labels first, to preference column aliases
        for (int i = 0; i < columnCount; i++) {
            if (columnLabel.equalsIgnoreCase(colLabels[i])) {
                columnIndex = i;
                break;
            }
        }

        final ColumnBase[] columns = resultMetaData.columns;

        // then bare column names, to preference simple
        // quoted column idents that *may* contain "."
        if (columnIndex < 0) {
            for (int i = 0; i < columnCount; i++) {
                if (columnLabel.equalsIgnoreCase(columns[i].getNameString())) {
                    columnIndex = i;
                    break;
                }
            }
        }

        // then table-qualified column names (again, quoted
        // table idents *may* contain "."
        // As a last resort, "fully" qualified column names
        // (we don't yet bother with catalog qualification)
        if (columnIndex < 0) {
            int position = columnLabel.indexOf('.');

            if (position < 0) {
                throw JDBCUtil.sqlException(
                    ErrorCode.JDBC_COLUMN_NOT_FOUND,
                    columnLabel);
            }

            for (int i = 0; i < columnCount; i++) {
                final String tabName = columns[i].getTableNameString();

                if (tabName == null || tabName.isEmpty()) {
                    continue;
                }

                final String colName = columns[i].getNameString();

                if (columnLabel.equalsIgnoreCase(tabName + '.' + colName)) {
                    columnIndex = i;
                    break;
                }

                final String schemName = columns[i].getSchemaNameString();

                if (schemName == null || schemName.isEmpty()) {
                    continue;
                }

                String match = new StringBuilder(schemName).append('.')
                        .append(tabName)
                        .append('.')
                        .append(colName)
                        .toString();

                if (columnLabel.equalsIgnoreCase(match)) {
                    columnIndex = i;
                    break;
                }
            }
        }

        if (columnIndex < 0) {
            throw JDBCUtil.sqlException(
                ErrorCode.JDBC_COLUMN_NOT_FOUND,
                columnLabel);
        }

        columnIndex++;

        if (columnMap == null) {
            columnMap = new IntValueHashMap<>();
        }

        columnMap.put(columnLabel, columnIndex);

        return columnIndex;
    }

    //--------------------------JDBC 2.0-----------------------------------
    //---------------------------------------------------------------------
    // Getters and Setters
    //---------------------------------------------------------------------

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a
     * {@code java.io.Reader} object.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @return a {@code java.io.Reader} object that contains the column
     * value; if the value is SQL {@code NULL}, the value returned is
     * {@code null} in the Java programming language.
     * @param columnIndex the first column is 1, the second is 2, ...
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     * @since JDK 1.2
     */
    public java.io.Reader getCharacterStream(
            int columnIndex)
            throws SQLException {

        checkColumn(columnIndex);

        Type   sourceType = resultMetaData.columnTypes[columnIndex - 1];
        Object o          = getColumnInType(columnIndex, sourceType);

        if (o == null) {
            return null;
        }

        if (o instanceof ClobDataID) {
            return ((ClobDataID) o).getCharacterStream(session);
        } else if (o instanceof Clob) {
            return ((Clob) o).getCharacterStream();
        } else if (o instanceof String) {
            return new StringReader((String) o);
        }

        throw JDBCUtil.sqlException(ErrorCode.X_42561);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a
     * {@code java.io.Reader} object.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return a {@code java.io.Reader} object that contains the column
     * value; if the value is SQL {@code NULL}, the value returned is
     * {@code null} in the Java programming language
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     * @since JDK 1.2
     */
    public java.io.Reader getCharacterStream(
            String columnLabel)
            throws SQLException {
        return getCharacterStream(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a
     * {@code java.math.BigDecimal} with full precision.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value (full precision);
     * if the value is SQL {@code NULL}, the value returned is
     * {@code null} in the Java programming language.
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     * @since JDK 1.2
     */
    public BigDecimal getBigDecimal(int columnIndex) throws SQLException {

        checkColumn(columnIndex);

        Type targetType = resultMetaData.columnTypes[columnIndex - 1];

        switch (targetType.typeCode) {

            case Types.SQL_NUMERIC :
            case Types.SQL_DECIMAL :
                break;

            case Types.TINYINT :
            case Types.SQL_SMALLINT :
            case Types.SQL_INTEGER :
            case Types.SQL_BIGINT :
                targetType = Type.SQL_DECIMAL;
                break;

            case Types.SQL_DOUBLE :
            default :
                targetType = Type.SQL_DECIMAL_DEFAULT;
                break;
        }

        return (BigDecimal) getColumnInType(columnIndex, targetType);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a
     * {@code java.math.BigDecimal} with full precision.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return the column value (full precision);
     * if the value is SQL {@code NULL}, the value returned is
     * {@code null} in the Java programming language.
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs or this method is
     *            called on a closed result set
     * @since JDK 1.2
     */
    public BigDecimal getBigDecimal(String columnLabel) throws SQLException {
        return getBigDecimal(findColumn(columnLabel));
    }

    //---------------------------------------------------------------------
    // Traversal/Positioning
    //---------------------------------------------------------------------

    /**
     * Retrieves whether the cursor is before the first row in
     * this {@code ResultSet} object.
     * <p>
     * <strong>Note:</strong>Support for the {@code isBeforeFirst} method
     * is optional for {@code ResultSet}s with a result
     * set type of {@code TYPE_FORWARD_ONLY}
     *
     * @return {@code true} if the cursor is before the first row;
     * {@code false} if the cursor is at any other position or the
     * result set contains no rows
     * @throws SQLException if a database access error occurs or this method is
     *            called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public boolean isBeforeFirst() throws SQLException {

        checkClosed();

        if (isOnInsertRow) {
            return false;
        }

        return navigator.isBeforeFirst();
    }

    /**
     * Retrieves whether the cursor is after the last row in
     * this {@code ResultSet} object.
     * <p>
     * <strong>Note:</strong>Support for the {@code isAfterLast} method
     * is optional for {@code ResultSet}s with a result
     * set type of {@code TYPE_FORWARD_ONLY}
     *
     * @return {@code true} if the cursor is after the last row;
     * {@code false} if the cursor is at any other position or the
     * result set contains no rows
     * @throws SQLException if a database access error occurs or this method is
     *         called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public boolean isAfterLast() throws SQLException {

        // At afterLast condition exists when resultset has been traversed and
        // the current row is null.  iCurrentRow should also be set to
        // afterlast but no need to test
        checkClosed();

        if (isOnInsertRow) {
            return false;
        }

        return navigator.isAfterLast();
    }

    /**
     * Retrieves whether the cursor is on the first row of
     * this {@code ResultSet} object.
     * <p>
     * <strong>Note:</strong>Support for the {@code isFirst} method
     * is optional for {@code ResultSet}s with a result
     * set type of {@code TYPE_FORWARD_ONLY}
     *
     * @return {@code true} if the cursor is on the first row;
     * {@code false} otherwise
     * @throws SQLException if a database access error occurs or this method is
     *         called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public boolean isFirst() throws SQLException {

        checkClosed();

        if (isOnInsertRow) {
            return false;
        }

        return navigator.isFirst();
    }

    /**
     * Retrieves whether the cursor is on the last row of
     * this {@code ResultSet} object.
     *  <strong>Note:</strong> Calling the method {@code isLast} may be expensive
     * because the JDBC driver
     * might need to fetch ahead one row in order to determine
     * whether the current row is the last row in the result set.
     * <p>
     * <strong>Note:</strong> Support for the {@code isLast} method
     * is optional for {@code ResultSet}s with a result
     * set type of {@code TYPE_FORWARD_ONLY}
     *
     * @return {@code true} if the cursor is on the last row;
     * {@code false} otherwise
     * @throws SQLException if a database access error occurs or this method is
     *            called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public boolean isLast() throws SQLException {

        checkClosed();

        if (isOnInsertRow) {
            return false;
        }

        return navigator.isLast();
    }

    /**
     * Moves the cursor to the front of
     * this {@code ResultSet} object, just before the
     * first row. This method has no effect if the result set contains no rows.
     *
     * @throws SQLException if a database access error
     * occurs; this method is called on a closed result set or the
     * result set type is {@code TYPE_FORWARD_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void beforeFirst() throws SQLException {

        checkClosed();
        checkNotForwardOnly();

        if (isOnInsertRow || isRowUpdated) {
            throw JDBCUtil.sqlExceptionSQL(ErrorCode.X_24513);
        }

        navigator.beforeFirst();
    }

    /**
     * Moves the cursor to the end of
     * this {@code ResultSet} object, just after the
     * last row. This method has no effect if the result set contains no rows.
     * @throws SQLException if a database access error
     * occurs; this method is called on a closed result set
     * or the result set type is {@code TYPE_FORWARD_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void afterLast() throws SQLException {

        checkClosed();
        checkNotForwardOnly();

        if (isOnInsertRow || isRowUpdated) {
            throw JDBCUtil.sqlExceptionSQL(ErrorCode.X_24513);
        }

        navigator.afterLast();
    }

    /**
     * Moves the cursor to the first row in
     * this {@code ResultSet} object.
     *
     * @return {@code true} if the cursor is on a valid row;
     * {@code false} if there are no rows in the result set
     * @throws SQLException if a database access error
     * occurs; this method is called on a closed result set
     * or the result set type is {@code TYPE_FORWARD_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public boolean first() throws SQLException {

        checkClosed();
        checkNotForwardOnly();

        if (isOnInsertRow || isRowUpdated) {
            throw JDBCUtil.sqlExceptionSQL(ErrorCode.X_24513);
        }

        return navigator.first();
    }

    /**
     * Moves the cursor to the last row in
     * this {@code ResultSet} object.
     *
     * @return {@code true} if the cursor is on a valid row;
     * {@code false} if there are no rows in the result set
     * @throws SQLException if a database access error
     * occurs; this method is called on a closed result set
     * or the result set type is {@code TYPE_FORWARD_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public boolean last() throws SQLException {

        checkClosed();
        checkNotForwardOnly();

        if (isOnInsertRow || isRowUpdated) {
            throw JDBCUtil.sqlExceptionSQL(ErrorCode.X_24513);
        }

        return navigator.last();
    }

    /**
     * Retrieves the current row number.  The first row is number 1, the
     * second number 2, and so on.
     * <p>
     * <strong>Note:</strong>Support for the {@code getRow} method
     * is optional for {@code ResultSet}s with a result
     * set type of {@code TYPE_FORWARD_ONLY}
     *
     * @return the current row number; {@code 0} if there is no current row
     * @throws SQLException if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public int getRow() throws SQLException {

        checkClosed();

        if (navigator.isAfterLast()) {
            return 0;
        }

        return navigator.getRowNumber() + 1;
    }

    /**
     * Moves the cursor to the given row number in
     * this {@code ResultSet} object.
     *
     * <p>If the row number is positive, the cursor moves to
     * the given row number with respect to the
     * beginning of the result set.  The first row is row 1, the second
     * is row 2, and so on.
     *
     * <p>If the given row number is negative, the cursor moves to
     * an absolute row position with respect to
     * the end of the result set.  For example, calling the method
     * {@code absolute(-1)} positions the
     * cursor on the last row; calling the method {@code absolute(-2)}
     * moves the cursor to the next-to-last row, and so on.
     *
     * <p>If the row number specified is zero, the cursor is moved to
     * before the first row.
     *
     * <p>An attempt to position the cursor beyond the first/last row in
     * the result set leaves the cursor before the first row or after
     * the last row.
     *
     * <p><B>Note:</B> Calling {@code absolute(1)} is the same
     * as calling {@code first()}. Calling {@code absolute(-1)}
     * is the same as calling {@code last()}.
     *
     * @param row the number of the row to which the cursor should move.
     *        A value of zero indicates that the cursor will be positioned
     *        before the first row; a positive number indicates the row number
     *        counting from the beginning of the result set; a negative number
     *        indicates the row number counting from the end of the result set
     * @return {@code true} if the cursor is moved to a position in this
     * {@code ResultSet} object;
     * {@code false} if the cursor is before the first row or after the
     * last row
     * @throws SQLException if a database access error
     * occurs; this method is called on a closed result set
     * or the result set type is {@code TYPE_FORWARD_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public boolean absolute(int row) throws SQLException {

        checkClosed();
        checkNotForwardOnly();

        if (isOnInsertRow || isRowUpdated) {
            throw JDBCUtil.sqlExceptionSQL(ErrorCode.X_24513);
        }

        if (row > 0) {
            row--;
        } else if (row == 0) {
            return navigator.beforeFirst();
        } else {
            row += navigator.getSize();

            if (row < 0) {
                navigator.beforeFirst();

                return false;
            }
        }

        return navigator.absolute(row);
    }

    /**
     * Moves the cursor a relative number of rows, either positive or negative.
     * Attempting to move beyond the first/last row in the
     * result set positions the cursor before/after
     * the first/last row. Calling {@code relative(0)} is valid, but does
     * not change the cursor position.
     *
     * <p>Note: Calling the method {@code relative(1)}
     * is identical to calling the method {@code next()} and
     * calling the method {@code relative(-1)} is identical
     * to calling the method {@code previous()}.
     *
     * @param rows an {@code int} specifying the number of rows to
     *        move from the current row; a positive number moves the cursor
     *        forward; a negative number moves the cursor backward
     * @return {@code true} if the cursor is on a row;
     *         {@code false} otherwise
     * @throws SQLException if a database access error occurs;  this method
     * is called on a closed result set or the result set type is
     *            {@code TYPE_FORWARD_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public boolean relative(int rows) throws SQLException {

        checkClosed();
        checkNotForwardOnly();

        if (isOnInsertRow || isRowUpdated) {
            throw JDBCUtil.sqlExceptionSQL(ErrorCode.X_24513);
        }

        return navigator.relative(rows);
    }

    /**
     * Moves the cursor to the previous row in this
     * {@code ResultSet} object.
     * <p>
     * When a call to the {@code previous} method returns {@code false},
     * the cursor is positioned before the first row.  Any invocation of a
     * {@code ResultSet} method which requires a current row will result in a
     * {@code SQLException} being thrown.
     * <p>
     * If an input stream is open for the current row, a call to the method
     * {@code previous} will implicitly close it.  A {@code ResultSet}
     *  object's warning change is cleared when a new row is read.
     *
     * @return {@code true} if the cursor is now positioned on a valid row;
     * {@code false} if the cursor is positioned before the first row
     * @throws SQLException if a database access error
     * occurs; this method is called on a closed result set
     * or the result set type is {@code TYPE_FORWARD_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public boolean previous() throws SQLException {

        checkClosed();
        checkNotForwardOnly();

        if (isOnInsertRow || isRowUpdated) {
            throw JDBCUtil.sqlExceptionSQL(ErrorCode.X_24513);
        }

        rootWarning = null;

        return navigator.previous();
    }

    //---------------------------------------------------------------------
    // Properties
    //---------------------------------------------------------------------

    /**
     * Gives a hint as to the direction in which the rows in this
     * {@code ResultSet} object will be processed.
     * The initial value is determined by the
     * {@code Statement} object
     * that produced this {@code ResultSet} object.
     * The fetch direction may be changed at any time.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not need this hint. However, as mandated by the JDBC standard,
     * an SQLException is thrown if the result set type is TYPE_FORWARD_ONLY
     * and a fetch direction other than FETCH_FORWARD is requested.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param direction an {@code int} specifying the suggested
     *        fetch direction; one of {@code ResultSet.FETCH_FORWARD},
     *        {@code ResultSet.FETCH_REVERSE}, or
     *        {@code ResultSet.FETCH_UNKNOWN}
     * @throws SQLException if a database access error occurs; this
     * method is called on a closed result set or
     * the result set type is {@code TYPE_FORWARD_ONLY} and the fetch
     * direction is not {@code FETCH_FORWARD}
     * @since JDK 1.2
     * @see JDBCStatement#setFetchDirection
     * @see #getFetchDirection
     */
    public void setFetchDirection(int direction) throws SQLException {

        checkClosed();

        switch (direction) {

            case ResultSet.FETCH_FORWARD : {
                break;
            }

            case ResultSet.FETCH_REVERSE :
            case ResultSet.FETCH_UNKNOWN : {
                checkNotForwardOnly();
                break;
            }

            default : {
                throw JDBCUtil.notSupported();
            }
        }
    }

    /**
     * Retrieves the fetch direction for this
     * {@code ResultSet} object.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not depend on fetch direction and always returns
     * {@code FETCH_FORWARD}, but the value has no real meaning.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @return the current fetch direction for this {@code ResultSet} object
     * @throws SQLException if a database access error occurs
     * or this method is called on a closed result set
     * @since JDK 1.2
     * @see #setFetchDirection
     */
    public int getFetchDirection() throws SQLException {
        checkClosed();

        return ResultSet.FETCH_FORWARD;
    }

    /**
     * Gives the JDBC driver a hint as to the number of rows that should
     * be fetched from the database when more rows are needed for this
     * {@code ResultSet} object.
     * If the fetch size specified is zero, the JDBC driver
     * ignores the value and is free to make its own best guess as to what
     * the fetch size should be.  The default value is set by the
     * {@code Statement} object
     * that created the result set.  The fetch size may be changed at any time.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB may not build and return a result set as a
     * whole. Therefore the supplied, non-zero, fetch size value is used for
     * some ResultSet objects.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param rows the number of rows to fetch
     * @throws SQLException if a database access error occurs; this method
     * is called on a closed result set or the
     * condition {@code rows >= 0} is not satisfied
     * @since JDK 1.2
     * @see #getFetchSize
     * @see JDBCStatement#setFetchSize
     * @see JDBCStatement#getFetchSize
     */
    public void setFetchSize(int rows) throws SQLException {
        if (rows < 0) {
            throw JDBCUtil.outOfRangeArgument();
        }
    }

    /**
     * Retrieves the fetch size for this
     * {@code ResultSet} object.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB may not build and return a result set as a
     * whole. The actual fetch size for this result set is returned.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @return the current fetch size for this {@code ResultSet} object
     * @throws SQLException if a database access error occurs
     * or this method is called on a closed result set
     * @since JDK 1.2
     * @see #setFetchSize
     * @see JDBCStatement#getFetchSize
     * @see JDBCStatement#setFetchSize
     */
    public int getFetchSize() throws SQLException {
        checkClosed();

        return fetchSize;
    }

    /**
     * Retrieves the type of this {@code ResultSet} object.
     * The type is determined by the {@code Statement} object
     * that created the result set.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB accurately
     * reports the actual runtime scrollability of this result set instance.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @return {@code ResultSet.TYPE_FORWARD_ONLY},
     *         {@code ResultSet.TYPE_SCROLL_INSENSITIVE},
     *         or {@code ResultSet.TYPE_SCROLL_SENSITIVE}
     * @throws SQLException if a database access error occurs
     * or this method is called on a closed result set
     * @since JDK 1.2
     */
    public int getType() throws SQLException {
        checkClosed();

        return ResultProperties.getJDBCScrollability(rsProperties);
    }

    /**
     * Retrieves the concurrency mode of this {@code ResultSet} object.
     * The concurrency used is determined by the
     * {@code Statement} object that created the result set.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports updatable result sets and accurately reports the actual
     * runtime concurrency of this result set instance.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @return the concurrency type, either
     *         {@code ResultSet.CONCUR_READ_ONLY}
     *         or {@code ResultSet.CONCUR_UPDATABLE}
     * @throws SQLException if a database access error occurs
     * or this method is called on a closed result set
     * @since JDK 1.2
     */
    public int getConcurrency() throws SQLException {
        checkClosed();

        return ResultProperties.getJDBCConcurrency(rsProperties);
    }

    //---------------------------------------------------------------------
    // Updates
    //---------------------------------------------------------------------

    /**
     * Retrieves whether the current row has been updated.  The value returned
     * depends on whether or not the result set can detect updates.
     * <p>
     * <strong>Note:</strong> Support for the {@code rowUpdated} method is optional with a result set
     * concurrency of {@code CONCUR_READ_ONLY}
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports updatable result sets and accurately reports the actual
     * value.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @return {@code true} if the current row is detected to
     * have been visibly updated by the owner or another; {@code false} otherwise
     * @throws SQLException if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @see JDBCDatabaseMetaData#updatesAreDetected
     * @since JDK 1.2
     */
    public boolean rowUpdated() throws SQLException {

        checkClosed();

        if (isUpdatable) {
            Object[] data = getCurrent();
            Integer status =
                (Integer) data[result.metaData.getColumnCount() + ResultMetaData.SysOffsets.rowStatus];

            if (status != null) {
                return status.intValue() == ResultMetaData.RowStatus.updated;
            }
        }

        return false;
    }

    /**
     * Retrieves whether the current row has had an insertion.
     * The value returned depends on whether or not this
     * {@code ResultSet} object can detect visible inserts.
     * <p>
     * <strong>Note:</strong> Support for the {@code rowInserted} method is optional with a result set
     * concurrency of {@code CONCUR_READ_ONLY}
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB always returns false as it does not add inserted rows to the original result set.

     * </div>
     * <!-- end release-specific documentation -->
     *
     * @return {@code true} if the current row is detected to
     * have been inserted; {@code false} otherwise
     * @throws SQLException if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @see JDBCDatabaseMetaData#insertsAreDetected
     * @since JDK 1.2
     */
    public boolean rowInserted() throws SQLException {
        checkClosed();

        return false;
    }

    /**
     * Retrieves whether a row has been deleted.  A deleted row may leave
     * a visible "hole" in a result set.  This method can be used to
     * detect holes in a result set.  The value returned depends on whether
     * or not this {@code ResultSet} object can detect deletions.
     * <p>
     * <strong>Note:</strong> Support for the {@code rowDeleted} method is optional with a result set
     * concurrency of {@code CONCUR_READ_ONLY}
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports updatable result sets and accurately reports the actual
     * value.
     * </div>
     * <!-- end release-specific documentation -->
     * @return {@code true} if the current row is detected to
     * have been deleted by the owner or another; {@code false} otherwise
     * @throws SQLException if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @see JDBCDatabaseMetaData#deletesAreDetected
     * @since JDK 1.2
     */
    public boolean rowDeleted() throws SQLException {

        checkClosed();

        if (isUpdatable) {
            Object[] data = getCurrent();
            Integer status =
                (Integer) data[result.metaData.getColumnCount() + ResultMetaData.SysOffsets.rowStatus];

            if (status != null) {
                return status.intValue() == ResultMetaData.RowStatus.deleted;
            }
        }

        return false;
    }

    /**
     * Updates the designated column with a {@code null} value.
     *
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow}
     * or {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateNull(int columnIndex) throws SQLException {
        setParameter(columnIndex, null);
    }

    /**
     * Updates the designated column with a {@code boolean} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateBoolean(int columnIndex, boolean x) throws SQLException {
        setParameter(columnIndex, x);
    }

    /**
     * Updates the designated column with a {@code byte} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateByte(int columnIndex, byte x) throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setIntParameter(columnIndex, x);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a {@code short} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateShort(int columnIndex, short x) throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setIntParameter(columnIndex, x);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with an {@code int} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateInt(int columnIndex, int x) throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setIntParameter(columnIndex, x);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a {@code long} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateLong(int columnIndex, long x) throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setLongParameter(columnIndex, x);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a {@code float} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateFloat(int columnIndex, float x) throws SQLException {
        Double value = Double.valueOf(x);

        setParameter(columnIndex, value);
    }

    /**
     * Updates the designated column with a {@code double} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateDouble(int columnIndex, double x) throws SQLException {
        Double value = Double.valueOf(x);

        setParameter(columnIndex, value);
    }

    /**
     * Updates the designated column with a {@code java.math.BigDecimal}
     * value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateBigDecimal(
            int columnIndex,
            BigDecimal x)
            throws SQLException {
        setParameter(columnIndex, x);
    }

    /**
     * Updates the designated column with a {@code String} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateString(int columnIndex, String x) throws SQLException {
        setParameter(columnIndex, x);
    }

    /**
     * Updates the designated column with a {@code byte} array value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateBytes(int columnIndex, byte[] x) throws SQLException {
        setParameter(columnIndex, x);
    }

    /**
     * Updates the designated column with a {@code java.sql.Date} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateDate(int columnIndex, Date x) throws SQLException {
        setParameter(columnIndex, x);
    }

    /**
     * Updates the designated column with a {@code java.sql.Time} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateTime(int columnIndex, Time x) throws SQLException {
        setParameter(columnIndex, x);
    }

    /**
     * Updates the designated column with a {@code java.sql.Timestamp}
     * value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateTimestamp(
            int columnIndex,
            Timestamp x)
            throws SQLException {
        setParameter(columnIndex, x);
    }

    /**
     * Updates the designated column with an ascii stream value, which will have
     * the specified number of bytes.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateAsciiStream(
            int columnIndex,
            java.io.InputStream x,
            int length)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setAsciiStream(columnIndex, x, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a binary stream value, which will have
     * the specified number of bytes.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateBinaryStream(
            int columnIndex,
            java.io.InputStream x,
            int length)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setBinaryStream(columnIndex, x, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a character stream value, which will have
     * the specified number of (CHECKME: characters?) bytes.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateCharacterStream(
            int columnIndex,
            java.io.Reader x,
            int length)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setCharacterStream(columnIndex, x, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with an {@code Object} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     * <p>
     * If the second argument is an {@code InputStream} then the stream must contain
     * the number of bytes specified by scaleOrLength.  If the second argument is a
     * {@code Reader} then the reader must contain the number of characters specified
     * by scaleOrLength. If these conditions are not true the driver will generate a
     * {@code SQLException} when the statement is executed.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @param scaleOrLength for an object of {@code java.math.BigDecimal} ,
     *          this is the number of digits after the decimal point. For
     *          Java Object types {@code InputStream} and {@code Reader},
     *          this is the length
     *          of the data in the stream or reader.  For all other types,
     *          this value will be ignored.
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateObject(
            int columnIndex,
            Object x,
            int scaleOrLength)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setObject(columnIndex, x, 0, scaleOrLength);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with an {@code Object} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateObject(int columnIndex, Object x) throws SQLException {
        setParameter(columnIndex, x);
    }

    /**
     * Updates the designated column with a {@code null} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateNull(String columnLabel) throws SQLException {
        updateNull(findColumn(columnLabel));
    }

    /**
     * Updates the designated column with a {@code boolean} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateBoolean(
            String columnLabel,
            boolean x)
            throws SQLException {
        updateBoolean(findColumn(columnLabel), x);
    }

    /**
     * Updates the designated column with a {@code byte} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateByte(String columnLabel, byte x) throws SQLException {
        updateByte(findColumn(columnLabel), x);
    }

    /**
     * Updates the designated column with a {@code short} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateShort(String columnLabel, short x) throws SQLException {
        updateShort(findColumn(columnLabel), x);
    }

    /**
     * Updates the designated column with an {@code int} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateInt(String columnLabel, int x) throws SQLException {
        updateInt(findColumn(columnLabel), x);
    }

    /**
     * Updates the designated column with a {@code long} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateLong(String columnLabel, long x) throws SQLException {
        updateLong(findColumn(columnLabel), x);
    }

    /**
     * Updates the designated column with a {@code float } value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateFloat(String columnLabel, float x) throws SQLException {
        updateFloat(findColumn(columnLabel), x);
    }

    /**
     * Updates the designated column with a {@code double} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateDouble(String columnLabel, double x) throws SQLException {
        updateDouble(findColumn(columnLabel), x);
    }

    /**
     * Updates the designated column with a {@code java.sql.BigDecimal}
     * value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateBigDecimal(
            String columnLabel,
            BigDecimal x)
            throws SQLException {
        updateBigDecimal(findColumn(columnLabel), x);
    }

    /**
     * Updates the designated column with a {@code String} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateString(String columnLabel, String x) throws SQLException {
        updateString(findColumn(columnLabel), x);
    }

    /**
     * Updates the designated column with a byte array value.
     *
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow}
     * or {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateBytes(String columnLabel, byte[] x) throws SQLException {
        updateBytes(findColumn(columnLabel), x);
    }

    /**
     * Updates the designated column with a {@code java.sql.Date} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateDate(String columnLabel, Date x) throws SQLException {
        updateDate(findColumn(columnLabel), x);
    }

    /**
     * Updates the designated column with a {@code java.sql.Time} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateTime(String columnLabel, Time x) throws SQLException {
        updateTime(findColumn(columnLabel), x);
    }

    /**
     * Updates the designated column with a {@code java.sql.Timestamp}
     * value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateTimestamp(
            String columnLabel,
            Timestamp x)
            throws SQLException {
        updateTimestamp(findColumn(columnLabel), x);
    }

    /**
     * Updates the designated column with an ascii stream value, which will have
     * the specified number of bytes.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateAsciiStream(
            String columnLabel,
            java.io.InputStream x,
            int length)
            throws SQLException {
        updateAsciiStream(findColumn(columnLabel), x, length);
    }

    /**
     * Updates the designated column with a binary stream value, which will have
     * the specified number of bytes.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateBinaryStream(
            String columnLabel,
            java.io.InputStream x,
            int length)
            throws SQLException {
        updateBinaryStream(findColumn(columnLabel), x, length);
    }

    /**
     * Updates the designated column with a character stream value, which will have
     * the specified number of (CHECKME: characters?) bytes.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param reader the {@code java.io.Reader} object containing
     *        the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateCharacterStream(
            String columnLabel,
            java.io.Reader reader,
            int length)
            throws SQLException {
        updateCharacterStream(findColumn(columnLabel), reader, length);
    }

    /**
     * Updates the designated column with an {@code Object} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     * <p>
     * If the second argument is an {@code InputStream} then the stream must contain
     * the number of bytes specified by scaleOrLength.  If the second argument is a
     * {@code Reader} then the reader must contain the number of characters specified
     * by scaleOrLength. If these conditions are not true the driver will generate a
     * {@code SQLException} when the statement is executed.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @param scaleOrLength for an object of {@code java.math.BigDecimal} ,
     *          this is the number of digits after the decimal point. For
     *          Java Object types {@code InputStream} and {@code Reader},
     *          this is the length
     *          of the data in the stream or reader.  For all other types,
     *          this value will be ignored.
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateObject(
            String columnLabel,
            Object x,
            int scaleOrLength)
            throws SQLException {
        updateObject(findColumn(columnLabel), x, scaleOrLength);
    }

    /**
     * Updates the designated column with an {@code Object} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateObject(String columnLabel, Object x) throws SQLException {
        updateObject(findColumn(columnLabel), x);
    }

    /**
     * Inserts the contents of the insert row into this
     * {@code ResultSet} object and into the database.
     * The cursor must be on the insert row when this method is called.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @throws SQLException if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY},
     * this method is called on a closed result set,
     * if this method is called when the cursor is not on the insert row,
     * or if not all of non-nullable columns in
     * the insert row have been given a non-null value
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void insertRow() throws SQLException {
        performInsert();
    }

    /**
     * Updates the underlying database with the new contents of the
     * current row of this {@code ResultSet} object.
     * This method cannot be called when the cursor is on the insert row.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature. <p>
     *
     * After updating any values in the current row, it is not possible to
     * move the cursor position without calling this method, or alternatively
     * calling cancelRowUpdates() to abandon the row update. <p>
     * After calling this method, the {@code getXXX} methods return the
     * updated values for the updated row.
     * </p>
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @throws SQLException if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY};
     *  this method is called on a closed result set or
     * if this method is called when the cursor is on the insert row
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void updateRow() throws SQLException {
        performUpdate();
    }

    /**
     * Deletes the current row from this {@code ResultSet} object
     * and from the underlying database.  This method cannot be called when
     * the cursor is on the insert row.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature. <p>
     *
     * After a successful call to this method, the row is deleted.
     * All column values returned by {@code getXXX} methods are null for a
     * row that has been deleted.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @throws SQLException if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY};
     * this method is called on a closed result set
     * or if this method is called when the cursor is on the insert row
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void deleteRow() throws SQLException {
        performDelete();
    }

    /**
     * Refreshes the current row with its most recent value in
     * the database.  This method cannot be called when
     * the cursor is on the insert row.
     *
     * <P>The {@code refreshRow} method provides a way for an
     * application to
     * explicitly tell the JDBC driver to refetch a row(s) from the
     * database.  An application may want to call {@code refreshRow} when
     * caching or prefetching is being done by the JDBC driver to
     * fetch the latest value of a row from the database.  The JDBC driver
     * may actually refresh multiple rows at once if the fetch size is
     * greater than one.
     *
     * <P> All values are refetched subject to the transaction isolation
     * level and cursor sensitivity.  If {@code refreshRow} is called after
     * calling an updater method, but before calling
     * the method {@code updateRow}, then the
     * updates made to the row are lost.  Calling the method
     * {@code refreshRow} frequently will likely slow performance.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @throws SQLException if a database access error
     * occurs; this method is called on a closed result set;
     * the result set type is {@code TYPE_FORWARD_ONLY} or if this
     * method is called when the cursor is on the insert row
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method or this method is not supported for the specified result
     * set type and result set concurrency.
     * @since JDK 1.2
     */
    public void refreshRow() throws SQLException {
        clearUpdates();
    }

    /**
     * Cancels the updates made to the current row in this
     * {@code ResultSet} object.
     * This method may be called after calling an
     * updater method(s) and before calling
     * the method {@code updateRow} to roll back
     * the updates made to a row.  If no updates have been made or
     * {@code updateRow} has already been called, this method has no
     * effect.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @throws SQLException if a database access error
     *            occurs; this method is called on a closed result set;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or if this method is called when the cursor is
     *            on the insert row
     *  @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void cancelRowUpdates() throws SQLException {
        clearUpdates();
    }

    /**
     * Moves the cursor to the insert row.  The current cursor position is
     * remembered while the cursor is positioned on the insert row.
     *
     * The insert row is a special row associated with an updatable
     * result set.  It is essentially a buffer where a new row may
     * be constructed by calling the updater methods prior to
     * inserting the row into the result set.
     *
     * Only the updater, getter,
     * and {@code insertRow} methods may be
     * called when the cursor is on the insert row. All of the columns in
     * a result set must be given a value each time this method is
     * called before calling {@code insertRow}.
     * An updater method must be called before a
     * getter method can be called on a column value.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @throws SQLException if a database access error occurs; this
     * method is called on a closed result set
     * or the result set concurrency is {@code CONCUR_READ_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void moveToInsertRow() throws SQLException {
        startInsert();
    }

    /**
     * Moves the cursor to the remembered cursor position, usually the
     * current row.  This method has no effect if the cursor is not on
     * the insert row.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature.
     * </div>
     * <!-- end release-specific documentation -->
     * @throws SQLException if a database access error occurs; this
     * method is called on a closed result set
     *  or the result set concurrency is {@code CONCUR_READ_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public void moveToCurrentRow() throws SQLException {
        endInsert();
    }

    /**
     * Retrieves the {@code Statement} object that produced this
     * {@code ResultSet} object.
     * If the result set was generated some other way, such as by a
     * {@code DatabaseMetaData} method, this method  may return
     * {@code null}.
     *
     * @return the {@code Statement} object that produced
     * this {@code ResultSet} object or {@code null}
     * if the result set was produced some other way
     * @throws SQLException if a database access error occurs
     * or this method is called on a closed result set
     * @since JDK 1.2
     */
    public Statement getStatement() throws SQLException {
        checkClosed();

        return (Statement) statement;
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as an {@code Object}
     * in the Java programming language.
     * If the value is an SQL {@code NULL},
     * the driver returns a Java {@code null}.
     * This method uses the given {@code Map} object
     * for the custom mapping of the
     * SQL structured or distinct type that is being retrieved.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature, but ignores the map.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param map a {@code java.util.Map} object that contains the mapping
     * from SQL type names to classes in the Java programming language
     * @return an {@code Object} in the Java programming language
     * representing the SQL value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public Object getObject(int columnIndex, Map map) throws SQLException {
        return getObject(columnIndex);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code Ref} object
     * in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not support this feature; this method always
     * throws an {@code SQLException} stating that the operation is not
     * supported.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a {@code Ref} object representing an SQL {@code REF}
     *         value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public Ref getRef(int columnIndex) throws SQLException {
        throw JDBCUtil.notSupported();
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code Blob} object
     * in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB 2.0 supports this feature for objects of type BLOB and BINARY.
     * The Blob returned for BINARY objects is a memory object. The Blob
     * return for BLOB objects is not held entirely in memory. Its contents are
     * fetched from the database when its getXXX() methods are called.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a {@code Blob} object representing the SQL
     *         {@code BLOB} value in the specified column
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public Blob getBlob(int columnIndex) throws SQLException {

        checkColumn(columnIndex);

        Type   sourceType = resultMetaData.columnTypes[columnIndex - 1];
        Object o          = getColumnInType(columnIndex, sourceType);

        if (o == null) {
            return null;
        }

        if (o instanceof BlobDataID) {
            if (memoryLobs) {
                long length = ((BlobDataID) o).length(session);

                if (length > Integer.MAX_VALUE) {
                    throw JDBCUtil.sqlException(ErrorCode.X_22001);
                }

                byte[] bytes = ((BlobDataID) o).getBytes(
                    session,
                    0,
                    (int) length);

                return new JDBCBlob(bytes);
            }

            JDBCBlobClient blob = new JDBCBlobClient(session, (BlobDataID) o);

            if (isUpdatable) {
                if (resultMetaData.colIndexes[columnIndex - 1] > 0
                        && resultMetaData.columns[columnIndex - 1].isWriteable()) {
                    blob.setWritable(this, columnIndex - 1);
                }
            }

            return blob;
        } else if (o instanceof Blob) {
            return (Blob) o;
        } else if (o instanceof BinaryData) {
            byte[] b = getBytes(columnIndex);

            return new JDBCBlob(b);
        }

        throw JDBCUtil.sqlException(ErrorCode.X_42561);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code Clob} object
     * in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB 2.0 supports this feature for objects of type CLOB and
     * the variations of CHAR.
     * The Clob returned for CHAR objects is a memory object. The Clob
     * return for CLOB objects is not held entirely in memory. Its contents are
     * fetched from the database when its getXXX() methods are called.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a {@code Clob} object representing the SQL
     *         {@code CLOB} value in the specified column
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public Clob getClob(int columnIndex) throws SQLException {

        checkColumn(columnIndex);

        Type   sourceType = resultMetaData.columnTypes[columnIndex - 1];
        Object o          = getColumnInType(columnIndex, sourceType);

        if (o == null) {
            return null;
        }

        if (o instanceof ClobDataID) {
            if (memoryLobs) {
                long length = ((ClobDataID) o).length(session);

                if (length > Integer.MAX_VALUE) {
                    throw JDBCUtil.sqlException(ErrorCode.X_22001);
                }

                String s = ((ClobDataID) o).getSubString(
                    session,
                    0,
                    (int) length);

                return new JDBCClob(s);
            }

            JDBCClobClient clob = new JDBCClobClient(session, (ClobDataID) o);

            if (isUpdatable) {
                if (resultMetaData.colIndexes[columnIndex - 1] > 0
                        && resultMetaData.columns[columnIndex - 1].isWriteable()) {
                    clob.setWritable(this, columnIndex - 1);
                }
            }

            return clob;
        } else if (o instanceof Clob) {
            return (Clob) o;
        } else if (o instanceof String) {
            return new JDBCClob((String) o);
        }

        throw JDBCUtil.sqlException(ErrorCode.X_42561);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as an {@code Array} object
     * in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * From version 2.0, HSQLDB supports array types.
     *
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return an {@code Array} object representing the SQL
     *         {@code ARRAY} value in the specified column
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public Array getArray(int columnIndex) throws SQLException {

        checkColumn(columnIndex);

        Type     type = resultMetaData.columnTypes[columnIndex - 1];
        Object[] data = (Object[]) getCurrent()[columnIndex - 1];

        if (!type.isArrayType()) {
            throw JDBCUtil.sqlException(ErrorCode.X_42561);
        }

        if (trackNull(data)) {
            return null;
        }

        return new JDBCArray(data, type.collectionBaseType(), type, connection);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as an {@code Object}
     * in the Java programming language.
     * If the value is an SQL {@code NULL},
     * the driver returns a Java {@code null}.
     * This method uses the specified {@code Map} object for
     * custom mapping if appropriate.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB supports this feature. But the Map parameter is ignored.
     *
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param map a {@code java.util.Map} object that contains the mapping
     * from SQL type names to classes in the Java programming language
     * @return an {@code Object} representing the SQL value in the
     *         specified column
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public Object getObject(String columnLabel, Map map) throws SQLException {
        return getObject(findColumn(columnLabel), map);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code Ref} object
     * in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * Including 2.0, HSQLDB does not support reference types; this method
     * always throws an {@code SQLException} stating that the operation
     * is not supported.
     * </div>
     * <!-- end release-specific documentation -->
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return a {@code Ref} object representing the SQL {@code REF}
     *         value in the specified column
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public Ref getRef(String columnLabel) throws SQLException {
        return getRef(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code Blob} object
     * in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB 2.0 supports this feature for objects of type BLOB and BINARY.
     * The Blob returned for BINARY objects is a memory object. The Blob
     * return for BLOB objects is not held entirely in memory. Its contents are
     * fetched from the database when its getXXX() methods are called.
     *
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return a {@code Blob} object representing the SQL {@code BLOB}
     *         value in the specified column
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public Blob getBlob(String columnLabel) throws SQLException {
        return getBlob(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code Clob} object
     * in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB 2.0 supports this feature for objects of type CLOB and
     * the variations of CHAR.
     * The Clob returned for CHAR objects is a memory object. The Clob
     * return for CLOB objects is not held entirely in memory. Its contents are
     * fetched from the database when its getXXX() methods are called.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return a {@code Clob} object representing the SQL {@code CLOB}
     * value in the specified column
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public Clob getClob(String columnLabel) throws SQLException {
        return getClob(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as an {@code Array} object
     * in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * From version 2.0, HSQLDB supports array types.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @return an {@code Array} object representing the SQL {@code ARRAY} value in
     *         the specified column
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.2
     */
    public Array getArray(String columnLabel) throws SQLException {
        return getArray(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code java.sql.Date} object
     * in the Java programming language.
     * This method uses the given calendar to construct an appropriate millisecond
     * value for the date if the underlying database does not store
     * timezone information.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param cal the {@code java.util.Calendar} object
     * to use in constructing the date
     * @return the column value as a {@code java.sql.Date} object;
     * if the value is SQL {@code NULL},
     * the value returned is {@code null} in the Java programming language
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @since JDK 1.2
     */
    public Date getDate(int columnIndex, Calendar cal) throws SQLException {

        TimestampData t = (TimestampData) getColumnInType(
            columnIndex,
            Type.SQL_DATE);

        if (t == null) {
            return null;
        }

        return (Date) Type.SQL_DATE.convertSQLToJava(session, t, cal);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code java.sql.Date} object
     * in the Java programming language.
     * This method uses the given calendar to construct an appropriate millisecond
     * value for the date if the underlying database does not store
     * timezone information.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param cal the {@code java.util.Calendar} object
     * to use in constructing the date
     * @return the column value as a {@code java.sql.Date} object;
     * if the value is SQL {@code NULL},
     * the value returned is {@code null} in the Java programming language
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @since JDK 1.2
     */
    public Date getDate(String columnLabel, Calendar cal) throws SQLException {
        return getDate(findColumn(columnLabel), cal);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code java.sql.Time} object
     * in the Java programming language.
     * This method uses the given calendar to construct an appropriate millisecond
     * value for the time if the underlying database does not store
     * timezone information.
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * The JDBC specification for this method is vague. HSQLDB interprets the
     * specification as follows:
     *
     * <ol>
     * <li>If the SQL type of the column is WITH TIME ZONE, then the UTC value
     * of the returned java.sql.Time object is the UTC of the SQL value without
     * modification. In other words, the Calendar object is not used.</li>
     * <li>If the SQL type of the column is WITHOUT TIME ZONE, then the UTC
     * value of the returned java.sql.Time is correct for the given Calendar's
     * time zone.</li>
     * <li>If the cal argument is null, it it ignored and the method returns
     * the same Object as the method without the Calendar parameter.</li>
     * </ol>
     * </div>
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param cal the {@code java.util.Calendar} object
     * to use in constructing the time
     * @return the column value as a {@code java.sql.Time} object;
     * if the value is SQL {@code NULL},
     * the value returned is {@code null} in the Java programming language
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @since JDK 1.2
     */
    public Time getTime(int columnIndex, Calendar cal) throws SQLException {

        TimeData t = (TimeData) getColumnInType(columnIndex, Type.SQL_TIME);

        if (t == null) {
            return null;
        }

        return (Time) Type.SQL_TIME.convertSQLToJava(session, t, cal);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code java.sql.Time} object
     * in the Java programming language.
     * This method uses the given calendar to construct an appropriate millisecond
     * value for the time if the underlying database does not store
     * timezone information.
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * The JDBC specification for this method is vague. HSQLDB interprets the
     * specification as follows:
     *
     * <ol>
     * <li>If the SQL type of the column is WITH TIME ZONE, then the UTC value
     * of the returned java.sql.Time object is the UTC of the SQL value without
     * modification. In other words, the Calendar object is not used.</li>
     * <li>If the SQL type of the column is WITHOUT TIME ZONE, then the
     * UTC value of the returned java.sql.Time will represent the correct
     * time for the time zone (including daylight saving time) of the given
     * Calendar. </li>
     * <li>If the cal argument is null, it it ignored and the method returns
     * the same Object as the method without the Calendar parameter.</li>
     * </ol>
     * </div>
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param cal the {@code java.util.Calendar} object
     * to use in constructing the time
     * @return the column value as a {@code java.sql.Time} object;
     * if the value is SQL {@code NULL},
     * the value returned is {@code null} in the Java programming language
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @since JDK 1.2
     */
    public Time getTime(String columnLabel, Calendar cal) throws SQLException {
        return getTime(findColumn(columnLabel), cal);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code java.sql.Timestamp} object
     * in the Java programming language.
     * This method uses the given calendar to construct an appropriate millisecond
     * value for the timestamp if the underlying database does not store
     * timezone information.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * The JDBC specification for this method is vague. HSQLDB interprets the
     * specification as follows:
     *
     * <ol>
     * <li>If the SQL type of the column is WITH TIME ZONE, then the UTC value
     * of the returned java.sql.Timestamp object is the UTC of the SQL value
     * without modification. In other words, the Calendar object is not used.
     * </li>
     * <li>If the SQL type of the column is WITHOUT TIME ZONE, then the
     * UTC value of the returned java.sql.Timestamp will represent the correct
     * timestamp for the time zone (including daylight saving time) of the given
     * Calendar object. </li>
     * <li>In this case, if the cal argument is null, then the default Calendar
     * of the JVM is used, which results in the same Object as one returned by the
     * getTimestamp() methods without the Calendar parameter.</li>
     * </ol>
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param cal the {@code java.util.Calendar} object
     * to use in constructing the timestamp
     * @return the column value as a {@code java.sql.Timestamp} object;
     * if the value is SQL {@code NULL},
     * the value returned is {@code null} in the Java programming language
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @since JDK 1.2
     */
    public Timestamp getTimestamp(
            int columnIndex,
            Calendar cal)
            throws SQLException {

        TimestampData tsd;
        Object        value = getColumnValue(columnIndex);

        if (value == null) {
            return null;
        }

        if (resultMetaData.columnTypes[columnIndex - 1].typeCode
                == Types.SQL_TIMESTAMP_WITH_TIME_ZONE) {
            tsd = (TimestampData) value;

            Timestamp ts = new Timestamp(tsd.getMillis());

            ts.setNanos(tsd.getNanos());

            return ts;
        }

        tsd = (TimestampData) getColumnInType(columnIndex, Type.SQL_TIMESTAMP);

        return (Timestamp) Type.SQL_TIMESTAMP.convertSQLToJava(
            session,
            tsd,
            cal);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code java.sql.Timestamp} object
     * in the Java programming language.
     * This method uses the given calendar to construct an appropriate millisecond
     * value for the timestamp if the underlying database does not store
     * timezone information.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * The JDBC specification for this method is vague. HSQLDB interprets the
     * specification as follows:
     *
     * <ol>
     * <li>If the SQL type of the column is WITH TIME ZONE, then the UTC value
     * of the returned java.sql.Timestamp object is the UTC of the SQL value
     * without modification. In other words, the Calendar object is not used.
     * </li>
     * <li>If the SQL type of the column is WITHOUT TIME ZONE, then the UTC
     * value of the returned java.sql.Timestamp is correct for the given
     * Calendar object.</li>
     * <li>If the cal argument is null, it it ignored and the method returns
     * the same Object as the method without the Calendar parameter.</li>
     * </ol>
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param cal the {@code java.util.Calendar} object
     * to use in constructing the date
     * @return the column value as a {@code java.sql.Timestamp} object;
     * if the value is SQL {@code NULL},
     * the value returned is {@code null} in the Java programming language
     * @throws SQLException if the columnLabel is not valid or
     * if a database access error occurs
     * or this method is called on a closed result set
     * @since JDK 1.2
     */
    public Timestamp getTimestamp(
            String columnLabel,
            Calendar cal)
            throws SQLException {
        return getTimestamp(findColumn(columnLabel), cal);
    }

    //-------------------------- JDBC 3.0 ----------------------------------------

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code java.net.URL}
     * object in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not support the datalink type; this method
     * always throws an {@code SQLException} stating that the operation is
     * not supported.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the index of the column 1 is the first, 2 is the second,...
     * @return the column value as a {@code java.net.URL} object;
     * if the value is SQL {@code NULL},
     * the value returned is {@code null} in the Java programming language
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs; this method
     * is called on a closed result set or if a URL is malformed
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.4, HSQLDB 1.7.0
     */
    public java.net.URL getURL(int columnIndex) throws SQLException {
        throw JDBCUtil.notSupported();
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code java.net.URL}
     * object in the Java programming language.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not support the datalink type; this method
     * always throws an {@code SQLException} stating that the operation is
     * not supported.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @return the column value as a {@code java.net.URL} object;
     * if the value is SQL {@code NULL},
     * the value returned is {@code null} in the Java programming language
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs; this method
     * is called on a closed result set or if a URL is malformed
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.4, HSQLDB 1.7.0
     */
    public java.net.URL getURL(String columnLabel) throws SQLException {
        throw JDBCUtil.notSupported();
    }

    /**
     * Updates the designated column with a {@code java.sql.Ref} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not support reference types; this method
     * always throws an {@code SQLException} stating that the operation is
     * not supported.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.4, HSQLDB 1.7.0
     */
    public void updateRef(int columnIndex, java.sql.Ref x) throws SQLException {
        throw JDBCUtil.notSupported();
    }

    /**
     * Updates the designated column with a {@code java.sql.Ref} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not support reference types; this method
     * always throws an {@code SQLException} stating that the operation is
     * not supported.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.4, HSQLDB 1.7.0
     */
    public void updateRef(
            String columnLabel,
            java.sql.Ref x)
            throws SQLException {
        throw JDBCUtil.notSupported();
    }

    /**
     * Updates the designated column with a {@code java.sql.Blob} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB 2.0 supports this feature for columns of type BLOB.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.4, HSQLDB 1.7.0
     */
    public void updateBlob(
            int columnIndex,
            java.sql.Blob x)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setBlobParameter(columnIndex, x);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a {@code java.sql.Blob} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB 2.0 supports this feature for columns of type BLOB.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.4, HSQLDB 1.7.0
     */
    public void updateBlob(
            String columnLabel,
            java.sql.Blob x)
            throws SQLException {
        int columnIndex = findColumn(columnLabel);

        updateBlob(columnIndex, x);
    }

    /**
     * Updates the designated column with a {@code java.sql.Clob} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB 2.0 supports this feature for columns of type CLOB.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.4, HSQLDB 1.7.0
     */
    public void updateClob(
            int columnIndex,
            java.sql.Clob x)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setClobParameter(columnIndex, x);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a {@code java.sql.Clob} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB 2.0 supports this feature for columns of type CLOB.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.4, HSQLDB 1.7.0
     */
    public void updateClob(
            String columnLabel,
            java.sql.Clob x)
            throws SQLException {
        int columnIndex = findColumn(columnLabel);

        updateClob(columnIndex, x);
    }

    /**
     * Updates the designated column with a {@code java.sql.Array} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not support array types; this method always
     * throws an {@code SQLException} stating that the operation is not
     * supported.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.4, HSQLDB 1.7.0
     */
    public void updateArray(
            int columnIndex,
            java.sql.Array x)
            throws SQLException {
        setParameter(columnIndex, x);
    }

    /**
     * Updates the designated column with a {@code java.sql.Array} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not support array types; this method always
     * throws an {@code SQLException} stating that the operation is not
     * supported.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.4, HSQLDB 1.7.0
     */
    public void updateArray(
            String columnLabel,
            java.sql.Array x)
            throws SQLException {
        int columnIndex = findColumn(columnLabel);

        updateArray(columnIndex, x);
    }

    //------------------------- JDBC 4.0 -----------------------------------

    /**
     * Retrieves the value of the designated column in the current row of this
     * {@code ResultSet} object as a {@code java.sql.RowId} object in the Java
     * programming language.
     *
     * @param columnIndex the first column is 1, the second 2, ...
     * @return the column value; if the value is a SQL {@code NULL} the
     *     value returned is {@code null}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public RowId getRowId(int columnIndex) throws SQLException {
        throw JDBCUtil.notSupported();
    }

    /**
     * Retrieves the value of the designated column in the current row of this
     * {@code ResultSet} object as a {@code java.sql.RowId} object in the Java
     * programming language.
     *
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not support the RowId type; this method always
     * throws an {@code SQLException} stating that the operation is not
     * supported.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @return the column value ; if the value is a SQL {@code NULL} the
     *     value returned is {@code null}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public RowId getRowId(String columnLabel) throws SQLException {
        throw JDBCUtil.notSupported();
    }

    /**
     * Updates the designated column with a {@code RowId} value. The updater
     * methods are used to update column values in the current row or the insert
     * row. The updater methods do not update the underlying database; instead
     * the {@code updateRow} or {@code insertRow} methods are called
     * to update the database.
     *
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not support the RowId type; this method always
     * throws an {@code SQLException} stating that the operation is not
     * supported.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnIndex the first column is 1, the second 2, ...
     * @param x the column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateRowId(int columnIndex, RowId x) throws SQLException {
        throw JDBCUtil.notSupported();
    }

    /**
     * Updates the designated column with a {@code RowId} value. The updater
     * methods are used to update column values in the current row or the insert
     * row. The updater methods do not update the underlying database; instead
     * the {@code updateRow} or {@code insertRow} methods are called
     * to update the database.
     *
     *
     * <!-- start release-specific documentation -->
     * <div class="ReleaseSpecificDocumentation">
     * <p class="rshead">HSQLDB-Specific Information:</p>
     *
     * HSQLDB does not support the RowId type; this method always
     * throws an {@code SQLException} stating that the operation is not
     * supported.
     * </div>
     * <!-- end release-specific documentation -->
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param x the column value
     * @throws SQLException if a database access error occurs,
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateRowId(String columnLabel, RowId x) throws SQLException {
        throw JDBCUtil.notSupported();
    }

    /**
     * Retrieves the holdability of this {@code ResultSet} object
     * @return  either {@code ResultSet.HOLD_CURSORS_OVER_COMMIT} or {@code ResultSet.CLOSE_CURSORS_AT_COMMIT}
     * @throws SQLException if a database access error occurs
     * or this method is called on a closed result set
     * @since JDK 1.6, HSQLDB 2.0
     */
    public int getHoldability() throws SQLException {
        checkClosed();

        return ResultProperties.getJDBCHoldability(rsProperties);
    }

    /**
     * Retrieves whether this {@code ResultSet} object has been closed.
     * A {@code ResultSet} is closed if the
     * method close has been called on it, or if it is automatically closed.
     *
     * @return true if this {@code ResultSet} object is closed; false if it is still open
     * @throws SQLException if a database access error occurs
     * @since JDK 1.6, HSQLDB 2.0
     */
    public boolean isClosed() throws SQLException {
        return navigator == null;
    }

    /**
     * Updates the designated column with a {@code String} value.
     * It is intended for use when updating {@code NCHAR},{@code NVARCHAR}
     * and {@code LONGNVARCHAR} columns.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnIndex the first column is 1, the second 2, ...
     * @param nString the value for the column to be updated
     * @throws SQLException if the columnIndex is not valid;
     * if the driver does not support national
     *         character sets;  if the driver can detect that a data conversion
     *  error could occur; this method is called on a closed result set;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or if a database access error occurs
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateNString(
            int columnIndex,
            String nString)
            throws SQLException {
        updateString(columnIndex, nString);
    }

    /**
     * Updates the designated column with a {@code String} value.
     * It is intended for use when updating {@code NCHAR},{@code NVARCHAR}
     * and {@code LONGNVARCHAR} columns.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param nString the value for the column to be updated
     * @throws SQLException if the columnLabel is not valid;
     * if the driver does not support national
     *         character sets;  if the driver can detect that a data conversion
     *  error could occur; this method is called on a closed result set;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     *  or if a database access error occurs
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateNString(
            String columnLabel,
            String nString)
            throws SQLException {
        updateString(columnLabel, nString);
    }

    /**
     * Updates the designated column with a {@code java.sql.NClob} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnIndex the first column is 1, the second 2, ...
     * @param nClob the value for the column to be updated
     * @throws SQLException if the columnIndex is not valid;
     * if the driver does not support national
     *         character sets;  if the driver can detect that a data conversion
     *  error could occur; this method is called on a closed result set;
     * if a database access error occurs or
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateNClob(int columnIndex, NClob nClob) throws SQLException {
        updateClob(columnIndex, nClob);
    }

    /**
     * Updates the designated column with a {@code java.sql.NClob} value.
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param nClob the value for the column to be updated
     * @throws SQLException if the columnLabel is not valid;
     * if the driver does not support national
     *         character sets;  if the driver can detect that a data conversion
     *  error could occur; this method is called on a closed result set;
     *  if a database access error occurs or
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateNClob(
            String columnLabel,
            NClob nClob)
            throws SQLException {
        updateClob(columnLabel, nClob);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code NClob} object
     * in the Java programming language.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a {@code NClob} object representing the SQL
     *         {@code NCLOB} value in the specified column
     * @throws SQLException if the columnIndex is not valid;
     * if the driver does not support national
     *         character sets;  if the driver can detect that a data conversion
     *  error could occur; this method is called on a closed result set
     * or if a database access error occurs
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public NClob getNClob(int columnIndex) throws SQLException {

        String s = getString(columnIndex);

        return s == null
               ? null
               : new JDBCNClob(s);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a {@code NClob} object
     * in the Java programming language.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @return a {@code NClob} object representing the SQL {@code NCLOB}
     * value in the specified column
     * @throws SQLException if the columnLabel is not valid;
   * if the driver does not support national
     *         character sets;  if the driver can detect that a data conversion
     *  error could occur; this method is called on a closed result set
     * or if a database access error occurs
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public NClob getNClob(String columnLabel) throws SQLException {
        return getNClob(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in  the current row of
     *  this {@code ResultSet} as a
     * {@code java.sql.SQLXML} object in the Java programming language.
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a {@code SQLXML} object that maps an {@code SQL XML} value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public SQLXML getSQLXML(int columnIndex) throws SQLException {

        checkColumn(columnIndex);

        SQLXML sqlxml;
        int    type = resultMetaData.columnTypes[columnIndex - 1].typeCode;

        switch (type) {

            case Types.SQL_XML : {
                Object object = getObject(columnIndex);

                if (object == null) {
                    sqlxml = null;
                } else if (object instanceof SQLXML) {
                    sqlxml = (SQLXML) object;
                } else {
                    throw JDBCUtil.notSupported();
                }

                break;
            }

            case Types.SQL_CLOB : {
                Clob clob = getClob(columnIndex);

                if (clob == null) {
                    sqlxml = null;
                } else {
                    sqlxml = new JDBCSQLXML(clob.getCharacterStream());
                }

                break;
            }

            case Types.SQL_CHAR :
            case Types.SQL_VARCHAR : {
                java.io.Reader reader = getCharacterStream(columnIndex);

                if (reader == null) {
                    sqlxml = null;
                } else {
                    sqlxml = new JDBCSQLXML(reader);
                }

                break;
            }

            case Types.SQL_NCHAR :
            case Types.SQL_NVARCHAR : {
                java.io.Reader nreader = getNCharacterStream(columnIndex);

                if (nreader == null) {
                    sqlxml = null;
                } else {
                    sqlxml = new JDBCSQLXML(nreader);
                }

                break;
            }

            case Types.SQL_BLOB : {
                Blob blob = getBlob(columnIndex);

                if (blob == null) {
                    sqlxml = null;
                } else {
                    sqlxml = new JDBCSQLXML(blob.getBinaryStream());
                }

                break;
            }

            case Types.SQL_BINARY :
            case Types.SQL_VARBINARY : {
                java.io.InputStream inputStream = getBinaryStream(columnIndex);

                if (inputStream == null) {
                    sqlxml = null;
                } else {
                    sqlxml = new JDBCSQLXML(inputStream);
                }

                break;
            }

            case Types.OTHER :
            case Types.JAVA_OBJECT : {
                Object data = getObject(columnIndex);

                if (data == null) {
                    sqlxml = null;
                } else if (data instanceof SQLXML) {
                    sqlxml = (SQLXML) data;
                } else if (data instanceof String) {
                    sqlxml = new JDBCSQLXML((String) data);
                } else if (data instanceof byte[]) {
                    sqlxml = new JDBCSQLXML((byte[]) data);
                } else if (data instanceof Blob) {
                    Blob blob = (Blob) data;

                    sqlxml = new JDBCSQLXML(blob.getBinaryStream());
                } else if (data instanceof Clob) {
                    Clob clob = (Clob) data;

                    sqlxml = new JDBCSQLXML(clob.getCharacterStream());
                } else {
                    throw JDBCUtil.notSupported();
                }

                break;
            }

            default : {
                throw JDBCUtil.notSupported();
            }
        }

        return sqlxml;
    }

    /**
     * Retrieves the value of the designated column in  the current row of
     *  this {@code ResultSet} as a
     * {@code java.sql.SQLXML} object in the Java programming language.
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @return a {@code SQLXML} object that maps an {@code SQL XML} value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public SQLXML getSQLXML(String columnLabel) throws SQLException {
        return getSQLXML(findColumn(columnLabel));
    }

    /**
     * Updates the designated column with a {@code java.sql.SQLXML} value.
     * The updater
     * methods are used to update column values in the current row or the insert
     * row. The updater methods do not update the underlying database; instead
     * the {@code updateRow} or {@code insertRow} methods are called
     * to update the database.
     *
     * @param columnIndex the first column is 1, the second 2, ...
     * @param xmlObject the value for the column to be updated
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs; this method
     *  is called on a closed result set;
     * the {@code java.xml.transform.Result},
     *  {@code Writer} or {@code OutputStream} has not been closed
     * for the {@code SQLXML} object;
     *  if there is an error processing the XML value or
     * the result set concurrency is {@code CONCUR_READ_ONLY}.  The {@code getCause} method
     *  of the exception may provide a more detailed exception, for example, if the
     *  stream does not contain valid XML.
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateSQLXML(
            int columnIndex,
            SQLXML xmlObject)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setSQLXML(columnIndex, xmlObject);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a {@code java.sql.SQLXML} value.
     * The updater
     * methods are used to update column values in the current row or the insert
     * row. The updater methods do not update the underlying database; instead
     * the {@code updateRow} or {@code insertRow} methods are called
     * to update the database.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param xmlObject the column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs; this method
     *  is called on a closed result set;
     * the {@code java.xml.transform.Result},
     *  {@code Writer} or {@code OutputStream} has not been closed
     * for the {@code SQLXML} object;
     *  if there is an error processing the XML value or
     * the result set concurrency is {@code CONCUR_READ_ONLY}.  The {@code getCause} method
     *  of the exception may provide a more detailed exception, for example, if the
     *  stream does not contain valid XML.
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateSQLXML(
            String columnLabel,
            SQLXML xmlObject)
            throws SQLException {
        updateSQLXML(findColumn(columnLabel), xmlObject);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code String} in the Java programming language.
     * It is intended for use when
     * accessing  {@code NCHAR},{@code NVARCHAR}
     * and {@code LONGNVARCHAR} columns.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public String getNString(int columnIndex) throws SQLException {
        return getString(columnIndex);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as
     * a {@code String} in the Java programming language.
     * It is intended for use when
     * accessing  {@code NCHAR},{@code NVARCHAR}
     * and {@code LONGNVARCHAR} columns.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @return the column value; if the value is SQL {@code NULL}, the
     * value returned is {@code null}
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public String getNString(String columnLabel) throws SQLException {
        return getString(findColumn(columnLabel));
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a
     * {@code java.io.Reader} object.
     * It is intended for use when
     * accessing  {@code NCHAR},{@code NVARCHAR}
     * and {@code LONGNVARCHAR} columns.
     *
     * @return a {@code java.io.Reader} object that contains the column
     * value; if the value is SQL {@code NULL}, the value returned is
     * {@code null} in the Java programming language.
     * @param columnIndex the first column is 1, the second is 2, ...
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public java.io.Reader getNCharacterStream(
            int columnIndex)
            throws SQLException {
        return getCharacterStream(columnIndex);
    }

    /**
     * Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object as a
     * {@code java.io.Reader} object.
     * It is intended for use when
     * accessing  {@code NCHAR},{@code NVARCHAR}
     * and {@code LONGNVARCHAR} columns.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @return a {@code java.io.Reader} object that contains the column
     * value; if the value is SQL {@code NULL}, the value returned is
     * {@code null} in the Java programming language
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public java.io.Reader getNCharacterStream(
            String columnLabel)
            throws SQLException {
        return getCharacterStream(findColumn(columnLabel));
    }

    /**
     * Updates the designated column with a character stream value, which will have
     * the specified number of bytes.   The
     * driver does the necessary conversion from Java character format to
     * the national character set in the database.
     * It is intended for use when
     * updating  {@code NCHAR},{@code NVARCHAR}
     * and {@code LONGNVARCHAR} columns.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY} or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateNCharacterStream(
            int columnIndex,
            java.io.Reader x,
            long length)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setCharacterStream(columnIndex, x, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a character stream value, which will have
     * the specified number of bytes.  The
     * driver does the necessary conversion from Java character format to
     * the national character set in the database.
     * It is intended for use when
     * updating  {@code NCHAR},{@code NVARCHAR}
     * and {@code LONGNVARCHAR} columns.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param reader the {@code java.io.Reader} object containing
     *        the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY} or this method is called on a closed result set
     *  @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateNCharacterStream(
            String columnLabel,
            java.io.Reader reader,
            long length)
            throws SQLException {
        updateCharacterStream(columnLabel, reader, length);
    }

// --------------------------- Added: JAVA 1.6 (Mustang) Build 86 -------------------------

    /**
     * Updates the designated column with an ascii stream value, which will have
     * the specified number of bytes.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateAsciiStream(
            int columnIndex,
            java.io.InputStream x,
            long length)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setAsciiStream(columnIndex, x, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a binary stream value, which will have
     * the specified number of bytes.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateBinaryStream(
            int columnIndex,
            java.io.InputStream x,
            long length)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setBinaryStream(columnIndex, x, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a character stream value, which will have
     * the specified number of bytes.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateCharacterStream(
            int columnIndex,
            java.io.Reader x,
            long length)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setCharacterStream(columnIndex, x, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with an ascii stream value, which will have
     * the specified number of bytes.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param x the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateAsciiStream(
            String columnLabel,
            java.io.InputStream x,
            long length)
            throws SQLException {

        int columnIndex = findColumn(columnLabel);

        startUpdate(columnIndex);
        preparedStatement.setAsciiStream(columnIndex, x, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a binary stream value, which will have
     * the specified number of bytes.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param x the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateBinaryStream(
            String columnLabel,
            java.io.InputStream x,
            long length)
            throws SQLException {

        int columnIndex = findColumn(columnLabel);

        startUpdate(columnIndex);
        preparedStatement.setBinaryStream(columnIndex, x, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a character stream value, which will have
     * the specified number of bytes.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param reader the {@code java.io.Reader} object containing
     *        the new column value
     * @param length the length of the stream
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateCharacterStream(
            String columnLabel,
            java.io.Reader reader,
            long length)
            throws SQLException {

        int columnIndex = findColumn(columnLabel);

        startUpdate(columnIndex);
        preparedStatement.setCharacterStream(columnIndex, reader, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column using the given input stream, which
     * will have the specified number of bytes.
     *
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param inputStream An object that contains the data to set the parameter
     * value to.
     * @param length the number of bytes in the parameter data.
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateBlob(
            int columnIndex,
            InputStream inputStream,
            long length)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setBlob(columnIndex, inputStream, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column using the given input stream, which
     * will have the specified number of bytes.
     *
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param inputStream An object that contains the data to set the parameter
     * value to.
     * @param length the number of bytes in the parameter data.
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateBlob(
            String columnLabel,
            InputStream inputStream,
            long length)
            throws SQLException {

        int columnIndex = findColumn(columnLabel);

        startUpdate(columnIndex);
        preparedStatement.setBlob(columnIndex, inputStream, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column using the given {@code Reader}
     * object, which is the given number of characters long.
     * When a very large UNICODE value is input to a {@code LONGVARCHAR}
     * parameter, it may be more practical to send it via a
     * {@code java.io.Reader} object. The JDBC driver will
     * do any necessary conversion from UNICODE to the database char format.
     *
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param reader An object that contains the data to set the parameter value to.
     * @param length the number of characters in the parameter data.
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateClob(
            int columnIndex,
            Reader reader,
            long length)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setClob(columnIndex, reader, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column using the given {@code Reader}
     * object, which is the given number of characters long.
     * When a very large UNICODE value is input to a {@code LONGVARCHAR}
     * parameter, it may be more practical to send it via a
     * {@code java.io.Reader} object.  The JDBC driver will
     * do any necessary conversion from UNICODE to the database char format.
     *
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param reader An object that contains the data to set the parameter value to.
     * @param length the number of characters in the parameter data.
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateClob(
            String columnLabel,
            Reader reader,
            long length)
            throws SQLException {

        int columnIndex = findColumn(columnLabel);

        startUpdate(columnIndex);
        preparedStatement.setClob(columnIndex, reader, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column using the given {@code Reader}
     * object, which is the given number of characters long.
     * When a very large UNICODE value is input to a {@code LONGVARCHAR}
     * parameter, it may be more practical to send it via a
     * {@code java.io.Reader} object. The JDBC driver will
     * do any necessary conversion from UNICODE to the database char format.
     *
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnIndex the first column is 1, the second 2, ...
     * @param reader An object that contains the data to set the parameter value to.
     * @param length the number of characters in the parameter data.
     * @throws SQLException if the columnIndex is not valid;
     * if the driver does not support national
     * character sets;  if the driver can detect that a data conversion
     * error could occur; this method is called on a closed result set,
     * if a database access error occurs or
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     *  @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     *  this method
     *  @since JDK 1.6, HSQLDB 2.0
     */
    public void updateNClob(
            int columnIndex,
            Reader reader,
            long length)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setClob(columnIndex, reader, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column using the given {@code Reader}
     * object, which is the given number of characters long.
     * When a very large UNICODE value is input to a {@code LONGVARCHAR}
     * parameter, it may be more practical to send it via a
     * {@code java.io.Reader} object. The JDBC driver will
     * do any necessary conversion from UNICODE to the database char format.
     *
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is the
     * name of the column
     * @param reader An object that contains the data to set the parameter value to.
     * @param length the number of characters in the parameter data.
     * @throws SQLException if the columnLabel is not valid;
     * if the driver does not support national
     *         character sets;  if the driver can detect that a data conversion
     *  error could occur; this method is called on a closed result set;
     *  if a database access error occurs or
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.6, HSQLDB 2.0
     */
    public void updateNClob(
            String columnLabel,
            Reader reader,
            long length)
            throws SQLException {

        int columnIndex = findColumn(columnLabel);

        startUpdate(columnIndex);
        preparedStatement.setClob(columnIndex, reader, length);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a character stream value.
     * The data will be read from the stream
     * as needed until end-of-stream is reached.  The
     * driver does the necessary conversion from Java character format to
     * the national character set in the database.
     * It is intended for use when
     * updating  {@code NCHAR},{@code NVARCHAR}
     * and {@code LONGNVARCHAR} columns.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateNCharacterStream} which takes a length parameter.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param reader the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY} or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateNCharacterStream(
            int columnIndex,
            java.io.Reader reader)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setCharacterStream(columnIndex, reader);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a character stream value.
     * The data will be read from the stream
     * as needed until end-of-stream is reached.  The
     * driver does the necessary conversion from Java character format to
     * the national character set in the database.
     * It is intended for use when
     * updating  {@code NCHAR},{@code NVARCHAR}
     * and {@code LONGNVARCHAR} columns.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateNCharacterStream} which takes a length parameter.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param reader the {@code java.io.Reader} object containing
     *        the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY} or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateNCharacterStream(
            String columnLabel,
            java.io.Reader reader)
            throws SQLException {

        int columnIndex = findColumn(columnLabel);

        startUpdate(columnIndex);
        preparedStatement.setCharacterStream(columnIndex, reader);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with an ascii stream value.
     * The data will be read from the stream
     * as needed until end-of-stream is reached.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateAsciiStream} which takes a length parameter.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateAsciiStream(
            int columnIndex,
            java.io.InputStream x)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setAsciiStream(columnIndex, x);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a binary stream value.
     * The data will be read from the stream
     * as needed until end-of-stream is reached.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateBinaryStream} which takes a length parameter.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateBinaryStream(
            int columnIndex,
            java.io.InputStream x)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setBinaryStream(columnIndex, x);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a character stream value.
     * The data will be read from the stream
     * as needed until end-of-stream is reached.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateCharacterStream} which takes a length parameter.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateCharacterStream(
            int columnIndex,
            java.io.Reader x)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setCharacterStream(columnIndex, x);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with an ascii stream value.
     * The data will be read from the stream
     * as needed until end-of-stream is reached.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateAsciiStream} which takes a length parameter.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateAsciiStream(
            String columnLabel,
            java.io.InputStream x)
            throws SQLException {

        int columnIndex = findColumn(columnLabel);

        startUpdate(columnIndex);
        preparedStatement.setAsciiStream(columnIndex, x);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a binary stream value.
     * The data will be read from the stream
     * as needed until end-of-stream is reached.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateBinaryStream} which takes a length parameter.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param x the new column value
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateBinaryStream(
            String columnLabel,
            java.io.InputStream x)
            throws SQLException {

        int columnIndex = findColumn(columnLabel);

        startUpdate(columnIndex);
        preparedStatement.setBinaryStream(columnIndex, x);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with a character stream value.
     * The data will be read from the stream
     * as needed until end-of-stream is reached.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateCharacterStream} which takes a length parameter.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param reader the {@code java.io.Reader} object containing
     *        the new column value
     * @throws SQLException if the columnLabel is not valid; if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateCharacterStream(
            String columnLabel,
            java.io.Reader reader)
            throws SQLException {

        int columnIndex = findColumn(columnLabel);

        startUpdate(columnIndex);
        preparedStatement.setCharacterStream(columnIndex, reader);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column using the given input stream. The data will be read from the stream
     * as needed until end-of-stream is reached.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateBlob} which takes a length parameter.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param inputStream An object that contains the data to set the parameter
     * value to.
     * @throws SQLException if the columnIndex is not valid; if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateBlob(
            int columnIndex,
            InputStream inputStream)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setBlob(columnIndex, inputStream);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column using the given input stream. The data will be read from the stream
     * as needed until end-of-stream is reached.
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     *   <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateBlob} which takes a length parameter.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param inputStream An object that contains the data to set the parameter
     * value to.
     * @throws SQLException if the columnLabel is not valid; if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateBlob(
            String columnLabel,
            InputStream inputStream)
            throws SQLException {

        int columnIndex = findColumn(columnLabel);

        startUpdate(columnIndex);
        preparedStatement.setBlob(columnIndex, inputStream);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column using the given {@code Reader}
     * object.
     *  The data will be read from the stream
     * as needed until end-of-stream is reached.  The JDBC driver will
     * do any necessary conversion from UNICODE to the database char format.
     *
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     *   <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateClob} which takes a length parameter.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param reader An object that contains the data to set the parameter value to.
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateClob(int columnIndex, Reader reader) throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setClob(columnIndex, reader);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column using the given {@code Reader}
     * object.
     *  The data will be read from the stream
     * as needed until end-of-stream is reached.  The JDBC driver will
     * do any necessary conversion from UNICODE to the database char format.
     *
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateClob} which takes a length parameter.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param reader An object that contains the data to set the parameter value to.
     * @throws SQLException if the columnLabel is not valid; if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateClob(
            String columnLabel,
            Reader reader)
            throws SQLException {

        int columnIndex = findColumn(columnLabel);

        startUpdate(columnIndex);
        preparedStatement.setClob(columnIndex, reader);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column using the given {@code Reader}
     *
     * The data will be read from the stream
     * as needed until end-of-stream is reached.  The JDBC driver will
     * do any necessary conversion from UNICODE to the database char format.
     *
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateNClob} which takes a length parameter.
     *
     * @param columnIndex the first column is 1, the second 2, ...
     * @param reader An object that contains the data to set the parameter value to.
     * @throws SQLException if the columnIndex is not valid;
     * if the driver does not support national
     * character sets;  if the driver can detect that a data conversion
     * error could occur; this method is called on a closed result set,
     * if a database access error occurs or
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateNClob(
            int columnIndex,
            Reader reader)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setClob(columnIndex, reader);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column using the given {@code Reader}
     * object.
     * The data will be read from the stream
     * as needed until end-of-stream is reached.  The JDBC driver will
     * do any necessary conversion from UNICODE to the database char format.
     *
     * <p>
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * {@code updateNClob} which takes a length parameter.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.  If the SQL AS clause was not specified, then the label is the name of the column
     * @param reader An object that contains the data to set the parameter value to.
     * @throws SQLException if the columnLabel is not valid; if the driver does not support national
     *         character sets;  if the driver can detect that a data conversion
     *  error could occur; this method is called on a closed result set;
     *  if a database access error occurs or
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since 1.6
     */
    public void updateNClob(
            String columnLabel,
            Reader reader)
            throws SQLException {

        int columnIndex = findColumn(columnLabel);

        startUpdate(columnIndex);
        preparedStatement.setClob(columnIndex, reader);

        isRowUpdated = true;
    }

    // --------------------- java.sql.Wrapper implementation -------------------

    /**
     * Returns an object that implements the given interface to allow access to
     * non-standard methods, or standard methods not exposed by the proxy.
     *
     * If the receiver implements the interface then the result is the receiver
     * or a proxy for the receiver. If the receiver is a wrapper
     * and the wrapped object implements the interface then the result is the
     * wrapped object or a proxy for the wrapped object. Otherwise return
     * the result of calling {@code unwrap} recursively on the wrapped object
     * or a proxy for that result. If the receiver is not a
     * wrapper and does not implement the interface, then an {@code SQLException} is thrown.
     *
     * @param iface A Class defining an interface that the result must implement.
     * @return an object that implements the interface. May be a proxy for the actual implementing object.
     * @throws java.sql.SQLException If no object found that implements the interface
     * @since JDK 1.6, HSQLDB 2.0
     */
    @SuppressWarnings("unchecked")
    public <T> T unwrap(java.lang.Class<T> iface) throws java.sql.SQLException {

        if (isWrapperFor(iface)) {
            return (T) this;
        }

        throw JDBCUtil.invalidArgument("iface: " + iface);
    }

    /**
     * Returns true if this either implements the interface argument or is directly or indirectly a wrapper
     * for an object that does. Returns false otherwise. If this implements the interface then return true,
     * else if this is a wrapper then return the result of recursively calling {@code isWrapperFor} on the wrapped
     * object. If this does not implement the interface and is not a wrapper, return false.
     * This method should be implemented as a low-cost operation compared to {@code unwrap} so that
     * callers can use this method to avoid expensive {@code unwrap} calls that may fail. If this method
     * returns true then calling {@code unwrap} with the same argument should succeed.
     *
     * @param iface a Class defining an interface.
     * @return true if this implements the interface or directly or indirectly wraps an object that does.
     * @throws java.sql.SQLException  if an error occurs while determining whether this is a wrapper
     * for an object with the given interface.
     * @since JDK 1.6, HSQLDB 2.0
     */
    public boolean isWrapperFor(
            java.lang.Class<?> iface)
            throws java.sql.SQLException {
        return (iface != null && iface.isAssignableFrom(this.getClass()));
    }

//------------------------- JDBC 4.1 -----------------------------------

    /**
     * <p>Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object and will convert from the
     * SQL type of the column to the requested Java data type, if the
     * conversion is supported. If the conversion is not
     * supported  or null is specified for the type, a
     * {@code SQLException} is thrown.
     * <p>
     * At a minimum, an implementation must support the conversions defined in
     * Appendix B, Table B-3 and conversion of appropriate user defined SQL
     * types to a Java type which implements {@code SQLData}, or {@code Struct}.
     * Additional conversions may be supported and are vendor defined.
     * @param <T> the type of the class modeled by this Class object
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param type Class representing the Java data type to convert the designated
     * column to.
     * @return an instance of {@code type} holding the column value
     * @throws SQLException if conversion is not supported, type is null or
     *         another error occurs. The getCause() method of the
     * exception may provide a more detailed exception, for example, if
     * a conversion error occurs
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.7, HSQLDB 2.0.1
     */
    public <T> T getObject(int columnIndex, Class<T> type) throws SQLException {

        if (type == null) {
            throw JDBCUtil.nullArgument();
        }

        final Object source = getColumnValue(columnIndex);

        if (wasNullValue) {
            return null;
        }

        Object o = null;

        switch (type.getName()) {

            case "int" :
            case "java.lang.Integer" :
                o = getInt(columnIndex);
                break;

            case "double" :
            case "java.lang.Double" :
                o = getDouble(columnIndex);
                break;

            case "boolean" :
            case "java.lang.Boolean" :
                o = getBoolean(columnIndex);
                break;

            case "byte" :
            case "java.lang.Byte" :
                o = getByte(columnIndex);
                break;

            case "short" :
            case "java.lang.Short" :
                o = getShort(columnIndex);
                break;

            case "long" :
            case "java.lang.Long" :
                o = getLong(columnIndex);
                break;

            case "[B" :
                o = getBytes(columnIndex);
                break;

            case "java.lang.Object" :
                o = getObject(columnIndex);
                break;

            case "java.math.BigDecimal" :
                o = getBigDecimal(columnIndex);
                break;

            case "java.sql.Blob" :
                o = getBlob(columnIndex);
                break;

            case "java.sql.Clob" :
                o = getClob(columnIndex);
                break;

            case "java.lang.String" :
            case "java.lang.CharSequence" :
                o = getString(columnIndex);
                break;

            case "java.sql.Date" : {
                o = getDate(columnIndex);
                break;
            }

            case "java.sql.Time" : {
                o = getTime(columnIndex);
                break;
            }

            case "java.sql.Timestamp" : {
                o = getTimestamp(columnIndex);
                break;
            }

            case "java.util.UUID" : {
                Type columnType = resultMetaData.columnTypes[columnIndex - 1];

                if (columnType.isUUIDType()) {
                    o = Type.SQL_GUID.convertSQLToJava(session, source);
                } else {
                    Object value = Type.SQL_GUID.convertToTypeJDBC(
                        session,
                        source,
                        columnType);

                    o = Type.SQL_GUID.convertSQLToJava(session, value);
                }

                break;
            }

            case "java.time.Instant" : {
                Type columnType = resultMetaData.columnTypes[columnIndex - 1];

                if (columnType.isDateOrTimestampType()) {
                    TimestampData v = (TimestampData) source;

                    o = ((DateTimeType) columnType).toInstant(session, v);
                }

                break;
            }

            case "java.time.LocalDate" : {
                Type columnType = resultMetaData.columnTypes[columnIndex - 1];

                if (columnType.isDateOrTimestampType()) {
                    TimestampData v = (TimestampData) source;

                    o = ((DateTimeType) columnType).toLocalDate(session, v);
                }

                break;
            }

            case "java.time.LocalTime" : {
                Type columnType = resultMetaData.columnTypes[columnIndex - 1];

                if (columnType.isTimeType()) {
                    TimeData v = (TimeData) source;

                    o = ((DateTimeType) columnType).toLocalTime(session, v);
                } else if (columnType.isTimestampType()) {
                    TimestampData v = (TimestampData) source;

                    o = ((DateTimeType) columnType).toLocalTime(session, v);
                }

                break;
            }

            case "java.time.LocalDateTime" : {
                Type columnType = resultMetaData.columnTypes[columnIndex - 1];

                if (columnType.isDateOrTimestampType()) {
                    TimestampData v = (TimestampData) source;

                    o = ((DateTimeType) columnType).toLocalDateTime(session, v);
                }

                break;
            }

            case "java.time.OffsetTime" : {
                Type columnType = resultMetaData.columnTypes[columnIndex - 1];

                if (columnType.isTimeType()) {
                    TimeData v = (TimeData) source;

                    o = ((DateTimeType) columnType).toOffsetTime(session, v);
                } else if (columnType.isTimestampType()) {
                    TimestampData v = (TimestampData) source;

                    o = ((DateTimeType) columnType).toOffsetTime(session, v);
                }

                break;
            }

            case "java.time.OffsetDateTime" : {
                Type columnType = resultMetaData.columnTypes[columnIndex - 1];

                if (columnType.isDateOrTimestampType()) {
                    TimestampData v = (TimestampData) source;

                    o = ((DateTimeType) columnType).toOffsetDateTime(
                        session,
                        v);
                }

                break;
            }

            case "java.time.Duration" : {
                Type sourceType = resultMetaData.columnTypes[columnIndex - 1];

                if (!sourceType.isIntervalDaySecondType()) {
                    break;
                }

                IntervalSecondData v = (IntervalSecondData) source;

                o = Duration.ofSeconds(v.getSeconds(), v.getNanos());
                break;
            }

            case "java.time.Period" : {
                Type sourceType = resultMetaData.columnTypes[columnIndex - 1];

                if (!sourceType.isIntervalYearMonthType()) {
                    break;
                }

                IntervalMonthData v      = (IntervalMonthData) source;
                int               months = v.getMonths();

                if (sourceType.typeCode == Types.SQL_INTERVAL_MONTH) {
                    o = Period.ofMonths(months);
                } else {
                    o = Period.of(months / 12, months % 12, 0);
                }

                break;
            }

            case "java.sql.Array" : {
                o = getArray(columnIndex);
                break;
            }
        }

        if (o == null) {
            throw JDBCUtil.sqlException(ErrorCode.X_42561);
        }

        return (T) o;
    }

    /**
     * <p>Retrieves the value of the designated column in the current row
     * of this {@code ResultSet} object and will convert from the
     * SQL type of the column to the requested Java data type, if the
     * conversion is supported. If the conversion is not
     * supported  or null is specified for the type, a
     * {@code SQLException} is thrown.
     * <p>
     * At a minimum, an implementation must support the conversions defined in
     * Appendix B, Table B-3 and conversion of appropriate user defined SQL
     * types to a Java type which implements {@code SQLData}, or {@code Struct}.
     * Additional conversions may be supported and are vendor defined.
     *
     * @param columnLabel the label for the column specified with the SQL AS clause.
     * If the SQL AS clause was not specified, then the label is the name
     * of the column
     * @param type Class representing the Java data type to convert the designated
     * column to.
     * @param <T> the type of the class modeled by this Class object
     * @return an instance of {@code type} holding the column value
     * @throws SQLException if conversion is not supported, type is null or
     *         another error occurs. The getCause() method of the
     * exception may provide a more detailed exception, for example, if
     * a conversion error occurs
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     * this method
     * @since JDK 1.7, HSQLDB 2.0.1
     */
    public <T> T getObject(
            String columnLabel,
            Class<T> type)
            throws SQLException {
        return getObject(findColumn(columnLabel), type);
    }

//------------------------- JDBC 4.2 -----------------------------------

    /**
     * Updates the designated column with an {@code Object} value.
     *
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *<p>
     * If the second argument is an {@code InputStream} then the stream must contain
     * the number of bytes specified by scaleOrLength.  If the second argument is a
     * {@code Reader} then the reader must contain the number of characters specified
     * by scaleOrLength. If these conditions are not true the driver will generate a
     * {@code SQLException} when the statement is executed.
     *<p>
     * The default implementation will throw {@code SQLFeatureNotSupportedException}
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @param targetSqlType the SQL type to be sent to the database
     * @param scaleOrLength for an object of {@code java.math.BigDecimal} ,
     *          this is the number of digits after the decimal point. For
     *          Java Object types {@code InputStream} and {@code Reader},
     *          this is the length
     *          of the data in the stream or reader.  For all other types,
     *          this value will be ignored.
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not
     * support this method; if the JDBC driver does not support the specified targetSqlType
     * @see JDBCType
     * @see SQLType
     * @since 1.8
     */
    public void updateObject(
            int columnIndex,
            Object x,
            SQLType targetSqlType,
            int scaleOrLength)
            throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setObject(
            columnIndex,
            x,
            targetSqlType,
            scaleOrLength);

        isRowUpdated = true;
    }

    /**
     * Updates the designated column with an {@code Object} value.
     *
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *<p>
     * If the second argument is an {@code InputStream} then the stream must
     * contain number of bytes specified by scaleOrLength.  If the second
     * argument is a {@code Reader} then the reader must contain the number
     * of characters specified by scaleOrLength. If these conditions are not
     * true the driver will generate a
     * {@code SQLException} when the statement is executed.
     *<p>
     * The default implementation will throw {@code SQLFeatureNotSupportedException}
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is
     * the name of the column
     * @param x the new column value
     * @param targetSqlType the SQL type to be sent to the database
     * @param scaleOrLength for an object of {@code java.math.BigDecimal} ,
     *          this is the number of digits after the decimal point. For
     *          Java Object types {@code InputStream} and {@code Reader},
     *          this is the length
     *          of the data in the stream or reader.  For all other types,
     *          this value will be ignored.
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not
     * support this method; if the JDBC driver does not support the specified targetSqlType
     * @see JDBCType
     * @see SQLType
     * @since 1.8
     */
    public void updateObject(
            String columnLabel,
            Object x,
            SQLType targetSqlType,
            int scaleOrLength)
            throws SQLException {
        updateObject(findColumn(columnLabel), x, targetSqlType, scaleOrLength);
    }

    /**
     * Updates the designated column with an {@code Object} value.
     *
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *<p>
     * The default implementation will throw {@code SQLFeatureNotSupportedException}
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param x the new column value
     * @param targetSqlType the SQL type to be sent to the database
     * @throws SQLException if the columnIndex is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not
     * support this method; if the JDBC driver does not support the specified targetSqlType
     * @see JDBCType
     * @see SQLType
     * @since 1.8
     */
    public void updateObject(
            int columnIndex,
            Object x,
            SQLType targetSqlType)
            throws SQLException {
        preparedStatement.setObject(columnIndex, x, targetSqlType);
    }

    /**
     * Updates the designated column with an {@code Object} value.
     *
     * The updater methods are used to update column values in the
     * current row or the insert row.  The updater methods do not
     * update the underlying database; instead the {@code updateRow} or
     * {@code insertRow} methods are called to update the database.
     *<p>
     * The default implementation will throw {@code SQLFeatureNotSupportedException}
     *
     * @param columnLabel the label for the column specified with the SQL AS
     * clause.  If the SQL AS clause was not specified, then the label is
     * the name of the column
     * @param x the new column value
     * @param targetSqlType the SQL type to be sent to the database
     * @throws SQLException if the columnLabel is not valid;
     * if a database access error occurs;
     * the result set concurrency is {@code CONCUR_READ_ONLY}
     * or this method is called on a closed result set
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not
     * support this method; if the JDBC driver does not support the specified targetSqlType
     * @see JDBCType
     * @see SQLType
     * @since 1.8
     */
    public void updateObject(
            String columnLabel,
            Object x,
            SQLType targetSqlType)
            throws SQLException {
        updateObject(findColumn(columnLabel), x, targetSqlType);
    }

    private Object getTimestampWithZone(int columnIndex) throws SQLException {

        TimestampData v = (TimestampData) getColumnInType(
            columnIndex,
            Type.SQL_TIMESTAMP_WITH_TIME_ZONE);

        if (v == null) {
            return null;
        }

        return Type.SQL_TIMESTAMP_WITH_TIME_ZONE.convertSQLToJava(session, v);
    }

    private Object getTimeWithZone(int columnIndex) throws SQLException {

        DateTimeType columnType =
            (DateTimeType) resultMetaData.columnTypes[columnIndex - 1];
        TimeData v = (TimeData) getColumnValue(columnIndex);

        if (v == null) {
            return null;
        }

        return columnType.convertSQLToJava(session, v);
    }

//------------------------ Internal Implementation -----------------------------

    /** The internal representation. */
    private RowSetNavigator navigator;

    /** The internal representation. */
    protected ResultMetaData resultMetaData;

    /**
     * Translation of INTERVAL types
     */
    private boolean translateTTIType;

    /** How many columns does this ResultSet have? */
    private int columnCount;

    /** Did the last getXXX method encounter a null value? */
    private boolean wasNullValue;

    /** The ResultSetMetaData object for this ResultSet */
    private ResultSetMetaData resultSetMetaData;

    /** Accelerates fineColumn {@code Map<columnName, columnIndex>} */
    private IntValueHashMap<String> columnMap;

    /** The first warning in the chain. Null if there are no warnings. */
    private SQLWarning rootWarning;

    /** If true, retrieves a lob as an in-memory object - this breaks updatability*/
    private boolean memoryLobs;

    // -------------------------- Package Attributes ----------------------------

    /**
     * The Statement that generated this result. Null if the result is
     * from DatabaseMetaData
     */
    JDBCStatementBase statement;

    /**
     * Session or ClientConnection
     */
    SessionInterface session;

    /** JDBCConnection for this. */
    JDBCConnection connection;

    /**
     * The scrollability / scroll sensitivity type of this result.
     */
    boolean isScrollable;

    /** The updatability of this result. */
    boolean isUpdatable;

    /** The insertability of this result. */
    boolean isInsertable;
    int     rsProperties;
    int     fetchSize;

    /** Statement is closed when its result set is closed */
    boolean autoClose;

    /** The underlying result. */
    public Result result;

    //-------------------------- Private Methods -------------------------------

    /**
     * Fetches the current row of the result set.
     *
     * @throws SQLException when result set is closed; result set is empty;
     *   result set is before first; result set is after last; no row data is
     *   available.
     * @return Object[]
     */
    protected Object[] getCurrent() throws SQLException {

        if (navigator == null) {
            throw JDBCUtil.sqlException(ErrorCode.X_24501);
        } else if (navigator.isEmpty()) {
            throw JDBCUtil.sqlException(
                ErrorCode.X_24504,
                ErrorCode.M_RS_EMPTY);
        } else if (navigator.isBeforeFirst()) {
            throw JDBCUtil.sqlException(
                ErrorCode.X_24504,
                ErrorCode.M_RS_BEFORE_FIRST);
        } else if (navigator.isAfterLast()) {
            throw JDBCUtil.sqlException(
                ErrorCode.X_24504,
                ErrorCode.M_RS_AFTER_LAST);
        } else if (isOnInsertRow) {
            throw JDBCUtil.sqlException(
                ErrorCode.X_24504,
                ErrorCode.M_RS_ON_INSERT_ROW);
        }

        Object[] data = navigator.getCurrent();

        if (data == null) {
            throw JDBCUtil.sqlException(ErrorCode.X_24501);
        }

        return data;
    }

    /**
     * Internal closed state check.
     *
     * @throws SQLException when this result set is closed
     */
    private void checkClosed() throws SQLException {
        if (navigator == null || navigator.isClosed()) {
            throw JDBCUtil.sqlException(ErrorCode.X_24501);
        }
    }

    /**
     * Internal column index validity check.
     *
     * @param columnIndex to check
     * @throws SQLException when this ResultSet has no such column
     */
    protected void checkColumn(int columnIndex) throws SQLException {

        if (navigator == null) {
            throw JDBCUtil.sqlException(ErrorCode.X_24501);
        }

        if (columnIndex < 1 || columnIndex > columnCount) {
            throw JDBCUtil.sqlException(
                ErrorCode.JDBC_COLUMN_NOT_FOUND,
                String.valueOf(columnIndex));
        }
    }

    /**
     * Internal wasNull tracker.
     *
     * @param o the Object to track
     * @return boolean
     */
    protected boolean trackNull(Object o) {
        return (wasNullValue = (o == null));
    }

    /**
     * Internal get value.
     */
    protected Object getColumnValue(int columnIndex) throws SQLException {

        Object value;

        if (isOnInsertRow) {
            checkColumn(columnIndex);

            if (preparedStatement.parameterSet[columnIndex - 1]) {
                value = preparedStatement.parameterValues[columnIndex - 1];
            } else {
                throw JDBCUtil.sqlException(ErrorCode.JDBC_PARAMETER_NOT_SET);
            }
        } else {
            Object[] rowData = getCurrent();

            checkColumn(columnIndex);

            value = rowData[columnIndex - 1];
        }

        trackNull(value);

        return value;
    }

    /**
     * Internal value converter. <p>
     *
     * All trivially successful getXXX methods eventually go through this
     * method, converting if necessary from the source type to the
     * requested type.  <p>
     *
     * Conversion to the JDBC representation, if different, is handled by the
     * calling methods.
     *
     * @param columnIndex of the column value for which to perform the
     *                 conversion
     * @param targetType the org.hsqldb.types.Type object for targetType
     * @return an Object of the requested targetType, representing the value of the
     *       specified column
     * @throws SQLException when there is no rowData, the column index is
     *    invalid, or the conversion cannot be performed
     */
    protected Object getColumnInType(
            int columnIndex,
            Type targetType)
            throws SQLException {

        Object value      = getColumnValue(columnIndex);
        Type   sourceType = resultMetaData.columnTypes[columnIndex - 1];

        if (value == null) {
            return null;
        }

        if (translateTTIType && targetType.isIntervalType()) {
            targetType = ((IntervalType) targetType).getCharacterType();
        }

        if (sourceType.typeCode != targetType.typeCode) {
            try {
                value = targetType.convertToTypeJDBC(
                    session,
                    value,
                    sourceType);
            } catch (Exception e) {
                String stringValue = (value instanceof Number
                                      || value instanceof String)
                                     ? value.toString()
                                     : "instance of "
                                       + value.getClass().getName();
                String msg = "from SQL type " + sourceType.getNameString()
                             + " to " + targetType.getJDBCClassName()
                             + ", value: " + stringValue;

                throw JDBCUtil.sqlException(ErrorCode.X_42561, msg);
            }
        }

        return value;
    }

    private void checkNotForwardOnly() throws SQLException {
        if (!isScrollable) {
            throw JDBCUtil.notSupported();
        }
    }

    //-------------------------- Updates --------------------------------------
    JDBCPreparedStatement preparedStatement;
    boolean               isRowUpdated;
    boolean               isOnInsertRow;

    /**
     * -2 after moveToInsertRow()
     * -1 before any updateXXX()
     * row number after any updateXXX()
     * -1 after updateRow(), clearUpdate() or moveToCurrentRow();
     */
    int currentUpdateRowNumber = -1;

    private void checkUpdatable() throws SQLException {

        checkClosed();

        if (!isUpdatable) {
            throw JDBCUtil.notUpdatableColumn();
        }
    }

    private void checkUpdatable(int columnIndex) throws SQLException {

        checkColumn(columnIndex);

        if (resultMetaData.colIndexes[--columnIndex] == -1) {
            throw JDBCUtil.notUpdatableColumn();
        }

        if (!resultMetaData.columns[columnIndex].isWriteable()) {
            throw JDBCUtil.notUpdatableColumn();
        }
    }

    void startUpdate(int columnIndex) throws SQLException {

        checkClosed();

        if (!isUpdatable) {
            throw JDBCUtil.notUpdatableColumn();
        }

        checkUpdatable(columnIndex);

        if (isRowUpdated) {
            return;
        }

        if (!isOnInsertRow) {
            if (rowDeleted()) {
                throw JDBCUtil.sqlException(
                    ErrorCode.X_0U000,
                    ErrorCode.M_RS_ROW_DELETED);
            }

            if (currentUpdateRowNumber != navigator.getRowNumber()) {
                preparedStatement.clearParameters();
            }

            currentUpdateRowNumber = navigator.getRowNumber();
        }
    }

    private void clearUpdates() throws SQLException {

        checkUpdatable();
        preparedStatement.clearParameters();

        currentUpdateRowNumber = -1;
        isRowUpdated           = false;
    }

    private void startInsert() throws SQLException {

        if (isRowUpdated) {
            throw JDBCUtil.sqlExceptionSQL(ErrorCode.X_24504);
        }

        checkUpdatable();

        // check insertable
        currentUpdateRowNumber = -2;
        isOnInsertRow          = true;
    }

    private void endInsert() throws SQLException {

        checkUpdatable();
        preparedStatement.clearParameters();

        currentUpdateRowNumber = -1;
        isOnInsertRow          = false;
    }

    private void setParameter(int columnIndex, Object x) throws SQLException {

        startUpdate(columnIndex);
        preparedStatement.setParameter(columnIndex, x);

        isRowUpdated = true;
    }

    private void performUpdate() throws SQLException {

        Object[] rowData = getCurrent();

        if (!isRowUpdated) {
            return;
        }

        preparedStatement.parameterValues[columnCount + ResultMetaData.SysOffsets.rowId] =
            rowData[columnCount + ResultMetaData.SysOffsets.rowId];
        preparedStatement.parameterValues[columnCount + ResultMetaData.SysOffsets.rowNum] =
            navigator.getRowNumber();

        for (int i = 0; i < columnCount; i++) {
            boolean set = preparedStatement.parameterSet[i];

            preparedStatement.resultOut.metaData.columnTypes[i] = set
                    ? preparedStatement.parameterTypes[i]
                    : Type.SQL_ALL_TYPES;
        }

        preparedStatement.resultOut.setActionType(
            ResultConstants.UPDATE_CURSOR);
        preparedStatement.resultOut.setResultId(navigator.getId());

        try {
            preparedStatement.fetchResult();

            if (connection.isNetConn) {
                Result          r                = preparedStatement.resultIn;
                RowSetNavigator updatesNavigator = r.getNavigator();

                updatesNavigator.beforeFirst();

                while (updatesNavigator.next()) {
                    Object[] updateData = updatesNavigator.getCurrent();
                    Number rowNumber =
                        (Number) updateData[columnCount + ResultMetaData.SysOffsets.rowNum];

                    ((RowSetNavigatorClient) navigator).setData(
                        rowNumber.intValue(),
                        updateData);
                }
            }
        } finally {
            preparedStatement.clearParameters();

            rootWarning = preparedStatement.getWarnings();

            preparedStatement.clearWarnings();

            isRowUpdated = false;
        }
    }

    private void performInsert() throws SQLException {

        checkUpdatable();

        for (int i = 0; i < columnCount; i++) {
            boolean set = preparedStatement.parameterSet[i];

            if (!set) {
                if (!resultMetaData.columns[i].isIdentity()) {
                    throw JDBCUtil.sqlException(ErrorCode.X_24515);
                }
            }

            preparedStatement.resultOut.metaData.columnTypes[i] =
                preparedStatement.parameterTypes[i];
        }

        preparedStatement.resultOut.setActionType(
            ResultConstants.INSERT_CURSOR);
        preparedStatement.resultOut.setResultId(navigator.getId());

        try {
            preparedStatement.fetchResult();
        } finally {
            preparedStatement.clearParameters();

            rootWarning = preparedStatement.getWarnings();

            preparedStatement.clearWarnings();

            isRowUpdated = false;
        }
    }

    private void performDelete() throws SQLException {

        checkUpdatable();

        if (isOnInsertRow) {
            throw JDBCUtil.sqlExceptionSQL(ErrorCode.X_24504);
        }

        if (rowDeleted()) {
            return;
        }

        preparedStatement.parameterValues[columnCount + ResultMetaData.SysOffsets.rowId] =
            getCurrent()[columnCount + ResultMetaData.SysOffsets.rowId];
        preparedStatement.parameterValues[columnCount + ResultMetaData.SysOffsets.rowNum] =
            navigator.getRowNumber();

        preparedStatement.resultOut.setActionType(
            ResultConstants.DELETE_CURSOR);
        preparedStatement.resultOut.setResultId(navigator.getId());

        try {
            preparedStatement.fetchResult();

            if (connection.isNetConn) {
                Result          r                = preparedStatement.resultIn;
                RowSetNavigator updatesNavigator = r.getNavigator();

                updatesNavigator.beforeFirst();

                while (updatesNavigator.next()) {
                    Object[] updateData = updatesNavigator.getCurrent();
                    Number rowNumber =
                        (Number) updateData[columnCount + ResultMetaData.SysOffsets.rowNum];

                    ((RowSetNavigatorClient) navigator).setData(
                        rowNumber.intValue(),
                        updateData);
                }
            }
        } finally {
            preparedStatement.clearParameters();

            rootWarning = preparedStatement.getWarnings();

            preparedStatement.clearWarnings();
        }
    }

    //-------------------------- Other Methods --------------------------------
    // HSQLDB Specific
    RowSetNavigator getNavigator() {
        return navigator;
    }

    void setNavigator(RowSetNavigator navigator) {
        this.navigator = navigator;
    }

    //------------------------------ Constructors -------------------------------

    /**
     * Constructs a new {@code JDBCResultSet} object using the specified
     * navigator and {@code org.hsqldb.result.ResultMetaData}.
     *
     * @param conn JDBCConnection
     * @param s the statement
     * @param r the internal result form that the new
     *   {@code JDBCResultSet} represents
     * @param metaData the connection properties
     */
    public JDBCResultSet(
            JDBCConnection conn,
            JDBCStatementBase s,
            Result r,
            ResultMetaData metaData) {

        this(conn, r, metaData);

        this.statement = s;
        isScrollable   = ResultProperties.isScrollable(rsProperties);

        if (ResultProperties.isUpdatable(rsProperties)) {
            isUpdatable  = true;
            isInsertable = true;

            for (int i = 0; i < metaData.colIndexes.length; i++) {
                if (metaData.colIndexes[i] < 0) {
                    isInsertable = false;
                    break;
                }
            }

            preparedStatement = new JDBCPreparedStatement(s.connection, result);
        }
    }

    public JDBCResultSet(
            JDBCConnection conn,
            Result r,
            ResultMetaData metaData) {

        this.session    = conn == null
                          ? null
                          : conn.sessionProxy;
        this.result     = r;
        this.connection = conn;
        rsProperties    = r.rsProperties;
        navigator       = r.getNavigator();
        resultMetaData  = metaData;
        columnCount     = resultMetaData.getColumnCount();

        if (conn != null) {
            translateTTIType = conn.isTranslateTTIType;
            memoryLobs       = conn.isMemoryLobs;
        }
    }

    /**
     * Factory method returns a new {@code JDBCResultSet} object for
     * use with user defined functions that return a ResultSet object.
     * See {@code org.hsqldb.jdbc.JDBCArrayBasic} for usage example.
     *
     * @param r the internal result form that the new
     *   {@code JDBCResultSet} represents
     * @param metaData the connection properties
     * @return ResultSet
     */
    public static JDBCResultSet newJDBCResultSet(
            Result r,
            ResultMetaData metaData) {
        return new JDBCResultSetBasic(r, metaData);
    }

    public static JDBCResultSet newEmptyResultSet() {

        ResultMetaData md     = ResultMetaData.newResultMetaData(1);
        ColumnBase     column = new ColumnBase(null, null, null, "C1");

        column.setType(Type.SQL_INTEGER);

        md.columnTypes[0] = Type.SQL_INTEGER;
        md.columns[0]     = column;

        Result r = Result.newSingleColumnResult(md);

        return newJDBCResultSet(r, md);
    }

    static class JDBCResultSetBasic extends JDBCResultSet {

        JDBCResultSetBasic(Result r, ResultMetaData metaData) {
            super(null, r, metaData);
        }

        protected Object getColumnInType(
                int columnIndex,
                Type targetType)
                throws SQLException {

            Object[] rowData = getCurrent();
            Type     sourceType;
            Object   value;

            checkColumn(columnIndex);

            sourceType = resultMetaData.columnTypes[--columnIndex];
            value      = rowData[columnIndex];

            if (trackNull(value)) {
                return null;
            }

            if (sourceType.typeCode != targetType.typeCode) {
                throw JDBCUtil.sqlException(ErrorCode.X_42561);
            }

            return value;
        }

        public Date getDate(int columnIndex) throws SQLException {
            return (Date) getColumnInType(columnIndex, Type.SQL_DATE);
        }

        public Time getTime(int columnIndex) throws SQLException {
            return (Time) getColumnInType(columnIndex, Type.SQL_DATE);
        }

        public Timestamp getTimestamp(int columnIndex) throws SQLException {
            return (Timestamp) getColumnInType(columnIndex, Type.SQL_DATE);
        }

        public java.io.InputStream getBinaryStream(
                int columnIndex)
                throws SQLException {
            throw JDBCUtil.notSupported();
        }

        public java.io.Reader getCharacterStream(
                int columnIndex)
                throws SQLException {
            throw JDBCUtil.notSupported();
        }

        public Blob getBlob(int columnIndex) throws SQLException {

            checkColumn(columnIndex);

            Type   sourceType = resultMetaData.columnTypes[columnIndex - 1];
            Object o          = getColumnInType(columnIndex, sourceType);

            if (o == null) {
                return null;
            }

            if (o instanceof Blob) {
                return (Blob) o;
            } else if (o instanceof byte[]) {
                return new JDBCBlob((byte[]) o);
            }

            throw JDBCUtil.sqlException(ErrorCode.X_42561);
        }

        public Clob getClob(int columnIndex) throws SQLException {

            checkColumn(columnIndex);

            Type   sourceType = resultMetaData.columnTypes[columnIndex - 1];
            Object o          = getColumnInType(columnIndex, sourceType);

            if (o == null) {
                return null;
            }

            if (o instanceof Clob) {
                return (Clob) o;
            } else if (o instanceof String) {
                return new JDBCClob((String) o);
            }

            throw JDBCUtil.sqlException(ErrorCode.X_42561);
        }

        public Time getTime(int columnIndex, Calendar cal) throws SQLException {
            throw JDBCUtil.notSupported();
        }

        public Timestamp getTimestamp(
                int columnIndex,
                Calendar cal)
                throws SQLException {
            throw JDBCUtil.notSupported();
        }
    }
}
