#!/bin/sh

# This tool initialises a deployment of imip-agent, creating data stores and
# published data directories, creating and initialising databases, and setting
# filesystem permissions. It is configured using the contents of the config.sh
# script.
#
# Copyright (C) 2015, 2016 Paul Boddie <paul@boddie.org.uk>
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 3 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program.  If not, see <http://www.gnu.org/licenses/>.

DIRNAME=`dirname "$0"`
CONFIG="$DIRNAME/config.sh"

if [ -e "$CONFIG" ]; then
    . "$CONFIG"
else
    CONFIG=/etc/imip-agent/config.sh
    . "$CONFIG"
fi

SCHEMA="$DIRNAME/../conf/postgresql/schema.sql"

if [ ! -e "$SCHEMA" ]; then
    SCHEMA=/etc/imip-agent/postgresql/schema.sql
fi

PROGNAME=`basename "$0"`

if [ "$1" = "--help" ]; then
    cat 1>&2 <<EOF
Usage: $PROGNAME

Initialise stored and published data directories at...

  * $INSTALL_DIR
  * $WEB_INSTALL_DIR

...respectively.

Set permissions to the user and group respectively given as $IMIP_AGENT_USER
and $IMIP_AGENT_GROUP.

Within the stored data directory, the following directories will be created
(with STORE_TYPE currently set as "$STORE_TYPE"):

  * $INSTALL_DIR/preferences
EOF

    if [ "$STORE_TYPE" = "file" ]; then
        cat 1>&2 <<EOF
  * $INSTALL_DIR/journal (if STORE_TYPE is "file")
  * $INSTALL_DIR/store (if STORE_TYPE is "file")
EOF
    fi

    cat 1>&2 <<EOF

Within the published data directory the following directory will be created:

  * $WEB_INSTALL_DIR/static
EOF

    if [ "$STORE_TYPE" = "postgresql" ]; then
        cat 1>&2 <<EOF

With STORE_TYPE set as "database", a database schema will be initialised for the
following database:

  * $POSTGRESQL_DB
EOF
    fi

    cat 1>&2 <<EOF

See $CONFIG for the settings used as described above.
EOF
    exit 1
fi

# Test for a privileged user.

if [ `whoami` != 'root' ]; then
    cat 1>&2 <<EOF
You will need to become a privileged user using su or sudo to run this program
because it changes file ownership and may also switch users to run database
administration commands.
EOF
    exit 1
fi

# Create necessary directories regardless of store type.

echo "Creating preferences and static Web directories..." 1>&2

for DIR in "$INSTALL_DIR"/preferences "$WEB_INSTALL_DIR"/static ; do
    mkdir -p "$DIR"
    chown "$IMIP_AGENT_USER" "$DIR"
    chgrp "$IMIP_AGENT_GROUP" "$DIR"
    chmod g+ws "$DIR"
done

# Initialise a file store.

if [ "$STORE_TYPE" = "file" ]; then

    echo "Creating store and journal directories..." 1>&2

    for DIR in "$INSTALL_DIR"/store "$INSTALL_DIR"/journal ; do
        mkdir -p "$DIR"
        chown "$IMIP_AGENT_USER" "$DIR"
        chgrp "$IMIP_AGENT_GROUP" "$DIR"
        chmod g+ws "$DIR"
    done

# Initialise a PostgreSQL store.

elif [ "$STORE_TYPE" = "postgresql" ]; then

    # Check for the database.

    echo "Checking for the database ${POSTGRESQL_DB}..." 1>&2

    if $AS_POSTGRES psql -tA -c 'select datname from pg_database' postgres | grep -q ^"$POSTGRESQL_DB"$ ; then
        cat 1>&2 <<EOF
Database $POSTGRESQL_DB already exists.
EOF
        exit 1
    fi

    # Attempt to create the database.

    echo "Creating database ${POSTGRESQL_DB}..." 1>&2

    if ! $AS_POSTGRES createdb "$POSTGRESQL_DB" ; then
        cat 1>&2 <<EOF
Could not create database $POSTGRESQL_DB using createdb.
EOF
        exit 1
    fi

    # Attempt to initialise the schema.

    echo "Initialising the schema for database ${POSTGRESQL_DB}..." 1>&2

    if ! $AS_POSTGRES psql -q -f "$SCHEMA" "$POSTGRESQL_DB" ; then
        cat 1>&2 <<EOF
Could not initialise schema in database $POSTGRESQL_DB using psql.
EOF
        exit 1
    fi

    # For each user needing to connect, attempt to create a role and grant it
    # privileges on the tables.

    for USER in $POSTGRESQL_USERS ; do

        echo "Creating a database user for ${USER}..." 1>&2

        if ! $AS_POSTGRES createuser -D -R -S "$USER" ; then
            cat 1>&2 <<EOF
Could not create database user $USER using createuser.
EOF
        fi

        echo "Granting privileges to database user for ${USER}..." 1>&2

        if ! $AS_POSTGRES psql -Atc '\dt' "$POSTGRESQL_DB" \
           | cut -d '|' -f 2 \
           | xargs -I{} $AS_POSTGRES psql -q -c "grant all privileges on table {} to \"$USER\"" "$POSTGRESQL_DB" ; then

            cat 1>&2 <<EOF
Could not grant permissions for schema in database $POSTGRESQL_DB to $USER
using psql.
EOF
        fi
    done
fi
