#include "CommandlineOptions.hh"
#include "PointConfiguration.hh"
#include "Symmetry.hh"
#include "Chirotope.hh"
#include "Incidences.hh"
#include "PlacingTriang.hh"
#include "RegularityCheck.hh"

int main(const int argc, const char** argv) {

  using namespace topcom;
  
  CommandlineOptions::init(argc, argv);
  
  MessageStreams::forced() << std::endl;
  MessageStreams::forced() << "------------------------------------------------------------------\n";
  MessageStreams::forced() << " checking regularity of triangulations \n";
  MessageStreams::forced() << " TOPCOM client: " << CommandlineOptions::client() << '\n';
  MessageStreams::forced() << "------------------------------------------------------------------\n";
  MessageStreams::forced() << std::endl;

  RegularityCheck::init();

  PointConfiguration points;
  if (!points.read(std::cin)) {
    MessageStreams::forced() << "error while reading point configuration" << std::endl;
    return 1;
  }
  if (points.rank() < points.rowdim()) {
    MessageStreams::verbose() << "point configuration has " << points.rowdim() << " rows of rank " << points.rank() << std::endl;
    points.transform_to_full_rank();
    MessageStreams::verbose() << "resulting no of rows after transformation: " << points.rank() << std::endl;
    points.pretty_print(MessageStreams::verbose());
  }
    MessageStreams::verbose() << "read point configuration with " << points.no()
			      << " points in rank " << points.rank() << std::endl;
  
  SymmetryGroup symmetries(points.no());
  symmetries.read(std::cin);
  Chirotope chiro(points, false);
  SimplicialComplex triang;
  size_type count(0);
  size_type count_nonregs(0);
  if (CommandlineOptions::memopt()) {
      MessageStreams::verbose() << "no preprocessing of simplex table to save memory" << std::endl;
  }
  else {
    MessageStreams::verbose() << "preprocessing simplex table up to rank " << points.rank() << " ..." << std::endl;
    SimplicialComplex::preprocess_index_table(points.no(), 0, points.rank(), chiro, true);
    MessageStreams::verbose() << "... done." << std::endl;
  }
  MessageStreams::verbose() << "computing cocircuits ..." << std::endl;
  Cocircuits cocircuits(chiro);
  MessageStreams::verbose() << "... done." << std::endl;
  MessageStreams::verbose() << "computing facets ..." << std::endl;
  Facets facets(cocircuits);
  MessageStreams::verbose() << "... done." << std::endl;
  MessageStreams::verbose() << "computing incidences ..." << std::endl;
  Incidences incidences(chiro, facets);
  MessageStreams::verbose() << "... done." << std::endl;
  MessageStreams::verbose() << "checking input triangulations for regularity ..." << std::endl;
  while (triang.read(std::cin)) {
    ++count;
    RegularityCheck regcheck(points, chiro, incidences, triang);
    if (!regcheck.is_regular()) {
      ++count_nonregs;
      if (CommandlineOptions::debug()) {
	MessageStreams::result() << "T[" << count << "] := " << triang << " is non-regular." << std::endl;
      }
      else {
	MessageStreams::result() << count << ": " << triang << " -> non-regular." << std::endl;
      }
    }
    else {
      if (CommandlineOptions::debug()) {
	MessageStreams::result() << "T[" << count << "] := " << triang << " is regular." << std::endl;
      }
      else {
	MessageStreams::result() << count << ": " << triang << " -> regular." << std::endl;
      }
      if (CommandlineOptions::output_heights()) {
	if (CommandlineOptions::debug()) {
	  MessageStreams::result() << "h[" << count
				   << "] := " << regcheck.heights() << ";\n";
	}
	else {
	  MessageStreams::result() << count << ": " << regcheck.heights() << ";\n";
	}
      }
    }
    MessageStreams::verbose() << "checked " << count << " triangulations, " 
			      << count_nonregs << " non-regular so far." << std::endl;
  }
  MessageStreams::verbose() << "... no more triangulations found." << std::endl;
  RegularityCheck::term();
  return 0;
}


// eof checkregularity.cc
