/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import java.io.IOException;
import java.util.Deque;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.Predicate;
import javax.annotation.Nullable;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.hints.HintsDescriptor;
import org.apache.cassandra.hints.HintsWriter;
import org.apache.cassandra.hints.InputPosition;
import org.apache.cassandra.hints.PendingHintsInfo;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.schema.Schema;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.SyncUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class HintsStore {
    private static final Logger logger = LoggerFactory.getLogger(HintsStore.class);
    public final UUID hostId;
    private final File hintsDirectory;
    private final ImmutableMap<String, Object> writerParams;
    private final Map<HintsDescriptor, InputPosition> dispatchPositions;
    private final Deque<HintsDescriptor> dispatchDequeue;
    private final Queue<HintsDescriptor> corruptedFiles;
    private final Map<HintsDescriptor, Long> hintsExpirations;
    private volatile long lastUsedTimestamp;
    private volatile HintsWriter hintsWriter;

    private HintsStore(UUID hostId, File hintsDirectory, ImmutableMap<String, Object> writerParams, List<HintsDescriptor> descriptors) {
        this.hostId = hostId;
        this.hintsDirectory = hintsDirectory;
        this.writerParams = writerParams;
        this.dispatchPositions = new ConcurrentHashMap<HintsDescriptor, InputPosition>();
        this.dispatchDequeue = new ConcurrentLinkedDeque<HintsDescriptor>(descriptors);
        this.corruptedFiles = new ConcurrentLinkedQueue<HintsDescriptor>();
        this.hintsExpirations = new ConcurrentHashMap<HintsDescriptor, Long>();
        this.lastUsedTimestamp = descriptors.stream().mapToLong(d -> d.timestamp).max().orElse(0L);
    }

    static HintsStore create(UUID hostId, File hintsDirectory, ImmutableMap<String, Object> writerParams, List<HintsDescriptor> descriptors) {
        descriptors.sort((d1, d2) -> Long.compare(d1.timestamp, d2.timestamp));
        return new HintsStore(hostId, hintsDirectory, writerParams, descriptors);
    }

    @VisibleForTesting
    int getDispatchQueueSize() {
        return this.dispatchDequeue.size();
    }

    @VisibleForTesting
    int getHintsExpirationsMapSize() {
        return this.hintsExpirations.size();
    }

    InetAddressAndPort address() {
        return StorageService.instance.getEndpointForHostId(this.hostId);
    }

    @Nullable
    PendingHintsInfo getPendingHintsInfo() {
        Iterator<HintsDescriptor> descriptors = this.dispatchDequeue.iterator();
        int queueSize = 0;
        long minTimestamp = Long.MAX_VALUE;
        long maxTimestamp = Long.MIN_VALUE;
        while (descriptors.hasNext()) {
            HintsDescriptor descriptor = descriptors.next();
            minTimestamp = Math.min(minTimestamp, descriptor.timestamp);
            maxTimestamp = Math.max(maxTimestamp, descriptor.timestamp);
            ++queueSize;
        }
        if (queueSize == 0) {
            return null;
        }
        return new PendingHintsInfo(this.hostId, queueSize, minTimestamp, maxTimestamp);
    }

    public long findOldestHintTimestamp() {
        HintsDescriptor desc = this.dispatchDequeue.peekFirst();
        if (desc != null) {
            return desc.timestamp;
        }
        HintsWriter writer = this.getWriter();
        if (writer != null) {
            return writer.descriptor().timestamp;
        }
        return Long.MAX_VALUE;
    }

    boolean isLive() {
        InetAddressAndPort address = this.address();
        return address != null && FailureDetector.instance.isAlive(address);
    }

    HintsDescriptor poll() {
        return this.dispatchDequeue.poll();
    }

    void offerFirst(HintsDescriptor descriptor) {
        this.dispatchDequeue.offerFirst(descriptor);
    }

    void offerLast(HintsDescriptor descriptor) {
        this.dispatchDequeue.offerLast(descriptor);
    }

    void deleteAllHints() {
        HintsDescriptor descriptor;
        while ((descriptor = this.poll()) != null) {
            this.cleanUp(descriptor);
            this.delete(descriptor);
        }
        while ((descriptor = this.corruptedFiles.poll()) != null) {
            this.cleanUp(descriptor);
            this.delete(descriptor);
        }
    }

    void deleteExpiredHints(long now) {
        this.deleteHints(it -> this.hasExpired((HintsDescriptor)it, now));
    }

    private boolean hasExpired(HintsDescriptor descriptor, long now) {
        Long cachedExpiresAt = this.hintsExpirations.get(descriptor);
        if (null != cachedExpiresAt) {
            return cachedExpiresAt <= now;
        }
        File hintFile = new File(this.hintsDirectory, descriptor.fileName());
        if (!hintFile.exists() || hintFile.lastModified() == 0L) {
            return false;
        }
        long ttl = hintFile.lastModified() + (long)Schema.instance.largestGcgs();
        this.hintsExpirations.put(descriptor, ttl);
        return ttl <= now;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void deleteHints(Predicate<HintsDescriptor> predicate) {
        HashSet<HintsDescriptor> removeSet = new HashSet<HintsDescriptor>();
        try {
            for (HintsDescriptor descriptor : Iterables.concat(this.dispatchDequeue, this.corruptedFiles)) {
                if (!predicate.test(descriptor)) continue;
                this.cleanUp(descriptor);
                removeSet.add(descriptor);
                this.delete(descriptor);
            }
        }
        finally {
            this.dispatchDequeue.removeAll(removeSet);
            this.corruptedFiles.removeAll(removeSet);
        }
    }

    void delete(HintsDescriptor descriptor) {
        File hintsFile = descriptor.file(this.hintsDirectory);
        if (hintsFile.tryDelete()) {
            logger.info("Deleted hint file {}", (Object)descriptor.fileName());
        } else if (hintsFile.exists()) {
            logger.error("Failed to delete hint file {}", (Object)descriptor.fileName());
        } else {
            logger.info("Already deleted hint file {}", (Object)descriptor.fileName());
        }
        descriptor.checksumFile(this.hintsDirectory).tryDelete();
    }

    boolean hasFiles() {
        return !this.dispatchDequeue.isEmpty();
    }

    InputPosition getDispatchOffset(HintsDescriptor descriptor) {
        return this.dispatchPositions.get(descriptor);
    }

    void markDispatchOffset(HintsDescriptor descriptor, InputPosition inputPosition) {
        this.dispatchPositions.put(descriptor, inputPosition);
    }

    long getTotalFileSize() {
        long total = 0L;
        for (HintsDescriptor descriptor : Iterables.concat(this.dispatchDequeue, this.corruptedFiles)) {
            total += descriptor.hintsFileSize(this.hintsDirectory);
        }
        HintsWriter currentWriter = this.getWriter();
        if (null != currentWriter) {
            total += currentWriter.descriptor().hintsFileSize(this.hintsDirectory);
        }
        return total;
    }

    void cleanUp(HintsDescriptor descriptor) {
        this.dispatchPositions.remove(descriptor);
        this.hintsExpirations.remove(descriptor);
    }

    void markCorrupted(HintsDescriptor descriptor) {
        this.corruptedFiles.add(descriptor);
    }

    boolean isWriting() {
        return this.hintsWriter != null;
    }

    HintsWriter getOrOpenWriter() {
        if (this.hintsWriter == null) {
            this.hintsWriter = this.openWriter();
        }
        return this.hintsWriter;
    }

    HintsWriter getWriter() {
        return this.hintsWriter;
    }

    private HintsWriter openWriter() {
        this.lastUsedTimestamp = Math.max(Clock.Global.currentTimeMillis(), this.lastUsedTimestamp + 1L);
        HintsDescriptor descriptor = new HintsDescriptor(this.hostId, this.lastUsedTimestamp, this.writerParams);
        try {
            return HintsWriter.create(this.hintsDirectory, descriptor);
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, descriptor.fileName());
        }
    }

    void closeWriter() {
        if (this.hintsWriter != null) {
            this.hintsWriter.close();
            this.offerLast(this.hintsWriter.descriptor());
            this.hintsWriter = null;
            SyncUtil.trySyncDir(this.hintsDirectory);
        }
    }

    void fsyncWriter() {
        if (this.hintsWriter != null) {
            this.hintsWriter.fsync();
        }
    }
}

