/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.transport;

import com.google.common.primitives.Ints;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.Channel;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.exceptions.OverloadedException;
import org.apache.cassandra.exceptions.OversizedCQLMessageException;
import org.apache.cassandra.metrics.ClientMessageSizeMetrics;
import org.apache.cassandra.metrics.ClientMetrics;
import org.apache.cassandra.net.AbstractMessageHandler;
import org.apache.cassandra.net.FrameDecoder;
import org.apache.cassandra.net.FrameEncoder;
import org.apache.cassandra.net.ResourceLimits;
import org.apache.cassandra.net.ShareableBytes;
import org.apache.cassandra.transport.ClientResourceLimits;
import org.apache.cassandra.transport.ConnectionStage;
import org.apache.cassandra.transport.Dispatcher;
import org.apache.cassandra.transport.Envelope;
import org.apache.cassandra.transport.Flusher;
import org.apache.cassandra.transport.Message;
import org.apache.cassandra.transport.ProtocolException;
import org.apache.cassandra.transport.ProtocolVersion;
import org.apache.cassandra.transport.QueueBackpressure;
import org.apache.cassandra.transport.ServerConnection;
import org.apache.cassandra.transport.messages.ErrorMessage;
import org.apache.cassandra.utils.MonotonicClock;
import org.apache.cassandra.utils.NoSpamLogger;
import org.apache.cassandra.utils.concurrent.NonBlockingRateLimiter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CQLMessageHandler<M extends Message>
extends AbstractMessageHandler {
    private static final Logger logger = LoggerFactory.getLogger(CQLMessageHandler.class);
    private static final NoSpamLogger noSpamLogger = NoSpamLogger.getLogger(logger, 1L, TimeUnit.SECONDS);
    public static final int LARGE_MESSAGE_THRESHOLD = 131071;
    public static final TimeUnit RATE_LIMITER_DELAY_UNIT = TimeUnit.NANOSECONDS;
    static final String MULTI_FRAME_AUTH_ERROR_MESSAGE_PREFIX = "The connection is not yet in a valid state to process multi frame CQL Messages, usually thismeans that authentication is still pending. ";
    private final QueueBackpressure queueBackpressure;
    private final Envelope.Decoder envelopeDecoder;
    private final Message.Decoder<M> messageDecoder;
    private final FrameEncoder.PayloadAllocator payloadAllocator;
    private final MessageConsumer<M> dispatcher;
    private final ErrorHandler errorHandler;
    private final boolean throwOnOverload;
    private final ProtocolVersion version;
    private final NonBlockingRateLimiter requestRateLimiter;
    long channelPayloadBytesInFlight;
    private int consecutiveMessageErrors = 0;
    private final ServerConnection serverConnection;

    CQLMessageHandler(Channel channel, ServerConnection serverConnection, ProtocolVersion version, FrameDecoder decoder, Envelope.Decoder envelopeDecoder, Message.Decoder<M> messageDecoder, MessageConsumer<M> dispatcher, FrameEncoder.PayloadAllocator payloadAllocator, int queueCapacity, QueueBackpressure queueBackpressure, ClientResourceLimits.ResourceProvider resources, AbstractMessageHandler.OnHandlerClosed onClosed, ErrorHandler errorHandler, boolean throwOnOverload) {
        super(decoder, channel, 131071, queueCapacity, resources.endpointLimit(), resources.globalLimit(), resources.endpointWaitQueue(), resources.globalWaitQueue(), onClosed);
        this.serverConnection = serverConnection;
        this.envelopeDecoder = envelopeDecoder;
        this.messageDecoder = messageDecoder;
        this.payloadAllocator = payloadAllocator;
        this.dispatcher = dispatcher;
        this.queueBackpressure = queueBackpressure;
        this.errorHandler = errorHandler;
        this.throwOnOverload = throwOnOverload;
        this.version = version;
        this.requestRateLimiter = resources.requestRateLimiter();
    }

    @Override
    public boolean process(FrameDecoder.Frame frame) throws IOException {
        this.consecutiveMessageErrors = 0;
        return super.process(frame);
    }

    @Override
    protected boolean processOneContainedMessage(ShareableBytes bytes, ResourceLimits.Limit endpointReserve, ResourceLimits.Limit globalReserve) {
        ByteBuffer buf = bytes.get();
        Envelope.Decoder.HeaderExtractionResult extracted = this.envelopeDecoder.extractHeader(buf);
        if (!extracted.isSuccess()) {
            return this.handleProtocolException(extracted.error(), buf, extracted.streamId(), extracted.bodyLength());
        }
        Envelope.Header header = extracted.header();
        if (header.version != this.version) {
            ProtocolException error = new ProtocolException(String.format("Invalid message version. Got %s but previousmessages on this connection had version %s", header.version, this.version));
            return this.handleProtocolException(error, buf, header.streamId, header.bodySizeInBytes);
        }
        int messageSize = Ints.checkedCast((long)header.bodySizeInBytes);
        ClientResourceLimits.Overload backpressure = ClientResourceLimits.Overload.NONE;
        if (this.throwOnOverload) {
            if (!this.acquireCapacity(header, endpointReserve, globalReserve)) {
                this.discardAndThrow(endpointReserve, globalReserve, buf, header, messageSize, ClientResourceLimits.Overload.BYTES_IN_FLIGHT);
                return true;
            }
            if (DatabaseDescriptor.getNativeTransportRateLimitingEnabled() && !this.requestRateLimiter.tryReserve()) {
                backpressure = ClientResourceLimits.Overload.REQUESTS;
            } else if (!this.dispatcher.hasQueueCapacity()) {
                backpressure = ClientResourceLimits.Overload.QUEUE_TIME;
            }
            if (backpressure != ClientResourceLimits.Overload.NONE) {
                this.release(header);
                this.discardAndThrow(endpointReserve, globalReserve, buf, header, messageSize, backpressure);
                return true;
            }
        } else {
            long delay = -1L;
            if (!this.acquireCapacityAndQueueOnFailure(header, endpointReserve, globalReserve)) {
                backpressure = ClientResourceLimits.Overload.BYTES_IN_FLIGHT;
            }
            if (DatabaseDescriptor.getNativeTransportRateLimitingEnabled()) {
                delay = this.requestRateLimiter.reserveAndGetDelay(RATE_LIMITER_DELAY_UNIT);
                if (backpressure == ClientResourceLimits.Overload.NONE && delay > 0L) {
                    backpressure = ClientResourceLimits.Overload.REQUESTS;
                }
            }
            if (backpressure == ClientResourceLimits.Overload.NONE && !this.dispatcher.hasQueueCapacity() && (delay = this.queueBackpressure.markAndGetDelay(RATE_LIMITER_DELAY_UNIT)) > 0L) {
                backpressure = ClientResourceLimits.Overload.QUEUE_TIME;
            }
            if (backpressure != ClientResourceLimits.Overload.NONE) {
                if (this.processRequestAndUpdateMetrics(bytes, header, messageSize, backpressure)) {
                    if (this.decoder.isActive()) {
                        ClientMetrics.instance.pauseConnection();
                    }
                    if (delay > 0L) {
                        this.scheduleConnectionWakeupTask(delay, RATE_LIMITER_DELAY_UNIT);
                    }
                }
                return false;
            }
        }
        return this.processRequestAndUpdateMetrics(bytes, header, messageSize, ClientResourceLimits.Overload.NONE);
    }

    private boolean processRequestAndUpdateMetrics(ShareableBytes bytes, Envelope.Header header, int messageSize, ClientResourceLimits.Overload backpressure) {
        this.channelPayloadBytesInFlight += (long)messageSize;
        this.incrementReceivedMessageMetrics(messageSize);
        return this.processRequest(this.composeRequest(header, bytes), backpressure);
    }

    private void discardAndThrow(ResourceLimits.Limit endpointReserve, ResourceLimits.Limit globalReserve, ByteBuffer buf, Envelope.Header header, int messageSize, ClientResourceLimits.Overload overload) {
        ClientMetrics.instance.markRequestDiscarded();
        this.logOverload(endpointReserve, globalReserve, header, messageSize);
        OverloadedException exception = CQLMessageHandler.buildOverloadedException(endpointReserve, globalReserve, this.requestRateLimiter, overload);
        this.handleError((Throwable)exception, header);
        this.incrementReceivedMessageMetrics(messageSize);
        buf.position(buf.position() + 9 + messageSize);
    }

    public static OverloadedException buildOverloadedException(ResourceLimits.Limit endpointReserve, ResourceLimits.Limit globalReserve, NonBlockingRateLimiter requestRateLimiter, ClientResourceLimits.Overload overload) {
        return CQLMessageHandler.buildOverloadedException(() -> String.format("Endpoint: %d/%d bytes, Global: %d/%d bytes.", endpointReserve.using(), endpointReserve.limit(), globalReserve.using(), globalReserve.limit()), requestRateLimiter, overload);
    }

    public static OverloadedException buildOverloadedException(Supplier<String> endpointLimits, NonBlockingRateLimiter requestRateLimiter, ClientResourceLimits.Overload overload) {
        switch (overload) {
            case REQUESTS: {
                return new OverloadedException(String.format("Request breached global limit of %d requests/second. Server is currently in an overloaded state and cannot accept more requests.", requestRateLimiter.getRate()));
            }
            case BYTES_IN_FLIGHT: {
                return new OverloadedException(String.format("Request breached limit on bytes in flight. (%s)Server is currently in an overloaded state and cannot accept more requests.", endpointLimits.get()));
            }
            case QUEUE_TIME: {
                return new OverloadedException(String.format("Request has spent over %s time of the maximum timeout %dms in the queue", DatabaseDescriptor.getNativeTransportQueueMaxItemAgeThreshold(), DatabaseDescriptor.getNativeTransportTimeout(TimeUnit.MILLISECONDS)));
            }
        }
        throw new IllegalArgumentException("Overload exception should not have been thrown with " + overload);
    }

    private void logOverload(ResourceLimits.Limit endpointReserve, ResourceLimits.Limit globalReserve, Envelope.Header header, int messageSize) {
        logger.trace("Discarded request of size {} with {} bytes in flight on channel. Using {}/{} bytes of endpoint limit and {}/{} bytes of global limit. Global rate limiter: {} Header: {}", new Object[]{messageSize, this.channelPayloadBytesInFlight, endpointReserve.using(), endpointReserve.limit(), globalReserve.using(), globalReserve.limit(), this.requestRateLimiter, header});
    }

    private boolean handleProtocolException(ProtocolException exception, ByteBuffer buf, int streamId, long expectedMessageLength) {
        if (expectedMessageLength < 0L || ++this.consecutiveMessageErrors > DatabaseDescriptor.getConsecutiveMessageErrorsThreshold()) {
            if (!exception.isFatal()) {
                exception = ProtocolException.toFatalException(exception);
            }
            this.handleError((Throwable)exception, streamId);
            return false;
        }
        this.handleError((Throwable)exception, streamId);
        buf.position(Math.min(buf.limit(), buf.position() + 9 + Ints.checkedCast((long)expectedMessageLength)));
        return true;
    }

    private void incrementReceivedMessageMetrics(int messageSize) {
        ++this.receivedCount;
        this.receivedBytes += (long)(messageSize + 9);
        ClientMessageSizeMetrics.bytesReceived.inc((long)(messageSize + 9));
        ClientMessageSizeMetrics.bytesReceivedPerRequest.update(messageSize + 9);
    }

    private Envelope composeRequest(Envelope.Header header, ShareableBytes bytes) {
        ByteBuffer buf = bytes.get();
        int idx = buf.position() + 9;
        int end = idx + Ints.checkedCast((long)header.bodySizeInBytes);
        ByteBuf body = Unpooled.wrappedBuffer((ByteBuffer)buf.slice());
        body.readerIndex(9);
        body.retain();
        buf.position(end);
        return new Envelope(header, body);
    }

    protected boolean processRequest(Envelope request, ClientResourceLimits.Overload backpressure) {
        Object message = null;
        try {
            message = this.messageDecoder.decode(this.channel, request);
            this.dispatcher.dispatch(this.channel, message, this::toFlushItem, backpressure);
            this.consecutiveMessageErrors = 0;
            return true;
        }
        catch (Exception e2) {
            ProtocolException e2;
            if (message != null) {
                request.release();
            }
            boolean continueProcessing = true;
            if (++this.consecutiveMessageErrors > DatabaseDescriptor.getConsecutiveMessageErrorsThreshold()) {
                if (!(e2 instanceof ProtocolException)) {
                    logger.debug("Error decoding CQL message", (Throwable)e2);
                    e2 = new ProtocolException("Error encountered decoding CQL message: " + e2.getMessage());
                }
                e2 = ProtocolException.toFatalException(e2);
                continueProcessing = false;
            }
            this.handleErrorAndRelease(e2, request.header);
            return continueProcessing;
        }
    }

    private void handleErrorAndRelease(Throwable t, Envelope.Header header) {
        this.release(header);
        this.handleError(t, header);
    }

    private void handleError(Throwable t, Envelope.Header header) {
        this.handleError(t, header.streamId);
    }

    private void handleError(Throwable t, int streamId) {
        this.errorHandler.accept(ErrorMessage.wrap(t, streamId));
    }

    private void handleError(Throwable t) {
        this.errorHandler.accept(t);
    }

    private Flusher.FlushItem.Framed toFlushItem(Channel channel, Message.Request request, Message.Response response) {
        Envelope responseFrame = response.encode(request.getSource().header.version);
        int responseSize = CQLMessageHandler.envelopeSize(responseFrame.header);
        ClientMessageSizeMetrics.bytesSent.inc((long)responseSize);
        ClientMessageSizeMetrics.bytesSentPerResponse.update(responseSize);
        return new Flusher.FlushItem.Framed(channel, responseFrame, request.getSource(), this.payloadAllocator, this::release);
    }

    private void release(Flusher.FlushItem<Envelope> flushItem) {
        this.release(flushItem.request.header);
        flushItem.request.release();
        ((Envelope)flushItem.response).release();
    }

    private void release(Envelope.Header header) {
        this.releaseCapacity(Ints.checkedCast((long)header.bodySizeInBytes));
        this.channelPayloadBytesInFlight -= header.bodySizeInBytes;
    }

    @Override
    protected boolean processFirstFrameOfLargeMessage(FrameDecoder.IntactFrame frame, ResourceLimits.Limit endpointReserve, ResourceLimits.Limit globalReserve) throws IOException {
        ShareableBytes bytes = frame.contents;
        ByteBuffer buf = bytes.get();
        try {
            Envelope.Decoder.HeaderExtractionResult extracted = this.envelopeDecoder.extractHeader(buf);
            if (!extracted.isSuccess()) {
                this.handleError(ProtocolException.toFatalException(extracted.error()));
                return false;
            }
            Envelope.Header header = extracted.header();
            int messageSize = Ints.checkedCast((long)header.bodySizeInBytes);
            this.receivedBytes += (long)buf.remaining();
            if (this.serverConnection != null && this.serverConnection.stage() != ConnectionStage.READY) {
                this.handleError(ProtocolException.toFatalException(new OversizedAuthMessageException("The connection is not yet in a valid state to process multi frame CQL Messages, usually thismeans that authentication is still pending. type = " + header.type + ", size = " + header.bodySizeInBytes)));
                ClientMetrics.instance.markRequestDiscarded();
                return false;
            }
            LargeMessage largeMessage = new LargeMessage(header);
            if ((long)messageSize > DatabaseDescriptor.getNativeTransportMaxMessageSizeInBytes()) {
                ClientMetrics.instance.markRequestDiscarded();
                largeMessage.markTooBig();
            } else if (this.throwOnOverload) {
                if (!this.acquireCapacity(header, endpointReserve, globalReserve)) {
                    ClientMetrics.instance.markRequestDiscarded();
                    this.logOverload(endpointReserve, globalReserve, header, messageSize);
                    largeMessage.markOverloaded(ClientResourceLimits.Overload.BYTES_IN_FLIGHT);
                } else {
                    ClientResourceLimits.Overload backpressure = ClientResourceLimits.Overload.NONE;
                    if (DatabaseDescriptor.getNativeTransportRateLimitingEnabled() && !this.requestRateLimiter.tryReserve()) {
                        backpressure = ClientResourceLimits.Overload.REQUESTS;
                    } else if (!this.dispatcher.hasQueueCapacity()) {
                        backpressure = ClientResourceLimits.Overload.QUEUE_TIME;
                    }
                    if (backpressure != ClientResourceLimits.Overload.NONE) {
                        ClientMetrics.instance.markRequestDiscarded();
                        this.logOverload(endpointReserve, globalReserve, header, messageSize);
                        largeMessage.markOverloaded(backpressure);
                        this.largeMessage = largeMessage;
                        largeMessage.supply(frame);
                        return true;
                    }
                }
            } else if (this.acquireCapacityAndQueueOnFailure(header, endpointReserve, globalReserve)) {
                long delay = -1L;
                ClientResourceLimits.Overload backpressure = ClientResourceLimits.Overload.NONE;
                if (DatabaseDescriptor.getNativeTransportRateLimitingEnabled() && (delay = this.requestRateLimiter.reserveAndGetDelay(RATE_LIMITER_DELAY_UNIT)) > 0L) {
                    backpressure = ClientResourceLimits.Overload.REQUESTS;
                }
                if (backpressure == ClientResourceLimits.Overload.NONE && !this.dispatcher.hasQueueCapacity() && (delay = this.queueBackpressure.markAndGetDelay(RATE_LIMITER_DELAY_UNIT)) > 0L) {
                    backpressure = ClientResourceLimits.Overload.QUEUE_TIME;
                }
                if (delay > 0L) {
                    this.largeMessage = largeMessage;
                    largeMessage.markBackpressure(backpressure);
                    largeMessage.supply(frame);
                    if (this.decoder.isActive()) {
                        ClientMetrics.instance.pauseConnection();
                    }
                    this.scheduleConnectionWakeupTask(delay, RATE_LIMITER_DELAY_UNIT);
                    return false;
                }
            } else {
                return false;
            }
            this.largeMessage = largeMessage;
            largeMessage.supply(frame);
            return true;
        }
        catch (Exception e) {
            throw new IOException("Error decoding CQL Message", e);
        }
    }

    @Override
    protected String id() {
        return this.channel.id().asShortText();
    }

    private void scheduleConnectionWakeupTask(long waitLength, TimeUnit unit) {
        this.channel.eventLoop().schedule(() -> {
            try {
                if (!this.decoder.isActive()) {
                    this.decoder.reactivate();
                    if (this.decoder.isActive()) {
                        ClientMetrics.instance.unpauseConnection();
                    }
                }
            }
            catch (Throwable t) {
                this.fatalExceptionCaught(t);
            }
        }, waitLength, unit);
    }

    private boolean acquireCapacityAndQueueOnFailure(Envelope.Header header, ResourceLimits.Limit endpointReserve, ResourceLimits.Limit globalReserve) {
        int bytesRequired = Ints.checkedCast((long)header.bodySizeInBytes);
        long currentTimeNanos = MonotonicClock.Global.approxTime.now();
        return this.acquireCapacity(endpointReserve, globalReserve, bytesRequired, currentTimeNanos, Long.MAX_VALUE);
    }

    private boolean acquireCapacity(Envelope.Header header, ResourceLimits.Limit endpointReserve, ResourceLimits.Limit globalReserve) {
        int bytesRequired = Ints.checkedCast((long)header.bodySizeInBytes);
        return this.acquireCapacity(endpointReserve, globalReserve, bytesRequired) == ResourceLimits.Outcome.SUCCESS;
    }

    @Override
    protected void processCorruptFrame(FrameDecoder.CorruptFrame frame) {
        ++this.corruptFramesUnrecovered;
        String error = String.format("%s invalid, unrecoverable CRC mismatch detected in frame %s. Read %d, Computed %d", this.id(), frame.isRecoverable() ? "body" : "header", frame.readCRC, frame.computedCRC);
        noSpamLogger.error(error, new Object[0]);
        if (!frame.isSelfContained) {
            if (null == this.largeMessage) {
                this.receivedBytes += (long)frame.frameSize;
            } else {
                this.processSubsequentFrameOfLargeMessage(frame);
            }
        }
        this.handleError(ProtocolException.toFatalException(new ProtocolException(error)));
    }

    @Override
    protected void fatalExceptionCaught(Throwable cause) {
        this.decoder.discard();
        logger.warn("Unrecoverable exception caught in CQL message processing pipeline, closing the connection", cause);
        this.channel.close();
    }

    static int envelopeSize(Envelope.Header header) {
        return 9 + Ints.checkedCast((long)header.bodySizeInBytes);
    }

    static class OversizedAuthMessageException
    extends ProtocolException {
        OversizedAuthMessageException(String message) {
            super(message);
        }
    }

    private class LargeMessage
    extends AbstractMessageHandler.LargeMessage<Envelope.Header> {
        private static final long EXPIRES_AT = Long.MAX_VALUE;
        private ClientResourceLimits.Overload overload;
        private ClientResourceLimits.Overload backpressure;
        private boolean tooBig;

        private LargeMessage(Envelope.Header header) {
            super(CQLMessageHandler.envelopeSize(header), header, Long.MAX_VALUE, false);
            this.overload = ClientResourceLimits.Overload.NONE;
            this.backpressure = ClientResourceLimits.Overload.NONE;
            this.tooBig = false;
        }

        private Envelope assembleFrame() {
            ByteBuf body = Unpooled.wrappedBuffer((ByteBuffer[])((ByteBuffer[])this.buffers.stream().map(ShareableBytes::get).toArray(ByteBuffer[]::new)));
            body.readerIndex(9);
            body.retain();
            return new Envelope((Envelope.Header)this.header, body);
        }

        private void markOverloaded(ClientResourceLimits.Overload overload) {
            this.overload = overload;
        }

        private void markBackpressure(ClientResourceLimits.Overload backpressure) {
            this.backpressure = backpressure;
        }

        private void markTooBig() {
            this.tooBig = true;
        }

        @Override
        protected void onIntactFrame(FrameDecoder.IntactFrame frame) {
            if (this.tooBig || this.overload != ClientResourceLimits.Overload.NONE) {
                frame.consume();
            } else {
                super.onIntactFrame(frame);
            }
        }

        @Override
        protected void onCorruptFrame() {
            if (!(this.isExpired || this.isCorrupt || this.tooBig)) {
                this.releaseBuffers();
                if (this.overload != ClientResourceLimits.Overload.BYTES_IN_FLIGHT) {
                    CQLMessageHandler.this.releaseCapacity(this.size);
                }
            }
            this.isCorrupt = true;
            this.isExpired |= MonotonicClock.Global.approxTime.isAfter(this.expiresAtNanos);
        }

        @Override
        protected void onComplete() {
            if (this.tooBig) {
                CQLMessageHandler.this.handleError((Throwable)this.buildOversizedCQLMessageException(((Envelope.Header)this.header).bodySizeInBytes), (Envelope.Header)this.header);
            } else if (this.overload != ClientResourceLimits.Overload.NONE) {
                if (this.overload == ClientResourceLimits.Overload.BYTES_IN_FLIGHT) {
                    CQLMessageHandler.this.handleError((Throwable)CQLMessageHandler.buildOverloadedException(CQLMessageHandler.this.endpointReserveCapacity, CQLMessageHandler.this.globalReserveCapacity, CQLMessageHandler.this.requestRateLimiter, this.overload), (Envelope.Header)this.header);
                } else {
                    CQLMessageHandler.this.handleErrorAndRelease(CQLMessageHandler.buildOverloadedException(CQLMessageHandler.this.endpointReserveCapacity, CQLMessageHandler.this.globalReserveCapacity, CQLMessageHandler.this.requestRateLimiter, this.overload), (Envelope.Header)this.header);
                }
            } else if (!this.isCorrupt) {
                CQLMessageHandler.this.processRequest(this.assembleFrame(), this.backpressure);
            }
        }

        @Override
        protected void abort() {
            if (!this.isCorrupt && !this.tooBig && this.overload == ClientResourceLimits.Overload.NONE) {
                this.releaseBuffers();
            }
            if (this.overload == ClientResourceLimits.Overload.NONE || this.overload == ClientResourceLimits.Overload.BYTES_IN_FLIGHT) {
                CQLMessageHandler.this.releaseCapacity(this.size);
            }
        }

        private OversizedCQLMessageException buildOversizedCQLMessageException(long messageBodySize) {
            return new OversizedCQLMessageException("CQL Message of size " + messageBodySize + " bytes exceeds allowed maximum of " + DatabaseDescriptor.getNativeTransportMaxMessageSizeInBytes() + " bytes");
        }
    }

    static interface ErrorHandler {
        public void accept(Throwable var1);
    }

    static interface MessageConsumer<M extends Message> {
        public void dispatch(Channel var1, M var2, Dispatcher.FlushItemConverter var3, ClientResourceLimits.Overload var4);

        public boolean hasQueueCapacity();
    }
}

