/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.k8s.overlord;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import io.fabric8.kubernetes.api.model.Pod;
import io.fabric8.kubernetes.api.model.PodStatus;
import io.fabric8.kubernetes.api.model.batch.v1.Job;
import io.fabric8.kubernetes.client.dsl.LogWatch;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.Nullable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.druid.indexer.TaskLocation;
import org.apache.druid.indexer.TaskStatus;
import org.apache.druid.indexing.common.task.Task;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.java.util.common.concurrent.Execs;
import org.apache.druid.java.util.emitter.EmittingLogger;
import org.apache.druid.k8s.overlord.common.JobResponse;
import org.apache.druid.k8s.overlord.common.K8sTaskId;
import org.apache.druid.k8s.overlord.common.KubernetesPeonClient;
import org.apache.druid.tasklogs.TaskLogs;
import org.checkerframework.checker.nullness.qual.MonotonicNonNull;

public class KubernetesPeonLifecycle {
    private static final EmittingLogger log = new EmittingLogger(KubernetesPeonLifecycle.class);
    private final AtomicReference<State> state = new AtomicReference<State>(State.NOT_STARTED);
    private final K8sTaskId taskId;
    private final TaskLogs taskLogs;
    private final Task task;
    private final KubernetesPeonClient kubernetesClient;
    private final ObjectMapper mapper;
    private final TaskStateListener stateListener;
    private final SettableFuture<Boolean> taskStartedSuccessfullyFuture;
    private final long logSaveTimeoutMs;
    private @MonotonicNonNull LogWatch logWatch;
    private final AtomicReference<TaskLocation> taskLocationRef = new AtomicReference();

    protected KubernetesPeonLifecycle(Task task, K8sTaskId taskId, KubernetesPeonClient kubernetesClient, TaskLogs taskLogs, ObjectMapper mapper, TaskStateListener stateListener, long logSaveTimeoutMs) {
        this.task = task;
        this.taskId = taskId;
        this.kubernetesClient = kubernetesClient;
        this.taskLogs = taskLogs;
        this.mapper = mapper;
        this.stateListener = stateListener;
        this.taskStartedSuccessfullyFuture = SettableFuture.create();
        this.logSaveTimeoutMs = logSaveTimeoutMs;
    }

    protected synchronized TaskStatus run(Job job, long launchTimeout, long timeout, boolean useDeepStorageForTaskPayload) throws IllegalStateException, IOException {
        try {
            this.updateState(new State[]{State.NOT_STARTED}, State.PENDING);
            if (useDeepStorageForTaskPayload) {
                this.writeTaskPayload(this.task);
            }
            this.taskLocationRef.set(null);
            this.kubernetesClient.launchPeonJobAndWaitForStart(job, this.task, launchTimeout, TimeUnit.MILLISECONDS);
            TaskStatus taskStatus = this.join(timeout);
            return taskStatus;
        }
        catch (Exception e) {
            log.info("Failed to run task: %s", new Object[]{this.taskId.getOriginalTaskId()});
            if (!this.taskStartedSuccessfullyFuture.isDone()) {
                this.taskStartedSuccessfullyFuture.set((Object)false);
            }
            throw e;
        }
        finally {
            this.stopTask();
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void writeTaskPayload(Task task) throws IOException {
        Path file = null;
        try {
            file = Files.createTempFile(this.taskId.getOriginalTaskId(), "task.json", new FileAttribute[0]);
            FileUtils.writeStringToFile((File)file.toFile(), (String)this.mapper.writeValueAsString((Object)task), (Charset)Charset.defaultCharset());
            this.taskLogs.pushTaskPayload(task.getId(), file.toFile());
            if (file == null) return;
        }
        catch (Exception e) {
            try {
                log.error("Failed to write task payload for task: %s", new Object[]{this.taskId.getOriginalTaskId()});
                throw new RuntimeException(e);
            }
            catch (Throwable throwable) {
                if (file == null) throw throwable;
                Files.deleteIfExists(file);
                throw throwable;
            }
        }
        Files.deleteIfExists(file);
        return;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    protected synchronized TaskStatus join(long timeout) throws IllegalStateException {
        TaskStatus taskStatus;
        try {
            this.updateState(new State[]{State.NOT_STARTED, State.PENDING}, State.RUNNING);
            this.taskStartedSuccessfullyFuture.set((Object)true);
            JobResponse jobResponse = this.kubernetesClient.waitForPeonJobCompletion(this.taskId, timeout, TimeUnit.MILLISECONDS);
            taskStatus = this.getTaskStatus(jobResponse.getJobDuration());
        }
        catch (Exception e) {
            try {
                if (!this.taskStartedSuccessfullyFuture.isDone()) {
                    this.taskStartedSuccessfullyFuture.set((Object)false);
                }
                throw e;
            }
            catch (Throwable throwable) {
                try {
                    this.saveLogs();
                }
                catch (Exception e2) {
                    log.warn((Throwable)e2, "Log processing failed for task [%s]", new Object[]{this.taskId});
                }
                this.stopTask();
                throw throwable;
            }
        }
        try {
            this.saveLogs();
        }
        catch (Exception e) {
            log.warn((Throwable)e, "Log processing failed for task [%s]", new Object[]{this.taskId});
        }
        this.stopTask();
        return taskStatus;
    }

    protected void shutdown() {
        State currentState = this.state.get();
        if (State.PENDING.equals((Object)currentState) || State.RUNNING.equals((Object)currentState) || State.STOPPED.equals((Object)currentState)) {
            this.kubernetesClient.deletePeonJob(this.taskId);
        }
    }

    protected Optional<InputStream> streamLogs() {
        if (!State.RUNNING.equals((Object)this.state.get())) {
            return Optional.absent();
        }
        return this.kubernetesClient.getPeonLogs(this.taskId);
    }

    protected State getState() {
        return this.state.get();
    }

    protected TaskLocation getTaskLocation() {
        State currentState = this.state.get();
        if (State.PENDING.equals((Object)currentState) || State.NOT_STARTED.equals((Object)currentState)) {
            return TaskLocation.unknown();
        }
        if (this.taskLocationRef.get() == null) {
            Optional<Pod> maybePod = this.kubernetesClient.getPeonPod(this.taskId.getK8sJobName());
            if (!maybePod.isPresent()) {
                log.warn("Could not get task location from k8s for task [%s].", new Object[]{this.taskId});
                return TaskLocation.unknown();
            }
            Pod pod = (Pod)maybePod.get();
            PodStatus podStatus = pod.getStatus();
            if (podStatus == null || podStatus.getPodIP() == null) {
                log.warn("Could not get task location from k8s for task [%s].", new Object[]{this.taskId});
                return TaskLocation.unknown();
            }
            this.taskLocationRef.set(TaskLocation.create((String)podStatus.getPodIP(), (int)8100, (int)8091, (boolean)Boolean.parseBoolean(pod.getMetadata().getAnnotations().getOrDefault("tls.enabled", "false")), (String)(pod.getMetadata() != null ? pod.getMetadata().getName() : "")));
        }
        return this.taskLocationRef.get();
    }

    private TaskStatus getTaskStatus(long duration) {
        TaskStatus taskStatus;
        try {
            Optional maybeTaskStatusStream = this.taskLogs.streamTaskStatus(this.taskId.getOriginalTaskId());
            if (maybeTaskStatusStream.isPresent()) {
                taskStatus = (TaskStatus)this.mapper.readValue(IOUtils.toString((InputStream)((InputStream)maybeTaskStatusStream.get()), (Charset)StandardCharsets.UTF_8), TaskStatus.class);
            } else {
                log.info("Peon for task [%s] did not push its task status. Check k8s logs and events for the pod to see what happened.", new Object[]{this.taskId});
                taskStatus = TaskStatus.failure((String)this.taskId.getOriginalTaskId(), (String)"Peon did not report status successfully.");
            }
        }
        catch (IOException e) {
            log.error((Throwable)e, "Failed to load task status for task [%s]", new Object[]{this.taskId.getOriginalTaskId()});
            taskStatus = TaskStatus.failure((String)this.taskId.getOriginalTaskId(), (String)StringUtils.format((String)"error loading status: %s", (Object[])new Object[]{e.getMessage()}));
        }
        return taskStatus.withDuration(duration);
    }

    protected void startWatchingLogs() {
        if (this.logWatch != null) {
            log.debug("There is already a log watcher for %s", new Object[]{this.taskId.getOriginalTaskId()});
            return;
        }
        Optional maybeLogWatch = this.executeWithTimeout(() -> this.kubernetesClient.getPeonLogWatcher(this.taskId), this.logSaveTimeoutMs, "initializing K8s LogWatch", "LogWatch failed to initialize. Peon may not be able to stream and persist task logs.  If this continues to happen, check Kubernetes server logs for potential errors.");
        if (maybeLogWatch != null && maybeLogWatch.isPresent()) {
            this.logWatch = (LogWatch)maybeLogWatch.get();
        }
    }

    protected void saveLogs() {
        try {
            Path file = Files.createTempFile(this.taskId.getOriginalTaskId(), "log", new FileAttribute[0]);
            try {
                this.startWatchingLogs();
                if (this.logWatch != null) {
                    this.executeWithTimeout(() -> {
                        FileUtils.copyInputStreamToFile((InputStream)this.logWatch.getOutput(), (File)file.toFile());
                        return null;
                    }, this.logSaveTimeoutMs, "coyping and persisting task logs", "This failure does not have any impact on the ingestion work done by the task, but the logs may be partial or innaccessible. If  this continues to happen, check Kubernetes server logs for potential errors.");
                } else {
                    log.debug("Log stream not found for %s", new Object[]{this.taskId.getOriginalTaskId()});
                    FileUtils.writeStringToFile((File)file.toFile(), (String)StringUtils.format((String)"Peon for task [%s] did not report any logs. Check k8s metrics and events for the pod to see what happened.", (Object[])new Object[]{this.taskId}), (Charset)Charset.defaultCharset());
                }
                this.taskLogs.pushTaskLog(this.taskId.getOriginalTaskId(), file.toFile());
            }
            catch (IOException e) {
                log.error((Throwable)e, "Failed to stream logs for task [%s]", new Object[]{this.taskId.getOriginalTaskId()});
            }
            finally {
                if (this.logWatch != null) {
                    this.logWatch.close();
                }
                Files.deleteIfExists(file);
            }
        }
        catch (IOException e) {
            log.warn((Throwable)e, "Failed to manage temporary log file for task [%s]", new Object[]{this.taskId.getOriginalTaskId()});
        }
    }

    private void stopTask() {
        if (!State.STOPPED.equals((Object)this.state.get())) {
            this.updateState(new State[]{State.NOT_STARTED, State.PENDING, State.RUNNING}, State.STOPPED);
        }
    }

    private void updateState(State[] acceptedStates, State targetState) {
        Preconditions.checkState((boolean)Arrays.stream(acceptedStates).anyMatch(s -> this.state.compareAndSet((State)((Object)s), targetState)), (String)"Task [%s] failed to run: invalid peon lifecycle state transition [%s]->[%s]", (Object)this.taskId.getOriginalTaskId(), (Object)((Object)this.state.get()), (Object)((Object)targetState));
        this.stateListener.stateChanged(this.state.get(), this.taskId.getOriginalTaskId());
    }

    protected ListenableFuture<Boolean> getTaskStartedSuccessfullyFuture() {
        return this.taskStartedSuccessfullyFuture;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    private <T> T executeWithTimeout(Callable<T> callable, long timeoutMillis, String operationName, String errorMessage) {
        ExecutorService executor = Executors.newSingleThreadExecutor(Execs.makeThreadFactory((String)("k8s-peon-lifecycle-util-" + this.taskId.getOriginalTaskId() + "-%d")));
        try {
            Future<T> future = executor.submit(callable);
            T t = future.get(timeoutMillis, TimeUnit.MILLISECONDS);
            return t;
        }
        catch (TimeoutException e) {
            log.warn("Operation[%s] for task[%s] timed out after [%d] ms with error[%s].", new Object[]{operationName, this.taskId.getOriginalTaskId(), timeoutMillis, errorMessage});
        }
        catch (InterruptedException e) {
            log.warn("Operation[%s] for task[%s] was interrupted with error[%s].", new Object[]{operationName, this.taskId.getOriginalTaskId(), errorMessage});
        }
        catch (Exception e) {
            log.error((Throwable)e, "Error during operation[%s] for task[%s]: %s", new Object[]{operationName, this.taskId, errorMessage});
        }
        finally {
            executor.shutdownNow();
        }
        return null;
    }

    protected static enum State {
        NOT_STARTED,
        PENDING,
        RUNNING,
        STOPPED;

    }

    @FunctionalInterface
    public static interface TaskStateListener {
        public void stateChanged(State var1, String var2);
    }
}

