/*
 * Copyright 2015 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.pedjak.gradle.plugins.dockerizedtest

import com.github.dockerjava.api.DockerClient
import com.github.dockerjava.core.DefaultDockerClientConfig
import com.github.dockerjava.core.DockerClientBuilder
import com.github.dockerjava.netty.NettyDockerCmdExecFactory
import org.apache.commons.lang3.SystemUtils
import org.apache.maven.artifact.versioning.ComparableVersion
import org.gradle.api.Action
import org.gradle.api.GradleException
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.internal.file.DefaultFileCollectionFactory
import org.gradle.api.tasks.testing.Test
import org.gradle.initialization.DefaultBuildCancellationToken
import org.gradle.internal.concurrent.DefaultExecutorFactory
import org.gradle.internal.concurrent.ExecutorFactory
import org.gradle.internal.operations.BuildOperationExecutor
import org.gradle.internal.remote.Address
import org.gradle.internal.remote.ConnectionAcceptor
import org.gradle.internal.remote.MessagingServer
import org.gradle.internal.remote.ObjectConnection
import org.gradle.internal.remote.internal.ConnectCompletion
import org.gradle.internal.remote.internal.IncomingConnector
import org.gradle.internal.remote.internal.hub.MessageHubBackedObjectConnection
import org.gradle.internal.remote.internal.inet.MultiChoiceAddress
import org.gradle.internal.time.Clock
import org.gradle.process.internal.JavaExecHandleFactory
import org.gradle.process.internal.worker.DefaultWorkerProcessFactory

import javax.inject.Inject

class DockerizedTestPlugin implements Plugin<Project> {

    def supportedVersion = '4.8'
    def currentUser
    def messagingServer
    def static workerSemaphore = new DefaultWorkerSemaphore()
    def memoryManager = new com.pedjak.gradle.plugins.dockerizedtest.NoMemoryManager()

    @Inject
    DockerizedTestPlugin(MessagingServer messagingServer) {
        this.currentUser = SystemUtils.IS_OS_WINDOWS ? "0" : "id -u".execute().text.trim()
        this.messagingServer = new MessageServer(messagingServer.connector, messagingServer.executorFactory)
    }

    void configureTest(project, test) {
        def ext = test.extensions.create("docker", DockerizedTestExtension, [] as Object[])
        def startParameter = project.gradle.startParameter
        ext.volumes = ["$startParameter.gradleUserHomeDir": "$startParameter.gradleUserHomeDir",
                       "$project.projectDir"              : "$project.projectDir"]
        ext.user = currentUser
        test.doFirst {
            def extension = test.extensions.docker

            if (extension?.image) {

                workerSemaphore.applyTo(test.project)
                test.testExecuter = new com.pedjak.gradle.plugins.dockerizedtest.TestExecuter(newProcessBuilderFactory(project, extension, test.processBuilderFactory), actorFactory, moduleRegistry, services.get(BuildOperationExecutor), services.get(Clock));

                if (!extension.client) {
                    extension.client = createDefaultClient()
                }
            }

        }
    }

    DockerClient createDefaultClient() {
        DockerClientBuilder.getInstance(DefaultDockerClientConfig.createDefaultConfigBuilder())
                .withDockerCmdExecFactory(new NettyDockerCmdExecFactory())
                .build()
    }

    void apply(Project project) {

        boolean unsupportedVersion = new ComparableVersion(project.gradle.gradleVersion).compareTo(new ComparableVersion(supportedVersion)) < 0
        if (unsupportedVersion) throw new GradleException("dockerized-test plugin requires Gradle ${supportedVersion}+")

        project.tasks.withType(Test).each { test -> configureTest(project, test) }
        project.tasks.whenTaskAdded { task ->
            if (task instanceof Test) configureTest(project, task)
        }
    }

    def newProcessBuilderFactory(project, extension, defaultProcessBuilderFactory) {

        def executorFactory = new DefaultExecutorFactory()
        def executor = executorFactory.create("Docker container link")
        def buildCancellationToken = new DefaultBuildCancellationToken()

        def defaultfilecollectionFactory = new DefaultFileCollectionFactory(project.fileResolver, null)
        def execHandleFactory = [newJavaExec: { ->
            new DockerizedJavaExecHandleBuilder(
              extension, project.fileResolver, defaultfilecollectionFactory,
              executor, buildCancellationToken, workerSemaphore)
        }] as JavaExecHandleFactory
        new DefaultWorkerProcessFactory(defaultProcessBuilderFactory.loggingManager,
                messagingServer,
                defaultProcessBuilderFactory.workerImplementationFactory.classPathRegistry,
                defaultProcessBuilderFactory.idGenerator,
                defaultProcessBuilderFactory.gradleUserHomeDir,
                defaultProcessBuilderFactory.workerImplementationFactory.temporaryFileProvider,
                execHandleFactory,
                defaultProcessBuilderFactory.workerImplementationFactory.jvmVersionDetector,
                defaultProcessBuilderFactory.outputEventListener,
                memoryManager
        )
    }

    class MessageServer implements MessagingServer {
        def IncomingConnector connector;
        def ExecutorFactory executorFactory;

        public MessageServer(IncomingConnector connector, ExecutorFactory executorFactory) {
            this.connector = connector;
            this.executorFactory = executorFactory;
        }

        public ConnectionAcceptor accept(Action<ObjectConnection> action) {
            return new ConnectionAcceptorDelegate(connector.accept(new ConnectEventAction(action, executorFactory), true))
        }


    }

    class ConnectEventAction implements Action<ConnectCompletion> {
        def action;
        def executorFactory;

        public ConnectEventAction(Action<ObjectConnection> action, executorFactory) {
            this.executorFactory = executorFactory
            this.action = action
        }

        public void execute(ConnectCompletion completion) {
            action.execute(new MessageHubBackedObjectConnection(executorFactory, completion));
        }
    }

    class ConnectionAcceptorDelegate implements ConnectionAcceptor {

        MultiChoiceAddress address

        @Delegate
        ConnectionAcceptor delegate

        ConnectionAcceptorDelegate(ConnectionAcceptor delegate) {
            this.delegate = delegate
        }

        Address getAddress() {
            synchronized (delegate)
            {
                if (address == null) {
                    def remoteAddresses = NetworkInterface.networkInterfaces.findAll {
                        try {
                            return it.up && !it.loopback
                        } catch (SocketException ex) {
                            logger.warn("Unable to inspect interface " + it)
                            return false
                        }
                    }*.inetAddresses*.collect { it }.flatten()
                    def original = delegate.address
                    address = new MultiChoiceAddress(original.canonicalAddress, original.port, remoteAddresses)
                }
            }
            address
        }
    }

}
