/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.task.assigner;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import org.apache.helix.model.ClusterConfig;
import org.apache.helix.model.InstanceConfig;
import org.apache.helix.model.LiveInstance;
import org.apache.helix.task.TaskConfig;
import org.apache.helix.task.assigner.TaskAssignResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AssignableInstance {
    private static final Logger logger = LoggerFactory.getLogger(AssignableInstance.class);
    public static final String DEFAULT_QUOTA_TYPE = "DEFAULT";
    private static final int fitnessScoreFactor = 1000;
    private Set<String> _currentAssignments;
    private ClusterConfig _clusterConfig;
    private InstanceConfig _instanceConfig;
    private LiveInstance _liveInstance;
    private Map<String, Map<String, Integer>> _totalCapacity;
    private Map<String, Map<String, Integer>> _usedCapacity;

    public AssignableInstance(ClusterConfig clusterConfig, InstanceConfig instanceConfig, LiveInstance liveInstance) {
        if (clusterConfig == null || instanceConfig == null || liveInstance == null) {
            throw new IllegalArgumentException(String.format("ClusterConfig, InstanceConfig, LiveInstance cannot be null! ClusterConfig null: %s, InstanceConfig null: %s, LiveInstance null: %s", clusterConfig == null, instanceConfig == null, liveInstance == null));
        }
        if (!instanceConfig.getInstanceName().equals(liveInstance.getInstanceName())) {
            throw new IllegalArgumentException(String.format("Instance name from LiveInstance (%s) and InstanceConfig (%s) don't match!", liveInstance.getInstanceName(), instanceConfig.getInstanceName()));
        }
        this._clusterConfig = clusterConfig;
        this._instanceConfig = instanceConfig;
        this._liveInstance = liveInstance;
        this._currentAssignments = new HashSet<String>();
        this._totalCapacity = new HashMap<String, Map<String, Integer>>();
        this._usedCapacity = new HashMap<String, Map<String, Integer>>();
        this.refreshTotalCapacity();
    }

    private void refreshTotalCapacity() {
        HashMap<String, Map<String, Integer>> tempTotalCapacity = new HashMap<String, Map<String, Integer>>();
        Map<String, String> typeQuotaRatio = this._clusterConfig.getTaskQuotaRatioMap();
        Map<String, String> resourceCapacity = this._liveInstance.getResourceCapacityMap();
        if (resourceCapacity == null) {
            resourceCapacity = new HashMap<String, String>();
            resourceCapacity.put(LiveInstance.InstanceResourceType.TASK_EXEC_THREAD.name(), Integer.toString(this._liveInstance.getCurrentTaskThreadPoolSize()));
            logger.debug("No resource capacity provided in LiveInstance {}, assuming default capacity: {}", (Object)this._instanceConfig.getInstanceName(), resourceCapacity);
        }
        if (typeQuotaRatio == null) {
            typeQuotaRatio = new HashMap<String, String>();
            typeQuotaRatio.put(DEFAULT_QUOTA_TYPE, Integer.toString(1));
            logger.debug("No quota type ratio provided in LiveInstance {}, assuming default ratio: {}", (Object)this._instanceConfig.getInstanceName(), typeQuotaRatio);
        }
        logger.debug("Updating capacity for AssignableInstance {}. Resource Capacity: {}; Type Quota Ratio: {}", new Object[]{this._instanceConfig.getInstanceName(), resourceCapacity, typeQuotaRatio});
        try {
            for (Map.Entry<String, String> resEntry : resourceCapacity.entrySet()) {
                String resourceType = resEntry.getKey();
                int capacity = Integer.valueOf(resEntry.getValue());
                if (!this._totalCapacity.containsKey(resourceType)) {
                    logger.debug("Adding InstanceResourceType {}", (Object)resourceType);
                    this._usedCapacity.put(resourceType, new HashMap());
                }
                tempTotalCapacity.put(resourceType, new HashMap());
                int totalRatio = 0;
                for (String string : typeQuotaRatio.values()) {
                    totalRatio += Integer.valueOf(string).intValue();
                }
                for (Map.Entry entry : typeQuotaRatio.entrySet()) {
                    String quotaType = (String)entry.getKey();
                    int quotaRatio = Integer.valueOf((String)entry.getValue());
                    int quota = Math.round((float)capacity * (float)quotaRatio / (float)totalRatio);
                    if (capacity != 0 && quotaRatio != 0 && quota == 0) {
                        quota = 1;
                    }
                    ((Map)tempTotalCapacity.get(resourceType)).put(quotaType, quota);
                    if (this._usedCapacity.get(resourceType).containsKey(quotaType)) continue;
                    logger.debug("Adding QuotaType {} for resource {}", (Object)quotaType, (Object)resourceType);
                    this._usedCapacity.get(resourceType).put(quotaType, 0);
                }
                this._usedCapacity.get(resourceType).keySet().retainAll(typeQuotaRatio.keySet());
            }
            this._totalCapacity = tempTotalCapacity;
            this._usedCapacity.keySet().retainAll(resourceCapacity.keySet());
            logger.debug("Finished updating capacity for AssignableInstance {}. Current capacity {}. Current usage: {}", new Object[]{this._instanceConfig.getInstanceName(), this._totalCapacity, this._usedCapacity});
        }
        catch (Exception e) {
            logger.error("Failed to update capacity for AssignableInstance {}, still using current capacity {}. Current usage: {}", new Object[]{this._instanceConfig.getInstanceName(), this._totalCapacity, this._usedCapacity, e});
        }
    }

    public void updateConfigs(ClusterConfig clusterConfig, InstanceConfig instanceConfig, LiveInstance liveInstance) {
        logger.debug("Updating configs for AssignableInstance {}", (Object)this._instanceConfig.getInstanceName());
        boolean refreshCapacity = false;
        if (clusterConfig != null && clusterConfig.getTaskQuotaRatioMap() != null) {
            if (!clusterConfig.getTaskQuotaRatioMap().equals(this._clusterConfig.getTaskQuotaRatioMap())) {
                refreshCapacity = true;
            }
            this._clusterConfig = clusterConfig;
            logger.debug("Updated cluster config");
        }
        if (liveInstance != null) {
            if (!this._instanceConfig.getInstanceName().equals(liveInstance.getInstanceName())) {
                logger.error("Cannot update live instance with different instance name. Current: {}; new: {}", (Object)this._instanceConfig.getInstanceName(), (Object)liveInstance.getInstanceName());
            } else {
                if (liveInstance.getResourceCapacityMap() != null && !liveInstance.getResourceCapacityMap().equals(this._liveInstance.getResourceCapacityMap())) {
                    refreshCapacity = true;
                }
                this._liveInstance = liveInstance;
                logger.debug("Updated live instance");
            }
        }
        if (instanceConfig != null) {
            if (!this._instanceConfig.getInstanceName().equals(instanceConfig.getInstanceName())) {
                logger.error("Cannot update instance config with different instance name. Current: {}; new: {}", (Object)this._instanceConfig.getInstanceName(), (Object)instanceConfig.getInstanceName());
            } else {
                this._instanceConfig = instanceConfig;
                logger.debug("Updated instance config");
            }
        }
        if (refreshCapacity) {
            this.refreshTotalCapacity();
        }
        logger.debug("Updated configs for AssignableInstance {}", (Object)this._instanceConfig.getInstanceName());
    }

    public synchronized TaskAssignResult tryAssign(TaskConfig task, String quotaType) throws IllegalArgumentException {
        int usage;
        int capacity;
        if (task == null) {
            throw new IllegalArgumentException("Task is null!");
        }
        if (this._currentAssignments.contains(task.getId())) {
            logger.debug("Task: {} of quotaType: {} is already assigned to this instance. Instance name: {}", new Object[]{task.getId(), quotaType, this.getInstanceName()});
            return new TaskAssignResult(task, quotaType, this, false, 0, TaskAssignResult.FailureReason.TASK_ALREADY_ASSIGNED, String.format("Task %s is already assigned to this instance. Need to release it first", task.getId()));
        }
        String resourceType = LiveInstance.InstanceResourceType.TASK_EXEC_THREAD.name();
        if (!this._totalCapacity.containsKey(resourceType)) {
            logger.debug("AssignableInstance does not support the given resourceType: {}. Task: {}, quotaType: {}, Instance name: {}", new Object[]{resourceType, task.getId(), quotaType, this.getInstanceName()});
            return new TaskAssignResult(task, quotaType, this, false, 0, TaskAssignResult.FailureReason.NO_SUCH_RESOURCE_TYPE, String.format("Requested resource type %s not supported. Available resource types: %s", resourceType, this._totalCapacity.keySet()));
        }
        if (quotaType == null || quotaType.equals("")) {
            quotaType = DEFAULT_QUOTA_TYPE;
        }
        if (!this._totalCapacity.get(resourceType).containsKey(quotaType)) {
            logger.debug("AssignableInstance does not support the given quotaType: {}. Task: {}, quotaType: {}, Instance name: {}. Task will be assigned as DEFAULT type.", new Object[]{quotaType, task.getId(), quotaType, this.getInstanceName()});
            quotaType = DEFAULT_QUOTA_TYPE;
        }
        if ((capacity = this._totalCapacity.get(resourceType).get(quotaType).intValue()) <= (usage = this._usedCapacity.get(resourceType).get(quotaType).intValue())) {
            logger.debug("AssignableInstance does not have enough capacity for quotaType: {}. Task: {}, quotaType: {}, Instance name: {}. Total capacity: {} Current usage: {}", new Object[]{quotaType, task.getId(), quotaType, this.getInstanceName(), capacity, usage});
            return new TaskAssignResult(task, quotaType, this, false, 0, TaskAssignResult.FailureReason.INSUFFICIENT_QUOTA, String.format("Insufficient quota %s::%s. Capacity: %s, Current Usage: %s", resourceType, quotaType, capacity, usage));
        }
        int fitness = Math.round((float)(capacity - usage) / (float)capacity * 1000.0f);
        return new TaskAssignResult(task, quotaType, this, true, fitness, null, "");
    }

    public synchronized void assign(TaskAssignResult result) throws IllegalStateException {
        if (!result.isSuccessful()) {
            throw new IllegalStateException("Cannot assign a failed result: " + result);
        }
        if (!result.getInstanceName().equals(this.getInstanceName())) {
            throw new IllegalStateException(String.format("Cannot assign a result for a different instance. This instance: %s; Result: %s", this.getInstanceName(), result));
        }
        if (this._currentAssignments.contains(result.getTaskConfig().getId())) {
            throw new IllegalStateException("Cannot double assign task " + result.getTaskConfig().getId());
        }
        this._currentAssignments.add(result.getTaskConfig().getId());
        String resourceType = LiveInstance.InstanceResourceType.TASK_EXEC_THREAD.name();
        String quotaType = result.getQuotaType();
        if (this._usedCapacity.containsKey(resourceType)) {
            if (this._usedCapacity.get(resourceType).containsKey(quotaType)) {
                int curUsage = this._usedCapacity.get(resourceType).get(quotaType);
                this._usedCapacity.get(resourceType).put(quotaType, curUsage + 1);
            } else {
                int curUsage = this._usedCapacity.get(resourceType).get(DEFAULT_QUOTA_TYPE);
                this._usedCapacity.get(resourceType).put(DEFAULT_QUOTA_TYPE, curUsage + 1);
            }
        } else {
            logger.debug("Task's requested resource type is not supported. TaskConfig: %s; UsedCapacity: %s; ResourceType: %s", new Object[]{result.getTaskConfig(), this._usedCapacity, resourceType});
        }
        logger.debug("Assigned task {} to instance {}", (Object)result.getTaskConfig().getId(), (Object)this._instanceConfig.getInstanceName());
    }

    public synchronized void release(TaskConfig taskConfig, String quotaType) {
        if (!this._currentAssignments.contains(taskConfig.getId())) {
            logger.debug("Task {} is not assigned on instance {}", (Object)taskConfig.getId(), (Object)this._instanceConfig.getInstanceName());
            return;
        }
        String resourceType = LiveInstance.InstanceResourceType.TASK_EXEC_THREAD.name();
        if (this._usedCapacity.containsKey(resourceType)) {
            if (this._usedCapacity.get(resourceType).containsKey(quotaType)) {
                int curUsage = this._usedCapacity.get(resourceType).get(quotaType);
                this._usedCapacity.get(resourceType).put(quotaType, curUsage - 1);
            } else {
                int curUsage = this._usedCapacity.get(resourceType).get(DEFAULT_QUOTA_TYPE);
                this._usedCapacity.get(resourceType).put(DEFAULT_QUOTA_TYPE, curUsage - 1);
            }
        }
        this._currentAssignments.remove(taskConfig.getId());
        logger.debug("Released task {} from instance {}", (Object)taskConfig.getId(), (Object)this._instanceConfig.getInstanceName());
    }

    public TaskAssignResult restoreTaskAssignResult(String taskId, TaskConfig taskConfig, String quotaType) {
        TaskAssignResult assignResult = new TaskAssignResult(taskConfig, quotaType, this, true, 1000, null, "Recovered TaskAssignResult from current state");
        try {
            this.assign(assignResult);
        }
        catch (IllegalStateException e) {
            logger.error("Failed to restore current TaskAssignResult for task {}.", (Object)taskId, (Object)e);
            return new TaskAssignResult(taskConfig, quotaType, this, false, 1000, null, "Recovered TaskAssignResult from current state");
        }
        return assignResult;
    }

    public Set<String> getCurrentAssignments() {
        return this._currentAssignments;
    }

    public String getInstanceName() {
        return this._instanceConfig.getInstanceName();
    }

    public Map<String, Map<String, Integer>> getTotalCapacity() {
        return this._totalCapacity;
    }

    public Map<String, Map<String, Integer>> getUsedCapacity() {
        return this._usedCapacity;
    }
}

