/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.spark.sql

import scala.reflect.runtime.universe.{typeTag, TypeTag}

import org.apache.spark.internal.Logging
import org.apache.spark.sql.api.java._
import org.apache.spark.sql.connect.common.UdfUtils
import org.apache.spark.sql.expressions.{ScalarUserDefinedFunction, UserDefinedFunction}
import org.apache.spark.sql.types.DataType

/**
 * Functions for registering user-defined functions. Use `SparkSession.udf` to access this:
 *
 * {{{
 *   spark.udf
 * }}}
 *
 * @since 3.5.0
 */
class UDFRegistration(session: SparkSession) extends Logging {

  /**
   * Registers a user-defined function (UDF), for a UDF that's already defined using the Dataset
   * API (i.e. of type UserDefinedFunction). To change a UDF to nondeterministic, call the API
   * `UserDefinedFunction.asNondeterministic()`. To change a UDF to nonNullable, call the API
   * `UserDefinedFunction.asNonNullable()`.
   *
   * Example:
   * {{{
   *   val foo = udf(() => Math.random())
   *   spark.udf.register("random", foo.asNondeterministic())
   *
   *   val bar = udf(() => "bar")
   *   spark.udf.register("stringLit", bar.asNonNullable())
   * }}}
   *
   * @param name
   *   the name of the UDF.
   * @param udf
   *   the UDF needs to be registered.
   * @return
   *   the registered UDF.
   *
   * @since 3.5.0
   */
  def register(name: String, udf: UserDefinedFunction): UserDefinedFunction = {
    udf.withName(name) match {
      case scalarUdf: ScalarUserDefinedFunction =>
        session.registerUdf(scalarUdf.toProto)
        scalarUdf
      case other =>
        throw new UnsupportedOperationException(
          s"Registering a UDF of type " +
            s"${other.getClass.getSimpleName} is currently unsupported.")
    }
  }

  // scalastyle:off line.size.limit

  /* register 0-22 were generated by this script:
    (0 to 22).foreach { x =>
      val params = (1 to x).map(num => s"A$num").mkString(", ")
      val typeTags = (1 to x).map(i => s"A$i: TypeTag").foldLeft("RT: TypeTag")(_ + ", " + _)
      println(s"""
        |/**
        | * Registers a deterministic Scala closure of $x arguments as user-defined function (UDF).
        | * @tparam RT return type of UDF.
        | * @since 3.5.0
        | */
        |def register[$typeTags](name: String, func: ($params) => RT): UserDefinedFunction = {
        |  register(name, functions.udf(func))
        |}""".stripMargin)
    }
   */

  /**
   * Registers a deterministic Scala closure of 0 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[RT: TypeTag](name: String, func: () => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(func, typeTag[RT])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 1 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[RT: TypeTag, A1: TypeTag](name: String, func: (A1) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(func, typeTag[RT], typeTag[A1])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 2 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[RT: TypeTag, A1: TypeTag, A2: TypeTag](
      name: String,
      func: (A1, A2) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(func, typeTag[RT], typeTag[A1], typeTag[A2])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 3 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[RT: TypeTag, A1: TypeTag, A2: TypeTag, A3: TypeTag](
      name: String,
      func: (A1, A2, A3) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(func, typeTag[RT], typeTag[A1], typeTag[A2], typeTag[A3])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 4 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[RT: TypeTag, A1: TypeTag, A2: TypeTag, A3: TypeTag, A4: TypeTag](
      name: String,
      func: (A1, A2, A3, A4) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 5 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[RT: TypeTag, A1: TypeTag, A2: TypeTag, A3: TypeTag, A4: TypeTag, A5: TypeTag](
      name: String,
      func: (A1, A2, A3, A4, A5) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 6 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag](name: String, func: (A1, A2, A3, A4, A5, A6) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 7 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag](
      name: String,
      func: (A1, A2, A3, A4, A5, A6, A7) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 8 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag](
      name: String,
      func: (A1, A2, A3, A4, A5, A6, A7, A8) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 9 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag](
      name: String,
      func: (A1, A2, A3, A4, A5, A6, A7, A8, A9) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 10 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag,
      A10: TypeTag](
      name: String,
      func: (A1, A2, A3, A4, A5, A6, A7, A8, A9, A10) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9],
      typeTag[A10])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 11 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag,
      A10: TypeTag,
      A11: TypeTag](
      name: String,
      func: (A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9],
      typeTag[A10],
      typeTag[A11])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 12 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag,
      A10: TypeTag,
      A11: TypeTag,
      A12: TypeTag](
      name: String,
      func: (A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9],
      typeTag[A10],
      typeTag[A11],
      typeTag[A12])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 13 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag,
      A10: TypeTag,
      A11: TypeTag,
      A12: TypeTag,
      A13: TypeTag](
      name: String,
      func: (A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13) => RT)
      : UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9],
      typeTag[A10],
      typeTag[A11],
      typeTag[A12],
      typeTag[A13])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 14 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag,
      A10: TypeTag,
      A11: TypeTag,
      A12: TypeTag,
      A13: TypeTag,
      A14: TypeTag](
      name: String,
      func: (A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14) => RT)
      : UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9],
      typeTag[A10],
      typeTag[A11],
      typeTag[A12],
      typeTag[A13],
      typeTag[A14])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 15 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag,
      A10: TypeTag,
      A11: TypeTag,
      A12: TypeTag,
      A13: TypeTag,
      A14: TypeTag,
      A15: TypeTag](
      name: String,
      func: (A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15) => RT)
      : UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9],
      typeTag[A10],
      typeTag[A11],
      typeTag[A12],
      typeTag[A13],
      typeTag[A14],
      typeTag[A15])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 16 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag,
      A10: TypeTag,
      A11: TypeTag,
      A12: TypeTag,
      A13: TypeTag,
      A14: TypeTag,
      A15: TypeTag,
      A16: TypeTag](
      name: String,
      func: (A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15, A16) => RT)
      : UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9],
      typeTag[A10],
      typeTag[A11],
      typeTag[A12],
      typeTag[A13],
      typeTag[A14],
      typeTag[A15],
      typeTag[A16])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 17 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag,
      A10: TypeTag,
      A11: TypeTag,
      A12: TypeTag,
      A13: TypeTag,
      A14: TypeTag,
      A15: TypeTag,
      A16: TypeTag,
      A17: TypeTag](
      name: String,
      func: (A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15, A16, A17) => RT)
      : UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9],
      typeTag[A10],
      typeTag[A11],
      typeTag[A12],
      typeTag[A13],
      typeTag[A14],
      typeTag[A15],
      typeTag[A16],
      typeTag[A17])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 18 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag,
      A10: TypeTag,
      A11: TypeTag,
      A12: TypeTag,
      A13: TypeTag,
      A14: TypeTag,
      A15: TypeTag,
      A16: TypeTag,
      A17: TypeTag,
      A18: TypeTag](
      name: String,
      func: (
          A1,
          A2,
          A3,
          A4,
          A5,
          A6,
          A7,
          A8,
          A9,
          A10,
          A11,
          A12,
          A13,
          A14,
          A15,
          A16,
          A17,
          A18) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9],
      typeTag[A10],
      typeTag[A11],
      typeTag[A12],
      typeTag[A13],
      typeTag[A14],
      typeTag[A15],
      typeTag[A16],
      typeTag[A17],
      typeTag[A18])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 19 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag,
      A10: TypeTag,
      A11: TypeTag,
      A12: TypeTag,
      A13: TypeTag,
      A14: TypeTag,
      A15: TypeTag,
      A16: TypeTag,
      A17: TypeTag,
      A18: TypeTag,
      A19: TypeTag](
      name: String,
      func: (
          A1,
          A2,
          A3,
          A4,
          A5,
          A6,
          A7,
          A8,
          A9,
          A10,
          A11,
          A12,
          A13,
          A14,
          A15,
          A16,
          A17,
          A18,
          A19) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9],
      typeTag[A10],
      typeTag[A11],
      typeTag[A12],
      typeTag[A13],
      typeTag[A14],
      typeTag[A15],
      typeTag[A16],
      typeTag[A17],
      typeTag[A18],
      typeTag[A19])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 20 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag,
      A10: TypeTag,
      A11: TypeTag,
      A12: TypeTag,
      A13: TypeTag,
      A14: TypeTag,
      A15: TypeTag,
      A16: TypeTag,
      A17: TypeTag,
      A18: TypeTag,
      A19: TypeTag,
      A20: TypeTag](
      name: String,
      func: (
          A1,
          A2,
          A3,
          A4,
          A5,
          A6,
          A7,
          A8,
          A9,
          A10,
          A11,
          A12,
          A13,
          A14,
          A15,
          A16,
          A17,
          A18,
          A19,
          A20) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9],
      typeTag[A10],
      typeTag[A11],
      typeTag[A12],
      typeTag[A13],
      typeTag[A14],
      typeTag[A15],
      typeTag[A16],
      typeTag[A17],
      typeTag[A18],
      typeTag[A19],
      typeTag[A20])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 21 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag,
      A10: TypeTag,
      A11: TypeTag,
      A12: TypeTag,
      A13: TypeTag,
      A14: TypeTag,
      A15: TypeTag,
      A16: TypeTag,
      A17: TypeTag,
      A18: TypeTag,
      A19: TypeTag,
      A20: TypeTag,
      A21: TypeTag](
      name: String,
      func: (
          A1,
          A2,
          A3,
          A4,
          A5,
          A6,
          A7,
          A8,
          A9,
          A10,
          A11,
          A12,
          A13,
          A14,
          A15,
          A16,
          A17,
          A18,
          A19,
          A20,
          A21) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9],
      typeTag[A10],
      typeTag[A11],
      typeTag[A12],
      typeTag[A13],
      typeTag[A14],
      typeTag[A15],
      typeTag[A16],
      typeTag[A17],
      typeTag[A18],
      typeTag[A19],
      typeTag[A20],
      typeTag[A21])
    register(name, udf)
  }

  /**
   * Registers a deterministic Scala closure of 22 arguments as user-defined function (UDF).
   * @tparam RT
   *   return type of UDF.
   * @since 3.5.0
   */
  def register[
      RT: TypeTag,
      A1: TypeTag,
      A2: TypeTag,
      A3: TypeTag,
      A4: TypeTag,
      A5: TypeTag,
      A6: TypeTag,
      A7: TypeTag,
      A8: TypeTag,
      A9: TypeTag,
      A10: TypeTag,
      A11: TypeTag,
      A12: TypeTag,
      A13: TypeTag,
      A14: TypeTag,
      A15: TypeTag,
      A16: TypeTag,
      A17: TypeTag,
      A18: TypeTag,
      A19: TypeTag,
      A20: TypeTag,
      A21: TypeTag,
      A22: TypeTag](
      name: String,
      func: (
          A1,
          A2,
          A3,
          A4,
          A5,
          A6,
          A7,
          A8,
          A9,
          A10,
          A11,
          A12,
          A13,
          A14,
          A15,
          A16,
          A17,
          A18,
          A19,
          A20,
          A21,
          A22) => RT): UserDefinedFunction = {
    val udf = ScalarUserDefinedFunction(
      func,
      typeTag[RT],
      typeTag[A1],
      typeTag[A2],
      typeTag[A3],
      typeTag[A4],
      typeTag[A5],
      typeTag[A6],
      typeTag[A7],
      typeTag[A8],
      typeTag[A9],
      typeTag[A10],
      typeTag[A11],
      typeTag[A12],
      typeTag[A13],
      typeTag[A14],
      typeTag[A15],
      typeTag[A16],
      typeTag[A17],
      typeTag[A18],
      typeTag[A19],
      typeTag[A20],
      typeTag[A21],
      typeTag[A22])
    register(name, udf)
  }

  //  (0 to 22).foreach { i =>
  //    val extTypeArgs = (0 to i).map(_ => "_").mkString(", ")
  //    val version = "3.5.0"
  //    println(s"""
  //               |/**
  //               | * Register a deterministic Java UDF$i instance as user-defined function (UDF).
  //               | * @since $version
  //               | */
  //               |def register(name: String, f: UDF$i[$extTypeArgs], returnType: DataType): Unit = {
  //               |  val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
  //               |  register(name, udf)
  //               |}""".stripMargin)
  //  }

  /**
   * Register a deterministic Java UDF0 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(name: String, f: UDF0[_], returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF1 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(name: String, f: UDF1[_, _], returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF2 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(name: String, f: UDF2[_, _, _], returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF3 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(name: String, f: UDF3[_, _, _, _], returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF4 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(name: String, f: UDF4[_, _, _, _, _], returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF5 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(name: String, f: UDF5[_, _, _, _, _, _], returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF6 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(name: String, f: UDF6[_, _, _, _, _, _, _], returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF7 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(name: String, f: UDF7[_, _, _, _, _, _, _, _], returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF8 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(name: String, f: UDF8[_, _, _, _, _, _, _, _, _], returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF9 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF9[_, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF10 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF10[_, _, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF11 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF11[_, _, _, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF12 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF12[_, _, _, _, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF13 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF13[_, _, _, _, _, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF14 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF14[_, _, _, _, _, _, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF15 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF15[_, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF16 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF16[_, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF17 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF17[_, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF18 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF18[_, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF19 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF19[_, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF20 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF20[_, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF21 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF21[_, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }

  /**
   * Register a deterministic Java UDF22 instance as user-defined function (UDF).
   * @since 3.5.0
   */
  def register(
      name: String,
      f: UDF22[_, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _, _],
      returnType: DataType): Unit = {
    val udf = ScalarUserDefinedFunction(UdfUtils.wrap(f), returnType)
    register(name, udf)
  }
  // scalastyle:on line.size.limit
}
