# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

import re

import pytest

import tvm
from tvm import te
from tvm.script import tir as T
from tvm.topi.arm_cpu.pstate_attributes import SMEAttributes

from tvm.target.codegen import llvm_version_major


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.parametrize(
    "dtype",
    ["float", "float16", "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"],
)
def test_mul(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        B = te.placeholder(m, dtype=type, name="B")
        C = te.compute((m), lambda i: A[i] * B[i], name="C")
        s = te.create_schedule([C.op])

        f = tvm.build(s, [A, B, C], target)

        # Verify we see SVE load instructions and mul instructions using z registers
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)
        matches = re.findall(
            r"mul\tz[0-9].[shdb],( p[0-9]/[m],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )

        assert len(loads) > 1
        assert len(matches) > 1

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.parametrize(
    "dtype",
    ["float", "float16", "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"],
)
def test_add(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        B = te.placeholder(m, dtype=type, name="B")
        C = te.compute((m), lambda i: A[i] + B[i], name="C")
        s = te.create_schedule([C.op])

        f = tvm.build(s, [A, B, C], target)

        # Verify we see SVE load instructions and add instructions using z registers
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)
        matches = re.findall(
            r"add\tz[0-9].[shdb],( p[0-9]/[m],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )

        assert len(loads) > 1
        assert len(matches) > 1

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.parametrize(
    "dtype",
    ["float", "float16", "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"],
)
def test_sub(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        B = te.placeholder(m, dtype=type, name="B")
        C = te.compute((m), lambda i: A[i] - B[i], name="C")
        s = te.create_schedule([C.op])

        f = tvm.build(s, [A, B, C], target)

        # Verify we see SVE load instructions and sub instructions using z registers
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)
        matches = re.findall(
            r"sub\tz[0-9].[shdb],( p[0-9]/[m],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )

        assert len(loads) > 1
        assert len(matches) > 1

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.parametrize(
    "dtype",
    ["float", "float16", "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"],
)
def test_muladd(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        B = te.placeholder(m, dtype=type, name="B")
        C = te.placeholder(m, dtype=type, name="C")
        D = te.compute((m), lambda i: A[i] * B[i] + C[i], name="D")
        s = te.create_schedule([D.op])

        f = tvm.build(s, [A, B, C, D], target)

        # Verify we see SVE load instructions and either mad or mla instructions using z registers
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)
        matches = re.findall(
            r"mad|mla\tz[0-9].[shdb],( p[0-9]/[m],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )

        assert len(loads) > 1
        assert len(matches) > 1

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.parametrize(
    "dtype",
    ["float", "float16", "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"],
)
def test_max(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        B = te.placeholder(m, dtype=type, name="B")
        C = te.compute((m), lambda i: tvm.te.max(A[i], B[i]))
        s = te.create_schedule([C.op])

        f = tvm.build(s, [A, B, C], target)

        # Verify we see SVE load instructions and cmgt + sel instructions or a max instruction, all using z registers
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)
        compare = re.findall(
            r"cmgt\tp[0-9].[shdb],( p[0-9]/[zm],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )
        select = re.findall("sel\tz[0-9].[shdb], p[0-9], z[0-9].[shdb], z[0-9].[shdb]", assembly)
        max = re.findall(
            r"max\tz[0-9].[shdb],( p[0-9]/[zm],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )

        assert len(loads) > 1
        assert (len(compare) > 1 and len(select) == len(compare)) or len(max) > 1

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.parametrize(
    "dtype",
    ["float", "float16", "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"],
)
def test_min(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        B = te.placeholder(m, dtype=type, name="B")
        C = te.compute((m), lambda i: tvm.te.min(A[i], B[i]))
        s = te.create_schedule([C.op])

        f = tvm.build(s, [A, B, C], target)

        # Verify we see SVE load instructions and cmgt + sel instructions or a min instruction, all using z registers
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)
        compare = re.findall(
            r"cmgt\tp[0-9].[shdb],( p[0-9]/[zm],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )
        select = re.findall("sel\tz[0-9].[shdb], p[0-9], z[0-9].[shdb], z[0-9].[shdb]", assembly)
        min = re.findall(
            r"min\tz[0-9].[shdb],( p[0-9]/[zm],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )

        assert len(loads) > 1
        assert (len(compare) > 1 and len(select) == len(compare)) or len(min) > 1

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.parametrize(
    "dtype",
    ["float", "float16", "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"],
)
def test_div(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        B = te.placeholder(m, dtype=type, name="B")
        C = te.compute((m), lambda i: tvm.te.div(A[i], B[i]))
        s = te.create_schedule([C.op])

        f = tvm.build(s, [A, B, C], target)

        # Verify we see SVE load instructions and div instructions using z registers
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)
        matches = re.findall(
            r"div\tz[0-9].[shdb],( p[0-9]/[m],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )

        assert len(loads) > 1
        assert len(matches) > 1

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.parametrize(
    "dtype", ["uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"]
)
def test_mod(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        B = te.placeholder(m, dtype=type, name="B")
        C = te.compute((m), lambda i: tvm.te.floormod(A[i], B[i]), name="C")
        s = te.create_schedule([C.op])

        f = tvm.build(s, [A, B, C], target)

        # Verify we see SVE load instructions and mls instructions using z registers
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)
        matches = re.findall(
            r"mls\tz[0-9].[shdb],( p[0-9]/[m],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )

        assert len(loads) > 1
        assert len(matches) > 0

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.parametrize(
    "dtype",
    ["float", "float16", "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"],
)
def test_eq(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        B = te.placeholder(m, dtype=type, name="B")
        C = te.compute((m), lambda i: A[i] == B[i], name="C")
        s = te.create_schedule([C.op])

        f = tvm.build(s, [A, B, C], target)

        # Verify we see SVE load instructions and cmpeq or cmeq instructions using z registers
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)
        matches = re.findall(
            r"cm(p)?eq\tp[0-9].[shdb],( p[0-9]/[zm],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )

        assert len(loads) > 1
        assert len(matches) > 1

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.parametrize(
    "dtype",
    ["float", "float16", "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"],
)
def test_neq(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        B = te.placeholder(m, dtype=type, name="B")
        C = te.compute((m), lambda i: A[i] != B[i], name="C")
        s = te.create_schedule([C.op])

        f = tvm.build(s, [A, B, C], target)

        # Verify we see SVE load instructions and cmpgt, cmgt, cmpne or cmne instructions, all using z registers
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)
        matches = re.findall(
            r"cm(p)?(gt|ne)\tp[0-9].[shdb],( p[0-9]/[zm],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )

        assert len(loads) > 1
        assert len(matches) > 1

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.parametrize(
    "dtype", ["uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"]
)
def test_or(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        B = te.placeholder(m, dtype=type, name="B")
        C = te.compute((m), lambda i: A[i] | B[i], name="C")
        s = te.create_schedule([C.op])

        f = tvm.build(s, [A, B, C], target)

        # Verify we see SVE load instructions and orr instructions using z registers
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)
        matches = re.findall(
            r"orr\tz[0-9].[shdb],( p[0-9]/[zm],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )

        assert len(loads) > 1
        assert len(matches) > 1

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.parametrize(
    "dtype", ["uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"]
)
def test_and(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        B = te.placeholder(m, dtype=type, name="B")
        C = te.compute((m), lambda i: A[i] & B[i], name="C")
        s = te.create_schedule([C.op])

        f = tvm.build(s, [A, B, C], target)

        # Verify we see SVE load instructions and and instructions using z registers
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)
        matches = re.findall(
            r"and\tz[0-9].[shdb],( p[0-9]/[zm],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )

        assert len(loads) > 1
        assert len(matches) > 1

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.parametrize(
    "dtype", ["uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"]
)
def test_not(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        C = te.compute((m), lambda i: ~A[i], name="C")
        s = te.create_schedule([C.op])

        f = tvm.build(s, [A, C], target)

        # Verify we see SVE load instructions and eor instructions using z registers
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)
        matches = re.findall(
            r"eor\tz[0-9].[shdb],( p[0-9]/[zm],)? z[0-9].[shdb], z[0-9].[shdb]", assembly
        )

        assert len(loads) > 1
        assert len(matches) > 1

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 15, reason="Test requires an LLVM version of at least 15 to target SVE"
)
@pytest.mark.xfail(
    reason="Awaiting llvm support for gathered loads",
    strict=True,
)
@pytest.mark.parametrize(
    "dtype", ["uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64"]
)
def test_memcpy(dtype):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    def check_correct_assembly(type):
        m = te.var("m")
        A = te.placeholder(m, dtype=type, name="A")
        B = te.placeholder(m, dtype="int32", name="B")
        C = te.compute((m), lambda i: A[B[i]], name="C")
        s = te.create_schedule([C.op])

        f = tvm.build(s, [A, B, C], target)

        # Verify we see gather instructions in the assembly
        assembly = f.get_source("asm")
        loads = re.findall("ld1[whdb]	{ z", assembly)

        assert len(loads) > 0

    check_correct_assembly(type=dtype)


@pytest.mark.skipif(
    llvm_version_major() < 11, reason="Vscale is not supported in earlier versions of LLVM"
)
def test_codegen_vscale():
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"
    vscale = tvm.tir.vscale()

    @T.prim_func
    def main(A: T.Buffer((5,), "int32")):
        for i in range(5):
            A[i] = 2 * vscale

    build_mod = tvm.build(main, target=target)
    llvm = build_mod.get_source()

    assert re.findall(r"llvm.vscale.i32", llvm), "No vscale in generated LLVM."


@pytest.mark.skipif(
    llvm_version_major() < 11, reason="Vscale is not supported in earlier versions of LLVM"
)
def test_scalable_buffer_load_store():
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    @T.prim_func
    def my_func(a: T.handle, b: T.handle):
        A = T.match_buffer(a, (128,), "float32")
        B = T.match_buffer(b, (128,), "float32")
        T.func_attr({"global_symbol": "my_module", "tir.noalias": True})
        B[T.ramp(0, 1, 4 * T.vscale())] = A[T.ramp(0, 1, 4 * T.vscale())]

    mod = tvm.build(my_func, target=target)
    llvm = mod.get_source("ll")

    assert re.findall(r"load <vscale x 4 x float>", llvm), "No scalable load in generated LLVM."
    assert re.findall(r" store <vscale x 4 x float>", llvm), "No scalable store in generated LLVM."


@pytest.mark.skipif(
    llvm_version_major() < 11, reason="Vscale is not supported in earlier versions of LLVM"
)
def test_scalable_broadcast():
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sve"

    @T.prim_func
    def my_func(a: T.handle):
        A = T.match_buffer(a, (128,), "float32")
        T.func_attr({"global_symbol": "my_module", "tir.noalias": True})
        A[T.ramp(0, 1, 4 * T.vscale())] = T.broadcast(1, 4 * T.vscale())

    mod = tvm.build(my_func, target=target)
    llvm = mod.get_source("ll")

    assert re.findall(
        r"shufflevector \(<vscale x 4 x float> insertelement \(<vscale x 4 x float>", llvm
    ), "No scalable broadcast in generated LLVM."
    assert re.findall(r" store <vscale x 4 x float>", llvm), "No scalable store in generated LLVM."


@pytest.mark.skipif(
    llvm_version_major() < 16, reason="Test requires an LLVM version of at least 16 to target SME"
)
@pytest.mark.parametrize(
    "attr_key,attr_value,expected",
    [
        (
            SMEAttributes.STREAMING_MODE,
            SMEAttributes.StreamingModeValues.ENABLED,
            "aarch64_pstate_sm_enabled",
        ),
        (
            SMEAttributes.STREAMING_MODE,
            SMEAttributes.StreamingModeValues.COMPATIBLE,
            "aarch64_pstate_sm_compatible",
        ),
        (SMEAttributes.ZA_STORAGE, SMEAttributes.ZAStorageValues.NEW, "aarch64_pstate_za_new"),
        (
            SMEAttributes.ZA_STORAGE,
            SMEAttributes.ZAStorageValues.SHARED,
            "aarch64_pstate_za_shared",
        ),
    ],
)
def test_function_attributes(attr_key, attr_value, expected):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sme"

    @T.prim_func
    def prim_func(a: T.handle, c: T.handle):
        T.func_attr({"global_symbol": "main", "tir.noalias": T.bool(True)})
        A = T.match_buffer(a, (16,), "float32")
        C = T.match_buffer(c, (1,), "float32")

        with T.block("extern"):
            T.block_attr({attr_key: attr_value})
            for i in range(16):
                C[0] += A[i]

    func = tvm.build(prim_func, target=target)
    ll = func.get_source("ll")

    # Check that the attribute exists
    attr = re.findall(rf".*{expected}*.", ll)
    assert attr, f"Function attribute {expected} was not found in generated LLVM IR"

    # Check this attribute is used on the "compute" function
    func_attr_label = attr[0].split(" ")[1]
    found_compute_func = False
    for match in re.findall(rf".*{func_attr_label}*.", ll):
        if "_compute_" in match:
            found_compute_func = True

    assert found_compute_func, (
        f"The attribute {expected} was found to be under the label {func_attr_label}, "
        "but it was not used by the 'compute' scope function."
    )


def test_unsupported_function_attribute_type():
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sme"

    @T.prim_func
    def prim_func(a: T.handle, c: T.handle):
        T.func_attr({"global_symbol": "main", "tir.noalias": T.bool(True)})
        A = T.match_buffer(a, (16,), "float32")
        C = T.match_buffer(c, (1,), "float32")

        with T.block("extern"):
            T.block_attr({SMEAttributes.STREAMING_MODE: True})
            with T.block("root"):
                for i in range(16):
                    C[0] += A[i]

    err_msg = f"Expect {SMEAttributes.STREAMING_MODE} to have a String value but was IntImm"
    with pytest.raises(tvm.error.TVMError, match=err_msg):
        tvm.build(prim_func, target=target)


@pytest.mark.parametrize(
    "attr_key,attr_value",
    [
        (SMEAttributes.STREAMING_MODE, SMEAttributes.StreamingModeValues.ENABLED),
        (SMEAttributes.ZA_STORAGE, SMEAttributes.ZAStorageValues.NEW),
    ],
)
def test_unsupported_multiple_function_attributes(attr_key, attr_value):
    target = "llvm -mtriple=aarch64-linux-gnu -mattr=+sme"

    @T.prim_func
    def prim_func(a: T.handle, c: T.handle):
        A = T.match_buffer(a, (16,), "float32")
        C = T.match_buffer(c, (1,), "float32")

        with T.block("root"):
            with T.block("extern"):
                T.block_attr({attr_key: attr_value})
                for i in range(16):
                    C[0] += A[i] * 2
            with T.block("extern2"):
                T.block_attr({attr_key: attr_value})
                for i in range(16):
                    C[0] += A[i] * 3

    err_msg = f"Multiple definitions of {attr_key} attribute found in the function default_function_compute_"
    with pytest.raises(tvm.error.TVMError, match=err_msg):
        tvm.build(prim_func, target=target)


if __name__ == "__main__":
    tvm.testing.main()
