/*
 * ResourceHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.3
 * Created by suhler on 99/07/09
 * Last modified by suhler on 04/12/30 12:37:55
 */

package sunlabs.brazil.handler;

import sunlabs.brazil.server.FileHandler;
import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.http.HttpInputStream;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.io.File;
import java.io.FileInputStream;
import java.util.Properties;

/**
 * Handler for serving documents out of the jar file.
 * 
 * Look for url's as resources, presumably in the same "jar"
 * file as the class files.
 * This allows an entire web site to be included in the jar file.
 * A typical way to use this handler (with java 1.2+) is as follows:
 *<ul>
 *<li>Add an existing web site to the jar file with:
 *    <pre>jar uf [jar file] [web site]</pre>
 *<li>Create a config file, and add it to the jar file as:
 *    <code>sunlabs/brazil/main/config</code>.  See
 *    {@link sunlabs.brazil.server.Main Main.java} for more info.
 *<li>Create a <i>startup</i> file containing:
 *    <pre>Main-Class: sunlabs.brazil.server.Main</pre>, 
 *    and add it to the manifest with:
 *    <pre>jar ufm [jar file] [startup file]</pre>
 *<li>Start the server with:
 *    <pre>java -jar [jar file] [optional server options....]</pre>
 *</ul>
 *
 *<p>
 * if no suffix is provided, and the "directory" exists, a redirect
 * is issued, to add-on the trailing slash.
 *
 * The following server properties are used:
 * <dl class=props>
 * <dt>root	<dd> The document root path within the jar file
 * <dt>prefix, suffix, glob, match
 * <dd>Specify the URL that triggers this handler.
 * (See {@link MatchString}).
 * <dt>default	<dd> The default file name for url references ending in /
 * <dt>mime.xxx <dd> The mime type for suffix xxx.
 * </dl>
 *
 * @author Stephen Uhler
 * @version %V% ResourceHandler.java
 */

public class ResourceHandler
    implements Handler
{
    private static final String ROOT = "root"; 
    private static final String DEFAULT = "default"; 
    private static final String MISSING = "default"; 

    String root;		// prepend to all url's
    MatchString isMine;            // check for matching url
    String defaultName;		// default name for directory url's

    public boolean
    init(Server server, String prefix) {
	isMine = new MatchString(prefix, server.props);
	root = server.props.getProperty(prefix + ROOT,
		    server.props.getProperty(ROOT,"/"));
	if (!root.endsWith("/")) {
	    root += "/";
	}
        defaultName = server.props.getProperty(prefix + DEFAULT, "index.html");
	return true;
    }

    public boolean
    respond(Request request)  throws IOException {
	if (!isMine.match(request.url)) {
	    return false;
        }
	String resource = root + request.url.substring(1);
	if (resource.endsWith("/")) {
	    resource += defaultName;
	}
	request.log(Server.LOG_DIAGNOSTIC, "Looking for resource: " + resource);
	int index = resource.lastIndexOf(".");
	String type = "";
	Class cl = request.getClass();
	if (index > 0) {
	    String suffix = resource.substring(index+1);
	    type = request.props.getProperty(FileHandler.MIME +"."+ suffix,"");
	} else if (cl.getResourceAsStream(resource + "/" + defaultName)
		!= null) {
	    request.log(Server.LOG_DIAGNOSTIC, "Redirecting buy adding a /");
	    request.redirect(request.url + "/", null);
	    return true;
	}
	     
        request.log(Server.LOG_DIAGNOSTIC, "Resource type: " + type);
	if (type.equals("")) {
	    request.log(Server.LOG_INFORMATIONAL, request.url +
		    " has an invalid type");
	    return false;
	}
	InputStream in = cl.getResourceAsStream(resource);
	if (in == null) {
	    request.log(Server.LOG_DIAGNOSTIC, "Resource not found in archive");
	    return false;
	}
	request.sendResponse(in, -1, type, 200);
	return true;
    }

    private static final Class myClass = ResourceHandler.class;
    
    static String
    getProperty(Properties props, String prefix, String name, String def)
    {
	return props.getProperty(prefix + name, props.getProperty(name, def));
    }

    static String
    getResourcePath(Properties props, String prefix, String file)
    {
	File f = new File(file);
	if (f.isAbsolute() == false) {
	    String root = getProperty(props, prefix, FileHandler.ROOT, ".");
	    f = new File(root, file);
	}
	return f.getPath();
    }

    /**
     * Look for a file in the filesystem.  If its not there, see if
     * we can find a resource in our jar file.  Relative paths
     * are resolved with respect to the document root.
     *
     * @param props	where to look for server root property
     * @param prefix	"
     * @param file	The pseudo file to find as a resource
     * @return		The input stream (or null)
     */

    public static InputStream
    getResourceStream(Properties props, String prefix, String file)
	throws IOException
    {
	String path = getResourcePath(props, prefix, file);
	File f = new File(path);
	if (f.isFile()) {
	    return new FileInputStream(f);
	}

	/*
	 * Form absolute path and collapse all relative directories;
	 * getResourceAsStream() won't work otherwise.
         *
         * File.getCanonicalPath() would be nice to use, but under Windows
         * it doesn't understand '/' and it prepends a "C:\" to the result.
         * Class.getResourceAsStream requires '/'.
	 */

        path = FileHandler.urlToPath(path.replace('\\', '/'));
	InputStream in = myClass.getResourceAsStream(path.replace('\\', '/'));
	if (in == null) {
	    throw new FileNotFoundException(file);
	}
	return in;
    }

    public static String
    getResourceString(Properties props, String prefix, String file)
	throws IOException
    {
	InputStream in = getResourceStream(props, prefix, file);
	ByteArrayOutputStream out = new ByteArrayOutputStream();
	new HttpInputStream(in).copyTo(out);
        in.close();
	return out.toString();
    }

    /**
     * Find a file blob as a resource in our jar file (experimental).
     * @param props	where to look for server root property
     * @param prefix	"
     * @param file	The pseudo file to find as a resource
     * @return		The data, if available, or raises an exception.
     */

    public static byte[]
    getResourceBytes(Properties props, String prefix, String file)
	throws IOException
    {
	InputStream in = getResourceStream(props, prefix, file);
	ByteArrayOutputStream out = new ByteArrayOutputStream();
	new HttpInputStream(in).copyTo(out);
        in.close();
	return out.toByteArray();
    }
}
