package Text::GenderFromName;

# Text::GenderFromName.pm
#
# Jon Orwant, <orwant@readable.com>
#
# 10 Mar 97 - created
#
# Copyright 1997, 2001 Jon Orwant.  All rights reserved.
# This program is free software; you can redistribute it and/or
# modify it under the same terms as Perl itself.
#
# Version 0.2.  Module list status is "Rdpf."
#
# Two big changes between this and version 0.1:
#
# First, the gender() subroutine now returns "m" if none of the rules 
# trigger, instead of "undef".  This isn't meant to be sexist, but merely 
# to reflect that the original rules were organized such that male names
# were meant to "fall through" to the end.
# 
# Second, I've appended a big list of names that gets searched first.
# This is counter to the spirit of the original code -- the whole point
# was that it used heuristics to guess at the gender rather than a 
# huge list of names.  But enough people have requested this that I 
# decided to add in the huge list, disk space be darned.

require 5;

require Exporter;
@ISA = qw( Exporter );

=head1 NAME

C<Text::GenderFromName> - Guess the gender of a "Christian" first name.

=head1 SYNOPSIS

    use Text::GenderFromName;
 
    print gender("Jon");    # prints "m"

Text::GenderFromName is available at a CPAN site near you.

=head1 DESCRIPTION

This module provides a lone function: C<gender()>, which returns
one of three values: "m" for male, "f" for female", or UNDEF if
it doesn't know.  For instance, gender("Chris") is UNDEF.

The original code assumed a default of male, and I am happy to
contribute to the destruction of the oppressive Patriarchy by
returning an UNDEF value if no rule triggers.  Ha ha!  Seriously, it'll
be useful to know when C<gender()> has no clue.

For the curious, I ran Text::GenderFromName on The Perl Journal's
subscriber list.  The result?  

   Male:    68%
   Female:  32%

=head1 BUGS

C<gender()> can never be perfect.  

With the addition of the huge word lists, some of the regex rules are
now redundant, so gender() may take several more milliseconds to
return than it could.

I'm sure that many of these rules could return immediately upon
firing.  However, the original author arranged them in a very
deliberate order, with more specific rules at the end overruling
earlier rules.  Consequently, I can't turn all of these rules into the
speedier form C<return "f" if /.../> without throwing away the meaning
of the ordering.  

The rules should probably be made case-insensitive, but I bet there's
some funky situation in which that'll lose.

=head1 AUTHOR

Jon Orwant

Readable Publications, Inc.

orwant@readable.com

This is an adaptation of an 8/91 awk script by Scott Pakin in the
December 91 issue of Computer Language Monthly.

Small contributions by Andrew Langmead and John Strickler.  Thanks 
to Bob Baldwin, Matt Bishop, and Daniel Klein for their lists of names.

=cut

@EXPORT = qw(gender);
$VERSION = 0.2;

%female_names = ("aaren",0,"aarika",0,"abagael",0,"abagail",0,"abbe",0,"abbey",0,"abbi",0,"abbie",0,"abby",0,"abbye",0,"abigael",0,"abigail",0,"abigale",0,"abra",0,"ada",0,"adah",0,"adaline",0,"adan",0,"adara",0,"adda",0,"addi",0,"addia",0,"addie",0,"addy",0,"adel",0,"adela",0,"adelaida",0,"adelaide",0,"adele",0,"adelheid",0,"adelice",0,"adelina",0,"adelind",0,"adeline",0,"adella",0,"adelle",0,"adena",0,"adey",0,"adi",0,"adiana",0,"adina",0,"adora",0,"adore",0,"adoree",0,"adorne",0,"adrea",0,"adria",0,"adriaens",0,"adrian",0,"adriana",0,"adriane",0,"adrianna",0,"adrianne",0,"adriena",0,"adrienne",0,"aeriel",0,"aeriela",0,"aeriell",0,"afton",0,"ag",0,"agace",0,"agata",0,"agatha",0,"agathe",0,"aggi",0,"aggie",0,"aggy",0,"agna",0,"agnella",0,"agnes",0,"agnese",0,"agnesse",0,"agneta",0,"agnola",0,"agretha",0,"aida",0,"aidan",0,"aigneis",0,"aila",0,"aile",0,"ailee",0,"aileen",0,"ailene",0,"ailey",0,"aili",0,"ailina",0,"ailis",0,"ailsun",0,"ailyn",0,"aime",0,"aimee",0,"aimil",0,"aindrea",0,"ainslee",0,"ainsley",0,"ainslie",0,"ajay",0,"alaine",0,"alameda",0,"alana",0,"alanah",0,"alane",0,"alanna",0,"alayne",0,"alberta",0,"albertina",0,"albertine",0,"albina",0,"alecia",0,"aleda",0,"aleece",0,"aleen",0,"alejandra",0,"alejandrina",0,"alena",0,"alene",0,"alessandra",0,"aleta",0,"alethea",0,"alex",0,"alexa",0,"alexandra",0,"alexandrina",0,"alexi",0,"alexia",0,"alexina",0,"alexine",0,"alexis",0,"alfi",0,"alfie",0,"alfreda",0,"alfy",0,"ali",0,"alia",0,"alica",0,"alice",0,"alicea",0,"alicia",0,"alida",0,"alidia",0,"alie",0,"alika",0,"alikee",0,"alina",0,"aline",0,"alis",0,"alisa",0,"alisha",0,"alison",0,"alissa",0,"alisun",0,"alix",0,"aliza",0,"alla",0,"alleen",0,"allegra",0,"allene",0,"alli",0,"allianora",0,"allie",0,"allina",0,"allis",0,"allison",0,"allissa",0,"allix",0,"allsun",0,"allx",0,"ally",0,"allyce",0,"allyn",0,"allys",0,"allyson",0,"alma",0,"almeda",0,"almeria",0,"almeta",0,"almira",0,"almire",0,"aloise",0,"aloisia",0,"aloysia",0,"alta",0,"althea",0,"alvera",0,"alverta",0,"alvina",0,"alvinia",0,"alvira",0,"alyce",0,"alyda",0,"alys",0,"alysa",0,"alyse",0,"alysia",0,"alyson",0,"alyss",0,"alyssa",0,"amabel",0,"amabelle",0,"amalea",0,"amalee",0,"amaleta",0,"amalia",0,"amalie",0,"amalita",0,"amalle",0,"amanda",0,"amandi",0,"amandie",0,"amandy",0,"amara",0,"amargo",0,"amata",0,"amber",0,"amberly",0,"ambur",0,"ame",0,"amelia",0,"amelie",0,"amelina",0,"ameline",0,"amelita",0,"ami",0,"amie",0,"amii",0,"amil",0,"amitie",0,"amity",0,"ammamaria",0,"amy",0,"amye",0,"ana",0,"anabal",0,"anabel",0,"anabella",0,"anabelle",0,"analiese",0,"analise",0,"anallese",0,"anallise",0,"anastasia",0,"anastasie",0,"anastassia",0,"anatola",0,"andee",0,"andeee",0,"anderea",0,"andi",0,"andie",0,"andra",0,"andrea",0,"andreana",0,"andree",0,"andrei",0,"andria",0,"andriana",0,"andriette",0,"andromache",0,"andy",0,"anestassia",0,"anet",0,"anett",0,"anetta",0,"anette",0,"ange",0,"angel",0,"angela",0,"angele",0,"angelia",0,"angelica",0,"angelika",0,"angelina",0,"angeline",0,"angelique",0,"angelita",0,"angelle",0,"angie",0,"angil",0,"angy",0,"ania",0,"anica",0,"anissa",0,"anita",0,"anitra",0,"anjanette",0,"anjela",0,"ann",0,"ann-marie",0,"anna",0,"anna-diana",0,"anna-diane",0,"anna-maria",0,"annabal",0,"annabel",0,"annabela",0,"annabell",0,"annabella",0,"annabelle",0,"annadiana",0,"annadiane",0,"annalee",0,"annaliese",0,"annalise",0,"annamaria",0,"annamarie",0,"anne",0,"anne-corinne",0,"anne-marie",0,"annecorinne",0,"anneliese",0,"annelise",0,"annemarie",0,"annetta",0,"annette",0,"anni",0,"annice",0,"annie",0,"annis",0,"annissa",0,"annmaria",0,"annmarie",0,"annnora",0,"annora",0,"anny",0,"anselma",0,"ansley",0,"anstice",0,"anthe",0,"anthea",0,"anthia",0,"anthiathia",0,"antoinette",0,"antonella",0,"antonetta",0,"antonia",0,"antonie",0,"antonietta",0,"antonina",0,"anya",0,"appolonia",0,"april",0,"aprilette",0,"ara",0,"arabel",0,"arabela",0,"arabele",0,"arabella",0,"arabelle",0,"arda",0,"ardath",0,"ardeen",0,"ardelia",0,"ardelis",0,"ardella",0,"ardelle",0,"arden",0,"ardene",0,"ardenia",0,"ardine",0,"ardis",0,"ardisj",0,"ardith",0,"ardra",0,"ardyce",0,"ardys",0,"ardyth",0,"aretha",0,"ariadne",0,"ariana",0,"aridatha",0,"ariel",0,"ariela",0,"ariella",0,"arielle",0,"arlana",0,"arlee",0,"arleen",0,"arlen",0,"arlena",0,"arlene",0,"arleta",0,"arlette",0,"arleyne",0,"arlie",0,"arliene",0,"arlina",0,"arlinda",0,"arline",0,"arluene",0,"arly",0,"arlyn",0,"arlyne",0,"aryn",0,"ashely",0,"ashia",0,"ashien",0,"ashil",0,"ashla",0,"ashlan",0,"ashlee",0,"ashleigh",0,"ashlen",0,"ashley",0,"ashli",0,"ashlie",0,"ashly",0,"asia",0,"astra",0,"astrid",0,"astrix",0,"atalanta",0,"athena",0,"athene",0,"atlanta",0,"atlante",0,"auberta",0,"aubine",0,"aubree",0,"aubrette",0,"aubrey",0,"aubrie",0,"aubry",0,"audi",0,"audie",0,"audra",0,"audre",0,"audrey",0,"audrie",0,"audry",0,"audrye",0,"audy",0,"augusta",0,"auguste",0,"augustina",0,"augustine",0,"aundrea",0,"aura",0,"aurea",0,"aurel",0,"aurelea",0,"aurelia",0,"aurelie",0,"auria",0,"aurie",0,"aurilia",0,"aurlie",0,"auroora",0,"aurora",0,"aurore",0,"austin",0,"austina",0,"austine",0,"ava",0,"aveline",0,"averil",0,"averyl",0,"avie",0,"avis",0,"aviva",0,"avivah",0,"avril",0,"avrit",0,"ayn",0,"bab",0,"babara",0,"babb",0,"babbette",0,"babbie",0,"babette",0,"babita",0,"babs",0,"bambi",0,"bambie",0,"bamby",0,"barb",0,"barbabra",0,"barbara",0,"barbara-anne",0,"barbaraanne",0,"barbe",0,"barbee",0,"barbette",0,"barbey",0,"barbi",0,"barbie",0,"barbra",0,"barby",0,"bari",0,"barrie",0,"barry",0,"basia",0,"bathsheba",0,"batsheva",0,"bea",0,"beatrice",0,"beatrisa",0,"beatrix",0,"beatriz",0,"bebe",0,"becca",0,"becka",0,"becki",0,"beckie",0,"becky",0,"bee",0,"beilul",0,"beitris",0,"bekki",0,"bel",0,"belia",0,"belicia",0,"belinda",0,"belita",0,"bell",0,"bella",0,"bellanca",0,"belle",0,"bellina",0,"belva",0,"belvia",0,"bendite",0,"benedetta",0,"benedicta",0,"benedikta",0,"benetta",0,"benita",0,"benni",0,"bennie",0,"benny",0,"benoite",0,"berenice",0,"beret",0,"berget",0,"berna",0,"bernadene",0,"bernadette",0,"bernadina",0,"bernadine",0,"bernardina",0,"bernardine",0,"bernelle",0,"bernete",0,"bernetta",0,"bernette",0,"berni",0,"bernice",0,"bernie",0,"bernita",0,"berny",0,"berri",0,"berrie",0,"berry",0,"bert",0,"berta",0,"berte",0,"bertha",0,"berthe",0,"berti",0,"bertie",0,"bertina",0,"bertine",0,"berty",0,"beryl",0,"beryle",0,"bess",0,"bessie",0,"bessy",0,"beth",0,"bethanne",0,"bethany",0,"bethena",0,"bethina",0,"betsey",0,"betsy",0,"betta",0,"bette",0,"bette-ann",0,"betteann",0,"betteanne",0,"betti",0,"bettina",0,"bettine",0,"betty",0,"bettye",0,"beulah",0,"bev",0,"beverie",0,"beverlee",0,"beverley",0,"beverlie",0,"beverly",0,"bevvy",0,"bianca",0,"bianka",0,"bibbie",0,"bibby",0,"bibbye",0,"bibi",0,"biddie",0,"biddy",0,"bidget",0,"bili",0,"bill",0,"billi",0,"billie",0,"billy",0,"billye",0,"binni",0,"binnie",0,"binny",0,"bird",0,"birdie",0,"birgit",0,"birgitta",0,"blair",0,"blaire",0,"blake",0,"blakelee",0,"blakeley",0,"blanca",0,"blanch",0,"blancha",0,"blanche",0,"blinni",0,"blinnie",0,"blinny",0,"bliss",0,"blisse",0,"blithe",0,"blondell",0,"blondelle",0,"blondie",0,"blondy",0,"blythe",0,"bobbe",0,"bobbee",0,"bobbette",0,"bobbi",0,"bobbie",0,"bobby",0,"bobbye",0,"bobette",0,"bobina",0,"bobine",0,"bobinette",0,"bonita",0,"bonnee",0,"bonni",0,"bonnibelle",0,"bonnie",0,"bonny",0,"brana",0,"brandais",0,"brande",0,"brandea",0,"brandi",0,"brandice",0,"brandie",0,"brandise",0,"brandy",0,"breanne",0,"brear",0,"bree",0,"breena",0,"bren",0,"brena",0,"brenda",0,"brenn",0,"brenna",0,"brett",0,"bria",0,"briana",0,"brianna",0,"brianne",0,"bride",0,"bridget",0,"bridgette",0,"bridie",0,"brier",0,"brietta",0,"brigid",0,"brigida",0,"brigit",0,"brigitta",0,"brigitte",0,"brina",0,"briney",0,"brinn",0,"brinna",0,"briny",0,"brit",0,"brita",0,"britney",0,"britni",0,"britt",0,"britta",0,"brittan",0,"brittaney",0,"brittani",0,"brittany",0,"britte",0,"britteny",0,"brittne",0,"brittney",0,"brittni",0,"brook",0,"brooke",0,"brooks",0,"brunhilda",0,"brunhilde",0,"bryana",0,"bryn",0,"bryna",0,"brynn",0,"brynna",0,"brynne",0,"buffy",0,"bunni",0,"bunnie",0,"bunny",0,"cacilia",0,"cacilie",0,"cahra",0,"cairistiona",0,"caitlin",0,"caitrin",0,"cal",0,"calida",0,"calla",0,"calley",0,"calli",0,"callida",0,"callie",0,"cally",0,"calypso",0,"cam",0,"camala",0,"camel",0,"camella",0,"camellia",0,"cami",0,"camila",0,"camile",0,"camilla",0,"camille",0,"cammi",0,"cammie",0,"cammy",0,"candace",0,"candi",0,"candice",0,"candida",0,"candide",0,"candie",0,"candis",0,"candra",0,"candy",0,"caprice",0,"cara",0,"caralie",0,"caren",0,"carena",0,"caresa",0,"caressa",0,"caresse",0,"carey",0,"cari",0,"caria",0,"carie",0,"caril",0,"carilyn",0,"carin",0,"carina",0,"carine",0,"cariotta",0,"carissa",0,"carita",0,"caritta",0,"carla",0,"carlee",0,"carleen",0,"carlen",0,"carlene",0,"carley",0,"carlie",0,"carlin",0,"carlina",0,"carline",0,"carlita",0,"carlota",0,"carlotta",0,"carly",0,"carlye",0,"carlyn",0,"carlynn",0,"carlynne",0,"carma",0,"carmel",0,"carmela",0,"carmelia",0,"carmelina",0,"carmelita",0,"carmella",0,"carmelle",0,"carmen",0,"carmencita",0,"carmina",0,"carmine",0,"carmita",0,"carmon",0,"caro",0,"carol",0,"carol-jean",0,"carola",0,"carolan",0,"carolann",0,"carole",0,"carolee",0,"carolin",0,"carolina",0,"caroline",0,"caroljean",0,"carolyn",0,"carolyne",0,"carolynn",0,"caron",0,"carree",0,"carri",0,"carrie",0,"carrissa",0,"carroll",0,"carry",0,"cary",0,"caryl",0,"caryn",0,"casandra",0,"casey",0,"casi",0,"casie",0,"cass",0,"cassandra",0,"cassandre",0,"cassandry",0,"cassaundra",0,"cassey",0,"cassi",0,"cassie",0,"cassondra",0,"cassy",0,"catarina",0,"cate",0,"caterina",0,"catha",0,"catharina",0,"catharine",0,"cathe",0,"cathee",0,"catherin",0,"catherina",0,"catherine",0,"cathi",0,"cathie",0,"cathleen",0,"cathlene",0,"cathrin",0,"cathrine",0,"cathryn",0,"cathy",0,"cathyleen",0,"cati",0,"catie",0,"catina",0,"catlaina",0,"catlee",0,"catlin",0,"catrina",0,"catriona",0,"caty",0,"caye",0,"cayla",0,"cecelia",0,"cecil",0,"cecile",0,"ceciley",0,"cecilia",0,"cecilla",0,"cecily",0,"ceil",0,"cele",0,"celene",0,"celesta",0,"celeste",0,"celestia",0,"celestina",0,"celestine",0,"celestyn",0,"celestyna",0,"celia",0,"celie",0,"celina",0,"celinda",0,"celine",0,"celinka",0,"celisse",0,"celka",0,"celle",0,"cesya",0,"chad",0,"chanda",0,"chandal",0,"chandra",0,"channa",0,"chantal",0,"chantalle",0,"charil",0,"charin",0,"charis",0,"charissa",0,"charisse",0,"charita",0,"charity",0,"charla",0,"charlean",0,"charleen",0,"charlena",0,"charlene",0,"charline",0,"charlot",0,"charlotta",0,"charlotte",0,"charmain",0,"charmaine",0,"charmane",0,"charmian",0,"charmine",0,"charmion",0,"charo",0,"charyl",0,"chastity",0,"chelsae",0,"chelsea",0,"chelsey",0,"chelsie",0,"chelsy",0,"cher",0,"chere",0,"cherey",0,"cheri",0,"cherianne",0,"cherice",0,"cherida",0,"cherie",0,"cherilyn",0,"cherilynn",0,"cherin",0,"cherise",0,"cherish",0,"cherlyn",0,"cherri",0,"cherrita",0,"cherry",0,"chery",0,"cherye",0,"cheryl",0,"cheslie",0,"chiarra",0,"chickie",0,"chicky",0,"chiquia",0,"chiquita",0,"chlo",0,"chloe",0,"chloette",0,"chloris",0,"chris",0,"chrissie",0,"chrissy",0,"christa",0,"christabel",0,"christabella",0,"christal",0,"christalle",0,"christan",0,"christean",0,"christel",0,"christen",0,"christi",0,"christian",0,"christiana",0,"christiane",0,"christie",0,"christin",0,"christina",0,"christine",0,"christy",0,"christye",0,"christyna",0,"chrysa",0,"chrysler",0,"chrystal",0,"chryste",0,"chrystel",0,"cicely",0,"cicily",0,"ciel",0,"cilka",0,"cinda",0,"cindee",0,"cindelyn",0,"cinderella",0,"cindi",0,"cindie",0,"cindra",0,"cindy",0,"cinnamon",0,"cissiee",0,"cissy",0,"clair",0,"claire",0,"clara",0,"clarabelle",0,"clare",0,"claresta",0,"clareta",0,"claretta",0,"clarette",0,"clarey",0,"clari",0,"claribel",0,"clarice",0,"clarie",0,"clarinda",0,"clarine",0,"clarissa",0,"clarisse",0,"clarita",0,"clary",0,"claude",0,"claudelle",0,"claudetta",0,"claudette",0,"claudia",0,"claudie",0,"claudina",0,"claudine",0,"clea",0,"clem",0,"clemence",0,"clementia",0,"clementina",0,"clementine",0,"clemmie",0,"clemmy",0,"cleo",0,"cleopatra",0,"clerissa",0,"clio",0,"clo",0,"cloe",0,"cloris",0,"clotilda",0,"clovis",0,"codee",0,"codi",0,"codie",0,"cody",0,"coleen",0,"colene",0,"coletta",0,"colette",0,"colleen",0,"collen",0,"collete",0,"collette",0,"collie",0,"colline",0,"colly",0,"con",0,"concettina",0,"conchita",0,"concordia",0,"conni",0,"connie",0,"conny",0,"consolata",0,"constance",0,"constancia",0,"constancy",0,"constanta",0,"constantia",0,"constantina",0,"constantine",0,"consuela",0,"consuelo",0,"cookie",0,"cora",0,"corabel",0,"corabella",0,"corabelle",0,"coral",0,"coralie",0,"coraline",0,"coralyn",0,"cordelia",0,"cordelie",0,"cordey",0,"cordi",0,"cordie",0,"cordula",0,"cordy",0,"coreen",0,"corella",0,"corena",0,"corenda",0,"corene",0,"coretta",0,"corette",0,"corey",0,"cori",0,"corie",0,"corilla",0,"corina",0,"corine",0,"corinna",0,"corinne",0,"coriss",0,"corissa",0,"corliss",0,"corly",0,"cornela",0,"cornelia",0,"cornelle",0,"cornie",0,"corny",0,"correna",0,"correy",0,"corri",0,"corrianne",0,"corrie",0,"corrina",0,"corrine",0,"corrinne",0,"corry",0,"cortney",0,"cory",0,"cosetta",0,"cosette",0,"costanza",0,"courtenay",0,"courtnay",0,"courtney",0,"crin",0,"cris",0,"crissie",0,"crissy",0,"crista",0,"cristabel",0,"cristal",0,"cristen",0,"cristi",0,"cristie",0,"cristin",0,"cristina",0,"cristine",0,"cristionna",0,"cristy",0,"crysta",0,"crystal",0,"crystie",0,"cthrine",0,"cyb",0,"cybil",0,"cybill",0,"cymbre",0,"cynde",0,"cyndi",0,"cyndia",0,"cyndie",0,"cyndy",0,"cynthea",0,"cynthia",0,"cynthie",0,"cynthy",0,"dacey",0,"dacia",0,"dacie",0,"dacy",0,"dael",0,"daffi",0,"daffie",0,"daffy",0,"dagmar",0,"dahlia",0,"daile",0,"daisey",0,"daisi",0,"daisie",0,"daisy",0,"dale",0,"dalenna",0,"dalia",0,"dalila",0,"dallas",0,"daloris",0,"damara",0,"damaris",0,"damita",0,"dana",0,"danell",0,"danella",0,"danette",0,"dani",0,"dania",0,"danica",0,"danice",0,"daniela",0,"daniele",0,"daniella",0,"danielle",0,"danika",0,"danila",0,"danit",0,"danita",0,"danna",0,"danni",0,"dannie",0,"danny",0,"dannye",0,"danya",0,"danyelle",0,"danyette",0,"daphene",0,"daphna",0,"daphne",0,"dara",0,"darb",0,"darbie",0,"darby",0,"darcee",0,"darcey",0,"darci",0,"darcie",0,"darcy",0,"darda",0,"dareen",0,"darell",0,"darelle",0,"dari",0,"daria",0,"darice",0,"darla",0,"darleen",0,"darlene",0,"darline",0,"darlleen",0,"daron",0,"darrelle",0,"darryl",0,"darsey",0,"darsie",0,"darya",0,"daryl",0,"daryn",0,"dasha",0,"dasi",0,"dasie",0,"dasya",0,"datha",0,"daune",0,"daveen",0,"daveta",0,"davida",0,"davina",0,"davine",0,"davita",0,"dawn",0,"dawna",0,"dayle",0,"dayna",0,"ddene",0,"de",0,"deana",0,"deane",0,"deanna",0,"deanne",0,"deb",0,"debbi",0,"debbie",0,"debby",0,"debee",0,"debera",0,"debi",0,"debor",0,"debora",0,"deborah",0,"debra",0,"dede",0,"dedie",0,"dedra",0,"dee",0,"dee dee",0,"deeann",0,"deeanne",0,"deedee",0,"deena",0,"deerdre",0,"deeyn",0,"dehlia",0,"deidre",0,"deina",0,"deirdre",0,"del",0,"dela",0,"delcina",0,"delcine",0,"delia",0,"delila",0,"delilah",0,"delinda",0,"dell",0,"della",0,"delly",0,"delora",0,"delores",0,"deloria",0,"deloris",0,"delphine",0,"delphinia",0,"demeter",0,"demetra",0,"demetria",0,"demetris",0,"dena",0,"deni",0,"denice",0,"denise",0,"denna",0,"denni",0,"dennie",0,"denny",0,"deny",0,"denys",0,"denyse",0,"deonne",0,"desdemona",0,"desirae",0,"desiree",0,"desiri",0,"deva",0,"devan",0,"devi",0,"devin",0,"devina",0,"devinne",0,"devon",0,"devondra",0,"devonna",0,"devonne",0,"devora",0,"di",0,"diahann",0,"dian",0,"diana",0,"diandra",0,"diane",0,"diane-marie",0,"dianemarie",0,"diann",0,"dianna",0,"dianne",0,"diannne",0,"didi",0,"dido",0,"diena",0,"dierdre",0,"dina",0,"dinah",0,"dinnie",0,"dinny",0,"dion",0,"dione",0,"dionis",0,"dionne",0,"dita",0,"dix",0,"dixie",0,"dniren",0,"dode",0,"dodi",0,"dodie",0,"dody",0,"doe",0,"doll",0,"dolley",0,"dolli",0,"dollie",0,"dolly",0,"dolores",0,"dolorita",0,"doloritas",0,"domeniga",0,"dominga",0,"domini",0,"dominica",0,"dominique",0,"dona",0,"donella",0,"donelle",0,"donetta",0,"donia",0,"donica",0,"donielle",0,"donna",0,"donnajean",0,"donnamarie",0,"donni",0,"donnie",0,"donny",0,"dora",0,"doralia",0,"doralin",0,"doralyn",0,"doralynn",0,"doralynne",0,"dore",0,"doreen",0,"dorelia",0,"dorella",0,"dorelle",0,"dorena",0,"dorene",0,"doretta",0,"dorette",0,"dorey",0,"dori",0,"doria",0,"dorian",0,"dorice",0,"dorie",0,"dorine",0,"doris",0,"dorisa",0,"dorise",0,"dorita",0,"doro",0,"dorolice",0,"dorolisa",0,"dorotea",0,"doroteya",0,"dorothea",0,"dorothee",0,"dorothy",0,"dorree",0,"dorri",0,"dorrie",0,"dorris",0,"dorry",0,"dorthea",0,"dorthy",0,"dory",0,"dosi",0,"dot",0,"doti",0,"dotti",0,"dottie",0,"dotty",0,"dre",0,"dreddy",0,"dredi",0,"drona",0,"dru",0,"druci",0,"drucie",0,"drucill",0,"drucy",0,"drusi",0,"drusie",0,"drusilla",0,"drusy",0,"dulce",0,"dulcea",0,"dulci",0,"dulcia",0,"dulciana",0,"dulcie",0,"dulcine",0,"dulcinea",0,"dulcy",0,"dulsea",0,"dusty",0,"dyan",0,"dyana",0,"dyane",0,"dyann",0,"dyanna",0,"dyanne",0,"dyna",0,"dynah",0,"eachelle",0,"eada",0,"eadie",0,"eadith",0,"ealasaid",0,"eartha",0,"easter",0,"eba",0,"ebba",0,"ebonee",0,"ebony",0,"eda",0,"eddi",0,"eddie",0,"eddy",0,"ede",0,"edee",0,"edeline",0,"eden",0,"edi",0,"edie",0,"edin",0,"edita",0,"edith",0,"editha",0,"edithe",0,"ediva",0,"edna",0,"edwina",0,"edy",0,"edyth",0,"edythe",0,"effie",0,"eileen",0,"eilis",0,"eimile",0,"eirena",0,"ekaterina",0,"elaina",0,"elaine",0,"elana",0,"elane",0,"elayne",0,"elberta",0,"elbertina",0,"elbertine",0,"eleanor",0,"eleanora",0,"eleanore",0,"electra",0,"eleen",0,"elena",0,"elene",0,"eleni",0,"elenore",0,"eleonora",0,"eleonore",0,"elfie",0,"elfreda",0,"elfrida",0,"elfrieda",0,"elga",0,"elianora",0,"elianore",0,"elicia",0,"elie",0,"elinor",0,"elinore",0,"elisa",0,"elisabet",0,"elisabeth",0,"elisabetta",0,"elise",0,"elisha",0,"elissa",0,"elita",0,"eliza",0,"elizabet",0,"elizabeth",0,"elka",0,"elke",0,"ella",0,"elladine",0,"elle",0,"ellen",0,"ellene",0,"ellette",0,"elli",0,"ellie",0,"ellissa",0,"elly",0,"ellyn",0,"ellynn",0,"elmira",0,"elna",0,"elnora",0,"elnore",0,"eloisa",0,"eloise",0,"elonore",0,"elora",0,"elsa",0,"elsbeth",0,"else",0,"elset",0,"elsey",0,"elsi",0,"elsie",0,"elsinore",0,"elspeth",0,"elsy",0,"elva",0,"elvera",0,"elvina",0,"elvira",0,"elwira",0,"elyn",0,"elyse",0,"elysee",0,"elysha",0,"elysia",0,"elyssa",0,"em",0,"ema",0,"emalee",0,"emalia",0,"emelda",0,"emelia",0,"emelina",0,"emeline",0,"emelita",0,"emelyne",0,"emera",0,"emilee",0,"emili",0,"emilia",0,"emilie",0,"emiline",0,"emily",0,"emlyn",0,"emlynn",0,"emlynne",0,"emma",0,"emmalee",0,"emmaline",0,"emmalyn",0,"emmalynn",0,"emmalynne",0,"emmeline",0,"emmey",0,"emmi",0,"emmie",0,"emmy",0,"emmye",0,"emogene",0,"emyle",0,"emylee",0,"engracia",0,"enid",0,"enrica",0,"enrichetta",0,"enrika",0,"enriqueta",0,"eolanda",0,"eolande",0,"eran",0,"erda",0,"erena",0,"erica",0,"ericha",0,"ericka",0,"erika",0,"erin",0,"erina",0,"erinn",0,"erinna",0,"erma",0,"ermengarde",0,"ermentrude",0,"ermina",0,"erminia",0,"erminie",0,"erna",0,"ernaline",0,"ernesta",0,"ernestine",0,"ertha",0,"eryn",0,"esma",0,"esmaria",0,"esme",0,"esmeralda",0,"essa",0,"essie",0,"essy",0,"esta",0,"estel",0,"estele",0,"estell",0,"estella",0,"estelle",0,"ester",0,"esther",0,"estrella",0,"estrellita",0,"ethel",0,"ethelda",0,"ethelin",0,"ethelind",0,"etheline",0,"ethelyn",0,"ethyl",0,"etta",0,"etti",0,"ettie",0,"etty",0,"eudora",0,"eugenia",0,"eugenie",0,"eugine",0,"eula",0,"eulalie",0,"eunice",0,"euphemia",0,"eustacia",0,"eva",0,"evaleen",0,"evangelia",0,"evangelin",0,"evangelina",0,"evangeline",0,"evania",0,"evanne",0,"eve",0,"eveleen",0,"evelina",0,"eveline",0,"evelyn",0,"evey",0,"evie",0,"evita",0,"evonne",0,"evvie",0,"evvy",0,"evy",0,"eyde",0,"eydie",0,"ezmeralda",0,"fae",0,"faina",0,"faith",0,"fallon",0,"fan",0,"fanchette",0,"fanchon",0,"fancie",0,"fancy",0,"fanechka",0,"fania",0,"fanni",0,"fannie",0,"fanny",0,"fanya",0,"fara",0,"farah",0,"farand",0,"farica",0,"farra",0,"farrah",0,"farrand",0,"faun",0,"faunie",0,"faustina",0,"faustine",0,"fawn",0,"fawne",0,"fawnia",0,"fay",0,"faydra",0,"faye",0,"fayette",0,"fayina",0,"fayre",0,"fayth",0,"faythe",0,"federica",0,"fedora",0,"felecia",0,"felicdad",0,"felice",0,"felicia",0,"felicity",0,"felicle",0,"felipa",0,"felisha",0,"felita",0,"feliza",0,"fenelia",0,"feodora",0,"ferdinanda",0,"ferdinande",0,"fern",0,"fernanda",0,"fernande",0,"fernandina",0,"ferne",0,"fey",0,"fiann",0,"fianna",0,"fidela",0,"fidelia",0,"fidelity",0,"fifi",0,"fifine",0,"filia",0,"filide",0,"filippa",0,"fina",0,"fiona",0,"fionna",0,"fionnula",0,"fiorenze",0,"fleur",0,"fleurette",0,"flo",0,"flor",0,"flora",0,"florance",0,"flore",0,"florella",0,"florence",0,"florencia",0,"florentia",0,"florenza",0,"florette",0,"flori",0,"floria",0,"florida",0,"florie",0,"florina",0,"florinda",0,"floris",0,"florri",0,"florrie",0,"florry",0,"flory",0,"flossi",0,"flossie",0,"flossy",0,"flss",0,"fran",0,"francene",0,"frances",0,"francesca",0,"francine",0,"francisca",0,"franciska",0,"francoise",0,"francyne",0,"frank",0,"frankie",0,"franky",0,"franni",0,"frannie",0,"franny",0,"frayda",0,"fred",0,"freda",0,"freddi",0,"freddie",0,"freddy",0,"fredelia",0,"frederica",0,"fredericka",0,"frederique",0,"fredi",0,"fredia",0,"fredra",0,"fredrika",0,"freida",0,"frieda",0,"friederike",0,"fulvia",0,"gabbey",0,"gabbi",0,"gabbie",0,"gabey",0,"gabi",0,"gabie",0,"gabriel",0,"gabriela",0,"gabriell",0,"gabriella",0,"gabrielle",0,"gabriellia",0,"gabrila",0,"gaby",0,"gae",0,"gael",0,"gail",0,"gale",0,"gale ",0,"galina",0,"garland",0,"garnet",0,"garnette",0,"gates",0,"gavra",0,"gavrielle",0,"gay",0,"gaye",0,"gayel",0,"gayla",0,"gayle",0,"gayleen",0,"gaylene",0,"gaynor",0,"gelya",0,"gena",0,"gene",0,"geneva",0,"genevieve",0,"genevra",0,"genia",0,"genna",0,"genni",0,"gennie",0,"gennifer",0,"genny",0,"genovera",0,"genvieve",0,"george",0,"georgeanna",0,"georgeanne",0,"georgena",0,"georgeta",0,"georgetta",0,"georgette",0,"georgia",0,"georgiana",0,"georgianna",0,"georgianne",0,"georgie",0,"georgina",0,"georgine",0,"geralda",0,"geraldine",0,"gerda",0,"gerhardine",0,"geri",0,"gerianna",0,"gerianne",0,"gerladina",0,"germain",0,"germaine",0,"germana",0,"gerri",0,"gerrie",0,"gerrilee",0,"gerry",0,"gert",0,"gerta",0,"gerti",0,"gertie",0,"gertrud",0,"gertruda",0,"gertrude",0,"gertrudis",0,"gerty",0,"giacinta",0,"giana",0,"gianina",0,"gianna",0,"gigi",0,"gilberta",0,"gilberte",0,"gilbertina",0,"gilbertine",0,"gilda",0,"gilemette",0,"gill",0,"gillan",0,"gilli",0,"gillian",0,"gillie",0,"gilligan",0,"gilly",0,"gina",0,"ginelle",0,"ginevra",0,"ginger",0,"ginni",0,"ginnie",0,"ginnifer",0,"ginny",0,"giorgia",0,"giovanna",0,"gipsy",0,"giralda",0,"gisela",0,"gisele",0,"gisella",0,"giselle",0,"giuditta",0,"giulia",0,"giulietta",0,"giustina",0,"gizela",0,"glad",0,"gladi",0,"gladys",0,"gleda",0,"glen",0,"glenda",0,"glenine",0,"glenn",0,"glenna",0,"glennie",0,"glennis",0,"glori",0,"gloria",0,"gloriana",0,"gloriane",0,"glory",0,"glyn",0,"glynda",0,"glynis",0,"glynnis",0,"gnni",0,"godiva",0,"golda",0,"goldarina",0,"goldi",0,"goldia",0,"goldie",0,"goldina",0,"goldy",0,"grace",0,"gracia",0,"gracie",0,"grata",0,"gratia",0,"gratiana",0,"gray",0,"grayce",0,"grazia",0,"greer",0,"greta",0,"gretal",0,"gretchen",0,"grete",0,"gretel",0,"grethel",0,"gretna",0,"gretta",0,"grier",0,"griselda",0,"grissel",0,"guendolen",0,"guenevere",0,"guenna",0,"guglielma",0,"gui",0,"guillema",0,"guillemette",0,"guinevere",0,"guinna",0,"gunilla",0,"gus",0,"gusella",0,"gussi",0,"gussie",0,"gussy",0,"gusta",0,"gusti",0,"gustie",0,"gusty",0,"gwen",0,"gwendolen",0,"gwendolin",0,"gwendolyn",0,"gweneth",0,"gwenette",0,"gwenneth",0,"gwenni",0,"gwennie",0,"gwenny",0,"gwenora",0,"gwenore",0,"gwyn",0,"gwyneth",0,"gwynne",0,"gypsy",0,"hadria",0,"hailee",0,"haily",0,"haleigh",0,"halette",0,"haley",0,"hali",0,"halie",0,"halimeda",0,"halley",0,"halli",0,"hallie",0,"hally",0,"hana",0,"hanna",0,"hannah",0,"hanni",0,"hannie",0,"hannis",0,"hanny",0,"happy",0,"harlene",0,"harley",0,"harli",0,"harlie",0,"harmonia",0,"harmonie",0,"harmony",0,"harri",0,"harrie",0,"harriet",0,"harriett",0,"harrietta",0,"harriette",0,"harriot",0,"harriott",0,"hatti",0,"hattie",0,"hatty",0,"hayley",0,"hazel",0,"heath",0,"heather",0,"heda",0,"hedda",0,"heddi",0,"heddie",0,"hedi",0,"hedvig",0,"hedvige",0,"hedwig",0,"hedwiga",0,"hedy",0,"heida",0,"heidi",0,"heidie",0,"helaina",0,"helaine",0,"helen",0,"helen-elizabeth",0,"helena",0,"helene",0,"helenelizabeth",0,"helenka",0,"helga",0,"helge",0,"helli",0,"heloise",0,"helsa",0,"helyn",0,"hendrika",0,"henka",0,"henrie",0,"henrieta",0,"henrietta",0,"henriette",0,"henryetta",0,"hephzibah",0,"hermia",0,"hermina",0,"hermine",0,"herminia",0,"hermione",0,"herta",0,"hertha",0,"hester",0,"hesther",0,"hestia",0,"hetti",0,"hettie",0,"hetty",0,"hilary",0,"hilda",0,"hildagard",0,"hildagarde",0,"hilde",0,"hildegaard",0,"hildegarde",0,"hildy",0,"hillary",0,"hilliary",0,"hinda",0,"holli",0,"hollie",0,"holly",0,"holly-anne",0,"hollyanne",0,"honey",0,"honor",0,"honoria",0,"hope",0,"horatia",0,"hortense",0,"hortensia",0,"hulda",0,"hyacinth",0,"hyacintha",0,"hyacinthe",0,"hyacinthia",0,"hyacinthie",0,"hynda",0,"ianthe",0,"ibbie",0,"ibby",0,"ida",0,"idalia",0,"idalina",0,"idaline",0,"idell",0,"idelle",0,"idette",0,"ileana",0,"ileane",0,"ilene",0,"ilise",0,"ilka",0,"illa",0,"ilsa",0,"ilse",0,"ilysa",0,"ilyse",0,"ilyssa",0,"imelda",0,"imogen",0,"imogene",0,"imojean",0,"ina",0,"indira",0,"ines",0,"inesita",0,"inessa",0,"inez",0,"inga",0,"ingaberg",0,"ingaborg",0,"inge",0,"ingeberg",0,"ingeborg",0,"inger",0,"ingrid",0,"ingunna",0,"inna",0,"iolande",0,"iolanthe",0,"iona",0,"iormina",0,"ira",0,"irena",0,"irene",0,"irina",0,"iris",0,"irita",0,"irma",0,"isa",0,"isabeau",0,"isabel",0,"isabelita",0,"isabella",0,"isabelle",0,"isadora",0,"isahella",0,"iseabal",0,"isidora",0,"isis",0,"isobel",0,"issi",0,"issie",0,"issy",0,"ivett",0,"ivette",0,"ivie",0,"ivonne",0,"ivory",0,"ivy",0,"izabel",0,"jacenta",0,"jacinda",0,"jacinta",0,"jacintha",0,"jacinthe",0,"jackelyn",0,"jacki",0,"jackie",0,"jacklin",0,"jacklyn",0,"jackquelin",0,"jackqueline",0,"jacky",0,"jaclin",0,"jaclyn",0,"jacquelin",0,"jacqueline",0,"jacquelyn",0,"jacquelynn",0,"jacquenetta",0,"jacquenette",0,"jacquetta",0,"jacquette",0,"jacqui",0,"jacquie",0,"jacynth",0,"jada",0,"jade",0,"jaime",0,"jaimie",0,"jaine",0,"jami",0,"jamie",0,"jamima",0,"jammie",0,"jan",0,"jana",0,"janaya",0,"janaye",0,"jandy",0,"jane",0,"janean",0,"janeczka",0,"janeen",0,"janel",0,"janela",0,"janella",0,"janelle",0,"janene",0,"janenna",0,"janessa",0,"janet",0,"janeta",0,"janetta",0,"janette",0,"janeva",0,"janey",0,"jania",0,"janice",0,"janie",0,"janifer",0,"janina",0,"janine",0,"janis",0,"janith",0,"janka",0,"janna",0,"jannel",0,"jannelle",0,"janot",0,"jany",0,"jaquelin",0,"jaquelyn",0,"jaquenetta",0,"jaquenette",0,"jaquith",0,"jasmin",0,"jasmina",0,"jasmine",0,"jayme",0,"jaymee",0,"jayne",0,"jaynell",0,"jazmin",0,"jean",0,"jeana",0,"jeane",0,"jeanelle",0,"jeanette",0,"jeanie",0,"jeanine",0,"jeanna",0,"jeanne",0,"jeannette",0,"jeannie",0,"jeannine",0,"jehanna",0,"jelene",0,"jemie",0,"jemima",0,"jemimah",0,"jemmie",0,"jemmy",0,"jen",0,"jena",0,"jenda",0,"jenelle",0,"jeni",0,"jenica",0,"jeniece",0,"jenifer",0,"jeniffer",0,"jenilee",0,"jenine",0,"jenn",0,"jenna",0,"jennee",0,"jennette",0,"jenni",0,"jennica",0,"jennie",0,"jennifer",0,"jennilee",0,"jennine",0,"jenny",0,"jeralee",0,"jere",0,"jeri",0,"jermaine",0,"jerrie",0,"jerrilee",0,"jerrilyn",0,"jerrine",0,"jerry",0,"jerrylee",0,"jess",0,"jessa",0,"jessalin",0,"jessalyn",0,"jessamine",0,"jessamyn",0,"jesse",0,"jesselyn",0,"jessi",0,"jessica",0,"jessie",0,"jessika",0,"jessy",0,"jewel",0,"jewell",0,"jewelle",0,"jill",0,"jillana",0,"jillane",0,"jillayne",0,"jilleen",0,"jillene",0,"jilli",0,"jillian",0,"jillie",0,"jilly",0,"jinny",0,"jo",0,"jo ann",0,"jo-ann",0,"jo-anne",0,"joan",0,"joana",0,"joane",0,"joanie",0,"joann",0,"joanna",0,"joanne",0,"joannes",0,"jobey",0,"jobi",0,"jobie",0,"jobina",0,"joby",0,"jobye",0,"jobyna",0,"jocelin",0,"joceline",0,"jocelyn",0,"jocelyne",0,"jodee",0,"jodi",0,"jodie",0,"jody",0,"joeann",0,"joela",0,"joelie",0,"joell",0,"joella",0,"joelle",0,"joellen",0,"joelly",0,"joellyn",0,"joelynn",0,"joete",0,"joey",0,"johanna",0,"johannah",0,"johna",0,"johnath",0,"johnette",0,"johnna",0,"joice",0,"jojo",0,"jolee",0,"joleen",0,"jolene",0,"joletta",0,"joli",0,"jolie",0,"joline",0,"joly",0,"jolyn",0,"jolynn",0,"jonell",0,"joni",0,"jonie",0,"jonis",0,"jordain",0,"jordan",0,"jordana",0,"jordanna",0,"jorey",0,"jori",0,"jorie",0,"jorrie",0,"jorry",0,"joscelin",0,"josee",0,"josefa",0,"josefina",0,"josepha",0,"josephina",0,"josephine",0,"josey",0,"josi",0,"josie",0,"josselyn",0,"josy",0,"jourdan",0,"joy",0,"joya",0,"joyan",0,"joyann",0,"joyce",0,"joycelin",0,"joye",0,"joyous",0,"jsandye",0,"juana",0,"juanita",0,"judi",0,"judie",0,"judith",0,"juditha",0,"judy",0,"judye",0,"juieta",0,"julee",0,"juli",0,"julia",0,"juliana",0,"juliane",0,"juliann",0,"julianna",0,"julianne",0,"julie",0,"julienne",0,"juliet",0,"julieta",0,"julietta",0,"juliette",0,"julina",0,"juline",0,"julissa",0,"julita",0,"june",0,"junette",0,"junia",0,"junie",0,"junina",0,"justina",0,"justine",0,"justinn",0,"jyoti",0,"kacey",0,"kacie",0,"kacy",0,"kaela",0,"kai",0,"kaia",0,"kaila",0,"kaile",0,"kailey",0,"kaitlin",0,"kaitlyn",0,"kaitlynn",0,"kaja",0,"kakalina",0,"kala",0,"kaleena",0,"kali",0,"kalie",0,"kalila",0,"kalina",0,"kalinda",0,"kalindi",0,"kalli",0,"kally",0,"kameko",0,"kamila",0,"kamilah",0,"kamillah",0,"kandace",0,"kandy",0,"kania",0,"kanya",0,"kara",0,"kara-lynn",0,"karalee",0,"karalynn",0,"kare",0,"karee",0,"karel",0,"karen",0,"karena",0,"kari",0,"karia",0,"karie",0,"karil",0,"karilynn",0,"karin",0,"karina",0,"karine",0,"kariotta",0,"karisa",0,"karissa",0,"karita",0,"karla",0,"karlee",0,"karleen",0,"karlen",0,"karlene",0,"karlie",0,"karlotta",0,"karlotte",0,"karly",0,"karlyn",0,"karmen",0,"karna",0,"karol",0,"karola",0,"karole",0,"karolina",0,"karoline",0,"karoly",0,"karon",0,"karrah",0,"karrie",0,"karry",0,"kary",0,"karyl",0,"karylin",0,"karyn",0,"kasey",0,"kass",0,"kassandra",0,"kassey",0,"kassi",0,"kassia",0,"kassie",0,"kat",0,"kata",0,"katalin",0,"kate",0,"katee",0,"katerina",0,"katerine",0,"katey",0,"kath",0,"katha",0,"katharina",0,"katharine",0,"katharyn",0,"kathe",0,"katherina",0,"katherine",0,"katheryn",0,"kathi",0,"kathie",0,"kathleen",0,"kathlin",0,"kathrine",0,"kathryn",0,"kathryne",0,"kathy",0,"kathye",0,"kati",0,"katie",0,"katina",0,"katine",0,"katinka",0,"katleen",0,"katlin",0,"katrina",0,"katrine",0,"katrinka",0,"katti",0,"kattie",0,"katuscha",0,"katusha",0,"katy",0,"katya",0,"kay",0,"kaycee",0,"kaye",0,"kayla",0,"kayle",0,"kaylee",0,"kayley",0,"kaylil",0,"kaylyn",0,"keeley",0,"keelia",0,"keely",0,"kelcey",0,"kelci",0,"kelcie",0,"kelcy",0,"kelila",0,"kellen",0,"kelley",0,"kelli",0,"kellia",0,"kellie",0,"kellina",0,"kellsie",0,"kelly",0,"kellyann",0,"kelsey",0,"kelsi",0,"kelsy",0,"kendra",0,"kendre",0,"kenna",0,"keri",0,"keriann",0,"kerianne",0,"kerri",0,"kerrie",0,"kerrill",0,"kerrin",0,"kerry",0,"kerstin",0,"kesley",0,"keslie",0,"kessia",0,"kessiah",0,"ketti",0,"kettie",0,"ketty",0,"kevina",0,"kevyn",0,"ki",0,"kiah",0,"kial",0,"kiele",0,"kiersten",0,"kikelia",0,"kiley",0,"kim",0,"kimberlee",0,"kimberley",0,"kimberli",0,"kimberly",0,"kimberlyn",0,"kimbra",0,"kimmi",0,"kimmie",0,"kimmy",0,"kinna",0,"kip",0,"kipp",0,"kippie",0,"kippy",0,"kira",0,"kirbee",0,"kirbie",0,"kirby",0,"kiri",0,"kirsten",0,"kirsteni",0,"kirsti",0,"kirstin",0,"kirstyn",0,"kissee",0,"kissiah",0,"kissie",0,"kit",0,"kitti",0,"kittie",0,"kitty",0,"kizzee",0,"kizzie",0,"klara",0,"klarika",0,"klarrisa",0,"konstance",0,"konstanze",0,"koo",0,"kora",0,"koral",0,"koralle",0,"kordula",0,"kore",0,"korella",0,"koren",0,"koressa",0,"kori",0,"korie",0,"korney",0,"korrie",0,"korry",0,"kris",0,"krissie",0,"krissy",0,"krista",0,"kristal",0,"kristan",0,"kriste",0,"kristel",0,"kristen",0,"kristi",0,"kristien",0,"kristin",0,"kristina",0,"kristine",0,"kristy",0,"kristyn",0,"krysta",0,"krystal",0,"krystalle",0,"krystle",0,"krystyna",0,"kyla",0,"kyle",0,"kylen",0,"kylie",0,"kylila",0,"kylynn",0,"kym",0,"kynthia",0,"kyrstin",0,"l;urette",0,"la verne",0,"lacee",0,"lacey",0,"lacie",0,"lacy",0,"ladonna",0,"laetitia",0,"laina",0,"lainey",0,"lana",0,"lanae",0,"lane",0,"lanette",0,"laney",0,"lani",0,"lanie",0,"lanita",0,"lanna",0,"lanni",0,"lanny",0,"lara",0,"laraine",0,"lari",0,"larina",0,"larine",0,"larisa",0,"larissa",0,"lark",0,"laryssa",0,"latashia",0,"latia",0,"latisha",0,"latrena",0,"latrina",0,"laura",0,"lauraine",0,"laural",0,"lauralee",0,"laure",0,"lauree",0,"laureen",0,"laurel",0,"laurella",0,"lauren",0,"laurena",0,"laurene",0,"lauretta",0,"laurette",0,"lauri",0,"laurianne",0,"laurice",0,"laurie",0,"lauryn",0,"lavena",0,"laverna",0,"laverne",0,"lavina",0,"lavinia",0,"lavinie",0,"layla",0,"layne",0,"layney",0,"lea",0,"leah",0,"leandra",0,"leann",0,"leanna",0,"leanor",0,"leanora",0,"lebbie",0,"leda",0,"lee",0,"leeann",0,"leeanne",0,"leela",0,"leelah",0,"leena",0,"leesa",0,"leese",0,"legra",0,"leia",0,"leigh",0,"leigha",0,"leila",0,"leilah",0,"leisha",0,"lela",0,"lelah",0,"leland",0,"lelia",0,"lena",0,"lenee",0,"lenette",0,"lenka",0,"lenna",0,"lenora",0,"lenore",0,"leodora",0,"leoine",0,"leola",0,"leoline",0,"leona",0,"leonanie",0,"leone",0,"leonelle",0,"leonie",0,"leonora",0,"leonore",0,"leontine",0,"leontyne",0,"leora",0,"leshia",0,"lesley",0,"lesli",0,"leslie",0,"lesly",0,"lesya",0,"leta",0,"lethia",0,"leticia",0,"letisha",0,"letitia",0,"letizia",0,"letta",0,"letti",0,"lettie",0,"letty",0,"lexi",0,"lexie",0,"lexine",0,"lexis",0,"lexy",0,"leyla",0,"lezlie",0,"lia",0,"lian",0,"liana",0,"liane",0,"lianna",0,"lianne",0,"lib",0,"libbey",0,"libbi",0,"libbie",0,"libby",0,"licha",0,"lida",0,"lidia",0,"liesa",0,"lil",0,"lila",0,"lilah",0,"lilas",0,"lilia",0,"lilian",0,"liliane",0,"lilias",0,"lilith",0,"lilla",0,"lilli",0,"lillian",0,"lillis",0,"lilllie",0,"lilly",0,"lily",0,"lilyan",0,"lin",0,"lina",0,"lind",0,"linda",0,"lindi",0,"lindie",0,"lindsay",0,"lindsey",0,"lindsy",0,"lindy",0,"linea",0,"linell",0,"linet",0,"linette",0,"linn",0,"linnea",0,"linnell",0,"linnet",0,"linnie",0,"linzy",0,"lira",0,"lisa",0,"lisabeth",0,"lisbeth",0,"lise",0,"lisetta",0,"lisette",0,"lisha",0,"lishe",0,"lissa",0,"lissi",0,"lissie",0,"lissy",0,"lita",0,"liuka",0,"liv",0,"liva",0,"livia",0,"livvie",0,"livvy",0,"livvyy",0,"livy",0,"liz",0,"liza",0,"lizabeth",0,"lizbeth",0,"lizette",0,"lizzie",0,"lizzy",0,"loella",0,"lois",0,"loise",0,"lola",0,"loleta",0,"lolita",0,"lolly",0,"lona",0,"lonee",0,"loni",0,"lonna",0,"lonni",0,"lonnie",0,"lora",0,"lorain",0,"loraine",0,"loralee",0,"loralie",0,"loralyn",0,"loree",0,"loreen",0,"lorelei",0,"lorelle",0,"loren",0,"lorena",0,"lorene",0,"lorenza",0,"loretta",0,"lorettalorna",0,"lorette",0,"lori",0,"loria",0,"lorianna",0,"lorianne",0,"lorie",0,"lorilee",0,"lorilyn",0,"lorinda",0,"lorine",0,"lorita",0,"lorna",0,"lorne",0,"lorraine",0,"lorrayne",0,"lorri",0,"lorrie",0,"lorrin",0,"lorry",0,"lory",0,"lotta",0,"lotte",0,"lotti",0,"lottie",0,"lotty",0,"lou",0,"louella",0,"louisa",0,"louise",0,"louisette",0,"loutitia",0,"lu",0,"luce",0,"luci",0,"lucia",0,"luciana",0,"lucie",0,"lucienne",0,"lucila",0,"lucilia",0,"lucille",0,"lucina",0,"lucinda",0,"lucine",0,"lucita",0,"lucky",0,"lucretia",0,"lucy",0,"ludovika",0,"luella",0,"luelle",0,"luisa",0,"luise",0,"lula",0,"lulita",0,"lulu",0,"lura",0,"lurette",0,"lurleen",0,"lurlene",0,"lurline",0,"lusa",0,"luz",0,"lyda",0,"lydia",0,"lydie",0,"lyn",0,"lynda",0,"lynde",0,"lyndel",0,"lyndell",0,"lyndsay",0,"lyndsey",0,"lyndsie",0,"lyndy",0,"lynea",0,"lynelle",0,"lynett",0,"lynette",0,"lynn",0,"lynna",0,"lynne",0,"lynnea",0,"lynnell",0,"lynnelle",0,"lynnet",0,"lynnett",0,"lynnette",0,"lynsey",0,"lyssa",0,"mab",0,"mabel",0,"mabelle",0,"mable",0,"mada",0,"madalena",0,"madalyn",0,"maddalena",0,"maddi",0,"maddie",0,"maddy",0,"madel",0,"madelaine",0,"madeleine",0,"madelena",0,"madelene",0,"madelin",0,"madelina",0,"madeline",0,"madella",0,"madelle",0,"madelon",0,"madelyn",0,"madge",0,"madlen",0,"madlin",0,"madonna",0,"mady",0,"mae",0,"maegan",0,"mag",0,"magda",0,"magdaia",0,"magdalen",0,"magdalena",0,"magdalene",0,"maggee",0,"maggi",0,"maggie",0,"maggy",0,"mahala",0,"mahalia",0,"maia",0,"maible",0,"maiga",0,"maighdiln",0,"mair",0,"maire",0,"maisey",0,"maisie",0,"maitilde",0,"mala",0,"malanie",0,"malena",0,"malia",0,"malina",0,"malinda",0,"malinde",0,"malissa",0,"malissia",0,"mallissa",0,"mallorie",0,"mallory",0,"malorie",0,"malory",0,"malva",0,"malvina",0,"malynda",0,"mame",0,"mamie",0,"manda",0,"mandi",0,"mandie",0,"mandy",0,"manon",0,"manya",0,"mara",0,"marabel",0,"marcela",0,"marcelia",0,"marcella",0,"marcelle",0,"marcellina",0,"marcelline",0,"marchelle",0,"marci",0,"marcia",0,"marcie",0,"marcile",0,"marcille",0,"marcy",0,"mareah",0,"maren",0,"marena",0,"maressa",0,"marga",0,"margalit",0,"margalo",0,"margaret",0,"margareta",0,"margarete",0,"margaretha",0,"margarethe",0,"margaretta",0,"margarette",0,"margarita",0,"margaux",0,"marge",0,"margeaux",0,"margery",0,"marget",0,"margette",0,"margi",0,"margie",0,"margit",0,"margo",0,"margot",0,"margret",0,"marguerite",0,"margy",0,"mari",0,"maria",0,"mariam",0,"marian",0,"mariana",0,"mariann",0,"marianna",0,"marianne",0,"maribel",0,"maribelle",0,"maribeth",0,"marice",0,"maridel",0,"marie",0,"marie-ann",0,"marie-jeanne",0,"marieann",0,"mariejeanne",0,"mariel",0,"mariele",0,"marielle",0,"mariellen",0,"marietta",0,"mariette",0,"marigold",0,"marijo",0,"marika",0,"marilee",0,"marilin",0,"marillin",0,"marilyn",0,"marin",0,"marina",0,"marinna",0,"marion",0,"mariquilla",0,"maris",0,"marisa",0,"mariska",0,"marissa",0,"marita",0,"maritsa",0,"mariya",0,"marj",0,"marja",0,"marje",0,"marji",0,"marjie",0,"marjorie",0,"marjory",0,"marjy",0,"marketa",0,"marla",0,"marlane",0,"marleah",0,"marlee",0,"marleen",0,"marlena",0,"marlene",0,"marley",0,"marlie",0,"marline",0,"marlo",0,"marlyn",0,"marna",0,"marne",0,"marney",0,"marni",0,"marnia",0,"marnie",0,"marquita",0,"marrilee",0,"marris",0,"marrissa",0,"marsha",0,"marsiella",0,"marta",0,"martelle",0,"martguerita",0,"martha",0,"marthe",0,"marthena",0,"marti",0,"martica",0,"martie",0,"martina",0,"martita",0,"marty",0,"martynne",0,"mary",0,"marya",0,"maryann",0,"maryanna",0,"maryanne",0,"marybelle",0,"marybeth",0,"maryellen",0,"maryjane",0,"maryjo",0,"maryl",0,"marylee",0,"marylin",0,"marylinda",0,"marylou",0,"marylynne",0,"maryrose",0,"marys",0,"marysa",0,"masha",0,"matelda",0,"mathilda",0,"mathilde",0,"matilda",0,"matilde",0,"matti",0,"mattie",0,"matty",0,"maud",0,"maude",0,"maudie",0,"maura",0,"maure",0,"maureen",0,"maureene",0,"maurene",0,"maurine",0,"maurise",0,"maurita",0,"maurizia",0,"mavis",0,"mavra",0,"max",0,"maxi",0,"maxie",0,"maxine",0,"maxy",0,"may",0,"maybelle",0,"maye",0,"mead",0,"meade",0,"meagan",0,"meaghan",0,"meara",0,"mechelle",0,"meg",0,"megan",0,"megen",0,"meggi",0,"meggie",0,"meggy",0,"meghan",0,"meghann",0,"mehetabel",0,"mei",0,"mel",0,"mela",0,"melamie",0,"melania",0,"melanie",0,"melantha",0,"melany",0,"melba",0,"melesa",0,"melessa",0,"melicent",0,"melina",0,"melinda",0,"melinde",0,"melisa",0,"melisande",0,"melisandra",0,"melisenda",0,"melisent",0,"melissa",0,"melisse",0,"melita",0,"melitta",0,"mella",0,"melli",0,"mellicent",0,"mellie",0,"mellisa",0,"mellisent",0,"melloney",0,"melly",0,"melodee",0,"melodie",0,"melody",0,"melonie",0,"melony",0,"melosa",0,"melva",0,"mercedes",0,"merci",0,"mercie",0,"mercy",0,"meredith",0,"meredithe",0,"meridel",0,"meridith",0,"meriel",0,"merilee",0,"merilyn",0,"meris",0,"merissa",0,"merl",0,"merla",0,"merle",0,"merlina",0,"merline",0,"merna",0,"merola",0,"merralee",0,"merridie",0,"merrie",0,"merrielle",0,"merrile",0,"merrilee",0,"merrili",0,"merrill",0,"merrily",0,"merry",0,"mersey",0,"meryl",0,"meta",0,"mia",0,"micaela",0,"michaela",0,"michaelina",0,"michaeline",0,"michaella",0,"michal",0,"michel",0,"michele",0,"michelina",0,"micheline",0,"michell",0,"michelle",0,"micki",0,"mickie",0,"micky",0,"midge",0,"mignon",0,"mignonne",0,"miguela",0,"miguelita",0,"mikaela",0,"mil",0,"mildred",0,"mildrid",0,"milena",0,"milicent",0,"milissent",0,"milka",0,"milli",0,"millicent",0,"millie",0,"millisent",0,"milly",0,"milzie",0,"mimi",0,"min",0,"mina",0,"minda",0,"mindy",0,"minerva",0,"minetta",0,"minette",0,"minna",0,"minnaminnie",0,"minne",0,"minni",0,"minnie",0,"minny",0,"minta",0,"miof mela",0,"miquela",0,"mira",0,"mirabel",0,"mirabella",0,"mirabelle",0,"miran",0,"miranda",0,"mireielle",0,"mireille",0,"mirella",0,"mirelle",0,"miriam",0,"mirilla",0,"mirna",0,"misha",0,"missie",0,"missy",0,"misti",0,"misty",0,"mitzi",0,"modesta",0,"modestia",0,"modestine",0,"modesty",0,"moina",0,"moira",0,"moll",0,"mollee",0,"molli",0,"mollie",0,"molly",0,"mommy",0,"mona",0,"monah",0,"monica",0,"monika",0,"monique",0,"mora",0,"moreen",0,"morena",0,"morgan",0,"morgana",0,"morganica",0,"morganne",0,"morgen",0,"moria",0,"morissa",0,"morna",0,"moselle",0,"moyna",0,"moyra",0,"mozelle",0,"muffin",0,"mufi",0,"mufinella",0,"muire",0,"mureil",0,"murial",0,"muriel",0,"murielle",0,"myra",0,"myrah",0,"myranda",0,"myriam",0,"myrilla",0,"myrle",0,"myrlene",0,"myrna",0,"myrta",0,"myrtia",0,"myrtice",0,"myrtie",0,"myrtle",0,"nada",0,"nadean",0,"nadeen",0,"nadia",0,"nadine",0,"nadiya",0,"nady",0,"nadya",0,"nalani",0,"nan",0,"nana",0,"nananne",0,"nance",0,"nancee",0,"nancey",0,"nanci",0,"nancie",0,"nancy",0,"nanete",0,"nanette",0,"nani",0,"nanice",0,"nanine",0,"nannette",0,"nanni",0,"nannie",0,"nanny",0,"nanon",0,"naoma",0,"naomi",0,"nara",0,"nari",0,"nariko",0,"nat",0,"nata",0,"natala",0,"natalee",0,"natalie",0,"natalina",0,"nataline",0,"natalya",0,"natasha",0,"natassia",0,"nathalia",0,"nathalie",0,"natividad",0,"natka",0,"natty",0,"neala",0,"neda",0,"nedda",0,"nedi",0,"neely",0,"neila",0,"neile",0,"neilla",0,"neille",0,"nelia",0,"nelie",0,"nell",0,"nelle",0,"nelli",0,"nellie",0,"nelly",0,"nerissa",0,"nerita",0,"nert",0,"nerta",0,"nerte",0,"nerti",0,"nertie",0,"nerty",0,"nessa",0,"nessi",0,"nessie",0,"nessy",0,"nesta",0,"netta",0,"netti",0,"nettie",0,"nettle",0,"netty",0,"nevsa",0,"neysa",0,"nichol",0,"nichole",0,"nicholle",0,"nicki",0,"nickie",0,"nicky",0,"nicol",0,"nicola",0,"nicole",0,"nicolea",0,"nicolette",0,"nicoli",0,"nicolina",0,"nicoline",0,"nicolle",0,"nikaniki",0,"nike",0,"niki",0,"nikki",0,"nikkie",0,"nikoletta",0,"nikolia",0,"nina",0,"ninetta",0,"ninette",0,"ninnetta",0,"ninnette",0,"ninon",0,"nissa",0,"nisse",0,"nissie",0,"nissy",0,"nita",0,"nixie",0,"noami",0,"noel",0,"noelani",0,"noell",0,"noella",0,"noelle",0,"noellyn",0,"noelyn",0,"noemi",0,"nola",0,"nolana",0,"nolie",0,"nollie",0,"nomi",0,"nona",0,"nonah",0,"noni",0,"nonie",0,"nonna",0,"nonnah",0,"nora",0,"norah",0,"norean",0,"noreen",0,"norene",0,"norina",0,"norine",0,"norma",0,"norri",0,"norrie",0,"norry",0,"novelia",0,"nydia",0,"nyssa",0,"octavia",0,"odele",0,"odelia",0,"odelinda",0,"odella",0,"odelle",0,"odessa",0,"odetta",0,"odette",0,"odilia",0,"odille",0,"ofelia",0,"ofella",0,"ofilia",0,"ola",0,"olenka",0,"olga",0,"olia",0,"olimpia",0,"olive",0,"olivette",0,"olivia",0,"olivie",0,"oliy",0,"ollie",0,"olly",0,"olva",0,"olwen",0,"olympe",0,"olympia",0,"olympie",0,"ondrea",0,"oneida",0,"onida",0,"oona",0,"opal",0,"opalina",0,"opaline",0,"ophelia",0,"ophelie",0,"ora",0,"oralee",0,"oralia",0,"oralie",0,"oralla",0,"oralle",0,"orel",0,"orelee",0,"orelia",0,"orelie",0,"orella",0,"orelle",0,"oriana",0,"orly",0,"orsa",0,"orsola",0,"ortensia",0,"otha",0,"othelia",0,"othella",0,"othilia",0,"othilie",0,"ottilie",0,"page",0,"paige",0,"paloma",0,"pam",0,"pamela",0,"pamelina",0,"pamella",0,"pammi",0,"pammie",0,"pammy",0,"pandora",0,"pansie",0,"pansy",0,"paola",0,"paolina",0,"papagena",0,"pat",0,"patience",0,"patrica",0,"patrice",0,"patricia",0,"patrizia",0,"patsy",0,"patti",0,"pattie",0,"patty",0,"paula",0,"paule",0,"pauletta",0,"paulette",0,"pauli",0,"paulie",0,"paulina",0,"pauline",0,"paulita",0,"pauly",0,"pavia",0,"pavla",0,"pearl",0,"pearla",0,"pearle",0,"pearline",0,"peg",0,"pegeen",0,"peggi",0,"peggie",0,"peggy",0,"pen",0,"penelopa",0,"penelope",0,"penni",0,"pennie",0,"penny",0,"pepi",0,"pepita",0,"peri",0,"peria",0,"perl",0,"perla",0,"perle",0,"perri",0,"perrine",0,"perry",0,"persis",0,"pet",0,"peta",0,"petra",0,"petrina",0,"petronella",0,"petronia",0,"petronilla",0,"petronille",0,"petunia",0,"phaedra",0,"phaidra",0,"phebe",0,"phedra",0,"phelia",0,"phil",0,"philipa",0,"philippa",0,"philippe",0,"philippine",0,"philis",0,"phillida",0,"phillie",0,"phillis",0,"philly",0,"philomena",0,"phoebe",0,"phylis",0,"phyllida",0,"phyllis",0,"phyllys",0,"phylys",0,"pia",0,"pier",0,"pierette",0,"pierrette",0,"pietra",0,"piper",0,"pippa",0,"pippy",0,"polly",0,"pollyanna",0,"pooh",0,"poppy",0,"portia",0,"pris",0,"prisca",0,"priscella",0,"priscilla",0,"prissie",0,"pru",0,"prudence",0,"prudi",0,"prudy",0,"prue",0,"queenie",0,"quentin",0,"querida",0,"quinn",0,"quinta",0,"quintana",0,"quintilla",0,"quintina",0,"rachael",0,"rachel",0,"rachele",0,"rachelle",0,"rae",0,"raeann",0,"raf",0,"rafa",0,"rafaela",0,"rafaelia",0,"rafaelita",0,"rahal",0,"rahel",0,"raina",0,"raine",0,"rakel",0,"ralina",0,"ramona",0,"ramonda",0,"rana",0,"randa",0,"randee",0,"randene",0,"randi",0,"randie",0,"randy",0,"ranee",0,"rani",0,"rania",0,"ranice",0,"ranique",0,"ranna",0,"raphaela",0,"raquel",0,"raquela",0,"rasia",0,"rasla",0,"raven",0,"ray",0,"raychel",0,"raye",0,"rayna",0,"raynell",0,"rayshell",0,"rea",0,"reba",0,"rebbecca",0,"rebe",0,"rebeca",0,"rebecca",0,"rebecka",0,"rebeka",0,"rebekah",0,"rebekkah",0,"ree",0,"reeba",0,"reena",0,"reeta",0,"reeva",0,"regan",0,"reggi",0,"reggie",0,"regina",0,"regine",0,"reiko",0,"reina",0,"reine",0,"remy",0,"rena",0,"renae",0,"renata",0,"renate",0,"rene",0,"renee",0,"renell",0,"renelle",0,"renie",0,"rennie",0,"reta",0,"retha",0,"revkah",0,"rey",0,"reyna",0,"rhea",0,"rheba",0,"rheta",0,"rhetta",0,"rhiamon",0,"rhianna",0,"rhianon",0,"rhoda",0,"rhodia",0,"rhodie",0,"rhody",0,"rhona",0,"rhonda",0,"riane",0,"riannon",0,"rianon",0,"rica",0,"ricca",0,"rici",0,"ricki",0,"rickie",0,"ricky",0,"riki",0,"rikki",0,"rina",0,"risa",0,"rita",0,"riva",0,"rivalee",0,"rivi",0,"rivkah",0,"rivy",0,"roana",0,"roanna",0,"roanne",0,"robbi",0,"robbie",0,"robbin",0,"robby",0,"robbyn",0,"robena",0,"robenia",0,"roberta",0,"robin",0,"robina",0,"robinet",0,"robinett",0,"robinetta",0,"robinette",0,"robinia",0,"roby",0,"robyn",0,"roch",0,"rochell",0,"rochella",0,"rochelle",0,"rochette",0,"roda",0,"rodi",0,"rodie",0,"rodina",0,"rois",0,"romola",0,"romona",0,"romonda",0,"romy",0,"rona",0,"ronalda",0,"ronda",0,"ronica",0,"ronna",0,"ronni",0,"ronnica",0,"ronnie",0,"ronny",0,"roobbie",0,"rora",0,"rori",0,"rorie",0,"rory",0,"ros",0,"rosa",0,"rosabel",0,"rosabella",0,"rosabelle",0,"rosaleen",0,"rosalia",0,"rosalie",0,"rosalind",0,"rosalinda",0,"rosalinde",0,"rosaline",0,"rosalyn",0,"rosalynd",0,"rosamond",0,"rosamund",0,"rosana",0,"rosanna",0,"rosanne",0,"rose",0,"roseann",0,"roseanna",0,"roseanne",0,"roselia",0,"roselin",0,"roseline",0,"rosella",0,"roselle",0,"rosemaria",0,"rosemarie",0,"rosemary",0,"rosemonde",0,"rosene",0,"rosetta",0,"rosette",0,"roshelle",0,"rosie",0,"rosina",0,"rosita",0,"roslyn",0,"rosmunda",0,"rosy",0,"row",0,"rowe",0,"rowena",0,"roxana",0,"roxane",0,"roxanna",0,"roxanne",0,"roxi",0,"roxie",0,"roxine",0,"roxy",0,"roz",0,"rozalie",0,"rozalin",0,"rozamond",0,"rozanna",0,"rozanne",0,"roze",0,"rozele",0,"rozella",0,"rozelle",0,"rozina",0,"rubetta",0,"rubi",0,"rubia",0,"rubie",0,"rubina",0,"ruby",0,"ruperta",0,"ruth",0,"ruthann",0,"ruthanne",0,"ruthe",0,"ruthi",0,"ruthie",0,"ruthy",0,"ryann",0,"rycca",0,"saba",0,"sabina",0,"sabine",0,"sabra",0,"sabrina",0,"sacha",0,"sada",0,"sadella",0,"sadie",0,"sadye",0,"saidee",0,"sal",0,"salaidh",0,"sallee",0,"salli",0,"sallie",0,"sally",0,"sallyann",0,"sallyanne",0,"saloma",0,"salome",0,"salomi",0,"sam",0,"samantha",0,"samara",0,"samaria",0,"sammy",0,"sande",0,"sandi",0,"sandie",0,"sandra",0,"sandy",0,"sandye",0,"sapphira",0,"sapphire",0,"sara",0,"sara-ann",0,"saraann",0,"sarah",0,"sarajane",0,"saree",0,"sarena",0,"sarene",0,"sarette",0,"sari",0,"sarina",0,"sarine",0,"sarita",0,"sascha",0,"sasha",0,"sashenka",0,"saudra",0,"saundra",0,"savina",0,"sayre",0,"scarlet",0,"scarlett",0,"sean",0,"seana",0,"seka",0,"sela",0,"selena",0,"selene",0,"selestina",0,"selia",0,"selie",0,"selina",0,"selinda",0,"seline",0,"sella",0,"selle",0,"selma",0,"sena",0,"sephira",0,"serena",0,"serene",0,"shae",0,"shaina",0,"shaine",0,"shalna",0,"shalne",0,"shana",0,"shanda",0,"shandee",0,"shandeigh",0,"shandie",0,"shandra",0,"shandy",0,"shane",0,"shani",0,"shanie",0,"shanna",0,"shannah",0,"shannen",0,"shannon",0,"shanon",0,"shanta",0,"shantee",0,"shara",0,"sharai",0,"shari",0,"sharia",0,"sharity",0,"sharl",0,"sharla",0,"sharleen",0,"sharlene",0,"sharline",0,"sharon",0,"sharona",0,"sharron",0,"sharyl",0,"shaun",0,"shauna",0,"shawn",0,"shawna",0,"shawnee",0,"shay",0,"shayla",0,"shaylah",0,"shaylyn",0,"shaylynn",0,"shayna",0,"shayne",0,"shea",0,"sheba",0,"sheela",0,"sheelagh",0,"sheelah",0,"sheena",0,"sheeree",0,"sheila",0,"sheila-kathryn",0,"sheilah",0,"sheilakathryn",0,"shel",0,"shela",0,"shelagh",0,"shelba",0,"shelbi",0,"shelby",0,"shelia",0,"shell",0,"shelley",0,"shelli",0,"shellie",0,"shelly",0,"shena",0,"sher",0,"sheree",0,"sheri",0,"sherie",0,"sherill",0,"sherilyn",0,"sherline",0,"sherri",0,"sherrie",0,"sherry",0,"sherye",0,"sheryl",0,"shina",0,"shir",0,"shirl",0,"shirlee",0,"shirleen",0,"shirlene",0,"shirley",0,"shirline",0,"shoshana",0,"shoshanna",0,"siana",0,"sianna",0,"sib",0,"sibbie",0,"sibby",0,"sibeal",0,"sibel",0,"sibella",0,"sibelle",0,"sibilla",0,"sibley",0,"sibyl",0,"sibylla",0,"sibylle",0,"sidoney",0,"sidonia",0,"sidonnie",0,"sigrid",0,"sile",0,"sileas",0,"silva",0,"silvana",0,"silvia",0,"silvie",0,"simona",0,"simone",0,"simonette",0,"simonne",0,"sindee",0,"siobhan",0,"sioux",0,"siouxie",0,"sisely",0,"sisile",0,"sissie",0,"sissy",0,"siusan",0,"sofia",0,"sofie",0,"sondra",0,"sonia",0,"sonja",0,"sonni",0,"sonnie",0,"sonnnie",0,"sonny",0,"sonya",0,"sophey",0,"sophi",0,"sophia",0,"sophie",0,"sophronia",0,"sorcha",0,"sosanna",0,"stace",0,"stacee",0,"stacey",0,"staci",0,"stacia",0,"stacie",0,"stacy",0,"stafani",0,"star",0,"starla",0,"starlene",0,"starlin",0,"starr",0,"stefa",0,"stefania",0,"stefanie",0,"steffane",0,"steffi",0,"steffie",0,"stella",0,"stepha",0,"stephana",0,"stephani",0,"stephanie",0,"stephannie",0,"stephenie",0,"stephi",0,"stephie",0,"stephine",0,"stesha",0,"stevana",0,"stevena",0,"stoddard",0,"storm",0,"stormi",0,"stormie",0,"stormy",0,"sue",0,"suellen",0,"sukey",0,"suki",0,"sula",0,"sunny",0,"sunshine",0,"susan",0,"susana",0,"susanetta",0,"susann",0,"susanna",0,"susannah",0,"susanne",0,"susette",0,"susi",0,"susie",0,"susy",0,"suzann",0,"suzanna",0,"suzanne",0,"suzette",0,"suzi",0,"suzie",0,"suzy",0,"sybil",0,"sybila",0,"sybilla",0,"sybille",0,"sybyl",0,"sydel",0,"sydelle",0,"sydney",0,"sylvia",0,"tabatha",0,"tabbatha",0,"tabbi",0,"tabbie",0,"tabbitha",0,"tabby",0,"tabina",0,"tabitha",0,"taffy",0,"talia",0,"tallia",0,"tallie",0,"tallou",0,"tallulah",0,"tally",0,"talya",0,"talyah",0,"tamar",0,"tamara",0,"tamarah",0,"tamarra",0,"tamera",0,"tami",0,"tamiko",0,"tamma",0,"tammara",0,"tammi",0,"tammie",0,"tammy",0,"tamqrah",0,"tamra",0,"tana",0,"tandi",0,"tandie",0,"tandy",0,"tanhya",0,"tani",0,"tania",0,"tanitansy",0,"tansy",0,"tanya",0,"tara",0,"tarah",0,"tarra",0,"tarrah",0,"taryn",0,"tasha",0,"tasia",0,"tate",0,"tatiana",0,"tatiania",0,"tatum",0,"tawnya",0,"tawsha",0,"ted",0,"tedda",0,"teddi",0,"teddie",0,"teddy",0,"tedi",0,"tedra",0,"teena",0,"teirtza",0,"teodora",0,"tera",0,"teresa",0,"terese",0,"teresina",0,"teresita",0,"teressa",0,"teri",0,"teriann",0,"terra",0,"terri",0,"terri-jo",0,"terrie",0,"terrijo",0,"terry",0,"terrye",0,"tersina",0,"terza",0,"tess",0,"tessa",0,"tessi",0,"tessie",0,"tessy",0,"thalia",0,"thea",0,"theadora",0,"theda",0,"thekla",0,"thelma",0,"theo",0,"theodora",0,"theodosia",0,"theresa",0,"therese",0,"theresina",0,"theresita",0,"theressa",0,"therine",0,"thia",0,"thomasa",0,"thomasin",0,"thomasina",0,"thomasine",0,"tiena",0,"tierney",0,"tiertza",0,"tiff",0,"tiffani",0,"tiffanie",0,"tiffany",0,"tiffi",0,"tiffie",0,"tiffy",0,"tilda",0,"tildi",0,"tildie",0,"tildy",0,"tillie",0,"tilly",0,"tim",0,"timi",0,"timmi",0,"timmie",0,"timmy",0,"timothea",0,"tina",0,"tine",0,"tiphani",0,"tiphanie",0,"tiphany",0,"tish",0,"tisha",0,"tobe",0,"tobey",0,"tobi",0,"toby",0,"tobye",0,"toinette",0,"toma",0,"tomasina",0,"tomasine",0,"tomi",0,"tommi",0,"tommie",0,"tommy",0,"toni",0,"tonia",0,"tonie",0,"tony",0,"tonya",0,"tonye",0,"tootsie",0,"torey",0,"tori",0,"torie",0,"torrie",0,"tory",0,"tova",0,"tove",0,"tracee",0,"tracey",0,"traci",0,"tracie",0,"tracy",0,"trenna",0,"tresa",0,"trescha",0,"tressa",0,"tricia",0,"trina",0,"trish",0,"trisha",0,"trista",0,"trix",0,"trixi",0,"trixie",0,"trixy",0,"truda",0,"trude",0,"trudey",0,"trudi",0,"trudie",0,"trudy",0,"trula",0,"tuesday",0,"twila",0,"twyla",0,"tybi",0,"tybie",0,"tyne",0,"ula",0,"ulla",0,"ulrica",0,"ulrika",0,"ulrikaumeko",0,"ulrike",0,"umeko",0,"una",0,"ursa",0,"ursala",0,"ursola",0,"ursula",0,"ursulina",0,"ursuline",0,"uta",0,"val",0,"valaree",0,"valaria",0,"vale",0,"valeda",0,"valencia",0,"valene",0,"valenka",0,"valentia",0,"valentina",0,"valentine",0,"valera",0,"valeria",0,"valerie",0,"valery",0,"valerye",0,"valida",0,"valina",0,"valli",0,"vallie",0,"vally",0,"valma",0,"valry",0,"van",0,"vanda",0,"vanessa",0,"vania",0,"vanna",0,"vanni",0,"vannie",0,"vanny",0,"vanya",0,"veda",0,"velma",0,"velvet",0,"venita",0,"venus",0,"vera",0,"veradis",0,"vere",0,"verena",0,"verene",0,"veriee",0,"verile",0,"verina",0,"verine",0,"verla",0,"verna",0,"vernice",0,"veronica",0,"veronika",0,"veronike",0,"veronique",0,"vevay",0,"vi",0,"vicki",0,"vickie",0,"vicky",0,"victoria",0,"vida",0,"viki",0,"vikki",0,"vikky",0,"vilhelmina",0,"vilma",0,"vin",0,"vina",0,"vinita",0,"vinni",0,"vinnie",0,"vinny",0,"viola",0,"violante",0,"viole",0,"violet",0,"violetta",0,"violette",0,"virgie",0,"virgina",0,"virginia",0,"virginie",0,"vita",0,"vitia",0,"vitoria",0,"vittoria",0,"viv",0,"viva",0,"vivi",0,"vivia",0,"vivian",0,"viviana",0,"vivianna",0,"vivianne",0,"vivie",0,"vivien",0,"viviene",0,"vivienne",0,"viviyan",0,"vivyan",0,"vivyanne",0,"vonni",0,"vonnie",0,"vonny",0,"vyky",0,"wallie",0,"wallis",0,"walliw",0,"wally",0,"waly",0,"wanda",0,"wandie",0,"wandis",0,"waneta",0,"wanids",0,"wenda",0,"wendeline",0,"wendi",0,"wendie",0,"wendy",0,"wendye",0,"wenona",0,"wenonah",0,"whitney",0,"wileen",0,"wilhelmina",0,"wilhelmine",0,"wilie",0,"willa",0,"willabella",0,"willamina",0,"willetta",0,"willette",0,"willi",0,"willie",0,"willow",0,"willy",0,"willyt",0,"wilma",0,"wilmette",0,"wilona",0,"wilone",0,"wilow",0,"windy",0,"wini",0,"winifred",0,"winna",0,"winnah",0,"winne",0,"winni",0,"winnie",0,"winnifred",0,"winny",0,"winona",0,"winonah",0,"wren",0,"wrennie",0,"wylma",0,"wynn",0,"wynne",0,"wynnie",0,"wynny",0,"xaviera",0,"xena",0,"xenia",0,"xylia",0,"xylina",0,"yalonda",0,"yasmeen",0,"yasmin",0,"yelena",0,"yetta",0,"yettie",0,"yetty",0,"yevette",0,"ynes",0,"ynez",0,"yoko",0,"yolanda",0,"yolande",0,"yolane",0,"yolanthe",0,"yoshi",0,"yoshiko",0,"yovonnda",0,"ysabel",0,"yvette",0,"yvonne",0,"zabrina",0,"zahara",0,"zandra",0,"zaneta",0,"zara",0,"zarah",0,"zaria",0,"zarla",0,"zea",0,"zelda",0,"zelma",0,"zena",0,"zenia",0,"zia",0,"zilvia",0,"zita",0,"zitella",0,"zoe",0,"zola",0,"zonda",0,"zondra",0,"zonnya",0,"zora",0,"zorah",0,"zorana",0,"zorina",0,"zorine",0,"zsa zsa",0,"zsazsa",0,"zulema",0,"zuzana");

%male_names = ("aaron",0,"ab",0,"abba",0,"abbe",0,"abbey",0,"abbie",0,"abbot",0,"abbott",0,"abby",0,"abdel",0,"abdul",0,"abe",0,"abel",0,"abelard",0,"abeu",0,"abey",0,"abie",0,"abner",0,"abraham",0,"abrahan",0,"abram",0,"abramo",0,"abran",0,"ad",0,"adair",0,"adam",0,"adamo",0,"adams",0,"adan",0,"addie",0,"addison",0,"addy",0,"ade",0,"adelbert",0,"adham",0,"adlai",0,"adler",0,"ado",0,"adolf",0,"adolph",0,"adolphe",0,"adolpho",0,"adolphus",0,"adrian",0,"adriano",0,"adrien",0,"agosto",0,"aguie",0,"aguistin",0,"aguste",0,"agustin",0,"aharon",0,"ahmad",0,"ahmed",0,"ailbert",0,"akim",0,"aksel",0,"al",0,"alain",0,"alair",0,"alan",0,"aland",0,"alano",0,"alanson",0,"alard",0,"alaric",0,"alasdair",0,"alastair",0,"alasteir",0,"alaster",0,"alberik",0,"albert",0,"alberto",0,"albie",0,"albrecht",0,"alden",0,"aldin",0,"aldis",0,"aldo",0,"aldon",0,"aldous",0,"aldric",0,"aldrich",0,"aldridge",0,"aldus",0,"aldwin",0,"alec",0,"alejandro",0,"alejoa",0,"aleksandr",0,"alessandro",0,"alex",0,"alexander",0,"alexandr",0,"alexandre",0,"alexandro",0,"alexandros",0,"alexei",0,"alexio",0,"alexis",0,"alf",0,"alfie",0,"alfons",0,"alfonse",0,"alfonso",0,"alford",0,"alfred",0,"alfredo",0,"alfy",0,"algernon",0,"ali",0,"alic",0,"alick",0,"alisander",0,"alistair",0,"alister",0,"alix",0,"allan",0,"allard",0,"allayne",0,"allen",0,"alley",0,"alleyn",0,"allie",0,"allin",0,"allister",0,"allistir",0,"allyn",0,"aloin",0,"alon",0,"alonso",0,"alonzo",0,"aloysius",0,"alphard",0,"alphonse",0,"alphonso",0,"alric",0,"aluin",0,"aluino",0,"alva",0,"alvan",0,"alvie",0,"alvin",0,"alvis",0,"alvy",0,"alwin",0,"alwyn",0,"alyosha",0,"amble",0,"ambros",0,"ambrose",0,"ambrosi",0,"ambrosio",0,"ambrosius",0,"amby",0,"amerigo",0,"amery",0,"amory",0,"amos",0,"anatol",0,"anatole",0,"anatollo",0,"ancell",0,"anders",0,"anderson",0,"andie",0,"andonis",0,"andras",0,"andre",0,"andrea",0,"andreas",0,"andrej",0,"andres",0,"andrew",0,"andrey",0,"andris",0,"andros",0,"andrus",0,"andy",0,"ange",0,"angel",0,"angeli",0,"angelico",0,"angelo",0,"angie",0,"angus",0,"ansel",0,"ansell",0,"anselm",0,"anson",0,"anthony",0,"antin",0,"antoine",0,"anton",0,"antone",0,"antoni",0,"antonin",0,"antonino",0,"antonio",0,"antonius",0,"antons",0,"antony",0,"any",0,"ara",0,"araldo",0,"arch",0,"archaimbaud",0,"archambault",0,"archer",0,"archibald",0,"archibaldo",0,"archibold",0,"archie",0,"archy",0,"arel",0,"ari",0,"arie",0,"ariel",0,"arin",0,"ario",0,"aristotle",0,"arlan",0,"arlen",0,"arley",0,"arlin",0,"arman",0,"armand",0,"armando",0,"armin",0,"armstrong",0,"arnaldo",0,"arne",0,"arney",0,"arni",0,"arnie",0,"arnold",0,"arnoldo",0,"arnuad",0,"arny",0,"aron",0,"arri",0,"arron",0,"art",0,"artair",0,"arte",0,"artemas",0,"artemis",0,"artemus",0,"arther",0,"arthur",0,"artie",0,"artur",0,"arturo",0,"artus",0,"arty",0,"arv",0,"arvie",0,"arvin",0,"arvy",0,"asa",0,"ase",0,"ash",0,"ashbey",0,"ashby",0,"asher",0,"ashley",0,"ashlin",0,"ashton",0,"aube",0,"auberon",0,"aubert",0,"aubrey",0,"augie",0,"august",0,"augustin",0,"augustine",0,"augusto",0,"augustus",0,"augy",0,"aurthur",0,"austen",0,"austin",0,"ave",0,"averell",0,"averil",0,"averill",0,"avery",0,"avictor",0,"avigdor",0,"avram",0,"avrom",0,"ax",0,"axe",0,"axel",0,"aylmar",0,"aylmer",0,"aymer",0,"bail",0,"bailey",0,"bailie",0,"baillie",0,"baily",0,"baird",0,"bald",0,"balduin",0,"baldwin",0,"bale",0,"ban",0,"bancroft",0,"bank",0,"banky",0,"bar",0,"barbabas",0,"barclay",0,"bard",0,"barde",0,"barn",0,"barnabas",0,"barnabe",0,"barnaby",0,"barnard",0,"barnebas",0,"barnett",0,"barney",0,"barnie",0,"barny",0,"baron",0,"barr",0,"barret",0,"barrett",0,"barri",0,"barrie",0,"barris",0,"barron",0,"barry",0,"bart",0,"bartel",0,"barth",0,"barthel",0,"bartholemy",0,"bartholomeo",0,"bartholomeus",0,"bartholomew",0,"bartie",0,"bartlet",0,"bartlett",0,"bartolemo",0,"bartolomeo",0,"barton",0,"bartram",0,"barty",0,"bary",0,"baryram",0,"base",0,"basil",0,"basile",0,"basilio",0,"basilius",0,"bastian",0,"bastien",0,"bat",0,"batholomew",0,"baudoin",0,"bax",0,"baxie",0,"baxter",0,"baxy",0,"bay",0,"bayard",0,"beale",0,"bealle",0,"bear",0,"bearnard",0,"beau",0,"beaufort",0,"beauregard",0,"beck",0,"beltran",0,"ben",0,"bendick",0,"bendicty",0,"bendix",0,"benedetto",0,"benedick",0,"benedict",0,"benedicto",0,"benedikt",0,"bengt",0,"beniamino",0,"benito",0,"benjamen",0,"benjamin",0,"benji",0,"benjie",0,"benjy",0,"benn",0,"bennett",0,"bennie",0,"benny",0,"benoit",0,"benson",0,"bent",0,"bentlee",0,"bentley",0,"benton",0,"benyamin",0,"ber",0,"berk",0,"berke",0,"berkeley",0,"berkie",0,"berkley",0,"berkly",0,"berky",0,"bern",0,"bernard",0,"bernardo",0,"bernarr",0,"berne",0,"bernhard",0,"bernie",0,"berny",0,"bert",0,"berti",0,"bertie",0,"berton",0,"bertram",0,"bertrand",0,"bertrando",0,"berty",0,"bev",0,"bevan",0,"bevin",0,"bevon",0,"bil",0,"bill",0,"billie",0,"billy",0,"bing",0,"bink",0,"binky",0,"birch",0,"birk",0,"biron",0,"bjorn",0,"blaine",0,"blair",0,"blake",0,"blane",0,"blayne",0,"bo",0,"bob",0,"bobbie",0,"bobby",0,"bogart",0,"bogey",0,"boigie",0,"bond",0,"bondie",0,"bondon",0,"bondy",0,"bone",0,"boniface",0,"boone",0,"boonie",0,"boony",0,"boot",0,"boote",0,"booth",0,"boothe",0,"bord",0,"borden",0,"bordie",0,"bordy",0,"borg",0,"boris",0,"bourke",0,"bowie",0,"boy",0,"boyce",0,"boycey",0,"boycie",0,"boyd",0,"brad",0,"bradan",0,"brade",0,"braden",0,"bradford",0,"bradley",0,"bradly",0,"bradney",0,"brady",0,"bram",0,"bran",0,"brand",0,"branden",0,"brander",0,"brandon",0,"brandtr",0,"brandy",0,"brandyn",0,"brannon",0,"brant",0,"brantley",0,"bren",0,"brendan",0,"brenden",0,"brendin",0,"brendis",0,"brendon",0,"brennan",0,"brennen",0,"brent",0,"bret",0,"brett",0,"brew",0,"brewer",0,"brewster",0,"brian",0,"briano",0,"briant",0,"brice",0,"brien",0,"brig",0,"brigg",0,"briggs",0,"brigham",0,"brion",0,"brit",0,"britt",0,"brnaba",0,"brnaby",0,"brock",0,"brockie",0,"brocky",0,"brod",0,"broddie",0,"broddy",0,"broderic",0,"broderick",0,"brodie",0,"brody",0,"brok",0,"bron",0,"bronnie",0,"bronny",0,"bronson",0,"brook",0,"brooke",0,"brooks",0,"brose",0,"bruce",0,"brucie",0,"bruis",0,"bruno",0,"bryan",0,"bryant",0,"bryanty",0,"bryce",0,"bryn",0,"bryon",0,"buck",0,"buckie",0,"bucky",0,"bud",0,"budd",0,"buddie",0,"buddy",0,"buiron",0,"burch",0,"burg",0,"burgess",0,"burk",0,"burke",0,"burl",0,"burlie",0,"burnaby",0,"burnard",0,"burr",0,"burt",0,"burtie",0,"burton",0,"burty",0,"butch",0,"byram",0,"byran",0,"byrann",0,"byrle",0,"byrom",0,"byron",0,"cad",0,"caddric",0,"caesar",0,"cal",0,"caldwell",0,"cale",0,"caleb",0,"calhoun",0,"callean",0,"calv",0,"calvin",0,"cam",0,"cameron",0,"camey",0,"cammy",0,"car",0,"carce",0,"care",0,"carey",0,"carl",0,"carleton",0,"carlie",0,"carlin",0,"carling",0,"carlo",0,"carlos",0,"carly",0,"carlyle",0,"carmine",0,"carney",0,"carny",0,"carolus",0,"carr",0,"carrol",0,"carroll",0,"carson",0,"cart",0,"carter",0,"carver",0,"cary",0,"caryl",0,"casar",0,"case",0,"casey",0,"cash",0,"caspar",0,"casper",0,"cass",0,"cassie",0,"cassius",0,"caz",0,"cazzie",0,"cchaddie",0,"cece",0,"cecil",0,"cecilio",0,"cecilius",0,"ced",0,"cedric",0,"cello",0,"cesar",0,"cesare",0,"cesaro",0,"chad",0,"chadd",0,"chaddie",0,"chaddy",0,"chadwick",0,"chaim",0,"chalmers",0,"chan",0,"chance",0,"chancey",0,"chandler",0,"chane",0,"chariot",0,"charles",0,"charley",0,"charlie",0,"charlton",0,"chas",0,"chase",0,"chaunce",0,"chauncey",0,"che",0,"chen",0,"ches",0,"chester",0,"cheston",0,"chet",0,"chev",0,"chevalier",0,"chevy",0,"chic",0,"chick",0,"chickie",0,"chicky",0,"chico",0,"chilton",0,"chip",0,"chris",0,"chrisse",0,"chrissie",0,"chrissy",0,"christian",0,"christiano",0,"christie",0,"christoffer",0,"christoforo",0,"christoper",0,"christoph",0,"christophe",0,"christopher",0,"christophorus",0,"christos",0,"christy",0,"chrisy",0,"chrotoem",0,"chucho",0,"chuck",0,"cirillo",0,"cirilo",0,"ciro",0,"cirstoforo",0,"claiborn",0,"claiborne",0,"clair",0,"claire",0,"clarance",0,"clare",0,"clarence",0,"clark",0,"clarke",0,"claudell",0,"claudian",0,"claudianus",0,"claudio",0,"claudius",0,"claus",0,"clay",0,"clayborn",0,"clayborne",0,"claybourne",0,"clayson",0,"clayton",0,"cleavland",0,"clem",0,"clemens",0,"clement",0,"clemente",0,"clementius",0,"clemmie",0,"clemmy",0,"cleon",0,"clerc",0,"clerkclaude",0,"cletis",0,"cletus",0,"cleve",0,"cleveland",0,"clevey",0,"clevie",0,"cliff",0,"clifford",0,"clim",0,"clint",0,"clive",0,"cly",0,"clyde",0,"clyve",0,"clywd",0,"cob",0,"cobb",0,"cobbie",0,"cobby",0,"codi",0,"codie",0,"cody",0,"cointon",0,"colan",0,"colas",0,"colby",0,"cole",0,"coleman",0,"colet",0,"colin",0,"collin",0,"colman",0,"colver",0,"con",0,"conan",0,"conant",0,"conn",0,"conney",0,"connie",0,"connor",0,"conny",0,"conrad",0,"conrade",0,"conrado",0,"conroy",0,"consalve",0,"constantin",0,"constantine",0,"constantino",0,"conway",0,"coop",0,"cooper",0,"corbet",0,"corbett",0,"corbie",0,"corbin",0,"corby",0,"cord",0,"cordell",0,"cordie",0,"cordy",0,"corey",0,"cori",0,"cornall",0,"cornelius",0,"cornell",0,"corney",0,"cornie",0,"corny",0,"correy",0,"corrie",0,"cort",0,"cortie",0,"corty",0,"cory",0,"cos",0,"cosimo",0,"cosme",0,"cosmo",0,"costa",0,"court",0,"courtnay",0,"courtney",0,"cozmo",0,"craggie",0,"craggy",0,"craig",0,"crawford",0,"creigh",0,"creight",0,"creighton",0,"crichton",0,"cris",0,"cristian",0,"cristiano",0,"cristobal",0,"crosby",0,"cross",0,"cull",0,"cullan",0,"cullen",0,"culley",0,"cullie",0,"cullin",0,"cully",0,"culver",0,"curcio",0,"curr",0,"curran",0,"currey",0,"currie",0,"curry",0,"curt",0,"curtice",0,"curtis",0,"cy",0,"cyril",0,"cyrill",0,"cyrille",0,"cyrillus",0,"cyrus",0,"d'arcy",0,"dael",0,"dag",0,"dagny",0,"dal",0,"dale",0,"dalis",0,"dall",0,"dallas",0,"dalli",0,"dallis",0,"dallon",0,"dalston",0,"dalt",0,"dalton",0,"dame",0,"damian",0,"damiano",0,"damien",0,"damon",0,"dan",0,"dana",0,"dane",0,"dani",0,"danie",0,"daniel",0,"dannel",0,"dannie",0,"danny",0,"dante",0,"danya",0,"dar",0,"darb",0,"darbee",0,"darby",0,"darcy",0,"dare",0,"daren",0,"darill",0,"darin",0,"dario",0,"darius",0,"darn",0,"darnall",0,"darnell",0,"daron",0,"darrel",0,"darrell",0,"darren",0,"darrick",0,"darrin",0,"darryl",0,"darwin",0,"daryl",0,"daryle",0,"dav",0,"dave",0,"daven",0,"davey",0,"david",0,"davidde",0,"davide",0,"davidson",0,"davie",0,"davin",0,"davis",0,"davon",0,"davy",0,"de witt",0,"dean",0,"deane",0,"decca",0,"deck",0,"del",0,"delainey",0,"delaney",0,"delano",0,"delbert",0,"dell",0,"delmar",0,"delmer",0,"delmor",0,"delmore",0,"demetre",0,"demetri",0,"demetris",0,"demetrius",0,"demott",0,"den",0,"dene",0,"denis",0,"dennet",0,"denney",0,"dennie",0,"dennis",0,"dennison",0,"denny",0,"denver",0,"denys",0,"der",0,"derby",0,"derek",0,"derick",0,"derk",0,"dermot",0,"derrek",0,"derrick",0,"derrik",0,"derril",0,"derron",0,"derry",0,"derward",0,"derwin",0,"des",0,"desi",0,"desmond",0,"desmund",0,"dev",0,"devin",0,"devland",0,"devlen",0,"devlin",0,"devy",0,"dew",0,"dewain",0,"dewey",0,"dewie",0,"dewitt",0,"dex",0,"dexter",0,"diarmid",0,"dick",0,"dickie",0,"dicky",0,"diego",0,"dieter",0,"dietrich",0,"dilan",0,"dill",0,"dillie",0,"dillon",0,"dilly",0,"dimitri",0,"dimitry",0,"dino",0,"dion",0,"dionisio",0,"dionysus",0,"dirk",0,"dmitri",0,"dolf",0,"dolph",0,"dom",0,"domenic",0,"domenico",0,"domingo",0,"dominic",0,"dominick",0,"dominik",0,"dominique",0,"don",0,"donal",0,"donall",0,"donalt",0,"donaugh",0,"donavon",0,"donn",0,"donnell",0,"donnie",0,"donny",0,"donovan",0,"dore",0,"dorey",0,"dorian",0,"dorie",0,"dory",0,"doug",0,"dougie",0,"douglas",0,"douglass",0,"dougy",0,"dov",0,"doy",0,"doyle",0,"drake",0,"drew",0,"dru",0,"drud",0,"drugi",0,"duane",0,"dud",0,"dudley",0,"duff",0,"duffie",0,"duffy",0,"dugald",0,"duke",0,"dukey",0,"dukie",0,"duky",0,"dun",0,"dunc",0,"duncan",0,"dunn",0,"dunstan",0,"dur",0,"durand",0,"durant",0,"durante",0,"durward",0,"dwain",0,"dwayne",0,"dwight",0,"dylan",0,"eadmund",0,"eal",0,"eamon",0,"earl",0,"earle",0,"earlie",0,"early",0,"earvin",0,"eb",0,"eben",0,"ebeneser",0,"ebenezer",0,"eberhard",0,"eberto",0,"ed",0,"edan",0,"edd",0,"eddie",0,"eddy",0,"edgar",0,"edgard",0,"edgardo",0,"edik",0,"edlin",0,"edmon",0,"edmund",0,"edouard",0,"edsel",0,"eduard",0,"eduardo",0,"eduino",0,"edvard",0,"edward",0,"edwin",0,"efrem",0,"efren",0,"egan",0,"egbert",0,"egon",0,"egor",0,"el",0,"elbert",0,"elden",0,"eldin",0,"eldon",0,"eldredge",0,"eldridge",0,"eli",0,"elia",0,"elias",0,"elihu",0,"elijah",0,"eliot",0,"elisha",0,"ellary",0,"ellerey",0,"ellery",0,"elliot",0,"elliott",0,"ellis",0,"ellswerth",0,"ellsworth",0,"ellwood",0,"elmer",0,"elmo",0,"elmore",0,"elnar",0,"elroy",0,"elston",0,"elsworth",0,"elton",0,"elvin",0,"elvis",0,"elvyn",0,"elwin",0,"elwood",0,"elwyn",0,"ely",0,"em",0,"emanuel",0,"emanuele",0,"emelen",0,"emerson",0,"emery",0,"emile",0,"emilio",0,"emlen",0,"emlyn",0,"emmanuel",0,"emmerich",0,"emmery",0,"emmet",0,"emmett",0,"emmit",0,"emmott",0,"emmy",0,"emory",0,"engelbert",0,"englebert",0,"ennis",0,"enoch",0,"enos",0,"enrico",0,"enrique",0,"ephraim",0,"ephrayim",0,"ephrem",0,"erasmus",0,"erastus",0,"erek",0,"erhard",0,"erhart",0,"eric",0,"erich",0,"erick",0,"erie",0,"erik",0,"erin",0,"erl",0,"ermanno",0,"ermin",0,"ernest",0,"ernesto",0,"ernestus",0,"ernie",0,"ernst",0,"erny",0,"errick",0,"errol",0,"erroll",0,"erskine",0,"erv",0,"ervin",0,"erwin",0,"esdras",0,"esme",0,"esra",0,"esteban",0,"estevan",0,"etan",0,"ethan",0,"ethe",0,"ethelbert",0,"ethelred",0,"etienne",0,"ettore",0,"euell",0,"eugen",0,"eugene",0,"eugenio",0,"eugenius",0,"eustace",0,"ev",0,"evan",0,"evelin",0,"evelyn",0,"even",0,"everard",0,"evered",0,"everett",0,"evin",0,"evyn",0,"ewan",0,"eward",0,"ewart",0,"ewell",0,"ewen",0,"ezechiel",0,"ezekiel",0,"ezequiel",0,"eziechiele",0,"ezra",0,"ezri",0,"fabe",0,"faber",0,"fabian",0,"fabiano",0,"fabien",0,"fabio",0,"fair",0,"fairfax",0,"fairleigh",0,"fairlie",0,"falito",0,"falkner",0,"far",0,"farlay",0,"farlee",0,"farleigh",0,"farley",0,"farlie",0,"farly",0,"farr",0,"farrel",0,"farrell",0,"farris",0,"faulkner",0,"fax",0,"federico",0,"fee",0,"felic",0,"felice",0,"felicio",0,"felike",0,"feliks",0,"felipe",0,"felix",0,"felizio",0,"feodor",0,"ferd",0,"ferdie",0,"ferdinand",0,"ferdy",0,"fergus",0,"ferguson",0,"fernando",0,"ferrel",0,"ferrell",0,"ferris",0,"fidel",0,"fidelio",0,"fidole",0,"field",0,"fielding",0,"fields",0,"filbert",0,"filberte",0,"filberto",0,"filip",0,"filippo",0,"filmer",0,"filmore",0,"fin",0,"findlay",0,"findley",0,"finlay",0,"finley",0,"finn",0,"fitz",0,"fitzgerald",0,"flem",0,"fleming",0,"flemming",0,"fletch",0,"fletcher",0,"flin",0,"flinn",0,"flint",0,"florian",0,"flory",0,"floyd",0,"flynn",0,"fons",0,"fonsie",0,"fonz",0,"fonzie",0,"forbes",0,"ford",0,"forest",0,"forester",0,"forrest",0,"forrester",0,"forster",0,"foss",0,"foster",0,"fowler",0,"fran",0,"francesco",0,"franchot",0,"francis",0,"francisco",0,"franciskus",0,"francklin",0,"francklyn",0,"francois",0,"frank",0,"frankie",0,"franklin",0,"franklyn",0,"franky",0,"frannie",0,"franny",0,"frans",0,"fransisco",0,"frants",0,"franz",0,"franzen",0,"frasco",0,"fraser",0,"frasier",0,"frasquito",0,"fraze",0,"frazer",0,"frazier",0,"fred",0,"freddie",0,"freddy",0,"fredek",0,"frederic",0,"frederich",0,"frederick",0,"frederico",0,"frederigo",0,"frederik",0,"fredric",0,"fredrick",0,"free",0,"freedman",0,"freeland",0,"freeman",0,"freemon",0,"fremont",0,"friedrich",0,"friedrick",0,"fritz",0,"fulton",0,"gabbie",0,"gabby",0,"gabe",0,"gabi",0,"gabie",0,"gabriel",0,"gabriele",0,"gabriello",0,"gaby",0,"gael",0,"gaelan",0,"gage",0,"gail",0,"gaile",0,"gal",0,"gale",0,"galen",0,"gallagher",0,"gallard",0,"galvan",0,"galven",0,"galvin",0,"gamaliel",0,"gan",0,"gannie",0,"gannon",0,"ganny",0,"gar",0,"garald",0,"gard",0,"gardener",0,"gardie",0,"gardiner",0,"gardner",0,"gardy",0,"gare",0,"garek",0,"gareth",0,"garey",0,"garfield",0,"garik",0,"garner",0,"garold",0,"garrard",0,"garrek",0,"garret",0,"garreth",0,"garrett",0,"garrick",0,"garrik",0,"garrot",0,"garrott",0,"garry",0,"garth",0,"garv",0,"garvey",0,"garvin",0,"garvy",0,"garwin",0,"garwood",0,"gary",0,"gaspar",0,"gaspard",0,"gasparo",0,"gasper",0,"gaston",0,"gaultiero",0,"gauthier",0,"gav",0,"gavan",0,"gaven",0,"gavin",0,"gawain",0,"gawen",0,"gay",0,"gayelord",0,"gayle",0,"gayler",0,"gaylor",0,"gaylord",0,"gearalt",0,"gearard",0,"gene",0,"geno",0,"geoff",0,"geoffrey",0,"geoffry",0,"georas",0,"geordie",0,"georg",0,"george",0,"georges",0,"georgi",0,"georgie",0,"georgy",0,"gerald",0,"gerard",0,"gerardo",0,"gerek",0,"gerhard",0,"gerhardt",0,"geri",0,"gerick",0,"gerik",0,"germain",0,"germaine",0,"germayne",0,"gerome",0,"gerrard",0,"gerri",0,"gerrie",0,"gerry",0,"gery",0,"gherardo",0,"giacobo",0,"giacomo",0,"giacopo",0,"gian",0,"gianni",0,"giavani",0,"gib",0,"gibb",0,"gibbie",0,"gibby",0,"gideon",0,"giff",0,"giffard",0,"giffer",0,"giffie",0,"gifford",0,"giffy",0,"gil",0,"gilbert",0,"gilberto",0,"gilburt",0,"giles",0,"gill",0,"gilles",0,"ginger",0,"gino",0,"giordano",0,"giorgi",0,"giorgio",0,"giovanni",0,"giraldo",0,"giraud",0,"giselbert",0,"giulio",0,"giuseppe",0,"giustino",0,"giusto",0,"glen",0,"glenden",0,"glendon",0,"glenn",0,"glyn",0,"glynn",0,"godard",0,"godart",0,"goddard",0,"goddart",0,"godfree",0,"godfrey",0,"godfry",0,"godwin",0,"gonzales",0,"gonzalo",0,"goober",0,"goran",0,"goraud",0,"gordan",0,"gorden",0,"gordie",0,"gordon",0,"gordy",0,"gothart",0,"gottfried",0,"grace",0,"gradeigh",0,"gradey",0,"grady",0,"graehme",0,"graeme",0,"graham",0,"graig",0,"gram",0,"gran",0,"grange",0,"granger",0,"grannie",0,"granny",0,"grant",0,"grantham",0,"granthem",0,"grantley",0,"granville",0,"gray",0,"greg",0,"gregg",0,"greggory",0,"gregoire",0,"gregoor",0,"gregor",0,"gregorio",0,"gregorius",0,"gregory",0,"grenville",0,"griff",0,"griffie",0,"griffin",0,"griffith",0,"griffy",0,"gris",0,"griswold",0,"griz",0,"grove",0,"grover",0,"gualterio",0,"guglielmo",0,"guido",0,"guilbert",0,"guillaume",0,"guillermo",0,"gun",0,"gunar",0,"gunner",0,"guntar",0,"gunter",0,"gunther",0,"gus",0,"guss",0,"gustaf",0,"gustav",0,"gustave",0,"gustavo",0,"gustavus",0,"guthrey",0,"guthrie",0,"guthry",0,"guy",0,"had",0,"hadlee",0,"hadleigh",0,"hadley",0,"hadrian",0,"hagan",0,"hagen",0,"hailey",0,"haily",0,"hakeem",0,"hakim",0,"hal",0,"hale",0,"haleigh",0,"haley",0,"hall",0,"hallsy",0,"halsey",0,"halsy",0,"ham",0,"hamel",0,"hamid",0,"hamil",0,"hamilton",0,"hamish",0,"hamlen",0,"hamlin",0,"hammad",0,"hamnet",0,"hanan",0,"hank",0,"hans",0,"hansiain",0,"hanson",0,"harald",0,"harbert",0,"harcourt",0,"hardy",0,"harlan",0,"harland",0,"harlen",0,"harley",0,"harlin",0,"harman",0,"harmon",0,"harold",0,"haroun",0,"harp",0,"harper",0,"harris",0,"harrison",0,"harry",0,"hart",0,"hartley",0,"hartwell",0,"harv",0,"harvey",0,"harwell",0,"harwilll",0,"hasheem",0,"hashim",0,"haskel",0,"haskell",0,"haslett",0,"hastie",0,"hastings",0,"hasty",0,"haven",0,"hayden",0,"haydon",0,"hayes",0,"hayward",0,"haywood",0,"hayyim",0,"haze",0,"hazel",0,"hazlett",0,"heall",0,"heath",0,"hebert",0,"hector",0,"heindrick",0,"heinrick",0,"heinrik",0,"henderson",0,"hendrick",0,"hendrik",0,"henri",0,"henrik",0,"henry",0,"herb",0,"herbert",0,"herbie",0,"herby",0,"herc",0,"hercule",0,"hercules",0,"herculie",0,"heriberto",0,"herman",0,"hermann",0,"hermie",0,"hermon",0,"hermy",0,"hernando",0,"herold",0,"herrick",0,"hersch",0,"herschel",0,"hersh",0,"hershel",0,"herve",0,"hervey",0,"hew",0,"hewe",0,"hewet",0,"hewett",0,"hewie",0,"hewitt",0,"heywood",0,"hi",0,"hieronymus",0,"hilario",0,"hilarius",0,"hilary",0,"hill",0,"hillard",0,"hillary",0,"hillel",0,"hillery",0,"hilliard",0,"hillie",0,"hillier",0,"hilly",0,"hillyer",0,"hilton",0,"hinze",0,"hiram",0,"hirsch",0,"hobard",0,"hobart",0,"hobey",0,"hobie",0,"hodge",0,"hoebart",0,"hogan",0,"holden",0,"hollis",0,"holly",0,"holmes",0,"holt",0,"homer",0,"homere",0,"homerus",0,"horace",0,"horacio",0,"horatio",0,"horatius",0,"horst",0,"hort",0,"horten",0,"horton",0,"howard",0,"howey",0,"howie",0,"hoyt",0,"hube",0,"hubert",0,"huberto",0,"hubey",0,"hubie",0,"huey",0,"hugh",0,"hughie",0,"hugibert",0,"hugo",0,"hugues",0,"humbert",0,"humberto",0,"humfrey",0,"humfrid",0,"humfried",0,"humphrey",0,"hunfredo",0,"hunt",0,"hunter",0,"huntington",0,"huntlee",0,"huntley",0,"hurlee",0,"hurleigh",0,"hurley",0,"husain",0,"husein",0,"hussein",0,"hy",0,"hyatt",0,"hyman",0,"hymie",0,"iago",0,"iain",0,"ian",0,"ibrahim",0,"ichabod",0,"iggie",0,"iggy",0,"ignace",0,"ignacio",0,"ignacius",0,"ignatius",0,"ignaz",0,"ignazio",0,"igor",0,"ike",0,"ikey",0,"ilaire",0,"ilario",0,"immanuel",0,"ingamar",0,"ingar",0,"ingelbert",0,"ingemar",0,"inger",0,"inglebert",0,"inglis",0,"ingmar",0,"ingra",0,"ingram",0,"ingrim",0,"inigo",0,"inness",0,"innis",0,"iorgo",0,"iorgos",0,"iosep",0,"ira",0,"irv",0,"irvin",0,"irvine",0,"irving",0,"irwin",0,"irwinn",0,"isa",0,"isaac",0,"isaak",0,"isac",0,"isacco",0,"isador",0,"isadore",0,"isaiah",0,"isak",0,"isiahi",0,"isidor",0,"isidore",0,"isidoro",0,"isidro",0,"israel",0,"issiah",0,"itch",0,"ivan",0,"ivar",0,"ive",0,"iver",0,"ives",0,"ivor",0,"izaak",0,"izak",0,"izzy",0,"jabez",0,"jack",0,"jackie",0,"jackson",0,"jacky",0,"jacob",0,"jacobo",0,"jacques",0,"jae",0,"jaime",0,"jaimie",0,"jake",0,"jakie",0,"jakob",0,"jamaal",0,"jamal",0,"james",0,"jameson",0,"jamesy",0,"jamey",0,"jamie",0,"jamil",0,"jamill",0,"jamison",0,"jammal",0,"jan",0,"janek",0,"janos",0,"jarad",0,"jard",0,"jareb",0,"jared",0,"jarib",0,"jarid",0,"jarrad",0,"jarred",0,"jarret",0,"jarrett",0,"jarrid",0,"jarrod",0,"jarvis",0,"jase",0,"jasen",0,"jason",0,"jasper",0,"jasun",0,"javier",0,"jay",0,"jaye",0,"jayme",0,"jaymie",0,"jayson",0,"jdavie",0,"jean",0,"jecho",0,"jed",0,"jedd",0,"jeddy",0,"jedediah",0,"jedidiah",0,"jeff",0,"jefferey",0,"jefferson",0,"jeffie",0,"jeffrey",0,"jeffry",0,"jeffy",0,"jehu",0,"jeno ",0,"jens",0,"jephthah",0,"jerad",0,"jerald",0,"jeramey",0,"jeramie",0,"jere",0,"jereme",0,"jeremiah",0,"jeremias",0,"jeremie",0,"jeremy",0,"jermain",0,"jermaine",0,"jermayne",0,"jerome",0,"jeromy",0,"jerri",0,"jerrie",0,"jerrold",0,"jerrome",0,"jerry",0,"jervis",0,"jess",0,"jesse",0,"jessee",0,"jessey",0,"jessie",0,"jesus",0,"jeth",0,"jethro",0,"jim",0,"jimmie",0,"jimmy",0,"jo",0,"joachim",0,"joaquin",0,"job",0,"jock",0,"jocko",0,"jodi",0,"jodie",0,"jody",0,"joe",0,"joel",0,"joey",0,"johan",0,"johann",0,"johannes",0,"john",0,"johnathan",0,"johnathon",0,"johnnie",0,"johnny",0,"johny",0,"jon",0,"jonah",0,"jonas",0,"jonathan",0,"jonathon",0,"jone",0,"jordan",0,"jordon",0,"jorgan",0,"jorge",0,"jory",0,"jose",0,"joseito",0,"joseph",0,"josh",0,"joshia",0,"joshua",0,"joshuah",0,"josiah",0,"josias",0,"jourdain",0,"jozef",0,"juan",0,"jud",0,"judah",0,"judas",0,"judd",0,"jude",0,"judon",0,"jule",0,"jules",0,"julian",0,"julie",0,"julio",0,"julius",0,"justen",0,"justin",0,"justinian",0,"justino",0,"justis",0,"justus",0,"kahaleel",0,"kahlil",0,"kain",0,"kaine",0,"kaiser",0,"kale",0,"kaleb",0,"kalil",0,"kalle",0,"kalvin",0,"kane",0,"kareem",0,"karel",0,"karim",0,"karl",0,"karlan",0,"karlens",0,"karlik",0,"karlis",0,"karney",0,"karoly",0,"kaspar",0,"kasper",0,"kayne",0,"kean",0,"keane",0,"kearney",0,"keary",0,"keefe",0,"keefer",0,"keelby",0,"keen",0,"keenan",0,"keene",0,"keir",0,"keith",0,"kelbee",0,"kelby",0,"kele",0,"kellby",0,"kellen",0,"kelley",0,"kelly",0,"kelsey",0,"kelvin",0,"kelwin",0,"ken",0,"kendal",0,"kendall",0,"kendell",0,"kendrick",0,"kendricks",0,"kenn",0,"kennan",0,"kennedy",0,"kenneth",0,"kennett",0,"kennie",0,"kennith",0,"kenny",0,"kenon",0,"kent",0,"kenton",0,"kenyon",0,"ker",0,"kerby",0,"kerk",0,"kermie",0,"kermit",0,"kermy",0,"kerr",0,"kerry",0,"kerwin",0,"kerwinn",0,"kev",0,"kevan",0,"keven",0,"kevin",0,"kevon",0,"khalil",0,"kiel",0,"kienan",0,"kile",0,"kiley",0,"kilian",0,"killian",0,"killie",0,"killy",0,"kim",0,"kimball",0,"kimbell",0,"kimble",0,"kin",0,"kincaid",0,"king",0,"kingsley",0,"kingsly",0,"kingston",0,"kinnie",0,"kinny",0,"kinsley",0,"kip",0,"kipp",0,"kippar",0,"kipper",0,"kippie",0,"kippy",0,"kirby",0,"kirk",0,"kit",0,"klaus",0,"klemens",0,"klement",0,"kleon",0,"kliment",0,"knox",0,"koenraad",0,"konrad",0,"konstantin",0,"konstantine",0,"korey",0,"kort",0,"kory",0,"kris",0,"krisha",0,"krishna",0,"krishnah",0,"krispin",0,"kristian",0,"kristo",0,"kristofer",0,"kristoffer",0,"kristofor",0,"kristoforo",0,"kristopher",0,"kristos",0,"kurt",0,"kurtis",0,"ky",0,"kyle",0,"kylie",0,"laird",0,"lalo",0,"lamar",0,"lambert",0,"lammond",0,"lamond",0,"lamont",0,"lance",0,"lancelot",0,"land",0,"lane",0,"laney",0,"langsdon",0,"langston",0,"lanie",0,"lannie",0,"lanny",0,"larry",0,"lars",0,"laughton",0,"launce",0,"lauren",0,"laurence",0,"laurens",0,"laurent",0,"laurie",0,"lauritz",0,"law",0,"lawrence",0,"lawry",0,"lawton",0,"lay",0,"layton",0,"lazar",0,"lazare",0,"lazaro",0,"lazarus",0,"lee",0,"leeland",0,"lefty",0,"leicester",0,"leif",0,"leigh",0,"leighton",0,"lek",0,"leland",0,"lem",0,"lemar",0,"lemmie",0,"lemmy",0,"lemuel",0,"lenard",0,"lenci",0,"lennard",0,"lennie",0,"leo",0,"leon",0,"leonard",0,"leonardo",0,"leonerd",0,"leonhard",0,"leonid",0,"leonidas",0,"leopold",0,"leroi",0,"leroy",0,"les",0,"lesley",0,"leslie",0,"lester",0,"leupold",0,"lev",0,"levey",0,"levi",0,"levin",0,"levon",0,"levy",0,"lew",0,"lewes",0,"lewie",0,"lewiss",0,"lezley",0,"liam",0,"lief",0,"lin",0,"linc",0,"lincoln",0,"lind",0,"lindon",0,"lindsay",0,"lindsey",0,"lindy",0,"link",0,"linn",0,"linoel",0,"linus",0,"lion",0,"lionel",0,"lionello",0,"lisle",0,"llewellyn",0,"lloyd",0,"llywellyn",0,"lock",0,"locke",0,"lockwood",0,"lodovico",0,"logan",0,"lombard",0,"lon",0,"lonnard",0,"lonnie",0,"lonny",0,"lorant",0,"loren",0,"lorens",0,"lorenzo",0,"lorin",0,"lorne",0,"lorrie",0,"lorry",0,"lothaire",0,"lothario",0,"lou",0,"louie",0,"louis",0,"lovell",0,"lowe",0,"lowell",0,"lowrance",0,"loy",0,"loydie",0,"luca",0,"lucais",0,"lucas",0,"luce",0,"lucho",0,"lucian",0,"luciano",0,"lucias",0,"lucien",0,"lucio",0,"lucius",0,"ludovico",0,"ludvig",0,"ludwig",0,"luigi",0,"luis",0,"lukas",0,"luke",0,"lutero",0,"luther",0,"ly",0,"lydon",0,"lyell",0,"lyle",0,"lyman",0,"lyn",0,"lynn",0,"lyon",0,"mac",0,"mace",0,"mack",0,"mackenzie",0,"maddie",0,"maddy",0,"madison",0,"magnum",0,"mahmoud",0,"mahmud",0,"maison",0,"maje",0,"major",0,"mal",0,"malachi",0,"malchy",0,"malcolm",0,"mallory",0,"malvin",0,"man",0,"mandel",0,"manfred",0,"mannie",0,"manny",0,"mano",0,"manolo",0,"manuel",0,"mar",0,"marc",0,"marcel",0,"marcello",0,"marcellus",0,"marcelo",0,"marchall",0,"marco",0,"marcos",0,"marcus",0,"marietta",0,"marijn",0,"mario",0,"marion",0,"marius",0,"mark",0,"markos",0,"markus",0,"marlin",0,"marlo",0,"marlon",0,"marlow",0,"marlowe",0,"marmaduke",0,"marsh",0,"marshal",0,"marshall",0,"mart",0,"martainn",0,"marten",0,"martie",0,"martin",0,"martino",0,"marty",0,"martyn",0,"marv",0,"marve",0,"marven",0,"marvin",0,"marwin",0,"mason",0,"massimiliano",0,"massimo",0,"mata",0,"mateo",0,"mathe",0,"mathew",0,"mathian",0,"mathias",0,"matias",0,"matt",0,"matteo",0,"matthaeus",0,"mattheus",0,"matthew",0,"matthias",0,"matthieu",0,"matthiew",0,"matthus",0,"mattias",0,"mattie",0,"matty",0,"maurice",0,"mauricio",0,"maurie",0,"maurise",0,"maurits",0,"maurizio",0,"maury",0,"max",0,"maxie",0,"maxim",0,"maximilian",0,"maximilianus",0,"maximilien",0,"maximo",0,"maxwell",0,"maxy",0,"mayer",0,"maynard",0,"mayne",0,"maynord",0,"mayor",0,"mead",0,"meade",0,"meier",0,"meir",0,"mel",0,"melvin",0,"melvyn",0,"menard",0,"mendel",0,"mendie",0,"mendy",0,"meredeth",0,"meredith",0,"merell",0,"merill",0,"merle",0,"merrel",0,"merrick",0,"merrill",0,"merry",0,"merv",0,"mervin",0,"merwin",0,"merwyn",0,"meryl",0,"meyer",0,"mic",0,"micah",0,"michael",0,"michail",0,"michal",0,"michale",0,"micheal",0,"micheil",0,"michel",0,"michele",0,"mick",0,"mickey",0,"mickie",0,"micky",0,"miguel",0,"mikael",0,"mike",0,"mikel",0,"mikey",0,"mikkel",0,"mikol",0,"mile",0,"miles",0,"mill",0,"millard",0,"miller",0,"milo",0,"milt",0,"miltie",0,"milton",0,"milty",0,"miner",0,"minor",0,"mischa",0,"mitch",0,"mitchael",0,"mitchel",0,"mitchell",0,"moe",0,"mohammed",0,"mohandas",0,"mohandis",0,"moise",0,"moises",0,"moishe",0,"monro",0,"monroe",0,"montague",0,"monte",0,"montgomery",0,"monti",0,"monty",0,"moore",0,"mord",0,"mordecai",0,"mordy",0,"morey",0,"morgan",0,"morgen",0,"morgun",0,"morie",0,"moritz",0,"morlee",0,"morley",0,"morly",0,"morrie",0,"morris",0,"morry",0,"morse",0,"mort",0,"morten",0,"mortie",0,"mortimer",0,"morton",0,"morty",0,"mose",0,"moses",0,"moshe",0,"moss",0,"mozes",0,"muffin",0,"muhammad",0,"munmro",0,"munroe",0,"murdoch",0,"murdock",0,"murray",0,"murry",0,"murvyn",0,"my",0,"myca",0,"mycah",0,"mychal",0,"myer",0,"myles",0,"mylo",0,"myron",0,"myrvyn",0,"myrwyn",0,"nahum",0,"nap",0,"napoleon",0,"nappie",0,"nappy",0,"nat",0,"natal",0,"natale",0,"nataniel",0,"nate",0,"nathan",0,"nathanael",0,"nathanial",0,"nathaniel",0,"nathanil",0,"natty",0,"neal",0,"neale",0,"neall",0,"nealon",0,"nealson",0,"nealy",0,"ned",0,"neddie",0,"neddy",0,"neel",0,"nefen",0,"nehemiah",0,"neil",0,"neill",0,"neils",0,"nels",0,"nelson",0,"nero",0,"neron",0,"nester",0,"nestor",0,"nev",0,"nevil",0,"nevile",0,"neville",0,"nevin",0,"nevins",0,"newton",0,"nial",0,"niall",0,"niccolo",0,"nicholas",0,"nichole",0,"nichols",0,"nick",0,"nickey",0,"nickie",0,"nicko",0,"nickola",0,"nickolai",0,"nickolas",0,"nickolaus",0,"nicky",0,"nico",0,"nicol",0,"nicola",0,"nicolai",0,"nicolais",0,"nicolas",0,"nicolis",0,"niel",0,"niels",0,"nigel",0,"niki",0,"nikita",0,"nikki",0,"niko",0,"nikola",0,"nikolai",0,"nikolaos",0,"nikolas",0,"nikolaus",0,"nikolos",0,"nikos",0,"nil",0,"niles",0,"nils",0,"nilson",0,"niven",0,"noach",0,"noah",0,"noak",0,"noam",0,"nobe",0,"nobie",0,"noble",0,"noby",0,"noe",0,"noel",0,"nolan",0,"noland",0,"noll",0,"nollie",0,"nolly",0,"norbert",0,"norbie",0,"norby",0,"norman",0,"normand",0,"normie",0,"normy",0,"norrie",0,"norris",0,"norry",0,"north",0,"northrop",0,"northrup",0,"norton",0,"nowell",0,"nye",0,"oates",0,"obadiah",0,"obadias",0,"obed",0,"obediah",0,"oberon",0,"obidiah",0,"obie",0,"oby",0,"octavius",0,"ode",0,"odell",0,"odey",0,"odie",0,"odo",0,"ody",0,"ogdan",0,"ogden",0,"ogdon",0,"olag",0,"olav",0,"ole",0,"olenolin",0,"olin",0,"oliver",0,"olivero",0,"olivier",0,"oliviero",0,"ollie",0,"olly",0,"olvan",0,"omar",0,"omero",0,"onfre",0,"onfroi",0,"onofredo",0,"oran",0,"orazio",0,"orbadiah",0,"oren",0,"orin",0,"orion",0,"orlan",0,"orland",0,"orlando",0,"orran",0,"orren",0,"orrin",0,"orson",0,"orton",0,"orv",0,"orville",0,"osbert",0,"osborn",0,"osborne",0,"osbourn",0,"osbourne",0,"osgood",0,"osmond",0,"osmund",0,"ossie",0,"oswald",0,"oswell",0,"otes",0,"othello",0,"otho",0,"otis",0,"otto",0,"owen",0,"ozzie",0,"ozzy",0,"pablo",0,"pace",0,"packston",0,"paco",0,"pacorro",0,"paddie",0,"paddy",0,"padget",0,"padgett",0,"padraic",0,"padraig",0,"padriac",0,"page",0,"paige",0,"pail",0,"pall",0,"palm",0,"palmer",0,"panchito",0,"pancho",0,"paolo",0,"papageno",0,"paquito",0,"park",0,"parke",0,"parker",0,"parnell",0,"parrnell",0,"parry",0,"parsifal",0,"pascal",0,"pascale",0,"pasquale",0,"pat",0,"pate",0,"paten",0,"patin",0,"paton",0,"patric",0,"patrice",0,"patricio",0,"patrick",0,"patrizio",0,"patrizius",0,"patsy",0,"patten",0,"pattie",0,"pattin",0,"patton",0,"patty",0,"paul",0,"paulie",0,"paulo",0,"pauly",0,"pavel",0,"pavlov",0,"paxon",0,"paxton",0,"payton",0,"peadar",0,"pearce",0,"pebrook",0,"peder",0,"pedro",0,"peirce",0,"pembroke",0,"pen",0,"penn",0,"pennie",0,"penny",0,"penrod",0,"pepe",0,"pepillo",0,"pepito",0,"perceval",0,"percival",0,"percy",0,"perice",0,"perkin",0,"pernell",0,"perren",0,"perry",0,"pete",0,"peter",0,"peterus",0,"petey",0,"petr",0,"peyter",0,"peyton",0,"phil",0,"philbert",0,"philip",0,"phillip",0,"phillipe",0,"phillipp",0,"phineas",0,"phip",0,"pierce",0,"pierre",0,"pierson",0,"pieter",0,"pietrek",0,"pietro",0,"piggy",0,"pincas",0,"pinchas",0,"pincus",0,"piotr",0,"pip",0,"pippo",0,"pooh",0,"port",0,"porter",0,"portie",0,"porty",0,"poul",0,"powell",0,"pren",0,"prent",0,"prentice",0,"prentiss",0,"prescott",0,"preston",0,"price",0,"prince",0,"prinz",0,"pryce",0,"puff",0,"purcell",0,"putnam",0,"putnem",0,"pyotr",0,"quent",0,"quentin",0,"quill",0,"quillan",0,"quincey",0,"quincy",0,"quinlan",0,"quinn",0,"quint",0,"quintin",0,"quinton",0,"quintus",0,"rab",0,"rabbi",0,"rabi",0,"rad",0,"radcliffe",0,"raddie",0,"raddy",0,"rafael",0,"rafaellle",0,"rafaello",0,"rafe",0,"raff",0,"raffaello",0,"raffarty",0,"rafferty",0,"rafi",0,"ragnar",0,"raimondo",0,"raimund",0,"raimundo",0,"rainer",0,"raleigh",0,"ralf",0,"ralph",0,"ram",0,"ramon",0,"ramsay",0,"ramsey",0,"rance",0,"rancell",0,"rand",0,"randal",0,"randall",0,"randell",0,"randi",0,"randie",0,"randolf",0,"randolph",0,"randy",0,"ransell",0,"ransom",0,"raoul",0,"raphael",0,"raul",0,"ravi",0,"ravid",0,"raviv",0,"rawley",0,"ray",0,"raymond",0,"raymund",0,"raynard",0,"rayner",0,"raynor",0,"read",0,"reade",0,"reagan",0,"reagen",0,"reamonn",0,"red",0,"redd",0,"redford",0,"reece",0,"reed",0,"rees",0,"reese",0,"reg",0,"regan",0,"regen",0,"reggie",0,"reggis",0,"reggy",0,"reginald",0,"reginauld",0,"reid",0,"reidar",0,"reider",0,"reilly",0,"reinald",0,"reinaldo",0,"reinaldos",0,"reinhard",0,"reinhold",0,"reinold",0,"reinwald",0,"rem",0,"remington",0,"remus",0,"renado",0,"renaldo",0,"renard",0,"renato",0,"renaud",0,"renault",0,"rene",0,"reube",0,"reuben",0,"reuven",0,"rex",0,"rey",0,"reynard",0,"reynold",0,"reynolds",0,"rhett",0,"rhys",0,"ric",0,"ricard",0,"ricardo",0,"riccardo",0,"rice",0,"rich",0,"richard",0,"richardo",0,"richart",0,"richie",0,"richmond",0,"richmound",0,"richy",0,"rick",0,"rickard",0,"rickert",0,"rickey",0,"ricki",0,"rickie",0,"ricky",0,"ricoriki",0,"rik",0,"rikki",0,"riley",0,"rinaldo",0,"ring",0,"ringo",0,"riobard",0,"riordan",0,"rip",0,"ripley",0,"ritchie",0,"roarke",0,"rob",0,"robb",0,"robbert",0,"robbie",0,"robby",0,"robers",0,"robert",0,"roberto",0,"robin",0,"robinet",0,"robinson",0,"rochester",0,"rock",0,"rockey",0,"rockie",0,"rockwell",0,"rocky",0,"rod",0,"rodd",0,"roddie",0,"roddy",0,"roderic",0,"roderich",0,"roderick",0,"roderigo",0,"rodge",0,"rodger",0,"rodney",0,"rodolfo",0,"rodolph",0,"rodolphe",0,"rodrick",0,"rodrigo",0,"rodrique",0,"rog",0,"roger",0,"rogerio",0,"rogers",0,"roi",0,"roland",0,"rolando",0,"roldan",0,"roley",0,"rolf",0,"rolfe",0,"rolland",0,"rollie",0,"rollin",0,"rollins",0,"rollo",0,"rolph",0,"roma",0,"romain",0,"roman",0,"romeo",0,"ron",0,"ronald",0,"ronnie",0,"ronny",0,"rooney",0,"roosevelt",0,"rorke",0,"rory",0,"rosco",0,"roscoe",0,"ross",0,"rossie",0,"rossy",0,"roth",0,"rourke",0,"rouvin",0,"rowan",0,"rowen",0,"rowland",0,"rowney",0,"roy",0,"royal",0,"royall",0,"royce",0,"rriocard",0,"rube",0,"ruben",0,"rubin",0,"ruby",0,"rudd",0,"ruddie",0,"ruddy",0,"rudie",0,"rudiger",0,"rudolf",0,"rudolfo",0,"rudolph",0,"rudy",0,"rudyard",0,"rufe",0,"rufus",0,"ruggiero",0,"rupert",0,"ruperto",0,"ruprecht",0,"rurik",0,"russ",0,"russell",0,"rustie",0,"rustin",0,"rusty",0,"rutger",0,"rutherford",0,"rutledge",0,"rutter",0,"ruttger",0,"ruy",0,"ryan",0,"ryley",0,"ryon",0,"ryun",0,"sal",0,"saleem",0,"salem",0,"salim",0,"salmon",0,"salomo",0,"salomon",0,"salomone",0,"salvador",0,"salvatore",0,"salvidor",0,"sam",0,"sammie",0,"sammy",0,"sampson",0,"samson",0,"samuel",0,"samuele",0,"sancho",0,"sander",0,"sanders",0,"sanderson",0,"sandor",0,"sandro",0,"sandy",0,"sanford",0,"sanson",0,"sansone",0,"sarge",0,"sargent",0,"sascha",0,"sasha",0,"saul",0,"sauncho",0,"saunder",0,"saunders",0,"saunderson",0,"saundra",0,"sauveur",0,"saw",0,"sawyer",0,"sawyere",0,"sax",0,"saxe",0,"saxon",0,"say",0,"sayer",0,"sayers",0,"sayre",0,"sayres",0,"scarface",0,"schuyler",0,"scot",0,"scott",0,"scotti",0,"scottie",0,"scotty",0,"seamus",0,"sean",0,"sebastian",0,"sebastiano",0,"sebastien",0,"see",0,"selby",0,"selig",0,"serge",0,"sergeant",0,"sergei",0,"sergent",0,"sergio",0,"seth",0,"seumas",0,"seward",0,"seymour",0,"shadow",0,"shae",0,"shaine",0,"shalom",0,"shamus",0,"shanan",0,"shane",0,"shannan",0,"shannon",0,"shaughn",0,"shaun",0,"shaw",0,"shawn",0,"shay",0,"shayne",0,"shea",0,"sheff",0,"sheffie",0,"sheffield",0,"sheffy",0,"shelby",0,"shelden",0,"shell",0,"shelley",0,"shellysheldon",0,"shelton",0,"shem",0,"shep",0,"shepard",0,"shepherd",0,"sheppard",0,"shepperd",0,"sheridan",0,"sherlock",0,"sherlocke",0,"sherm",0,"sherman",0,"shermie",0,"shermy",0,"sherwin",0,"sherwood",0,"sherwynd",0,"sholom",0,"shurlock",0,"shurlocke",0,"shurwood",0,"si",0,"sibyl",0,"sid",0,"sidnee",0,"sidney",0,"siegfried",0,"siffre",0,"sig",0,"sigfrid",0,"sigfried",0,"sigismond",0,"sigismondo",0,"sigismund",0,"sigismundo",0,"sigmund",0,"sigvard",0,"silas",0,"silvain",0,"silvan",0,"silvano",0,"silvanus",0,"silvester",0,"silvio",0,"sim",0,"simeon",0,"simmonds",0,"simon",0,"simone",0,"sinclair",0,"sinclare",0,"siward",0,"skell",0,"skelly",0,"skip",0,"skipp",0,"skipper",0,"skippie",0,"skippy",0,"skipton",0,"sky",0,"skye",0,"skylar",0,"skyler",0,"slade",0,"sloan",0,"sloane",0,"sly",0,"smith",0,"smitty",0,"sol",0,"sollie",0,"solly",0,"solomon",0,"somerset",0,"son",0,"sonnie",0,"sonny",0,"spence",0,"spencer",0,"spense",0,"spenser",0,"spike",0,"stacee",0,"stacy",0,"staffard",0,"stafford",0,"staford",0,"stan",0,"standford",0,"stanfield",0,"stanford",0,"stanislas",0,"stanislaus",0,"stanislaw",0,"stanleigh",0,"stanley",0,"stanly",0,"stanton",0,"stanwood",0,"stavro",0,"stavros",0,"stearn",0,"stearne",0,"stefan",0,"stefano",0,"steffen",0,"stephan",0,"stephanus",0,"stephen",0,"sterling",0,"stern",0,"sterne",0,"steve",0,"steven",0,"stevie",0,"stevy",0,"steward",0,"stewart",0,"stillman",0,"stillmann",0,"stinky",0,"stirling",0,"stu",0,"stuart",0,"sullivan",0,"sully",0,"sumner",0,"sunny",0,"sutherlan",0,"sutherland",0,"sutton",0,"sven",0,"svend",0,"swen",0,"syd",0,"sydney",0,"sylas",0,"sylvan",0,"sylvester",0,"syman",0,"symon",0,"tab",0,"tabb",0,"tabbie",0,"tabby",0,"taber",0,"tabor",0,"tad",0,"tadd",0,"taddeo",0,"taddeusz",0,"tadeas",0,"tadeo",0,"tades",0,"tadio",0,"tailor",0,"tait",0,"taite",0,"talbert",0,"talbot",0,"tallie",0,"tally",0,"tam",0,"tamas",0,"tammie",0,"tammy",0,"tan",0,"tann",0,"tanner",0,"tanney",0,"tannie",0,"tanny",0,"tarrance",0,"tate",0,"taylor",0,"teador",0,"ted",0,"tedd",0,"teddie",0,"teddy",0,"tedie",0,"tedman",0,"tedmund",0,"temp",0,"temple",0,"templeton",0,"teodoor",0,"teodor",0,"teodorico",0,"teodoro",0,"terence",0,"terencio",0,"terrance",0,"terrel",0,"terrell",0,"terrence",0,"terri",0,"terrill",0,"terry",0,"thacher",0,"thaddeus",0,"thaddus",0,"thadeus",0,"thain",0,"thaine",0,"thane",0,"thatch",0,"thatcher",0,"thaxter",0,"thayne",0,"thebault",0,"thedric",0,"thedrick",0,"theo",0,"theobald",0,"theodor",0,"theodore",0,"theodoric",0,"thibaud",0,"thibaut",0,"thom",0,"thoma",0,"thomas",0,"thor",0,"thorin",0,"thorn",0,"thorndike",0,"thornie",0,"thornton",0,"thorny",0,"thorpe",0,"thorstein",0,"thorsten",0,"thorvald",0,"thurstan",0,"thurston",0,"tibold",0,"tiebold",0,"tiebout",0,"tiler",0,"tim",0,"timmie",0,"timmy",0,"timofei",0,"timoteo",0,"timothee",0,"timotheus",0,"timothy",0,"tirrell",0,"tito",0,"titos",0,"titus",0,"tobe",0,"tobiah",0,"tobias",0,"tobie",0,"tobin",0,"tobit",0,"toby",0,"tod",0,"todd",0,"toddie",0,"toddy",0,"toiboid",0,"tom",0,"tomas",0,"tomaso",0,"tome",0,"tomkin",0,"tomlin",0,"tommie",0,"tommy",0,"tonnie",0,"tony",0,"tore",0,"torey",0,"torin",0,"torr",0,"torrance",0,"torre",0,"torrence",0,"torrey",0,"torrin",0,"torry",0,"town",0,"towney",0,"townie",0,"townsend",0,"towny",0,"trace",0,"tracey",0,"tracie",0,"tracy",0,"traver",0,"travers",0,"travis",0,"travus",0,"trefor",0,"tremain",0,"tremaine",0,"tremayne",0,"trent",0,"trenton",0,"trev",0,"trevar",0,"trever",0,"trevor",0,"trey",0,"trip",0,"tripp",0,"tris",0,"tristam",0,"tristan",0,"troy",0,"trstram",0,"trueman",0,"trumaine",0,"truman",0,"trumann",0,"tuck",0,"tucker",0,"tuckie",0,"tucky",0,"tudor",0,"tull",0,"tulley",0,"tully",0,"turner",0,"ty",0,"tybalt",0,"tye",0,"tyler",0,"tymon",0,"tymothy",0,"tynan",0,"tyrone",0,"tyrus",0,"tyson",0,"udale",0,"udall",0,"udell",0,"ugo",0,"ulberto",0,"ulick",0,"ulises",0,"ulric",0,"ulrich",0,"ulrick",0,"ulysses",0,"umberto",0,"upton",0,"urbain",0,"urban",0,"urbano",0,"urbanus",0,"uri",0,"uriah",0,"uriel",0,"urson",0,"vachel",0,"vaclav",0,"vail",0,"val",0,"valdemar",0,"vale",0,"valentijn",0,"valentin",0,"valentine",0,"valentino",0,"valle",0,"van",0,"vance",0,"vanya",0,"vasili",0,"vasilis",0,"vasily",0,"vassili",0,"vassily",0,"vaughan",0,"vaughn",0,"verge",0,"vergil",0,"vern",0,"verne",0,"vernen",0,"verney",0,"vernon",0,"vernor",0,"vic",0,"vick",0,"victoir",0,"victor",0,"vidovic",0,"vidovik",0,"vin",0,"vince",0,"vincent",0,"vincents",0,"vincenty",0,"vincenz",0,"vinnie",0,"vinny",0,"vinson",0,"virge",0,"virgie",0,"virgil",0,"virgilio",0,"vite",0,"vito",0,"vittorio",0,"vlad",0,"vladamir",0,"vladimir",0,"von",0,"wade",0,"wadsworth",0,"wain",0,"wainwright",0,"wait",0,"waite",0,"waiter",0,"wake",0,"wakefield",0,"wald",0,"waldemar",0,"walden",0,"waldo",0,"waldon",0,"walker",0,"wallace",0,"wallache",0,"wallas",0,"wallie",0,"wallis",0,"wally",0,"walsh",0,"walt",0,"walther",0,"walton",0,"wang",0,"ward",0,"warde",0,"warden",0,"ware",0,"waring",0,"warner",0,"warren",0,"wash",0,"washington",0,"wat",0,"waverley",0,"waverly",0,"way",0,"waylan",0,"wayland",0,"waylen",0,"waylin",0,"waylon",0,"wayne",0,"web",0,"webb",0,"weber",0,"webster",0,"weidar",0,"weider",0,"welbie",0,"welby",0,"welch",0,"wells",0,"welsh",0,"wendall",0,"wendel",0,"wendell",0,"werner",0,"wernher",0,"wes",0,"wesley",0,"west",0,"westbrook",0,"westbrooke",0,"westleigh",0,"westley",0,"weston",0,"weylin",0,"wheeler",0,"whit",0,"whitaker",0,"whitby",0,"whitman",0,"whitney",0,"whittaker",0,"wiatt",0,"wilbert",0,"wilbur",0,"wilburt",0,"wilden",0,"wildon",0,"wilek",0,"wiley",0,"wilfred",0,"wilfrid",0,"wilhelm",0,"will",0,"willard",0,"willdon",0,"willem",0,"willey",0,"willi",0,"william",0,"willie",0,"willis",0,"willy",0,"wilmar",0,"wilmer",0,"wilt",0,"wilton",0,"win",0,"windham",0,"winfield",0,"winfred",0,"winifield",0,"winn",0,"winnie",0,"winny",0,"winslow",0,"winston",0,"winthrop",0,"wit",0,"wittie",0,"witty",0,"wolf",0,"wolfgang",0,"wolfie",0,"wolfy",0,"wood",0,"woodie",0,"woodman",0,"woodrow",0,"woody",0,"worden",0,"worth",0,"worthington",0,"worthy",0,"wright",0,"wyatan",0,"wyatt",0,"wye",0,"wylie",0,"wyn",0,"wyndham",0,"wynn",0,"xavier",0,"xenos",0,"xerxes",0,"xever",0,"ximenes",0,"ximenez",0,"xymenes",0,"yale",0,"yanaton",0,"yance",0,"yancey",0,"yancy",0,"yank",0,"yankee",0,"yard",0,"yardley",0,"yehudi",0,"yehudit",0,"yorgo",0,"yorgos",0,"york",0,"yorke",0,"yorker",0,"yul",0,"yule",0,"yulma",0,"yuma",0,"yuri",0,"yurik",0,"yves",0,"yvon",0,"yvor",0,"zaccaria",0,"zach",0,"zacharia",0,"zachariah",0,"zacharias",0,"zacharie",0,"zachary",0,"zacherie",0,"zachery",0,"zack",0,"zackariah",0,"zak",0,"zane",0,"zared",0,"zeb",0,"zebadiah",0,"zebedee",0,"zebulen",0,"zebulon",0,"zechariah",0,"zed",0,"zedekiah",0,"zeke",0,"zelig",0,"zerk",0,"zollie",0,"zolly",0);

sub gender {     
    my ($gender) = "m";
    my ($name) = @_;     

    return "f" if exists $female_names{lc($name)};
    return "m" if exists $male_names{lc($name)};

    # Special cases added by orwant     
    return "m" if $name =~ /^Joh?n/; # Jon and John      
    return "m" if $name =~ /^Th?o(m|b)/; # Tom and Thomas and Tomas and Toby  
    return "m" if $name =~ /^Frank/;      
    return "m" if $name =~ /^Bil/;
    return "m" if $name =~ /^Hans/;
    return "m" if $name =~ /^Ron/;
    return "f" if $name =~ /^Ro(z|s)/;
    return "m" if $name =~ /^Walt/;
    return "m" if $name =~ /^Krishna/;
    return "f" if $name =~ /^Tri(c|sh)/; 
    return "m" if $name =~ /^Pas(c|qu)al$/; # Pascal and Pasqual
    return "f" if $name =~ /^Ellie/;
    return "m" if $name =~ /^Anfernee/;

    # Rules from original code
    $gender = "f" if $name =~ /^.*[aeiy]$/;    # most names ending in a/e/i/y are female
    $gender = "f" if $name =~ /^All?[iy]((ss?)|z)on$/; # Allison and variations
    $gender = "f" if $name =~ /een$/; # Cathleen, Eileen, Maureen
    $gender = "m" if $name =~ /^[^S].*r[rv]e?y?$/; # Barry, Larry, Perry
    $gender = "m" if $name =~ /^[^G].*v[ei]$/; # Clive, Dave, Steve
    $gender = "f" if $name =~ /^[^BD].*(b[iy]|y|via)nn?$/; # Carolyn, Gwendolyn, Vivian
    $gender = "m" if $name =~ /^[^AJKLMNP][^o][^eit]*([glrsw]ey|lie)$/; # Dewey, Stanley, Wesley
    $gender = "f" if $name =~ /^[^GKSW].*(th|lv)(e[rt])?$/; # Heather, Ruth, Velvet
    $gender = "m" if $name =~ /^[CGJWZ][^o][^dnt]*y$/; # Gregory, Jeremy, Zachary
    $gender = "m" if $name =~ /^.*[Rlr][abo]y$/; # Leroy, Murray, Roy
    $gender = "f" if $name =~ /^[AEHJL].*il.*$/; # Abigail, Jill, Lillian
    $gender = "f" if $name =~ /^.*[Jj](o|o?[ae]a?n.*)$/; # Janet, Jennifer, Joan
    $gender = "m" if $name =~ /^.*[GRguw][ae]y?ne$/; # Duane, Eugene, Rene
    $gender = "f" if $name =~ /^[FLM].*ur(.*[^eotuy])?$/; # Fleur, Lauren, Muriel
    $gender = "m" if $name =~ /^[CLMQTV].*[^dl][in]c.*[ey]$/; # Lance, Quincy, Vince
    $gender = "f" if $name =~ /^M[aei]r[^tv].*([^cklnos]|([^o]n))$/; # Margaret, Marylou, Miri;  
    $gender = "m" if $name =~ /^.*[ay][dl]e$/; # Clyde, Kyle, Pascale
    $gender = "m" if $name =~ /^[^o]*ke$/; # Blake, Luke, Mi;  
    $gender = "f" if $name =~ /^[CKS]h?(ar[^lst]|ry).+$/; # Carol, Karen, Shar;  
    $gender = "f" if $name =~ /^[PR]e?a([^dfju]|qu)*[lm]$/; # Pam, Pearl, Rachel
    $gender = "f" if $name =~ /^.*[Aa]nn.*$/; # Annacarol, Leann, Ruthann
    $gender = "f" if $name =~ /^.*[^cio]ag?h$/; # Deborah, Leah, Sarah
    $gender = "f" if $name =~ /^[^EK].*[grsz]h?an(ces)?$/; # Frances, Megan, Susan
    $gender = "f" if $name =~ /^[^P]*([Hh]e|[Ee][lt])[^s]*[ey].*[^t]$/; # Ethel, Helen, Gretchen
    $gender = "m" if $name =~ /^[^EL].*o(rg?|sh?)?(e|ua)$/; # George, Joshua, Theodore
    $gender = "f" if $name =~ /^[DP][eo]?[lr].*s$/; # Delores, Doris, Precious
    $gender = "m" if $name =~ /^[^JPSWZ].*[denor]n.*y$/; # Anthony, Henry, Rodney
    $gender = "f" if $name =~ /^K[^v]*i.*[mns]$/; # Karin, Kim, Kristin
    $gender = "m" if $name =~ /^Br[aou][cd].*[ey]$/; # Bradley, Brady, Bruce
    $gender = "f" if $name =~ /^[ACGK].*[deinx][^aor]s$/; # Agnes, Alexis, Glynis
    $gender = "m" if $name =~ /^[ILW][aeg][^ir]*e$/; # Ignace, Lee, Wallace
    $gender = "f" if $name =~ /^[^AGW][iu][gl].*[drt]$/; # Juliet, Mildred, Millicent
    $gender = "m" if $name =~ /^[ABEIUY][euz]?[blr][aeiy]$/; # Ari, Bela, Ira
    $gender = "f" if $name =~ /^[EGILP][^eu]*i[ds]$/; # Iris, Lois, Phyllis
    $gender = "m" if $name =~ /^[ART][^r]*[dhn]e?y$/; # Randy, Timothy, Tony
    $gender = "f" if $name =~ /^[BHL].*i.*[rtxz]$/; # Beatriz, Bridget, Harriet
    $gender = "m" if $name =~ /^.*oi?[mn]e$/; # Antoine, Jerome, Tyrone
    $gender = "m" if $name =~ /^D.*[mnw].*[iy]$/; # Danny, Demetri, Dondi
    return "m" if $name =~ /^[^BG](e[rst]|ha)[^il]*e$/; # Pete, Serge, Shane
    return "f" if $name =~ /^[ADFGIM][^r]*([bg]e[lr]|il|wn)$/; # Angel, Gail, Isabel
    return $gender;
}

1;


