#!/usr/bin/perl

package Amazon::API::Differences;

use strict;
use warnings;

use Data::Dumper;
use English qw(-no_match_vars);
use JSON::PP;
use Test::Differences;
use Test::More 'no_plan';

use parent qw(Class::Accessor::Fast Exporter);

__PACKAGE__->follow_best_practice;

__PACKAGE__->mk_accessors(qw(got expected array_key hash_key));

our @EXPORT_OK = qw(slurp_json diff_json);

caller or __PACKAGE__->main();

########################################################################
sub new {
########################################################################
  my ( $class, @args ) = @_;

  my $options = ref $args[0] ? $args[0] : {@args};

  my $self = $class->SUPER::new($options);

  my $got      = $self->get_got;
  my $expected = $self->get_expected;

  if ( $got && !ref $got ) {
    $self->set_got( slurp_json($got) );
  }

  if ( $expected && !ref $expected ) {
    $self->set_expected( slurp_json($expected) );
  }

  $self->sort_objects;

  return $self;
}

########################################################################
sub sort_object {
########################################################################
  my ( $obj, %args ) = @_;

  my ( $array_key, $hash_key ) = @args{qw(array_key hash_key)};

  my @list = @{ $array_key ? $obj->{$array_key} : $obj };
  @list = sort { $a->{$hash_key} cmp $b->{$hash_key} } @list;

  return [@list]
    if !$array_key;

  $obj->{$array_key} = [@list];

  return $obj;
}

########################################################################
sub sort_objects {
########################################################################
  my ($self) = @_;

  my %args = (
    hash_key  => $self->get_hash_key,
    array_key => $self->get_array_key,
  );

  return
    if !$args{hash_key} && !$args{array_key};

  $self->set_got( sort_obj( $self->get_got, %args ) );

  $self->set_expected( sort_obj( $self->get_expected, %args ) );

  return $self;
}

########################################################################
sub slurp_json {
########################################################################
  my ($file) = @_;

  local $RS = undef;

  open my $fh, '<', $file
    or die "could not open $file\n";

  my $json = <$fh>;
  close $fh;

  return JSON::PP->new->decode($json);
}

########################################################################
sub diff_json {
########################################################################
  my ($self) = @_;

  return eq_or_diff( $self->get_got, $self->get_expected );
}

########################################################################
sub main {
########################################################################

  my $diff = Amazon::API::Differences->new(
    got       => $ARGV[0],
    expected  => $ARGV[1],
    array_key => $ARGV[2],
    hash_key  => $ARGV[3]
  );

  return $diff->diff_json;
}

1;

__END__
