#
# This file is part of Dist-Zilla-Plugin-AuthorSignatureTest
#
# This software is copyright (c) 2017 by Michael Schout.
#
# This is free software; you can redistribute it and/or modify it under
# the same terms as the Perl 5 programming language system itself.
#
package Dist::Zilla::Plugin::AuthorSignatureTest;
$Dist::Zilla::Plugin::AuthorSignatureTest::VERSION = '0.04';
# ABSTRACT: An Author Distribution C<SIGNATURE> Test for L<Dist::Zilla>

use strict;
use warnings;

use Moose;

extends 'Dist::Zilla::Plugin::InlineFiles';
with (
    'Dist::Zilla::Role::PrereqSource',
    'Dist::Zilla::Role::FileMunger',
    'Dist::Zilla::Role::TextTemplate',
);

has force => (
    is       => 'ro',
    isa      => 'Bool',
    default  => sub { 0 },
);


sub register_prereqs {
    my $self = shift;
    $self->zilla->register_prereqs(
        { type => 'requires', phase => 'develop' },
        'Test::Signature' => 0);
}

sub munge_file {
    my $self = shift;
    my ($file) = @_;

    # depending on if ExtraTests ran first, the file might
    # be named xt/author/signature.t, or might be named t/author-signature.t
    return unless (
      $file->name eq 'xt/author/signature.t' ||
      $file->name eq 't/author-signature.t'
    );

    $file->content(
        $self->fill_in_string(
            $file->content,
            {
                force => $self->force ? q{force_} : q{},
            }
        )
    );
}

__PACKAGE__->meta->make_immutable;

=pod

=encoding UTF-8

=head1 NAME

Dist::Zilla::Plugin::AuthorSignatureTest - An Author Distribution C<SIGNATURE> Test for L<Dist::Zilla>

=head1 VERSION

version 0.04

=head1 SYNOPSIS

   [AuthorSignatureTest]
   force = false

=head1 DESCRIPTION

This is an extension of L<Dist::Zilla::Plugin::InlineFile>, providing the
following files:

   xt/author/signature.t - a standard Test::Signature test

This test uses L<Test::Signature> to test the SIGNATURE file in your dist.  If
L<Test::Signature> is not installed, the test is skipped.

=head1 ATTRIBUTES

=head2 force

E.g. C<force = true>

By default L<Test::Signature>::signature_ok does not make test fail if it cannot perform it,
for example, because of missing B<SIGNATURE> file, missing package L<Module::Signature>
or not able to make connection to the key server.

Turn on this feature to make test fail also in the above cases.

=for Pod::Coverage register_prereqs munge_file

=head1 SOURCE

The development version is on github at L<https://github.com/mschout/perl-dist-zilla-plugin-authorsignaturetest>
and may be cloned from L<https://github.com/mschout/perl-dist-zilla-plugin-authorsignaturetest.git>

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website
L<https://github.com/mschout/perl-dist-zilla-plugin-authorsignaturetest/issues>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

Michael Schout <mschout@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Michael Schout.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut

__DATA__

__[ xt/author/signature.t ]__
#!perl -w
# This file was automatically generated by Dist::Zilla::Plugin::AuthorSignatureTest

use strict;
use warnings;
use Test::More;

unless (eval { require Test::Signature; 1 }) {
    plan skip_all => 'Test::Signature is required for this test';
}

Test::Signature::signature_{{ $force }}ok();
done_testing;
