#!/usr/bin/env perl
# vim:ts=4:shiftwidth=4:expandtab

use strict;
use warnings;
use version;

use Test::More;

use App::RetroPAN;
use File::Temp qw/ tempdir /;

my $perl_version = version->new("5.026001");

use_ok 'App::RetroPAN';

my $ua = LWP::UserAgent->new();
my $req = HTTP::Request->new( GET => "https://fastapi.metacpan.org/v1/release/_search" );
my $res = $ua->request($req);

SKIP: {
    skip "Cannot reach API server" if !$res->is_success;

    ok $res->is_success, "API server reachable";

    my ($author, $dist, $url) = App::RetroPAN::find_module_on_date("Moose", "2011-01-01T00:00:00", $perl_version);

    is($dist, "Moose-1.21");
    is($author, "DROLSKY");
    like($url, qr/^http.*Moose-1\.21\.tar\.gz/);

    ($author, $dist, $url) = App::RetroPAN::find_module_on_date("Pod::Spell", "2002-01-01T00:00:00", $perl_version);
    is($dist, "Pod-Spell-1.01");
    is($author, "SBURKE");

    ($author, $dist, $url) = App::RetroPAN::find_module_on_date("Pod::Spell", "2016-02-22T00:00:00", $perl_version);
    is($dist, "Pod-Spell-1.19");
    is($author, "DOLMEN");

    ($author, $dist, $url) = App::RetroPAN::find_module_on_date("Class::C3", "2011-01-01T00:00:00", $perl_version);
    is($dist, "Class-C3-0.23");
    is($author, "FLORA");

    ($author, $dist, $url) = App::RetroPAN::find_module_on_date("DBI", "2015-01-01T00:00:00", $perl_version);
    is($dist, "DBI-1.632");
    is($author, "TIMB");

    my @deps = App::RetroPAN::find_module_dependencies("DROLSKY", "Moose-1.21", $perl_version);
    is_deeply(\@deps, [qw/ Test::Requires Test::Fatal Params::Util Data::OptList Sub::Name Class::MOP Task::Weaken Try::Tiny Package::DeprecationManager List::MoreUtils Sub::Exporter /]);

    my ($dists_required, $dist_to_url) = App::RetroPAN::find_deps_on_date_with_perl("2011-01-01T00:00:00", "Moose");
    isa_ok($dist_to_url, 'HASH');


    ($dists_required, $dist_to_url) = App::RetroPAN::find_deps_on_date_with_perl("2011-01-01T00:00:00", $perl_version, "Moose");
    isa_ok($dist_to_url, 'HASH');

    {
        is_deeply($dists_required, [qw|
            CHOCOLATE/Scope-Guard-0.20
            FLORA/Devel-GlobalDestruction-0.03
            DOY/Package-Stash-0.13
            FLORA/Algorithm-C3-0.08
            FLORA/Class-C3-0.23
            FLORA/MRO-Compat-0.11
            FLORA/Sub-Name-0.05
            DROLSKY/Class-MOP-1.11
            ADAMK/Task-Weaken-1.03
            DOY/Try-Tiny-0.09
            RJBS/Test-Fatal-0.003
            TOKUHIROM/Test-Requires-0.06
            DROLSKY/Package-DeprecationManager-0.10
            VPARSEVAL/List-MoreUtils-0.24
            RJBS/Sub-Install-0.925
            RJBS/Data-OptList-0.106
            ADAMK/Params-Util-1.03
            RJBS/Sub-Exporter-0.982
            DROLSKY/Moose-1.21
        |]);
    }

    my $td = tempdir( CLEANUP => 1 );

    App::RetroPAN::make_minicpan($td, $dists_required, $dist_to_url);
    ok -f "$td/modules/02packages.details.txt";

    ($dists_required, $dist_to_url) = App::RetroPAN::find_deps_on_date_with_perl("2015-01-01T00:00:00", $perl_version, "Catalyst");
    isa_ok($dist_to_url, 'HASH');

    {
        is_deeply($dists_required, [qw|
            MIYAGAWA/Plack-1.0033
            MIYAGAWA/Plack-Middleware-ReverseProxy-0.15
            MRAMBERG/Text-SimpleTable-2.03
            FLORA/Class-C3-Adopt-NEXT-0.13
            FULLERMD/CGI-Struct-1.21
            ETHER/MooseX-MethodAttributes-0.29
            SZABGAB/CGI-Simple-1.115
            SWEETKID/Plack-Middleware-RemoveRedundantBody-0.05
            RSAVAGE/Tree-Simple-1.24
            RSAVAGE/Tree-Simple-VisitorFactory-0.12
            EBHANSSEN/Test-Trap-v0.3.0
            ETHER/MooseX-ConfigFromFile-0.13
            BRICAS/Config-Any-0.24
            ETHER/MooseX-SimpleConfig-0.11
            ETHER/Moose-2.1403
            ETHER/MooseX-Getopt-0.65
            MIYAGAWA/Plack-0.9956
            FLORA/MooseX-Emulate-Class-Accessor-Fast-0.00903
            SIMONW/Module-Pluggable-5.1
            SWEETKID/Plack-Middleware-FixMissingBodyInRedirect-0.12
            FLORA/Plack-Test-ExternalServer-0.01
            INGY/Inline-0.78
            INGY/Pegex-0.59
            PARDUS/File-BaseDir-0.03
            PARDUS/File-MimeInfo-0.9
            URI/File-ReadBackwards-1.05
            FREW/IO-All-0.85
            INGY/YAML-LibYAML-0.55
            INGY/Inline-C-0.71
            DAGOLDEN/Dist-Zilla-Plugin-OSPrereqs-0.005
            APOCAL/Pod-Spell-CommonMistakes-1.002
            LEONT/ExtUtils-Helpers-0.022
            LEONT/ExtUtils-InstallPaths-0.010
            REHSACK/Config-AutoConf-0.309
            DAGOLDEN/inc-latest-0.500
            REHSACK/Data-Tumbler-0.008
            SHLOMIF/File-Find-Object-v0.2.12
            ISHIGAKI/Archive-Any-Lite-0.10
            AUDREYT/Algorithm-Diff-XS-0.04
            TYPESTER/Array-Diff-0.07
            DOMM/Pod-Strip-1.02
            DOMM/Module-ExtractUse-0.33
            SUKRIA/Test-TinyMocker-0.05
            XSAWYERX/MetaCPAN-API-0.44
            KENTNL/Dist-Zilla-Plugin-Git-Remote-Check-0.1.2
            DAGOLDEN/superclass-0.003
            DAGOLDEN/HTTP-Tiny-UA-0.003
            KENTNL/Net-Travis-API-0.001001
            KENTNL/Dist-Zilla-UtilRole-MaybeZilla-0.001000
            FREW/Log-Contextual-0.006004
            FREW/Data-Dumper-Concise-2.022
            KENTNL/Log-Contextual-LogDispatchouli-0.001000
            KENTNL/Dist-Zilla-Util-Git-Wrapper-0.004001
            CJM/Dist-Zilla-Plugin-Git-2.029
            IZUT/Date-Simple-3.03
            RUBYKAT/Dist-Zilla-Plugin-GitFmtChanges-0.005
            ALEXBIO/Dist-Zilla-Plugin-Test-CheckManifest-0.04
            ALEXBIO/Dist-Zilla-Plugin-Clean-0.07
            BBYRD/Dist-Zilla-TravisCI-1.12
            BBYRD/Dist-Zilla-Role-PluginBundle-Merged-1.00
            BBYRD/Dist-Zilla-PluginBundle-Prereqs-0.93
            RJBS/Dist-Zilla-5.029
            SWESTRUP/Dist-Zilla-Plugin-ReportPhase-0.03
            ADAMK/Module-Manifest-1.08
            ETHER/Test-DistManifest-1.012
            DOHERTY/Dist-Zilla-Plugin-Test-DistManifest-2.000004
            VPIT/autovivification-0.14
            THALJEF/criticism-1.02
            DOHERTY/utf8-all-0.015
            HIDEAKIO/autolocale-0.07
            CHOCOLATE/B-Hooks-OP-Annotation-0.44
            CHOCOLATE/true-0.18
            DEXTER/namespace-functions-0.0101
            VPIT/subs-auto-0.08
            DCONWAY/Toolkit-0.0.2
            RKITOVER/vendorlib-0.11
            BBYRD/sanity-1.03
            KENTNL/HTTP-Tiny-Mech-1.001001
            PEGI/WWW-Mechanize-GZip-0.12
            YANNK/Find-Lib-1.04
            ANDYA/IPC-ShareLite-0.17
            RJBS/Cache-Cache-1.07
            OALDERS/WWW-Mechanize-Cached-1.44
            PEGI/WWW-Mechanize-Cached-GZip-0.12
            BBYRD/Dist-Zilla-Role-MetaCPANInterfacer-0.97
            DOHERTY/Dist-Zilla-Plugin-InstallRelease-0.008
            ETHER/Dist-Zilla-Plugin-Test-CheckDeps-0.012
            ETHER/Dist-Zilla-Plugin-CheckBin-0.007
            RJBS/Pod-Weaver-4.010
            ALEXBIO/Dist-Zilla-Plugin-GitHub-0.39
            RKITOVER/Dist-Zilla-Plugin-Git-PushInitial-0.02
            ETHER/Dist-Zilla-Plugin-Run-0.028
            ETHER/Dist-Zilla-Plugin-BlockRelease-0.002
            ROKR/Config-Identity-0.0018
            RJBS/Software-License-0.103010
            GFUJI/Mouse-2.4.1
            BINGOS/Dist-Zilla-Plugin-Meta-Dynamic-Config-0.04
            CJM/Dist-Zilla-Plugins-CJM-4.26
            YANICK/Dist-Zilla-Plugin-Covenant-0.1.1
            JQUELIN/Dist-Zilla-Plugin-Prepender-1.112280
            ETHER/URI-1.65
            DROLSKY/URI-FromHash-0.04
            ETHER/MooseX-Types-URI-0.07
            HAARG/App-FatPacker-0.010002
            MLEHMANN/Perl-Strip-1.1
            MIYAGAWA/App-cpanminus-1.6930
            RSCHUPP/Module-ScanDeps-1.17
            MSCHILLI/libwww-perl-6.08
            RSCHUPP/PAR-Dist-0.49
            BINGOS/Module-Install-1.14
            PHRED/Archive-Zip-1.39
            SREZIC/Parse-CPAN-Packages-Fast-0.08
            LEONT/Dist-Zilla-Plugin-PPPort-0.007
            RWSTAUNER/Dist-Zilla-Role-DynamicConfig-1.002
            RWSTAUNER/Dist-Zilla-Role-Stash-Plugins-1.006
            RWSTAUNER/Dist-Zilla-Stash-PodWeaver-1.005
            ETHER/MooseX-Types-Common-0.001012
            DROLSKY/MooseX-Params-Validate-0.19
            RSRCHBOY/MooseX-Types-Tied-0.003
            ETHER/MooseX-Types-0.45
            ILMARI/Parse-Method-Signatures-1.003016
            DROLSKY/MooseX-AttributeHelpers-0.23
            ETHER/MooseX-Role-WithOverloading-0.16
            ETHER/Pod-Coverage-Moose-0.05
            ETHER/Devel-Declare-0.006017
            JROCKWAY/Context-Preserve-0.01
            FLORA/MooseX-Meta-TypeConstraint-ForceCoercion-0.01
            DRUOSO/MooseX-AuthorizedMethods-0.006
            DRUOSO/MooseX-TransactionalMethods-0.008
            ETHER/MooseX-Method-Signatures-0.48
            ETHER/MooseX-AlwaysCoerce-0.21
            DROLSKY/MooseX-ClassAttribute-0.27
            RSRCHBOY/MooseX-Util-0.003
            RSRCHBOY/MooseX-CurriedDelegation-0.002
            RSRCHBOY/MooseX-TraitFor-Meta-Class-BetterAnonClassNames-0.002002
            ETHER/MooseX-Types-LoadableClass-0.013
            RSRCHBOY/MooseX-AbstractMethod-0.004
            RSRCHBOY/MooseX-NewDefaults-0.004
            RSRCHBOY/Reindeer-0.017
            RSRCHBOY/Dist-Zilla-Role-RegisterStash-0.003
            BBB/File-NFSLock-1.27
            KENTNL/File-ShareDir-ProjectDistDir-1.000004
            XENO/Pod-Spell-1.15
            RJBS/Dist-Zilla-Plugin-TaskWeaver-0.101627
            DAGOLDEN/Exception-Class-TryCatch-1.13
            DAGOLDEN/Getopt-Lucid-1.05
            JTBRAUN/Parse-RecDescent-1.967009
            DAGOLDEN/Pod-WikiDoc-0.20
            RJBS/Pod-Elemental-Transformer-WikiDoc-0.093003
            MARCEL/Dist-Zilla-Plugin-Bugtracker-1.111080
            APOCAL/Pod-Weaver-Section-Support-1.007
            NUFFIN/Tie-RefHash-Weak-0.09
            FERREIRA/Devel-Hide-0.0009
            ETHER/Hash-Util-FieldHash-Compat-0.08
            DOY/MooseX-InsideOut-0.106
            DEXTER/Symbol-Util-0.0203
            DEXTER/constant-boolean-0.02
            DEXTER/Exception-Base-0.25
            DEXTER/Test-Assert-0.0504
            DEXTER/Test-Unit-Lite-0.1202
            DEXTER/MooseX-GlobRef-0.0701
            CHROMATIC/UNIVERSAL-can-1.20140328
            KENTNL/Dist-Zilla-Plugin-Prereqs-Soften-0.005001
            RJBS/Perl-Critic-Lax-0.011
            KENTNL/Dist-Zilla-Util-BundleInfo-1.001003
            RJBS/Pod-Elemental-0.103004
            RJBS/Pod-Elemental-Transformer-List-0.102000
            KENTNL/Dist-Zilla-Plugin-MetaProvides-2.001002
            KWILLIAMS/Path-Class-0.35
            KENTNL/Path-ScanINC-1.000002
            KENTNL/CPAN-Changes-Group-Dependencies-Details-0.001003
            KENTNL/CPAN-Changes-Dependencies-Details-0.001004
            FAYLAND/Net-GitHub-0.69
            ELLIOTJS/Perl-Critic-Compatibility-1.001
            YVES/Sereal-Decoder-3.004
            YVES/Sereal-Encoder-3.004
            INGY/Data-Denter-0.15
            ARISTOTLE/Bencode-1.4
            BOBTFISH/PHP-Serialization-0.34
            ILYAZ/FreezeThaw-0.5001
            TLINDEN/Config-General-2.56
            MIKEWONG/XML-Dumper-0.81
            DPARIS/Crypt-Blowfish-2.14
            ORCLEV/Convert-Bencode-1.03
            MIKO/Data-Taxi-0.96
            SALVA/Compress-PPMd-0.11
            IWADE/Convert-Bencode_XS-0.06
            TODDR/XML-Parser-2.43
            BJOERN/XML-SAX-Expat-0.51
            GRANTM/XML-Simple-2.20
            LDS/Crypt-CBC-2.33
            NEELY/Data-Serializer-0.60
            ROUZIER/Data-Serializer-Sereal-1.05
            ADAMK/File-PathList-1.04
            THALJEF/Perl-Critic-StricterSubs-0.03
            THALJEF/Perl-Critic-1.123
            MITHALDU/PPI-1.220
            WYANT/Perl-Critic-Policy-Variables-ProhibitUnusedVarsStricter-0.003
            ELLIOTJS/Perl-Critic-Swift-v1.0.3
            HAARG/CHI-0.58
            PETDANCE/Perl-Critic-Bangs-1.10
            KRYDE/Pod-MinimumVersion-50
            KRYDE/Perl-Critic-Pulp-88
            THALJEF/Perl-Critic-Deprecated-1.119
            RJBS/Perl-Critic-Tics-0.009
            NEILB/Time-Duration-Parse-0.11
            SHLOMIF/Digest-JHash-0.08
            MATEU/MooX-Types-MooseLike-0.27
            MATEU/MooX-Types-MooseLike-Numeric-1.02
            REHSACK/Hash-MoreUtils-0.05
            SORTIZ/LMDB_File-0.07
            GBARR/TimeDate-2.30
            ETHER/Test-Class-0.48
            KENTNL/CHI-Driver-LMDB-0.002003
            MARCELO/Perl-Critic-Itch-1.0
            THALJEF/Perl-Critic-More-1.003
            BRICAS/CPAN-Changes-0.30
            KENTNL/CPAN-Meta-Prereqs-Diff-0.001002
            KENTNL/CPAN-Changes-Group-Dependencies-Stats-0.002004
            KENTNL/Dist-Zilla-Role-BundleDeps-0.002001
            FAYLAND/Dist-Zilla-Plugin-PerlTidy-0.19
            KENTNL/Dist-Zilla-Util-CurrentCmd-0.002002
            RSRCHBOY/TAP-SimpleOutput-0.002
            RSRCHBOY/Test-Moose-More-0.025
            SWALTERS/autobox-Core-1.28
            RSRCHBOY/autobox-Junctions-0.001
            RSRCHBOY/MooseX-AttributeShortcuts-0.024
            DROLSKY/MooseX-StrictConstructor-0.19
            ROKR/Hash-Merge-Simple-0.051
            KENTNL/Dist-Zilla-Util-EmulatePhase-1.001001
            KENTNL/Dist-Zilla-PluginBundle-Author-KENTNL-2.022003
            DOLMEN/Dist-Zilla-Plugin-ReportVersions-Tiny-1.10
            DAGOLDEN/Dist-Zilla-Plugin-CheckExtraTests-0.025
            KENTNL/Module-Data-0.007
            KENTNL/Dist-Zilla-Plugin-Prereqs-MatchInstalled-1.001002
            KENTNL/Dist-Zilla-Plugin-Prereqs-MatchInstalled-All-1.001000
            DAGOLDEN/Dist-Zilla-Plugin-RemovePhasedPrereqs-0.002
            DAGOLDEN/Dist-Zilla-Plugin-Pod2Readme-0.001
            KGRENNAN/Net-OAuth-0.28
            ETHER/Dist-Zilla-Plugin-MungeFile-WithDataSection-0.007
            KENTNL/Git-Wrapper-Plus-0.004010
            KENTNL/Dist-Zilla-Plugin-Git-NextVersion-Sanitized-0.002002
            KENTNL/Dist-Zilla-Util-ConfigDumper-0.003006
            KENTNL/Test-File-ShareDir-1.000005
            KENTNL/Dist-Zilla-Role-Version-Sanitize-0.002001
            KENTNL/MooX-Lsub-0.002000
            KENTNL/Generic-Assertions-0.001001
            KENTNL/Dist-Zilla-Plugin-Author-KENTNL-RecommendFixes-0.004001
            JGOFF/Pod-Simple-LaTeX-0.06
            ILMARI/bareword-filehandles-0.003
            ZEFRAM/Lexical-SealRequireHints-0.007
            ILMARI/multidimensional-0.011
            HAARG/strictures-1.005005
            MATTP/Data-Perl-0.002009
            MATTP/MooX-HandlesVia-0.001005
            FRIEDO/namespace-sweep-0.001
            ZEFRAM/DynaLoader-Functions-0.002
            ZEFRAM/Devel-CallChecker-0.006
            VPIT/indirect-0.33
            ZEFRAM/Devel-CallParser-0.002
            DOY/Parse-Keyword-0.08
            TOBYINK/Type-Tiny-1.000005
            TOBYINK/Return-Type-0.005
            ZEFRAM/Data-Alias-1.18
            TOBYINK/Dist-Inkt-Role-Test-0.002
            TOBYINK/Dist-Inkt-Role-Release-0.004
            ISHIGAKI/App-CPANTS-Lint-0.05
            TOBYINK/Dist-Inkt-Role-Test-Kwalitee-0.002
            TOBYINK/Dist-Inkt-Role-Hg-0.004
            TOBYINK/MooseX-AttributeTags-0.002
            RGARCIA/Test-LongString-0.17
            TOBYINK/RDF-DOAP-0.012
            VOJ/RDF-NS-20140910
            TOBYINK/RDF-TriN3-0.206
            TOBYINK/PerlX-Maybe-XS-1.001
            TOBYINK/PerlX-Maybe-1.001
            TOBYINK/Scalar-Does-0.202
            TOBYINK/RDF-TrineX-Functions-0.005
            SALVA/Sort-Key-1.33
            TOBYINK/RDF-Prefixes-0.005
            TOBYINK/match-simple-0.009
            TOBYINK/match-simple-XS-0.001
            TOBYINK/Sub-Infix-0.004
            TOBYINK/RDF-TrineX-Serializer-MockTurtleSoup-0.006
            MSCHWERN/Object-ID-v0.1.2
            TOBYINK/lexical-underscore-0.003
            TOBYINK/MooX-Struct-0.013
            ETHER/MooseX-Traits-0.12
            DROLSKY/DateTime-1.12
            ETHER/MooseX-Types-DateTime-0.10
            DROLSKY/Dist-Zilla-Plugin-Conflicts-0.17
            ADAMK/Test-Inline-2.213
            DAGOLDEN/Dist-Zilla-Plugin-BumpVersionAfterRelease-0.008
            ETHER/Dist-Zilla-Plugin-EnsurePrereqsInstalled-0.008
            ETHER/MooseX-Types-Structured-0.30
            ZEFRAM/B-Hooks-OP-Check-0.19
            VPIT/Scope-Upper-0.25
            DAGOLDEN/Test-Number-Delta-1.06
            XAOC/ExtUtils-Depends-0.403
            FLORA/B-Hooks-OP-PPAddr-0.03
            VPIT/Variable-Magic-0.55
            ASH/TryCatch-1.003002
            TODDR/YAML-Syck-1.29
            DOWENS/JSON-DWIW-0.47
            ETHER/JSON-Any-1.38
            OVID/Test-JSON-0.11
            SHLOMIF/XML-LibXML-2.0117
            SIMONFLK/Test-MockModule-0.05
            DWHEELER/DBIx-Connector-0.53
            FXN/Algorithm-Combinatorics-0.27
            MAKAMAKA/Text-CSV-1.32
            SHLOMIF/Text-Aligner-0.12
            SHLOMIF/Text-Table-1.130
            ISHIGAKI/DBD-SQLite-1.46
            XENO/Dist-Zilla-Plugin-OurPkgVersion-0.005001
            DAGOLDEN/Dist-Zilla-Plugin-CheckMetaResources-0.001
            DAGOLDEN/Dist-Zilla-Plugin-InsertCopyright-0.005
            PERIGRIN/XML-NamespaceFactory-1.02
            PERIGRIN/XML-CommonNS-0.06
            TOBYINK/MooseX-ArrayRef-0.005
            ABW/XML-Namespace-0.02
            SHLOMIF/Error-0.17022
            JZUCKER/DBD-RAM-0.072
            DARNOLD/DBD-Amazon-0.10
            SDOWIDEIT/AnyData-0.11
            REHSACK/DBD-AnyData-0.110
            DARREN/DBD-Google-0.05
            REHSACK/SQL-Statement-1.405
            STEFFENW/DBD-PO-2.10
            TIMB/DBI-1.632
            HMBRAND/Text-CSV_XS-1.12
            HMBRAND/DBD-CSV-0.46
            PERIGRIN/XML-NamespaceSupport-1.11
            GRANTM/XML-SAX-Base-1.08
            GRANTM/XML-SAX-0.99
            RJBS/Data-UUID-1.220
            GWILLIAMS/RDF-Trine-1.011
            TOBYINK/Dist-Inkt-DOAP-0.022
            AUDREYT/Module-Signature-0.73
            TOBYINK/RDF-DOAP-Lite-0.002
            DAGOLDEN/Types-Path-Tiny-0.005
            KENTNL/Path-IsDev-1.001002
            KENTNL/Path-FindDev-0.5.2
            TOBYINK/Dist-Inkt-0.023
            DAGOLDEN/Dist-Zilla-Plugin-AppendExternalData-0.003
            DAGOLDEN/Test-Filename-0.03
            DAGOLDEN/Path-Iterator-Rule-1.008
            TOBYINK/Test-Tabs-0.005
            TOBYINK/Dist-Inkt-Profile-TOBYINK-0.023
            TOBYINK/Kavorka-0.036
            TOBYINK/Types-ReadOnly-0.002
            GFUJI/Hash-FieldHash-0.14
            TOBYINK/Exporter-Tiny-0.042
            TOBYINK/Type-Tie-0.009
            TOBYINK/Type-Tiny-XS-0.012
            EXODIST/Meta-Builder-0.003
            EXODIST/Exporter-Declare-0.113
            EXODIST/Fennec-Lite-0.004
            EXODIST/Mock-Quick-1.108
            EXODIST/Test-Exception-LessClever-0.006
            DRTECH/Search-Elasticsearch-1.17
            PHRED/Any-URI-Escape-0.01
            ARODLAND/FCGI-ProcManager-0.25
            ETHER/FCGI-0.77
            TOKUHIROM/Test-TCP-2.06
            MIYAGAWA/Devel-StackTrace-AsHTML-0.14
            GETTY/HTTP-Body-1.19
            GAAS/HTTP-Message-6.06
            DDICK/Test-MockTime-0.13
            KAZEBURO/Time-TZOffset-0.03
            KAZEBURO/POSIX-strftime-Compiler-0.40
            KAZEBURO/Apache-LogFormat-Compiler-0.32
            MIYAGAWA/Hash-MultiValue-0.15
            DOY/Stream-Buffered-0.03
            MIYAGAWA/Filesys-Notify-Simple-0.12
            MARKOV/MIME-Types-2.09
            MIYAGAWA/HTTP-Server-Simple-PSGI-0.16
            GAAS/LWP-Protocol-http10-6.03
            RWSTAUNER/Config-MVP-Slicer-0.302
            RWSTAUNER/Dist-Zilla-Config-Slicer-0.200
            MIYAGAWA/Dist-Zilla-Plugin-Prereqs-FromCPANfile-0.08
            RWSTAUNER/Dist-Zilla-Role-PluginBundle-PluginRemover-0.102
            MIYAGAWA/Dist-Zilla-Plugin-ReversionOnRelease-0.05
            DAGOLDEN/Dist-Zilla-Plugin-CheckChangesHasContent-0.007
            MIYAGAWA/Dist-Zilla-Plugin-LicenseFromModule-0.05
            MIYAGAWA/Dist-Zilla-Plugin-NameFromDirectory-0.03
            MIYAGAWA/Dist-Milla-v1.0.9
            MIYAGAWA/CGI-Compile-0.18
            MIYAGAWA/CGI-Emulate-PSGI-0.20
            KASEI/Class-Accessor-0.34
            RSAVAGE/Crypt-PasswdMD5-1.40
            CHANSEN/Authen-Simple-0.5
            CHANSEN/Authen-Simple-Passwd-0.6
            FLORA/HTTP-Request-AsCGI-1.2
            AVAR/Hijk-0.18
            DANKOGAI/URI-Escape-XS-0.12
            SRI/Mojolicious-5.70
            AVIF/Time-Duration-1.1
            PERLANCAR/Progress-Any-0.18
            SHARYANTO/Perinci-Sub-DepChecker-0.10
            PERLANCAR/Sah-0.9.31
            YANICK/File-ShareDir-Tarball-0.2.2
            SHARYANTO/Text-sprintfn-0.06
            ADAMK/Algorithm-Dependency-1.110
            SHARYANTO/Data-ModeMerge-0.30
            SHARYANTO/Version-Util-0.71
            NEILB/Lingua-EN-Numbers-Ordinate-1.03
            PERLANCAR/String-Indent-0.01
            CHOCOLATE/Scalar-Util-Numeric-0.40
            PERLANCAR/Data-Sah-0.36
            PERLANCAR/Perinci-Sub-Wrapper-0.67
            PERLANCAR/Package-MoreUtil-0.58
            PERLANCAR/Module-Path-More-0.27
            PERLANCAR/Riap-1.2.2
            PERLANCAR/Perinci-Access-Base-0.32
            CHAMAS/Tie-Cache-0.21
            ETJ/Event-1.23
            MUIR/IO-Event-0.813
            EXODIST/Test-SharedFork-0.29
            ANDYA/Data-Structure-Util-0.15
            MLEHMANN/Async-Interrupt-1.2
            MLEHMANN/Guard-1.023
            MLEHMANN/EV-4.18
            MLEHMANN/AnyEvent-7.08
            MUIR/File-Flock-2014.01
            CRENZ/Module-Find-0.12
            ETHER/Any-Moose-0.24
            ETHER/Crypt-Random-Source-0.10
            JAWNSY/Math-Random-ISAAC-1.004
            JAWNSY/Math-Random-ISAAC-XS-1.004
            MKANAT/Math-Random-Secure-0.06
            SHARYANTO/UUID-Random-Patch-UseMRS-0.01
            SHARYANTO/Crypt-Password-Util-0.09
            SHARYANTO/Unix-Passwd-File-0.12
            SHARYANTO/String-Wildcard-Bash-0.01
            PERLANCAR/Complete-Getopt-Long-0.27
            PERLER/UUID-Random-0.04
            SHARYANTO/Filesys-Cap-0.01
            PERLANCAR/Complete-Path-0.06
            PERLANCAR/Complete-Util-0.21
            SPROUT/Sub-Delete-1.00002
            SHARYANTO/Monkey-Patch-Action-0.04
            PERLANCAR/Complete-0.10
            PERLANCAR/Complete-Bash-0.15
            PERLANCAR/Perinci-Sub-Complete-0.70
            ZEFRAM/Module-List-0.003
            PERLANCAR/String-Trim-More-0.02
            PERLANCAR/Perinci-Object-0.21
            PERLANCAR/Perinci-Examples-0.44
            PERLANCAR/Perinci-Access-Perl-0.79
            PERLANCAR/Data-Dmp-0.05
            PERLANCAR/Data-Dump-OneLine-0.06
            SHARYANTO/Gen-Test-Rinci-FuncResult-0.04
            PERLANCAR/Perinci-Sub-Util-0.40
            SHARYANTO/Perinci-Sub-GetArgs-Array-0.14
            PERLANCAR/Getopt-Long-Util-0.79
            PERLANCAR/Rinci-1.1.70
            PERLANCAR/Perinci-Sub-Normalize-0.08
            PERLANCAR/Data-Clean-JSON-0.20
            SHARYANTO/Data-Sah-Normalize-0.01
            PERLANCAR/String-LineNumber-0.01
            PERLANCAR/Function-Fallback-CoreOrPP-0.06
            PERLANCAR/String-PerlQuote-0.01
            DAGOLDEN/Log-Any-1.02
            GFUJI/Data-Clone-0.004
            IBB/Acme-Damn-0.06
            PERLANCAR/Perl-osnames-0.11
            PERLANCAR/Perinci-Sub-GetArgs-Argv-0.60
            PERLANCAR/DefHash-1.0.8
            PERLANCAR/Test-Rinci-0.09
            DROLSKY/Dist-Zilla-Plugin-SurgicalPodWeaver-0.0023
            SHARYANTO/Log-Any-Adapter-Callback-0.09
            MICKEY/MetaCPAN-Client-1.008000
            ADAMK/asa-1.03
            NUFFIN/IO-Handle-Util-0.01
            SZABGAB/Parallel-ForkManager-1.07
            DROLSKY/Test-DependentModules-0.20
            HDP/Config-INI-Reader-Ordered-0.011
            BARBIE/Test-CPAN-Meta-JSON-0.15
            JROCKWAY/Test-YAML-Valid-0.04
            BARBIE/Test-CPAN-Meta-YAML-0.22
            ETHER/Dist-Zilla-Plugin-Test-CheckBreaks-0.011
            ETHER/Dist-Zilla-Plugin-Breaks-0.003
            DAGOLDEN/MooseX-Types-Stringlike-0.003
            ETHER/Dist-Zilla-Plugin-MakeMaker-Awesome-0.32
            ETHER/Dist-Zilla-Plugin-DynamicPrereqs-0.009
            ETHER/Dist-Zilla-Plugin-OptionalFeature-0.020
            ZOFFIX/WWW-Pastebin-PastebinCom-Create-1.003
            CFRANKS/Browser-Open-0.04
            KING/Clipboard-0.13
            JESSE/HTTP-Server-Simple-0.44
            GAAS/HTML-Form-6.03
            GAAS/libwww-perl-5.36
            GAAS/HTML-Parser-3.71
            GAAS/libwww-perl-5.22
            CJM/HTML-Tree-5.03
            ETHER/WWW-Mechanize-1.73
            ETHER/App-Nopaste-1.002
            FERREIRA/Carp-Always-0.13
            NEILB/Clone-PP-1.06
            BINGOS/Sort-Naturally-1.03
            GARU/Data-Printer-0.35
            DAGOLDEN/Devel-OverrideGlobalRequire-0.001
            ILMARI/Class-Unload-0.08
            DOY/Class-Refresh-0.07
            GAAS/Data-Dump-1.22
            NEILB/Carp-Assert-0.21
            MSTEVENS/Proc-InvokeEditor-1.07
            DOY/Reply-0.37
            RRWO/Pod-Readme-v1.1.2
            FAYLAND/Dist-Zilla-Plugin-ReadmeFromPod-0.30
            KENTNL/Dist-Zilla-Plugin-RewriteVersion-Sanitized-0.001004
            KENTNL/Dist-Zilla-Plugin-Test-Compile-PerFile-0.003000
            KENTNL/Dist-Zilla-Plugin-Git-NextRelease-0.004000
            WYANT/PPIx-Regexp-0.037
            DARREN/String-Format-1.17
            RJBS/Email-Address-1.905
            ELLIOTJS/PPIx-Utilities-1.001000
            RURBAN/B-Keywords-1.13
            SANKO/Readonly-2.00
            ROODE/Readonly-XS-1.05
            THALJEF/Test-Perl-Critic-1.02
            DOHERTY/Dist-Zilla-Plugin-Test-Synopsis-2.000005
            JQUELIN/Dist-Zilla-Plugin-Test-Perl-Critic-2.112410
            DOHERTY/Dist-Zilla-Plugin-Test-CPAN-Meta-JSON-0.003
            XENO/Dist-Zilla-Plugin-Test-Version-0.003001
            APOCAL/Pod-Weaver-Section-SeeAlso-1.003
            GFUJI/Test-Vars-0.005
            DOHERTY/Dist-Zilla-Plugin-Test-UnusedVars-2.000006
            ETHER/Dist-Zilla-Plugin-Test-EOL-0.16
            MSCHWERN/URI-Find-20140709
            APOCAL/Test-Pod-No404s-0.02
            RWSTAUNER/Dist-Zilla-Plugin-Test-Pod-No404s-1.001
            ETHER/MooseX-SlurpyConstructor-1.2
            ETHER/Dist-Zilla-Plugin-GenerateFile-ShareDir-0.005
            HAARG/Dist-Zilla-Plugin-CopyFilesFromRelease-0.004
            ETHER/Dist-Zilla-Plugin-AuthorityFromModule-0.004
            ETHER/Dist-Zilla-Plugin-CheckIssues-0.007
            ETHER/Dist-Zilla-Plugin-CheckSelfDependency-0.009
            ETHER/Dist-Zilla-Plugin-Test-NoTabs-0.09
            KEEDI/Pod-Weaver-Section-Contributors-0.009
            APOCAL/Dist-Zilla-Plugin-MinimumPerl-1.006
            BRADFITZ/Unicode-CheckUTF8-1.03
            SYP/Test-Mojibake-1.0
            SYP/Dist-Zilla-Plugin-MojibakeTests-0.7
            ETHER/Dist-Zilla-Role-File-ChangeNotification-0.005
            DSKOLL/IO-stringy-2.110
            RWSTAUNER/Pod-Markdown-2.002
            RTHOMPSON/Dist-Zilla-Plugin-ReadmeAnyFromPod-0.142470
            RJBS/Dist-Zilla-Plugin-Git-Describe-0.004
            ETHER/Dist-Zilla-Plugin-Test-Kwalitee-2.11
            KENTNL/Dist-Zilla-PluginBundle-Author-KENTNL-Lite-2.001000
            ETHER/Dist-Zilla-Plugin-MakeMaker-Fallback-0.016
            ETHER/Safe-Isa-1.000005
            ETHER/Dist-Zilla-Plugin-Git-Contributors-0.008
            ABRAXXA/Test-Portability-Files-0.06
            DOHERTY/Dist-Zilla-Plugin-Test-Portability-2.000005
            RTHOMPSON/Dist-Zilla-Plugin-CheckVersionIncrement-0.121750
            GAAS/Digest-SHA1-2.13
            GAAS/Digest-HMAC-1.03
            DOY/Data-Visitor-0.30
            MMIMS/Net-Twitter-4.01006
            DAVECROSS/WWW-Shorten-3.06
            MIYAGAWA/WWW-Shorten-Simple-0.01
            INGY/Spiffy-0.46
            INGY/Test-Base-0.88
            INGY/Test-YAML-1.05
            INGY/YAML-1.13
            SHLOMIF/Config-IniFiles-2.83
            BINGOS/Config-Auto-0.44
            RJBS/Config-INI-0.025
            DOHERTY/Dist-Zilla-Plugin-Twitter-0.026
            RJBS/Dist-Zilla-Plugin-CheckPrereqsIndexed-0.013
            KENTNL/Dist-Zilla-Plugin-MetaData-BuiltWith-1.004001
            RJBS/Config-MVP-2.200008
            RJBS/String-Truncate-1.100602
            RJBS/Pod-Elemental-PerlMunger-0.200002
            RJBS/Dist-Zilla-Plugin-PodWeaver-4.006
            XENO/Test-Version-1.004001
            DOHERTY/Dist-Zilla-Plugin-Test-CPAN-Changes-0.008
            KENTNL/Dist-Zilla-Plugin-Prereqs-Recommend-MatchInstalled-0.003001
            DAVIDO/Set-Scalar-1.29
            RTHOMPSON/Dist-Zilla-Plugin-CopyFilesFromBuild-0.103510
            DAGOLDEN/recommended-0.002
            DAGOLDEN/Test-TempDir-Tiny-0.002
            ROODE/Iterator-0.03
            ROODE/Iterator-Util-0.02
            SMUELLER/Class-XSAccessor-1.19
            SCHWIGON/Data-DPath-0.50
            KENTNL/Dist-Zilla-Util-Test-KENTNL-1.005000
            KENTNL/Dist-Zilla-Plugin-MetaProvides-Package-2.002000
            BBYRD/Dist-Zilla-Plugin-Config-Git-0.92
            SAPER/Devel-SimpleTrace-0.08
            KENTNL/Dist-Zilla-Util-ExpandINI-0.003000
            KENTNL/Dist-Zilla-App-Command-bakeini-0.002000
            KENTNL/MooseX-Has-Sugar-1.000004
            DAGOLDEN/File-chdir-0.1009
            PJF/IPC-System-Simple-1.25
            RSRCHBOY/Dist-Zilla-PluginBundle-Git-CheckFor-0.011
            DAGOLDEN/Version-Next-0.004
            DAGOLDEN/Dist-Zilla-Plugin-CheckStrictVersion-0.001
            APOCAL/Dist-Zilla-Plugin-Authority-1.009
            PHAYLON/syntax-0.004
            FREW/Syntax-Keyword-Junction-0.003008
            CHOCOLATE/Scope-Guard-0.20
            CHOCOLATE/autobox-2.82
            RJBS/Moose-Autobox-0.15
            DOHERTY/Dist-Zilla-Plugin-Test-MinimumVersion-2.000005
            DAGOLDEN/Dist-Zilla-Plugin-Test-ReportPrereqs-0.019
            ETHER/MooseX-Types-Path-Tiny-0.011
            NEILB/Sort-Versions-1.60
            GENEHACK/Git-Wrapper-0.038
            NEILB/Exporter-Lite-0.06
            MOTEMEN/Test-Deep-JSON-0.03
            DAGOLDEN/Dist-Zilla-Plugin-Prereqs-AuthorDeps-0.004
            ETHER/UNIVERSAL-isa-1.20140927
            LEEJO/CGI-4.13
            CHROMATIC/Test-MockObject-1.20140408
            AJGB/Dist-Zilla-Plugin-AutoMetaResources-1.20
            RJBS/App-Cmd-0.326
            RJBS/Throwable-0.200012
            MLEHMANN/Types-Serialiser-1.0
            MLEHMANN/JSON-XS-3.01
            MAKAMAKA/JSON-2.90
            RJBS/String-Errf-0.007
            RJBS/Role-HasMessage-0.006
            PHAYLON/Declare-Constraints-Simple-0.03
            DOY/ExtUtils-MakeMaker-Dist-Zilla-Develop-0.03
            NUFFIN/Tie-ToObject-0.03
            RSRCHBOY/MooseX-MarkAsMethods-0.15
            CHROMATIC/SUPER-1.20141117
            HAARG/Algorithm-C3-0.10
            SHAY/Class-Singleton-1.5
            DROLSKY/List-AllUtils-0.09
            BDFOY/Test-Output-1.03
            PETDANCE/Test-Taint-1.06
            DROLSKY/DateTime-TimeZone-1.83
            DROLSKY/DateTime-Locale-0.45
            DROLSKY/DateTime-Format-Strptime-1.56
            DROLSKY/Class-Factory-Util-1.7
            DROLSKY/DateTime-Format-Builder-0.81
            XMIKEW/DateTime-Format-MySQL-0.05
            HAARG/Role-Tiny-1.003004
            DCONWAY/Lingua-EN-Inflect-1.895
            DROLSKY/Specio-0.11
            ADAMK/Params-Coerce-0.14
            DOY/MooseX-NonMoose-0.26
            XENO/File-chmod-0.40
            ADAMK/Pod-Tests-1.19
            RSAVAGE/Config-Tiny-2.20
            ADAMK/Test-ClassAPI-1.06
            ADAMK/prefork-1.04
            ADAMK/File-Flat-1.04
            JHOBLITT/DateTime-Calendar-Mayan-0.0601
            RKINYON/DBM-Deep-2.0011
            MIYAGAWA/Data-Section-Simple-0.07
            TBONE/Locale-US-2.112150
            LDS/Devel-Cycle-1.12
            PETDANCE/Test-Memory-Cycle-1.04
            LBROCARD/DateTime-Stringify-4.08
            ALEXMV/Module-Refresh-0.17
            THEPLER/MooseX-Types-Path-Class-0.06
            DAGOLDEN/File-pushd-1.009
            RJBS/MooseX-SetOnce-0.200002
            PEVANS/List-UtilsBy-0.09
            KENTNL/Dist-Zilla-Role-Bootstrap-1.000003
            KENTNL/Dist-Zilla-Plugin-Bootstrap-lib-1.001000
            DAGOLDEN/Dist-Zilla-PluginBundle-DAGOLDEN-0.073
            ETHER/aliased-0.33
            ETHER/MooseX-LazyRequire-0.11
            DROLSKY/Log-Dispatch-2.44
            RJBS/Log-Dispatch-Array-1.003
            RJBS/String-Flogger-1.101245
            RJBS/Sub-Exporter-GlobExporter-0.004
            RJBS/Log-Dispatchouli-2.012
            RJBS/IO-TieCombine-1.004
            ETHER/Test-TempDir-0.09
            URI/File-Slurp-9999.19
            BOBTFISH/Directory-Scratch-0.15
            RSRCHBOY/Dist-Zilla-Plugin-ContributorsFromGit-0.014
            ETHER/Devel-PartialDump-0.17
            RJBS/MooseX-Types-Perl-0.101343
            XENO/Dist-Zilla-Plugin-Test-PodSpelling-2.006008
            ETHER/Class-Load-XS-0.09
            STBEY/Carp-Clan-6.04
            BINGOS/Log-Message-0.08
            BINGOS/Log-Message-Simple-0.10
            BINGOS/Term-UI-0.42
            MIYAGAWA/Term-Encoding-0.02
            LEONT/ExtUtils-Config-0.008
            LEONT/Devel-FindPerl-0.012
            NEILB/Module-Path-0.18
            RJBS/Perl-PrereqScanner-1.022
            RJBS/Sub-Exporter-ForMethods-0.100051
            ETHER/Dist-Zilla-Plugin-Test-CleanNamespaces-0.006
            ETHER/Dist-Zilla-Plugin-PromptIfStale-0.032
            ETHER/namespace-autoclean-0.23
            RJBS/Pod-Eventual-0.094001
            BINGOS/Dist-Zilla-Plugin-GithubMeta-0.46
            ETHER/Class-Method-Modifiers-2.11
            HAARG/Moo-1.006001
            RJBS/Role-Identifiable-0.007
            CHORNY/Tie-IxHash-1.23
            LEONT/File-Slurp-Tiny-0.003
            BDFOY/Perl-Version-1.013
            LEONT/Dist-Zilla-Plugin-ModuleBuildTiny-0.008
            ETHER/Dist-Zilla-Plugin-ModuleBuildTiny-Fallback-0.014
            ETHER/YAML-Tiny-1.64
            DROLSKY/Params-Validate-1.13
            RJBS/Sub-Exporter-0.987
            RJBS/Getopt-Long-Descriptive-0.098
            MSCHILLI/Net-HTTP-6.07
            GAAS/File-Listing-6.04
            PETDANCE/HTML-Tagset-3.20
            MIKEM/Net-SSLeay-1.66
            SULLR/IO-Socket-SSL-2.008
            GAAS/WWW-RobotRules-6.02
            GAAS/HTTP-Negotiate-6.01
            GAAS/HTTP-Cookies-6.01
            GAAS/HTTP-Daemon-6.01
            NBEBOUT/NTLM-1.09
            ABH/Mozilla-CA-20141217
            MSCHILLI/LWP-Protocol-https-6.06
            RJBS/CPAN-Uploader-0.103007
            ETHER/Dist-Zilla-Plugin-Test-Compile-2.051
            ETHER/Dist-Zilla-Plugin-Keywords-0.006
            ETHER/Module-Runtime-Conflicts-0.001
            ROBIN/PadWalker-2.0
            RCLAMP/Devel-Caller-2.06
            RCLAMP/Devel-LexAlias-0.05
            SHANCOCK/Perl-Tidy-20140711
            DOY/Eval-Closure-0.11
            RJBS/Config-MVP-Reader-INI-2.101463
            RJBS/String-RewritePrefix-0.007
            ETHER/Class-Load-0.22
            DROLSKY/Package-DeprecationManager-0.13
            DROLSKY/Module-Implementation-0.09
            GFUJI/Test-LeakTrace-0.15
            DOY/Package-Stash-XS-0.28
            DOY/Dist-CheckConflicts-0.11
            DOY/Package-Stash-0.37
            RGARCIA/Sub-Identify-0.09
            ILMARI/Devel-OverloadInfo-0.002
            RJBS/MooseX-OneArgNew-0.004
            RJBS/Mixin-Linewise-0.106
            ETHER/Test-Warnings-0.019
            ETHER/Test-Kwalitee-1.22
            ETHER/B-Hooks-EndOfScope-0.13
            RIBASUSHI/namespace-clean-0.25
            FREW/Sub-Exporter-Progressive-0.001011
            HAARG/Devel-GlobalDestruction-0.13
            ZEFRAM/Module-Runtime-0.014
            ETHER/Import-Into-1.002004
            ETHER/Test-CleanNamespaces-0.16
            ETHER/MooseX-Role-Parameterized-1.08
            RJBS/String-Formatter-0.093220
            DOHERTY/Dist-Zilla-Plugin-InstallGuide-1.200005
            RURBAN/B-Flags-0.11
            RURBAN/Opcodes-0.14
            RURBAN/B-C-1.52
            TOKUHIROM/Devel-CheckBin-0.02
            ETHER/Sub-Name-0.12
            NEILB/PAUSE-Permissions-0.10
            ISHIGAKI/Test-UseAllModules-0.17
            ISHIGAKI/Parse-LocalDistribution-0.14
            ISHIGAKI/Parse-PMFile-0.33
            ADAMK/File-HomeDir-1.00
            TYEMQ/Algorithm-Diff-1.1903
            OVID/Text-Diff-1.41
            DCANTRELL/Test-Differences-0.63
            TMTM/Class-Data-Inheritable-0.08
            DROLSKY/Devel-StackTrace-2.00
            ZOFFIX/Test-Synopsis-0.11
            BOBTFISH/Test-EOL-1.5
            DROLSKY/Exception-Class-1.39
            CHORNY/Test-Warn-0.30
            OVID/Test-Most-0.34
            ISHIGAKI/Test-PAUSE-Permissions-0.05
            KAZEBURO/Cwd-Guard-0.04
            DMUEY/File-Copy-Recursive-0.38
            ADAMK/File-Find-Rule-Perl-1.13
            RJBS/Test-MinimumVersion-0.101081
            ABIGAIL/Regexp-Common-2013031301
            TOKUHIROM/Spellunker-v0.4.0
            SYOHEX/Devel-CheckCompiler-0.05
            HIDEAKIO/Module-Build-XSUtil-0.14
            GOCCY/Compiler-Lexer-0.19
            ISHIGAKI/Perl-MinimumVersion-Fast-0.15
            TOKUHIROM/Test-MinimumVersion-Fast-0.04
            TOKUHIROM/Test-Requires-0.08
            RCLAMP/Number-Compare-0.03
            RCLAMP/Text-Glob-0.09
            RCLAMP/File-Find-Rule-0.33
            ADAMK/File-Remove-1.52
            ADAMK/Test-Object-0.07
            JSTOWE/TermReadKey-2.32
            ETHER/Dist-Zilla-Plugin-VerifyPhases-0.009
            ETHER/Dist-Zilla-PluginBundle-Author-ETHER-0.081
            ETHER/Hook-LexWrap-0.25
            ADAMK/Test-SubCalls-1.09
            GAAS/IO-String-1.08
            DAGOLDEN/PPI-XS-0.902
            HAYASHI/Term-ReadLine-Gnu-1.25
            RJBS/IPC-Run3-0.048
            KWILLIAMS/Probe-Perl-0.03
            ADAMK/Test-Script-1.07
            ADAMK/File-Which-1.09
            TODDR/IO-Tty-1.12
            TODDR/IPC-Run-0.94
            MSCHILLI/Log-Log4perl-1.46
            MSCHILLI/Archive-Tar-Wrapper-0.21
            ETHER/Dist-Zilla-Plugin-OnlyCorePrereqs-0.019
            CHANSEN/Unicode-UTF8-0.60
            DAGOLDEN/Path-Tiny-0.061
            MJD/Text-Template-1.46
            DAGOLDEN/Test-FailWarnings-0.008
            BOBTFISH/MRO-Compat-0.12
            RJBS/Data-Section-0.200006
            LEONT/CPAN-Meta-Check-0.009
            LEONT/Test-CheckDeps-0.010
            BBYRD/Software-License-CCpack-1.11
            REYNOLDS/IO-Capture-0.05
            MIYAGAWA/Module-CPANfile-1.1000
            ISHIGAKI/ExtUtils-MakeMaker-CPANfile-0.06
            ISHIGAKI/Module-CPANTS-Analyse-0.96
            SANBEG/Test-Directory-0.041
            REHSACK/Test-WriteVariants-0.010
            REHSACK/List-MoreUtils-0.402
            ADAMK/Task-Weaken-1.04
            GARU/Clone-0.37
            HAARG/local-lib-2.000015
            LEONT/Module-Build-Tiny-0.039
            APOCAL/Test-Pod-Spelling-CommonMistakes-1.001
            RENEEB/Test-CheckManifest-1.28
            ADAMK/Class-Inspector-1.28
            GWYN/File-ShareDir-Install-0.10
            REHSACK/File-ShareDir-1.102
            DOY/Dist-Zilla-Plugin-PerlVersionPrereqs-0.01
            DAGOLDEN/Class-Tiny-1.000
            BARBIE/Test-CPAN-Meta-0.23
            SARTAK/Test-Spelling-0.20
            DAGOLDEN/Capture-Tiny-0.27
            BOBTFISH/Test-NoTabs-1.3
            DOY/Try-Tiny-0.22
            RJBS/Test-Fatal-0.014
            RCLAMP/Pod-Coverage-0.23
            DAGOLDEN/Sub-Uplevel-0.24
            EXODIST/Test-Exception-0.35
            ANDK/Devel-Symdump-2.14
            NEILB/Test-Pod-Coverage-1.10
            LEONT/PerlIO-utf8_strict-0.005
            ADAMK/Params-Util-1.07
            RJBS/Data-OptList-0.109
            RJBS/Sub-Install-0.928
            ADAMK/Test-NoWarnings-1.04
            RJBS/Test-Deep-0.114
            LEONT/Module-Build-0.4210
            DWHEELER/Test-Pod-1.48
            RJBS/Pod-Coverage-TrustPod-0.100003
            CJM/IO-HTML-1.001
            GAAS/HTTP-Date-6.02
            JDB/Win32-Console-0.10
            GAAS/Encode-Locale-1.03
            GAAS/LWP-MediaTypes-6.02
            BDFOY/Business-ISBN-20001010
            CORION/Test-Without-Module-0.18
            MLEHMANN/common-sense-3.73
            RURBAN/Cpanel-JSON-XS-3.0113
            ETHER/JSON-MaybeXS-1.003003
            JJNAPIORK/Catalyst-Runtime-5.90075
        |], "Why re-invent the wheel?");
    }
}
done_testing();
