package Data::Zipcode::TW;
{
  $Data::Zipcode::TW::VERSION = '1.0';
}
# ABSTRACT: Taiwan zipcode data in a box.

use strict;

## cd doc; perl twzip.pl
my $ZIPCODE = {
  '花蓮縣鳳林' => 975,
  '西螺' => 648,
  '連江縣東引' => 212,
  882 => '澎湖縣望安',
  '彰化縣線西' => 507,
  '七堵區' => 206,
  '屏東縣恆春' => 946,
  '大村' => 515,
  512 => '彰化縣永靖',
  '番路' => 602,
  '臺北市中山區' => 104,
  '嘉義縣溪口' => 623,
  '宜蘭縣冬山' => 269,
  '中區' => 400,
  '新北市萬里區' => 207,
  722 => '臺南市佳里區',
  '桃園縣龜山' => 333,
  634 => '雲林縣褒忠',
  '西屯區' => 407,
  '平溪區' => 226,
  '臺東縣鹿野' => 955,
  '南投縣南投' => 540,
  978 => '花蓮縣瑞穗',
  '屏東縣滿州' => 947,
  '春日' => 942,
  '屏東縣枋山' => 941,
  742 => '臺南市大內區',
  325 => '桃園縣龍潭',
  '永和區' => 234,
  350 => '苗栗縣竹南',
  '山上區' => 743,
  '泰武' => 921,
  '莿桐' => 647,
  540 => '南投縣南投',
  211 => '連江縣莒光',
  337 => '桃園縣大園',
  '關山' => 956,
  '臺南市鹽水區' => 737,
  311 => '新竹縣五峰',
  982 => '花蓮縣卓溪',
  '西嶼' => 881,
  '臺中市大肚區' => 432,
  '龜山' => 333,
  825 => '高雄市橋頭區',
  '臺中市霧峰區' => 413,
  '新城' => 971,
  966 => '臺東縣達仁',
  '新化區' => 712,
  '屏東縣竹田' => 911,
  '臺中市烏日區' => 414,
  '新北市樹林區' => 238,
  236 => '新北市土城區',
  '造橋' => 361,
  '萬榮' => 979,
  '臺北市士林區' => 111,
  '杉林區' => 846,
  234 => '新北市永和區',
  '臺南市將軍區' => 725,
  '東港' => 928,
  '蘆竹' => 338,
  '水林' => 652,
  '大甲區' => 437,
  '桃園縣龍潭' => 325,
  '善化區' => 741,
  '屏東縣潮州' => 920,
  '前金區' => 801,
  '信義' => 556,
  '北投區' => 112,
  '古坑' => 646,
  943 => '屏東縣獅子',
  '內埔' => 912,
  '學甲區' => 726,
  '新北市石門區' => 253,
  '新營區' => 730,
  '蘆洲區' => 247,
  '苗栗' => 360,
  '達仁' => 966,
  '彰化縣埤頭' => 523,
  '苓雅區' => 802,
  438 => '臺中市外埔區',
  508 => '彰化縣和美',
  953 => '臺東縣延平',
  '大里區' => 412,
  221 => '新北市汐止區',
  '佳里區' => 722,
  558 => '南投縣鹿谷',
  '嘉義縣阿里山' => 605,
  '外埔區' => 438,
  260 => '宜蘭縣宜蘭',
  '金門縣金城' => 893,
  '屏東縣南州' => 926,
  '彰化縣埔鹽' => 516,
  638 => '雲林縣麥寮',
  734 => '臺南市六甲區',
  '民雄' => 621,
  '三峽區' => 237,
  506 => '彰化縣福興',
  116 => '臺北市文山區',
  '鼓山區' => 804,
  '南澳' => 272,
  '太平區' => 411,
  '桃園' => 330,
  308 => '新竹縣寶山',
  906 => '屏東縣高樹',
  '羅東' => 265,
  '臺南市大內區' => 742,
  '臺南市下營區' => 735,
  '苗栗縣西湖' => 368,
  '大寮區' => 831,
  '溪口' => 623,
  '苗栗縣三義' => 367,
  205 => '基隆市暖暖區',
  '後壁區' => 731,
  '苗栗縣苗栗' => 360,
  927 => '屏東縣林邊',
  '高雄市湖內區' => 829,
  '高雄市路竹區' => 821,
  '東區' => undef,
  652 => '雲林縣水林',
  975 => '花蓮縣鳳林',
  '嘉義縣民雄' => 621,
  '朴子' => 613,
  '南投縣國姓' => 544,
  '坪林區' => 232,
  '臺中市外埔區' => 438,
  '那瑪夏區' => 849,
  '池上' => 958,
  '土城區' => 236,
  '基隆市七堵區' => 206,
  '頭城' => 261,
  366 => '苗栗縣銅鑼',
  844 => '高雄市六龜區',
  '臺東縣臺東' => 950,
  '高雄市美濃區' => 843,
  '阿里山' => 605,
  272 => '宜蘭縣南澳',
  '苗栗縣南庄' => 353,
  '彰化縣花壇' => 503,
  400 => '臺中市中區',
  744 => '臺南市新市區',
  633 => '雲林縣土庫',
  '屏東縣新埤' => 925,
  '屏東縣泰武' => 921,
  '高雄市鼓山區' => 804,
  700 => '臺南市中西區',
  '臺東縣池上' => 958,
  '富里' => 983,
  413 => '臺中市霧峰區',
  932 => '屏東縣新園',
  '潮州' => 920,
  515 => '彰化縣大村',
  '新莊區' => 242,
  '臺南市玉井區' => 714,
  606 => '嘉義縣中埔',
  '梅山' => 603,
  724 => '臺南市七股區',
  '新竹縣芎林' => 307,
  432 => '臺中市大肚區',
  '竹北' => 302,
  '苗栗縣大湖' => 364,
  '基隆市仁愛區' => 200,
  '汐止區' => 221,
  '高雄市茂林區' => 851,
  '苗栗縣苑裡' => 358,
  708 => '臺南市安平區',
  110 => '臺北市信義區',
  '臺中市沙鹿區' => 433,
  333 => '桃園縣龜山',
  '大城' => 527,
  '鹿港' => 505,
  710 => '臺南市永康區',
  '新北市貢寮區' => 228,
  '宜蘭縣南澳' => 272,
  112 => '臺北市北投區',
  545 => '南投縣埔里',
  '北斗' => 521,
  '烏坵' => 896,
  '嘉義縣六腳' => 615,
  '屏東縣車城' => 944,
  646 => '雲林縣古坑',
  '南投縣集集' => 552,
  262 => '宜蘭縣礁溪',
  212 => '連江縣東引',
  '宜蘭縣蘇澳' => 270,
  '三芝區' => 252,
  542 => '南投縣草屯',
  '苗栗縣卓蘭' => 369,
  '長濱' => 962,
  '花蓮縣瑞穗' => 978,
  961 => '臺東縣成功',
  '臺中市潭子區' => 427,
  '彰化' => 500,
  '九如' => 904,
  '屏東縣崁頂' => 924,
  '六腳' => 615,
  227 => '新北市雙溪區',
  977 => '花蓮縣豐濱',
  526 => '彰化縣二林',
  '彰化縣溪湖' => 514,
  806 => '高雄市前鎮區',
  264 => '宜蘭縣員山',
  '新北市土城區' => 236,
  108 => '臺北市萬華區',
  '苗栗縣後龍' => 356,
  604 => '嘉義縣竹崎',
  556 => '南投縣信義',
  717 => '臺南市仁德區',
  '金山區' => 208,
  232 => '新北市坪林區',
  815 => '高雄市大社區',
  207 => '新北市萬里區',
  330 => '桃園縣桃園',
  '臺東縣達仁' => 966,
  '七美' => 883,
  '溪州' => 524,
  360 => '苗栗縣苗栗',
  '基隆市中山區' => 203,
  '元長' => 655,
  '新北市中和區' => 235,
  '臺北市松山區' => 105,
  '橫山' => 312,
  510 => '彰化縣員林',
  '馬公' => 880,
  '牡丹' => 945,
  '高雄市鳥松區' => 833,
  '彰化縣彰化' => 500,
  '苗栗縣公館' => 363,
  353 => '苗栗縣南庄',
  '基隆市信義區' => 201,
  '臺南市龍崎區' => 719,
  '釣魚台列嶼' => 290,
  946 => '屏東縣恆春',
  '溪湖' => 514,
  '金寧' => 892,
  '田中' => 520,
  '林口區' => 244,
  '臺南市西港區' => 723,
  303 => '新竹縣湖口',
  '士林區' => 111,
  '新北市淡水區' => 251,
  614 => '嘉義縣東石',
  '二崙' => 649,
  '茂林區' => 851,
  '板橋區' => 220,
  '大園' => 337,
  '臺中市中區' => 400,
  '高雄市大社區' => 815,
  '苗栗縣三灣' => 352,
  '龍井區' => 434,
  '新竹縣竹北' => 302,
  '新北市泰山區' => 243,
  941 => '屏東縣枋山',
  '新北市板橋區' => 220,
  '屏東縣高樹' => 906,
  '信義區' => undef,
  '連江縣南竿' => 209,
  951 => '臺東縣綠島',
  '新竹縣五峰' => 311,
  '臺南市安南區' => 709,
  '高雄市旗津區' => 805,
  '雲林縣古坑' => 646,
  '新港' => 616,
  '新北市蘆洲區' => 247,
  832 => '高雄市林園區',
  736 => '臺南市柳營區',
  '南竿' => 209,
  828 => '高雄市永安區',
  '彰化縣芳苑' => 528,
  '新興區' => 800,
  '宜蘭縣宜蘭' => 260,
  '中西區' => 700,
  '湖口' => 303,
  520 => '彰化縣田中',
  '雲林縣元長' => 655,
  424 => '臺中市和平區',
  '金峰' => 964,
  '旗津區' => 805,
  '湖內區' => 829,
  '新北市新店區' => 231,
  '四湖' => 654,
  '雲林縣林內' => 643,
  434 => '臺中市龍井區',
  '金門縣烏坵' => 896,
  '三星' => 266,
  '茄萣區' => 852,
  904 => '屏東縣九如',
  '梧棲區' => 435,
  649 => '雲林縣二崙',
  '樹林區' => 238,
  '芬園' => 502,
  '高雄市鹽埕區' => 803,
  648 => '雲林縣西螺',
  '大埔' => 607,
  '卓蘭' => 369,
  '桃園縣八德' => 334,
  '臺中市北區' => 404,
  727 => '臺南市北門區',
  '嘉義縣朴子' => 613,
  '新竹縣北埔' => 314,
  320 => '桃園縣中壢',
  '關西' => 306,
  913 => '屏東縣萬丹',
  811 => '高雄市楠梓區',
  '恆春' => 946,
  '社頭' => 511,
  '彰化縣秀水' => 504,
  '宜蘭縣大同' => 267,
  202 => '基隆市中正區',
  249 => '新北市八里區',
  361 => '苗栗縣造橋',
  '橋頭區' => 825,
  819 => '南海諸島南沙',
  894 => '金門縣烈嶼',
  '甲仙區' => 847,
  '水里' => 553,
  '宜蘭縣釣魚台列嶼' => 290,
  412 => '臺中市大里區',
  954 => '臺東縣卑南',
  '大埤' => 631,
  '義竹' => 624,
  '新埔' => 305,
  307 => '新竹縣芎林',
  801 => '高雄市前金區',
  '臺南市北門區' => 727,
  '林內' => 643,
  971 => '花蓮縣新城',
  830 => '高雄市鳳山區',
  847 => '高雄市甲仙區',
  704 => '臺南市北區',
  326 => '桃園縣楊梅',
  '仁愛' => 546,
  335 => '桃園縣大溪',
  621 => '嘉義縣民雄',
  '萬里區' => 207,
  524 => '彰化縣溪州',
  '銅鑼' => 366,
  552 => '南投縣集集',
  '臺中市太平區' => 411,
  '埔里' => 545,
  '臺南市山上區' => 743,
  890 => '金門縣金沙',
  115 => '臺北市南港區',
  '滿州' => 947,
  '屏東縣屏東' => 900,
  922 => '屏東縣來義',
  '高雄市燕巢區' => 824,
  602 => '嘉義縣番路',
  '七股區' => 724,
  612 => '嘉義縣太保',
  '大安區' => undef,
  '屏東縣三地門' => 901,
  701 => '臺南市東區',
  '三義' => 367,
  '東河' => 959,
  '宜蘭縣壯圍' => 263,
  '花蓮縣卓溪' => 982,
  '臺東縣成功' => 961,
  817 => '南海諸島東沙',
  '將軍區' => 725,
  '通霄' => 357,
  '彰化縣二水' => 530,
  731 => '臺南市後壁區',
  '五結' => 268,
  '苗栗縣通霄' => 357,
  '新北市鶯歌區' => 239,
  '望安' => 882,
  503 => '彰化縣花壇',
  '新竹縣寶山' => 308,
  '北區' => undef,
  '高雄市梓官區' => 826,
  '臺中市后里區' => 421,
  315 => '新竹縣峨眉',
  338 => '桃園縣蘆竹',
  '彰化縣員林' => 510,
  '埔鹽' => 516,
  222 => '新北市深坑區',
  '東山區' => 733,
  '彰化縣埔心' => 513,
  '花蓮縣吉安' => 973,
  '臺南市安定區' => 745,
  305 => '新竹縣新埔',
  '西區' => 403,
  '大武' => 965,
  928 => '屏東縣東港',
  623 => '嘉義縣溪口',
  544 => '南投縣國姓',
  '臺北市信義區' => 110,
  '宜蘭縣頭城' => 261,
  328 => '桃園縣觀音',
  239 => '新北市鶯歌區',
  '虎尾' => 632,
  269 => '宜蘭縣冬山',
  363 => '苗栗縣公館',
  '岡山區' => 820,
  733 => '臺南市東山區',
  630 => '雲林縣斗南',
  '南屯區' => 408,
  '南投縣仁愛' => 546,
  '鳳林' => 975,
  '桃園縣平鎮' => 324,
  '高雄市岡山區' => 820,
  608 => '嘉義縣水上',
  711 => '臺南市歸仁區',
  '臺中市大甲區' => 437,
  '瑪家' => 903,
  892 => '金門縣金寧',
  '中山區' => undef,
  '臺中市神岡區' => 429,
  '線西' => 507,
  546 => '南投縣仁愛',
  '東引' => 212,
  '嘉義縣大林' => 622,
  '臺南市左鎮區' => 713,
  '高雄市田寮區' => 823,
  '員林' => 510,
  522 => '彰化縣田尾',
  '成功' => 961,
  741 => '臺南市善化區',
  '臺南市中西區' => 700,
  '彰化縣竹塘' => 525,
  714 => '臺南市玉井區',
  313 => '新竹縣尖石',
  '臺中市梧棲區' => 435,
  '北埔' => 314,
  '南區' => undef,
  '南投縣草屯' => 542,
  '臺西' => 636,
  '桃源區' => 848,
  '苗栗縣銅鑼' => 366,
  '歸仁區' => 711,
  '桃園縣大溪' => 335,
  '楠梓區' => 811,
  '臺北市中正區' => 100,
  908 => '屏東縣長治',
  823 => '高雄市田寮區',
  903 => '屏東縣瑪家',
  247 => '新北市蘆洲區',
  '新北市瑞芳區' => 224,
  '南投縣嘉義市' => 600,
  '屏東縣里港' => 905,
  '宜蘭縣新竹市' => 300,
  '萬華區' => 108,
  401 => '臺中市東區',
  845 => '高雄市內門區',
  '苑裡' => 358,
  '新北市金山區' => 208,
  '竹田' => 911,
  973 => '花蓮縣吉安',
  '林園區' => 832,
  '桃園縣觀音' => 328,
  '文山區' => 116,
  '屏東縣獅子' => 943,
  '臺南市六甲區' => 734,
  '蘭嶼' => 952,
  '竹山' => 557,
  '宜蘭縣羅東' => 265,
  735 => '臺南市下營區',
  404 => '臺中市北區',
  516 => '彰化縣埔鹽',
  '新北市汐止區' => 221,
  '澎湖縣望安' => 882,
  '宜蘭縣礁溪' => 262,
  974 => '花蓮縣壽豐',
  '臺南市北區' => 704,
  702 => '臺南市南區',
  '屏東縣枋寮' => 940,
  '泰山區' => 243,
  '臺中市新社區' => 426,
  '南投縣鹿谷' => 558,
  '嘉義縣新港' => 616,
  '新北市三芝區' => 252,
  '高雄市旗山區' => 842,
  636 => '雲林縣臺西',
  '東勢區' => 423,
  209 => '連江縣南竿',
  357 => '苗栗縣通霄',
  '海端' => 957,
  '崙背' => 637,
  '石岡區' => 422,
  600 => '南投縣嘉義市',
  '臺北市南港區' => 115,
  336 => '桃園縣復興',
  '臺中市石岡區' => 422,
  '秀林' => 972,
  821 => '高雄市路竹區',
  963 => '臺東縣太麻里',
  '雲林縣斗六' => 640,
  '金湖' => 891,
  '新豐' => 304,
  '泰安' => 365,
  '白沙' => 884,
  '澎湖縣湖西' => 885,
  '花壇' => 503,
  '安樂區' => 204,
  505 => '彰化縣鹿港',
  '石門區' => 253,
  '柳營區' => 736,
  958 => '臺東縣池上',
  '屏東縣東港' => 928,
  '高雄市鳳山區' => 830,
  '田寮區' => 823,
  290 => '宜蘭縣釣魚台列嶼',
  625 => '嘉義縣布袋',
  '臺南市官田區' => 720,
  210 => '連江縣北竿',
  712 => '臺南市新化區',
  '三重區' => 241,
  406 => '臺中市北屯區',
  '南投縣魚池' => 555,
  '五股區' => 248,
  530 => '彰化縣二水',
  '金門縣金寧' => 892,
  '安南區' => 709,
  365 => '苗栗縣泰安',
  '新竹縣橫山' => 312,
  '楊梅' => 326,
  654 => '雲林縣四湖',
  '峨眉' => 315,
  252 => '新北市三芝區',
  883 => '澎湖縣七美',
  '彰化縣永靖' => 512,
  655 => '雲林縣元長',
  '鹽水區' => 737,
  421 => '臺中市后里區',
  '雲林縣斗南' => 630,
  '新店區' => 231,
  '獅潭' => 354,
  '和美' => 508,
  '彰化縣田中' => 520,
  '澎湖縣馬公' => 880,
  '北門區' => 727,
  105 => '臺北市松山區',
  925 => '屏東縣新埤',
  840 => '高雄市大樹區',
  '大湖' => 364,
  '臺南市善化區' => 741,
  111 => '臺北市士林區',
  '臺南市東區' => 701,
  '桃園縣新屋' => 327,
  408 => '臺中市南屯區',
  '雲林縣二崙' => 649,
  '烏來區' => 233,
  945 => '屏東縣牡丹',
  242 => '新北市新莊區',
  '金沙' => 890,
  206 => '基隆市七堵區',
  '花蓮縣壽豐' => 974,
  959 => '臺東縣東河',
  358 => '苗栗縣苑裡',
  '雲林縣口湖' => 653,
  '臺北市萬華區' => 108,
  737 => '臺南市鹽水區',
  '彰化縣田尾' => 522,
  '高雄市新興區' => 800,
  805 => '高雄市旗津區',
  437 => '臺中市大甲區',
  220 => '新北市板橋區',
  '中寮' => 541,
  846 => '高雄市杉林區',
  '大社區' => 815,
  '光復' => 976,
  '水上' => 608,
  '後龍' => 356,
  '南投' => 540,
  '南投縣竹山' => 557,
  '臺南市永康區' => 710,
  '高雄市大樹區' => 840,
  '新北市三峽區' => 237,
  226 => '新北市平溪區',
  '六龜區' => 844,
  527 => '彰化縣大城',
  '高雄市大寮區' => 831,
  833 => '高雄市鳥松區',
  '彰化縣二林' => 526,
  '卓溪' => 982,
  '臺南市歸仁區' => 711,
  '龍潭' => 325,
  926 => '屏東縣南州',
  '新竹縣新埔' => 305,
  '南投縣水里' => 553,
  '頭份' => 351,
  '嘉義市' => 600,
  '臺南市東山區' => 733,
  '臺南市南區' => 702,
  715 => '臺南市楠西區',
  364 => '苗栗縣大湖',
  '嘉義縣竹崎' => 604,
  653 => '雲林縣口湖',
  950 => '臺東縣臺東',
  '嘉義縣番路' => 602,
  '高雄市楠梓區' => 811,
  '臺南市後壁區' => 731,
  '新北市永和區' => 234,
  509 => '彰化縣伸港',
  '新北市坪林區' => 232,
  976 => '花蓮縣光復',
  367 => '苗栗縣三義',
  716 => '臺南市南化區',
  940 => '屏東縣枋寮',
  '彰化縣北斗' => 521,
  '深坑區' => 222,
  '臺南市七股區' => 724,
  743 => '臺南市山上區',
  523 => '彰化縣埤頭',
  720 => '臺南市官田區',
  324 => '桃園縣平鎮',
  723 => '臺南市西港區',
  244 => '新北市林口區',
  637 => '雲林縣崙背',
  351 => '苗栗縣頭份',
  '口湖' => 653,
  '大同' => 267,
  '南投縣名間' => 551,
  '基隆市中正區' => 202,
  '雲林縣北港' => 651,
  901 => '屏東縣三地門',
  '基隆市安樂區' => 204,
  '新竹市' => 300,
  '高雄市小港區' => 812,
  '前鎮區' => 806,
  '雲林縣水林' => 652,
  541 => '南投縣中寮',
  '楠西區' => 715,
  881 => '澎湖縣西嶼',
  '內湖區' => 114,
  312 => '新竹縣橫山',
  507 => '彰化縣線西',
  '萬巒' => 923,
  237 => '新北市三峽區',
  '臺南市麻豆區' => 721,
  '三地門' => 901,
  '萬丹' => 913,
  '高雄市內門區' => 845,
  '左營區' => 813,
  100 => '臺北市中正區',
  965 => '臺東縣大武',
  300 => '宜蘭縣新竹市',
  824 => '高雄市燕巢區',
  '新竹縣竹東' => 310,
  800 => '高雄市新興區',
  '太保' => 612,
  '南投縣埔里' => 545,
  '屏東縣新園' => 932,
  '新北市八里區' => 249,
  '大肚區' => 432,
  '屏東縣萬巒' => 923,
  '觀音' => 328,
  '苗栗縣泰安' => 365,
  '臺中市北屯區' => 406,
  605 => '嘉義縣阿里山',
  '臺南市白河區' => 732,
  '大林' => 622,
  '名間' => 551,
  607 => '嘉義縣大埔',
  '琉球' => 929,
  '綠島' => 951,
  813 => '高雄市左營區',
  '暖暖區' => 205,
  851 => '高雄市茂林區',
  '車城' => 944,
  235 => '新北市中和區',
  436 => '臺中市清水區',
  '屏東縣長治' => 908,
  '金城' => 893,
  '壯圍' => 263,
  '大溪' => 335,
  '西湖' => 368,
  '豐原區' => 420,
  '新社區' => 426,
  265 => '宜蘭縣羅東',
  '豐濱' => 977,
  '中壢' => 320,
  '貢寮區' => 228,
  '基隆市暖暖區' => 205,
  '內門區' => 845,
  '嘉義縣水上' => 608,
  '臺北市文山區' => 116,
  '嘉義縣鹿草' => 611,
  '南化區' => 716,
  200 => '基隆市仁愛區',
  '高雄市前鎮區' => 806,
  '桃園縣中壢' => 320,
  '永安區' => 828,
  '高雄市桃源區' => 848,
  525 => '彰化縣竹塘',
  '臺東縣金峰' => 964,
  '瑞芳區' => 224,
  '桃園縣楊梅' => 326,
  '臺中市大里區' => 412,
  '復興' => 336,
  '彰化縣伸港' => 509,
  557 => '南投縣竹山',
  '麻豆區' => 721,
  718 => '臺南市關廟區',
  '臺南市柳營區' => 736,
  '臺南市仁德區' => 717,
  983 => '花蓮縣富里',
  '新北市三重區' => 241,
  803 => '高雄市鹽埕區',
  '新北市林口區' => 244,
  '高雄市杉林區' => 846,
  '埤頭' => 523,
  '五峰' => 311,
  '下營區' => 735,
  '伸港' => 509,
  911 => '屏東縣竹田',
  427 => '臺中市潭子區',
  '臺南市佳里區' => 722,
  '新竹縣峨眉' => 315,
  '苗栗縣獅潭' => 354,
  907 => '屏東縣鹽埔',
  '南沙' => 819,
  '莒光' => 211,
  '麟洛' => 909,
  '屏東縣佳冬' => 931,
  '中和區' => 235,
  '官田區' => 720,
  '新屋' => 327,
  '宜蘭縣員山' => 264,
  '新竹縣尖石' => 313,
  224 => '新北市瑞芳區',
  '芳苑' => 528,
  '大內區' => 742,
  '屏東縣來義' => 922,
  352 => '苗栗縣三灣',
  '左鎮區' => 713,
  '來義' => 922,
  '新竹縣新豐' => 304,
  251 => '新北市淡水區',
  426 => '臺中市新社區',
  '臺南市楠西區' => 715,
  369 => '苗栗縣卓蘭',
  '彰化縣鹿港' => 505,
  '臺中市大安區' => 439,
  651 => '雲林縣北港',
  '嘉義縣太保' => 612,
  956 => '臺東縣關山',
  '彰化縣大城' => 527,
  947 => '屏東縣滿州',
  '高雄市那瑪夏區' => 849,
  '澎湖縣白沙' => 884,
  920 => '屏東縣潮州',
  428 => '臺中市大雅區',
  '嘉義縣東石' => 614,
  '東沙' => 817,
  '永靖' => 512,
  '臺東' => 950,
  '吉安' => 973,
  '臺南市安平區' => 708,
  500 => '彰化縣彰化',
  '臺北市大同區' => 103,
  '桃園縣桃園' => 330,
  '南港區' => 115,
  '南庄' => 353,
  414 => '臺中市烏日區',
  843 => '高雄市美濃區',
  '臺東縣卑南' => 954,
  '鹿草' => 611,
  263 => '宜蘭縣壯圍',
  632 => '雲林縣虎尾',
  '褒忠' => 634,
  '高雄市甲仙區' => 847,
  514 => '彰化縣溪湖',
  725 => '臺南市將軍區',
  931 => '屏東縣佳冬',
  513 => '彰化縣埔心',
  615 => '嘉義縣六腳',
  '新北市烏來區' => 233,
  '南投縣中寮' => 541,
  '潭子區' => 427,
  '花蓮縣玉里' => 981,
  '二水' => 530,
  '彰化縣溪州' => 524,
  635 => '雲林縣東勢',
  '雲林縣臺西' => 636,
  979 => '花蓮縣萬榮',
  '冬山' => 269,
  '阿蓮區' => 822,
  '宜蘭縣五結' => 268,
  709 => '臺南市安南區',
  '鳳山區' => 830,
  310 => '新竹縣竹東',
  '臺東縣大武' => 965,
  '玉井區' => 714,
  '屏東縣牡丹' => 945,
  '屏東縣麟洛' => 909,
  '新北市平溪區' => 226,
  '安平區' => 708,
  '鶯歌區' => 239,
  '臺東縣太麻里' => 963,
  '二林' => 526,
  '鹽埔' => 907,
  '臺中市西屯區' => 407,
  '烏日區' => 414,
  '花蓮縣豐濱' => 977,
  '竹塘' => 525,
  267 => '宜蘭縣大同',
  '東石' => 614,
  826 => '高雄市梓官區',
  354 => '苗栗縣獅潭',
  '六甲區' => 734,
  '新埤' => 925,
  '霧峰區' => 413,
  '鹽埕區' => 803,
  '臺中市和平區' => 424,
  745 => '臺南市安定區',
  '金門縣金沙' => 890,
  '臺中市龍井區' => 434,
  603 => '嘉義縣梅山',
  891 => '金門縣金湖',
  '苗栗縣頭份' => 351,
  '花蓮縣富里' => 983,
  730 => '臺南市新營區',
  921 => '屏東縣泰武',
  233 => '新北市烏來區',
  '公館' => 363,
  '臺北市內湖區' => 114,
  '連江縣莒光' => 211,
  368 => '苗栗縣西湖',
  '三灣' => 352,
  '仁愛區' => 200,
  '臺中市大雅區' => 428,
  '永康區' => 710,
  '松山區' => 105,
  '壽豐' => 974,
  611 => '嘉義縣鹿草',
  849 => '高雄市那瑪夏區',
  '新北市五股區' => 248,
  208 => '新北市金山區',
  511 => '彰化縣社頭',
  842 => '高雄市旗山區',
  '臺中市南區' => 402,
  '竹東' => 310,
  '高雄市橋頭區' => 825,
  '高雄市三民區' => 807,
  '花蓮縣萬榮' => 979,
  '田尾' => 522,
  '高雄市阿蓮區' => 822,
  '南海諸島南沙' => 819,
  114 => '臺北市內湖區',
  '鹿野' => 955,
  429 => '臺中市神岡區',
  '員山' => 264,
  '高雄市前金區' => 801,
  '桃園縣蘆竹' => 338,
  '高雄市永安區' => 828,
  '屏東縣瑪家' => 903,
  '魚池' => 555,
  '雲林縣東勢' => 635,
  '大樹區' => 840,
  '新北市雙溪區' => 227,
  241 => '新北市三重區',
  812 => '高雄市小港區',
  '屏東縣春日' => 942,
  327 => '桃園縣新屋',
  '臺中市東區' => 401,
  '雲林縣土庫' => 633,
  944 => '屏東縣車城',
  732 => '臺南市白河區',
  '臺南市新營區' => 730,
  '臺東縣關山' => 956,
  957 => '臺東縣海端',
  '燕巢區' => 824,
  622 => '嘉義縣大林',
  970 => '花蓮縣花蓮',
  '彰化縣福興' => 506,
  '佳冬' => 931,
  '竹崎' => 604,
  '仁武區' => 814,
  '斗六' => 640,
  104 => '臺北市中山區',
  '北竿' => 210,
  502 => '彰化縣芬園',
  314 => '新竹縣北埔',
  '八德' => 334,
  '梓官區' => 826,
  '屏東縣內埔' => 912,
  '花蓮縣新城' => 971,
  553 => '南投縣水里',
  '嘉義縣義竹' => 624,
  555 => '南投縣魚池',
  '高雄市茄萣區' => 852,
  '臺東縣海端' => 957,
  '臺中市豐原區' => 420,
  270 => '宜蘭縣蘇澳',
  831 => '高雄市大寮區',
  '高雄市林園區' => 832,
  521 => '彰化縣北斗',
  433 => '臺中市沙鹿區',
  880 => '澎湖縣馬公',
  647 => '雲林縣莿桐',
  '鹿谷' => 558,
  '臺南市南化區' => 716,
  334 => '桃園縣八德',
  '埔心' => 513,
  '小港區' => 812,
  '連江縣北竿' => 210,
  103 => '臺北市大同區',
  '土庫' => 633,
  '芎林' => 307,
  201 => '基隆市信義區',
  '雲林縣褒忠' => 634,
  '鳥松區' => 833,
  423 => '臺中市東勢區',
  719 => '臺南市龍崎區',
  '雲林縣四湖' => 654,
  902 => '屏東縣霧臺',
  827 => '高雄市彌陀區',
  266 => '宜蘭縣三星',
  820 => '高雄市岡山區',
  '彰化縣大村' => 515,
  422 => '臺中市石岡區',
  '布袋' => 625,
  '高雄市彌陀區' => 827,
  905 => '屏東縣里港',
  '頭屋' => 362,
  '嘉義縣中埔' => 606,
  '烈嶼' => 894,
  '竹南' => 350,
  '斗南' => 630,
  302 => '新竹縣竹北',
  '桃園縣復興' => 336,
  '雲林縣麥寮' => 638,
  '南海諸島東沙' => 817,
  '崁頂' => 924,
  '臺南市新化區' => 712,
  '花蓮縣秀林' => 972,
  '白河區' => 732,
  '沙鹿區' => 433,
  '臺中市東勢區' => 423,
  402 => '臺中市南區',
  893 => '金門縣金城',
  '雲林縣西螺' => 648,
  616 => '嘉義縣新港',
  640 => '雲林縣斗六',
  '金門縣烈嶼' => 894,
  804 => '高雄市鼓山區',
  896 => '金門縣烏坵',
  '屏東縣琉球' => 929,
  '關廟區' => 718,
  '臺東縣蘭嶼' => 952,
  962 => '臺東縣長濱',
  '新竹縣關西' => 306,
  '澎湖縣西嶼' => 881,
  '秀水' => 504,
  '臺南市新市區' => 744,
  '澎湖縣七美' => 883,
  '雲林縣莿桐' => 647,
  '林邊' => 927,
  '北港' => 651,
  '仁德區' => 717,
  '安定區' => 745,
  '太麻里' => 963,
  439 => '臺中市大安區',
  362 => '苗栗縣頭屋',
  411 => '臺中市太平區',
  '霧臺' => 902,
  '屏東縣萬丹' => 913,
  '淡水區' => 251,
  '和平區' => 424,
  942 => '屏東縣春日',
  955 => '臺東縣鹿野',
  '里港' => 905,
  407 => '臺中市西屯區',
  '神岡區' => 429,
  '石碇區' => 223,
  912 => '屏東縣內埔',
  '彰化縣芬園' => 502,
  '臺東縣延平' => 953,
  '臺東縣長濱' => 962,
  '清水區' => 436,
  923 => '屏東縣萬巒',
  '大雅區' => 428,
  822 => '高雄市阿蓮區',
  231 => '新北市新店區',
  243 => '新北市泰山區',
  551 => '南投縣名間',
  '美濃區' => 843,
  '彰化縣和美' => 508,
  504 => '彰化縣秀水',
  981 => '花蓮縣玉里',
  '新北市新莊區' => 242,
  '集集' => 552,
  '雲林縣虎尾' => 632,
  '南州' => 926,
  726 => '臺南市學甲區',
  '花蓮縣花蓮' => 970,
  '桃園縣大園' => 337,
  '卑南' => 954,
  106 => '臺北市大安區',
  964 => '臺東縣金峰',
  203 => '基隆市中山區',
  261 => '宜蘭縣頭城',
  '雲林縣大埤' => 631,
  '雲林縣崙背' => 637,
  '后里區' => 421,
  '苗栗縣頭屋' => 362,
  '高雄市六龜區' => 844,
  624 => '嘉義縣義竹',
  '龍崎區' => 719,
  '延平' => 953,
  '平鎮' => 324,
  '玉里' => 981,
  '嘉義縣大埔' => 607,
  204 => '基隆市安樂區',
  '金門縣金湖' => 891,
  '枋山' => 941,
  435 => '臺中市梧棲區',
  '花蓮' => 970,
  814 => '高雄市仁武區',
  '高雄市苓雅區' => 802,
  '苗栗縣造橋' => 361,
  924 => '屏東縣崁頂',
  885 => '澎湖縣湖西',
  884 => '澎湖縣白沙',
  228 => '新北市貢寮區',
  '高樹' => 906,
  268 => '宜蘭縣五結',
  829 => '高雄市湖內區',
  '福興' => 506,
  223 => '新北市石碇區',
  613 => '嘉義縣朴子',
  721 => '臺南市麻豆區',
  '宜蘭' => 260,
  420 => '臺中市豐原區',
  '蘇澳' => 270,
  '礁溪' => 262,
  952 => '臺東縣蘭嶼',
  238 => '新北市樹林區',
  '花蓮縣光復' => 976,
  253 => '新北市石門區',
  '南投縣信義' => 556,
  '獅子' => 943,
  '臺北市大安區' => 106,
  852 => '高雄市茄萣區',
  643 => '雲林縣林內',
  '瑞穗' => 978,
  '湖西' => 885,
  '屏東縣林邊' => 927,
  '臺東縣東河' => 959,
  '寶山' => 308,
  '長治' => 908,
  713 => '臺南市左鎮區',
  '高雄市仁武區' => 814,
  '枋寮' => 940,
  '臺東縣綠島' => 951,
  '嘉義縣布袋' => 625,
  '新北市深坑區' => 222,
  '尖石' => 313,
  '臺北市北投區' => 112,
  929 => '屏東縣琉球',
  '新園' => 932,
  '新市區' => 744,
  '高雄市左營區' => 813,
  '新竹縣湖口' => 303,
  304 => '新竹縣新豐',
  '北屯區' => 406,
  631 => '雲林縣大埤',
  '西港區' => 723,
  '麥寮' => 638,
  '草屯' => 542,
  '屏東縣九如' => 904,
  '屏東' => 900,
  '旗山區' => 842,
  306 => '新竹縣關西',
  '臺中市清水區' => 436,
  '新北市石碇區' => 223,
  '雙溪區' => 227,
  '屏東縣鹽埔' => 907,
  '路竹區' => 821,
  909 => '屏東縣麟洛',
  403 => '臺中市西區',
  '屏東縣霧臺' => 902,
  848 => '高雄市桃源區',
  '東勢' => 635,
  '臺中市西區' => 403,
  '國姓' => 544,
  '彌陀區' => 827,
  '苗栗縣竹南' => 350,
  '臺南市學甲區' => 726,
  '三民區' => 807,
  '八里區' => 249,
  900 => '屏東縣屏東',
  '中正區' => undef,
  '嘉義縣梅山' => 603,
  972 => '花蓮縣秀林',
  '大同區' => 103,
  '彰化縣社頭' => 511,
  248 => '新北市五股區',
  '宜蘭縣三星' => 266,
  356 => '苗栗縣後龍',
  802 => '高雄市苓雅區',
  '臺中市南屯區' => 408,
  '臺南市關廟區' => 718,
  528 => '彰化縣芳苑',
  '中埔' => 606,
  807 => '高雄市三民區'
};

sub new {
    return bless {}, $_[0];
}

sub get {
    return $ZIPCODE->{$_[1]};
}

1;

__END__

=encoding utf8

=head1 NAME

Data::Zipcode::TW

=head1 VERSION

version 1.0

=head1 SYNOPSIS

    ### Lookup
    say Data::Zipcode::TW->get('瑞穗'); #=> 978

    ### Reverse Lookup
    say Data::Zipcode::TW->get('404');  #=> 臺中市北區

    ### Ambiguous
    say Data::Zipcode::TW->get('中區'); #=> undef

    ### Disambiguated
    say Data::Zipcode::TW->get('臺中市北區'); #=> 404
    say Data::Zipcode::TW->get('臺南市北區'); #=> 704

    ### Not Found
    say Data::Zipcode::TW->get('這裡那裡');   #=> undef

    ### As Object Method
    my $zipcode = Data::Zipcode::TW->new;
    say $zipcode->get('新竹縣竹北');      #=> 302

=head1 DESCRIPTION

This module provides Taiwan Zipcode data. The original zip data can be downloaded
from L<http://www.post.gov.tw/post/internet/down/index.html?ID=190108>.

=head1 METHODS

=head2 get( $query )

This method takes a simple, non-ref scalar (String or Number) and returns a simple
scalar (String or Number) or undef.

This method is the only method for the class and the instances of the class, it
requires one argument, that is the query string.

When the query string is a number, it performs a reverse lookup (a
zipcode-to-area-name lookup), and returns a long area name. Otherwise it is
assumed the string means a long or short area name (described below), and returns
the zipcode number.

Please reference this official table provided by the postal office website at
 L<http://download.post.gov.tw/post/download/臺灣地區郵遞區號前3碼一覽表_9912.txt>
for the area names that can be used in the query string.

Short names are those with zipcode numbers in the table, they are usually town
names. Long names are short names prepened by their city/county names. For
example, C<大安區> or C<瑞穗> are short names, while C<臺北市大安區> is a long
name. Notice there are two C<大安區> in the table, one with zipcode 106, the
other with zipcode 439.

Spaces must be removed in the query, you cannot use C<"竹 北"> to perform the
query, use C<"竹北"> instead.

If the given C<$query> is an ambiguours short name, C<undef> is returned.

=head1 AUTHOR

Kang-min Liu C<< <gugod@gugod.org> >>

=head1 COPYRIGHT

Copyright (c) 2011 Kang-min Liu C<< <gugod@gugod.org> >>.

=head1 LICENCE

The MIT License

=head1 PROJECT DEVELOPMENT

See L<https://github.com/gugod/Data-Zipcode-TW>

=head1 DISCLAIMER OF WARRANTY

BECAUSE THIS SOFTWARE IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE LAW. EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOFTWARE IS WITH
YOU. SHOULD THE SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL
NECESSARY SERVICING, REPAIR, OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE SOFTWARE AS PERMITTED BY THE ABOVE LICENCE, BE
LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE
THE SOFTWARE (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING
RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
FAILURE OF THE SOFTWARE TO OPERATE WITH ANY OTHER SOFTWARE), EVEN IF
SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGES.

=cut
