package XML::Grammar::Screenplay::FromProto;
$XML::Grammar::Screenplay::FromProto::VERSION = '0.22.6';
use strict;
use warnings;
use autodie;

use Carp ();

use MooX 'late';

extends("XML::Grammar::FictionBase::TagsTree2XML");

my $screenplay_ns =
q{http://web-cpan.berlios.de/modules/XML-Grammar-Screenplay/screenplay-xml-0.2/};


sub _write_Element_Text
{
    my ( $self, $elem ) = @_;

    if ( $elem->_short_isa("Saying") )
    {
        $self->_output_tag_with_childs(
            {
                start => [ "saying", 'character' => $elem->character() ],
                elem  => $elem,
            },
        );

        return;
    }
    elsif ( $elem->_short_isa("Description") )
    {
        $self->_output_tag_with_childs(
            {
                start => ["description"],
                elem  => $elem,
            },
        );

        return;
    }
    else
    {
        $self->_write_elem_childs($elem);

        return;
    }
}

sub _paragraph_tag
{
    return "para";
}

sub _handle_elem_of_name_code_blk
{
    my ( $self, $elem ) = @_;

    my $good = sub {
        my ( $k, $v ) = @_;
        my $input_v = $elem->lookup_attr($k);
        die
"wrong value \"$input_v\" for attribute \"$k\" for element \"<code_blk>\""
            if ( ref($v) eq "" ? ( $v ne $input_v ) : ( $input_v !~ /$v/ ) );
        return ( $k, $input_v );
    };

    my $out_cb = sub {
        $self->_output_tag(
            {
                start => [
                    "code_blk",
                    $good->( "syntax",   "text" ),
                    $good->( "tag_role", qr#\A(?:asciiart|code_block)\z#, ),
                    $good->( "title",    qr#.#ms ),
                    $good->( "alt",      qr#.#ms ),
                ],
                in => sub {
                    my $inner_text = $elem->_first()->_first();

                    foreach my $cnt ( 1 .. 2 )
                    {
                        my $x;

                        eval { $x = $inner_text->_first(); };

                        if ($x)
                        {
                            $inner_text = $x;
                        }

                    }
                    die if ( ref($inner_text) ne "" );
                    $inner_text =~ s/\A(?:\r?\n)*//ms;
                    $inner_text =~ s/(?:^\r?\n)*\z//ms;
                    my @lines = split /^/ms, $inner_text;
                LINES:
                    foreach my $l (@lines)
                    {
                        # next LINES if $l =~ /\A\r?\n/
                        $l =~ s#^\|##ms
                            or die qq#code_blk line did not start with a '|'#;
                    }
                    return $self->_write_elem(
                        {
                            elem => ( join "", @lines )
                        }
                    );
                },
            }
        );
    };
    return (
        ( $self->_writer->within_element( $self->_paragraph_tag ) )
        ? $out_cb->()
        : $self->_output_tag(
            {
                start => [ $self->_paragraph_tag, ],
                in    => $out_cb,
            }
        )
    );

    return;
}

sub _handle_elem_of_name_img
{
    my ( $self, $elem ) = @_;

    my $image_cb = sub {
        return $self->_output_tag_with_childs(
            {
                start => [
                    "image",
                    "url"   => $elem->lookup_attr("src"),
                    "alt"   => $elem->lookup_attr("alt"),
                    "title" => $elem->lookup_attr("title"),
                ],
                elem => $elem,
            }
        );
    };

    return (
        ( $self->_writer->within_element( $self->_paragraph_tag ) )
        ? $image_cb->()
        : $self->_output_tag(
            {
                start => [ "para", ],
                in    => $image_cb,
            }
        )
    );

}

sub _handle_elem_of_name_a
{
    my ( $self, $elem ) = @_;

    $self->_output_tag_with_childs(
        {
            start => [
                "a",
                [ "http://www.w3.org/1999/xlink", "href" ],
                $elem->lookup_attr("href"),
            ],
            elem => $elem,
        }
    );

    return;
}

sub _handle_elem_of_name_section
{
    my ( $self, $elem ) = @_;

    return $self->_handle_elem_of_name_s($elem);
}

sub _bold_tag_name
{
    return "bold";
}

sub _italics_tag_name
{
    return "italics";
}

sub _write_scene_main
{
    my ( $self, $scene ) = @_;

    my $id = $scene->lookup_attr("id");

    if ( !defined($id) )
    {
        Carp::confess("Unspecified id for scene!");
    }

    my $title = $scene->lookup_attr("title");
    my $lang  = $scene->lookup_attr("lang");
    my @t     = ( defined($title) ? ( title => $title ) : () );
    if ( defined($lang) )
    {
        push @t, ( [ $self->_get_xml_xml_ns, 'lang' ] => $lang );
    }

    $self->_output_tag_with_childs(
        {
            'start' => [ "scene", id => $id, @t ],
            elem    => $scene,
        }
    );

    return;
}

sub _get_default_xml_ns
{
    return $screenplay_ns;
}

sub _convert_write_content
{
    my ( $self, $tree ) = @_;

    my $writer = $self->_writer;

    $writer->startTag( [ $screenplay_ns, "document" ] );
    $writer->startTag( [ $screenplay_ns, "head" ] );
    $writer->endTag();
    $writer->startTag( [ $screenplay_ns, "body" ], "id" => "index", );

    $self->_write_scene( { scene => $tree } );

    # Ending the body
    $writer->endTag();

    $writer->endTag();

    return;
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

XML::Grammar::Screenplay::FromProto - module that converts well-formed
text representing a screenplay to an XML format.

=head2 new()

Accepts no arguments so far. May take some time as the grammar is compiled
at that point.

=head2 meta()

Internal - (to settle pod-coverage.).

=head2 $self->convert({ source => { file => $path_to_file } })

Converts the file $path_to_file to XML and returns it.

=head1 VERSION

version 0.22.6

=for :stopwords cpan testmatrix url bugtracker rt cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 SUPPORT

=head2 Websites

The following websites have more information about this module, and may be of help to you. As always,
in addition to those websites please use your favorite search engine to discover more resources.

=over 4

=item *

MetaCPAN

A modern, open-source CPAN search engine, useful to view POD in HTML format.

L<https://metacpan.org/release/XML-Grammar-Fiction>

=item *

RT: CPAN's Bug Tracker

The RT ( Request Tracker ) website is the default bug/issue tracking system for CPAN.

L<https://rt.cpan.org/Public/Dist/Display.html?Name=XML-Grammar-Fiction>

=item *

CPANTS

The CPANTS is a website that analyzes the Kwalitee ( code metrics ) of a distribution.

L<http://cpants.cpanauthors.org/dist/XML-Grammar-Fiction>

=item *

CPAN Testers

The CPAN Testers is a network of smoke testers who run automated tests on uploaded CPAN distributions.

L<http://www.cpantesters.org/distro/X/XML-Grammar-Fiction>

=item *

CPAN Testers Matrix

The CPAN Testers Matrix is a website that provides a visual overview of the test results for a distribution on various Perls/platforms.

L<http://matrix.cpantesters.org/?dist=XML-Grammar-Fiction>

=item *

CPAN Testers Dependencies

The CPAN Testers Dependencies is a website that shows a chart of the test results of all dependencies for a distribution.

L<http://deps.cpantesters.org/?module=XML::Grammar::Fiction>

=back

=head2 Bugs / Feature Requests

Please report any bugs or feature requests by email to C<bug-xml-grammar-fiction at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/Public/Bug/Report.html?Queue=XML-Grammar-Fiction>. You will be automatically notified of any
progress on the request by the system.

=head2 Source Code

The code is open to the world, and available for you to hack on. Please feel free to browse it and play
with it, or whatever. If you want to contribute patches, please send me a diff or prod me to pull
from your repository :)

L<https://github.com/shlomif/fiction-xml>

  git clone git://github.com/shlomif/fiction-xml.git

=head1 AUTHOR

Shlomi Fish <shlomif@cpan.org>

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website
L<https://github.com/shlomif/fiction-xml/issues>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2007 by Shlomi Fish.

This is free software, licensed under:

  The MIT (X11) License

=cut
