;; DFA scheduling description for ST40-300.
;; Copyright (C) 2004-2013 Free Software Foundation, Inc.

;; This file is part of GCC.

;; GCC is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 3, or (at your option)
;; any later version.

;; GCC is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GCC; see the file COPYING3.  If not see
;; <http://www.gnu.org/licenses/>.

;; Load and store instructions save a cycle if they are aligned on a
;; four byte boundary.  Using a function unit for stores encourages
;; gcc to separate load and store instructions by one instruction,
;; which makes it more likely that the linker will be able to word
;; align them when relaxing.

;; The following description models the ST40-300 pipeline using the DFA based
;; scheduler.

;; Two automata are defined to reduce number of states
;; which a single large automaton will have. (Factoring)

(define_automaton "sh4_300_inst_pipeline,sh4_300_fpu_pipe")

;; This unit is basically the decode unit of the processor.
;; Since SH4 is a dual issue machine,it is as if there are two
;; units so that any insn can be processed by either one
;; of the decoding unit.
(define_cpu_unit "sh4_300_pipe_01,sh4_300_pipe_02" "sh4_300_inst_pipeline")

;; The floating point units.
(define_cpu_unit "sh4_300_fpt,sh4_300_fpu,sh4_300_fds" "sh4_300_fpu_pipe")

;; integer multiplier unit
(define_cpu_unit "sh4_300_mul" "sh4_300_inst_pipeline")

;; LS unit
(define_cpu_unit "sh4_300_ls" "sh4_300_inst_pipeline")

;; The address calculator used for branch instructions.
;; This will be reserved after "issue" of branch instructions
;; and this is to make sure that no two branch instructions
;; can be issued in parallel.
(define_cpu_unit "sh4_300_br" "sh4_300_inst_pipeline")

;; ----------------------------------------------------
;; This reservation is to simplify the dual issue description.

(define_reservation  "sh4_300_issue"  "sh4_300_pipe_01|sh4_300_pipe_02")

(define_reservation "all" "sh4_300_pipe_01+sh4_300_pipe_02")

;;(define_insn_reservation "nil" 0 (eq_attr "type" "nil") "nothing")

;; MOV RM,RN / MOV #imm8,RN / STS PR,RN
(define_insn_reservation "sh4_300_mov" 0
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "move,movi8,prget"))
  "sh4_300_issue")

;; Fixed STS from MACL / MACH
(define_insn_reservation "sh4_300_mac_gp" 0
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "mac_gp"))
  "sh4_300_issue+sh4_300_mul")

;; Fixed LDS to MACL / MACH
(define_insn_reservation "sh4_300_gp_mac" 1
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "gp_mac"))
  "sh4_300_issue+sh4_300_mul")

;; Instructions without specific resource requirements with latency 1.
(define_insn_reservation "sh4_300_simple_arith" 1
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "mt_group,arith,dyn_shift,prset"))
  "sh4_300_issue")

;; Load and store instructions have no alignment peculiarities for the ST40-300,
;; but they use the load-store unit, which they share with the fmove type
;; insns (fldi[01]; fmov frn,frm; flds; fsts; fabs; fneg) .
;; Loads have a latency of three.

;; Load Store instructions.
(define_insn_reservation "sh4_300_load" 3
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "load,pcload,load_si,pcload_si,pload"))
  "sh4_300_issue+sh4_300_ls")

(define_insn_reservation "sh4_300_mac_load" 3
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "mem_mac"))
  "sh4_300_issue+sh4_300_ls+sh4_300_mul")

(define_insn_reservation "sh4_300_fload" 4
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "fload,pcfload"))
  "sh4_300_issue+sh4_300_ls+sh4_300_fpt")

;; sh_adjust_cost describes the reduced latency of the feeding insns of a store.
;; The latency of an auto-increment register is 1; the latency of the memory
;; output is not actually considered here anyway.
(define_insn_reservation "sh4_300_store" 1
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "store,pstore"))
  "sh4_300_issue+sh4_300_ls")

(define_insn_reservation "sh4_300_fstore" 1
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "fstore"))
  "sh4_300_issue+sh4_300_ls+sh4_300_fpt")

;; Fixed STS.L from MACL / MACH
(define_insn_reservation "sh4_300_mac_store" 1
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "mac_mem"))
  "sh4_300_issue+sh4_300_mul+sh4_300_ls")

(define_insn_reservation "sh4_300_gp_fpul" 2
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "gp_fpul"))
  "sh4_300_issue+sh4_300_fpt")

(define_insn_reservation "sh4_300_fpul_gp" 1
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "fpul_gp"))
  "sh4_300_issue+sh4_300_fpt")

;; Branch (BF,BF/S,BT,BT/S,BRA)
;; Branch Far (JMP,RTS,BRAF)
;; Group:	BR
;; When displacement is 0 for BF / BT, we have effectively conditional
;; execution of one instruction, without pipeline disruption.
;; Otherwise, the latency depends on prediction success.
;; We can't really do much with the latency, even if we could express it,
;; but the pairing restrictions are useful to take into account.
;; ??? If the branch is likely, and not paired with a preceding insn,
;; or likely and likely not predicted, we might want to fill the delay slot.
;; However, there appears to be no machinery to make the compiler
;; recognize these scenarios.
(define_insn_reservation "sh4_300_branch"  1
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "cbranch,jump,return,jump_ind"))
  "sh4_300_issue+sh4_300_br")

;; RTE
(define_insn_reservation "sh4_300_return_from_exp" 9
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "rte"))
  "sh4_300_pipe_01+sh4_300_pipe_02*9")

;; OCBP, OCBWB
;; Group:	CO
;; Latency: 	1-5
;; Issue Rate: 	1
;; cwb is used for the sequence
;;	ocbwb  @%0
;;	extu.w %0,%2
;;	or     %1,%2
;;	mov.l  %0,@%2
;; This description is likely inexact, but this pattern should not actually
;; appear when compiling for sh4-300; we should use isbi instead.
;; If a -mtune option is added later, we should use the icache array
;; dispatch method instead.
(define_insn_reservation "sh4_300_ocbwb"  3
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "cwb"))
  "all*3")

;; JSR,BSR,BSRF
;; Calls have a mandatory delay slot, which we'd like to fill with an insn
;; that can be paired with the call itself.
;; Scheduling runs before reorg, so we approximate this by saying that we
;; want the call to be paired with a preceding insn.
;; In most cases, the insn that loads the address of the call should have
;; a nonzero latency (mov rn,rm doesn't make sense since we could use rn
;; for the address then).  Thus, a preceding insn that can be paired with
;; a call should be eligible for the delay slot.
;;
;; calls introduce a longisch delay that is likely to flush the pipelines
;; of the caller's instructions.  Ordinary functions tend to end with a
;; load to restore a register (in the delay slot of rts), while sfuncs
;; tend to end with an EX or MT insn.  But that is not actually relevant,
;; since there are no instructions that contend for memory access early.
;; We could, of course, provide exact scheduling information for specific
;; sfuncs, if that should prove useful.
(define_insn_reservation "sh4_300_call" 16
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "call,sfunc"))
  "sh4_300_issue+sh4_300_br,all*15")

;; FMOV.S / FMOV.D
(define_insn_reservation "sh4_300_fmov" 1
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "fmove"))
  "sh4_300_issue+sh4_300_fpt")

;; LDS to FPSCR
(define_insn_reservation "sh4_300_fpscr_load" 8
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "gp_fpscr"))
  "sh4_300_issue+sh4_300_fpu+sh4_300_fpt")

;; LDS.L to FPSCR
(define_insn_reservation "sh4_300_fpscr_load_mem" 8
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type"  "mem_fpscr"))
  "sh4_300_issue+sh4_300_fpu+sh4_300_fpt+sh4_300_ls")


;; Fixed point multiplication (DMULS.L DMULU.L MUL.L MULS.W,MULU.W)
(define_insn_reservation "multi" 2
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "smpy,dmpy"))
  "sh4_300_issue+sh4_300_mul")

;; FPCHG, FRCHG, FSCHG
(define_insn_reservation "fpscr_toggle"  1
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "fpscr_toggle"))
  "sh4_300_issue+sh4_300_fpu+sh4_300_fpt")

;; FCMP/EQ, FCMP/GT
(define_insn_reservation "fp_cmp"  3
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "fp_cmp,dfp_cmp"))
  "sh4_300_issue+sh4_300_fpu")

;; Single precision floating point (FADD,FLOAT,FMAC,FMUL,FSUB,FTRC)
;; Double-precision floating-point (FADD,FCNVDS,FCNVSD,FLOAT,FSUB,FTRC)
(define_insn_reservation "fp_arith"  6
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "fp,ftrc_s,dfp_arith,dfp_conv"))
  "sh4_300_issue+sh4_300_fpu")

;; Single Precision FDIV/SQRT
(define_insn_reservation "fp_div" 19
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "fdiv"))
  "sh4_300_issue+sh4_300_fpu+sh4_300_fds,sh4_300_fds*15")

;; Double-precision floating-point FMUL
(define_insn_reservation "dfp_mul" 9
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "dfp_mul"))
  "sh4_300_issue+sh4_300_fpu,sh4_300_fpu*3")

;; Double precision FDIV/SQRT
(define_insn_reservation "dp_div" 35
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "dfdiv"))
  "sh4_300_issue+sh4_300_fpu+sh4_300_fds,sh4_300_fds*31")

;; ??? We don't really want these for sh4-300.
;; this pattern itself is likely to finish in 3 cycles, but also
;; to disrupt branch prediction for taken branches for the following
;; condbranch.
(define_insn_reservation "sh4_300_arith3" 5
  (and (eq_attr "pipe_model" "sh4_300")
       (eq_attr "type" "arith3"))
  "sh4_300_issue,all*4")

;; arith3b insns without brach redirection make use of the 0-offset 0-latency
;; branch feature, and thus schedule the same no matter if the branch is taken
;; or not.  If the branch is redirected, the taken branch might take longer,
;; but then, we don't have to take the next branch.
;; ??? should we suppress branch redirection for sh4-300 to improve branch
;; target hit rates?
(define_insn_reservation "arith3b" 2
  (and (eq_attr "pipe_model" "sh4")
       (eq_attr "type" "arith3"))
  "issue,all")
