#!/usr/bin/env python

# GMSK modulation and demodulation.  
#
#
# Copyright 2005 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

# See http://noether.uoregon.edu/~jl/gmsk
# See gnuradio-examples/python/gmsk for examples

from gnuradio import gr
from math import pi

class gmsk_mod(gr.hier_block):

	def __init__(self, fg, sps = 8, symbol_rate = 1625000.0 / 6.0,
	   bt = 0.3, p_size = 1024):
		"""
		Hierarchical block for Gaussian Minimum Shift Key (GMSK)
		modulation.

		The input is a byte stream (unsigned char) and the
		output is the complex modulated signal at baseband.

		@param fg: flow graph
		@type fg: flow graph
		@param sps: samples per symbol
		@type sps: integer
		@param symbol_rate: symbols per second
		@type symbol_rate: float
		@param bt: Gaussian filter bandwidth * symbol time
		@type bt: float
		@param p_size: packet size (data to send between syncs)
		@type p_size: integer
		"""

		sample_rate = sps * symbol_rate	# samples per second
		ntaps = 2 * sps			# up to 3 bits in filter at once
		sensitivity = (pi / 2) / sps	# phase change per bit = pi / 2

		# Input is a byte stream.
		self.framer = gr.simple_framer(p_size)
	
		# Turn it into NRZ data.
		self.nrz = gr.bytes_to_syms()

		# Make samples per symbol copies
		#interp_taps = (1,) * sps
		interp_taps = (8,0,0,0,0,0,0,0)
		self.interp = gr.interp_fir_filter_fff(sps, interp_taps)

		# Form Gaussian filter
		gaussian_taps = gr.firdes.gaussian(
		   1.0,		# gain
		   sample_rate,
		   symbol_rate,
		   bt,		# bandwidth * symbol time
		   ntaps	# number of taps
		)
		self.gaussian_filter = gr.fir_filter_fff(1, gaussian_taps)
	
		# FM modulation
		self.fmmod = gr.frequency_modulator_fc(sensitivity)
		
		# Connect
		fg.connect(self.framer, self.nrz, self.interp,
		   self.gaussian_filter, self.fmmod)

		# Initialize base class
		gr.hier_block.__init__(self, fg, self.framer, self.fmmod)


class gmsk_demod(gr.hier_block):
	def __init__(self, fg, sps = 8, symbol_rate = 1625000.0 / 6.0,
	   p_size = 1024):
		"""
		Hierarchical block for Gaussian Minimum Shift Key (GMSK)
		demodulation.

		The input is the complex modulated signal at baseband
		and the output is a stream of bytes.

		@param fg: flow graph
		@type fg: flow graph
		@param sps: samples per symbol
		@type sps: integer
		@param symbol_rate: symbols per second
		@type symbol_rate: float
		@param p_size: packet size
		@type p_size: integer
		"""
		
		# Demodulate FM
		sensitivity = (pi / 2) / sps
		self.fmdemod = gr.quadrature_demod_cf(1.0 / sensitivity)

		# Integrate over a bit length to smooth noise
		#integrate_taps = (1.0 / sps,) * sps
		integrate_taps = (1.0,0,0,0,0,0,0,0,0)
		self.integrate_filter = gr.fir_filter_fff(1, integrate_taps)

		# Bit slice, try and find the center of the bit from the sync
		# field placed by the framer, output p_size bytes at a time.
		self.correlator = gr.simple_correlator(p_size)

		# Connect
		fg.connect(self.fmdemod, self.integrate_filter, self.correlator)

		# Initialize base class
		gr.hier_block.__init__(self, fg, self.fmdemod, self.correlator)

# vim:ts=8
