
/* atari_mac_sdl.c - Main C file for 
 *  Macintosh OS X SDL port of Atari800
 *  Mark Grebe <atarimac@cox.net>
 *  
 *  Based on the SDL port of Atari 800 by:
 *  Jacek Poplawski <jacekp@linux.com.pl>
 *
 * Copyright (C) 2002-2003 Mark Grebe
 * Copyright (C) 2001-2003 Atari800 development team (see DOC/CREDITS)
 *
 * This file is part of the Atari800 emulator project which emulates
 * the Atari 400, 800, 800XL, 130XE, and 5200 8-bit computers.
 *
 * Atari800 is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Atari800 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Atari800; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <SDL.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

// Atari800 includes
#include "config.h"
#include "input.h"
#include "colours.h"
#include "mac_colours.h"
#include "platform.h"
#include "ui.h"
#include "ataripcx.h"
#include "pokeysnd.h"
#include "gtia.h"
#include "antic.h"
#include "diskled.h"
#include "devices.h"
#include "cpu.h"
#include "memory.h"
#include "pia.h"
#include "sndsave.h"
#include "statesav.h"
#include "log.h"
#include "cartridge.h"
#include "sio.h"
#include "mac_rdevice.h"

/* Local variables that control the display and sound modes.  They need to be visable externally
   so the preferences and menu manager Objective-C files may access them. */
int sound_enabled = TRUE;
int useBuiltinPalette = TRUE;
int adjustPalette = FALSE;
int paletteBlack = 0;
int paletteWhite = 0xf0;
int paletteIntensity = 80;
int paletteColorShift = 40;
static int SDL_ATARI_BPP = 0;   // 0 - autodetect
int FULLSCREEN;
int DOUBLESIZE;
int scaleFactor = 3;
int lockFullscreenSize = 1;
int SHOWFPS = 0;
int GRAB_MOUSE = 0;
int WIDTH_MODE;  /* Width mode, and the constants that define it */
#define SHORT_WIDTH_MODE 0
#define DEFAULT_WIDTH_MODE 1
#define FULL_WIDTH_MODE 2

/* Local variables that control the speed of emulation */
int speed_limit = 1;
int pauseEmulator = 0;

/* Semaphore flags used for signalling between the Objective-C menu managers and the emulator. */
int requestDoubleSizeChange = 0;
int requestWidthModeChange = 0;
int requestFullScreenChange = 0;
int requestGrabMouse = 0;
int requestScreenshot = 0;
int requestInterlacedScreenshot = 0;
int requestColdReset = 0;
int requestWarmReset = 0;
int requestSaveState = 0;
int requestLoadState = 0;
int requestLimitChange = 0;
int requestSoundEnabledChange = 0;
int requestSoundStereoChange = 0;
int requestSoundRecordingChange = 0;
int requestFpsChange = 0;
int requestPrefsChange = 0;
int requestPauseEmulator = 0;
int requestMonitor = 0;
int requestQuit = 0;
int requestCaptionChange = 0;

/* Filenames for saving and loading the state of the emulator */
char saveFilename[FILENAME_MAX];
char loadFilename[FILENAME_MAX];

/* Filename for loading the color palette */
char paletteFilename[FILENAME_MAX];

/* Externs for misc. emulator functions */
extern int stereo_enabled;
extern char sio_filename[8][FILENAME_MAX];
extern int refresh_rate;
extern double deltatime;
double real_deltatime;
extern int alt_function;
extern UBYTE break_cim;

/* Externs from preferences_c.c which indicate if user has changed certain preferences */
extern int displaySizeChanged;
extern int showfpsChanged;
extern int artifChanged;
extern int paletteChanged;
extern int machineTypeChanged;
extern int patchFlagsChanged;
extern int keyboardJoystickChanged;
extern int hardDiskChanged;
extern int hifiSoundChanged;
extern int fileToLoad;
extern int cart_type;

/* Routines used to notify Menu Manager objects of changes in configuration casued
   by key presses or Fullscreen user interface */
extern void SDLMainLoadStartupFile(void);
extern void UpdateMediaManagerInfo(void);
extern void MediaManagerRunDiskManagement(void);
extern void MediaManagerInsertCartridge(void);
extern void MediaManagerRemoveCartridge(void);
extern void MediaManagerLoadExe(void);
extern void MediaManagerInsertDisk(int diskNum);
extern void MediaManagerRemoveDisk(int diskNum);
extern void SetDisplayManagerDoubleSize(int doubleSize);
extern void SetDisplayManagerWidthMode(int widthMode);
extern void SetDisplayManagerFps(int fpsOn);
extern void SetDisplayManagerGrabMouse(int mouseOn);
extern void SetControlManagerLimit(int limit);
extern int ControlManagerFatalError(void);
extern void ControlManagerSaveState(void); 
extern void ControlManagerLoadState(void);
extern void ControlManagerPauseEmulator(void);
extern void ControlManagerHideApp(void);
extern void ControlManagerAboutApp(void);
extern void ControlManagerMiniturize(void);
extern void ControlManagerShowHelp(void);
extern int ControlManagerMonitorRun(void);
extern void SetSoundManagerEnable(int soundEnabled);
extern void SetSoundManagerStereo(int soundStereo);
extern void SetSoundManagerRecording(int soundRecording);
void SoundManagerStereoDo(void);
void SoundManagerRecordingDo(void);
extern int RtConfigLoad(char *rtconfig_filename);
extern void CalculatePrefsChanged();
extern void InitializeUI(void);
extern void ClearScreen(UBYTE * screen);
extern void CenterPrint(UBYTE * screen, int fg, int bg, char *string, int y);
extern void RunPreferences(void);


/* Local Routines */
void SDL_Sound_Start(void); 
void PauseAudio(int pause);
void CreateWindowCaption(void);

// joystick emulation - Future enhancement will allow user to set these in 
//   Preferences.

int SDL_TRIG_0 = SDLK_LCTRL;
int SDL_TRIG_0_B = SDLK_KP0;
int SDL_JOY_0_LEFT = SDLK_KP4;
int SDL_JOY_0_RIGHT = SDLK_KP6;
int SDL_JOY_0_DOWN = SDLK_KP2;
int SDL_JOY_0_UP = SDLK_KP8;
int SDL_JOY_0_LEFTUP = SDLK_KP7;
int SDL_JOY_0_RIGHTUP = SDLK_KP9;
int SDL_JOY_0_LEFTDOWN = SDLK_KP1;
int SDL_JOY_0_RIGHTDOWN = SDLK_KP3;

int SDL_TRIG_1 = SDLK_TAB;
int SDL_TRIG_1_B = SDLK_LSHIFT;
int SDL_JOY_1_LEFT = SDLK_a;
int SDL_JOY_1_RIGHT = SDLK_d;
int SDL_JOY_1_DOWN = SDLK_x;
int SDL_JOY_1_UP = SDLK_w;
int SDL_JOY_1_LEFTUP = SDLK_q;
int SDL_JOY_1_RIGHTUP = SDLK_e;
int SDL_JOY_1_LEFTDOWN = SDLK_z;
int SDL_JOY_1_RIGHTDOWN = SDLK_c;

int SDL_IsJoyKey[SDLK_LAST];  // Array to determine if keypress is for joystick.

// real joysticks - Pointers to SDL structures
SDL_Joystick *joystick0 = NULL;
SDL_Joystick *joystick1 = NULL;
SDL_Joystick *joystick2 = NULL;
SDL_Joystick *joystick3 = NULL;
int joystick0_nbuttons, joystick1_nbuttons;
int joystick2_nbuttons, joystick3_nbuttons;
int joystick0_nsticks, joystick0_nhats;
int joystick1_nsticks, joystick1_nhats;
int joystick2_nsticks, joystick2_nhats;
int joystick3_nsticks, joystick3_nhats;

#define minjoy 10000            // real joystick tolerancy
#define NUM_JOYSTICKS    4
int JOYSTICK_MODE[NUM_JOYSTICKS];
int JOYSTICK_AUTOFIRE[NUM_JOYSTICKS];
#define NO_JOYSTICK      0
#define KEYPAD_JOYSTICK  1
#define USERDEF_JOYSTICK 2
#define SDL0_JOYSTICK    3
#define SDL1_JOYSTICK    4
#define SDL2_JOYSTICK    5
#define SDL3_JOYSTICK    6
#define MOUSE_EMULATION  7

#define JOY_TYPE_STICK   0
#define JOY_TYPE_HAT     1
#define JOY_TYPE_BUTTONS 2
int joystick0TypePref = JOY_TYPE_HAT;
int joystick1TypePref = JOY_TYPE_HAT;
int joystick2TypePref = JOY_TYPE_HAT;
int joystick3TypePref = JOY_TYPE_HAT;
int joystickType0, joystickType1;
int joystickType2, joystickType3;
int joystick0Num, joystick1Num;
int joystick2Num, joystick3Num;
int paddlesXAxisOnly = 0;

#define AKEY_BUTTON1  		0x100
#define AKEY_BUTTON2  		0x200
#define AKEY_BUTTON_START  	0x300
#define AKEY_BUTTON_SELECT 	0x400
#define AKEY_BUTTON_OPTION 	0x500
#define AKEY_BUTTON_RESET	0x600
#define AKEY_JOYSTICK_UP	0x700
#define AKEY_JOYSTICK_DOWN	0x701
#define AKEY_JOYSTICK_LEFT	0x702
#define AKEY_JOYSTICK_RIGHT	0x703
#define MAX_JOYSTICK_BUTTONS 24
int joystickButtonKey[4][MAX_JOYSTICK_BUTTONS];
int joystick5200ButtonKey[4][MAX_JOYSTICK_BUTTONS];

/* Mouse and joystick related stuff from Input.c */
int key_code = AKEY_NONE;
int key_break = 0;
int key_shift = 0;
int key_consol = CONSOL_NONE;
int pad_key_shift = 0;
int pad_key_control = 0;
int pad_key_consol = CONSOL_NONE;

int joy_autofire[4] = {AUTOFIRE_OFF, AUTOFIRE_OFF, AUTOFIRE_OFF, AUTOFIRE_OFF};

int joy_block_opposite_directions = 1;

int joy_5200_min = 6;
int joy_5200_center = 114;
int joy_5200_max = 220;

int mouse_mode = MOUSE_OFF;
int mouse_port = 0;
int mouse_delta_x = 0;
int mouse_delta_y = 0;
int mouse_buttons = 0;
int mouse_speed = 3;
int mouse_pot_min = 1;
int mouse_pot_max = 228;
int mouse_pen_ofs_h = 42;
int mouse_pen_ofs_v = 2;
int mouse_joy_inertia = 10;

static UBYTE STICK[4];
static UBYTE TRIG_input[4];
static int mouse_x = 0;
static int mouse_y = 0;
static int mouse_move_x = 0;
static int mouse_move_y = 0;
static int mouse_pen_show_pointer = 0;
static UBYTE mouse_amiga_codes[16] = {
    0x00, 0x02, 0x0a, 0x08,
    0x01, 0x03, 0x0b, 0x09,
    0x05, 0x07, 0x0f, 0x0d,
    0x04, 0x06, 0x0e, 0x0c
    };
static UBYTE mouse_st_codes[16] = {
    0x00, 0x02, 0x03, 0x01,
    0x08, 0x0a, 0x0b, 0x09,
    0x0c, 0x0e, 0x0f, 0x0d,
    0x04, 0x06, 0x07, 0x05
    };
static int max_scanline_counter;
static int scanline_counter;
#define MOUSE_SHIFT 4

/* Prefs */
extern int prefsArgc;
extern char *prefsArgv[];

// sound 
#define FRAGSIZE        11      // 1<<FRAGSIZE is size of sound buffer
static int dsprate = 44100;

// video
SDL_Surface *MainScreen = NULL;
SDL_Color colors[256];          // palette
Uint16 Palette16[256];          // 16-bit palette
Uint32 Palette32[256];          // 32-bit palette
static int our_width, our_height, our_bpp; // The variables for storing screen width
static char windowCaption[40];

// keyboard variables 
Uint8 *kbhits;
static int last_key_break = 0;
static int last_key_code = AKEY_NONE;
static int CONTROL = 0x00;

/*------------------------------------------------------------------------------
*  Sound_Update - Empty fucntion needed by emulator.  Actual sound processing
*   done by SDL sound functions.
*-----------------------------------------------------------------------------*/
int Sound_Update(void)
{
    return 0;
}                               // fake function

/*------------------------------------------------------------------------------
*  SetPalette - Call to set the Pallete used by the emulator.  Uses the global
*   variable colors[].
*-----------------------------------------------------------------------------*/
static void SetPalette()
{
    SDL_SetPalette(MainScreen, SDL_LOGPAL | SDL_PHYSPAL, colors, 0, 256);
}

/*------------------------------------------------------------------------------
*  CalcPalette - Call to calculate a b/w or color pallete for use by the
*   emulator.  May be replaced by ability to use user defined palletes.
*-----------------------------------------------------------------------------*/
void CalcPalette()
{
    int i, rgb;
    Uint32 c;
    for (i = 0; i < 256; i++) {
         rgb = colortable[i];
         colors[i].r = (rgb & 0x00ff0000) >> 16;
         colors[i].g = (rgb & 0x0000ff00) >> 8;
         colors[i].b = (rgb & 0x000000ff) >> 0;
        }
        
    for (i = 0; i < 256; i++) {
        c = SDL_MapRGB(MainScreen->format, colors[i].r, colors[i].g,
                       colors[i].b);
        switch (MainScreen->format->BitsPerPixel) {
            case 16:
                Palette16[i] = (Uint16) c;
                break;
            case 32:
                Palette32[i] = (Uint32) c;
                break;
            }
        }

}

/*------------------------------------------------------------------------------
*  SetVideoMode - Call to set new video mode, using SDL functions.
*-----------------------------------------------------------------------------*/
void SetVideoMode(int w, int h, int bpp)
{
    PauseAudio(1);

    if (FULLSCREEN)
        MainScreen = SDL_SetVideoMode(w, h, 8, SDL_FULLSCREEN | SDL_ANYFORMAT);
    else {
        MainScreen = SDL_SetVideoMode(w, h, bpp, SDL_ANYFORMAT);
        SDL_WM_SetCaption(windowCaption,NULL);
        }

    PauseAudio(0);
        
    if (MainScreen == NULL) {
        Aprint("Setting Video Mode: %ix%ix%i FAILED", w, h, bpp);
        Aflushlog();
        exit(-1);
        }
}

/*------------------------------------------------------------------------------
*  SetNewVideoMode - Call used when emulator changes video size or windowed/full
*    screen.
*-----------------------------------------------------------------------------*/
void SetNewVideoMode(int w, int h, int bpp)
{
    float ww, hh;

    if ((h < ATARI_HEIGHT) || (w < ATARI_WIDTH)) {
        h = ATARI_HEIGHT;
        w = ATARI_WIDTH;
        }

    // aspect ratio, floats needed
    ww = w;
    hh = h;
    if (FULLSCREEN && lockFullscreenSize) {
        if (ww * 0.75 < hh)
            hh = ww * 0.75;
        else
            ww = hh / 0.75;
        }
    else {
        switch (WIDTH_MODE) {
            case SHORT_WIDTH_MODE:
                if (ww * 0.75 < hh)
                    hh = ww * 0.75;
                else
                    ww = hh / 0.75;
                break;
            case DEFAULT_WIDTH_MODE:
                if (ww / 1.4 < hh)
                    hh = ww / 1.4;
                else
                    ww = hh * 1.4;
                break;
            case FULL_WIDTH_MODE:
                if (ww / 1.6 < hh)
                    hh = ww / 1.6;
                else
                    ww = hh * 1.6;
                break;
            }
        }
        w = ww;
        h = hh;
        w = w / 8;
        w = w * 8;
        h = h / 8;
        h = h * 8;

        SDL_ShowCursor(SDL_ENABLE); // show mouse cursor 
            
        if (GRAB_MOUSE)
            SDL_WM_GrabInput(SDL_GRAB_OFF);
            
        if (FULLSCREEN && lockFullscreenSize) 
            SetVideoMode(2*w, 2*h, bpp);
        else if (DOUBLESIZE)
            SetVideoMode(scaleFactor*w, scaleFactor*h, bpp);
        else
            SetVideoMode(w, h, bpp);

        SDL_ATARI_BPP = MainScreen->format->BitsPerPixel;
        if (bpp == 0) {
            Aprint("detected %ibpp", SDL_ATARI_BPP);
        if ((SDL_ATARI_BPP != 8) && (SDL_ATARI_BPP != 16)
            && (SDL_ATARI_BPP != 32)) {
            Aprint
                ("it's unsupported, so setting 8bit mode (slow conversion)");
            SetVideoMode(w, h, 8);
            }
        }

    SetPalette();
    
    if (GRAB_MOUSE)
        SDL_WM_GrabInput(SDL_GRAB_ON);

    if ( FULLSCREEN || GRAB_MOUSE)
        SDL_ShowCursor(SDL_DISABLE);    // hide mouse cursor 
    else
        SDL_ShowCursor(SDL_ENABLE); // show mouse cursor 
        
    if (FULLSCREEN)
        SDL_WarpMouse(MainScreen->w/2,MainScreen->h/2);
        
    /* Clear the alternate page, so the first redraw is entire screen */
    if (atari_screen_b)
        memset(atari_screen_b, 0, (ATARI_HEIGHT * ATARI_WIDTH));
}

/*------------------------------------------------------------------------------
*  SwitchFullscreen - Called by user interface to switch between Fullscreen and 
*    windowed modes.
*-----------------------------------------------------------------------------*/
void SwitchFullscreen()
{
    FULLSCREEN = 1 - FULLSCREEN;
    if (!FULLSCREEN) {
        CreateWindowCaption();
        UpdateMediaManagerInfo();
        SetSoundManagerStereo(stereo_enabled);
        SetSoundManagerRecording(IsSoundFileOpen());
        }
    SetNewVideoMode(our_width, our_height,
                    MainScreen->format->BitsPerPixel);
    Atari_DisplayScreen((UBYTE *) atari_screen);
}

/*------------------------------------------------------------------------------
*  SwitchDoubleSize - Called by user interface to switch between Single (336x240) 
*    and DoubleSize (672x480) screens. (Dimensions for normal width)
*-----------------------------------------------------------------------------*/
void SwitchDoubleSize()
{
    DOUBLESIZE = 1 - DOUBLESIZE;
    if (!(FULLSCREEN && lockFullscreenSize)) {
        SetNewVideoMode(our_width, our_height,
                        MainScreen->format->BitsPerPixel);
        Atari_DisplayScreen((UBYTE *) atari_screen);
        }
}

/*------------------------------------------------------------------------------
*  SwitchShowFps - Called by user interface to switch between showing and not 
*    showing Frame per second speed for the emulator in windowed mode.
*-----------------------------------------------------------------------------*/
void SwitchShowFps()
{
    SHOWFPS = 1 - SHOWFPS;
    if (!SHOWFPS && !FULLSCREEN)
        SDL_WM_SetCaption(windowCaption,NULL);
}

/*------------------------------------------------------------------------------
*  SwitchGrabMouse - Called by user interface to switch between grabbing the
*    mouse for Atari emulation or not.  Only works in Windowed mode, as mouse
*    is always grabbed in Full Screen mode.
*-----------------------------------------------------------------------------*/
void SwitchGrabMouse()
{
    GRAB_MOUSE = 1 - GRAB_MOUSE;
    if (GRAB_MOUSE) {
        if (mouse_mode != MOUSE_OFF) {
            if (!FULLSCREEN)
                SDL_ShowCursor(SDL_DISABLE);    // hide mouse cursor 
                SDL_WM_GrabInput(SDL_GRAB_ON);
                }
            }
    else {
        if (!FULLSCREEN)
            SDL_ShowCursor(SDL_ENABLE); // show mouse cursor 
        SDL_WM_GrabInput(SDL_GRAB_OFF);
        }
}

/*------------------------------------------------------------------------------
*  SwitchWidth - Called by user interface to cycle between the 3 width modes,
*    short, normal, and full.  In single mode the sizes are:
*     Short - 320x240
*     Normal -  336x240
*     Full - 384x240
*-----------------------------------------------------------------------------*/
void SwitchWidth()
{
    WIDTH_MODE++;
    if (WIDTH_MODE > FULL_WIDTH_MODE)
        WIDTH_MODE = SHORT_WIDTH_MODE;
    if (!(FULLSCREEN && lockFullscreenSize)) {
        SetNewVideoMode(our_width, our_height,
                        MainScreen->format->BitsPerPixel);
        Atari_DisplayScreen((UBYTE *) atari_screen);
    }
}

/*------------------------------------------------------------------------------
*  PauseAudio - This function pauses and unpauses the Audio output of the
*    emulator.  Used when prefs window is up, other menu windows, changing
*    from full screen to windowed, etc.  Number of calls to unpause must match
*    the number of calls to pause before unpausing will actually occur.
*-----------------------------------------------------------------------------*/
void PauseAudio(int pause)
{
    static int pauseCount = 0;
    
    if (pause) {
        if (pauseCount == 0)
            SDL_PauseAudio(1);
        pauseCount++;
        }
    else {
        if (pauseCount > 0) {
            if (pauseCount == 1)
               SDL_PauseAudio(0);
            pauseCount--;
            }
        }
}

/*------------------------------------------------------------------------------
*  SDL_Sound_Update - This function is the callback function called by SDL to
*    update sound.  It calls the Pokey function to get the new sound, then
*    calls the SDL mixer to output it.
*-----------------------------------------------------------------------------*/
void SDL_Sound_Update(void *userdata, Uint8 * stream, int len)
{
    Uint8 dsp_buffer[1 << FRAGSIZE];

    if (len > 1 << FRAGSIZE)
        len = 1 << FRAGSIZE;
        
    /* If there is no cartridge on the 5200, make sure there is no sound */
    if ((machine_type == MACHINE_5200) && (cart_type == CART_NONE))
        memset (dsp_buffer, 127, len);
    else
        Pokey_process(dsp_buffer, len);
        
    if (sound_enabled)
        bcopy(dsp_buffer, stream, len);
    else
        memset (stream, 127, len);
}

/*------------------------------------------------------------------------------
*  SDL_Sound_Start - This function starts the SDL audio device.*-----------------------------------------------------------------------------*/
void SDL_Sound_Start() 
{
    SDL_AudioSpec desired, obtained;
        
    desired.freq = dsprate;
    desired.format = AUDIO_U8;
    desired.samples = 1 << (FRAGSIZE-1);
    desired.callback = SDL_Sound_Update;
    desired.userdata = NULL;
    desired.channels = 2;

    if (SDL_OpenAudio(&desired, &obtained) < 0) {
        Aprint("Problem with audio: %s", SDL_GetError());
        Aprint("Sound is disabled.");
        sound_enabled = FALSE;
                return;
            }
}

/*------------------------------------------------------------------------------
*  SDL_Sound_Initialise - This function initializes the SDL sound.  The
*    command line arguments are not used here.
*-----------------------------------------------------------------------------*/
void SDL_Sound_Initialise(int *argc, char *argv[])
{
    int i, j;

    for (i = j = 1; i < *argc; i++) {
        if (strcmp(argv[i], "-sound") == 0)
            sound_enabled = TRUE;
        else if (strcmp(argv[i], "-nosound") == 0)
            sound_enabled = FALSE;
        else if (strcmp(argv[i], "-dsprate") == 0)
            sscanf(argv[++i], "%d", &dsprate);
        else {
            if (strcmp(argv[i], "-help") == 0) {
                Aprint("\t-sound           Enable sound\n"
                       "\t-nosound         Disable sound\n"
                       "\t-dsprate <rate>  Set DSP rate in Hz\n"
                      );
            }
            argv[j++] = argv[i];
        }
    }
    *argc = j;

    SDL_Sound_Start();
    Pokey_sound_init(FREQ_17_EXACT, dsprate, 2,0);
}

/*------------------------------------------------------------------------------
*  SDL_Sound_Initialise - This function is modeled after a similar one in 
*    ui.c.  It is used to start/stop recording of raw sound.
*-----------------------------------------------------------------------------*/
void SDL_Sound_Recording()
{
    static int record_num=0;
    char buf[128];

    if (! IsSoundFileOpen()) {
        sprintf(buf,"%d.raw",record_num);
        if (OpenSoundFile(buf)) {
            SetSoundManagerRecording(1);
            }
        }
    else {
        CloseSoundFile();
        SetSoundManagerRecording(0);
        record_num++;
        }
}

/*------------------------------------------------------------------------------
*  Atari_Keyboard - This function is called once per main loop to handle 
*    keyboard input.  Like the original SDL version, this has been derived from
*    the original routines in input.c.
*-----------------------------------------------------------------------------*/
int Atari_Keyboard(void)
{
    static int lastkey = AKEY_NONE, key_pressed = 0;
    SDL_Event event;

    /* Poll for SDL events.  All we want here are Keydown and Keyup events,
       and the quit event. */
    if (SDL_PollEvent(&event)) {
        switch (event.type) {
            case SDL_KEYDOWN:
                lastkey = event.key.keysym.sym;
                key_pressed = 1;
                break;
            case SDL_KEYUP:
                lastkey = event.key.keysym.sym;
                key_pressed = 0;
                if (lastkey == SDLK_CAPSLOCK)
                    key_pressed = 1;
                break;
            case SDL_QUIT:
                return AKEY_EXIT;
                break;
            }
        }
    kbhits = SDL_GetKeyState(NULL);

    if (kbhits == NULL) {
        Aprint("oops, kbhits is NULL!");
        Aflushlog();
        exit(-1);
    }
    
    /*  Determine if the shift key is pressed */
    if ((kbhits[SDLK_LSHIFT]) || (kbhits[SDLK_RSHIFT]))
        key_shift = 1;
    else
        key_shift = 0;

    /* Determine if the control key is pressed */
    if ((kbhits[SDLK_LCTRL]) || (kbhits[SDLK_RCTRL]))
        CONTROL = AKEY_CTRL;
    else
        CONTROL = 0;

    /*  If the command key is pressed, in fullscreen, execute the 
        emulators built-in UI, otherwise, do the command key equivelents
        here.  The reason for this, is due to the SDL lib construction,
        all key events go to the SDL app, so the menus don't handle them
        directly */
    alt_function = -1;
    if (kbhits[SDLK_LMETA]) {
        if (key_pressed) {
            switch(lastkey) {
            case SDLK_f:
                SwitchFullscreen();
                break;
            case SDLK_COMMA:
                RunPreferences();
                break;
            case SDLK_j:
                SetDisplayManagerDoubleSize(!DOUBLESIZE);
                SwitchDoubleSize();
                break;
            case SDLK_k:
                if (!FULLSCREEN)
                    SwitchShowFps();
                break;
            case SDLK_g:
                SetDisplayManagerWidthMode((WIDTH_MODE+1)%3);
                SwitchWidth();
                break;
            case SDLK_r:
                if (FULLSCREEN)
                    alt_function = MENU_RUN;
                else 
                    MediaManagerLoadExe();
                break;
            case SDLK_y:
                if (FULLSCREEN)
                    alt_function = MENU_SYSTEM;
                break;
            case SDLK_e:
                if (FULLSCREEN)
                    alt_function = MENU_SOUND;
                else 
                    SoundManagerStereoDo();
                break;
            case SDLK_p:
                ControlManagerPauseEmulator();
                break;
            case SDLK_t:
                if (FULLSCREEN)
                    alt_function = MENU_SOUND_RECORDING;
                else 
                    SoundManagerRecordingDo();
                break;
            case SDLK_a:
                if (FULLSCREEN)
                    alt_function = MENU_ABOUT;
                else
                    ControlManagerAboutApp();
                break;
            case SDLK_s:
                if (FULLSCREEN)
                    alt_function = MENU_SAVESTATE;
                else 
                    ControlManagerSaveState();
                break;
            case SDLK_d:
                if (FULLSCREEN)
                    alt_function = MENU_DISK;
                else 
                    MediaManagerRunDiskManagement();
                break;
            case SDLK_l:
                if (FULLSCREEN)
                    alt_function = MENU_LOADSTATE;
                else 
                    ControlManagerLoadState();
                break;
            case SDLK_o:
                if (FULLSCREEN)
                    alt_function = MENU_CARTRIDGE;
                else {
                    if (key_shift)
                        MediaManagerRemoveCartridge();
                    else
                        MediaManagerInsertCartridge();
                    }
                break;
            case SDLK_1:
            case SDLK_2:
            case SDLK_3:
            case SDLK_4:
            case SDLK_5:
            case SDLK_6:
            case SDLK_7:
            case SDLK_8:
               if (!FULLSCREEN) {
                    if (CONTROL)
                        MediaManagerRemoveDisk(lastkey - SDLK_1 + 1);
                    else
                        MediaManagerInsertDisk(lastkey - SDLK_1 + 1);
                    }
                break;
            case SDLK_0:
                if (!FULLSCREEN)
                    if (CONTROL)
                        MediaManagerRemoveDisk(0);
                break;
            case SDLK_q:
                return AKEY_EXIT;
                break;
            case SDLK_h:
                ControlManagerHideApp();
                break;
            case SDLK_m:
                ControlManagerMiniturize();
                break;
            case SDLK_SLASH:
                if (key_shift)
                    ControlManagerShowHelp();
                break;
            }
        }

        key_pressed = 0;
        if (alt_function != -1)
            return AKEY_UI;
        else
            return AKEY_NONE;
    }


    /* Handle the Atari Console Keys */
    key_consol = CONSOL_NONE;
    if (kbhits[SDLK_F2])
        key_consol &= (~CONSOL_OPTION);
    if (kbhits[SDLK_F3])
        key_consol &= (~CONSOL_SELECT);
    if (kbhits[SDLK_F4])
        key_consol &= (~CONSOL_START);

    if (key_pressed == 0) {
        return AKEY_NONE;
        }

    /* Handle the key translation for shifted characters */
    if (key_shift)
        switch (lastkey) {
        case SDLK_a:
            return AKEY_A;
        case SDLK_b:
            return AKEY_B;
        case SDLK_c:
            return AKEY_C;
        case SDLK_d:
            return AKEY_D;
        case SDLK_e:
            return AKEY_E;
        case SDLK_f:
            return AKEY_F;
        case SDLK_g:
            return AKEY_G;
        case SDLK_h:
            return AKEY_H;
        case SDLK_i:
            return AKEY_I;
        case SDLK_j:
            return AKEY_J;
        case SDLK_k:
            return AKEY_K;
        case SDLK_l:
            return AKEY_L;
        case SDLK_m:
            return AKEY_M;
        case SDLK_n:
            return AKEY_N;
        case SDLK_o:
            return AKEY_O;
        case SDLK_p:
            return AKEY_P;
        case SDLK_q:
            return AKEY_Q;
        case SDLK_r:
            return AKEY_R;
        case SDLK_s:
            return AKEY_S;
        case SDLK_t:
            return AKEY_T;
        case SDLK_u:
            return AKEY_U;
        case SDLK_v:
            return AKEY_V;
        case SDLK_w:
            return AKEY_W;
        case SDLK_x:
            return AKEY_X;
        case SDLK_y:
            return AKEY_Y;
        case SDLK_z:
            return AKEY_Z;
        case SDLK_SEMICOLON:
            return AKEY_COLON;
        case SDLK_F5:
            return AKEY_COLDSTART;
        case SDLK_1:
            return AKEY_EXCLAMATION;
        case SDLK_2:
            return AKEY_AT;
        case SDLK_3:
            return AKEY_HASH;
        case SDLK_4:
            return AKEY_DOLLAR;
        case SDLK_5:
            return AKEY_PERCENT;
        case SDLK_6:
            return AKEY_CARET;
        case SDLK_7:
            return AKEY_AMPERSAND;
        case SDLK_8:
            return AKEY_ASTERISK;
        case SDLK_9:
            return AKEY_PARENLEFT;
        case SDLK_0:
            return AKEY_PARENRIGHT;
        case SDLK_EQUALS:
            return AKEY_PLUS;
        case SDLK_MINUS:
            return AKEY_UNDERSCORE;
        case SDLK_QUOTE:
            return AKEY_DBLQUOTE;
        case SDLK_SLASH:
            return AKEY_QUESTION;
        case SDLK_COMMA:
            return AKEY_LESS;
        case SDLK_PERIOD:
            return AKEY_GREATER;
        case SDLK_BACKSLASH:
            return AKEY_BAR;
        case SDLK_F10:
            key_pressed = 0;
            return AKEY_SCREENSHOT_INTERLACE;
        case SDLK_INSERT:
            return AKEY_INSERT_LINE;
        case SDLK_CAPSLOCK:
            key_pressed = 0;
            return AKEY_CAPSLOCK;
        }
    /* Handle the key translation for non-shifted characters 
       First check to make sure it isn't an emulated joystick key,
       since if it is, it should be ignored.  */
    else {
        if (SDL_IsJoyKey[lastkey])
            return AKEY_NONE;
        if (machine_type == MACHINE_5200)
            {
            if ((key_consol & CONSOL_START) == 0)
                return(AKEY_5200_START);

            switch(lastkey) {
            case SDLK_p:
                return(AKEY_5200_PAUSE);	/* pause */
            case SDLK_r:
                return(AKEY_5200_RESET);	/* reset (5200 has no warmstart) */
            case SDLK_0:
            case SDLK_KP0:
                return(AKEY_5200_0);		/* controller numpad keys (0-9) */
            case SDLK_1:
            case SDLK_KP1:
                return(AKEY_5200_1);
            case SDLK_2:
            case SDLK_KP2:
                return(AKEY_5200_2);
            case SDLK_3:
            case SDLK_KP3:
                return(AKEY_5200_3);
            case SDLK_4:
            case SDLK_KP4:
                return(AKEY_5200_4);
            case SDLK_5:
            case SDLK_KP5:
                return(AKEY_5200_5);
            case SDLK_6:
            case SDLK_KP6:
                return(AKEY_5200_6);
            case SDLK_7:
            case SDLK_KP7:
                return(AKEY_5200_7);
            case SDLK_8:
            case SDLK_KP8:
                return(AKEY_5200_8);
            case SDLK_9:
            case SDLK_KP9:
                return(AKEY_5200_9);
            case SDLK_MINUS:
            case SDLK_KP_MINUS:
                return(AKEY_5200_HASH);	/* # key on 5200 controller */
            case SDLK_ASTERISK:
            case SDLK_KP_MULTIPLY:
                return(AKEY_5200_ASTERISK);	/* * key on 5200 controller */
            case SDLK_F9:
            case SDLK_F8:
            case SDLK_F10:
            case SDLK_F11:
            case SDLK_F1:
            case SDLK_F5:
            case SDLK_LEFT:
            case SDLK_RIGHT:
            case SDLK_UP:
            case SDLK_DOWN:
            case SDLK_ESCAPE:
            case SDLK_TAB:
            case SDLK_RETURN:
                break;
            default:
                return AKEY_NONE;
                }
            }
        switch (lastkey) {
        case SDLK_a:
            return AKEY_a;
        case SDLK_b:
            return AKEY_b;
        case SDLK_c:
            return AKEY_c;
        case SDLK_d:
            return AKEY_d;
        case SDLK_e:
            return AKEY_e;
        case SDLK_f:
            return AKEY_f;
        case SDLK_g:
            return AKEY_g;
        case SDLK_h:
            return AKEY_h;
        case SDLK_i:
            return AKEY_i;
        case SDLK_j:
            return AKEY_j;
        case SDLK_k:
            return AKEY_k;
        case SDLK_l:
            return AKEY_l;
        case SDLK_m:
            return AKEY_m;
        case SDLK_n:
            return AKEY_n;
        case SDLK_o:
            return AKEY_o;
        case SDLK_p:
            return AKEY_p;
        case SDLK_q:
            return AKEY_q;
        case SDLK_r:
            return AKEY_r;
        case SDLK_s:
            return AKEY_s;
        case SDLK_t:
            return AKEY_t;
        case SDLK_u:
            return AKEY_u;
        case SDLK_v:
            return AKEY_v;
        case SDLK_w:
            return AKEY_w;
        case SDLK_x:
            return AKEY_x;
        case SDLK_y:
            return AKEY_y;
        case SDLK_z:
            return AKEY_z;
        case SDLK_SEMICOLON:
            return AKEY_SEMICOLON;
        case SDLK_F5:
            return AKEY_WARMSTART;
        case SDLK_0:
            return AKEY_0;
        case SDLK_1:
            return AKEY_1;
        case SDLK_2:
            return AKEY_2;
        case SDLK_3:
            return AKEY_3;
        case SDLK_4:
            return AKEY_4;
        case SDLK_5:
            return AKEY_5;
        case SDLK_6:
            return AKEY_6;
        case SDLK_7:
            return AKEY_7;
        case SDLK_8:
            return AKEY_8;
        case SDLK_9:
            return AKEY_9;
        case SDLK_COMMA:
            return AKEY_COMMA;
        case SDLK_PERIOD:
            return AKEY_FULLSTOP;
        case SDLK_EQUALS:
            return AKEY_EQUAL;
        case SDLK_MINUS:
            return AKEY_MINUS;
        case SDLK_QUOTE:
            return AKEY_QUOTE;
        case SDLK_SLASH:
            return AKEY_SLASH;
        case SDLK_BACKSLASH:
            return AKEY_BACKSLASH;
        case SDLK_LEFTBRACKET:
            return AKEY_BRACKETLEFT;
        case SDLK_RIGHTBRACKET:
            return AKEY_BRACKETRIGHT;
        case SDLK_F7:
            key_pressed = 0;
            requestLimitChange=1;
            return AKEY_NONE;
        case SDLK_F8:
            key_pressed = 0;
            requestMonitor = TRUE;
            if (!Atari800_Exit(TRUE)) 
                return AKEY_EXIT;
            else
                return AKEY_NONE;
        case SDLK_F10:
            key_pressed = 0;
            return AKEY_SCREENSHOT;
        case SDLK_F11:
            if (!FULLSCREEN)
                 SwitchGrabMouse();
        case SDLK_INSERT:
            return AKEY_INSERT_CHAR;
        case SDLK_CAPSLOCK:
            key_pressed = 0;
            return AKEY_CAPSTOGGLE;
            }
        }

    /* Handle the key translation for special characters 
       First check to make sure it isn't an emulated joystick key,
       since if it is, it should be ignored.  */ 
    if (SDL_IsJoyKey[lastkey])
        return AKEY_NONE;
    switch (lastkey) {
    case SDLK_END:
        return AKEY_ATARI;
    case SDLK_PAGEDOWN:
        return AKEY_HELP;
    case SDLK_PAGEUP:
        return AKEY_CAPSLOCK;
    case SDLK_HOME:
        return AKEY_CLEAR;
    case SDLK_PAUSE:
    case SDLK_BACKQUOTE:
        return AKEY_BREAK;
    case SDLK_CAPSLOCK:
        if (key_shift)
            return AKEY_CAPSLOCK;
        else
            return AKEY_CAPSTOGGLE;
    case SDLK_SPACE:
        return AKEY_SPACE;
    case SDLK_BACKSPACE:
        return AKEY_BACKSPACE;
    case SDLK_RETURN:
        return AKEY_RETURN;
    case SDLK_F9:
        return AKEY_EXIT;
    case SDLK_F1:
        return AKEY_UI;
    case SDLK_LEFT:
        return AKEY_LEFT;
    case SDLK_RIGHT:
        return AKEY_RIGHT;
    case SDLK_UP:
        return AKEY_UP;
    case SDLK_DOWN:
        return AKEY_DOWN;
    case SDLK_ESCAPE:
        return AKEY_ESCAPE;
    case SDLK_TAB:
        if (key_shift)
            return AKEY_SETTAB;
        else if (CONTROL)
            return AKEY_CLRTAB;
        else
            return AKEY_TAB;
    case SDLK_DELETE:
        if (key_shift)
            return AKEY_DELETE_LINE;
        else
            return AKEY_DELETE_CHAR;
    case SDLK_INSERT:
        if (key_shift)
            return AKEY_INSERT_LINE;
        else
            return AKEY_INSERT_CHAR;
    }
    return AKEY_NONE;
}

/*------------------------------------------------------------------------------
*  Check_SDL_Joysticks - Initialize the use of up to two joysticks from the SDL
*   Library.  
*-----------------------------------------------------------------------------*/
void Check_SDL_Joysticks()
{
    if (joystick0 != NULL) {
        Aprint("Joystick 0 is %s", SDL_JoystickName(0));
        joystick0_nbuttons = SDL_JoystickNumButtons(joystick0);
        joystick0_nsticks = SDL_JoystickNumAxes(joystick0)/2;
        joystick0_nhats = SDL_JoystickNumHats(joystick0);
        if (joystick0TypePref == JOY_TYPE_STICK) {
            if (joystick0_nsticks) 
                joystickType0 = JOY_TYPE_STICK;
            else if (joystick0_nhats) {
                Aprint("Joystick 0: Analog not available, using digital");
                joystickType0 = JOY_TYPE_HAT;
                }
            else {
                Aprint("Joystick 0: Analog and Digital not available, using buttons");
                joystickType0 = JOY_TYPE_BUTTONS;
                }
            }
        else if (joystick0TypePref == JOY_TYPE_HAT) {
            if (joystick0_nhats)
                joystickType0 = JOY_TYPE_HAT;
            else if (joystick0_nsticks) {
                Aprint("Joystick 0: Digital not available, using analog");
                joystickType0 = JOY_TYPE_STICK;
                }
            else {
                Aprint("Joystick 0: Analog and Digital not available, using buttons");
                joystickType0 = JOY_TYPE_BUTTONS;
                }
            }
        else
            joystickType0 = JOY_TYPE_BUTTONS;
        }
        
    if (joystick1 != NULL) {
        Aprint("Joystick 1 is %s", SDL_JoystickName(1));
        joystick1_nbuttons = SDL_JoystickNumButtons(joystick1);
        joystick1_nsticks = SDL_JoystickNumAxes(joystick1)/2;
        joystick1_nhats = SDL_JoystickNumHats(joystick1);
        if (joystick1TypePref == JOY_TYPE_STICK) {
            if (joystick1_nsticks)
                joystickType1 = JOY_TYPE_STICK;
            else if (joystick1_nhats) {
                Aprint("Joystick 1: Analog not available, using digital");
                joystickType1 = JOY_TYPE_HAT;
                }
            else {
                Aprint("Joystick 1: Analog and Digital not available, using buttons");
                joystickType1 = JOY_TYPE_BUTTONS;
                }
            }
        else if (joystick1TypePref == JOY_TYPE_HAT) {
            if (joystick1_nhats)
                joystickType1 = JOY_TYPE_HAT;
            else if (joystick1_nsticks) {
                Aprint("Joystick 1: Digital not available, using analog");
                joystickType1 = JOY_TYPE_STICK;
                }
            else {
                Aprint("Joystick 1: Analog and Digital not available, using buttons");
                joystickType1 = JOY_TYPE_BUTTONS;
                }
            }
        else
            joystickType1 = JOY_TYPE_BUTTONS;
        }

    if (joystick2 != NULL) {
        Aprint("Joystick 2 is %s", SDL_JoystickName(2));
        joystick2_nbuttons = SDL_JoystickNumButtons(joystick2);
        joystick2_nsticks = SDL_JoystickNumAxes(joystick2)/2;
        joystick2_nhats = SDL_JoystickNumHats(joystick2);
        if (joystick2TypePref == JOY_TYPE_STICK) {
            if (joystick2_nsticks)
                joystickType2 = JOY_TYPE_STICK;
            else if (joystick2_nhats) {
                Aprint("Joystick 2: Analog not available, using digital");
                joystickType2 = JOY_TYPE_HAT;
                }
            else {
                Aprint("Joystick 2: Analog and Digital not available, using buttons");
                joystickType2 = JOY_TYPE_BUTTONS;
                }
            }
        else if (joystick2TypePref == JOY_TYPE_HAT) {
            if (joystick2_nhats)
                joystickType2 = JOY_TYPE_HAT;
            else if (joystick2_nsticks) {
                Aprint("Joystick 2: Digital not available, using analog");
                joystickType2 = JOY_TYPE_STICK;
                }
            else {
                Aprint("Joystick 2: Analog and Digital not available, using buttons");
                joystickType2 = JOY_TYPE_BUTTONS;
                }
            }
        else
            joystickType2 = JOY_TYPE_BUTTONS;
        }

    if (joystick3 != NULL) {
        Aprint("Joystick 3 is %s", SDL_JoystickName(3));
        joystick3_nbuttons = SDL_JoystickNumButtons(joystick3);
        joystick3_nsticks = SDL_JoystickNumAxes(joystick3)/2;
        joystick3_nhats = SDL_JoystickNumHats(joystick3);
        if (joystick1TypePref == JOY_TYPE_STICK) {
            if (joystick3_nsticks)
                joystickType3 = JOY_TYPE_STICK;
            else if (joystick3_nhats) {
                Aprint("Joystick 3: Analog not available, using digital");
                joystickType3 = JOY_TYPE_HAT;
                }
            else {
                Aprint("Joystick 3: Analog and Digital not available, using buttons");
                joystickType3 = JOY_TYPE_BUTTONS;
                }
            }
        else if (joystick3TypePref == JOY_TYPE_HAT) {
            if (joystick3_nhats)
                joystickType3 = JOY_TYPE_HAT;
            else if (joystick3_nsticks) {
                Aprint("Joystick 3: Digital not available, using analog");
                joystickType3 = JOY_TYPE_STICK;
                }
            else {
                Aprint("Joystick 3: Analog and Digital not available, using buttons");
                joystickType3 = JOY_TYPE_BUTTONS;
                }
            }
        else
            joystickType3 = JOY_TYPE_BUTTONS;
        }
}

/*------------------------------------------------------------------------------
*  Open_SDL_Joysticks - Initialize the use of up to two joysticks from the SDL
*   Library.  
*-----------------------------------------------------------------------------*/
void Open_SDL_Joysticks()
{
    joystick0 = SDL_JoystickOpen(0);
    joystick1 = SDL_JoystickOpen(1);
    joystick2 = SDL_JoystickOpen(2);
    joystick3 = SDL_JoystickOpen(3);
    Check_SDL_Joysticks();
}

/*------------------------------------------------------------------------------
*  Remove_Bad_SDL_Joysticks - If the user has requested joysticks in the
*   preferences that aren't plugged in, make sure that it is switch to no 
*   joystick for that port.
*-----------------------------------------------------------------------------*/
void Remove_Bad_SDL_Joysticks()
{
    int i;

    if (joystick0 == NULL) {
        for (i=0;i<NUM_JOYSTICKS;i++) {
            if (JOYSTICK_MODE[i] == SDL0_JOYSTICK)
                JOYSTICK_MODE[i] = NO_JOYSTICK;
            }
        Aprint("joystick 0 not found");
        }

    if (joystick1 == NULL) {
        for (i=0;i<NUM_JOYSTICKS;i++) {
            if (JOYSTICK_MODE[i] == SDL1_JOYSTICK)
                JOYSTICK_MODE[i] = NO_JOYSTICK;
            }
        Aprint("joystick 1 not found");
        }

    if (joystick2 == NULL) {
        for (i=0;i<NUM_JOYSTICKS;i++) {
            if (JOYSTICK_MODE[i] == SDL2_JOYSTICK)
                JOYSTICK_MODE[i] = NO_JOYSTICK;
            }
        Aprint("joystick 2 not found");
        }

    if (joystick3 == NULL) {
        for (i=0;i<NUM_JOYSTICKS;i++) {
            if (JOYSTICK_MODE[i] == SDL3_JOYSTICK)
                JOYSTICK_MODE[i] = NO_JOYSTICK;
            }
        Aprint("joystick 3 not found");
        }
}

/*------------------------------------------------------------------------------
*  Init_SDL_Joykeys - Initializes the array that is used to determine if a 
*    keypress is used by the emulated joysticks.
*-----------------------------------------------------------------------------*/
void Init_SDL_Joykeys()
{
    int i;
    int keypadJoystick = FALSE;
    int userDefJoystick = FALSE;

    for (i=0;i<SDLK_LAST;i++)
        SDL_IsJoyKey[i] = 0;
    
    /*  Check to see if any of the joystick ports use the Keypad as joystick */
    for (i=0;i<NUM_JOYSTICKS;i++)
        if (JOYSTICK_MODE[i] == KEYPAD_JOYSTICK)
            keypadJoystick = TRUE;
    /*  Check to see if any of the joystick ports use the User defined keys
          as joystick */
    for (i=0;i<NUM_JOYSTICKS;i++)
        if (JOYSTICK_MODE[i] == USERDEF_JOYSTICK)
            userDefJoystick = TRUE;
     
    /* If used as a joystick, set the keypad keys as emulated */               
    if (keypadJoystick) {
        SDL_IsJoyKey[SDL_TRIG_0] = 1;
        SDL_IsJoyKey[SDL_TRIG_0_B] = 1;
        SDL_IsJoyKey[SDL_JOY_0_LEFT] = 1;
        SDL_IsJoyKey[SDL_JOY_0_RIGHT] = 1;
        SDL_IsJoyKey[SDL_JOY_0_DOWN] = 1;
        SDL_IsJoyKey[SDL_JOY_0_UP] = 1;
        SDL_IsJoyKey[SDL_JOY_0_LEFTUP] = 1;
        SDL_IsJoyKey[SDL_JOY_0_RIGHTUP] = 1;
        SDL_IsJoyKey[SDL_JOY_0_LEFTDOWN] = 1;
        SDL_IsJoyKey[SDL_JOY_0_RIGHTDOWN] = 1;
    }

    /* If used as a joystick, set the user defined keys as emulated */               
    if (userDefJoystick) {
        SDL_IsJoyKey[SDL_TRIG_1] = 1;
        SDL_IsJoyKey[SDL_TRIG_1_B] = 1;
        SDL_IsJoyKey[SDL_JOY_1_LEFT] = 1;
        SDL_IsJoyKey[SDL_JOY_1_RIGHT] = 1;
        SDL_IsJoyKey[SDL_JOY_1_DOWN] = 1;
        SDL_IsJoyKey[SDL_JOY_1_UP] = 1;
        SDL_IsJoyKey[SDL_JOY_1_LEFTUP] = 1;
        SDL_IsJoyKey[SDL_JOY_1_RIGHTUP] = 1;
        SDL_IsJoyKey[SDL_JOY_1_LEFTDOWN] = 1;
        SDL_IsJoyKey[SDL_JOY_1_RIGHTDOWN] = 1;
    }
}

/*------------------------------------------------------------------------------
*  Init_Joysticks - Initializes the joysticks for the emulator.  The command
*    line args are currently unused.
*-----------------------------------------------------------------------------*/
void Init_Joysticks(int *argc, char *argv[])
{
    Open_SDL_Joysticks();
    Remove_Bad_SDL_Joysticks();
    Init_SDL_Joykeys();        
}

/*------------------------------------------------------------------------------
*  Atari_Initialise - Initializes the SDL video and sound functions.  The command
*    line args are currently unused.
*-----------------------------------------------------------------------------*/
void Atari_Initialise(int *argc, char *argv[])
{
    int i, j;
    int no_joystick;
    int help_only = FALSE;

    no_joystick = 0;
    our_width = ATARI_WIDTH;
    our_height = ATARI_HEIGHT;
    our_bpp = SDL_ATARI_BPP;

    for (i = j = 1; i < *argc; i++) {
                if (strcmp(argv[i], "-nojoystick") == 0) {
            no_joystick = 1;
            i++;
        }
        else if (strcmp(argv[i], "-fullscreen") == 0) {
            FULLSCREEN = 1;
        }
        else if (strcmp(argv[i], "-windowed") == 0) {
            FULLSCREEN = 0;
        }
        else if (strcmp(argv[i], "-double") == 0) {
            DOUBLESIZE= 1;
        }
        else if (strcmp(argv[i], "-single") == 0) {
            DOUBLESIZE = 0;
        }
        else {
            if (strcmp(argv[i], "-help") == 0) {
                help_only = TRUE;
                Aprint("\t-nojoystick      Disable joystick");
                Aprint("\t-fullscreen      Run fullscreen");
                Aprint("\t-windowed        Run in window");
                Aprint("\t-double          Run double size mode");
                Aprint("\t-single          Run single size mode");
            }
            argv[j++] = argv[i];
        }
    }
    *argc = j;

    i = SDL_INIT_VIDEO | SDL_INIT_JOYSTICK | SDL_INIT_AUDIO;
    if (SDL_Init(i) != 0) {
        Aprint("SDL_Init FAILED");
        Aprint(SDL_GetError());
        Aflushlog();
        exit(-1);
    }
    atexit(SDL_Quit);

    SDL_Sound_Initialise(argc, argv);

    if (help_only)
        return;     /* return before changing the gfx mode */

    SetNewVideoMode(our_width, our_height, our_bpp);
    CalcPalette();
    SetPalette();

    if (no_joystick == 0)
        Init_Joysticks(argc, argv);
}

/*------------------------------------------------------------------------------
*  Atari_Exit - Exits the emulator.  Someday, it will allow you to run the 
*     built in monitor.  Right now, if monitor is called for, displays fatal
*     error dialog box.
*-----------------------------------------------------------------------------*/
int Atari_Exit(int run_monitor)
{
    int restart;
    int action;

    RDevice_Exit();

    if (run_monitor) {
        if (FULLSCREEN) {
            // Need to bring us back from Fullscreen 
            SwitchFullscreen();
            }
        if (requestMonitor || !break_cim) { 
            /* run the monitor....*/  
            restart = ControlManagerMonitorRun();
            requestMonitor = FALSE;
            break_cim = 0;
            }
        else {
            /* Run the crash dialogue */
            action = ControlManagerFatalError();
            if (action == 1) {
                Warmstart();
                restart = TRUE;
                }
            else if (action == 2) {
                Coldstart();
                restart = TRUE;
                }
            else if (action == 3) {
                restart = ControlManagerMonitorRun();
                }
            else if (action == 4) {
                CART_Remove();
                UpdateMediaManagerInfo();
                Coldstart();
                restart = TRUE;
                }
            else if (action == 5) {
                SIO_Dismount(1);
                UpdateMediaManagerInfo();
                Coldstart();
                restart = TRUE;
                }
            else {
                restart = FALSE;
                }
            }
        }
    else
        restart = FALSE;

    if (restart) {
        /* set up graphics and all the stuff */
        return 1;
    }
    else
        return(0);
#if 0        
    if (run_monitor) {
        if (FULLSCREEN) {
            // Need to bring us back from Fullscreen 
            SwitchFullscreen();
        }
        action = ControlManagerFatalError();
        if (action == 1) {
            Warmstart();
            restart = TRUE;
            }
        else if (action == 2) {
            Coldstart();
            restart = TRUE;
            }
        else {
            restart = FALSE;
            }
        }
#endif    
}

/*------------------------------------------------------------------------------
*  DisplayWithoutScaling8bpp - Displays the emulated atari screen, in single
*    size mode, with 8 bits per pixel.
*-----------------------------------------------------------------------------*/
void DisplayWithoutScaling8bpp(Uint8 * screen, int jumped, int width,
                                 int first_row, int last_row)
{
    register Uint32 *start32;
    register int pitch4;
    int i;

    pitch4 = MainScreen->pitch / 4;
    start32 = (Uint32 *) MainScreen->pixels + (first_row * pitch4);

    screen = screen + jumped + (first_row * ATARI_WIDTH);
    i = last_row - first_row + 1;
    while (i > 0) {
        memcpy(start32, screen, width);
        screen += ATARI_WIDTH;
        start32 += pitch4;
        i--;
        }
}

/*------------------------------------------------------------------------------
*  DisplayWithoutScaling16bpp - Displays the emulated atari screen, in single
*    size mode, with 16 bits per pixel.
*-----------------------------------------------------------------------------*/
void DisplayWithoutScaling16bpp(Uint8 * screen, int jumped, int width,
                                 int first_row, int last_row)
{
    register Uint8 *fromPtr;
    register Uint16 *toPtr;
    register int i,j;
    register Uint32 *start32;
    register int pitch4;

    pitch4 = MainScreen->pitch / 4;
    start32 = (Uint32 *) MainScreen->pixels + (first_row * pitch4);

    screen = screen + jumped + (first_row * ATARI_WIDTH);
    i = last_row - first_row + 1;
    while (i > 0) {
        j = width;
        toPtr = (Uint16 *) start32;
        fromPtr = screen;
        while (j > 0) {
            *toPtr++ = Palette16[*fromPtr++];
            j--;
            }
        screen += ATARI_WIDTH;
        start32 += pitch4;
        i--;
        }
}

/*------------------------------------------------------------------------------
*  DisplayWithoutScaling32bpp - Displays the emulated atari screen, in single
*    size mode, with 32 bits per pixel.
*-----------------------------------------------------------------------------*/
void DisplayWithoutScaling32bpp(Uint8 * screen, int jumped, int width,
                                 int first_row, int last_row)
{
    register Uint8 *fromPtr;
    register Uint32 *toPtr;
    register int i,j;
    register Uint32 *start32;
    register int pitch4;

    pitch4 = MainScreen->pitch / 4;
    start32 = (Uint32 *) MainScreen->pixels + (first_row * pitch4);

    screen = screen + jumped + (first_row * ATARI_WIDTH);
    i = last_row - first_row + 1;
    while (i > 0) {
        j = width;
        toPtr = (Uint32 *) start32;
        fromPtr = screen;
        while (j > 0) {
            *toPtr++ = Palette32[*fromPtr++];
            j--;
            }
        screen += ATARI_WIDTH;
        start32 += pitch4;
        i--;
        }
}

/*------------------------------------------------------------------------------
*  DisplayWith2xScaling8bpp - Displays the emulated atari screen, in double
*    size mode, with 8 bits per pixel.
*-----------------------------------------------------------------------------*/
void DisplayWith2xScaling8bpp(Uint8 * screen, int jumped, int width,
                                 int first_row, int last_row)
{
    register Uint8 *fromptr, *toptr;
    register int i,j;
    register Uint32 *start32;
    register int pitch4;

    pitch4 = MainScreen->pitch / 4;
    start32 = (Uint32 *) MainScreen->pixels + (first_row * pitch4 * 2);

    screen = screen + jumped + (first_row * ATARI_WIDTH);
    i = last_row - first_row + 1;
    while (i > 0) {
        toptr = (Uint8 *) start32;
        fromptr = screen;
        j = width;
        while (j>0) {
           *toptr++ = *fromptr;
           *toptr++ = *fromptr++;
           j--;
           }
        start32 += pitch4;
        toptr = (Uint8 *) start32;
        fromptr = screen;
        j = width;
        while (j>0) {
           *toptr++ = *fromptr;
           *toptr++ = *fromptr++;
           j--;
           }
        screen += ATARI_WIDTH;
        start32 += pitch4;
        i--;
    }
}

/*------------------------------------------------------------------------------
*  DisplayWith3xScaling8bpp - Displays the emulated atari screen, in triple
*    size mode, with 8 bits per pixel.
*-----------------------------------------------------------------------------*/
void DisplayWith3xScaling8bpp(Uint8 * screen, int jumped, int width,
                                 int first_row, int last_row)
{
    register Uint8 *fromptr, *toptr;
    register int i,j;
    register Uint32 *start32;
    register int pitch4;

    pitch4 = MainScreen->pitch / 4;
    start32 = (Uint32 *) MainScreen->pixels + (first_row * pitch4 * 3);

    screen = screen + jumped + (first_row * ATARI_WIDTH);
    i = last_row - first_row + 1;
    while (i > 0) {
        toptr = (Uint8 *) start32;
        fromptr = screen;
        j = width;
        while (j>0) {
           *toptr++ = *fromptr;
           *toptr++ = *fromptr;
           *toptr++ = *fromptr++;
           j--;
           }
        start32 += pitch4;
        toptr = (Uint8 *) start32;
        fromptr = screen;
        j = width;
        while (j>0) {
           *toptr++ = *fromptr;
           *toptr++ = *fromptr;
           *toptr++ = *fromptr++;
           j--;
           }
        start32 += pitch4;
        toptr = (Uint8 *) start32;
        fromptr = screen;
        j = width;
        while (j>0) {
           *toptr++ = *fromptr;
           *toptr++ = *fromptr;
           *toptr++ = *fromptr++;
           j--;
           }
        screen += ATARI_WIDTH;
        start32 += pitch4;
        i--;
    }
}

/*------------------------------------------------------------------------------
*  DisplayWith4xScaling8bpp - Displays the emulated atari screen, in four times
*    size mode, with 8 bits per pixel.
*-----------------------------------------------------------------------------*/
void DisplayWith4xScaling8bpp(Uint8 * screen, int jumped, int width,
                                 int first_row, int last_row)
{
    register Uint8 *fromptr, *toptr;
    register int i,j;
    register Uint32 *start32;
    register int pitch4;

    pitch4 = MainScreen->pitch / 4;
    start32 = (Uint32 *) MainScreen->pixels + (first_row * pitch4 * 4);

    screen = screen + jumped + (first_row * ATARI_WIDTH);
    i = last_row - first_row + 1;
    while (i > 0) {
        toptr = (Uint8 *) start32;
        fromptr = screen;
        j = width;
        while (j>0) {
           *toptr++ = *fromptr;
           *toptr++ = *fromptr;
           *toptr++ = *fromptr;
           *toptr++ = *fromptr++;
           j--;
           }
        start32 += pitch4;
        toptr = (Uint8 *) start32;
        fromptr = screen;
        j = width;
        while (j>0) {
           *toptr++ = *fromptr;
           *toptr++ = *fromptr;
           *toptr++ = *fromptr;
           *toptr++ = *fromptr++;
           j--;
           }
        start32 += pitch4;
        toptr = (Uint8 *) start32;
        fromptr = screen;
        j = width;
        while (j>0) {
           *toptr++ = *fromptr;
           *toptr++ = *fromptr;
           *toptr++ = *fromptr;
           *toptr++ = *fromptr++;
           j--;
           }
        start32 += pitch4;
        toptr = (Uint8 *) start32;
        fromptr = screen;
        j = width;
        while (j>0) {
           *toptr++ = *fromptr;
           *toptr++ = *fromptr;
           *toptr++ = *fromptr;
           *toptr++ = *fromptr++;
           j--;
           }
        screen += ATARI_WIDTH;
        start32 += pitch4;
        i--;
    }
}

/*------------------------------------------------------------------------------
*  DisplayWith2xScaling16bpp - Displays the emulated atari screen, in double
*    size mode, with 16 bits per pixel.
*-----------------------------------------------------------------------------*/
void DisplayWith2xScaling16bpp(Uint8 * screen, int jumped, int width,
                                 int first_row, int last_row)
{
    register Uint8 *fromPtr;
    register Uint16 *toPtr;
    register Uint16 quad;
    register int i,j;
    register Uint32 *start32;
    register int pitch4;

    pitch4 = MainScreen->pitch / 4;
    start32 = (Uint32 *) MainScreen->pixels + (first_row * pitch4 * 2);

    screen = screen + jumped + (first_row * ATARI_WIDTH);
    i = last_row - first_row + 1;
    while (i > 0) {
        j = width;
        toPtr = (Uint16 *) start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette16[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        start32 += pitch4;
        j = width;
        toPtr = (Uint16 *) start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette16[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        screen += ATARI_WIDTH;
        start32 += pitch4;
        i--;
        }
}

/*------------------------------------------------------------------------------
*  DisplayWith3xScaling16bpp - Displays the emulated atari screen, in triple
*    size mode, with 16 bits per pixel.
*-----------------------------------------------------------------------------*/
void DisplayWith3xScaling16bpp(Uint8 * screen, int jumped, int width,
                                 int first_row, int last_row)
{
    register Uint8 *fromPtr;
    register Uint16 *toPtr;
    register Uint16 quad;
    register int i,j;
    register Uint32 *start32;
    register int pitch4;

    pitch4 = MainScreen->pitch / 4;
    start32 = (Uint32 *) MainScreen->pixels + (first_row * pitch4 * 3);

    screen = screen + jumped + (first_row * ATARI_WIDTH);
    i = last_row - first_row + 1;
    while (i > 0) {
        j = width;
        toPtr = (Uint16 *) start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette16[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        start32 += pitch4;
        j = width;
        toPtr = (Uint16 *) start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette16[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        start32 += pitch4;
        j = width;
        toPtr = (Uint16 *) start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette16[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        screen += ATARI_WIDTH;
        start32 += pitch4;
        i--;
        }
}

/*------------------------------------------------------------------------------
*  DisplayWith4xScaling16bpp - Displays the emulated atari screen, in four times
*    size mode, with 16 bits per pixel.
*-----------------------------------------------------------------------------*/
void DisplayWith4xScaling16bpp(Uint8 * screen, int jumped, int width,
                                 int first_row, int last_row)
{
    register Uint8 *fromPtr;
    register Uint16 *toPtr;
    register Uint16 quad;
    register int i,j;
    register Uint32 *start32;
    register int pitch4;

    pitch4 = MainScreen->pitch / 4;
    start32 = (Uint32 *) MainScreen->pixels + (first_row * pitch4 * 4);

    screen = screen + jumped + (first_row * ATARI_WIDTH);
    i = last_row - first_row + 1;
    while (i > 0) {
        j = width;
        toPtr = (Uint16 *) start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette16[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        start32 += pitch4;
        j = width;
        toPtr = (Uint16 *) start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette16[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        start32 += pitch4;
        j = width;
        toPtr = (Uint16 *) start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette16[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        start32 += pitch4;
        j = width;
        toPtr = (Uint16 *) start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette16[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        screen += ATARI_WIDTH;
        start32 += pitch4;
        i--;
        }
}

/*------------------------------------------------------------------------------
*  DisplayWith2xScaling32bpp - Displays the emulated atari screen, in double
*    size mode, with 32 bits per pixel.
*-----------------------------------------------------------------------------*/
void DisplayWith2xScaling32bpp(Uint8 * screen, int jumped, int width,
                                 int first_row, int last_row)
{
    register Uint8 *fromPtr;
    register Uint32 *toPtr;
    register Uint32 quad;
    register int i,j;
    register Uint32 *start32;
    register int pitch4;

    pitch4 = MainScreen->pitch / 4;
    start32 = (Uint32 *) MainScreen->pixels + first_row * pitch4 * 2;

    screen = screen + jumped + (first_row * ATARI_WIDTH);
    i = last_row - first_row + 1;
    while (i > 0) {
        j = width;
        toPtr = start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette32[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        start32 += pitch4;
        j = width;
        toPtr = start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette32[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        screen += ATARI_WIDTH;
        start32 += pitch4;
        i--;
        }
}

/*------------------------------------------------------------------------------
*  DisplayWith3xScaling32bpp - Displays the emulated atari screen, in triple
*    size mode, with 32 bits per pixel.
*-----------------------------------------------------------------------------*/
void DisplayWith3xScaling32bpp(Uint8 * screen, int jumped, int width,
                                 int first_row, int last_row)
{
    register Uint8 *fromPtr;
    register Uint32 *toPtr;
    register Uint32 quad;
    register int i,j;
    register Uint32 *start32;
    register int pitch4;

    pitch4 = MainScreen->pitch / 4;
    start32 = (Uint32 *) MainScreen->pixels + (first_row * pitch4 * 3);

    screen = screen + jumped + (first_row * ATARI_WIDTH);
    i = last_row - first_row + 1;
    while (i > 0) {
        j = width;
        toPtr = start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette32[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        start32 += pitch4;
        j = width;
        toPtr = start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette32[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        start32 += pitch4;
        j = width;
        toPtr = start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette32[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        screen += ATARI_WIDTH;
        start32 += pitch4;
        i--;
        }
}

/*------------------------------------------------------------------------------
*  DisplayWith4xScaling32bpp - Displays the emulated atari screen, in four times
*    size mode, with 32 bits per pixel.
*-----------------------------------------------------------------------------*/
void DisplayWith4xScaling32bpp(Uint8 * screen, int jumped, int width,
                                 int first_row, int last_row)
{
    register Uint8 *fromPtr;
    register Uint32 *toPtr;
    register Uint32 quad;
    register int i,j;
    register Uint32 *start32;
    register int pitch4;

    pitch4 = MainScreen->pitch / 4;
    start32 = (Uint32 *) MainScreen->pixels + (first_row * pitch4 * 4);

    screen = screen + jumped + (first_row * ATARI_WIDTH);
    i = last_row - first_row + 1;
    while (i > 0) {
        j = width;
        toPtr = start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette32[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        start32 += pitch4;
        j = width;
        toPtr = start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette32[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        start32 += pitch4;
        j = width;
        toPtr = start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette32[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        start32 += pitch4;
        j = width;
        toPtr = start32;
        fromPtr = screen;
        while (j > 0) {
            quad = Palette32[*fromPtr++];
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            *toPtr++ = quad;
            j--;
            }
        screen += ATARI_WIDTH;
        start32 += pitch4;
        i--;
        }
}

/*------------------------------------------------------------------------------
*  Display_Line_Equal - Determines if two display lines are equal.  Used as a 
*     test to determine which parts of the screen must be redrawn.
*-----------------------------------------------------------------------------*/
int Display_Line_Equal(ULONG *line1, ULONG *line2)
{
    int i;
    int equal = TRUE;
    
    i = ATARI_WIDTH/4;
    while (i>0) {
        if (*line1++ != *line2++) {
            equal = FALSE;
            break;
            }
        i--;
        }
    
    return(equal);      
}

/*------------------------------------------------------------------------------
*  Atari_DisplayScreen - Displays the Atari screen, redrawing only lines in the
*    middle of the screen that have changed.
*-----------------------------------------------------------------------------*/
void Atari_DisplayScreen(UBYTE * screen)
{
    int width, jumped;
    int first_row = 0;
    int last_row = ATARI_HEIGHT - 1;
    ULONG *line_start1, *line_start2;
    SDL_Rect dirty_rect;

    if (FULLSCREEN && lockFullscreenSize) {
        width = ATARI_WIDTH - 2 * 24 - 2 * 8;
        jumped = 24 + 8;
        }
    else {
        switch (WIDTH_MODE) {
        case SHORT_WIDTH_MODE:
            width = ATARI_WIDTH - 2 * 24 - 2 * 8;
            jumped = 24 + 8;
            break;
        case DEFAULT_WIDTH_MODE:
            width = ATARI_WIDTH - 2 * 24;
            jumped = 24;
            break;
        case FULL_WIDTH_MODE:
            width = ATARI_WIDTH;
            jumped = 0;
            break;
        default:
            Aprint("unsupported WIDTH_MODE");
            Aflushlog();
            exit(-1);
            break;
        }
    }

    /* Find the last row of the screen that changed since last redraw */
    line_start1 = atari_screen + (ATARI_WIDTH/4 * (ATARI_HEIGHT-1));
    line_start2 = atari_screen_b + (ATARI_WIDTH/4 * (ATARI_HEIGHT-1));
    last_row = ATARI_HEIGHT-1;
    while((0 < last_row) && Display_Line_Equal(line_start1, line_start2)) {
        last_row--;
        line_start1 -= ATARI_WIDTH/4;
        line_start2 -= ATARI_WIDTH/4;
        }
    /* Find the first row of the screen that changed since last redraw */
    line_start1 = atari_screen;
    line_start2 = atari_screen_b;
    while((first_row < last_row) && Display_Line_Equal(line_start1, line_start2)) {
        first_row++;
        line_start1 += ATARI_WIDTH/4;
        line_start2 += ATARI_WIDTH/4;
        }

    if (FULLSCREEN && lockFullscreenSize) {
        switch (MainScreen->format->BitsPerPixel) {
            case 8:
                DisplayWith2xScaling8bpp(screen, jumped, width, first_row, last_row);
                break;
            case 16:
                DisplayWith2xScaling16bpp(screen, jumped, width, first_row, last_row);
                break;
            case 32:
                DisplayWith2xScaling32bpp(screen, jumped, width, first_row, last_row);
                break;
            default:
                Aprint("unsupported color depth %i",
                       MainScreen->format->BitsPerPixel);
                Aprint
                    ("please set SDL_ATARI_BPP to 8 or 16 and recompile atari_sdl");
                Aflushlog();
                exit(-1);
            }
        }
    else if (DOUBLESIZE)
    {
        switch (MainScreen->format->BitsPerPixel) {
            case 8:
                if (scaleFactor == 2)
                    DisplayWith2xScaling8bpp(screen, jumped, width, first_row, last_row);
                else if (scaleFactor == 3)
                    DisplayWith3xScaling8bpp(screen, jumped, width, first_row, last_row);
                else
                    DisplayWith4xScaling8bpp(screen, jumped, width, first_row, last_row);
                break;
            case 16:
                if (scaleFactor == 2)
                    DisplayWith2xScaling16bpp(screen, jumped, width, first_row, last_row);
                else if (scaleFactor == 3)
                    DisplayWith3xScaling16bpp(screen, jumped, width, first_row, last_row);
                else
                    DisplayWith4xScaling16bpp(screen, jumped, width, first_row, last_row);
                break;
            case 32:
                if (scaleFactor == 2)
                    DisplayWith2xScaling32bpp(screen, jumped, width, first_row, last_row);
                else if (scaleFactor == 3)
                    DisplayWith3xScaling32bpp(screen, jumped, width, first_row, last_row);
                else
                    DisplayWith4xScaling32bpp(screen, jumped, width, first_row, last_row);
                break;
            default:
                Aprint("unsupported color depth %i",
                       MainScreen->format->BitsPerPixel);
                Aprint
                    ("please set SDL_ATARI_BPP to 8 or 16 and recompile atari_sdl");
                Aflushlog();
                exit(-1);
            }
    }
    else
    {
        switch (MainScreen->format->BitsPerPixel) {
            case 8:
                DisplayWithoutScaling8bpp(screen, jumped, width, first_row, last_row);
                break;
            case 16:
                DisplayWithoutScaling16bpp(screen, jumped, width, first_row, last_row);
                break;
            case 32:
                DisplayWithoutScaling32bpp(screen, jumped, width, first_row, last_row);
                break;
            default:
                Aprint("unsupported color depth %i",
                       MainScreen->format->BitsPerPixel);
                Aprint
                    ("please set SDL_ATARI_BPP to 8 or 16 and recompile atari_sdl");
                Aflushlog();
                exit(-1);
            }
    }
    
    /* Update the dirty part of the screen */
    dirty_rect.x = 0;
    if (FULLSCREEN && lockFullscreenSize)
    {
        dirty_rect.w = width*2;
        dirty_rect.h = (last_row - first_row + 1)*2;
        dirty_rect.y = first_row*2;
    }
    else if (DOUBLESIZE)
    {
        dirty_rect.w = width*scaleFactor;
        dirty_rect.h = (last_row - first_row + 1)*scaleFactor;
        dirty_rect.y = first_row*scaleFactor;
    }
    else
    {
        dirty_rect.w = width;
        dirty_rect.h = last_row - first_row + 1;
        dirty_rect.y = first_row;
    }   
    SDL_UpdateRects(MainScreen, 1, &dirty_rect);    
}

// two empty functions, needed by input.c and platform.h

int Atari_PORT(int num)
{
    return 0;
}

int Atari_TRIG(int num)
{
    return 0;
}

/*------------------------------------------------------------------------------
*  get_SDL_joystick_state - Convert SDL axis values into paddle values
*-----------------------------------------------------------------------------*/
Uint8 convert_SDL_to_Pot(int16 axis)
{
    Uint8 port;
    int16 naxis;
    
    naxis = -axis;
    
    if (naxis < 0) {
        naxis = -(naxis - 256);
        naxis = naxis >> 8;
        if (naxis > 128)
            naxis = 128;
        port = 128 - naxis;
        }
    else
        {
        naxis = naxis >> 8;
        if (naxis > 99)
            naxis = 99;
        port = 128 + naxis;
        }
    return(port);
}

/*------------------------------------------------------------------------------
*  convert_SDL_to_Pot5200 - Convert SDL axis values into analog 5200 joystick 
*    values
*-----------------------------------------------------------------------------*/
Uint8 convert_SDL_to_Pot5200(int16 axis)
{
    Uint8 port;
    
    if (axis < 0) {
        axis = -(axis + 256);
        axis = axis >> 8;
        if (axis > (joy_5200_center - joy_5200_min))
            axis = joy_5200_center - joy_5200_min;
        port = joy_5200_center - axis;
        }
    else
        {
        axis = axis >> 8;
        if (axis > (joy_5200_max - joy_5200_center))
            axis = joy_5200_max - joy_5200_center;
        port = joy_5200_center + axis;
        }
        
    return(port);
}

/*------------------------------------------------------------------------------
*  get_SDL_joystick_state - Return the states of one of the Attached HID 
*     joysticks.
*-----------------------------------------------------------------------------*/
int get_SDL_joystick_state(SDL_Joystick *joystick, int stickNum, int *x_pot, int *y_pot)
{
    int x;
    int y;

    x = SDL_JoystickGetAxis(joystick, stickNum*2);
    y = SDL_JoystickGetAxis(joystick, stickNum*2+1);

    if (machine_type == MACHINE_5200) {
        *x_pot = convert_SDL_to_Pot5200(x);
        *y_pot = convert_SDL_to_Pot5200(y);
        }
    else {
        *x_pot = convert_SDL_to_Pot(x);
        *y_pot = convert_SDL_to_Pot(y);
        }

    if (x > minjoy) {
        if (y < -minjoy)
            return STICK_UR;
        else if (y > minjoy)
            return STICK_LR;
        else
            return STICK_RIGHT;
    } else if (x < -minjoy) {
        if (y < -minjoy)
            return STICK_UL;
        else if (y > minjoy)
            return STICK_LL;
        else
            return STICK_LEFT;
    } else {
        if (y < -minjoy)
            return STICK_FORWARD;
        else if (y > minjoy)
            return STICK_BACK;
        else
            return STICK_CENTRE;
    }
}

/*------------------------------------------------------------------------------
*  get_SDL_hat_state - Return the states of one of the Attached HID 
*     joystick hats.
*-----------------------------------------------------------------------------*/
int get_SDL_hat_state(SDL_Joystick *joystick, int hatNum)
{
    int stick, astick = 0;

    stick = SDL_JoystickGetHat(joystick, hatNum);

    switch (stick) {
        case SDL_HAT_CENTERED:
            astick = STICK_CENTRE;
            break;
        case SDL_HAT_UP:
            astick = STICK_FORWARD;
            break;
        case SDL_HAT_RIGHT:
            astick = STICK_RIGHT;
            break;
        case SDL_HAT_DOWN:
            astick = STICK_BACK;
            break;
        case SDL_HAT_LEFT:
            astick = STICK_LEFT;
            break;
        case SDL_HAT_RIGHTUP:
            astick = STICK_UR;
            break;
        case SDL_HAT_RIGHTDOWN:
            astick = STICK_LR;
            break;
        case SDL_HAT_LEFTUP:
            astick = STICK_UL;
            break;
        case SDL_HAT_LEFTDOWN:
            astick = STICK_LL;
            break;
        }

    return(astick);
}

/*------------------------------------------------------------------------------
*  get_SDL_joy_buttons_state - Return the states of one of the Attached HID 
*     joystick based on four buttons assigned to joystick directions.
*-----------------------------------------------------------------------------*/
int get_SDL_joy_buttons_state(SDL_Joystick *joystick, int numberOfButtons)
{
    int stick = 0;
    int up = 0;
    int down = 0;
    int left = 0;
    int right = 0;
    int i;
    int joyindex;
    
    joyindex = SDL_JoystickIndex(joystick);

    if (machine_type != MACHINE_5200) {
        for (i = 0; i < numberOfButtons; i++) {
            if (SDL_JoystickGetButton(joystick, i)) {
                switch(joystickButtonKey[joyindex][i]) {
                    case AKEY_JOYSTICK_UP:
                        up = 1;
                        break;
                    case AKEY_JOYSTICK_DOWN:
                        down = 1;
                        break;
                    case AKEY_JOYSTICK_LEFT:
                        left = 1;
                        break;
                    case AKEY_JOYSTICK_RIGHT:
                        right = 1;
                        break;
                        }
                    }
                }
            }
    else {
        for (i = 0; i < numberOfButtons; i++) {
            if (SDL_JoystickGetButton(joystick, i)) {
                switch(joystick5200ButtonKey[joyindex][i]) {
                    case AKEY_JOYSTICK_UP:
                        up = 1;
                        break;
                    case AKEY_JOYSTICK_DOWN:
                        down = 1;
                        break;
                    case AKEY_JOYSTICK_LEFT:
                        left = 1;
                        break;
                    case AKEY_JOYSTICK_RIGHT:
                        right = 1;
                        break;
                        }
                    }
                }
            }
            
    if (right) {
        if (up)
            return STICK_UR;
        else if (down)
            return STICK_LR;
        else
            return STICK_RIGHT;
    } else if (left) {
        if (up)
            return STICK_UL;
        else if (down)
            return STICK_LL;
        else
            return STICK_LEFT;
    } else {
        if (up)
            return STICK_FORWARD;
        else if (down)
            return STICK_BACK;
        else
            return STICK_CENTRE;
    }

    return(stick);
}


/*------------------------------------------------------------------------------
*  SDL_Atari_PORT - Read the status of the four joystick ports.
*-----------------------------------------------------------------------------*/
void SDL_Atari_PORT(Uint8 * s0, Uint8 * s1, Uint8 *s2, Uint8 *s3)
{
    int stick[NUM_JOYSTICKS];
    static int last_stick[NUM_JOYSTICKS] =  {STICK_CENTRE, STICK_CENTRE, STICK_CENTRE, STICK_CENTRE};
    int i;
    int joy0_x_pot, joy0_y_pot, joy1_x_pot, joy1_y_pot;
    int joy2_x_pot, joy2_y_pot, joy3_x_pot, joy3_y_pot;

    if ((joystick0 != NULL) || (joystick1 != NULL) || (joystick2 != NULL) || (joystick3 != NULL)) // can only joystick1!=NULL ?
        {
        SDL_JoystickUpdate();
        }

    /* Get the basic joystick info, from either the keyboard or HID joysticks */
    for (i=0;i<NUM_JOYSTICKS;i++) {
        stick[i] = STICK_CENTRE;
            
        switch(JOYSTICK_MODE[i]) {
            case NO_JOYSTICK:
                break;
            case KEYPAD_JOYSTICK:
                if (kbhits[SDL_JOY_0_LEFT])
                    stick[i] = STICK_LEFT;
                if (kbhits[SDL_JOY_0_RIGHT])
                    stick[i] = STICK_RIGHT;
                if (kbhits[SDL_JOY_0_UP])
                    stick[i] = STICK_FORWARD;
                if (kbhits[SDL_JOY_0_DOWN])
                    stick[i] = STICK_BACK;
                if ((kbhits[SDL_JOY_0_LEFTUP])
                    || ((kbhits[SDL_JOY_0_LEFT]) && (kbhits[SDL_JOY_0_UP])))
                    stick[i] = STICK_UL;
                if ((kbhits[SDL_JOY_0_LEFTDOWN])
                    || ((kbhits[SDL_JOY_0_LEFT]) && (kbhits[SDL_JOY_0_DOWN])))
                    stick[i] = STICK_LL;
                if ((kbhits[SDL_JOY_0_RIGHTUP])
                    || ((kbhits[SDL_JOY_0_RIGHT]) && (kbhits[SDL_JOY_0_UP])))
                    stick[i] = STICK_UR;
                if ((kbhits[SDL_JOY_0_RIGHTDOWN])
                    || ((kbhits[SDL_JOY_0_RIGHT]) && (kbhits[SDL_JOY_0_DOWN])))
                    stick[i] = STICK_LR;
                break;
            case USERDEF_JOYSTICK:
                if (kbhits[SDL_JOY_1_LEFT])
                    stick[i] = STICK_LEFT;
                if (kbhits[SDL_JOY_1_RIGHT])
                    stick[i] = STICK_RIGHT;
                if (kbhits[SDL_JOY_1_UP])
                    stick[i] = STICK_FORWARD;
                if (kbhits[SDL_JOY_1_DOWN])
                    stick[i] = STICK_BACK;
                if ((kbhits[SDL_JOY_1_LEFTUP])
                    || ((kbhits[SDL_JOY_1_LEFT]) && (kbhits[SDL_JOY_1_UP])))
                    stick[i] = STICK_UL;
                if ((kbhits[SDL_JOY_1_LEFTDOWN])
                    || ((kbhits[SDL_JOY_1_LEFT]) && (kbhits[SDL_JOY_1_DOWN])))
                    stick[i] = STICK_LL;
                if ((kbhits[SDL_JOY_1_RIGHTUP])
                    || ((kbhits[SDL_JOY_1_RIGHT]) && (kbhits[SDL_JOY_1_UP])))
                    stick[i] = STICK_UR;
                if ((kbhits[SDL_JOY_1_RIGHTDOWN])
                    || ((kbhits[SDL_JOY_1_RIGHT]) && (kbhits[SDL_JOY_1_DOWN])))
                    stick[i] = STICK_LR;
                break;
            case SDL0_JOYSTICK:
                if (joystickType0 == JOY_TYPE_STICK)
                    stick[i] = get_SDL_joystick_state(joystick0, joystick0Num, 
                                                        &joy0_x_pot, &joy0_y_pot);
                else if (joystickType0 == JOY_TYPE_HAT)
                    stick[i] = get_SDL_hat_state(joystick0, joystick0Num);
                else
                    stick[i] = get_SDL_joy_buttons_state(joystick0, joystick0_nbuttons);
                break;
            case SDL1_JOYSTICK:
                if (joystickType1 == JOY_TYPE_STICK)
                    stick[i] = get_SDL_joystick_state(joystick1, joystick1Num, 
                                                        &joy1_x_pot, &joy1_y_pot);
                else if (joystickType1 == JOY_TYPE_HAT)
                    stick[i] = get_SDL_hat_state(joystick1, joystick1Num);
                else
                    stick[i] = get_SDL_joy_buttons_state(joystick1, joystick1_nbuttons);
                break;
            case SDL2_JOYSTICK:
                if (joystickType2 == JOY_TYPE_STICK)
                    stick[i] = get_SDL_joystick_state(joystick2, joystick2Num,
                                                        &joy2_x_pot, &joy2_y_pot);
                else if (joystickType2 == JOY_TYPE_HAT)
                    stick[i] = get_SDL_hat_state(joystick2, joystick2Num);
                else
                    stick[i] = get_SDL_joy_buttons_state(joystick2, joystick2_nbuttons);
                break;
            case SDL3_JOYSTICK:
                if (joystickType3 == JOY_TYPE_STICK)
                    stick[i] = get_SDL_joystick_state(joystick3, joystick3Num,
                                                        &joy3_x_pot, &joy3_y_pot);
                else if (joystickType3 == JOY_TYPE_HAT)
                    stick[i] = get_SDL_hat_state(joystick3, joystick3Num);
                else
                    stick[i] = get_SDL_joy_buttons_state(joystick3, joystick3_nbuttons);
                break;
            }
        
        /* Make sure that left/right and up/down aren't pressed at the same time
            from keyboard */
        if (joy_block_opposite_directions) {
            if ((stick[i] & 0x0c) == 0) {   /* right and left simultaneously */
                if (last_stick[i] & 0x04)   /* if wasn't left before, move left */
                    stick[i] |= 0x08;
                else            /* else move right */
                stick[i] |= 0x04;
                }
            else {
                last_stick[i] &= 0x03;
                last_stick[i] |= stick[i] & 0x0c;
                }
            if ((stick[i] & 0x03) == 0) {   /* up and down simultaneously */
                if (last_stick[i] & 0x01)   /* if wasn't up before, move up */
                    stick[i] |= 0x02;
                else                        /* else move down */
                    stick[i] |= 0x01;
                }
            else {
                last_stick[i] &= 0x0c;
                last_stick[i] |= stick[i] & 0x03;
                }
            }
        else
            last_stick[i] = stick[i];
        }

    /* handle  SDL joysticks as paddles in non-5200 */
    if (machine_type != MACHINE_5200) {
        for (i=0;i<NUM_JOYSTICKS;i++) {
            if ((JOYSTICK_MODE[i] == SDL0_JOYSTICK) && 
                (joystickType0 == JOY_TYPE_STICK)){
                POT_input[2*i] = joy0_x_pot;
                POT_input[2*i+1] = joy0_y_pot;
                if (paddlesXAxisOnly)
                    POT_input[2*i+1] = joy0_x_pot;
                }
            else if ((JOYSTICK_MODE[i] == SDL1_JOYSTICK) && 
                     (joystickType1 == JOY_TYPE_STICK)) {
                POT_input[2*i] = joy1_x_pot;
                POT_input[2*i+1] = joy1_y_pot;
                if (paddlesXAxisOnly)
                    POT_input[2*i+1] = joy1_x_pot;
                }
            else if ((JOYSTICK_MODE[i] == SDL2_JOYSTICK) && 
                     (joystickType2 == JOY_TYPE_STICK)) {
                POT_input[2*i] = joy2_x_pot;
                POT_input[2*i+1] = joy2_y_pot;
                if (paddlesXAxisOnly)
                    POT_input[2*i+1] = joy2_x_pot;
                }
            else if ((JOYSTICK_MODE[i] == SDL3_JOYSTICK) && 
                     (joystickType3 == JOY_TYPE_STICK)) {
                POT_input[2*i] = joy3_x_pot;
                POT_input[2*i+1] = joy3_y_pot;
                if (paddlesXAxisOnly)
                    POT_input[2*i+1] = joy3_x_pot;
                }
            else {
                POT_input[2*i] = 228;
                POT_input[2*i+1] = 228;
                }
            }
        }
        
    /* handle analog joysticks in Atari 5200 */
    else {
        for (i = 0; i < 4; i++) {
            if ((JOYSTICK_MODE[i] == SDL0_JOYSTICK) && 
                (joystickType0 == JOY_TYPE_STICK)) {
                POT_input[2*i] = joy0_x_pot;
                POT_input[2*i+1] = joy0_y_pot;
                }
            else if ((JOYSTICK_MODE[i] == SDL1_JOYSTICK) && 
                     (joystickType0 == JOY_TYPE_STICK)) {
                POT_input[2*i] = joy1_x_pot;
                POT_input[2*i+1] = joy1_y_pot;
                }
            else if ((JOYSTICK_MODE[i] == SDL2_JOYSTICK) && 
                     (joystickType0 == JOY_TYPE_STICK)) {
                POT_input[2*i] = joy2_x_pot;
                POT_input[2*i+1] = joy2_y_pot;
                }
            else if ((JOYSTICK_MODE[i] == SDL3_JOYSTICK) && 
                     (joystickType0 == JOY_TYPE_STICK)) {
                POT_input[2*i] = joy3_x_pot;
                POT_input[2*i+1] = joy3_y_pot;
                }
            else {
                if ((stick[i] & (STICK_CENTRE ^ STICK_LEFT)) == 0) 
                    POT_input[2 * i] = joy_5200_min;
                else if ((stick[i] & (STICK_CENTRE ^ STICK_RIGHT)) == 0) 
                    POT_input[2 * i] = joy_5200_max;
                else 
                    POT_input[2 * i] = joy_5200_center;
                if ((stick[i] & (STICK_CENTRE ^ STICK_FORWARD)) == 0) 
                    POT_input[2 * i + 1] = joy_5200_min;
                else if ((stick[i] & (STICK_CENTRE ^ STICK_BACK)) == 0) 
                    POT_input[2 * i + 1] = joy_5200_max;
                else 
                    POT_input[2 * i + 1] = joy_5200_center;
                }
            }
        }

    *s0 = stick[0];
    *s1 = stick[1];
    *s2 = stick[2];
    *s3 = stick[3];
}

/*------------------------------------------------------------------------------
*  get_SDL_joystick_state - Return the states of one of the Attached HID 
*     joysticks buttons.
*-----------------------------------------------------------------------------*/
int get_SDL_button_state(SDL_Joystick *joystick, int numberOfButtons, int *key_code, 
                         int *key_shift, int *key_control, int *key_consol)
{
    int i;
    int trig = 1;
    int joyindex;
    
    joyindex = SDL_JoystickIndex(joystick);
    
    *key_code = AKEY_NONE;
    
    if (machine_type != MACHINE_5200) {
        for (i = 0; i < numberOfButtons; i++) {
            if (SDL_JoystickGetButton(joystick, i)) {
                switch(joystickButtonKey[joyindex][i]) {
                    case AKEY_BUTTON1:
                        trig = 0;
                        break;
                    case AKEY_SHFT:
                        *key_shift = 1;
                        break;
                    case AKEY_CTRL:
                        *key_control = AKEY_CTRL;
                        break;
                    case AKEY_BUTTON_OPTION:
                        *key_consol &= (~CONSOL_OPTION);
                        break;
                    case AKEY_BUTTON_SELECT:
                        *key_consol &= (~CONSOL_SELECT);
                        break;
                    case AKEY_BUTTON_START:
                        *key_consol &= (~CONSOL_START);
                        break;
                    case AKEY_JOYSTICK_UP:
                    case AKEY_JOYSTICK_DOWN:
                    case AKEY_JOYSTICK_LEFT:
                    case AKEY_JOYSTICK_RIGHT:
                        break;
                    default:
                        *key_code = joystickButtonKey[joyindex][i];
                    }
                }
            }
        }
    else {
        for (i = 0; i < numberOfButtons; i++) {
            if (SDL_JoystickGetButton(joystick, i)) {
                switch(joystick5200ButtonKey[joyindex][i]) {
                    case AKEY_BUTTON1:
                        trig = 0;
                        break;
                    case AKEY_JOYSTICK_UP:
                    case AKEY_JOYSTICK_DOWN:
                    case AKEY_JOYSTICK_LEFT:
                    case AKEY_JOYSTICK_RIGHT:
                        break;
                    default:
                        *key_code = joystick5200ButtonKey[joyindex][i];
                    }
                }
            }
        }

    return(trig);
}

/*------------------------------------------------------------------------------
*  SDL_Atari_TRIG - Read the status of the four joystick buttons.
*-----------------------------------------------------------------------------*/
int SDL_Atari_TRIG(Uint8 * t0, Uint8 * t1, Uint8 *t2, Uint8 *t3, 
                   int *key_shift, int *key_control, int *key_consol)
{
    int trig[NUM_JOYSTICKS],i;
    int key_code0 = AKEY_NONE;
    int key_code1 = AKEY_NONE;
    int key_code2 = AKEY_NONE;
    int key_code3 = AKEY_NONE;
    
    *key_shift = 0;
    *key_control = 0;
    *key_consol = CONSOL_NONE;
        
    for (i=0;i<NUM_JOYSTICKS;i++) {
        switch(JOYSTICK_MODE[i]) {
            case NO_JOYSTICK:
                trig[i] = 1;
                break;
            case KEYPAD_JOYSTICK:
                if ((kbhits[SDL_TRIG_0]) || (kbhits[SDL_TRIG_0_B]))
                    trig[i] = 0;
                else
                    trig[i] = 1;
                break;
            case USERDEF_JOYSTICK:
                if ((kbhits[SDL_TRIG_1]) || (kbhits[SDL_TRIG_1_B]))
                    trig[i] = 0;
                else
                    trig[i] = 1;
                break;
            case SDL0_JOYSTICK:
                trig[i] = get_SDL_button_state(joystick0, joystick0_nbuttons, 
                                               &key_code0, key_shift, key_control,
                                               key_consol);
                break;
            case SDL1_JOYSTICK:
                trig[i] = get_SDL_button_state(joystick1, joystick1_nbuttons,
                                               &key_code1, key_shift, key_control,
                                               key_consol);
                break;
            case SDL2_JOYSTICK:
                trig[i] = get_SDL_button_state(joystick2, joystick2_nbuttons,
                                               &key_code2, key_shift, key_control,
                                               key_consol);
                break;
            case SDL3_JOYSTICK:
                trig[i] = get_SDL_button_state(joystick3, joystick3_nbuttons,
                                               &key_code3, key_shift, key_control,
                                               key_consol);
                break;
            }
        if ((JOYSTICK_AUTOFIRE[i] == AUTOFIRE_FIRE && !trig[i]) || 
            (JOYSTICK_AUTOFIRE[i] == AUTOFIRE_CONT))
            trig[i] = (nframes & 2) ? 1 : 0;
        }
           
    *t0 = trig[0];
    *t1 = trig[1];
    *t2 = trig[2];
    *t3 = trig[3];

    if (key_code0 == AKEY_NONE) {
        if (key_code1 == AKEY_NONE) {
            if (key_code2 == AKEY_NONE) 
                return(key_code3);
            else
                return(key_code2);
            }
        else
            return(key_code1);
        }
    else
        return(key_code0);
}

/* mouse_step is used in Amiga, ST, trak-ball and joystick modes.
   It moves mouse_x and mouse_y in the direction given by
   mouse_move_x and mouse_move_y.
   Bresenham's algorithm is used:
   if (abs(deltaX) >= abs(deltaY)) {
     if (deltaX == 0)
       return;
     MoveHorizontally();
     e -= abs(gtaY);
     if (e < 0) {
       e += abs(deltaX);
       MoveVertically();
     }
   }
   else {
     MoveVertically();
     e -= abs(deltaX);
     if (e < 0) {
       e += abs(deltaY);
       MoveHorizontally();
     }
   }
   Returned is stick code for the movement direction.
*/
static UBYTE mouse_step(void)
{
    static int e = 0;
    UBYTE r = STICK_CENTRE;
    int dx = mouse_move_x >= 0 ? mouse_move_x : -mouse_move_x;
    int dy = mouse_move_y >= 0 ? mouse_move_y : -mouse_move_y;
    if (dx >= dy) {
        if (mouse_move_x == 0)
            return r;
        if (mouse_move_x < 0) {
            r &= STICK_LEFT;
            mouse_x--;
            mouse_move_x += 1 << MOUSE_SHIFT;
            if (mouse_move_x > 0)
                mouse_move_x = 0;
        }
        else {
            r &= STICK_RIGHT;
            mouse_x++;
            mouse_move_x -= 1 << MOUSE_SHIFT;
            if (mouse_move_x < 0)
                mouse_move_x = 0;
        }
        e -= dy;
        if (e < 0) {
            e += dx;
            if (mouse_move_y < 0) {
                r &= STICK_FORWARD;
                mouse_y--;
                mouse_move_y += 1 << MOUSE_SHIFT;
                if (mouse_move_y > 0)
                    mouse_move_y = 0;
            }
            else {
                r &= STICK_BACK;
                mouse_y++;
                mouse_move_y -= 1 << MOUSE_SHIFT;
                if (mouse_move_y < 0)
                    mouse_move_y = 0;
            }
        }
    }
    else {
        if (mouse_move_y < 0) {
            r &= STICK_FORWARD;
            mouse_y--;
            mouse_move_y += 1 << MOUSE_SHIFT;
            if (mouse_move_y > 0)
                mouse_move_y = 0;
        }
        else {
            r &= STICK_BACK;
            mouse_y++;
            mouse_move_y -= 1 << MOUSE_SHIFT;
            if (mouse_move_y < 0)
                mouse_move_y = 0;
        }
        e -= dx;
        if (e < 0) {
            e += dy;
            if (mouse_move_x < 0) {
                r &= STICK_LEFT;
                mouse_x--;
                mouse_move_x += 1 << MOUSE_SHIFT;
                if (mouse_move_x > 0)
                    mouse_move_x = 0;
            }
            else {
                r &= STICK_RIGHT;
                mouse_x++;
                mouse_move_x -= 1 << MOUSE_SHIFT;
                if (mouse_move_x < 0)
                    mouse_move_x = 0;
            }
        }
    }
    return r;
}

/*------------------------------------------------------------------------------
*  SDL_Atari_Mouse - Read the status the device emulated by the mouse, if there
*     is one.  Emulated devices include paddles, light pens, etc...  Most of this
*     was brought over from input.c.  I hated to duplicate it, but the author
*     of the SDL port indicated there were problems with Atari_Frame and 
*     INPUT_Frame.
*-----------------------------------------------------------------------------*/
void SDL_Atari_Mouse(Uint8 *s, Uint8 *t)
{
    int STICK = STICK_CENTRE;
    int TRIG = 1;
    int i;
    static int last_mouse_buttons = 0;
    int sdl_mouse_buttons;

    mouse_buttons = 0;
    sdl_mouse_buttons = SDL_GetRelativeMouseState(&mouse_delta_x, &mouse_delta_y);

    if (sdl_mouse_buttons & SDL_BUTTON(SDL_BUTTON_RIGHT))
        mouse_buttons |= 2;
    if (sdl_mouse_buttons & SDL_BUTTON(SDL_BUTTON_LEFT))
        mouse_buttons |= 1;
        
    /* handle mouse */
    switch (mouse_mode) {
    case MOUSE_OFF:
        break;
    case MOUSE_PAD:
    case MOUSE_TOUCH:
    case MOUSE_KOALA:
        if (mouse_mode != MOUSE_PAD || machine_type == MACHINE_5200)
            mouse_x += mouse_delta_x * mouse_speed;
        else
            mouse_x -= mouse_delta_x * mouse_speed;
        if (mouse_x < mouse_pot_min << MOUSE_SHIFT)
            mouse_x = mouse_pot_min << MOUSE_SHIFT;
        else if (mouse_x > mouse_pot_max << MOUSE_SHIFT)
            mouse_x = mouse_pot_max << MOUSE_SHIFT;
        if (mouse_mode == MOUSE_KOALA || machine_type == MACHINE_5200)
            mouse_y += mouse_delta_y * mouse_speed;
        else
            mouse_y -= mouse_delta_y * mouse_speed;
        if (mouse_y < mouse_pot_min << MOUSE_SHIFT)
            mouse_y = mouse_pot_min << MOUSE_SHIFT;
        else if (mouse_y > mouse_pot_max << MOUSE_SHIFT)
            mouse_y = mouse_pot_max << MOUSE_SHIFT;
        POT_input[mouse_port << 1] = mouse_x >> MOUSE_SHIFT;
        POT_input[(mouse_port << 1) + 1] = mouse_y >> MOUSE_SHIFT;
        if (paddlesXAxisOnly && (mouse_mode == MOUSE_PAD))
            POT_input[(mouse_port << 1) + 1] = mouse_x >> MOUSE_SHIFT;

        if (machine_type == MACHINE_5200) {
            if (mouse_buttons & 1)
                TRIG_input[mouse_port] = 0;
            if (mouse_buttons & 2)
                SKSTAT &= ~8;
        }
        else
            STICK &= ~(mouse_buttons << 2);
        break;
    case MOUSE_PEN:
    case MOUSE_GUN:
        mouse_x += mouse_delta_x * mouse_speed;
        if (mouse_x < 0)
            mouse_x = 0;
        else if (mouse_x > 167 << MOUSE_SHIFT)
            mouse_x = 167 << MOUSE_SHIFT;
        mouse_y += mouse_delta_y * mouse_speed;
        if (mouse_y < 0)
            mouse_y = 0;
        else if (mouse_y > 119 << MOUSE_SHIFT)
            mouse_y = 119 << MOUSE_SHIFT;
        PENH_input = 44 + mouse_pen_ofs_h + (mouse_x >> MOUSE_SHIFT);
        PENV_input = 4 + mouse_pen_ofs_v + (mouse_y >> MOUSE_SHIFT);
        if ((mouse_buttons & 1) == (mouse_mode == MOUSE_PEN))
            STICK &= ~1;
        if ((mouse_buttons & 2) && !(last_mouse_buttons & 2))
            mouse_pen_show_pointer = !mouse_pen_show_pointer;
        break;
    case MOUSE_AMIGA:
    case MOUSE_ST:
    case MOUSE_TRAK:
        mouse_move_x += (mouse_delta_x * mouse_speed) >> 1;
        mouse_move_y += (mouse_delta_y * mouse_speed) >> 1;

        /* i = max(abs(mouse_move_x), abs(mouse_move_y)); */
        i = mouse_move_x >= 0 ? mouse_move_x : -mouse_move_x;
        if (mouse_move_y > i)
            i = mouse_move_y;
        else if (-mouse_move_y > i)
            i = -mouse_move_y;

        {
            UBYTE stick = STICK_CENTRE;
            if (i > 0) {
                i += (1 << MOUSE_SHIFT) - 1;
                i >>= MOUSE_SHIFT;
                if (i > 50)
                    max_scanline_counter = scanline_counter = 5;
                else
                    max_scanline_counter = scanline_counter = max_ypos / i;
                stick = mouse_step();
            }
            if (mouse_mode == MOUSE_TRAK) {
                /* bit 3 toggles - vertical movement, bit 2 = 0 - up */
                /* bit 1 toggles - horizontal movement, bit 0 = 0 - left */
                STICK = ((mouse_y & 1) << 3) | ((stick & 1) << 2)
                        | ((mouse_x & 1) << 1) | ((stick & 4) >> 2);
            }
            else {
                STICK = (mouse_mode == MOUSE_AMIGA ? mouse_amiga_codes : mouse_st_codes)
                            [(mouse_y & 3) * 4 + (mouse_x & 3)];
            }
        }

        if (mouse_buttons & 1)
            TRIG_input[mouse_port] = 0;
        if (mouse_buttons & 2)
            POT_input[mouse_port << 1] = 1;
        break;
    case MOUSE_JOY:
        mouse_move_x += mouse_delta_x * mouse_speed;
        mouse_move_y += mouse_delta_y * mouse_speed;
        if (mouse_move_x < -mouse_joy_inertia << MOUSE_SHIFT ||
            mouse_move_x > mouse_joy_inertia << MOUSE_SHIFT ||
            mouse_move_y < -mouse_joy_inertia << MOUSE_SHIFT ||
            mouse_move_y > mouse_joy_inertia << MOUSE_SHIFT) {
               mouse_move_x >>= 1;
               mouse_move_y >>= 1;
            }
        STICK &= mouse_step();
        if (machine_type == MACHINE_5200) {
            if ((STICK & (STICK_CENTRE ^ STICK_LEFT)) == 0)
                POT_input[(mouse_port << 1)] = joy_5200_min;
            else if ((STICK & (STICK_CENTRE ^ STICK_RIGHT)) == 0)
                POT_input[(mouse_port << 1)] = joy_5200_max;
            else
                POT_input[(mouse_port << 1)] = joy_5200_center;
            if ((STICK & (STICK_CENTRE ^ STICK_FORWARD)) == 0)
                POT_input[(mouse_port << 1) + 1] = joy_5200_min;
            else if ((STICK & (STICK_CENTRE ^ STICK_BACK)) == 0)
                POT_input[(mouse_port << 1) + 1] = joy_5200_max;
            else
                POT_input[(mouse_port << 1) + 1] = joy_5200_center;
            }
        if (mouse_buttons & 1)
            TRIG_input[mouse_port] = 0;
        break;
    default:
        break;
    }
    last_mouse_buttons = mouse_buttons;
        
    *s = STICK;
    *t = TRIG;
        
}

/*  Three blank functions that were defined in Input.c, but are replaced elsewhere
     now */
void INPUT_Frame()
{
}

void INPUT_DrawMousePointer()
{
}

void INPUT_Initialise(int *argc, char *argv[])
{
}

int joy_multijoy = 0;
static int joy_multijoy_no = 0;	/* number of selected joy */

void INPUT_SelectMultiJoy(int no)
{
	no &= 3;
	joy_multijoy_no = no;
	if (joy_multijoy && machine_type != MACHINE_5200) {
		PORT_input[0] = 0xf0 | STICK[no];
		TRIG[0] = TRIG_input[no];
	}
}


/*------------------------------------------------------------------------------
*  INPUT_Scanline - Handle per scanline processing for trackballs.  This was
*    brought from Input.c.
*-----------------------------------------------------------------------------*/
void INPUT_Scanline(void)
{
    if (mouse_mode == MOUSE_TRAK || mouse_mode == MOUSE_AMIGA || mouse_mode == MOUSE_ST) {
    if (--scanline_counter == 0) {
        UBYTE stick = mouse_step();
        if (mouse_mode == MOUSE_TRAK) {
            /* bit 3 toggles - vertical movement, bit 2 = 0 - up */
            /* bit 1 toggles - horizontal movement, bit 0 = 0 - left */
            STICK[mouse_port] = ((mouse_y & 1) << 3) | ((stick & 1) << 2)
                                | ((mouse_x & 1) << 1) | ((stick & 4) >> 2);
        }
        else {
            STICK[mouse_port] = (mouse_mode == MOUSE_AMIGA ? mouse_amiga_codes : mouse_st_codes)
                                [(mouse_y & 3) * 4 + (mouse_x & 3)];
        }
        PORT_input[0] = (STICK[1] << 4) | STICK[0];
        PORT_input[1] = (STICK[3] << 4) | STICK[2];
        scanline_counter = max_scanline_counter;
    }
    }
}

/*------------------------------------------------------------------------------
*  SDL_CenterMousePointer - Center the mouse pointer on the emulated 
*     screen.
*-----------------------------------------------------------------------------*/
void SDL_CenterMousePointer(void)
{
    switch (mouse_mode) {
    case MOUSE_PAD:
    case MOUSE_TOUCH:
    case MOUSE_KOALA:
        mouse_x = 114 << MOUSE_SHIFT;
        mouse_y = 114 << MOUSE_SHIFT;
        break;
    case MOUSE_PEN:
    case MOUSE_GUN:
        mouse_x = 84 << MOUSE_SHIFT;
        mouse_y = 60 << MOUSE_SHIFT;
        break;
    case MOUSE_AMIGA:
    case MOUSE_ST:
    case MOUSE_TRAK:
    case MOUSE_JOY:
        mouse_move_x = 0;
        mouse_move_y = 0;
        break;
    }
}

#define PLOT(dx, dy)    do {\
                            ptr[(dx) + ATARI_WIDTH * (dy)] ^= 0x0f0f;\
                            ptr[(dx) + ATARI_WIDTH * (dy) + ATARI_WIDTH / 2] ^= 0x0f0f;\
                        } while (0)

/*------------------------------------------------------------------------------
*  SDL_DrawMousePointer - Draws the mouse pointer in light pen/gun modes.
*-----------------------------------------------------------------------------*/
void SDL_DrawMousePointer(void)
{
    if ((mouse_mode == MOUSE_PEN || mouse_mode == MOUSE_GUN) && mouse_pen_show_pointer) {
        int x = mouse_x >> MOUSE_SHIFT;
        int y = mouse_y >> MOUSE_SHIFT;
        if (x >= 0 && x <= 167 && y >= 0 && y <= 119) {
            UWORD *ptr = & ((UWORD *) atari_screen)[12 + x + ATARI_WIDTH * y];
                        if (x >= 1) {
                            PLOT(-1, 0);
                            if (x>= 2)
                                PLOT(-2, 0);
                        }
                        if (x <= 166) {
                            PLOT(1, 0);
                            if (x<=165)
                                PLOT(2, 0);
                            }
            if (y >= 1) {
                PLOT(0, -1);
                if (y >= 2)
                    PLOT(0, -2);
            }
            if (y <= 118) {
                PLOT(0, 1);
                if (y <= 117)
                    PLOT(0, 2);
            }
        }
    }
}

/*------------------------------------------------------------------------------
*  CountFPS - Displays the Frames per second in windowed mode.
*-----------------------------------------------------------------------------*/
void CountFPS()
{
    static int ticks1 = 0, ticks2, shortframes;
        char title[40];
        char count[20];
        
    if (!FULLSCREEN && SHOWFPS) {    
        if (ticks1 == 0)
            ticks1 = SDL_GetTicks();
        ticks2 = SDL_GetTicks();
        shortframes++;
        if (ticks2 - ticks1 > 1000) {
            ticks1 = ticks2;
            strcpy(title,windowCaption);
            sprintf(count," - %3d fps",shortframes);
            strcat(title,count);
            SDL_WM_SetCaption(title,NULL);
            shortframes = 0;
            }
        }
}

/*------------------------------------------------------------------------------
*  ProcessMacMenus - Handle requested changes do to menu operations in the
*      Objective-C Cocoa code.
*-----------------------------------------------------------------------------*/
void ProcessMacMenus()
{
        
    if (requestDoubleSizeChange) {
        SwitchDoubleSize();
        requestDoubleSizeChange = 0;
        }
    if (requestFullScreenChange) {
         SwitchFullscreen();
         requestFullScreenChange = 0;
         }
    if (requestWidthModeChange) {
        SwitchWidth();
        requestWidthModeChange = 0;
        }                
    if (requestScreenshot) {
        Save_PCX_file(FALSE, Find_PCX_name());
        requestScreenshot = 0;
        }                
    if (requestInterlacedScreenshot) {
        Save_PCX_file(TRUE, Find_PCX_name());
        requestInterlacedScreenshot = 0;
        }                
    if (requestGrabMouse) {
        SwitchGrabMouse();
        requestGrabMouse = 0;
        }
    if (requestFpsChange) {
        SwitchShowFps();
        requestFpsChange = 0;
        }
    if (requestSoundEnabledChange) {
         sound_enabled = 1 - sound_enabled;
         requestSoundEnabledChange = 0;
         }
    if (requestSoundStereoChange) {
         stereo_enabled = 1 - stereo_enabled;
         requestSoundStereoChange = 0;
         }
    if (requestSoundRecordingChange) {
         SDL_Sound_Recording();
         requestSoundRecordingChange = 0;
         }
    if (requestLimitChange) {
         if (speed_limit == 1) 
            deltatime = 0;
         else
            deltatime = real_deltatime;
         speed_limit = 1 - speed_limit;
         requestLimitChange = 0;
         SetControlManagerLimit(speed_limit);
         }
    if (requestPauseEmulator) {
         pauseEmulator = 1-pauseEmulator;
         printf("Processing Pause request %d\n",pauseEmulator);
         if (pauseEmulator) 
            PauseAudio(1);
         else
            PauseAudio(0);
         requestPauseEmulator = 0;
         }
    if (requestColdReset) {
        Coldstart();
        requestColdReset = 0;
        }
    if (requestWarmReset) {
        Warmstart();
        requestWarmReset = 0;
        }
    if (requestSaveState) {
        SaveAtariState(saveFilename, "wb", TRUE);
        requestSaveState = 0;
        }
    if (requestLoadState) {
        ReadAtariState(loadFilename, "rb");
        requestLoadState = 0;
        }
    if (requestCaptionChange) {
        CreateWindowCaption();
        SDL_WM_SetCaption(windowCaption,NULL);
        requestCaptionChange = 0;
        }
}

/*------------------------------------------------------------------------------
*  ProcessMacPrefsChange - Handle requested changes do to preference window operations
*      in the Objective-C Cocoa code.
*-----------------------------------------------------------------------------*/
void ProcessMacPrefsChange()
{
    int retVal;
    
    if (requestPrefsChange) {
        CalculatePrefsChanged();
        RtConfigLoad(NULL);
        if (displaySizeChanged)
            {
            SetNewVideoMode(our_width, our_height,
            MainScreen->format->BitsPerPixel);
            Atari_DisplayScreen((UBYTE *) atari_screen);
            }
        if (showfpsChanged)
            {
            SHOWFPS = 1 - SHOWFPS;
            SwitchShowFps();
            }
        if (artifChanged)
            {
            ANTIC_UpdateArtifacting();
            }
        if (paletteChanged)
            {
            if (useBuiltinPalette) {
                Palette_Generate(paletteBlack, paletteWhite, paletteIntensity, paletteColorShift);
                Palette_Format(paletteBlack, paletteWhite, paletteIntensity);
                }
            else {
                retVal = read_palette(paletteFilename, adjustPalette);
                if (!retVal)
                    Palette_Generate(paletteBlack, paletteWhite, paletteIntensity, paletteColorShift);
                if (!retVal || adjustPalette)
                    Palette_Format(paletteBlack, paletteWhite, paletteIntensity);
                }
            CalcPalette();
            SetPalette();
            /* Clear the alternate page, so the first redraw is entire screen */
            if (atari_screen)
                memset(atari_screen, 0, (ATARI_HEIGHT * ATARI_WIDTH));
            }
        if (machineTypeChanged)
            {
            memset(atari_screen_b, 0, (ATARI_HEIGHT * ATARI_WIDTH));
            Atari800_InitialiseMachine();
            CreateWindowCaption();
            SDL_WM_SetCaption(windowCaption,NULL);
            if (machine_type == MACHINE_5200) 
                Atari_DisplayScreen((UBYTE *) atari_screen_b);
            }
        if (patchFlagsChanged)
            {
            Atari800_UpdatePatches();
            Coldstart();
            }
        if (keyboardJoystickChanged)
            Init_SDL_Joykeys();
        if (hardDiskChanged)
            Device_HHINIT();
        if (hifiSoundChanged) {
            Pokey_DoInit();
            Coldstart();
            }

    if (tv_mode == TV_PAL)
            deltatime = (1.0 / 50.0);
    else
            deltatime = (1.0 / 60.0);
            
    SetSoundManagerEnable(sound_enabled);
    SetSoundManagerStereo(stereo_enabled);
    SetSoundManagerRecording(IsSoundFileOpen());
    SetDisplayManagerDoubleSize(DOUBLESIZE);
    SetDisplayManagerWidthMode(WIDTH_MODE);
    SetDisplayManagerFps(SHOWFPS);
    SetDisplayManagerGrabMouse((mouse_mode != MOUSE_OFF));
    real_deltatime = deltatime;
    SetControlManagerLimit(speed_limit);
    UpdateMediaManagerInfo();
    Check_SDL_Joysticks();
    Remove_Bad_SDL_Joysticks();
    Init_SDL_Joykeys();
    if (speed_limit == 0)
        deltatime = 0;
        requestPrefsChange = 0;
        }
                    
}

/*------------------------------------------------------------------------------
* CreateWindowCaption - Change the window title in windowed mode based on type
*     of emulated machine.
*-----------------------------------------------------------------------------*/
void CreateWindowCaption()
{
    char *machineType;
    
    switch(machine_type) {
        case MACHINE_OSA:
            machineType = "OSA";
            break;
        case MACHINE_OSB:
            machineType = "OSB";
            break;
        case MACHINE_XLXE:
            machineType = "XL/XE";
            break;
        case MACHINE_5200:
            machineType = "5200";
            break;
        default:
            machineType = "";
            break;
        }
        
    if (ram_size == RAM_320_RAMBO)
        sprintf(windowCaption, "%s 320K RAMBO", machineType);
    else if (ram_size == RAM_320_COMPY_SHOP)
        sprintf(windowCaption, "%s 320K COMPY SHOP", machineType);
    else
        sprintf(windowCaption, "%s %dK", machineType, ram_size); 
}

/*------------------------------------------------------------------------------
* main - main function of emulator with main execution loop.
*-----------------------------------------------------------------------------*/
int main(int argc, char **argv)
{
    int keycode;
    int done = 0;
    int i;
    int retVal;

    stereo_enabled = FALSE; /* Turn this off here....otherwise games only come
                             from one channel...you only want this for demos mainly
                             anyway */
    
    argc = prefsArgc;
    argv = prefsArgv;

    if (!Atari800_Initialise(&argc, argv))
        return 3;

    CreateWindowCaption();
    SDL_WM_SetCaption(windowCaption,NULL);
    
    if (useBuiltinPalette) {
        Palette_Generate(paletteBlack, paletteWhite, paletteIntensity, paletteColorShift);
        Palette_Format(paletteBlack, paletteWhite, paletteIntensity);
        }
    else {
        retVal = read_palette(paletteFilename, adjustPalette);
        if (!retVal)
            Palette_Generate(paletteBlack, paletteWhite, paletteIntensity, paletteColorShift);
        if (!retVal || adjustPalette)
            Palette_Format(paletteBlack, paletteWhite, paletteIntensity);
        }
    CalcPalette();
    SetPalette();

    SetSoundManagerEnable(sound_enabled);
    SetSoundManagerStereo(stereo_enabled);
    SetSoundManagerRecording(IsSoundFileOpen());
    SetDisplayManagerDoubleSize(DOUBLESIZE);
    SetDisplayManagerWidthMode(WIDTH_MODE);
    SetDisplayManagerFps(SHOWFPS);
    SetDisplayManagerGrabMouse((mouse_mode != MOUSE_OFF));
    real_deltatime = deltatime;
    SetControlManagerLimit(speed_limit);
    UpdateMediaManagerInfo();
    if (speed_limit == 0)
        deltatime = 0;

    /* Clear the alternate page, so the first redraw is entire screen */
    memset(atari_screen_b, 0, (ATARI_HEIGHT * ATARI_WIDTH));

    SDL_PauseAudio(0);
    SDL_CenterMousePointer();
    SDL_JoystickEventState(SDL_IGNORE);

    if (fileToLoad) 
        SDLMainLoadStartupFile();

    /* First time thru, get the keystate since we will be doing joysticks first */
    kbhits = SDL_GetKeyState(NULL);

    while (!done) {
        /* Handle joysticks and paddles */
        SDL_Atari_PORT(&STICK[0], &STICK[1], &STICK[2], &STICK[3]);
        keycode = SDL_Atari_TRIG(&TRIG_input[0], &TRIG_input[1], &TRIG_input[2], &TRIG_input[3],
                                 &pad_key_shift, &pad_key_control, &pad_key_consol);
        for (i=0;i<4;i++) {
            if (JOYSTICK_MODE[i] == MOUSE_EMULATION)
                break;
            }
        if ((i<4) && (mouse_mode != MOUSE_OFF))
            SDL_Atari_Mouse(&STICK[i],&TRIG_input[i]);

        key_consol = CONSOL_NONE;
        switch(keycode) {
            case AKEY_NONE:
                keycode = Atari_Keyboard();
                break;
            case AKEY_BUTTON2:
                key_shift = 1;
                keycode = AKEY_NONE;
                break;
            case AKEY_5200_RESET:
                if (machine_type == MACHINE_5200)
                    keycode = AKEY_WARMSTART;
                break;
            case AKEY_BUTTON_RESET:
                if (machine_type != MACHINE_5200)
                    keycode = AKEY_WARMSTART;
                break;
            default:
                Atari_Keyboard(); /* Get the keyboard state (for shift and control), but 
                                      throw away the key, since the keypad key takes precedence */
                if (key_shift)
                    keycode |= AKEY_SHFT;
                if (CONTROL)
                    keycode |= AKEY_CTRL;
            }

        switch (keycode) {
        case AKEY_EXIT:
            done = 1;
            break;
        case AKEY_COLDSTART:
            Coldstart();
            break;
        case AKEY_WARMSTART:
            Warmstart();
            break;
        case AKEY_UI:
            if (FULLSCREEN) {
                PauseAudio(1);
                ui((UBYTE *) atari_screen);
                PauseAudio(0);
                }
            break;
        case AKEY_SCREENSHOT:
            Save_PCX_file(FALSE, Find_PCX_name());
            break;
        case AKEY_SCREENSHOT_INTERLACE:
            Save_PCX_file(TRUE, Find_PCX_name());
            break;
        case AKEY_BREAK:
            key_break = 1;
            if (!last_key_break) {
                if (IRQEN & 0x80) {
                    IRQST &= ~0x80;
                    GenerateIRQ();
                }
                break;
        default:
                key_break = 0;
                key_code = keycode;
                break;
            }
        }
        
        if (machine_type == MACHINE_5200) {
            if (key_shift & !last_key_break) {
		if (IRQEN & 0x80) {
		    IRQST &= ~0x80;
                    GenerateIRQ();
                    }
                }
             last_key_break = key_shift;
           }
        else
            last_key_break = key_break;

        key_code = key_code | CONTROL;
        
        SKSTAT |= 0xc;
        if (key_shift)
            SKSTAT &= ~8;
        if (key_code == AKEY_NONE)
            last_key_code = AKEY_NONE;
        if (key_code != AKEY_NONE) {
            SKSTAT &= ~4;
            if ((key_code ^ last_key_code) & ~AKEY_SHFTCTRL) {
                last_key_code = key_code;
                KBCODE = (UBYTE) key_code;
                if (IRQEN & 0x40) {
                    IRQST &= ~0x40;
                    GenerateIRQ();
                }
		else {
                    /* keyboard over-run */
                    SKSTAT &= ~0x40;
                    /* assert(IRQ != 0); */
		}
            }
        }

	if (joy_multijoy && machine_type != MACHINE_5200) {
		PORT_input[0] = 0xf0 | STICK[joy_multijoy_no];
		PORT_input[1] = 0xff;
		TRIG[0] = TRIG_input[joy_multijoy_no];
		TRIG[2] = TRIG[1] = 1;
		TRIG[3] = machine_type == MACHINE_XLXE ? cartA0BF_enabled : 1;
	}
	else {
		TRIG[0] = TRIG_input[0];
		TRIG[1] = TRIG_input[1];
		if (machine_type == MACHINE_XLXE) {
			TRIG[2] = 1;
			TRIG[3] = cartA0BF_enabled;
		}
		else {
			TRIG[2] = TRIG_input[2];
			TRIG[3] = TRIG_input[3];
		}
		PORT_input[0] = (STICK[1] << 4) | STICK[0];
		PORT_input[1] = (STICK[3] << 4) | STICK[2];
	}

        /* switch between screens to enable delta output */
        if (atari_screen==atari_screen1) {
            atari_screen = atari_screen2;
            atari_screen_b = atari_screen1;
        }
        else {
            atari_screen = atari_screen1;
            atari_screen_b = atari_screen2;
        }

        ProcessMacMenus();
                
        /* If emulator isn't paused, and 5200 has a cartridge */
        if (!pauseEmulator && !((machine_type == MACHINE_5200) && (cart_type == CART_NONE))) {
            Device_Frame();
            GTIA_Frame();
            ANTIC_Frame(TRUE);
            LED_Frame();
            SDL_DrawMousePointer();
            POKEY_Frame();
            nframes++;
            atari_sync();
            Atari_DisplayScreen((UBYTE *) atari_screen);
            CountFPS();
            }
        else if ((machine_type == MACHINE_5200) && (cart_type == CART_NONE)){
            /* Clear the screen if we are in 5200 mode, with no cartridge */
            InitializeUI();
            memset(atari_screen, 0, (ATARI_HEIGHT * ATARI_WIDTH));
            ClearScreen((UBYTE *) atari_screen_b);
            CenterPrint((UBYTE *) atari_screen_b, 0x9e, 0x94, "Atari 5200 Emulator", 11);
            CenterPrint((UBYTE *) atari_screen_b, 0x9e, 0x94, "Please Insert Cartridge", 12);
            Atari_DisplayScreen((UBYTE *) atari_screen_b);
            }
        else
            usleep(100000);

        ProcessMacPrefsChange();
            
        if (requestQuit)
            done = TRUE;
        }
    Atari800_Exit(FALSE);
    Aflushlog();
    return 0;
}

