/* 
 * File: gtrackinfo.c
 *
 * Copyright (c) 2005 Johannes Heimansberg
 *
 * Released under the GNU General Public License v2
 */

#include <gtk/gtk.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtktogglebutton.h>
#include "gtrackinfo.h"
#include "trackinfo.h"

static void gtrackinfo_class_init(GTrackinfoClass *klass);
static void gtrackinfo_init(GTrackinfo *gti);

GType gtrackinfo_get_type(void)
{
	static GType gti_type = 0;

	if (!gti_type) {
		static const GTypeInfo gti_info =
		{
			sizeof (GTrackinfoClass),
			NULL, /* base_init */
			NULL, /* base_finalize */
			(GClassInitFunc) gtrackinfo_class_init,
			NULL, /* class_finalize */
			NULL, /* class_data */
        	sizeof (GTrackinfo),
			0,
			(GInstanceInitFunc) gtrackinfo_init,
		};

		gti_type = g_type_register_static (GTK_TYPE_DIALOG, 
		                                   "GTrackinfo", &gti_info, 0);
	}
	return gti_type;
}

static void gtrackinfo_class_init(GTrackinfoClass *klass)
{
}

static void cb_close_trackinfo_dialog(GtkWidget *widget, gpointer pt)
{
	GtkWidget *gti = (GtkWidget *)pt;
	gtk_widget_destroy(gti);
}

static void gtrackinfo_init(GTrackinfo *gti)
{
	gtk_window_set_title(GTK_WINDOW(gti), "File details");

	gti->hbox = gtk_hbox_new(FALSE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(gti->hbox), 8);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(gti)->vbox),
		                   gti->hbox, FALSE, FALSE, 0);
		
	gti->table = gtk_table_new(3, 7, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(gti->table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(gti->table), 4);
	gtk_box_pack_start(GTK_BOX(gti->hbox), gti->table, TRUE, TRUE, 0);

	/* ... */
	gti->label_title = gtk_label_new("Title:");
	gtk_table_attach(GTK_TABLE (gti->table),
		             gti->label_title,
		             0, 1, 0, 1,
	                 GTK_SHRINK, GTK_SHRINK, 0, 0);

	gti->entry_title = gtk_entry_new();
	gtk_entry_set_editable(GTK_ENTRY(gti->entry_title), FALSE);

	gtk_table_attach_defaults(GTK_TABLE(gti->table), 
	                          gti->entry_title, 
	                          1, 2, 0, 1);

	/* ... */
	gti->label_artist = gtk_label_new("Artist:");
	gtk_table_attach(GTK_TABLE(gti->table), 
	                 gti->label_artist, 
	                 0, 1, 1, 2,
	                 GTK_SHRINK, GTK_SHRINK, 0, 0);

	gti->entry_artist = gtk_entry_new();
	gtk_entry_set_editable(GTK_ENTRY(gti->entry_artist), FALSE);
	gtk_table_attach_defaults(GTK_TABLE(gti->table), 
	                          gti->entry_artist, 
	                          1, 2, 1, 2);

	/* ... */
	gti->label_album = gtk_label_new("Album:");
	gtk_table_attach(GTK_TABLE(gti->table), 
	                 gti->label_album, 
	                 0, 1, 2, 3,
	                 GTK_SHRINK, GTK_SHRINK, 0, 0);

	gti->entry_album = gtk_entry_new();
	gtk_entry_set_editable(GTK_ENTRY(gti->entry_album), FALSE);
	gtk_table_attach_defaults(GTK_TABLE(gti->table), 
	                          gti->entry_album, 
	                          1, 2, 2, 3);

	/* ... */
	gti->label_date = gtk_label_new("Date:");
	gtk_table_attach(GTK_TABLE(gti->table), 
	                 gti->label_date, 
	                 0, 1, 3, 4,
	                 GTK_SHRINK, GTK_SHRINK, 0, 0);

	gti->entry_date = gtk_entry_new();
	gtk_entry_set_editable(GTK_ENTRY(gti->entry_date), FALSE);
	gtk_table_attach_defaults(GTK_TABLE(gti->table), 
	                          gti->entry_date, 
	                          1, 2, 3, 4);

	/* ... */
	gti->label_comment = gtk_label_new("Comment:");
	gtk_table_attach(GTK_TABLE(gti->table), 
	                 gti->label_comment, 
	                 0, 1, 4, 5,
	                 GTK_SHRINK, GTK_SHRINK, 0, 0);

	gti->entry_comment = gtk_entry_new();
	gtk_entry_set_editable(GTK_ENTRY(gti->entry_comment), FALSE);
	gtk_table_attach_defaults(GTK_TABLE(gti->table), 
	                          gti->entry_comment, 
	                          1, 2, 4, 5);

	/* ... */
	gti->label_genre = gtk_label_new("Genre:");
	gtk_table_attach(GTK_TABLE(gti->table), 
	                 gti->label_genre, 
	                 0, 1, 5, 6,
	                 GTK_SHRINK, GTK_SHRINK, 0, 0);

	gti->entry_genre = gtk_entry_new();
	gtk_entry_set_editable(GTK_ENTRY(gti->entry_genre), FALSE);
	gtk_table_attach_defaults(GTK_TABLE(gti->table), 
	                          gti->entry_genre, 
	                          1, 2, 5, 6);

	/* ... */
	gti->label_url = gtk_label_new("URL:");
	gtk_table_attach(GTK_TABLE (gti->table),
		             gti->label_url,
		             0, 1, 6, 7,
	                 GTK_SHRINK, GTK_SHRINK, 0, 0);

	gti->entry_url = gtk_entry_new();
	gtk_entry_set_editable(GTK_ENTRY(gti->entry_url), FALSE);
	gtk_table_attach_defaults(GTK_TABLE (gti->table),
		                      gti->entry_url,
		                      1, 3, 6, 7);

	/* ... */
	gti->frame = gtk_frame_new("File details");
	gtk_widget_set_size_request(gti->frame, 175, 100);
	gtk_table_attach(GTK_TABLE(gti->table), 
	                 gti->frame, 
	                 2, 3, 0, 6,
	                 GTK_SHRINK, GTK_EXPAND | GTK_FILL, 0, 0);

	gti->frame_vbox = gtk_vbox_new(4, FALSE);
	gtk_container_add(GTK_CONTAINER(gti->frame), gti->frame_vbox);

	gti->label_length = gtk_label_new("Length: unknown");
	gtk_container_add(GTK_CONTAINER(gti->frame_vbox), gti->label_length);
	gti->label_filetype = gtk_label_new("Filetype: unknown");
	gtk_container_add(GTK_CONTAINER(gti->frame_vbox), gti->label_filetype);
	gti->label_filesize = gtk_label_new("File size: N/A");
	gtk_container_add(GTK_CONTAINER(gti->frame_vbox), gti->label_filesize);
	gti->label_bitrate = gtk_label_new("Bit rate: unknown");
	gtk_container_add(GTK_CONTAINER(gti->frame_vbox), gti->label_bitrate);
	gti->label_samplerate = gtk_label_new("Sample rate: unknown");
	gtk_container_add(GTK_CONTAINER(gti->frame_vbox), gti->label_samplerate);
	gti->label_id = gtk_label_new("ID: unknown");
	gtk_container_add(GTK_CONTAINER(gti->frame_vbox), gti->label_id);

	/* ok button */
	gti->ok_button = gtk_button_new_from_stock(GTK_STOCK_OK);
	gtk_box_pack_start(GTK_BOX (GTK_DIALOG (gti)->action_area),
                       gti->ok_button, TRUE, TRUE, 0);
	g_signal_connect(G_OBJECT(gti->ok_button), "clicked",
	                 G_CALLBACK(cb_close_trackinfo_dialog), (gpointer)gti);

	gtk_widget_show_all(GTK_WIDGET(gti));
}

gboolean gtrackinfo_set_info(GTrackinfo *gti, trackinfo *info)
{
	char   tmp[256];
	gchar *bitrate, *samplerate, *playtime;

	gtk_entry_set_text(GTK_ENTRY(gti->entry_title),
	                   trackinfo_get_title_str(info));
	gtk_entry_set_text(GTK_ENTRY(gti->entry_artist),
	                   trackinfo_get_artist_str(info));
	gtk_entry_set_text(GTK_ENTRY(gti->entry_album),
	                   trackinfo_get_album_str(info));
	gtk_entry_set_text(GTK_ENTRY(gti->entry_date),
	                   trackinfo_get_date_str(info));
	gtk_entry_set_text(GTK_ENTRY(gti->entry_url),
	                   trackinfo_get_url_str(info));
	gtk_entry_set_text(GTK_ENTRY(gti->entry_comment),
	                   trackinfo_get_comment_str(info));
	gtk_entry_set_text(GTK_ENTRY(gti->entry_genre),
	                   trackinfo_get_genre_str(info));

	playtime = trackinfo_get_playtime_str_alloc(info);
	g_snprintf(tmp, 254, "Length: %s", playtime);
	g_free(playtime);
	gtk_label_set_text(GTK_LABEL(gti->label_length), tmp);

	bitrate = trackinfo_get_bitrate_str_alloc(info);
	if (trackinfo_is_vbr(info) == FALSE)
		g_snprintf(tmp, 254, "Bit rate: %s", bitrate);
	else
		g_snprintf(tmp, 254, "Avg. Bit rate: %s (VBR)", bitrate);
	g_free(bitrate);
	gtk_label_set_text(GTK_LABEL(gti->label_bitrate), tmp);

	samplerate = trackinfo_get_samplerate_str_alloc(info);
	g_snprintf(tmp, 254, "Sample rate: %s", samplerate);
	g_free(samplerate);
	gtk_label_set_text(GTK_LABEL(gti->label_samplerate), tmp);

	g_snprintf(tmp, 254, "File type: %s", trackinfo_get_mimetype_str(info));
	gtk_label_set_text(GTK_LABEL(gti->label_filetype), tmp);

	g_snprintf(tmp, 254, "File size: %d KiB", trackinfo_get_file_size(info));
	gtk_label_set_text(GTK_LABEL(gti->label_filesize), tmp);

	g_snprintf(tmp, 254, "ID: %d", trackinfo_get_id(info));
	gtk_label_set_text(GTK_LABEL(gti->label_id), tmp);
	return TRUE;
}

GtkWidget* gtrackinfo_new(void)
{
	return GTK_WIDGET(g_object_new(gtrackinfo_get_type(), NULL));
}
