/*
 * Logserver
 * Copyright (C) 2017-2025 Joel Reardon
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include "test.h"

#include "format_string.h"

TEST_CASE("fs no format") {
	FormatString fs;
	fs.init("hello");
	CHECK(fs == "hello");
	for (size_t i = 0; i < 5; ++i) {
		CHECK(fs.code(i) == 0);
	}
}

TEST_CASE("fs append") {
	FormatString fs;
	fs.init("hello ");
	fs.add("world", 6);
	CHECK(fs == "hello world");
	for (size_t i = 0; i < 11; ++i) {
		CHECK(fs.code(i) == (i >= 6 ? 6 : 0));
	}
}

TEST_CASE("fs multi append") {
	FormatString fs;
	fs.init("hello ");
	fs.add("there ", 2);
	fs.add("world", 6);
	CHECK(fs == "hello there world");
	for (size_t i = 0; i < 17; ++i) {
		if (i >= 12) CHECK(fs.code(i) == 6);
		else if (i >= 6) CHECK(fs.code(i) == 2);
		else CHECK(fs.code(i) == 0);
	}
}

TEST_CASE("fs cursors") {
	FormatString fs;
	fs.init("hello");
	fs.cursor(0);
	CHECK(fs.code(0) == G::CURSOR_COLOUR);
	CHECK(fs.code(1) == 0);
}

TEST_CASE("fs highlight") {
	FormatString fs;
	fs.init("hello ");
	fs.add("world", 6);
	fs.highlight();
	CHECK(fs == "hello world");
	for (size_t i = 0; i < 11; ++i) {
		CHECK(fs.code(i) == (i >= 6 ? 38 : 32));
	}
}

TEST_CASE("fs keyword") {
	FormatString fs;
	fs.init("hello there world");
	fs.mark(1, "hello");
	for (size_t i = 0; i < 11; ++i) {
		CHECK(fs.code(i) == (i >= 5 ? 0 : 1));
	}
}

TEST_CASE("fs multikeyword") {
	FormatString fs;
	fs.init("hello there world");
	fs.mark(1, "hello");
	fs.mark(3, "there");
	for (size_t i = 0; i < 11; ++i) {
		if (i < 5) CHECK(fs.code(i) == 1);
		else if (i >= 6 && i < 11) CHECK(fs.code(i) == 3);
		else CHECK(fs.code(i) == 0);
	}
}

TEST_CASE("fs colon") {
	FormatString fs;
	fs.init("some key: value");
	fs.colour_function();
	for (size_t i = 0; i < 14; ++i) {
		if (i >= 4 && i < 9) {
			CHECK(fs.code(i) == G::COLON_COLOUR);
		} else CHECK(fs.code(i) == 0);
	}
}

TEST_CASE("fs add pair") {
	FormatString fs;
	fs.init("hello");
	fs.add("world", "\5\4\3\2\1");
	for (size_t i = 0; i < 5; ++i) {
		CHECK(fs.code(i) == 0);
	}
	for (int i = 5; i < 10; ++i) {
		CHECK(fs.code(i) == 10 - i);
	}
	CHECK((string) fs == "helloworld");
}


TEST_CASE("fs add fs") {
	FormatString fs1, fs2, fs;
	fs1.add("hello", 3);
	fs2.add("world", 4);
	fs.add(fs1);
	fs.add(fs2);
	for (size_t i = 0; i < 5; ++i) {
		CHECK(fs.code(i) == 3);
	}
	for (size_t i = 5; i < 10; ++i) {
		CHECK(fs.code(i) == 4);
	}
	CHECK((string) fs == "helloworld");
}
