/* Copyright (c) 2000 Ben Woodard
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public Licence
 * as published by the Free Software Foundation; either version 2
 * of the Licence, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRENTY; without even the implied warrenty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public Licence in the COPYING file for more
 * details.
 *
 * You should have received a copy of the GNU Library General 
 * Public License along with the GNU C Library; see the file 
 * COPYING.LIB.  If not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */

#define _GNU_SOURCE
#include <assert.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/file.h>
#include <sys/poll.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <sys/un.h>
#include <tdb.h>
#include <unistd.h>
#include <wchar.h>
#include <wctype.h>

#include <glib.h>

#include "printsys.h"

/* *** - Insert the header files for any other ways to get 
   printer information here. */
#include "printcap.h"

#define ERRBUF_SIZE 100
#define LPS_MAGIC 0x1662
char errbuf[ERRBUF_SIZE];

/***** Local Function prototypes *************************/
// static void lps_perror(int fd, const char *str);
static int get_connection(LPS_Printer_t *printer, 
			  LPS_Pair_t *attributes,unsigned int *docid, 
			  int errfd);
static LPS_Error_t send_reference(LPS_Printer_t *printer, 
				  LPS_Pair_t *attributes,unsigned int *docid,
				  int srcfd,int errfd);
static int setup_connection(LPS_Printer_t *printer,
			    LPS_Pair_t *attributes,struct msghdr *msg,
			    unsigned int *docid);
static LPS_Pair_t *merge_attrib(const LPS_Pair_t *first, 
				const LPS_Pair_t *second);
static LPS_Pair_t *add_aggregate(LPS_Job_t *job,LPS_Pair_t *attrib,
				 int last);
static int init_request(LPS_System_t *sys);
static char *readall(int fd, int *curlen);
static char *asprintfa(char *base,const char *fmt, ...);
static int isnumber(const wchar_t *str);
/********************************************************/
LPS_System_t *lps_init(const void *initdata){
  LPS_System_t *newone=(LPS_System_t *)malloc(sizeof(LPS_System_t));
  if(newone!=NULL){
    memset(newone,0,sizeof(LPS_System_t));
    printcap_init(newone,initdata);
    newone->magic=LPS_MAGIC;
  }
  return newone;
}

LPS_Error_t lps_end(LPS_System_t *system){
  if(!system)
    return LPS_NOSYS;
  if(system->magic!=LPS_MAGIC)
    return LPS_BADMAGIC;
  system->magic=0;
  printcap_end(system);
  free(system);
  return LPS_OK;
}

wchar_t *lps_pair_lookup(LPS_Pair_t *pairs, const wchar_t *key){
  LPS_Pair_t *cur;
  if (!pairs)
    return NULL;
  
  for (cur = pairs; cur->key != NULL; cur++) {
    if (wcscmp(cur->key, key))
      continue;
    return cur->value;
  }

  return NULL;
}

LPS_Pair_t *lps_pair_update(LPS_Pair_t *pairs,const wchar_t *key, 
			    const wchar_t *value){
  int i=0;
  LPS_Pair_t *cur;
  if(pairs){
    for(cur=pairs;cur->key!=NULL;cur++,i++){
      if(!wcscmp(cur->key,key)){
	wchar_t *old_val=cur->value;
	cur->value=wcsdup(value);
	if(cur->value==NULL){
	  cur->value=old_val;
	  return NULL; /* FIXME: nothing checks for this atm. */
	}
	free(old_val);
	return pairs;
      }
    }
    cur=(LPS_Pair_t*)realloc(pairs,sizeof(LPS_Pair_t)*(i+2));
  }else{
    cur=(LPS_Pair_t*)malloc(sizeof(LPS_Pair_t)*2);
  }
  assert(cur!=NULL);
  cur[i+1].key=NULL;
  cur[i].key=wcsdup(key);
  cur[i].value=wcsdup(value);
  return cur;
}

void lps_free_pairs(LPS_Pair_t *doomed){
  LPS_Pair_t *cur;
  if(doomed==NULL)
    return;
  for(cur=doomed;cur->key!=NULL;cur++){
    free(cur->key);
    free(cur->value);
  }
  g_free(doomed);
}

const wchar_t **lps_get_printer_list(LPS_System_t *sys,const char *ns
				     __attribute__ ((unused))){
  if(sys->magic!=LPS_MAGIC)
    return NULL;
  return printcap_get_printer_list(sys);
}

void lps_free_printer_list(wchar_t **doomed){
  wchar_t **cur;
  if(doomed==NULL)
    return;
  for(cur=doomed;*cur!=NULL;cur++)
    free(*cur);
  free(doomed);
}

LPS_Printer_t *lps_get_printer (LPS_System_t *sys,const wchar_t *prname,
				const char *ns __attribute__ ((unused))){
  LPS_Printer_t *retval;
  if(sys->magic!=LPS_MAGIC)
    return NULL;
  retval=printcap_get_printer(sys,prname);
  if(retval!=NULL)
    retval->system=sys;
  return  retval;
}

void lps_free_printer(LPS_Printer_t *doomed){
  if(doomed==NULL)
    return;
  lps_free_printer_list(doomed->names);
  lps_free_pairs(doomed->fields);
  if(doomed->comments!=NULL)
    free(doomed->comments);
  free(doomed);
}

LPS_Printer_t *lps_create_printer(LPS_System_t *sys, wchar_t **prnames,
				  LPS_Pair_t *fields, wchar_t *comments){
  wchar_t **cur;
  LPS_Printer_t *newone;

  if(sys->magic!=LPS_MAGIC)
    return NULL;

  // make sure that the name is unique
  for(cur=prnames;*cur!=NULL;cur++)
    if(lps_get_printer(sys,*cur,NULL)!=NULL){
      sys->lps_errno=LPS_BADNAME;
      return NULL;
    }

  newone=(LPS_Printer_t*) malloc(sizeof(LPS_Printer_t));
  if(newone){
    newone->system=sys;
    newone->names=prnames;
    newone->fields=fields;
    /* XXX this field needs some sanity checking to make sure
       that it is reasonable */
    newone->comments=comments;
  }else{
    sys->lps_errno=LPS_NOMEM;
  }
  return newone;
}

LPS_Error_t lps_commit_printer(LPS_Printer_t *printer, 
			       const char *ns __attribute__ ((unused))){
  //make sure names are unique.
  wchar_t **cur;
  for(cur=printer->names;*cur!=NULL;cur++){
    LPS_Printer_t *testpr=lps_get_printer(printer->system,*cur,NULL);
    if(testpr!=NULL && testpr!=printer){
      printf("commit fail\n");
      return LPS_BADNAME;
    }
  }
  return printcap_commit_printer(printer);
}

LPS_Error_t lps_destroy_printer(LPS_Printer_t *printer, 
				const char *ns __attribute__ ((unused))){
  return printcap_destroy_printer(printer);
}

LPS_Pair_t *lps_lookup_doc_attrib(unsigned int docid){
  TDB_CONTEXT *tdb;
  TDB_DATA key;
  TDB_DATA rawdata;
  LPS_Pair_t *attrib=NULL;

  if((tdb=tdb_open(LPS_DOCDBPATH,0,0,O_RDONLY,0644))==NULL){
    /*     fprintf(stderr,"could not open tdb database %s: %s\n",
	   LPS_DOCDBPATH, sys_errlist[errno]); */
    return NULL;
  }
  key.dptr=(char*)&docid;
  key.dsize=sizeof(docid);
  rawdata=tdb_fetch(tdb,key);
  tdb_close(tdb);
  if(rawdata.dptr==NULL)
    return NULL;
  attrib=_lps_decode_pairs(rawdata.dptr,rawdata.dsize,attrib);
  return attrib;
}

LPS_Error_t lps_commit_doc_attrib(unsigned int docid, 
				  LPS_Pair_t *attrbs){
  TDB_CONTEXT *tdb;
  TDB_DATA key;
  TDB_DATA rawdata;
  LPS_Pair_t *attrib=NULL;

  if((tdb=tdb_open(LPS_DOCDBPATH,0,0,O_RDWR,0644))==NULL)
    return LPS_TDBOPENFAIL;

  key.dptr=(char*)&docid;
  key.dsize=sizeof(docid);
  rawdata=tdb_fetch(tdb,key);
  if(rawdata.dptr==NULL)
    return LPS_NODOC;
  attrib=_lps_decode_pairs(rawdata.dptr,rawdata.dsize,attrib);

  attrbs=merge_attrib(attrbs,attrib);
  lps_free_pairs(attrib);
  rawdata.dptr=_lps_encode_pairs(attrbs,&rawdata.dsize);
  if(tdb_store(tdb,key,rawdata,TDB_REPLACE)!=0)
    return LPS_TDBSTOREFAIL;
  tdb_close(tdb);
  return LPS_OK;
}

LPS_Job_t *lps_create_job(LPS_Printer_t *printer,LPS_Pair_t *attributes){
  LPS_Job_t *newone=(LPS_Job_t*) malloc(sizeof(LPS_Job_t));
  if(newone!=NULL){
    newone->printer=printer;
    newone->last_docid=0;
    newone->seqno=0;
    newone->attributes=attributes;
  }else{
    printer->system->lps_errno=LPS_NOMEM;
  }
  return newone;
}

void lps_free_job(LPS_Job_t *doomed){
  // printer is a reference to the object not the object itself
  // so don't free it here.
  lps_free_pairs(doomed->attributes);
}

static int setup_connection(LPS_Printer_t *printer,LPS_Pair_t *attributes,
			    struct msghdr *msg,unsigned int *docid){
  wchar_t *prname=lps_pr_name(printer);
  int sock=socket(AF_LOCAL,SOCK_STREAM,0);
  struct sockaddr_un *addr;
  struct iovec iov[2];
  ssize_t len_read;
  int save_errno;
  ssize_t i,len;

  iov[0].iov_base=(char*)"sub\n";
  iov[0].iov_len=4;
  iov[1].iov_base=prname;
  iov[1].iov_len=(wcslen(prname)+1)*sizeof(wchar_t);
  msg->msg_name=NULL;
  msg->msg_namelen=0;
  msg->msg_iov=iov;
  msg->msg_iovlen=2;
  msg->msg_flags=0;

  if(sock==-1){
    printer->system->lps_errno=LPS_SOCKET;
    return -1;
  }

  addr = malloc(sizeof(struct sockaddr_un) + strlen(LPS_SOCK_PATHNAME));
  assert(addr);
  memset(addr, 0, sizeof(struct sockaddr_un) + strlen(LPS_SOCK_PATHNAME));
  addr->sun_family=AF_LOCAL;
  strcpy(addr->sun_path, LPS_SOCK_PATHNAME);
  
  if(connect(sock, (struct sockaddr *) addr, SUN_LEN(addr))==-1){
    save_errno=errno;
    printer->system->lps_errno=LPS_SOCKET;
    close(sock);
    free(addr);
    errno=save_errno;
    return -1;
  }
  free(addr);

  if(sendmsg(sock,msg,0)==-1){
    save_errno=errno;
    printer->system->lps_errno=LPS_SENDPR;
    close(sock);
    errno=save_errno;
    return -1;
  }
  
  memset(errbuf,0,ERRBUF_SIZE);
  switch(len_read=read(sock,errbuf,ERRBUF_SIZE)){
  case 0:
    printer->system->lps_errno=LPS_CONNTERM;
    return -1;
  case -1:
    printer->system->lps_errno=LPS_READPR;
    save_errno=errno;
    close(sock);
    errno=save_errno;
    return -1;
  }

  g_strchomp(errbuf);
  if(strcmp(errbuf,"OK")){ 
    printer->system->lps_errno=LPS_PRNOTOK;
    close(sock);
    return -1;
  }

  if(attributes==NULL){
    i=0;
    len=sizeof(i);
    i=write(sock,&i,sizeof(i));
    if(i!=len) 
      goto bad_write;
  }else{
    ssize_t i;
    char *buf=_lps_encode_pairs(attributes,&i);
    if(buf==NULL){
      printer->system->lps_errno=LPS_NOMEM;
      close(sock);
      return -1;
    }
    if(write(sock,&i,sizeof(i))!=sizeof(i))
      goto bad_write;
    if(i!=write(sock,buf,i))
      goto bad_write;
  }

  printf("reading jobid\n"); //DEBUG
  switch(len_read=read(sock,errbuf,ERRBUF_SIZE-1)){
  case 0:
    printer->system->lps_errno=LPS_CONNTERM;
    close(sock);
    return -1;
  case -1:
    printer->system->lps_errno=LPS_READDOCID;
    close(sock);
    return -1;
  }
  errbuf[len_read]=0;

  printf("jobid=%s\n",errbuf); //DEBUG
  if(!strncmp(errbuf,"Err ",4)){
    printer->system->lps_errno=LPS_READDOCID;
    close(sock);
    return -1;
  }
  *docid=strtoul(errbuf,NULL,0);

  return sock;

 bad_write:
  save_errno=errno;
  switch(i){
  case 0:
    printer->system->lps_errno=LPS_CONNTERM;
    break;
  case -1:
    printer->system->lps_errno=LPS_SENDATTR;
    break;
  default:
    printer->system->lps_errno=LPS_ATTRSHRT;
  }
  close(sock);
  errno=save_errno;
  return -1;
}

static LPS_Error_t send_reference(LPS_Printer_t *printer, 
				  LPS_Pair_t *attributes,unsigned int *docid,
				  int srcfd,int errfd){
  struct msghdr msg;
  struct cmsghdr *fdhdr;
  union {
    struct cmsghdr cm;
    char control[CMSG_SPACE(sizeof(int))*2];
  } control_un;
  int fd;

  msg.msg_control=control_un.control;
  msg.msg_controllen=sizeof(control_un.control);

  fdhdr=CMSG_FIRSTHDR(&msg);
  fdhdr->cmsg_len=CMSG_LEN(sizeof(int)); 
  fdhdr->cmsg_level=SOL_SOCKET;
  fdhdr->cmsg_type=SCM_RIGHTS;
  *((int*)CMSG_DATA(fdhdr))=srcfd;

  fdhdr=CMSG_NXTHDR(&msg,fdhdr);
  fdhdr->cmsg_len=CMSG_LEN(sizeof(int)); 
  fdhdr->cmsg_level=SOL_SOCKET;
  fdhdr->cmsg_type=SCM_RIGHTS;
  *((int*)CMSG_DATA(fdhdr))=errfd;

  if((fd=setup_connection(printer,attributes,&msg,docid))!=-1)
    close(fd);
  return printer->system->lps_errno;
}  

static int get_connection(LPS_Printer_t *printer,LPS_Pair_t *attributes,
			  unsigned int *docid, int errfd){
  struct msghdr msg;
  struct cmsghdr *fdhdr;
  union {
    struct cmsghdr cm;
    char control[CMSG_SPACE(sizeof(int))];
  } control_un;

  msg.msg_control=control_un.control;
  msg.msg_controllen=sizeof(control_un.control);

  fdhdr=CMSG_FIRSTHDR(&msg);
  fdhdr->cmsg_len=CMSG_LEN(sizeof(int)); 
  fdhdr->cmsg_level=SOL_SOCKET;
  fdhdr->cmsg_type=SCM_RIGHTS;
  *((int*)CMSG_DATA(fdhdr))=errfd;

  return setup_connection(printer,attributes,&msg,docid);
}

static LPS_Pair_t *merge_attrib(const LPS_Pair_t *first, 
				const LPS_Pair_t *second){
  unsigned int i=0;
  unsigned int j=0;
  const LPS_Pair_t *cur;
  LPS_Pair_t *retval;
  if(first)
    for(cur=first;cur->key!=NULL;cur++,i++);
  if(second)
    for(cur=second;cur->key!=NULL;cur++,j++);
  if(i==0 && j==0)
    return NULL;
  retval=(LPS_Pair_t*)malloc(sizeof(LPS_Pair_t)*(i+j+1));
  if(retval){
    memset(retval,0,(i+j+1)*sizeof(LPS_Pair_t));
    memcpy(retval,first,(i+1)*sizeof(LPS_Pair_t));
    for(cur=second;cur->key;cur++){
      LPS_Pair_t *idx;
      for(idx=retval;idx->key;idx++){
	if(!wcscmp(cur->key,idx->key)){
	  idx->value=cur->value;
	  break;
	}
      } // loop through pre-existing pairs
      idx->key=cur->key;
      idx->value=cur->value;
    } // loop through new pairs
  }
  return retval;
}

static LPS_Pair_t *add_aggregate(LPS_Job_t *job,LPS_Pair_t *attrib,
				 int last){
  LPS_Pair_t *curattr=merge_attrib(job->attributes,attrib);
  GString *str=g_string_new("");
  wchar_t *tmp_dest1;

  if(job->last_docid!=0){
    g_string_sprintf(str,"%u",job->last_docid);
    curattr=lps_pair_update(curattr,LPS_AGG_PREV,
			    tmp_dest1=lps_promote(str->str));
    free(tmp_dest1);
  }

  g_string_sprintf(str,"%u",job->seqno);
  curattr=lps_pair_update(curattr,LPS_AGG_SEQNO,
			  tmp_dest1=lps_promote(str->str));
  free(tmp_dest1);

  if(last)
    curattr=lps_pair_update(curattr,LPS_AGG_LAST,L"true");
    
  /* identify which attributes were job vs. document attributes 
     so that we can reconstruct them later if need be. */
  if(job->attributes!=NULL && job->attributes->key!=NULL){
    size_t len=wcslen(job->attributes->key)+2;
    LPS_Pair_t *cur;

    tmp_dest1=malloc(len*sizeof(wchar_t));
    assert(tmp_dest1);

    wcscpy(tmp_dest1,job->attributes->key);
    for(cur=(job->attributes)+1;cur->key!=NULL;cur++){
      wcscat(tmp_dest1,L",");
      len+=wcslen(cur->key);
      tmp_dest1=realloc(tmp_dest1,len*sizeof(wchar_t));
      wcscat(tmp_dest1,cur->key);
    }
    curattr=lps_pair_update(curattr,LPS_JOB_ATTR,tmp_dest1);
    free(tmp_dest1);
  }

  g_string_free(str,TRUE);
  return curattr;
}

int lps_get_connection(LPS_Job_t *job,LPS_Pair_t *attributes,
		       unsigned int *docid,int errfd,int last){
  LPS_Pair_t *curattr=add_aggregate(job,attributes,last);
  int retval;

  retval=get_connection(job->printer,curattr,docid,errfd);
  free(curattr);
  return retval;
}

LPS_Error_t lps_send_ref(LPS_Job_t *job,LPS_Pair_t *attributes,
			 unsigned int *docid,int srcfd,int errfd,int last){
  LPS_Pair_t *curattr=add_aggregate(job,attributes,last);
  LPS_Error_t retval=send_reference(job->printer,curattr,docid,srcfd,errfd);
  free(curattr);
  return retval;
}

wchar_t *lps_promote(const char *str){
  mbstate_t ps;
  size_t len;
  wchar_t *newone;
  memset(&ps,0,sizeof(mbstate_t)); // needed to avoid an assertion
  len=mbsrtowcs(NULL,&str,0,&ps)+1;
  // needed to deal with incomplete chars at the end of the string
  memset(&ps,0,sizeof(mbstate_t)); 
  newone=(wchar_t*)malloc(len*sizeof(wchar_t));
  if(newone)
    mbsrtowcs(newone,&str,len,&ps);
  return newone;
}

char *lps_demote(const wchar_t *str){
  mbstate_t ps;
  size_t len;
  char *newone;

  memset(&ps,0,sizeof(mbstate_t)); 
  len=wcsrtombs(NULL,&str,0,&ps);
  memset(&ps,0,sizeof(mbstate_t)); 
  newone=(char*)malloc(len+1);
  //  printf("str=\"%s\"\tlen=%d\n",newone,len);
  if(newone){
    memset(newone,0,len+1);
    wcsrtombs(newone,&str,len,&ps);
  }
  return newone;
}

char **lps_demote_vec(const wchar_t **strs){
  const wchar_t **cur;
  char **newone;
  char **cur_str;

  for(cur=strs;*cur!=NULL;cur++);
  newone=(char**)malloc((cur-strs+1)*sizeof(char*));
  assert(newone);
  newone[cur-strs]=NULL;
  for(cur=strs,cur_str=newone;*cur!=NULL;cur++,cur_str++)
    *cur_str=lps_demote(*cur);
  return newone;
}

/* FIXME: This needs to deal with memory allocation failures
   better. */
wchar_t **lps_promote_vec(const char **strs){
  const char **cur;
  wchar_t **newone;
  wchar_t **cur_wcs;

  for(cur=strs;*cur!=NULL;cur++);
  newone=(wchar_t**)malloc((cur-strs+1)*sizeof(wchar_t*));
  assert(newone);
  newone[cur-strs]=NULL;
  for(cur=strs,cur_wcs=newone;*cur!=NULL;cur++,cur_wcs++)
    *cur_wcs=lps_promote(*cur); 
  return newone;
}

unsigned int lps_num_pairs(LPS_Pair_t *pairs){
  unsigned int num;
  for(num=0;pairs[num].key!=NULL;num++);
  return num;
}

ssize_t lps_block_write(int fd, const char *buf, size_t len){
  size_t orig_len = len;
  
  while (len) {
    ssize_t ret = write(fd, buf, len);
      
    if (ret == -1){
      if(errno==EINTR)
	continue;
      else
	return -1;
    }
      
    len -= (size_t)ret;
    buf += (size_t)ret;
  }
  
  return (orig_len);
}

int lps_send_data(int outfd,int infd){
  ssize_t len = -1;
  char buf[1024 * 8];
  
  while ((len = read(infd, buf, sizeof(buf))) != 0){
    struct pollfd tmp;
    int ret = -1;	    
    size_t bytes = len;
      
    if (len == -1)
      switch (errno){
      case EAGAIN: /* stdin is a pipe/socket */
	tmp.fd = infd;
	tmp.events = POLLIN;
	if ((ret = poll(&tmp, 1, -1))== -1 && errno!=EINTR){
	  return 0;
	} // intentionally falls through
      case EINTR:
	continue;
      default:
	return 0;
      }
      
    if (lps_block_write(outfd, buf, bytes) == -1)	{
      return 0;
    }
  }
   
  return 1;
}

unsigned int lps_filter_opts(LPS_System_t *sys,int argc, char **argv, 
			     LPS_Printer_t **printer, LPS_Pair_t **docattr){
  unsigned int docid;
  wchar_t *tmp;

  if(sys==NULL){
    fprintf(stderr,"Configuration Error: No print system\n");
    exit(EXIT_NOSYSTEM);
  }

  if(argc!=3){
    fprintf(stderr,"Internal Error: Parameters passed to %s were wrong.\n"
	    "Something is wrong with the spooler\n",argv[0]);
    exit(EXIT_BADPARAMS);
  }
 
  if((tmp=lps_promote(argv[1]))==NULL){
    fprintf(stderr,"Internal Error: Out of memory.\n");
    exit(EXIT_NOMEM);
  }

  if((*printer=lps_get_printer(sys,tmp,NULL))==NULL){
    fprintf(stderr,"Internal Error: Printer specified to %s was not "
	    "found.\n",argv[0]);
    exit(EXIT_BADPARAMS);
  }
  free(tmp);
  
  docid=strtoul(argv[2],NULL,10);
  if((*docattr=lps_lookup_doc_attrib(docid))==NULL){
    fprintf(stderr,"Internal Error: Document specified to %s was not "
	    "found.\n",argv[0]);
    exit(EXIT_BADPARAMS);
  }
  return docid;
}

char *lps_chdir_sd(LPS_Printer_t *printer, int *err){
  wchar_t *tmpwcs;
  char *spooldir;

  if((tmpwcs=lps_pr_lookup_field(printer,L"sd"))==NULL){
    *err=1;
    return NULL;
  }

  if((spooldir=lps_demote(tmpwcs))==NULL){
    *err=2;
    return NULL;
  }

  if(chdir(spooldir)==-1){
    *err=3;
    return NULL;
  }

  return spooldir;
}

void lps_fail_doc(unsigned int docid, LPS_Pair_t *docattr){
  docattr=lps_pair_update(docattr,L"state",L"failed");
  lps_commit_doc_attrib(docid,docattr);
}


static int init_request(LPS_System_t *sys){
  int sock=socket(AF_LOCAL,SOCK_STREAM,0);
  struct sockaddr_un *addr;

  if(sock==-1){
    sys->lps_errno=LPS_SOCKET;
    return -1;
  }

  addr = malloc(sizeof(struct sockaddr_un) + strlen(LPS_SOCK_PATHNAME));
  if(!addr){
    sys->lps_errno=LPS_NOMEM;
    close(sock);
    return -1;
  }
  memset(addr, 0, sizeof(struct sockaddr_un) + strlen(LPS_SOCK_PATHNAME));
  addr->sun_family=AF_LOCAL;
  strcpy(addr->sun_path, LPS_SOCK_PATHNAME);
  
  if(connect(sock, (struct sockaddr *) addr, SUN_LEN(addr))==-1){
    int save_errno=errno;
    sys->lps_errno=LPS_SOCKET;
    close(sock);
    free(addr);
    errno=save_errno;
    return -1;
  }
  free(addr);

  return sock;
}

int lps_request_printer_list(LPS_System_t *sys,const char *ns){
  size_t len=strlen(ns)+1;
  int sock=init_request(sys);
  struct iovec iov[3];
  struct msghdr msg;

  msg.msg_control=NULL;
  msg.msg_controllen=0;
  msg.msg_name=NULL;
  msg.msg_namelen=0;
  msg.msg_iov=iov;
  msg.msg_iovlen=3;
  msg.msg_flags=0;

  iov[0].iov_base=(char*)"lst\n";
  iov[0].iov_len=4;
  iov[1].iov_base=&len;
  iov[1].iov_len=sizeof(size_t);
  iov[2].iov_base=(char*)ns;
  iov[2].iov_len=len;

  if(sock==-1){
    sys->lps_errno=LPS_SOCKET;
    return -1;
  }

  if(sendmsg(sock,&msg,0)==-1){
    int save_errno=errno;
    sys->lps_errno=LPS_SENDPR;
    close(sock);
    errno=save_errno;
    return -1;
  }

  return sock;
}

#define BUFSIZE 10

static char *readall(int fd, int *curlen){
  char *retval=NULL;
  char *cur;
  char tmpbuf[BUFSIZE];
  int len_read=0;
  while((len_read=read(fd,tmpbuf,BUFSIZE))!=0){
    if(retval==NULL){
      cur=retval=malloc(len_read);
      *curlen=len_read;
    }else{
      retval=realloc(retval,len_read+*curlen);
      cur=retval+*curlen;
      *curlen+=len_read;
    }
    assert(cur);
    memcpy(cur,tmpbuf,len_read);
  }
  return retval;
}

wchar_t **lps_decode_printer_list(int fd, wchar_t **dest, 
				  unsigned int dest_len, int *more){
  static char *residue=NULL;
  static int residue_len=0;
  int len_read;
  char *curbuf=readall(fd,&len_read);
  wchar_t *cur;
  unsigned int idx=0;

  // push everything into the residue
  if(residue==NULL){
    residue=curbuf;
    residue_len=len_read;
  }else{
    residue=realloc(residue,residue_len+len_read);
    memcpy(residue+residue_len,curbuf,len_read);
    residue_len+=len_read;
  }

  // everything is in residue and we have the length
  while((cur=wcschr((wchar_t*)residue,L'\n'))!=NULL && idx+1<dest_len){
    *cur=0;
    dest[idx]=lps_promote(residue);
    dest[++idx]=NULL;
    residue_len-=(char*)cur-residue;
    memcpy(residue,cur+1,residue_len);
  }
  if(residue[0]==L'\0')
    *more=0;
  else
    *more=1;
  return dest;
}

int lps_request_printer(LPS_System_t *sys,const wchar_t *prname,
                        const char *ns){
  size_t len=strlen(ns)+1;
  int sock;
  if(sys->magic!=LPS_MAGIC)
    return -1;
  sock=init_request(sys);
  write(sock,"prn\n",4);
  write(sock,&len,sizeof(size_t));
  write(sock,ns,len);
  len=(wcslen(prname)+1)*sizeof(wchar_t);
  write(sock,&len,sizeof(size_t));
  write(sock,ns,len);
  return sock;
}

/* XXX this is not thread safe the way that it should be. The problem 
   is multiple threads may call this and therefore the static 
   variables will get overwritten. */
LPS_Printer_t *lps_decode_printer(LPS_System_t *sys,int fd){
  static char *residue=NULL;
  static unsigned int residue_len=0;
  static int got_len=0;
  static int cur_len;
  int len_read;
  char *curbuf=readall(fd,&len_read);
  LPS_Printer_t *dest;
  unsigned int total_len;

  /* XXX this is a bad thing to do here because it is impossible for 
     the calling application to tell if sys is bad or if there just
     isn't enough data yet. */
  if(sys->magic!=LPS_MAGIC)
    return NULL; 
  // push everything into the residue
  if(residue==NULL){
    residue=curbuf;
    residue_len=len_read;
  }else{
    residue=realloc(residue,residue_len+len_read);
    memcpy(residue+residue_len,curbuf,len_read);
    residue_len+=len_read;
  }

  if(got_len==0){
    // didn't get enough to do anything with
    if(!(residue_len>=sizeof(total_len)))
      return NULL;
    cur_len=*(int*)residue;
    residue_len-=sizeof(total_len);
    memcpy(residue,residue+sizeof(total_len),residue_len);
    got_len=1;
  }
  if(total_len<residue_len)
    return NULL;

  // at this point the residule should have the complete entry
  residue[residue_len]=0;
  dest=_lps_process_entry(residue);
  dest->system=sys;
  return dest;
}

void _lps_dump_printer(LPS_Printer_t *printer){
  printf("printer=%p\n",printer);
  if(printer){
    wchar_t **cur;
    printf("\tnames ");
    for(cur=printer->names;*cur!=NULL;cur++)
      printf("%ls ",*cur);
    printf("\n");
  }
  // dump the other fields as necessary
}

char *_lps_encode_pairs(LPS_Pair_t *pairs, size_t *length){
  LPS_Pair_t *curatt;
  char *curpos;
  char *buf;
  *length=0;
  for(curatt=pairs;curatt->key;curatt++)
    *length+=(wcslen(curatt->key)+wcslen(curatt->value))*sizeof(wchar_t)+
      2*sizeof(size_t);
  if((buf=curpos=malloc(*length))==NULL)
    return buf;
  for(curatt=pairs;curatt->key;curatt++){
    int len;
    len=(*(size_t*)curpos)=wcslen(curatt->key);
    curpos+=sizeof(size_t);
    memcpy(curpos,curatt->key,len*sizeof(wchar_t));
    curpos+=len*sizeof(wchar_t);

    len=(*(size_t*)curpos)=wcslen(curatt->value);
    curpos+=sizeof(size_t);
    memcpy(curpos,curatt->value,len*sizeof(wchar_t));
    curpos+=len*sizeof(wchar_t);
  }
  return buf;
}

LPS_Pair_t *_lps_decode_pairs(const char *attr_str, size_t attrib_size,
			      LPS_Pair_t *attrib){
  size_t cur;
  for(cur=0;cur<attrib_size;){
    size_t len=*(size_t*)(attr_str+cur);
    wchar_t *key,*value;

    cur+=sizeof(size_t);
    key=malloc(sizeof(wchar_t)*(len+1)); 
    assert(key);
    key[len]=0;
    memcpy(key,attr_str+cur,len*sizeof(wchar_t));
    cur+=len*sizeof(wchar_t);

    len=*(size_t*)(attr_str+cur);
    cur+=sizeof(size_t);
    value=malloc(sizeof(wchar_t)*(len+1)); 
    assert(value);
    value[len]=0;
    memcpy(value,attr_str+cur,len*sizeof(wchar_t));
    cur+=len*sizeof(wchar_t);

    attrib=lps_pair_update(attrib,key,value);
  }
  return attrib;
}

LPS_Printer_t *_lps_process_entry(char *buf){
  char *endpos = NULL;
  LPS_Printer_t *newone = NULL;
  char **kvstrings = NULL;
  char **curkv = NULL;
  GSList *fields = NULL;
  unsigned int num_fields;
  char **tmpvec;
  char *line_end = strchr(buf, '\n');
  char *curpos;
  char *commline = NULL;
  static GString *comments=NULL;

  if(comments==NULL)
    comments= g_string_new("");

  // skip over beginning blanks
  for (curpos = buf; *curpos==' ' || *curpos=='\t'; curpos++);

  // skip blank lines and comments
  if (*curpos == '#' || *curpos == '\n') {
    /* Append all comments and blank lines, so they can be 
       preserved if the file is saved. */
    int len;
    if(*curpos=='\n'){
      // if there is a blank line between the comments 
      // and the printer the comments don't belong with this
      // printer.
      comments=g_string_assign(comments,"");
    } else if (*curpos == '#') {
      /* Get just the comment line. */
      commline = (char *)malloc(line_end - curpos + 2);
      assert(commline);
      strncpy(commline, curpos, line_end - curpos + 2);
      commline[line_end - curpos + 1] = 0;
      g_string_append(comments, commline);
      free(commline);
    } 

    memcpy(buf, line_end + 1,len=strlen(line_end+1)+1);
    return NULL;
  }
  /* ends up on printers */
  newone = (LPS_Printer_t*)malloc(sizeof(LPS_Printer_t));
  assert(newone);
  
  /* Associate the preceding comment lines with this print
     queue. */
  if (comments->len>0) {
    newone->comments=lps_promote(comments->str);
    comments=g_string_assign(comments,"");
  }else
    newone->comments=NULL;

  /* handle multiple entries properly -NickM */
  *line_end = 0;

  // pick out the names
  if ((endpos = strchr(curpos, ':'))==NULL){
    fprintf(stderr,"Warning: invalid line in printcap skipped.\n");
    return NULL;
  }

  *endpos = 0;		// terminate the string
    

  tmpvec = g_strsplit(curpos, "|", 0);
  if (!tmpvec[0])
  {
    fprintf(stderr,"Warning: blank printer name in printcap skipped.\n");
    g_strfreev(tmpvec);
    return NULL;
  }

  newone->names=lps_promote_vec((const char **)tmpvec);
  g_strfreev(tmpvec);

  // get the rest of the fields
  kvstrings = g_strsplit(endpos + 1, ":", 0);
  for (curkv = kvstrings; *curkv != NULL; curkv++) {
    LPS_Pair_t *newkv = NULL;

    g_strstrip(*curkv);
    if (strlen(*curkv) == 0)
      continue;
    /* ends up in newone's printer data */
    newkv = (LPS_Pair_t*)malloc(sizeof(LPS_Pair_t));	

    assert(newkv);
    if ((endpos = strchr(*curkv, '='))) {
      *endpos = 0;
      newkv->value = lps_promote(endpos + 1);
    } else if ((endpos = strchr(*curkv, '#'))) {
      *endpos = 0;
      newkv->value = lps_promote(endpos + 1);
    } else if ((endpos = strchr(*curkv, '@'))) {
      *endpos = 0;
      newkv->value = lps_promote("FALSE");
    } else {
      newkv->value = lps_promote("TRUE");
    }

    newkv->key = lps_promote(*curkv);
    fields = g_slist_prepend(fields, newkv);
  }			 // end of for through all the key value pairs

  g_strfreev(kvstrings);

  num_fields = g_slist_length(fields);
  newone->fields = g_malloc(sizeof(LPS_Pair_t)*(num_fields+1));

  /* ends up in printer structure */
  newone->fields[num_fields].key =NULL;
  newone->fields[num_fields].value =NULL;

  for (num_fields--; fields; num_fields--) {
    GSList *save = fields->next;

    newone->fields[num_fields].key = ((LPS_Pair_t *) fields->data)->key;
    newone->fields[num_fields].value =
      ((LPS_Pair_t *) fields->data)->value;

    g_free(fields->data);
    g_slist_free_1(fields);
    fields = save;
  }

  // printf("before %d --%s--\n",strlen(buf),buf); //debug
  memcpy(buf,line_end+1,strlen(line_end+1)+1);
  //  printf("after %d --%s--\n",strlen(buf),buf); //debug
  //  fflush(stdout);
  return newone;
}

char *_lps_write_printcap_entry(LPS_Printer_t *curprinter){
  char *str = NULL;
  wchar_t **curname;
  LPS_Pair_t *curfield;

  /* The reason we use asprintfa in this function rather than 
     glib fucntions such as GString is we find that the current 
     version of glib doesn't handle wide character strings. it 
     prints the error 
     "GLib-WARNING **: g_printf_string_upper_bound(): 
     unable to handle wide char strings" */
  if (curprinter->comments!=NULL)
    str=asprintfa(str,"%S",curprinter->comments);
  
  for (curname=curprinter->names;*curname;curname++)
    str=asprintfa(str,"%S%c",*curname,*(curname+1)==NULL?':':'|');
  str=asprintfa(str,"\\\n");

  for(curfield=curprinter->fields;curfield->key!=NULL;curfield++)
    if(!wcscmp(L"TRUE",curfield->value))
      str=asprintfa(str,"\t:%S:%s",curfield->key,
		    curfield[1].key!=NULL?"\\\n":"\n");
    else 
      str=asprintfa(str,"\t:%S%c%S:%s",curfield->key,
		    isnumber(curfield->value)?'#':'=',curfield->value,
		    curfield[1].key!=NULL?"\\\n":"\n");
  str=asprintfa(str,"\n");
  return str;
}

static int isnumber(const wchar_t *str){
  for(;*str!=0;str++)
    if(!iswdigit(*str))
      return 0;
  return 1;
}

/* needed becase glib doesn't handle wchar_t's -- see above */
static char *asprintfa(char *base,const char *fmt, ...){
  va_list ap;
  char buf[1000]; // just to make it easier to debug
  char *retval;
  size_t len = 0;
  size_t base_len = 0;
  
  va_start(ap,fmt);
  memset(buf,0,1000);
  len=vsnprintf(buf, 1000, fmt, ap);
  va_end(ap);
  
  if(base){
      base_len = strlen(base);
    retval=realloc(base,len+base_len+1);
  }else{
    retval=malloc(len+1);
  }

  va_start(ap,fmt);
  vsprintf(retval+base_len,fmt,ap);
  va_end(ap);
  
  return retval;
}

int lps_pr_update_field( LPS_Printer_t *printer, 
			 const wchar_t *key , const wchar_t *value){
  if(wcschr(key,':') || wcschr(key,'\n') || wcschr(value,':') || 
	    wcschr(value,'\n'))
    return 0;
  printer->fields=lps_pair_update(printer->fields,key,value);
  return 1;
}
