# flake8: noqa
"""
The following contains a database of small molecules

Data for the G2/97 database are from
Raghavachari, Redfern, and Pople, J. Chem. Phys. Vol. 106, 1063 (1997).
See http://www.cse.anl.gov/Catalysis_and_Energy_Conversion/Computational_Thermochemistry.shtml for the original files.

All numbers are experimental values, except for coordinates, which are
MP2(full)/6-31G(d) optimized geometries (from http://www.cse.anl.gov/OldCHMwebsiteContent/compmat/G2-97.htm)

Atomic species:
ref: Curtiss et al. JCP 106, 1063 (1997).
'Enthalpy' is the experimental enthalpies of formation at 0K
'thermal correction' is the thermal corrections H(298)-H(0)

Molecular species:
ref: Staroverov et al. JCP 119, 12129 (2003)
'Enthalpy' is the experimental enthalpies of formation at 298K
'ZPE' is the zero-point energies
'thermal correction' is the thermal enthalpy corrections H(298K) - H_exp(0K)
ZPE and thermal corrections are estimated from B3LYP geometries and vibrations.

For details about G2-1 and G2-2 sets see doi:10.1063/1.477422.

Experimental ionization potentials are from http://srdata.nist.gov/cccbdb/
Information presented on these pages is considered public information
and may be distributed or copied http://www.nist.gov/public_affairs/disclaimer.cfm
"""

from ase.atoms import Atoms

atom_names = ['H','B','C','N','O','F','Al','Si','S','Cl']

molecule_names = ['BF3','BCl3','AlF3','AlCl3','CF4','CCl4','OCS','CS2','COF2','SiF4','SiCl4','N2O','ClNO','NF3','PF3','O3','F2O','ClF3','C2F4','C2Cl4','CF3CN','C3H4_C3v','C3H4_D2d','C3H4_C2v','C3H6_Cs','C3H6_D3h','C3H8','butadiene','2-butyne','methylenecyclopropane','bicyclobutane','cyclobutene','cyclobutane','isobutene','trans-butane','isobutane','C5H8','C6H6','H2CF2','HCF3','H2CCl2','HCCl3','H3CNH2','CH3CN','CH3NO2','CH3ONO','CH3SiH3','HCOOH','HCOOCH3','CH3CONH2','CH2NHCH2','NCCN','C2H6NH','CH3CH2NH2','H2CCO','CH2OCH2','CH3CHO','OCHCHO','CH3CH2OH','CH3OCH3','CH2SCH2','C2H6SO','CH3CH2SH','CH3SCH3','H2CCHF','CH3CH2Cl','H2CCHCl','H2CCHCN','CH3COCH3','CH3COOH','CH3COF','CH3COCl','C3H7Cl','C2H6CHOH','CH3CH2OCH3','C3H9N','C4H4O','C4H4S','C4H4NH','C5H5N','H2','SH','CCH','C2H3','CH3CO','H2COH','CH3O','CH3CH2O','CH3S','C2H5','C3H7','C3H9C','NO2']

data = {
 '2-butyne': {'CAS No.': 503173,
              'ZPE': 51.8731,
              'charges': None,
              'database': 'G2-2',
              'description': 'Dimethylacetylene (2-butyne, C4H6), D3h symm (eclipsed).',
              'enthalpy': 34.8,
              'magmoms': None,
              'name': 'C_4H_6 (2-butyne)',
              'positions': [[0.0, 0.0, 2.071955],
                            [0.0, 0.0, 0.60997],
                            [0.0, 0.0, -0.60997],
                            [0.0, 0.0, -2.071955],
                            [0.0, 1.020696, 2.464562],
                            [-0.883949, -0.510348, 2.464562],
                            [0.883949, -0.510348, 2.464562],
                            [0.0, 1.020696, -2.464562],
                            [0.883949, -0.510348, -2.464562],
                            [-0.883949, -0.510348, -2.464562]],
              'symbols': 'CCCCHHHHHH',
              'thermal correction': 4.2344},
 'Al': {'CAS No.': 7429905,
        'charges': None,
        'database': 'G2-2',
        'description': 'Al atom',
        'enthalpy': 78.23,
        'ionization energy': 5.99,
        'magmoms': [1.0],
        'name': 'Aluminium',
        'positions': [[0.0, 0.0, 0.0]],
        'symbols': 'Al',
        'thermal correction': 1.08},
 'AlCl3': {'CAS No.': 7446700,
           'ZPE': 2.9687,
           'charges': None,
           'database': 'G2-2',
           'description': 'AlCl3, Planar D3h symm.',
           'enthalpy': -139.7,
           'magmoms': None,
           'name': 'AlCl_3',
           'positions': [[0.0, 0.0, 0.0],
                         [0.0, 2.069041, 0.0],
                         [1.791842, -1.03452, 0.0],
                         [-1.791842, -1.03452, 0.0]],
           'symbols': 'AlClClCl',
           'thermal correction': 3.9464},
 'AlF3': {'CAS No.': 7784181,
          'ZPE': 4.8645,
          'charges': None,
          'database': 'G2-2',
          'description': 'AlF3, Planar D3h symm.',
          'enthalpy': -289.0,
          'magmoms': None,
          'name': 'AlF_3',
          'positions': [[0.0, 0.0, 0.0],
                        [0.0, 1.64472, 0.0],
                        [1.424369, -0.82236, 0.0],
                        [-1.424369, -0.82236, 0.0]],
          'symbols': 'AlFFF',
          'thermal correction': 3.3986},
 'B': {'CAS No.': 7440428,
       'charges': None,
       'database': 'G2-2',
       'description': 'B atom',
       'enthalpy': 136.2,
       'ionization energy': 8.3,
       'magmoms': [1.0],
       'name': 'Boron',
       'positions': [[0.0, 0.0, 0.0]],
       'symbols': 'B',
       'thermal correction': 0.29},
 'BCl3': {'CAS No.': 10294345,
          'ZPE': 4.6536,
          'charges': None,
          'database': 'G2-2',
          'description': 'BCl3, Planar D3h symm.',
          'enthalpy': -96.3,
          'magmoms': None,
          'name': 'BCl_3',
          'positions': [[0.0, 0.0, 0.0],
                        [0.0, 1.735352, 0.0],
                        [1.502859, -0.867676, 0.0],
                        [-1.502859, -0.867676, 0.0]],
          'symbols': 'BClClCl',
          'thermal correction': 3.3729},
 'BF3': {'CAS No.': 7637072,
         'ZPE': 7.8257,
         'charges': None,
         'database': 'G2-2',
         'description': 'BF3, Planar D3h symm.',
         'enthalpy': -271.4,
         'magmoms': None,
         'name': 'BF_3',
         'positions': [[0.0, 0.0, 0.0],
                       [0.0, 1.32176, 0.0],
                       [1.144678, -0.66088, 0.0],
                       [-1.144678, -0.66088, 0.0]],
         'symbols': 'BFFF',
         'thermal correction': 2.7893},
 'C': {'CAS No.': 7440440,
       'charges': None,
       'database': 'G2-1',
       'description': 'C atom',
       'enthalpy': 169.98,
       'ionization energy': 11.26,
       'magmoms': [2.0],
       'name': 'Carbon',
       'positions': [[0.0, 0.0, 0.0]],
       'symbols': 'C',
       'thermal correction': 0.25},
 'C2Cl4': {'CAS No.': 127184,
           'ZPE': 9.4628,
           'charges': None,
           'database': 'G2-2',
           'description': 'C2Cl4 (Cl2C=CCl2), D2h symm.',
           'enthalpy': -3.0,
           'magmoms': None,
           'name': 'C_2Cl_4',
           'positions': [[0.0, 0.0, 0.675402],
                         [0.0, 0.0, -0.675402],
                         [0.0, 1.448939, 1.589701],
                         [0.0, -1.448939, 1.589701],
                         [0.0, -1.448939, -1.589701],
                         [0.0, 1.448939, -1.589701]],
           'symbols': 'CCClClClCl',
           'thermal correction': 4.7132},
 'C2F4': {'CAS No.': 116143,
          'ZPE': 13.4118,
          'charges': None,
          'database': 'G2-2',
          'description': 'C2F4 (F2C=CF2), D2H symm.',
          'enthalpy': -157.4,
          'magmoms': None,
          'name': 'C_2F_4',
          'positions': [[0.0, 0.0, 0.66323],
                        [0.0, 0.0, -0.66323],
                        [0.0, 1.112665, 1.385652],
                        [0.0, -1.112665, 1.385652],
                        [0.0, 1.112665, -1.385652],
                        [0.0, -1.112665, -1.385652]],
          'symbols': 'CCFFFF',
          'thermal correction': 3.9037},
 'C2H3': {'CAS No.': 2669898,
          'ZPE': 22.5747,
          'charges': None,
          'database': 'G2-2',
          'description': "C2H3 radical, Cs symm, 2-A'.",
          'enthalpy': 71.6,
          'magmoms': [0.0, 1.0, 0.0, 0.0, 0.0],
          'name': 'C_2H_3 (2A)(vinyl)',
          'positions': [[0.049798, -0.576272, 0.0],
                        [0.049798, 0.710988, 0.0],
                        [-0.87675, -1.151844, 0.0],
                        [0.969183, -1.154639, 0.0],
                        [-0.690013, 1.498185, 0.0]],
          'symbols': 'CCHHH',
          'thermal correction': 2.5483},
 'C2H5': {'CAS No.': 2025561,
          'ZPE': 36.5675,
          'charges': None,
          'database': 'G2-2',
          'description': "C2H5 radical, Staggered, Cs symm, 2-A'.",
          'enthalpy': 28.9,
          'magmoms': [0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0],
          'name': 'C_2H_5 (2A)',
          'positions': [[-0.014359, -0.694617, 0.0],
                        [-0.014359, 0.794473, 0.0],
                        [1.006101, -1.104042, 0.0],
                        [-0.517037, -1.093613, 0.884839],
                        [-0.517037, -1.093613, -0.884839],
                        [0.100137, 1.346065, 0.923705],
                        [0.100137, 1.346065, -0.923705]],
          'symbols': 'CCHHHHH',
          'thermal correction': 3.0942},
 'C2H6CHOH': {'CAS No.': 67630,
              'ZPE': 66.5612,
              'charges': None,
              'database': 'G2-2',
              'description': 'Isopropyl alcohol, (CH3)2CH-OH, Gauche isomer, C1 symm.',
              'enthalpy': -65.2,
              'magmoms': None,
              'name': '(CH_3)_2CHOH (isopropanol)',
              'positions': [[0.027191, 1.363691, -0.167516],
                            [-0.000926, 0.036459, 0.370128],
                            [0.859465, 1.775647, 0.121307],
                            [0.007371, 0.082145, 1.470506],
                            [-1.313275, -0.563514, -0.088979],
                            [1.200721, -0.76448, -0.10492],
                            [-1.334005, -0.607253, -1.181009],
                            [1.202843, -0.807817, -1.197189],
                            [-2.147812, 0.054993, 0.247676],
                            [2.136462, -0.299324, 0.223164],
                            [-1.438709, -1.574275, 0.30834],
                            [1.177736, -1.784436, 0.289967]],
              'symbols': 'OCHHCCHHHHHH',
              'thermal correction': 4.0732},
 'C2H6NH': {'CAS No.': 124403,
            'ZPE': 57.0287,
            'charges': None,
            'database': 'G2-2',
            'description': 'Dimethylamine, (CH3)2NH, Cs symm.',
            'enthalpy': -4.4,
            'magmoms': None,
            'name': '(CH_3)_2NH (dimethylamine)',
            'positions': [[-0.02753, -0.224702, 1.20488],
                          [-0.02753, 0.59247, 0.0],
                          [-0.02753, -0.224702, -1.20488],
                          [0.791501, -0.962742, 1.248506],
                          [0.039598, 0.421182, 2.083405],
                          [-0.97222, -0.772987, 1.26175],
                          [0.805303, 1.17822, 0.0],
                          [0.791501, -0.962742, -1.248506],
                          [0.039598, 0.421182, -2.083405],
                          [-0.97222, -0.772987, -1.26175]],
            'symbols': 'CNCHHHHHHH',
            'thermal correction': 3.376},
 'C2H6SO': {'CAS No.': 67685,
            'ZPE': 48.8479,
            'charges': None,
            'database': 'G2-2',
            'description': 'Dimethylsulfoxide (CH3)2SO, Cs symm.',
            'enthalpy': -36.2,
            'magmoms': None,
            'name': '(CH_3)_2SO (dimethyl sulfoxide)',
            'positions': [[2e-06, 0.231838, -0.438643],
                          [2e-05, 1.500742, 0.379819],
                          [1.339528, -0.809022, 0.180717],
                          [-1.339548, -0.808992, 0.180718],
                          [1.255835, -0.896385, 1.266825],
                          [-2.279404, -0.313924, -0.068674],
                          [1.304407, -1.793327, -0.292589],
                          [2.279395, -0.313974, -0.068674],
                          [-1.304447, -1.793298, -0.292587],
                          [-1.255857, -0.896355, 1.266826]],
            'symbols': 'SOCCHHHHHH',
            'thermal correction': 4.1905},
 'C3H4_C2v': {'CAS No.': 2781853,
              'ZPE': 34.7603,
              'charges': None,
              'database': 'G2-2',
              'description': 'Cyclopropene (C3H4), C2v symm.',
              'enthalpy': 66.2,
              'magmoms': None,
              'name': 'C_3H_4 (cyclopropene)',
              'positions': [[0.0, 0.0, 0.858299],
                            [0.0, -0.650545, -0.498802],
                            [0.0, 0.650545, -0.498802],
                            [0.912438, 0.0, 1.456387],
                            [-0.912438, 0.0, 1.456387],
                            [0.0, -1.584098, -1.038469],
                            [0.0, 1.584098, -1.038469]],
              'symbols': 'CCCHHHH',
              'thermal correction': 2.6763},
 'C3H4_C3v': {'CAS No.': 74997,
              'ZPE': 34.2614,
              'charges': None,
              'database': 'G2-2',
              'description': 'Propyne (C3H4), C3v symm.',
              'enthalpy': 44.2,
              'magmoms': None,
              'name': 'CH_3CCH (propyne)',
              'positions': [[0.0, 0.0, 0.214947],
                            [0.0, 0.0, 1.43313],
                            [0.0, 0.0, -1.246476],
                            [0.0, 0.0, 2.498887],
                            [0.0, 1.021145, -1.636167],
                            [0.884337, -0.510572, -1.636167],
                            [-0.884337, -0.510572, -1.636167]],
              'symbols': 'CCCHHHH',
              'thermal correction': 3.1193},
 'C3H4_D2d': {'CAS No.': 463490,
              'ZPE': 34.1189,
              'charges': None,
              'database': 'G2-2',
              'description': 'Allene (C3H4), D2d symm.',
              'enthalpy': 45.5,
              'magmoms': None,
              'name': 'CH_2:C:CH_2 (allene)',
              'positions': [[0.0, 0.0, 0.0],
                            [0.0, 0.0, 1.31119],
                            [0.0, 0.0, -1.31119],
                            [0.0, 0.926778, 1.876642],
                            [0.0, -0.926778, 1.876642],
                            [0.926778, 0.0, -1.876642],
                            [-0.926778, 0.0, -1.876642]],
              'symbols': 'CCCHHHH',
              'thermal correction': 2.9744},
 'C3H6_Cs': {'CAS No.': 115071,
             'ZPE': 49.1836,
             'charges': None,
             'database': 'G2-2',
             'description': 'Propene (C3H6), Cs symm.',
             'enthalpy': 4.8,
             'magmoms': None,
             'name': 'CH_3CH:CH_2 (propylene)',
             'positions': [[1.29129, 0.133682, 0.0],
                           [0.0, 0.479159, 0.0],
                           [1.60116, -0.90742, 0.0],
                           [2.0808, 0.877337, 0.0],
                           [-0.263221, 1.536098, 0.0],
                           [-1.139757, -0.492341, 0.0],
                           [-0.776859, -1.523291, 0.0],
                           [-1.77554, -0.352861, 0.88042],
                           [-1.77554, -0.352861, -0.88042]],
             'symbols': 'CCHHHCHHH',
             'thermal correction': 3.1727},
 'C3H6_D3h': {'CAS No.': 75194,
              'ZPE': 50.2121,
              'charges': None,
              'database': 'G2-2',
              'description': 'Cyclopropane (C3H6), D3h symm.',
              'enthalpy': 12.7,
              'magmoms': None,
              'name': 'C_3H_6 (cyclopropane)',
              'positions': [[0.0, 0.866998, 0.0],
                            [0.750842, -0.433499, 0.0],
                            [-0.750842, -0.433499, 0.0],
                            [0.0, 1.455762, 0.910526],
                            [0.0, 1.455762, -0.910526],
                            [1.260727, -0.727881, -0.910526],
                            [1.260727, -0.727881, 0.910526],
                            [-1.260727, -0.727881, 0.910526],
                            [-1.260727, -0.727881, -0.910526]],
              'symbols': 'CCCHHHHHH',
              'thermal correction': 2.7272},
 'C3H7': {'CAS No.': 2025550,
          'ZPE': 54.2928,
          'charges': None,
          'database': 'G2-2',
          'description': "(CH3)2CH radical, Cs symm, 2-A'.",
          'enthalpy': 21.5,
          'magmoms': [1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
          'name': '(CH_3)_2CH (2A)',
          'positions': [[0.014223, 0.54385, 0.0],
                        [0.014223, -0.199742, 1.291572],
                        [0.014223, -0.199742, -1.291572],
                        [-0.32289, 1.575329, 0.0],
                        [0.221417, 0.459174, 2.138477],
                        [0.221417, 0.459174, -2.138477],
                        [-0.955157, -0.684629, 1.484633],
                        [0.767181, -0.995308, 1.286239],
                        [0.767181, -0.995308, -1.286239],
                        [-0.955157, -0.684629, -1.484633]],
          'symbols': 'CCCHHHHHHH',
          'thermal correction': 3.8435},
 'C3H7Cl': {'CAS No.': 540545,
            'ZPE': 58.6696,
            'charges': None,
            'database': 'G2-2',
            'description': 'Propyl chloride (CH3CH2CH2Cl), Cs symm.',
            'enthalpy': -31.5,
            'magmoms': None,
            'name': 'CH_3CH_2CH_2Cl (propyl chloride)',
            'positions': [[0.892629, -0.642344, 0.0],
                          [2.365587, -0.245168, 0.0],
                          [0.0, 0.582921, 0.0],
                          [0.663731, -1.252117, 0.879201],
                          [0.663731, -1.252117, -0.879201],
                          [3.005476, -1.130924, 0.0],
                          [-1.73281, 0.139743, 0.0],
                          [2.614882, 0.347704, -0.88473],
                          [2.614882, 0.347704, 0.88473],
                          [0.172881, 1.195836, 0.88646],
                          [0.172881, 1.195836, -0.88646]],
            'symbols': 'CCCHHHClHHHH',
            'thermal correction': 3.9885},
 'C3H8': {'CAS No.': 74986,
          'ZPE': 63.8008,
          'charges': None,
          'database': 'G2-2',
          'description': 'Propane (C3H8), C2v symm.',
          'enthalpy': -25.0,
          'magmoms': None,
          'name': 'C_3H_8 (propane)',
          'positions': [[0.0, 0.0, 0.587716],
                        [0.0, 1.266857, -0.260186],
                        [0.0, -1.266857, -0.260186],
                        [-0.876898, 0.0, 1.244713],
                        [0.876898, 0.0, 1.244713],
                        [0.0, 2.16615, 0.362066],
                        [0.0, -2.16615, 0.362066],
                        [0.883619, 1.304234, -0.904405],
                        [-0.883619, 1.304234, -0.904405],
                        [-0.883619, -1.304234, -0.904405],
                        [0.883619, -1.304234, -0.904405]],
          'symbols': 'CCCHHHHHHHH',
          'thermal correction': 3.4632},
 'C3H9C': {'CAS No.': 1605738,
           'ZPE': 71.7833,
           'charges': None,
           'database': 'G2-2',
           'description': 't-Butyl radical, (CH3)3C, C3v symm.',
           'enthalpy': 12.3,
           'magmoms': [1.0,
                       0.0,
                       0.0,
                       0.0,
                       0.0,
                       0.0,
                       0.0,
                       0.0,
                       0.0,
                       0.0,
                       0.0,
                       0.0,
                       0.0],
           'name': '(CH_3)_3C (t-butyl radical)',
           'positions': [[0.0, 0.0, 0.191929],
                         [0.0, 1.478187, -0.020866],
                         [1.280147, -0.739093, -0.020866],
                         [-1.280147, -0.739093, -0.020866],
                         [0.0, 1.731496, -1.093792],
                         [-0.887043, 1.945769, 0.417565],
                         [0.887043, 1.945769, 0.417565],
                         [1.49952, -0.865748, -1.093792],
                         [2.128607, -0.204683, 0.417565],
                         [1.241564, -1.741086, 0.417565],
                         [-1.49952, -0.865748, -1.093792],
                         [-1.241564, -1.741086, 0.417565],
                         [-2.128607, -0.204683, 0.417565]],
           'symbols': 'CCCCHHHHHHHHH',
           'thermal correction': 4.6662},
 'C3H9N': {'CAS No.': 75503,
           'ZPE': 74.1584,
           'charges': None,
           'database': 'G2-2',
           'description': 'Trimethyl Amine, (CH3)3N, C3v symm.',
           'enthalpy': -5.7,
           'magmoms': None,
           'name': '(CH_3)_3N (trimethylamine)',
           'positions': [[0.0, 0.0, 0.395846],
                         [0.0, 1.378021, -0.065175],
                         [1.193401, -0.689011, -0.065175],
                         [-1.193401, -0.689011, -0.065175],
                         [0.0, 1.461142, -1.167899],
                         [0.886156, 1.891052, 0.317655],
                         [-0.886156, 1.891052, 0.317655],
                         [1.265386, -0.730571, -1.167899],
                         [1.194621, -1.71296, 0.317655],
                         [2.080777, -0.178092, 0.317655],
                         [-1.265386, -0.730571, -1.167899],
                         [-2.080777, -0.178092, 0.317655],
                         [-1.194621, -1.71296, 0.317655]],
           'symbols': 'NCCCHHHHHHHHH',
           'thermal correction': 4.0631},
 'C4H4NH': {'CAS No.': 109977,
            'ZPE': 50.9688,
            'charges': None,
            'database': 'G2-2',
            'description': 'Pyrrole (Planar cyclic C4H4NH), C2v symm.',
            'enthalpy': 25.9,
            'magmoms': None,
            'name': 'C_4H_5N (pyrrole)',
            'positions': [[0.0, 0.0, 2.129296],
                          [0.0, 0.0, 1.118684],
                          [0.0, 1.124516, 0.333565],
                          [0.0, -1.124516, 0.333565],
                          [0.0, 0.708407, -0.983807],
                          [0.0, -0.708407, -0.983807],
                          [0.0, 2.112872, 0.770496],
                          [0.0, -2.112872, 0.770496],
                          [0.0, 1.357252, -1.849085],
                          [0.0, -1.357252, -1.849085]],
            'symbols': 'HNCCCCHHHH',
            'thermal correction': 3.1156},
 'C4H4O': {'CAS No.': 110009,
           'ZPE': 43.2116,
           'charges': None,
           'database': 'G2-2',
           'description': 'Furan (cyclic C4H4O), C2v symm.',
           'enthalpy': -8.3,
           'magmoms': None,
           'name': 'C_4H_4O (furan)',
           'positions': [[0.0, 0.0, 1.163339],
                         [0.0, 1.0947, 0.348039],
                         [0.0, -1.0947, 0.348039],
                         [0.0, 0.7132, -0.962161],
                         [0.0, -0.7132, -0.962161],
                         [0.0, 2.049359, 0.851113],
                         [0.0, -2.049359, 0.851113],
                         [0.0, 1.370828, -1.819738],
                         [0.0, -1.370828, -1.819738]],
           'symbols': 'OCCCCHHHH',
           'thermal correction': 2.948},
 'C4H4S': {'CAS No.': 110021,
           'ZPE': 41.2029,
           'charges': None,
           'database': 'G2-2',
           'description': 'Thiophene (cyclic C4H4S), C2v symm.',
           'enthalpy': 27.5,
           'magmoms': None,
           'name': 'C_4H_4S (thiophene)',
           'positions': [[0.0, 0.0, 1.189753],
                         [0.0, 1.233876, -0.001474],
                         [0.0, -1.233876, -0.001474],
                         [0.0, 0.709173, -1.272322],
                         [0.0, -0.709173, -1.272322],
                         [0.0, 2.275343, 0.291984],
                         [0.0, -2.275343, 0.291984],
                         [0.0, 1.321934, -2.167231],
                         [0.0, -1.321934, -2.167231]],
           'symbols': 'SCCCCHHHH',
           'thermal correction': 3.1702},
 'C5H5N': {'CAS No.': 110861,
           'ZPE': 54.823,
           'charges': None,
           'database': 'G2-2',
           'description': 'Pyridine (cyclic C5H5N), C2v symm.',
           'enthalpy': 33.6,
           'magmoms': None,
           'name': 'C_5H_5N (pyridine)',
           'positions': [[0.0, 0.0, 1.424672],
                         [0.0, 0.0, -1.386178],
                         [0.0, 1.144277, 0.720306],
                         [0.0, -1.144277, 0.720306],
                         [0.0, -1.196404, -0.672917],
                         [0.0, 1.196404, -0.672917],
                         [0.0, 0.0, -2.473052],
                         [0.0, 2.060723, 1.307477],
                         [0.0, -2.060723, 1.307477],
                         [0.0, -2.155293, -1.183103],
                         [0.0, 2.155293, -1.183103]],
           'symbols': 'NCCCCCHHHHH',
           'thermal correction': 3.3007},
 'C5H8': {'CAS No.': 157404,
          'ZPE': 70.9964,
          'charges': None,
          'database': 'G2-2',
          'description': 'Spiropentane (C5H8), D2d symm.',
          'enthalpy': 44.3,
          'magmoms': None,
          'name': 'C_5H_8 (spiropentane)',
          'positions': [[0.0, 0.0, 0.0],
                        [0.0, 0.762014, 1.265752],
                        [0.0, -0.762014, 1.265752],
                        [0.762014, 0.0, -1.265752],
                        [-0.762014, 0.0, -1.265752],
                        [-0.914023, 1.265075, 1.56809],
                        [0.914023, 1.265075, 1.56809],
                        [-0.914023, -1.265075, 1.56809],
                        [0.914023, -1.265075, 1.56809],
                        [1.265075, -0.914023, -1.56809],
                        [1.265075, 0.914023, -1.56809],
                        [-1.265075, -0.914023, -1.56809],
                        [-1.265075, 0.914023, -1.56809]],
          'symbols': 'CCCCCHHHHHHHH',
          'thermal correction': 3.7149},
 'C6H6': {'CAS No.': 71432,
          'ZPE': 61.9252,
          'charges': None,
          'database': 'G2-2',
          'description': 'Benzene (C6H6), D6h symm.',
          'enthalpy': 19.7,
          'ionization energy': 9.24,
          'magmoms': None,
          'name': 'C_6H_6 (benzene)',
          'positions': [[0.0, 1.395248, 0.0],
                        [1.20832, 0.697624, 0.0],
                        [1.20832, -0.697624, 0.0],
                        [0.0, -1.395248, 0.0],
                        [-1.20832, -0.697624, 0.0],
                        [-1.20832, 0.697624, 0.0],
                        [0.0, 2.48236, 0.0],
                        [2.149787, 1.24118, 0.0],
                        [2.149787, -1.24118, 0.0],
                        [0.0, -2.48236, 0.0],
                        [-2.149787, -1.24118, 0.0],
                        [-2.149787, 1.24118, 0.0]],
          'symbols': 'CCCCCCHHHHHH',
          'thermal correction': 3.3886,
          'vertical ionization energy': 9.25},
 'CCH': {'CAS No.': 2122487,
         'ZPE': 7.8533,
         'charges': None,
         'database': 'G2-2',
         'description': 'CCH radical, C*v symm.',
         'enthalpy': 135.1,
         'magmoms': [0.0, 1.0, 0.0],
         'name': 'CCH(Ethynyl radical)',
         'positions': [[0.0, 0.0, -0.462628],
                       [0.0, 0.0, 0.717162],
                       [0.0, 0.0, -1.527198]],
         'symbols': 'CCH',
         'thermal correction': 2.783},
 'CCl4': {'CAS No.': 56235,
          'ZPE': 5.7455,
          'charges': None,
          'database': 'G2-2',
          'description': 'CCl4, Td symm.',
          'enthalpy': -22.9,
          'magmoms': None,
          'name': 'CCl_4',
          'positions': [[0.0, 0.0, 0.0],
                        [1.02134, 1.02134, 1.02134],
                        [-1.02134, -1.02134, 1.02134],
                        [-1.02134, 1.02134, -1.02134],
                        [1.02134, -1.02134, -1.02134]],
          'symbols': 'CClClClCl',
          'thermal correction': 4.1754},
 'CF3CN': {'CAS No.': 353855,
           'ZPE': 14.102,
           'charges': None,
           'database': 'G2-2',
           'description': 'CF3CN, C3v symm.',
           'enthalpy': -118.4,
           'magmoms': None,
           'name': 'CF_3CN',
           'positions': [[0.0, 0.0, -0.32635],
                         [0.0, 0.0, 1.15083],
                         [0.0, 1.257579, -0.787225],
                         [1.089096, -0.62879, -0.787225],
                         [-1.089096, -0.62879, -0.787225],
                         [0.0, 0.0, 2.329741]],
           'symbols': 'CCFFFN',
           'thermal correction': 3.7996},
 'CF4': {'CAS No.': 75730,
         'ZPE': 10.5999,
         'charges': None,
         'database': 'G2-2',
         'description': 'CF4, Td symm.',
         'enthalpy': -223.0,
         'magmoms': None,
         'name': 'CF_4(Carbon tetrafluoride)',
         'positions': [[0.0, 0.0, 0.0],
                       [0.767436, 0.767436, 0.767436],
                       [-0.767436, -0.767436, 0.767436],
                       [-0.767436, 0.767436, -0.767436],
                       [0.767436, -0.767436, -0.767436]],
         'symbols': 'CFFFF',
         'thermal correction': 3.0717},
 'CH2NHCH2': {'CAS No.': 151564,
              'ZPE': 43.3728,
              'charges': None,
              'database': 'G2-2',
              'description': 'Aziridine (cyclic CH2-NH-CH2 ring), C2v symm.',
              'enthalpy': 30.2,
              'magmoms': None,
              'name': 'C_2H_4NH (aziridine)',
              'positions': [[-0.03845, -0.397326, 0.739421],
                            [-0.03845, 0.875189, 0.0],
                            [-0.03845, -0.397326, -0.739421],
                            [0.903052, 1.268239, 0.0],
                            [-0.955661, -0.604926, 1.280047],
                            [-0.955661, -0.604926, -1.280047],
                            [0.869409, -0.708399, 1.249033],
                            [0.869409, -0.708399, -1.249033]],
              'symbols': 'CNCHHHHH',
              'thermal correction': 2.6399},
 'CH2OCH2': {'CAS No.': 75218,
             'ZPE': 35.4204,
             'charges': None,
             'database': 'G2-2',
             'description': 'Oxirane (cyclic CH2-O-CH2 ring), C2v symm.',
             'enthalpy': -12.6,
             'magmoms': None,
             'name': 'C_2H_4O (Oxirane)',
             'positions': [[0.0, 0.73158, -0.375674],
                           [0.0, 0.0, 0.86095],
                           [0.0, -0.73158, -0.375674],
                           [0.919568, 1.268821, -0.594878],
                           [-0.919568, 1.268821, -0.594878],
                           [-0.919568, -1.268821, -0.594878],
                           [0.919568, -1.268821, -0.594878]],
             'symbols': 'COCHHHH',
             'thermal correction': 2.5816},
 'CH2SCH2': {'CAS No.': 420122,
             'ZPE': 33.9483,
             'charges': None,
             'database': 'G2-2',
             'description': 'Thiooxirane (cyclic CH2-S-CH2 ring), C2v symm.',
             'enthalpy': 19.6,
             'magmoms': None,
             'name': 'C_2H_4S (Thiirane)',
             'positions': [[0.0, -0.739719, -0.792334],
                           [0.0, 0.0, 0.863474],
                           [0.0, 0.739719, -0.792334],
                           [-0.91394, -1.250142, -1.076894],
                           [0.91394, -1.250142, -1.076894],
                           [0.91394, 1.250142, -1.076894],
                           [-0.91394, 1.250142, -1.076894]],
             'symbols': 'CSCHHHH',
             'thermal correction': 2.729},
 'CH3CH2Cl': {'CAS No.': 75003,
              'ZPE': 41.0686,
              'charges': None,
              'database': 'G2-2',
              'description': 'Ethyl chloride (CH3-CH2-Cl), Cs symm.',
              'enthalpy': -26.8,
              'magmoms': None,
              'name': 'C_2H_5Cl (ethyl chloride)',
              'positions': [[0.0, 0.807636, 0.0],
                            [1.505827, 0.647832, 0.0],
                            [-0.823553, -0.77997, 0.0],
                            [-0.344979, 1.341649, 0.885248],
                            [-0.344979, 1.341649, -0.885248],
                            [1.976903, 1.634877, 0.0],
                            [1.839246, 0.10425, 0.885398],
                            [1.839246, 0.10425, -0.885398]],
              'symbols': 'CCClHHHHH',
              'thermal correction': 3.1488},
 'CH3CH2NH2': {'CAS No.': 75047,
               'ZPE': 57.242,
               'charges': None,
               'database': 'G2-2',
               'description': 'Trans-Ethylamine (CH3-CH2-NH2), Cs symm.',
               'enthalpy': -11.3,
               'magmoms': None,
               'name': 'CH_3CH_2NH_2 (trans ethylamine)',
               'positions': [[1.210014, -0.353598, 0.0],
                             [0.0, 0.575951, 0.0],
                             [-1.305351, -0.087478, 0.0],
                             [2.14931, 0.208498, 0.0],
                             [1.201796, -0.99776, 0.884909],
                             [1.201796, -0.99776, -0.884909],
                             [0.034561, 1.230963, -0.876478],
                             [0.034561, 1.230963, 0.876478],
                             [-1.372326, -0.69834, 0.813132],
                             [-1.372326, -0.69834, -0.813132]],
               'symbols': 'CCNHHHHHHH',
               'thermal correction': 3.3678},
 'CH3CH2O': {'CAS No.': 2154509,
             'ZPE': 39.444,
             'charges': None,
             'database': 'G2-2',
             'description': "CH3CH2O radical, Cs symm, 2-A''.",
             'enthalpy': -3.7,
             'magmoms': [0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0],
             'name': 'CH_3CH_2O (Ethoxy radical)',
             'positions': [[1.004757, -0.568263, 0.0],
                           [0.0, 0.588691, 0.0],
                           [-1.260062, 0.000729, 0.0],
                           [0.146956, 1.204681, 0.896529],
                           [0.146956, 1.204681, -0.896529],
                           [2.019363, -0.1641, 0.0],
                           [0.86934, -1.186832, 0.888071],
                           [0.86934, -1.186832, -0.888071]],
             'symbols': 'CCOHHHHH',
             'thermal correction': 3.0158},
 'CH3CH2OCH3': {'CAS No.': 540670,
                'ZPE': 66.6936,
                'charges': None,
                'database': 'G2-2',
                'description': 'Methyl ethyl ether (CH3-CH2-O-CH3), Trans, Cs symm.',
                'enthalpy': -51.7,
                'magmoms': None,
                'name': 'C_2H_5OCH_3 (Methoxyethane)',
                'positions': [[0.006429, -0.712741, 0.0],
                              [0.0, 0.705845, 0.0],
                              [1.324518, -1.226029, 0.0],
                              [-1.442169, 1.160325, 0.0],
                              [0.530962, 1.086484, 0.886881],
                              [0.530962, 1.086484, -0.886881],
                              [1.241648, -2.313325, 0.0],
                              [1.881329, -0.905925, -0.89171],
                              [1.881329, -0.905925, 0.89171],
                              [-1.954863, 0.780605, -0.885855],
                              [-1.954863, 0.780605, 0.885855],
                              [-1.502025, 2.252083, 0.0]],
                'symbols': 'OCCCHHHHHHHH',
                'thermal correction': 4.1058},
 'CH3CH2OH': {'CAS No.': 64175,
              'ZPE': 49.3072,
              'charges': None,
              'database': 'G2-2',
              'description': 'Ethanol (trans, CH3CH2OH), Cs symm.',
              'enthalpy': -56.2,
              'magmoms': None,
              'name': 'CH_3CH_2OH (ethanol)',
              'positions': [[1.168181, -0.400382, 0.0],
                            [0.0, 0.559462, 0.0],
                            [-1.190083, -0.227669, 0.0],
                            [-1.946623, 0.381525, 0.0],
                            [0.042557, 1.207508, 0.886933],
                            [0.042557, 1.207508, -0.886933],
                            [2.115891, 0.1448, 0.0],
                            [1.128599, -1.037234, 0.885881],
                            [1.128599, -1.037234, -0.885881]],
              'symbols': 'CCOHHHHHH',
              'thermal correction': 3.3252},
 'CH3CH2SH': {'CAS No.': 75081,
              'ZPE': 46.1583,
              'charges': None,
              'database': 'G2-2',
              'description': 'ThioEthanol (CH3-CH2-SH), Cs symm.',
              'enthalpy': -11.1,
              'magmoms': None,
              'name': 'C_2H_5SH (ethanethiol)',
              'positions': [[1.514343, 0.679412, 0.0],
                            [0.0, 0.826412, 0.0],
                            [-0.756068, -0.831284, 0.0],
                            [-2.035346, -0.427738, 0.0],
                            [-0.32497, 1.376482, 0.885793],
                            [-0.32497, 1.376482, -0.885793],
                            [1.986503, 1.665082, 0.0],
                            [1.854904, 0.137645, 0.885494],
                            [1.854904, 0.137645, -0.885494]],
              'symbols': 'CCSHHHHHH',
              'thermal correction': 3.59},
 'CH3CHO': {'CAS No.': 75070,
            'ZPE': 34.2288,
            'charges': None,
            'database': 'G2-2',
            'description': 'Acetaldehyde (CH3CHO), Cs symm.',
            'enthalpy': -39.7,
            'magmoms': None,
            'name': 'CH_3CHO (acetaldehyde)',
            'positions': [[1.218055, 0.36124, 0.0],
                          [0.0, 0.464133, 0.0],
                          [-0.477241, 1.465295, 0.0],
                          [-0.948102, -0.700138, 0.0],
                          [-0.385946, -1.634236, 0.0],
                          [-1.596321, -0.652475, 0.880946],
                          [-1.596321, -0.652475, -0.880946]],
            'symbols': 'OCHCHHH',
            'thermal correction': 3.0428},
 'CH3CN': {'CAS No.': 75058,
           'ZPE': 28.0001,
           'charges': None,
           'database': 'G2-2',
           'description': 'Acetonitrile (CH3-CN), C3v symm.',
           'enthalpy': 18.0,
           'magmoms': None,
           'name': 'CH_3CN (methyl cyanide)',
           'positions': [[0.0, 0.0, -1.18693],
                         [0.0, 0.0, 0.273874],
                         [0.0, 0.0, 1.452206],
                         [0.0, 1.024986, -1.56237],
                         [0.887664, -0.512493, -1.56237],
                         [-0.887664, -0.512493, -1.56237]],
           'symbols': 'CCNHHH',
           'thermal correction': 2.8552},
 'CH3CO': {'CAS No.': 3170692,
           'ZPE': 26.607,
           'charges': None,
           'database': 'G2-2',
           'description': "CH3CO radical, HCCO cis, Cs symm, 2-A'.",
           'enthalpy': -2.4,
           'magmoms': [0.1, 0.6, 0.0, 0.0, 0.0, 0.3],
           'name': 'CH_3CO (2A)',
           'positions': [[-0.978291, -0.647814, 0.0],
                         [0.0, 0.506283, 0.0],
                         [-0.455551, -1.607837, 0.0],
                         [-1.617626, -0.563271, 0.881061],
                         [-1.617626, -0.563271, -0.881061],
                         [1.195069, 0.447945, 0.0]],
           'symbols': 'CCHHHO',
           'thermal correction': 3.0842},
 'CH3COCH3': {'CAS No.': 67641,
              'ZPE': 51.5587,
              'charges': None,
              'database': 'G2-2',
              'description': 'Acetone (CH3-CO-CH3), C2v symm.',
              'enthalpy': -51.9,
              'magmoms': None,
              'name': 'CH_3COCH_3 (acetone)',
              'positions': [[0.0, 0.0, 1.405591],
                            [0.0, 0.0, 0.17906],
                            [0.0, 1.28549, -0.616342],
                            [0.0, -1.28549, -0.616342],
                            [0.0, 2.134917, 0.066535],
                            [0.0, -2.134917, 0.066535],
                            [-0.881086, 1.331548, -1.264013],
                            [0.881086, 1.331548, -1.264013],
                            [0.881086, -1.331548, -1.264013],
                            [-0.881086, -1.331548, -1.264013]],
              'symbols': 'OCCCHHHHHH',
              'thermal correction': 3.9878},
 'CH3COCl': {'CAS No.': 75365,
             'ZPE': 29.1855,
             'charges': None,
             'database': 'G2-2',
             'description': 'Acetyl,Chloride (CH3COCl), HCCO cis, Cs symm.',
             'enthalpy': -58.0,
             'magmoms': None,
             'name': 'CH_3COCl (acetyl chloride)',
             'positions': [[0.0, 0.523878, 0.0],
                           [1.486075, 0.716377, 0.0],
                           [-0.452286, -1.217999, 0.0],
                           [-0.845539, 1.37494, 0.0],
                           [1.701027, 1.784793, 0.0],
                           [1.917847, 0.240067, 0.882679],
                           [1.917847, 0.240067, -0.882679]],
             'symbols': 'CCClOHHH',
             'thermal correction': 3.5235},
 'CH3COF': {'CAS No.': 557993,
            'ZPE': 30.2742,
            'charges': None,
            'database': 'G2-2',
            'description': 'Acetyl fluoride (CH3COF), HCCO cis, Cs symm.',
            'enthalpy': -105.7,
            'magmoms': None,
            'name': 'CH_3COF (acetyl fluoride)',
            'positions': [[0.0, 0.186396, 0.0],
                          [0.126651, 1.377199, 0.0],
                          [-1.24395, -0.382745, 0.0],
                          [1.049454, -0.876224, 0.0],
                          [2.035883, -0.417099, 0.0],
                          [0.924869, -1.508407, 0.881549],
                          [0.924869, -1.508407, -0.881549]],
            'symbols': 'COFCHHH',
            'thermal correction': 3.3126},
 'CH3CONH2': {'CAS No.': 60355,
              'ZPE': 45.2566,
              'charges': None,
              'database': 'G2-2',
              'description': 'Acetamide (CH3CONH2), C1 symm.',
              'enthalpy': -57.0,
              'magmoms': None,
              'name': 'CH_3CONH_2 (acetamide)',
              'positions': [[0.424546, 1.327024, 0.008034],
                            [0.077158, 0.149789, -0.004249],
                            [0.985518, -0.878537, -0.04891],
                            [-1.371475, -0.288665, -0.000144],
                            [0.707952, -1.824249, 0.169942],
                            [-1.997229, 0.584922, -0.175477],
                            [-1.560842, -1.03927, -0.771686],
                            [-1.632113, -0.723007, 0.969814],
                            [1.953133, -0.631574, 0.111866]],
              'symbols': 'OCNCHHHHH',
              'thermal correction': 3.9313},
 'CH3COOH': {'CAS No.': 64197,
             'ZPE': 38.167,
             'charges': None,
             'database': 'G2-2',
             'description': 'Acetic Acid (CH3COOH), Single bonds trans, Cs symm.',
             'enthalpy': -103.4,
             'magmoms': None,
             'name': 'CH_3COOH (acetic acid)',
             'positions': [[0.0, 0.15456, 0.0],
                           [0.166384, 1.360084, 0.0],
                           [-1.236449, -0.415036, 0.0],
                           [-1.867646, 0.333582, 0.0],
                           [1.073776, -0.892748, 0.0],
                           [2.048189, -0.408135, 0.0],
                           [0.968661, -1.528353, 0.881747],
                           [0.968661, -1.528353, -0.881747]],
             'symbols': 'COOHCHHH',
             'thermal correction': 3.477},
 'CH3NO2': {'CAS No.': 75525,
            'ZPE': 30.7568,
            'charges': None,
            'database': 'G2-2',
            'description': 'Nitromethane (CH3-NO2), Cs symm.',
            'enthalpy': -17.8,
            'magmoms': None,
            'name': 'CH_3NO_2 (nitromethane)',
            'positions': [[-0.114282, -1.314565, 0.0],
                          [0.0, 0.16648, 0.0],
                          [0.899565, -1.715256, 0.0],
                          [-0.640921, -1.607212, 0.904956],
                          [-0.640921, -1.607212, -0.904956],
                          [0.066748, 0.728232, -1.103775],
                          [0.066748, 0.728232, 1.103775]],
            'symbols': 'CNHHHOO',
            'thermal correction': 2.7887},
 'CH3O': {'CAS No.': 2143682,
          'ZPE': 22.4215,
          'charges': None,
          'database': 'G2-2',
          'description': "CH3O radical, Cs symm, 2-A'.",
          'enthalpy': 4.1,
          'magmoms': [0.0, 1.0, 0.0, 0.0, 0.0],
          'name': 'CH_3O CS (2A)(Methoxy radical)',
          'positions': [[-0.008618, -0.586475, 0.0],
                        [-0.008618, 0.799541, 0.0],
                        [1.055363, -0.868756, 0.0],
                        [-0.467358, -1.004363, 0.903279],
                        [-0.467358, -1.004363, -0.903279]],
          'symbols': 'COHHH',
          'thermal correction': 2.4969},
 'CH3OCH3': {'CAS No.': 115106,
             'ZPE': 49.1911,
             'charges': None,
             'database': 'G2-2',
             'description': 'DimethylEther (CH3-O-CH3), C2v symm.',
             'enthalpy': -44.0,
             'magmoms': None,
             'name': 'CH_3OCH_3 (dimethylether)',
             'positions': [[0.0, 1.165725, -0.19995],
                           [0.0, 0.0, 0.60011],
                           [0.0, -1.165725, -0.19995],
                           [0.0, 2.017769, 0.480203],
                           [0.891784, 1.21432, -0.840474],
                           [-0.891784, 1.21432, -0.840474],
                           [0.0, -2.017769, 0.480203],
                           [-0.891784, -1.21432, -0.840474],
                           [0.891784, -1.21432, -0.840474]],
             'symbols': 'COCHHHHHH',
             'thermal correction': 3.3139},
 'CH3ONO': {'CAS No.': 624919,
            'ZPE': 29.9523,
            'charges': None,
            'database': 'G2-2',
            'description': 'Methylnitrite (CH3-O-N=O), NOCH trans, ONOC cis, Cs symm.',
            'enthalpy': -15.9,
            'magmoms': None,
            'name': 'CH_3ONO (Methyl nitrite)',
            'positions': [[-1.316208, 0.309247, 0.0],
                          [0.0, 0.896852, 0.0],
                          [-1.985538, 1.166013, 0.0],
                          [-1.464336, -0.304637, 0.890672],
                          [-1.464336, -0.304637, -0.890672],
                          [1.045334, -0.022815, 0.0],
                          [0.686764, -1.178416, 0.0]],
            'symbols': 'COHHHNO',
            'thermal correction': 3.3641},
 'CH3S': {'CAS No.': 7175759,
          'ZPE': 21.9415,
          'charges': None,
          'database': 'G2-2',
          'description': "CH3S radical, Cs symm, 2-A'.",
          'enthalpy': 29.8,
          'magmoms': [0.0, 1.0, 0.0, 0.0, 0.0],
          'name': 'CH_3S (2A)(thiomethoxy)',
          'positions': [[-0.003856, 1.106222, 0.0],
                        [-0.003856, -0.692579, 0.0],
                        [1.043269, 1.427057, 0.0],
                        [-0.479217, 1.508437, 0.895197],
                        [-0.479217, 1.508437, -0.895197]],
          'symbols': 'CSHHH',
          'thermal correction': 2.6054},
 'CH3SCH3': {'CAS No.': 75183,
             'ZPE': 46.676,
             'charges': None,
             'database': 'G2-2',
             'description': 'Dimethyl ThioEther (CH3-S-CH3), C2v symm.',
             'enthalpy': -8.9,
             'magmoms': None,
             'name': 'CH_3SCH_3 (dimethyl sulfide)',
             'positions': [[0.0, 1.366668, -0.513713],
                           [0.0, 0.0, 0.664273],
                           [0.0, -1.366668, -0.513713],
                           [0.0, 2.296687, 0.057284],
                           [0.891644, 1.34568, -1.144596],
                           [-0.891644, 1.34568, -1.144596],
                           [0.0, -2.296687, 0.057284],
                           [-0.891644, -1.34568, -1.144596],
                           [0.891644, -1.34568, -1.144596]],
             'symbols': 'CSCHHHHHH',
             'thermal correction': 3.6929},
 'CH3SiH3': {'CAS No.': 992949,
             'ZPE': 37.6606,
             'charges': None,
             'database': 'G2-2',
             'description': 'Methylsilane (CH3-SiH3), C3v symm.',
             'enthalpy': -7.0,
             'magmoms': None,
             'name': 'CH_3SiH_3 (methyl silane)',
             'positions': [[0.0, 0.0, -1.244466],
                           [0.0, 0.0, 0.635703],
                           [0.0, -1.019762, -1.636363],
                           [-0.88314, 0.509881, -1.636363],
                           [0.88314, 0.509881, -1.636363],
                           [0.0, 1.391234, 1.158682],
                           [-1.204844, -0.695617, 1.158682],
                           [1.204844, -0.695617, 1.158682]],
             'symbols': 'CSiHHHHHH',
             'thermal correction': 3.2486},
 'COF2': {'CAS No.': 353504,
          'ZPE': 8.8215,
          'charges': None,
          'database': 'G2-2',
          'description': 'COF2, C2v symm.',
          'enthalpy': -149.1,
          'magmoms': None,
          'name': 'COF_2 (Carbonic difluoride)',
          'positions': [[0.0, 0.0, 1.330715],
                        [0.0, 0.0, 0.144358],
                        [0.0, 1.06949, -0.639548],
                        [0.0, -1.06949, -0.639548]],
          'symbols': 'OCFF',
          'thermal correction': 2.6619},
 'CS2': {'CAS No.': 75150,
         'ZPE': 4.338,
         'charges': None,
         'database': 'G2-2',
         'description': 'CS2, Linear, D*h symm.',
         'enthalpy': 28.0,
         'magmoms': None,
         'name': 'CS_2',
         'positions': [[0.0, 0.0, 1.561117],
                       [0.0, 0.0, 0.0],
                       [0.0, 0.0, -1.561117]],
         'symbols': 'SCS',
         'thermal correction': 2.5326},
 'Cl': {'CAS No.': 22537151,
        'charges': None,
        'database': 'G2-1',
        'description': 'Cl atom',
        'enthalpy': 28.59,
        'ionization energy': 12.97,
        'magmoms': [1.0],
        'name': 'Chlorine',
        'positions': [[0.0, 0.0, 0.0]],
        'symbols': 'Cl',
        'thermal correction': 1.1},
 'ClF3': {'CAS No.': 7790912,
          'ZPE': 4.2922,
          'charges': None,
          'database': 'G2-2',
          'description': 'ClF3, C2v symm.',
          'enthalpy': -38.0,
          'magmoms': None,
          'name': 'ClF_3',
          'positions': [[0.0, 0.0, 0.376796],
                        [0.0, 0.0, -1.258346],
                        [0.0, 1.714544, 0.27331],
                        [0.0, -1.714544, 0.27331]],
          'symbols': 'ClFFF',
          'thermal correction': 3.3289},
 'ClNO': {'CAS No.': 2696926,
          'ZPE': 4.0619,
          'charges': None,
          'database': 'G2-2',
          'description': 'ClNO, Cs symm.',
          'enthalpy': 12.4,
          'magmoms': None,
          'name': 'ClNO',
          'positions': [[-0.537724, -0.961291, 0.0],
                        [0.0, 0.997037, 0.0],
                        [1.142664, 1.170335, 0.0]],
          'symbols': 'ClNO',
          'thermal correction': 2.7039},
 'F': {'CAS No.': 14762948,
       'charges': None,
       'database': 'G2-1',
       'description': 'F atom',
       'enthalpy': 18.47,
       'ionization energy': 17.42,
       'magmoms': [1.0],
       'name': 'Fluorine',
       'positions': [[0.0, 0.0, 0.0]],
       'symbols': 'F',
       'thermal correction': 1.05},
 'F2O': {'CAS No.': 7783417,
         'ZPE': 3.4362,
         'charges': None,
         'database': 'G2-2',
         'description': 'F2O, C2v symm.',
         'enthalpy': 5.9,
         'magmoms': None,
         'name': 'F_2O',
         'positions': [[0.0, 1.110576, -0.273729],
                       [0.0, 0.0, 0.61589],
                       [0.0, -1.110576, -0.273729]],
         'symbols': 'FOF',
         'thermal correction': 2.5747},
 'H': {'CAS No.': 12385136,
       'charges': None,
       'database': 'G2-1',
       'description': 'H atom',
       'enthalpy': 51.63,
       'ionization energy': 13.6,
       'magmoms': [1.0],
       'name': 'Hydrogen',
       'positions': [[0.0, 0.0, 0.0]],
       'symbols': 'H',
       'thermal correction': 1.01},
 'H2': {'CAS No.': 1333740,
        'ZPE': 6.2908,
        'charges': None,
        'database': 'G2-2',
        'description': 'H2. D*h symm.',
        'enthalpy': 0.0,
        'ionization energy': 15.43,
        'magmoms': None,
        'name': 'H_2',
        'positions': [[0.0, 0.0, 0.368583], [0.0, 0.0, -0.368583]],
        'symbols': 'HH',
        'thermal correction': 2.0739},
 'H2CCHCN': {'CAS No.': 107131,
             'ZPE': 31.4081,
             'charges': None,
             'database': 'G2-2',
             'description': 'CyanoEthylene (H2C=CHCN), Cs symm.',
             'enthalpy': 43.2,
             'magmoms': None,
             'name': 'CH_2:CHCN (acrylonitrile)',
             'positions': [[-0.161594, -1.638625, 0.0],
                           [0.584957, -0.524961, 0.0],
                           [0.0, 0.782253, 0.0],
                           [-1.245203, -1.598169, 0.0],
                           [0.305973, -2.616405, 0.0],
                           [1.669863, -0.572107, 0.0],
                           [-0.467259, 1.867811, 0.0]],
             'symbols': 'CCCHHHN',
             'thermal correction': 3.2034},
 'H2CCHCl': {'CAS No.': 75014,
             'ZPE': 26.3554,
             'charges': None,
             'database': 'G2-2',
             'description': 'Vinyl chloride, H2C=CHCl, Cs symm.',
             'enthalpy': 8.9,
             'magmoms': None,
             'name': 'CH_2:CHCl (Chloroethene)',
             'positions': [[0.0, 0.756016, 0.0],
                           [1.303223, 1.028507, 0.0],
                           [-0.631555, -0.85498, 0.0],
                           [-0.771098, 1.516963, 0.0],
                           [2.056095, 0.249427, 0.0],
                           [1.632096, 2.061125, 0.0]],
             'symbols': 'CCClHHH',
             'thermal correction': 2.8269},
 'H2CCHF': {'CAS No.': 75025,
            'ZPE': 27.2785,
            'charges': None,
            'database': 'G2-2',
            'description': 'Vinyl fluoride (H2C=CHF), Cs symm.',
            'enthalpy': -33.2,
            'magmoms': None,
            'name': 'CH_2:CHF (Ethene, fluoro-)',
            'positions': [[0.0, 0.437714, 0.0],
                          [1.191923, -0.145087, 0.0],
                          [-1.148929, -0.278332, 0.0],
                          [-0.186445, 1.505778, 0.0],
                          [1.291348, -1.222833, 0.0],
                          [2.083924, 0.466279, 0.0]],
            'symbols': 'CCFHHH',
            'thermal correction': 2.7039},
 'H2CCO': {'CAS No.': 463514,
           'ZPE': 19.5984,
           'charges': None,
           'database': 'G2-2',
           'description': 'Ketene (H2C=C=O), C2v symm.',
           'enthalpy': -11.4,
           'magmoms': None,
           'name': 'CH_2CO (ketene)',
           'positions': [[0.0, 0.0, -1.21934],
                         [0.0, 0.0, 0.09892],
                         [0.0, 0.938847, -1.753224],
                         [0.0, -0.938847, -1.753224],
                         [0.0, 0.0, 1.27862]],
           'symbols': 'CCHHO',
           'thermal correction': 2.8075},
 'H2CCl2': {'CAS No.': 75092,
            'ZPE': 18.093,
            'charges': None,
            'database': 'G2-2',
            'description': 'Dichloromethane (H2CCl2), C2v symm.',
            'enthalpy': -22.8,
            'magmoms': None,
            'name': 'CH_2Cl_2 (Methylene chloride)',
            'positions': [[0.0, 0.0, 0.759945],
                          [0.0, 1.4742, -0.215115],
                          [0.0, -1.4742, -0.215115],
                          [-0.894585, 0.0, 1.377127],
                          [0.894585, 0.0, 1.377127]],
            'symbols': 'CClClHH',
            'thermal correction': 2.8527},
 'H2CF2': {'CAS No.': 75105,
           'ZPE': 20.2767,
           'charges': None,
           'database': 'G2-2',
           'description': 'Difluoromethane (H2CF2), C2v symm.',
           'enthalpy': -107.7,
           'magmoms': None,
           'name': 'CH_2F_2 (Methane, difluoro-)',
           'positions': [[0.0, 0.0, 0.502903],
                         [0.0, 1.109716, -0.290601],
                         [0.0, -1.109716, -0.290601],
                         [-0.908369, 0.0, 1.106699],
                         [0.908369, 0.0, 1.106699]],
           'symbols': 'CFFHH',
           'thermal correction': 2.5552},
 'H2COH': {'CAS No.': 2597435,
           'ZPE': 23.1294,
           'charges': None,
           'database': 'G2-2',
           'description': 'H2COH radical, C1 symm.',
           'enthalpy': -4.1,
           'magmoms': [0.7, 0.3, 0.0, 0.0, 0.0],
           'name': 'H_2COH (2A)(Hydroxymethyl radical)',
           'positions': [[0.687448, 0.029626, -0.082014],
                         [-0.672094, -0.125648, 0.030405],
                         [-1.09185, 0.740282, -0.095167],
                         [1.122783, 0.975263, 0.225993],
                         [1.221131, -0.888116, 0.118015]],
           'symbols': 'COHHH',
           'thermal correction': 2.6726},
 'H3CNH2': {'CAS No.': 74895,
            'ZPE': 39.5595,
            'charges': None,
            'database': 'G2-2',
            'description': 'Methylamine (H3C-NH2), Cs symm.',
            'enthalpy': -5.5,
            'magmoms': None,
            'name': 'CH_3NH_2 (methylamine)',
            'positions': [[0.051736, 0.704422, 0.0],
                          [0.051736, -0.759616, 0.0],
                          [-0.941735, 1.176192, 0.0],
                          [-0.458181, -1.099433, 0.81237],
                          [-0.458181, -1.099433, -0.81237],
                          [0.592763, 1.056727, 0.88067],
                          [0.592763, 1.056727, -0.88067]],
            'symbols': 'CNHHHHH',
            'thermal correction': 2.7428},
 'HCCl3': {'CAS No.': 67663,
           'ZPE': 12.1975,
           'charges': None,
           'database': 'G2-2',
           'description': 'Chloroform (HCCl3), C3v symm.',
           'enthalpy': -24.7,
           'magmoms': None,
           'name': 'CHCl_3',
           'positions': [[0.0, 0.0, 0.451679],
                         [0.0, 0.0, 1.537586],
                         [0.0, 1.681723, -0.083287],
                         [1.456415, -0.840862, -0.083287],
                         [-1.456415, -0.840862, -0.083287]],
           'symbols': 'CHClClCl',
           'thermal correction': 3.4262},
 'HCF3': {'CAS No.': 75467,
          'ZPE': 15.7072,
          'charges': None,
          'database': 'G2-2',
          'description': 'Trifluoromethane (HCF3), C3v symm.',
          'enthalpy': -166.6,
          'magmoms': None,
          'name': 'CHF_3 (Methane, trifluoro-)',
          'positions': [[0.0, 0.0, 0.341023],
                        [0.0, 0.0, 1.429485],
                        [0.0, 1.2582, -0.128727],
                        [1.089633, -0.6291, -0.128727],
                        [-1.089633, -0.6291, -0.128727]],
          'symbols': 'CHFFF',
          'thermal correction': 2.7717},
 'HCOOCH3': {'CAS No.': 107313,
             'ZPE': 38.3026,
             'charges': None,
             'database': 'G2-2',
             'description': 'Methyl formate (HCOOCH3), Cs symm.',
             'enthalpy': -85.0,
             'magmoms': None,
             'name': 'HCOOCH_3 (methyl formate)',
             'positions': [[-0.931209, -0.083866, 0.0],
                           [-0.711019, -1.278209, 0.0],
                           [0.0, 0.886841, 0.0],
                           [-1.92836, 0.374598, 0.0],
                           [1.356899, 0.397287, 0.0],
                           [1.980134, 1.288164, 0.0],
                           [1.541121, -0.206172, 0.889397],
                           [1.541121, -0.206172, -0.889397]],
             'symbols': 'COOHCHHH',
             'thermal correction': 3.4726},
 'HCOOH': {'CAS No.': 64186,
           'ZPE': 20.9525,
           'charges': None,
           'database': 'G2-2',
           'description': 'Formic Acid (HCOOH), HOCO cis, Cs symm.',
           'enthalpy': -90.5,
           'magmoms': None,
           'name': 'HCOOH (formic acid)',
           'positions': [[-1.040945, -0.436432, 0.0],
                         [0.0, 0.423949, 0.0],
                         [1.169372, 0.103741, 0.0],
                         [-0.64957, -1.335134, 0.0],
                         [-0.377847, 1.452967, 0.0]],
           'symbols': 'OCOHH',
           'thermal correction': 2.5853},
 'N': {'CAS No.': 17778880,
       'charges': None,
       'database': 'G2-1',
       'description': 'N atom',
       'enthalpy': 112.53,
       'ionization energy': 14.53,
       'magmoms': [3.0],
       'name': 'Nitrogen',
       'positions': [[0.0, 0.0, 0.0]],
       'symbols': 'N',
       'thermal correction': 1.04},
 'N2O': {'CAS No.': 10024972,
         'ZPE': 6.9748,
         'charges': None,
         'database': 'G2-2',
         'description': 'N2O, Cs symm.',
         'enthalpy': 19.6,
         'magmoms': None,
         'name': 'N_2O',
         'positions': [[0.0, 0.0, -1.231969],
                       [0.0, 0.0, -0.060851],
                       [0.0, 0.0, 1.131218]],
         'symbols': 'NNO',
         'thermal correction': 2.271},
 'NCCN': {'CAS No.': 460195,
          'ZPE': 10.2315,
          'charges': None,
          'database': 'G2-2',
          'description': 'Cyanogen (NCCN). D*h symm.',
          'enthalpy': 73.3,
          'magmoms': None,
          'name': 'NCCN (cyanogen)',
          'positions': [[0.0, 0.0, 1.875875],
                        [0.0, 0.0, 0.690573],
                        [0.0, 0.0, -0.690573],
                        [0.0, 0.0, -1.875875]],
          'symbols': 'NCCN',
          'thermal correction': 2.9336},
 'NF3': {'CAS No.': 7783542,
         'ZPE': 6.4477,
         'charges': None,
         'database': 'G2-2',
         'description': 'NF3, C3v symm.',
         'enthalpy': -31.6,
         'magmoms': None,
         'name': 'NF_3',
         'positions': [[0.0, 0.0, 0.489672],
                       [0.0, 1.238218, -0.126952],
                       [1.072328, -0.619109, -0.126952],
                       [-1.072328, -0.619109, -0.126952]],
         'symbols': 'NFFF',
         'thermal correction': 2.8301},
 'NO2': {'CAS No.': 10102440,
         'ZPE': 5.4631,
         'charges': None,
         'database': 'G2-2',
         'description': 'NO2 radical, C2v symm, 2-A1.',
         'enthalpy': 7.9,
         'magmoms': [1.0, 0.0, 0.0],
         'name': 'NO_2',
         'positions': [[0.0, 0.0, 0.332273],
                       [0.0, 1.118122, -0.14537],
                       [0.0, -1.118122, -0.14537]],
         'symbols': 'NOO',
         'thermal correction': 2.4366},
 'O': {'CAS No.': 17778802,
       'charges': None,
       'database': 'G2-1',
       'description': 'O atom',
       'enthalpy': 58.99,
       'ionization energy': 13.62,
       'magmoms': [2.0],
       'name': 'Oxygen',
       'positions': [[0.0, 0.0, 0.0]],
       'symbols': 'O',
       'thermal correction': 1.04},
 'O3': {'CAS No.': 10028156,
        'ZPE': 4.6178,
        'charges': None,
        'database': 'G2-2',
        'description': 'O3 (Ozone), C2v symm.',
        'enthalpy': 34.1,
        'magmoms': None,
        'name': 'O_3',
        'positions': [[0.0, 1.10381, -0.228542],
                      [0.0, 0.0, 0.457084],
                      [0.0, -1.10381, -0.228542]],
        'symbols': 'OOO',
        'thermal correction': 2.4479},
 'OCHCHO': {'CAS No.': 107222,
            'ZPE': 22.8426,
            'charges': None,
            'database': 'G2-2',
            'description': 'Glyoxal (O=CH-CH=O). Trans, C2h symm.',
            'enthalpy': -50.7,
            'magmoms': None,
            'name': 'HCOCOH (Oxalaldehyde)',
            'positions': [[0.0, 0.75643, 0.0],
                          [0.0, -0.75643, 0.0],
                          [1.04609, 1.389916, 0.0],
                          [-0.99994, 1.228191, 0.0],
                          [-1.04609, -1.389916, 0.0],
                          [0.99994, -1.228191, 0.0]],
            'symbols': 'CCOHOH',
            'thermal correction': 3.2518},
 'OCS': {'CAS No.': 463581,
         'ZPE': 5.7706,
         'charges': None,
         'database': 'G2-2',
         'description': 'O=C=S, Linear, C*v symm.',
         'enthalpy': -33.1,
         'magmoms': None,
         'name': 'COS',
         'positions': [[0.0, 0.0, -1.699243],
                       [0.0, 0.0, -0.520492],
                       [0.0, 0.0, 1.044806]],
         'symbols': 'OCS',
         'thermal correction': 2.3663},
 'P': {'CAS No.': 7723140,
       'charges': None,
       'database': 'G2-1',
       'description': 'P atom',
       'enthalpy': 75.42,
       'ionization energy': 10.49,
       'magmoms': [3.0],
       'name': 'Phosphorus',
       'positions': [[0.0, 0.0, 0.0]],
       'symbols': 'P',
       'thermal correction': 1.28},
 'PF3': {'CAS No.': 7783553,
         'ZPE': 5.2981,
         'charges': None,
         'database': 'G2-2',
         'description': 'PF3, C3v symm.',
         'enthalpy': -229.1,
         'magmoms': None,
         'name': 'PF_3',
         'positions': [[0.0, 0.0, 0.506767],
                       [0.0, 1.383861, -0.281537],
                       [1.198459, -0.691931, -0.281537],
                       [-1.198459, -0.691931, -0.281537]],
         'symbols': 'PFFF',
         'thermal correction': 3.1288},
 'S': {'CAS No.': 7704349,
       'charges': None,
       'database': 'G2-1',
       'description': 'S atom',
       'enthalpy': 65.66,
       'ionization energy': 10.36,
       'magmoms': [2.0],
       'name': 'Sulfur',
       'positions': [[0.0, 0.0, 0.0]],
       'symbols': 'S',
       'thermal correction': 1.05},
 'SH': {'CAS No.': 13940211,
        'ZPE': 3.7625,
        'charges': None,
        'database': 'G2-2',
        'description': 'SH radical, C*v symm.',
        'enthalpy': 34.2,
        'magmoms': [1.0, 0.0],
        'name': 'HS',
        'positions': [[0.0, 0.0, 0.079083], [0.0, 0.0, -1.26533]],
        'symbols': 'SH',
        'thermal correction': 2.0739},
 'Si': {'CAS No.': 7440213,
        'charges': None,
        'database': 'G2-1',
        'description': 'Si atom',
        'enthalpy': 106.6,
        'ionization energy': 8.15,
        'magmoms': [2.0],
        'name': 'Silicon',
        'positions': [[0.0, 0.0, 0.0]],
        'symbols': 'Si',
        'thermal correction': 0.76},
 'SiCl4': {'CAS No.': 10026047,
           'ZPE': 4.4396,
           'charges': None,
           'database': 'G2-2',
           'description': 'SiCl4, Td symm.',
           'enthalpy': -158.4,
           'magmoms': None,
           'name': 'SiCl_4 (Silane, tetrachloro-)',
           'positions': [[0.0, 0.0, 0.0],
                         [1.169349, 1.169349, 1.169349],
                         [-1.169349, -1.169349, 1.169349],
                         [1.169349, -1.169349, -1.169349],
                         [-1.169349, 1.169349, -1.169349]],
           'symbols': 'SiClClClCl',
           'thermal correction': 4.7182},
 'SiF4': {'CAS No.': 7783611,
          'ZPE': 7.8771,
          'charges': None,
          'database': 'G2-2',
          'description': 'SiF4, Td symm.',
          'enthalpy': -386.0,
          'magmoms': None,
          'name': 'SiF_4',
          'positions': [[0.0, 0.0, 0.0],
                        [0.912806, 0.912806, 0.912806],
                        [-0.912806, -0.912806, 0.912806],
                        [-0.912806, 0.912806, -0.912806],
                        [0.912806, -0.912806, -0.912806]],
          'symbols': 'SiFFFF',
          'thermal correction': 3.7054},
 'bicyclobutane': {'CAS No.': 157335,
                   'ZPE': 53.3527,
                   'charges': None,
                   'database': 'G2-2',
                   'description': 'Bicyclo[1.1.0]butane (C4H6), C2v symm.',
                   'enthalpy': 51.9,
                   'magmoms': None,
                   'name': 'C_4H_6 (bicyclobutane)',
                   'positions': [[0.0, 1.131343, 0.310424],
                                 [0.0, -1.131343, 0.310424],
                                 [0.747952, 0.0, -0.311812],
                                 [-0.747952, 0.0, -0.311812],
                                 [0.0, 1.237033, 1.397617],
                                 [0.0, 2.077375, -0.227668],
                                 [0.0, -1.237033, 1.397617],
                                 [0.0, -2.077375, -0.227668],
                                 [1.41441, 0.0, -1.161626],
                                 [-1.41441, 0.0, -1.161626]],
                   'symbols': 'CCCCHHHHHH',
                   'thermal correction': 2.9637},
 'butadiene': {'CAS No.': 106990,
               'ZPE': 52.6273,
               'charges': None,
               'database': 'G2-2',
               'description': 'Trans-1,3-butadiene (C4H6), C2h symm.',
               'enthalpy': 26.3,
               'magmoms': None,
               'name': 'CH_2CHCHCH_2 (butadiene)',
               'positions': [[0.605711, 1.74655, 0.0],
                             [0.605711, 0.404083, 0.0],
                             [-0.605711, -0.404083, 0.0],
                             [-0.605711, -1.74655, 0.0],
                             [1.527617, 2.317443, 0.0],
                             [-0.321132, 2.313116, 0.0],
                             [1.553503, -0.13364, 0.0],
                             [-1.553503, 0.13364, 0.0],
                             [0.321132, -2.313116, 0.0],
                             [-1.527617, -2.317443, 0.0]],
               'symbols': 'CCCCHHHHHH',
               'thermal correction': 3.5341},
 'cyclobutane': {'CAS No.': 287230,
                 'ZPE': 68.3314,
                 'charges': None,
                 'database': 'G2-2',
                 'description': 'Cyclobutane (C4H8), D2d symm.',
                 'enthalpy': 6.8,
                 'magmoms': None,
                 'name': 'C_4H_8 (cyclobutane)',
                 'positions': [[0.0, 1.071142, 0.147626],
                               [0.0, -1.071142, 0.147626],
                               [-1.071142, 0.0, -0.147626],
                               [1.071142, 0.0, -0.147626],
                               [0.0, 1.986858, -0.450077],
                               [0.0, 1.342921, 1.20752],
                               [0.0, -1.986858, -0.450077],
                               [0.0, -1.342921, 1.20752],
                               [-1.986858, 0.0, 0.450077],
                               [-1.342921, 0.0, -1.20752],
                               [1.986858, 0.0, 0.450077],
                               [1.342921, 0.0, -1.20752]],
                 'symbols': 'CCCCHHHHHHHH',
                 'thermal correction': 3.231},
 'cyclobutene': {'CAS No.': 822355,
                 'ZPE': 53.4105,
                 'charges': None,
                 'database': 'G2-2',
                 'description': 'Cyclobutene (C4H6), C2v symm.',
                 'enthalpy': 37.4,
                 'magmoms': None,
                 'name': 'C_4H_6 (cyclobutene)',
                 'positions': [[0.0, -0.672762, 0.811217],
                               [0.0, 0.672762, 0.811217],
                               [0.0, -0.78198, -0.696648],
                               [0.0, 0.78198, -0.696648],
                               [0.0, -1.422393, 1.597763],
                               [0.0, 1.422393, 1.597763],
                               [-0.88931, -1.239242, -1.142591],
                               [0.88931, -1.239242, -1.142591],
                               [0.88931, 1.239242, -1.142591],
                               [-0.88931, 1.239242, -1.142591]],
                 'symbols': 'CCCCHHHHHH',
                 'thermal correction': 3.0108},
 'isobutane': {'CAS No.': 75285,
               'ZPE': 81.105,
               'charges': None,
               'database': 'G2-2',
               'description': 'Isobutane (C4H10), C3v symm.',
               'enthalpy': -32.1,
               'magmoms': None,
               'name': 'C_4H_{10} (isobutane)',
               'positions': [[0.0, 0.0, 0.376949],
                             [0.0, 0.0, 1.475269],
                             [0.0, 1.45029, -0.096234],
                             [0.0, 1.493997, -1.190847],
                             [-0.885482, 1.984695, 0.261297],
                             [0.885482, 1.984695, 0.261297],
                             [1.255988, -0.725145, -0.096234],
                             [1.293839, -0.746998, -1.190847],
                             [2.161537, -0.225498, 0.261297],
                             [1.276055, -1.759198, 0.261297],
                             [-1.255988, -0.725145, -0.096234],
                             [-1.293839, -0.746998, -1.190847],
                             [-1.276055, -1.759198, 0.261297],
                             [-2.161537, -0.225498, 0.261297]],
               'symbols': 'CHCHHHCHHHCHHH',
               'thermal correction': 4.2282},
 'isobutene': {'CAS No.': 115117,
               'ZPE': 66.5693,
               'charges': None,
               'database': 'G2-2',
               'description': 'Isobutene (C4H8), Single bonds trans, C2v symm.',
               'enthalpy': -4.0,
               'magmoms': None,
               'name': 'C_4H_8 (1-Propene, 2-methyl-)',
               'positions': [[0.0, 0.0, 1.458807],
                             [0.0, 0.0, 0.119588],
                             [0.0, 0.924302, 2.028409],
                             [0.0, -0.924302, 2.028409],
                             [0.0, 1.272683, -0.678803],
                             [0.0, 2.153042, -0.031588],
                             [0.880211, 1.323542, -1.329592],
                             [-0.880211, 1.323542, -1.329592],
                             [0.0, -1.272683, -0.678803],
                             [0.0, -2.153042, -0.031588],
                             [-0.880211, -1.323542, -1.329592],
                             [0.880211, -1.323542, -1.329592]],
               'symbols': 'CCHHCHHHCHHH',
               'thermal correction': 3.9495},
 'methylenecyclopropane': {'CAS No.': 6142730,
                           'ZPE': 52.623,
                           'charges': None,
                           'database': 'G2-2',
                           'description': 'Methylenecyclopropane (C4H6), C2v symm.',
                           'enthalpy': 47.9,
                           'magmoms': None,
                           'name': 'C_4H_6 (methylene cyclopropane)',
                           'positions': [[0.0, 0.0, 0.315026],
                                         [0.0, -0.76792, -0.932032],
                                         [0.0, 0.76792, -0.932032],
                                         [0.0, 0.0, 1.640027],
                                         [-0.912794, -1.271789, -1.239303],
                                         [0.912794, -1.271789, -1.239303],
                                         [0.912794, 1.271789, -1.239303],
                                         [-0.912794, 1.271789, -1.239303],
                                         [0.0, -0.926908, 2.20564],
                                         [0.0, 0.926908, 2.20564]],
                           'symbols': 'CCCCHHHHHH',
                           'thermal correction': 3.2881},
 'trans-butane': {'CAS No.': 106978,
                  'ZPE': 81.398,
                  'charges': None,
                  'database': 'G2-2',
                  'description': 'Trans-butane (C4H10), C2h symm.',
                  'enthalpy': -30.0,
                  'magmoms': None,
                  'name': 'C_4H_{10} (Butane)',
                  'positions': [[0.702581, 1.820873, 0.0],
                                [0.702581, 0.296325, 0.0],
                                [-0.702581, -0.296325, 0.0],
                                [-0.702581, -1.820873, 0.0],
                                [1.719809, 2.22234, 0.0],
                                [-1.719809, -2.22234, 0.0],
                                [0.188154, 2.210362, 0.883614],
                                [0.188154, 2.210362, -0.883614],
                                [-0.188154, -2.210362, 0.883614],
                                [-0.188154, -2.210362, -0.883614],
                                [1.247707, -0.07266, -0.877569],
                                [1.247707, -0.07266, 0.877569],
                                [-1.247707, 0.07266, -0.877569],
                                [-1.247707, 0.07266, 0.877569]],
                  'symbols': 'CCCCHHHHHHHHHH',
                  'thermal correction': 4.2633}}


# all constituent atoms
atoms_g22 = []
for f in data.keys():
    s = Atoms(symbols=data[f]['symbols'],  # type: ignore
              positions=data[f]['positions'])  # type: ignore
    for a in s:
        atoms_g22.append(a.symbol)
# unique atoms
atoms_g22 = list(set(atoms_g22))

# add remaining atoms from G2_1
from ase.data.g2_1 import data as data1

for a in atoms_g22:
    if not a in data.keys():
        data[a] = data1[a]
