/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at <http://mozilla.org/MPL/2.0/>. */

// @flow

import type {
  SourcePacket,
  ResumedPacket,
  PausedPacket,
  ThreadClient,
  Actions,
  TabTarget,
} from "./types";

import { createPause, prepareSourcePayload } from "./create";
import sourceQueue from "../../utils/source-queue";

const CALL_STACK_PAGE_SIZE = 1000;

type Dependencies = {
  threadClient: ThreadClient,
  tabTarget: TabTarget,
  actions: Actions,
};

let actions: Actions;
let isInterrupted: boolean;

function addThreadEventListeners(client: ThreadClient) {
  Object.keys(clientEvents).forEach(eventName => {
    client.addListener(eventName, clientEvents[eventName].bind(null, client));
  });
}

function setupEvents(dependencies: Dependencies) {
  const threadClient = dependencies.threadClient;
  const tabTarget = dependencies.tabTarget;
  actions = dependencies.actions;
  sourceQueue.initialize(actions);

  addThreadEventListeners(threadClient);
  tabTarget.on("workerListChanged", workerListChanged);
}

async function paused(
  threadClient: ThreadClient,
  _: "paused",
  packet: PausedPacket
) {
  // If paused by an explicit interrupt, which are generated by the
  // slow script dialog and internal events such as setting
  // breakpoints, ignore the event.
  const { why } = packet;
  if (why.type === "interrupted" && !packet.why.onNext) {
    isInterrupted = true;
    return;
  }

  let response;
  try {
    // Eagerly fetch the frames
    response = await threadClient.getFrames(0, CALL_STACK_PAGE_SIZE);
  } catch (e) {
    console.log(e);
    return;
  }

  // NOTE: this happens if we fetch frames and then immediately navigate
  if (!response.hasOwnProperty("frames")) {
    return;
  }

  if (why.type != "alreadyPaused") {
    const pause = createPause(threadClient.actor, packet, response);
    await sourceQueue.flush();
    actions.paused(pause);
  }
}

function resumed(
  threadClient: ThreadClient,
  _: "resumed",
  packet: ResumedPacket
) {
  // NOTE: the client suppresses resumed events while interrupted
  // to prevent unintentional behavior.
  // see [client docs](../README.md#interrupted) for more information.
  if (isInterrupted) {
    isInterrupted = false;
    return;
  }

  actions.resumed(packet);
}

function newSource(
  threadClient: ThreadClient,
  _: "newSource",
  { source }: SourcePacket
) {
  sourceQueue.queue({
    type: "generated",
    data: prepareSourcePayload(threadClient, source),
  });
}

function workerListChanged() {
  actions.updateWorkers();
}

const clientEvents = {
  paused,
  resumed,
  newSource,
};

export { setupEvents, clientEvents, addThreadEventListeners };
