/***************************************************************
 * Name:      dxutils.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2020
 * License:   GPL3
 **************************************************************/

#include <wx/tokenzr.h>
#include <wx/url.h>
#include <wx/wfstream.h>
#include <wx/sstream.h>
#include "dxutils.h"
#include "data/event.h"
#include "config.h"
#ifdef HAVE_CURL
#include <curl/curl.h>

static size_t curlCallback(void *contents, size_t size, size_t nmemb, void *userp)
{
    (static_cast<std::string*>(userp))->append(static_cast<char*>(contents), size * nmemb);
    return size * nmemb;
}
#endif //HAVE_CURL

dxutils::dxutils()
{
}

wxString dxutils::formatDateTime(const wxDateTime &date, const wxString &formatDate, const wxString &formatTime, int layout)
{
    struct tm t = dateTimeToTm(date);
    wxString format;
    switch (layout) {
    case 0: format << formatDate; format << " "; format << formatTime; break;
    case 1: format << formatTime; format << " "; format << formatDate; break;
    case 2: format = formatDate; break;
    default: format = formatTime;
    }
    char mbstr[100];
    if(std::strftime(mbstr, sizeof(mbstr), format.mb_str(), &t))
    {
        return mbstr;
    }
    return _("Invalid format");
}

wxString dxutils::formatDate(const wxDateTime &date, const wxString &formatDate, bool replaceBrackets)
{
    wxString format = formatDate;
    if(replaceBrackets)
    {
        format.Replace("[","");
        format.Replace("]","");
        format.Replace("(","");
        format.Replace(")","");
    }
    struct tm t = dateTimeToTm(date);
    char mbstr[100];
    if(std::strftime(mbstr, sizeof(mbstr), format.mb_str(), &t))
    {
        return mbstr;
    }
    return _("Invalid format");
}

wxString dxutils::formatTime(const wxDateTime &date, const wxString &formatTime, bool replaceBrackets)
{
    wxString format = formatTime;
    if(replaceBrackets)
    {
        format.Replace("[","");
        format.Replace("]","");
        format.Replace("(","");
        format.Replace(")","");
    }
    struct tm t = dateTimeToTm(date);
    char mbstr[100];
    if(std::strftime(mbstr, sizeof(mbstr), format.mb_str(), &t))
    {
        return mbstr;
    }
    return _("Invalid format");
}

wxDateTime dxutils::easternSunday(int year)
{
    int a = year % 19;
    int b = year / 100;
    int c = year % 100;
    int d = b / 4;
    int e = b % 4;
    int f = (b+8) / 25;
    int g = (b-f+1) / 3;
    int h = (19*a + b - d - g + 15) % 30;
    int i = c / 4;
    int k = c % 4;
    int l = (32 + 2*e + 2*i - h - k) % 7;
    int m = (a + 11*h + 22*l) / 451;
    int p = (h + l - 7*m + 114) % 31;
    return wxDateTime(static_cast<wxDateTime::wxDateTime_t>(p+1), ((h + l - 7*m + 114) / 31) == 3 ? wxDateTime::Mar : wxDateTime::Apr, year);
}

wxDateTime::wxDateTime_t dxutils::lastdayofmonth(wxDateTime::Month month, int year)
{
    switch (month) {
    case wxDateTime::Jan: return 31;
    case wxDateTime::Feb: {
        if(wxDateTime::IsLeapYear(year)) return 29;
        else return 28;
    }
    case wxDateTime::Mar: return 31;
    case wxDateTime::Apr: return 30;
    case wxDateTime::May: return 31;
    case wxDateTime::Jun: return 30;
    case wxDateTime::Jul: return 31;
    case wxDateTime::Aug: return 31;
    case wxDateTime::Sep: return 30;
    case wxDateTime::Oct: return 31;
    case wxDateTime::Nov: return 30;
    case wxDateTime::Dec: return 31;
    default: return 0;
    }
}

wxDateTime::wxDateTime_t dxutils::weeksofmonth(const wxDateTime &date)
{
    return date.GetLastMonthDay().GetWeekOfMonth();
}

wxString dxutils::fontToString(const wxFont &font)
{
    return font.GetNativeFontInfoDesc();
}

wxFont dxutils::fontFromString(const wxString &fontstr)
{
    return wxFont(fontstr);
}

tm dxutils::dateTimeToTm(const wxDateTime &date)
{
    struct tm t;
    t.tm_year = date.GetYear()-1900;
    t.tm_mon = date.GetMonth();
    t.tm_mday = date.GetDay();
    t.tm_hour = date.GetHour();
    t.tm_min = date.GetMinute();
    t.tm_sec = date.GetSecond();
    t.tm_wday = date.GetWeekDay();
    t.tm_yday = date.GetDayOfYear()-1;
    return t;
}

size_t dxutils::numberOfChars(const wxString &text, char value)
{
    size_t result = 0;
    for(size_t i=0; i<text.Len(); ++i)
    {
        if(text[i] == value) result++;
    }
    return result;
}

int dxutils::recurrenceSelectionToType(int selection)
{
    switch (selection){
    case 0: return R_NONE;
    case 1: return R_ONCENOTDELETE;
    case 2: return R_ONCE;
    case 3: return R_MINUTES;
    case 4: return R_HOURLY;
    case 5: return R_DAILY;
    case 6: return R_DAILYBUSINESS;
    case 7: return R_WEEKLY;
    case 8: return R_WEEKLYBUSINESS;
    case 9: return R_WEEKLYPLUSBUSINESSDAY;
    case 10: return R_WEEKLYGIVENBUSINESSDAY;
    case 11: return R_2WEEKLY;
    case 12: return R_2WEEKLYBUSINESS;
    case 13: return R_2WEEKLYPLUSBUSINESSDAY;
    case 14: return R_2WEEKLYGIVENBUSINESSDAY;
    case 15: return R_MONTHLY;
    case 16: return R_MONTHLYBUSINESS;
    case 17: return R_MONTHLYPLUSBUSINESSDAY;
    case 18: return R_MONTHLYGIVENBUSINESSDAY;
    case 19: return R_MONTHLYATDAY;
    case 20: return R_QUARTERLY;
    case 21: return R_QUARTERLYBUSINESS;
    case 22: return R_QUARTERLYPLUSBUSINESSDAY;
    case 23: return R_QUARTERLYGIVENBUSINESSDAY;
    case 24: return R_YEARLY;
    case 25: return R_OWN;
    default: return R_ONCE;
    }
}

int dxutils::recurrenceTypeToSelection(int type)
{
    switch (type){
    case R_ONCE: return 2;
    case R_ONCENOTDELETE: return 1;
    case R_MINUTES: return 3;
    case R_HOURLY: return 4;
    case R_DAILY: return 5;
    case R_DAILYBUSINESS: return 6;
    case R_WEEKLY: return 7;
    case R_WEEKLYBUSINESS: return 8;
    case R_WEEKLYPLUSBUSINESSDAY: return 9;
    case R_WEEKLYGIVENBUSINESSDAY: return 10;
    case R_2WEEKLY: return 11;
    case R_2WEEKLYBUSINESS: return 12;
    case R_2WEEKLYPLUSBUSINESSDAY: return 13;
    case R_2WEEKLYGIVENBUSINESSDAY: return 14;
    case R_MONTHLY: return 15;
    case R_MONTHLYBUSINESS: return 16;
    case R_MONTHLYPLUSBUSINESSDAY: return 17;
    case R_MONTHLYGIVENBUSINESSDAY: return 18;
    case R_MONTHLYATDAY: return 19;
    case R_QUARTERLY: return 20;
    case R_QUARTERLYBUSINESS: return 21;
    case R_QUARTERLYPLUSBUSINESSDAY: return 22;
    case R_QUARTERLYGIVENBUSINESSDAY: return 23;
    case R_YEARLY: return 24;
    case R_OWN: return 25;
    case R_NONE: return 0;
    default: return 0;
    }
}

wxArrayString dxutils::reminders()
{
    wxArrayString reminders;
    reminders.Add(_("At set time"));
    reminders.Add(_("5 minutes before"));
    reminders.Add(_("15 minutes before"));
    reminders.Add(_("1 hour before"));
    reminders.Add(_("2 hours before"));
    reminders.Add(_("1 day before"));
    reminders.Add(_("2 days before"));
    reminders.Add(_("Custom"));
    return reminders;
}

wxArrayString dxutils::recurrences()
{
    wxArrayString recurrences;
    recurrences.Add(_("None (Do not delete)"));
    recurrences.Add(_("Once (Do not delete)"));
    recurrences.Add(_("Once"));
    recurrences.Add(_("Minutes"));
    recurrences.Add(_("Hourly"));
    recurrences.Add(_("Daily"));
    recurrences.Add(_("Daily at business day"));
    recurrences.Add(_("Weekly"));
    recurrences.Add(_("Weekly at business day"));
    recurrences.Add(_("Weekly plus given business day"));
    recurrences.Add(_("Weekly at given business day"));
    recurrences.Add(_("Fortnightly"));
    recurrences.Add(_("Fortnightly at business day"));
    recurrences.Add(_("Fortnightly plus given business day"));
    recurrences.Add(_("Fortnightly at given business day"));
    recurrences.Add(_("Monthly"));
    recurrences.Add(_("Monthly at business day"));
    recurrences.Add(_("Monthly plus given business day"));
    recurrences.Add(_("Monthly at given business day"));
    recurrences.Add(_("Monthly at given day"));
    recurrences.Add(_("Quarterly"));
    recurrences.Add(_("Quarterly at business day"));
    recurrences.Add(_("Quarterly plus given business day"));
    recurrences.Add(_("Quarterly at given business day"));
    recurrences.Add(_("Yearly"));
    recurrences.Add(_("Custom"));
    return recurrences;
}

long dxutils::textToInt(const wxString &text)
{
    long result = 0;
    wxString number = "";
    for(size_t i=0; i<text.Len(); ++i)
    {
        if(std::isdigit(static_cast<unsigned char>(text[i])))
        {
            number << text[i];
        }
        else
        {
            break;
        }
    }
    if(number.Len())
    {
        number.ToLong(&result);
    }
    return result;
}

wxString dxutils::fromStdString(const std::string &str)
{
    if(str.empty()) return "";
    return wxString::FromUTF8(str.c_str());
}

wxString dxutils::availableVersion()
{
#ifdef HAVE_CURL
    CURL *curl;
    curl = curl_easy_init();
    if(curl)
    {
        std::string response;
        curl_easy_setopt(curl, CURLOPT_URL, "https://dxsolutions.org/dxreminders-version");
        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0);
        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curlCallback);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, &response);
        CURLcode res = curl_easy_perform(curl);
        curl_easy_cleanup(curl);
        if(res == CURLE_OK)
        {
            return fromStdString(response).BeforeFirst('\n');
        }
    }
    return "0.00.0";
#else
    return "0.00.0";
#endif
}

int dxutils::versionToInt(const wxString &version)
{
    int result = 0;
    wxStringTokenizer tok(version.BeforeFirst('-'), ".");
    if(tok.CountTokens() == 3)
    {
        int i=0;
        long a = 0;
        while(tok.HasMoreTokens())
        {
            wxString token = tok.GetNextToken();
            switch(i) {
            case 0: token.ToLong(&a); result += a*1000; break;
            case 1: token.ToLong(&a); result += a*10; break;
            case 2: token.ToLong(&a); result += a; break;
            default: break;
            }
            i++;
        }
    }
    if(version.Contains("-"))
    {
        result += 1000;
    }
    return result;
}
