﻿/*  $Id$
 *
 *  Copyright (C) 2003 Choe Hwanjin(krisna@kldp.org)
 *  Copyright (c) 2006 Remco den Breeje <remco@sx.mine.nu>
 *  Copyright (c) 2008 Diego Ongaro <ongardie@gmail.com>
 *  Copyright (c) 2016 Landry Breuil <landry@xfce.org>
 *  Copyright (C) 2020 David Vachulka <arch_dvx@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <time.h>
#include <string.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <libxfce4ui/libxfce4ui.h>
#include <libxfce4util/libxfce4util.h>
#include <libxfce4panel/libxfce4panel.h>
#include "dxreminders.h"
#include "dxreminders-dialogs.h"

#define UPDATE_INTERVAL 1000
#define BUFFER_SIZE 32768

/* prototypes */
static void dxreminders_construct (XfcePanelPlugin *plugin);
static void dxreminders_start(dxremindersPlugin *dxreminders);

static gboolean dxreminders_ipcread(G_GNUC_UNUSED GIOChannel *channel, G_GNUC_UNUSED GIOCondition cond, dxremindersPlugin *dxreminders)
{
    DBG("Something to read");
    char buffer[BUFFER_SIZE];
    int rc = recv(dxreminders->sd, buffer, sizeof(buffer), 0);
    if(rc < 0)
    {
        DBG("recv() failed");
        dxreminders->sd = -1;
        return FALSE;
    }
    else if(rc == 0)
    {
        DBG("server closed the connection");
        dxreminders->sd = -1;
        return FALSE;
    }
    else
    {
        buffer[rc] = '\0';
    }
    gchar **pp1, **pp2;
    pp1 = pp2 = g_strsplit (buffer, "\r\n", 0);
    while(*pp1 != NULL)
    {
        if(g_str_has_prefix(*pp1,"<tooltip>"))
        {
            gchar **tts;
            tts = g_strsplit(*pp1,">",2);
            g_free(dxreminders->tooltip);
            dxreminders->tooltip = g_strdup(tts[1]);
            g_strfreev(tts);
        }
        if(g_str_has_prefix(*pp1,"<silenton>"))
        {
            dxreminders->silentmode = TRUE;
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM (dxreminders->silentcheck), dxreminders->silentmode);
        }
        if(g_str_has_prefix(*pp1,"<silentoff>"))
        {
            dxreminders->silentmode = FALSE;
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM (dxreminders->silentcheck), dxreminders->silentmode);
        }
        if(g_str_has_prefix(*pp1,"<gorestart>"))
        {
            if(dxreminders->pid != 0) killpg(dxreminders->pid, SIGKILL);
            g_usleep(750000);
            dxreminders_start(dxreminders);
        }
        pp1++;
    }
    g_strfreev(pp2);
    return TRUE;
}

static gboolean dxreminders_ipcerr(G_GNUC_UNUSED GIOChannel *channel, G_GNUC_UNUSED GIOCondition cond, dxremindersPlugin *dxreminders)
{
    dxreminders->sd = -1;
    return FALSE;
}

static gboolean dxreminders_ipchup(G_GNUC_UNUSED GIOChannel *channel, G_GNUC_UNUSED GIOCondition cond, dxremindersPlugin *dxreminders)
{
    dxreminders->sd = -1;
    return FALSE;
}


gboolean dxreminders_connectipc(dxremindersPlugin *dxreminders)
{
    if(dxreminders->sd == -1)
    {
        dxreminders->sd = socket(AF_UNIX, SOCK_STREAM, 0);
        if(dxreminders->sd < 0)
        {
            DBG("socket() failed");
            return FALSE;
        }
        int rc;
        struct sockaddr_un serveraddr;
        memset(&serveraddr, 0, sizeof(serveraddr));
        serveraddr.sun_family = AF_UNIX;
        gchar *filename = g_strconcat(g_get_home_dir(), "/.dxreminders.socket", NULL);
        strcpy(serveraddr.sun_path, filename);
        g_free(filename);
        rc = connect(dxreminders->sd, (struct sockaddr *)&serveraddr, SUN_LEN(&serveraddr));
        if(rc < 0)
        {
            DBG("connect() failed");
            dxreminders->sd = -1;
            return FALSE;
        }
        GIOChannel *channel = g_io_channel_unix_new(dxreminders->sd);
        g_io_add_watch(channel, G_IO_IN, (GIOFunc)dxreminders_ipcread, dxreminders);
        g_io_add_watch(channel, G_IO_ERR, (GIOFunc)dxreminders_ipcerr, dxreminders);
        g_io_add_watch(channel, G_IO_HUP, (GIOFunc)dxreminders_ipchup, dxreminders);
        g_io_channel_unref(channel);
        return TRUE;
    }
    return TRUE;
}

void dxreminders_save(XfcePanelPlugin *plugin, dxremindersPlugin *dxreminders)
{
    XfceRc *rc;
    gchar *file;
    /* get the config file location */
    file = xfce_panel_plugin_save_location(plugin, TRUE);
    if(G_UNLIKELY (file == NULL))
    {
        DBG("Failed to open config file");
        return;
    }
    /* open the config file, read/write */
    rc = xfce_rc_simple_open(file, FALSE);
    g_free(file);
    if(G_LIKELY(rc != NULL))
    {
        /* save the settings */
        xfce_rc_write_int_entry(rc, "layout", dxreminders->layout);
        xfce_rc_write_entry(rc, "date_font", dxreminders->date_font);
        xfce_rc_write_entry(rc, "time_font", dxreminders->time_font);
        xfce_rc_write_entry(rc, "date_format", dxreminders->date_format);
        xfce_rc_write_entry(rc, "time_format", dxreminders->time_format);
        xfce_rc_write_bool_entry(rc, "calendar", dxreminders->calendar);
        /* close the rc file */
        xfce_rc_close (rc);
    }
}

static void dxreminders_read(dxremindersPlugin *dxreminders)
{
    dxreminders->tooltip = NULL;
    dxreminders->pid = 0;
    dxreminders->sd = -1;
    dxreminders->calendar = TRUE;
    XfceRc *rc = NULL;
    gchar *file;
    dx_layout layout;
    const gchar *date_font, *time_font, *date_format, *time_format;
    /* load defaults */
    layout = LAYOUT_DATE_TIME;
    date_font = "Bitstream Vera Sans 8";
    time_font = "Bitstream Vera Sans 8";
    date_format = "%Y-%m-%d";
    time_format = "%H:%M:%S";
    /* get the plugin config file location */
    file = xfce_panel_plugin_save_location(dxreminders->plugin, TRUE);
    if(G_LIKELY(file != NULL))
    {
        /* open the config file, readonly */
        rc = xfce_rc_simple_open(file, TRUE);
        /* cleanup */
        g_free(file);
        if(G_LIKELY(rc != NULL))
        {
            layout = xfce_rc_read_int_entry(rc, "layout", layout);
            date_font = xfce_rc_read_entry(rc, "date_font", date_font);
            time_font = xfce_rc_read_entry(rc, "time_font", time_font);
            date_format = xfce_rc_read_entry(rc, "date_format", date_format);
            time_format = xfce_rc_read_entry(rc, "time_format", time_format);
            dxreminders->calendar = xfce_rc_read_bool_entry(rc, "calendar", TRUE);
        }
    }
    date_font   = g_strdup(date_font);
    time_font   = g_strdup(time_font);
    date_format = g_strdup(date_format);
    time_format = g_strdup(time_format);
    if(G_LIKELY(rc != NULL))
        xfce_rc_close(rc);
    /* set values in plugin struct */
    dxreminders_apply_layout(dxreminders, layout);
    dxreminders_apply_font(dxreminders, date_font, time_font);
    dxreminders_apply_format(dxreminders, date_format, time_format);
}

/*
 * change widgets orientation when the panel orientation changes
 */
static void dxreminders_set_mode(G_GNUC_UNUSED XfcePanelPlugin *plugin, XfcePanelPluginMode mode, dxremindersPlugin *dxreminders)
{
    GtkOrientation orientation = (mode == XFCE_PANEL_PLUGIN_MODE_VERTICAL) ? GTK_ORIENTATION_VERTICAL : GTK_ORIENTATION_HORIZONTAL;
    if(orientation == GTK_ORIENTATION_VERTICAL)
    {
        gtk_orientable_set_orientation(GTK_ORIENTABLE(dxreminders->box), GTK_ORIENTATION_HORIZONTAL);
        gtk_label_set_angle(GTK_LABEL(dxreminders->time_label), -90);
        gtk_label_set_angle(GTK_LABEL(dxreminders->date_label), -90);
        gtk_box_reorder_child(GTK_BOX(dxreminders->box), dxreminders->time_label, 0);
        gtk_box_reorder_child(GTK_BOX(dxreminders->box), dxreminders->date_label, 1);
    }
    else
    {
        gtk_orientable_set_orientation(GTK_ORIENTABLE(dxreminders->box), GTK_ORIENTATION_VERTICAL);
        gtk_label_set_angle(GTK_LABEL(dxreminders->time_label), 0);
        gtk_label_set_angle(GTK_LABEL(dxreminders->date_label), 0);
        gtk_box_reorder_child(GTK_BOX(dxreminders->box), dxreminders->date_label, 0);
        gtk_box_reorder_child(GTK_BOX(dxreminders->box), dxreminders->time_label, 1);
    }
}

static gboolean dxreminders_tooltip_timer(dxremindersPlugin *dxreminders)
{
    /* flag to dxreminders_query_tooltip that there is no longer an active timeout */
    dxreminders->tooltip_timeout_id = 0;
    /*
    * Run dxreminders_query_tooltip if the mouse is still there.
    * If it is run, it'll register *this* function to be called after another
    * timeout. If not, *this* function won't run again.
    */
    gtk_widget_trigger_tooltip_query(GTK_WIDGET(dxreminders->button));
    /* we don't want to automatically run again */
    return FALSE;
}

static gboolean dxreminders_running()
{
    gchar *filename = g_strconcat(g_get_home_dir(), "/dxreminders-", g_get_user_name(), NULL);
    if(g_file_test(filename, G_FILE_TEST_EXISTS) && !g_file_test(filename, G_FILE_TEST_IS_DIR))
    {
        struct flock fl;
        int fd;
        fd = open(filename, O_RDWR | O_CREAT, 0600);
        if(fd < 0)
        {
            perror(filename);
            _exit(1);
        }
        fl.l_start = 0;
        fl.l_len = 0;
        fl.l_type = F_WRLCK;
        fl.l_whence = SEEK_SET;
        if(fcntl(fd, F_SETLK, &fl) < 0)
        {
            g_free(filename);
            return TRUE;
        }
        unlink(filename);
    }
    g_free(filename);
    return FALSE;
}

static void dxreminders_start(dxremindersPlugin *dxreminders)
{
    if(!dxreminders_running())
    {
        DBG("dxreminders doesn't run");
        gint pid = fork();
        if(pid == 0)
        {
            setsid();
            int ret = system("dxreminders -p");
            if(ret == -1)
            {
                DBG("dxreminders start failed");
            }
        }
        else
        {
            dxreminders->pid = pid;
        }
    }
}

static void dxreminders_show_hide(GtkMenuItem *item, dxremindersPlugin *dxreminders)
{
    g_return_if_fail(GTK_IS_MENU_ITEM (item));
    g_return_if_fail(dxreminders != NULL);
    if(!dxreminders_running())
    {
        dxreminders_start(dxreminders);
        g_usleep(750000);
    }
    dxreminders_connectipc(dxreminders);
    if(dxreminders->sd == -1)
    {
        DBG("IPC doesn't connected");
        return;
    }
    int rc = send(dxreminders->sd, "<showhide>", 10, 0);
    if(rc < 0)
    {
        DBG("send() failed");
        return;
    }
    dxreminders_tooltip_timer(dxreminders);
}

static void dxreminders_newevent(GtkMenuItem *item, dxremindersPlugin *dxreminders)
{
    g_return_if_fail(GTK_IS_MENU_ITEM (item));
    g_return_if_fail(dxreminders != NULL);
    if(!dxreminders_running())
    {
        dxreminders_start(dxreminders);
        g_usleep(750000);
    }
    dxreminders_connectipc(dxreminders);
    if(dxreminders->sd == -1)
    {
        DBG("IPC doesn't connected");
        return;
    }
    int rc = send(dxreminders->sd, "<newevent>", 10, 0);
    if(rc < 0)
    {
        DBG("send() failed");
        return;
    }
    dxreminders_tooltip_timer(dxreminders);
}

static void dxreminders_restart(GtkMenuItem *item, dxremindersPlugin *dxreminders)
{
    g_return_if_fail(GTK_IS_MENU_ITEM (item));
    g_return_if_fail(dxreminders != NULL);
    if(dxreminders_running() && dxreminders->sd != -1)
    {
        int rc = send(dxreminders->sd, "<saveini>", 9, 0);
        if(rc < 0)
        {
            DBG("send() failed");
            return;
        }
        else
        {
            DBG("send saveini sucessfully");
        }
    }
//    if(dxreminders->pid != 0) killpg(dxreminders->pid, SIGKILL);
//    g_usleep(750000);
//    dxreminders_start(dxreminders);
}

static void dxreminders_on_click(dxremindersPlugin *dxreminders)
{
    if(!dxreminders_running())
    {
        dxreminders_start(dxreminders);
        g_usleep(750000);
    }
    dxreminders_connectipc(dxreminders);
    if(dxreminders->sd == -1)
    {
        DBG("IPC doesn't connected");
        return;
    }
    int rc = send(dxreminders->sd, dxreminders->calendar?"<calendar>":"<showhide>", 10, 0);
    if(rc < 0)
    {
        DBG("send() failed");
        return;
    }
    dxreminders_tooltip_timer(dxreminders);
}

static gboolean dxreminders_events_tooltip_timer(dxremindersPlugin *dxreminders)
{
    if(dxreminders_connectipc(dxreminders))
    {
        send(dxreminders->sd, "<tooltip>", 9, 0);
        DBG("send(tooltip) successfull on timer");
        return FALSE;
    }
    return TRUE;
}

static void dxreminders_events_tooltip(dxremindersPlugin *dxreminders)
{
    if(!dxreminders_running())
    {
        DBG("dxreminders doesn't run");
        g_timeout_add(750, (GSourceFunc)dxreminders_events_tooltip_timer, (gpointer)dxreminders);
        return;
    }
    if(dxreminders->sd == -1)
    {
        DBG("IPC doesn't connected");
        g_timeout_add(750, (GSourceFunc)dxreminders_events_tooltip_timer, (gpointer)dxreminders);
        return;
    }
    int rc = send(dxreminders->sd, "<tooltip>", 9, 0);
    if(rc < 0)
    {
        DBG("send() failed");
        g_timeout_add(750, (GSourceFunc)dxreminders_events_tooltip_timer, (gpointer)dxreminders);
        return;
    }
    DBG("send(tooltip) successfull");
}

static gboolean dxreminders_events_silentmode_timer(dxremindersPlugin *dxreminders)
{
    if(dxreminders_connectipc(dxreminders))
    {
        send(dxreminders->sd, "<silentmode>", 12, 0);
        DBG("send(silentmode) successfull on timer");
        return FALSE;
    }
    return TRUE;
}

static void dxreminders_events_silentmode(dxremindersPlugin *dxreminders)
{
    if(!dxreminders_running())
    {
        DBG("dxreminders doesn't run");
        g_timeout_add(750, (GSourceFunc)dxreminders_events_silentmode_timer, (gpointer)dxreminders);
        return;
    }
    if(dxreminders->sd == -1)
    {
        DBG("IPC doesn't connected");
        g_timeout_add(750, (GSourceFunc)dxreminders_events_silentmode_timer, (gpointer)dxreminders);
        return;
    }
    int rc = send(dxreminders->sd, "<silentmode>", 12, 0);
    if(rc < 0)
    {
        DBG("send() failed");
        g_timeout_add(750, (GSourceFunc)dxreminders_events_silentmode_timer, (gpointer)dxreminders);
        return;
    }
    DBG("send(silentmode) successfull");
}

static gboolean dxreminders_clicked(G_GNUC_UNUSED GtkWidget *widget, GdkEventButton *event, dxremindersPlugin *dxreminders)
{
    if(event->button != 1 || event->state & GDK_CONTROL_MASK)
        return FALSE;
    if(dxreminders == NULL)
        return FALSE;
    dxreminders_on_click(dxreminders);
    return TRUE;
}

/*
 * create the gtk-part of the dxreminders plugin
 */
static void dxreminders_create_widget(dxremindersPlugin *dxreminders)
{
    GtkOrientation orientation;
    orientation = xfce_panel_plugin_get_orientation(dxreminders->plugin);
    /* create button */
    dxreminders->button = xfce_panel_create_button();
    gtk_widget_show(dxreminders->button);
    /* create a box which can be easily adapted to the panel orientation */
    dxreminders->box = gtk_box_new(GTK_ORIENTATION_VERTICAL, 0);
    gtk_widget_show(dxreminders->box);
    gtk_container_add(GTK_CONTAINER(dxreminders->button), dxreminders->box);
    /* create time and date lines */
    dxreminders->time_label = gtk_label_new("");
    dxreminders->date_label = gtk_label_new("");
    gtk_label_set_justify(GTK_LABEL(dxreminders->time_label), GTK_JUSTIFY_CENTER);
    gtk_label_set_justify(GTK_LABEL(dxreminders->date_label), GTK_JUSTIFY_CENTER);
    /* add time and date lines to the box */
    gtk_box_pack_start(GTK_BOX(dxreminders->box), dxreminders->time_label, TRUE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(dxreminders->box), dxreminders->date_label, TRUE, FALSE, 0);
    /* connect widget signals to functions */
    g_signal_connect(dxreminders->button, "button-press-event", G_CALLBACK(dxreminders_clicked), dxreminders);
    /* set orientation according to the panel orientation */
    dxreminders_set_mode(dxreminders->plugin, (XfcePanelPluginMode)orientation, dxreminders);
}

static dxremindersPlugin *dxreminders_new(XfcePanelPlugin *plugin)
{
    dxremindersPlugin *dxreminders;
    /* allocate memory for the plugin structure */
    dxreminders = g_slice_new0(dxremindersPlugin);
    /* pointer to plugin */
    dxreminders->plugin = plugin;
    /* call widget-create function */
    dxreminders_create_widget(dxreminders);
    /* read the user settings */
    dxreminders_read(dxreminders);
    /* set date and time labels */
    dxreminders_update(dxreminders);
    return dxreminders;
}

static void dxreminders_free(XfcePanelPlugin *plugin, dxremindersPlugin *dxreminders)
{
    GtkWidget *dialog;
    /* check if the dialog is still open. if so, destroy it */
    dialog = g_object_get_data(G_OBJECT (plugin), "dialog");
    if(G_UNLIKELY (dialog != NULL)) gtk_widget_destroy(dialog);
    /* stop timeouts */
    if(dxreminders->timeout_id != 0)
        g_source_remove(dxreminders->timeout_id);
    /* close dxreminders */
    if(dxreminders_running())
    {
        if(dxreminders->sd != -1)
        {
            int rc = send(dxreminders->sd, "<quit>", 6, 0);
            if(rc < 0)
            {
                DBG("send() failed");
            }
        }
        if(dxreminders->pid != 0) killpg(dxreminders->pid, SIGKILL);
    }
    /* destroy the panel widgets */
    gtk_widget_destroy(dxreminders->button);
    /* cleanup the settings */
    g_free(dxreminders->date_font);
    g_free(dxreminders->time_font);
    g_free(dxreminders->date_format);
    g_free(dxreminders->time_format);
    g_free(dxreminders->tooltip);
    /* free the plugin structure */
    g_slice_free(dxremindersPlugin, dxreminders);
}

static gboolean dxreminders_size_changed(G_GNUC_UNUSED XfcePanelPlugin *plugin, G_GNUC_UNUSED gint size, G_GNUC_UNUSED dxremindersPlugin *dxreminders)
{
    return TRUE;
}

static void dxreminders_minicalendar(GtkCheckMenuItem *checkmenuitem, dxremindersPlugin *dxreminders)
{
    dxreminders->calendar = gtk_check_menu_item_get_active(checkmenuitem);
}

static void dxreminders_silentmode(GtkCheckMenuItem *checkmenuitem, dxremindersPlugin *dxreminders)
{
    dxreminders->silentmode = gtk_check_menu_item_get_active(checkmenuitem);
    if(!dxreminders_running())
    {
        dxreminders_start(dxreminders);
        g_usleep(750000);
    }
    dxreminders_connectipc(dxreminders);
    if(dxreminders->sd == -1)
    {
        DBG("IPC doesn't connected");
        return;
    }
    int rc;
    if(dxreminders->silentmode)
    {
        rc = send(dxreminders->sd, "<silenton>", 10, 0);
    }
    else
    {
        rc = send(dxreminders->sd, "<silentoff>", 11, 0);
    }
    if(rc < 0)
    {
        DBG("send() failed");
        return;
    }
}

static void dxreminders_construct(XfcePanelPlugin *plugin)
{
    dxremindersPlugin *dxreminders;
    /* setup transation domain */
    xfce_textdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");
    /* create the plugin */
    dxreminders = dxreminders_new(plugin);
    /* add the ebox to the panel */
    gtk_container_add(GTK_CONTAINER (plugin), dxreminders->button);
    /* show the panel's right-click menu on this ebox */
    xfce_panel_plugin_add_action_widget(plugin, dxreminders->button);
    /* connect plugin signals */
    g_signal_connect(G_OBJECT (plugin), "free-data", G_CALLBACK (dxreminders_free), dxreminders);
    g_signal_connect(G_OBJECT (plugin), "save", G_CALLBACK (dxreminders_save), dxreminders);
    g_signal_connect(G_OBJECT (plugin), "size-changed", G_CALLBACK (dxreminders_size_changed), dxreminders);
    g_signal_connect(G_OBJECT (plugin), "mode-changed", G_CALLBACK (dxreminders_set_mode), dxreminders);
    /* menu item enable mini calendar */
    GtkWidget * item = gtk_check_menu_item_new_with_label(_("Enable Mini Calendar"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM (item), dxreminders->calendar);
    xfce_panel_plugin_menu_insert_item(plugin, GTK_MENU_ITEM (item));
    gtk_widget_show(GTK_WIDGET (item));
    g_signal_connect(G_OBJECT (item), "toggled", G_CALLBACK (dxreminders_minicalendar), dxreminders);
    /* menu item enable silent mode */
    dxreminders->silentcheck = gtk_check_menu_item_new_with_label(_("Enable Silent Mode"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM (dxreminders->silentcheck), dxreminders->silentmode);
    xfce_panel_plugin_menu_insert_item(plugin, GTK_MENU_ITEM (dxreminders->silentcheck));
    gtk_widget_show(GTK_WIDGET (dxreminders->silentcheck));
    g_signal_connect(G_OBJECT (dxreminders->silentcheck), "toggled", G_CALLBACK (dxreminders_silentmode), dxreminders);
    /* menu item to show dialog for add new event */
    item = gtk_menu_item_new_with_label (_("New event"));
    xfce_panel_plugin_menu_insert_item(plugin, GTK_MENU_ITEM (item));
    gtk_widget_show(GTK_WIDGET (item));
    g_signal_connect(G_OBJECT (item), "activate", G_CALLBACK (dxreminders_newevent), dxreminders);
    /* menu item to show mainwindow of dxreminders */
    item = gtk_menu_item_new_with_label ("dxReminders");
    xfce_panel_plugin_menu_insert_item(plugin, GTK_MENU_ITEM (item));
    gtk_widget_show(GTK_WIDGET (item));
    g_signal_connect(G_OBJECT (item), "activate", G_CALLBACK (dxreminders_show_hide), dxreminders);
    /* menu item to restart dxreminders */
    item = gtk_menu_item_new_with_label (_("Restart dxReminders"));
    xfce_panel_plugin_menu_insert_item(plugin, GTK_MENU_ITEM (item));
    gtk_widget_show(GTK_WIDGET (item));
    g_signal_connect(G_OBJECT (item), "activate", G_CALLBACK (dxreminders_restart), dxreminders);
    /* show the configure menu item and connect signal */
    xfce_panel_plugin_menu_show_configure(plugin);
    g_signal_connect (G_OBJECT (plugin), "configure-plugin", G_CALLBACK (dxreminders_configure), dxreminders);
    /* show the about menu item and connect signal */
    xfce_panel_plugin_menu_show_about(plugin);
    g_signal_connect(G_OBJECT (plugin), "about", G_CALLBACK (dxreminders_about), NULL);
    /* check and run dxreminders */
    dxreminders_start(dxreminders);
    g_usleep(750000);
    dxreminders_events_tooltip(dxreminders);
    dxreminders_events_silentmode(dxreminders);
}

static inline guint dxreminders_wake_interval()
{
    GDateTime *dt = g_date_time_new_now_local();
    gint mseconds = g_date_time_get_microsecond(dt);
    g_date_time_unref(dt);
    return UPDATE_INTERVAL - mseconds/1000;
}

gboolean dxreminders_update(dxremindersPlugin *dxreminders)
{
    gchar *utf8str;
    /* stop timer */
    if(dxreminders->timeout_id)
    {
        g_source_remove(dxreminders->timeout_id);
    }
    if(dxreminders->layout != LAYOUT_TIME && dxreminders->date_format != NULL && GTK_IS_LABEL(dxreminders->date_label))
    {
        utf8str = dxreminders_do_utf8strftime(dxreminders->date_format, g_date_time_new_now_local());
        gtk_label_set_text(GTK_LABEL(dxreminders->date_label), utf8str);
        g_free(utf8str);
    }
    if(dxreminders->layout != LAYOUT_DATE && dxreminders->time_format != NULL && GTK_IS_LABEL(dxreminders->time_label))
    {
        utf8str = dxreminders_do_utf8strftime(dxreminders->time_format, g_date_time_new_now_local());
        gtk_label_set_text(GTK_LABEL(dxreminders->time_label), utf8str);
        g_free(utf8str);
    }
    dxreminders->timeout_id = g_timeout_add(dxreminders_wake_interval(), (GSourceFunc) dxreminders_update, dxreminders);
    return TRUE;
}

gchar *dxreminders_do_utf8strftime(const char *format, GDateTime *dt)
{
    gchar *utf8str = NULL;
    utf8str = g_date_time_format(dt, format);
    g_date_time_unref(dt);
    return utf8str;
}

static gboolean dxreminders_query_tooltip(G_GNUC_UNUSED GtkWidget *widget, G_GNUC_UNUSED gint x, G_GNUC_UNUSED gint y, G_GNUC_UNUSED gboolean keyboard_mode, GtkTooltip *tooltip, dxremindersPlugin *dxreminders)
{
    if(dxreminders->layout == LAYOUT_DATE || dxreminders->layout == LAYOUT_TIME)
    {
        gchar *utf8str;
        gchar *format = NULL;
        switch(dxreminders->layout)
        {
            case LAYOUT_TIME:
                format = dxreminders->date_format;
                break;
            case LAYOUT_DATE:
                format = dxreminders->time_format;
                break;
            default:
                break;
        }
        utf8str = g_strconcat(dxreminders_do_utf8strftime(format, g_date_time_new_now_local()),"\n\n",dxreminders->tooltip,NULL);
        gtk_tooltip_set_text(tooltip, utf8str);
        g_free(utf8str);
    }
    else
    {
        gtk_tooltip_set_text(tooltip, dxreminders->tooltip);
    }
    /* if there is no active timeout to update the tooltip, register one */
    if((dxreminders->layout == LAYOUT_DATE || dxreminders->layout == LAYOUT_TIME) && !dxreminders->tooltip_timeout_id)
    {
        dxreminders->tooltip_timeout_id = g_timeout_add(dxreminders_wake_interval(), (GSourceFunc)dxreminders_tooltip_timer, dxreminders);
    }
    return TRUE;
}

static void dxreminders_update_date_font(dxremindersPlugin *dxreminders)
{
#if GTK_CHECK_VERSION (3, 16, 0)
    GtkCssProvider *css_provider;
    gchar * css;
#if GTK_CHECK_VERSION (3, 20, 0)
    PangoFontDescription *font;
    font = pango_font_description_from_string(dxreminders->date_font);
    if(G_LIKELY(font))
    {
        css = g_strdup_printf("label { font-family: %s; font-size: %dpx; font-style: %s; font-weight: %s }",
                              pango_font_description_get_family (font),
                              pango_font_description_get_size (font) / PANGO_SCALE,
                              (pango_font_description_get_style(font) == PANGO_STYLE_ITALIC ||
                              pango_font_description_get_style(font) == PANGO_STYLE_OBLIQUE) ? "italic" : "normal",
                              (pango_font_description_get_weight(font) >= PANGO_WEIGHT_BOLD) ? "bold" : "normal");
        pango_font_description_free (font);
    }
    else
        css = g_strdup_printf("label { font: %s; }",
#else
    css = g_strdup_printf(".label { font: %s; }",
#endif
                              dxreminders->date_font);
    /* Setup Gtk style */
    DBG("css: %s",css);
    css_provider = gtk_css_provider_new ();
    gtk_css_provider_load_from_data (css_provider, css, strlen(css), NULL);
    gtk_style_context_add_provider(GTK_STYLE_CONTEXT(gtk_widget_get_style_context(GTK_WIDGET(dxreminders->date_label))),
                                   GTK_STYLE_PROVIDER (css_provider),
                                   GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
    g_free(css);
#else
    PangoFontDescription *font;
    font = pango_font_description_from_string(dxreminders->date_font);
    if(G_LIKELY(font))
    {
        gtk_widget_override_font(dxreminders->date_label, font);
        pango_font_description_free(font);
    }
#endif
}

static void dxreminders_update_time_font(dxremindersPlugin *dxreminders)
{
#if GTK_CHECK_VERSION (3, 16, 0)
    GtkCssProvider *css_provider;
    gchar * css;
#if GTK_CHECK_VERSION (3, 20, 0)
    PangoFontDescription *font;
    font = pango_font_description_from_string(dxreminders->time_font);
    if(G_LIKELY(font))
    {
        css = g_strdup_printf("label { font-family: %s; font-size: %dpx; font-style: %s; font-weight: %s }",
                          pango_font_description_get_family (font),
                          pango_font_description_get_size (font) / PANGO_SCALE,
                          (pango_font_description_get_style(font) == PANGO_STYLE_ITALIC ||
                           pango_font_description_get_style(font) == PANGO_STYLE_OBLIQUE) ? "italic" : "normal",
                          (pango_font_description_get_weight(font) >= PANGO_WEIGHT_BOLD) ? "bold" : "normal");
        pango_font_description_free (font);
    }
    else
        css = g_strdup_printf("label { font: %s; }",
#else
    css = g_strdup_printf(".label { font: %s; }",
#endif
                             dxreminders->time_font);
    /* Setup Gtk style */
    DBG("css: %s",css);
    css_provider = gtk_css_provider_new();
    gtk_css_provider_load_from_data(css_provider, css, strlen(css), NULL);
    gtk_style_context_add_provider(GTK_STYLE_CONTEXT (gtk_widget_get_style_context (GTK_WIDGET (dxreminders->time_label))),
                                   GTK_STYLE_PROVIDER (css_provider),
                                   GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
    g_free(css);
#else
    PangoFontDescription *font;
    font = pango_font_description_from_string(dxreminders->time_font);
    if(G_LIKELY(font))
    {
        gtk_widget_override_font(dxreminders->time_label, font);
        pango_font_description_free(font);
    }
#endif
}

void dxreminders_apply_font(dxremindersPlugin *dxreminders, const gchar *date_font_name, const gchar *time_font_name)
{
    if(date_font_name != NULL)
    {
        g_free(dxreminders->date_font);
        dxreminders->date_font = g_strdup(date_font_name);
        dxreminders_update_date_font(dxreminders);
    }

    if (time_font_name != NULL)
    {
        g_free(dxreminders->time_font);
        dxreminders->time_font = g_strdup(time_font_name);
        dxreminders_update_time_font(dxreminders);
    }
}

void dxreminders_apply_format(dxremindersPlugin *dxreminders, const gchar *date_format, const gchar *time_format)
{
    if(dxreminders == NULL)
        return;
    if(date_format != NULL)
    {
        g_free(dxreminders->date_format);
        dxreminders->date_format = g_strdup(date_format);
    }
    if(time_format != NULL)
    {
        g_free(dxreminders->time_format);
        dxreminders->time_format = g_strdup(time_format);
    }
}

void dxreminders_apply_layout(dxremindersPlugin *dxreminders, dx_layout layout)
{
    if(layout < LAYOUT_COUNT)
    {
        dxreminders->layout = layout;
    }
    /* hide labels based on layout-selection */
    gtk_widget_show(GTK_WIDGET(dxreminders->time_label));
    gtk_widget_show(GTK_WIDGET(dxreminders->date_label));
    switch(dxreminders->layout)
    {
        case LAYOUT_DATE:
            gtk_widget_hide(GTK_WIDGET(dxreminders->time_label));
            break;
        case LAYOUT_TIME:
            gtk_widget_hide(GTK_WIDGET(dxreminders->date_label));
            break;
        default:
            break;
    }
    /* update tooltip handler */
    if(dxreminders->tooltip_handler_id)
    {
        g_signal_handler_disconnect(dxreminders->button, dxreminders->tooltip_handler_id);
        dxreminders->tooltip_handler_id = 0;
    }
    gtk_widget_set_has_tooltip(GTK_WIDGET(dxreminders->button), TRUE);
    dxreminders->tooltip_handler_id = g_signal_connect(dxreminders->button, "query-tooltip", G_CALLBACK(dxreminders_query_tooltip), dxreminders);
    /* set order based on layout-selection */
    switch(dxreminders->layout)
    {
        case LAYOUT_TIME_DATE:
            gtk_box_reorder_child(GTK_BOX(dxreminders->box), dxreminders->time_label, 0);
            gtk_box_reorder_child(GTK_BOX(dxreminders->box), dxreminders->date_label, 1);
            break;
        default:
            gtk_box_reorder_child(GTK_BOX(dxreminders->box), dxreminders->time_label, 1);
            gtk_box_reorder_child(GTK_BOX(dxreminders->box), dxreminders->date_label, 0);
    }
}

/* register the plugin */
XFCE_PANEL_PLUGIN_REGISTER (dxreminders_construct);
