/*
 * Copyright (C) 2003  Robert Collins  <robertc@squid-cache.org>
 * 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * DO NOT ALTER THE NEXT LINE
 * arch-tag: 21df1c64-1d6a-4476-ad3a-e451889d3da8
 * 
 */

#include "ConfigArchSource.h"
#include <iostream>
#include <stdexcept>
#include <fstream>
#include "Path.h"

using namespace std;

/* ConfigArchSource */
StringOption ConfigArchSource::buildOptions("",'a',"arch-build-options", "Options to pass to tla get.");

ConfigArchSource::ConfigArchSource() : theUrl(), location()
{
}

ConfigArchSource::ConfigArchSource(string const &aUrl) : theUrl(aUrl)
{
    if (theUrl.find("arch://") == 0)
	theUrl.erase(0,7);
    /* We only support @ location syntax in full archive paths */
    if (theUrl.find("@") != theUrl.rfind("@"))
      {
	string::size_type pos = theUrl.find("@", theUrl.find("@") + 1);
	location = theUrl.substr(pos + 1);
	theUrl.erase(pos);
	if (Verbose())
	    cout << "Found archive " << theUrl << " location at " << location << endl;
      }
}

string
ConfigArchSource::url(bool const &stripScheme) const
{
    return (stripScheme ? "" : "arch://") + theUrl;
}

string
ConfigArchSource::archive() const
{
    return theUrl;
}

void
ConfigArchSource::maybeRegister() const
{
    string command = "tla whereis-archive `tla parse-package-name -a " + archive() + "` > /dev/null";
    if (run_query (command, "tla") != 0)
	run ("tla register-archive `tla parse-package-name -a " + archive() + "` " + location ,  "tla");
}

void
ConfigArchSource::get (string const &where) const throw(std::exception *)
{
    maybeRegister();
    string command = "tla get " + (string)buildOptions + " " + archive() + " " + where;
    run (command, "tla");
}

void
ConfigArchSource::update (string const &where) const throw(std::exception *)
{
    maybeRegister();
    string command = "tla update " + (string)buildOptions + " -d " + where + " " + archive();
    run (command, "tla");
}

int
ConfigArchSource::missing (string const &where) const throw(std::exception *)
{
    maybeRegister();
    string command = "tla missing " + (string)buildOptions + " -d " + where + " " + archive();
    return run_query (command, "tla");
}

int
ConfigArchSource::changes (string const &where) const throw(std::exception *)
{
    maybeRegister();
    string command = "tla changes " + (string)buildOptions + " -d " + where + " " + archive();
    return run_query (command, "tla");
}

string
ConfigArchSource::treeVersion(string const &where) const
{
    string command = "tla logs -f -r -d " + where + " | head -n1 > ,,config-tree";
    run(command, "tla");

    ifstream resultbuffer(",,config-tree");
    string result;
    resultbuffer >> result;
    resultbuffer.close();
    unlink(",,config-tree");
    return result;
}

void
ConfigArchSource::ignore (ConfigSource const* aSource, string const &where) const throw (std::exception *)
{
    if (dynamic_cast<ConfigArchSource const *>( aSource))
	return;
    
    ifstream projtreetest((where + "/{arch}/.arch-project-tree").c_str());
    if (projtreetest.is_open())
      {
	std::cerr << "Warning: '" << where << "' is already an arch project tree. Consider using an arch:// source in the config." << std::endl;
      }
      
    string command = "tla init-tree --nested -d " + where;
    run (command, "tla");
    ofstream taggingMethod((where + "/{arch}/=tagging-method").c_str());
    taggingMethod << dummyTaggingText;
    std::cout << "Created a dummy tagging file for " << where << "/{arch}/=tagging-method" << endl;
}

char const * ConfigArchSource::dummyTaggingText=
"# this is a GNU Arch configuration file. It's been placed here by\n"
"# config-manager because this directory was created as part of a\n"
"# build operation, to ensure that Arch ignores this sub-tree.\n"
"explicit\n"
"untagged-source precious\n"
"exclude ^.*$\n"
"junk ^$\n"
"precious ^.*$\n"
"backup ^$\n"
"unrecognized ^$\n"
"source ^$\n";

ConfigArchSource*
ConfigArchSource::Create(Path const &location) throw (std::exception *)
{
    ifstream projtreetest((location.fullName() + "/{arch}/++default-version").c_str());
    if (!projtreetest.is_open())
	return NULL;

    ConfigArchSource *result = new ConfigArchSource;
    result->theUrl = result->treeVersion(location.fullName());
    result->location = "";
    return result;
}
