#ifndef _CLIENT_H
#define _CLIENT_H 1

// A Client represents a single connection from a client (watching a single stream).

#include <stddef.h>
#include <time.h>
#include <string>

class ClientProto;
struct Stream;

// Digested statistics for writing to logs etc.
struct ClientStats {
	std::string url;
	int sock;
	int fwmark;
	std::string remote_addr;
	time_t connect_time;
	size_t bytes_sent;
	size_t bytes_lost;
	size_t num_loss_events;
};

struct Client {
	Client(int sock);

	// Serialization/deserialization.
	Client(const ClientProto &serialized, Stream *stream);
	ClientProto serialize() const;

	ClientStats get_stats() const;

	// The file descriptor associated with this socket.
	int sock;

	// The fwmark associated with this socket (or 0).
	int fwmark;

	// Some information only used for logging.
	std::string remote_addr;
	time_t connect_time;

	enum State { READING_REQUEST, SENDING_HEADER, SENDING_DATA, SENDING_ERROR, WAITING_FOR_KEYFRAME };
	State state;

	// The HTTP request, as sent by the client. If we are in READING_REQUEST,
	// this might not be finished.
	std::string request;

	// What stream we're connecting to; parsed from <request>.
	// Not relevant for READING_REQUEST.
	std::string url;
	Stream *stream;

	// The header we want to send. This is nominally a copy of Stream::header,
	// but since that might change on reconnects etc., we keep a local copy here.
	// Only relevant for SENDING_HEADER or SENDING_ERROR; blank otherwise.
	std::string header_or_error;

	// Number of bytes we've sent of the header. Only relevant for SENDING_HEADER
	// or SENDING_ERROR.
	size_t header_or_error_bytes_sent;

	// Number of bytes we are into the stream (ie., the end of last send).
	// -1 means we want to send from the end of the backlog (the normal case),
	// although only at a keyframe.
	// -2 means we want to send from the _beginning_ of the backlog.
	// Once we go into WAITING_FOR_KEYFRAME or SENDING_DATA, these negative
	// values will be translated to real numbers.
	size_t stream_pos;

	// Number of bytes we've sent of data. Only relevant for SENDING_DATA.
	size_t bytes_sent;

	// Number of times we've skipped forward due to the backlog being too big,
	// and how many bytes we've skipped over in all. Only relevant for SENDING_DATA.
	size_t bytes_lost, num_loss_events;
};

#endif  // !defined(_CLIENT_H)
