#include <ctype.h>
#include <string.h>
#include <string>
#include <vector>

#include "log.h"
#include "parse.h"

using namespace std;

vector<string> split_tokens(const string &line)
{
	vector<string> ret;
	string current_token;

	for (size_t i = 0; i < line.size(); ++i) {
		if (isspace(line[i])) {
			if (!current_token.empty()) {
				ret.push_back(current_token);
			}
			current_token.clear();
		} else {
			current_token.push_back(line[i]);
		}
	}
	if (!current_token.empty()) {
		ret.push_back(current_token);
	}
	return ret;
}

vector<string> split_lines(const string &str)
{
	vector<string> ret;
	string current_line;

	for (size_t i = 0; i < str.size(); ++i) {
		// Skip \r if followed by an \n.
		if (str[i] == '\r' && i < str.size() - 1 && str[i + 1] == '\n') {
			continue;
		}

		// End of the current line?
		if (str[i] == '\n') {
			if (!current_line.empty()) {
				ret.push_back(current_line);
			}
			current_line.clear();
		} else {
			current_line.push_back(str[i]);
		}
	}
	if (!current_line.empty()) {
		ret.push_back(current_line);
	}
	return ret;
}

HTTPHeaderMultimap extract_headers(const vector<string> &lines, const string &log_context)
{
	HTTPHeaderMultimap parameters;
	for (size_t i = 1; i < lines.size(); ++i) {
		size_t split = lines[i].find(":");
		if (split == string::npos) {
			log(WARNING, "[%s] Ignoring malformed HTTP response line '%s'",
			        log_context.c_str(), lines[i].c_str());
			continue;
		}

		string key(lines[i].begin(), lines[i].begin() + split);

		// Skip any spaces after the colon.
		do {
			++split;
		} while (split < lines[i].size() && (lines[i][split] == ' ' || lines[i][split] == '\t'));

		string value(lines[i].begin() + split, lines[i].end());

		parameters.insert(make_pair(key, value));
	}

	return parameters;
}

#define MAX_REQUEST_SIZE 16384  /* 16 kB. */

RequestParseStatus wait_for_double_newline(string *existing_data, const char *new_data, size_t new_data_size)
{
	// Guard against overlong requests gobbling up all of our space.
	if (existing_data->size() + new_data_size > MAX_REQUEST_SIZE) {
		return RP_OUT_OF_SPACE;
	}	

	// See if we have \r\n\r\n anywhere in the request. We start three bytes
	// before what we just appended, in case we just got the final character.
	size_t existing_data_bytes = existing_data->size();
	existing_data->append(string(new_data, new_data + new_data_size));
	
	const size_t start_at = (existing_data_bytes >= 3 ? existing_data_bytes - 3 : 0);
	const char *ptr = reinterpret_cast<char *>(
		memmem(existing_data->data() + start_at, existing_data->size() - start_at,
		       "\r\n\r\n", 4));
	if (ptr == nullptr) {
		return RP_NOT_FINISHED_YET;
	}
	if (ptr != existing_data->data() + existing_data->size() - 4) {
		return RP_EXTRA_DATA;
	}
	return RP_FINISHED;
}
