/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "publisher.h"

Publisher::Publisher(PumpController *pumpController,
                     QWidget *parent) : QWidget(parent)
{
    this->pController = pumpController;
    connect(pController, SIGNAL(postPublished()),
            this, SLOT(onPublishingOk()));
    connect(pController, SIGNAL(postPublishingFailed()),
            this, SLOT(onPublishingFailed()));

    // After receiving the list of lists, update the "Lists" submenus
    connect(pController, SIGNAL(listsListReceived(QVariantList)),
            this, SLOT(updateListsMenus(QVariantList)));


    this->editingMode = false; // False unless set from setEditingMode
                               // after clicking "Edit" in a post

    this->setFocusPolicy(Qt::StrongFocus); // To keep the publisher from getting focus by accident


    this->audienceSelectorTo = new AudienceSelector(pController, "to", this);
    connect(audienceSelectorTo, SIGNAL(audienceSelected(QString,QStringList)),
            this, SLOT(updateToCcFields(QString,QStringList)));
    this->audienceSelectorCC = new AudienceSelector(pController, "cc", this);
    connect(audienceSelectorCC, SIGNAL(audienceSelected(QString,QStringList)),
            this, SLOT(updateToCcFields(QString,QStringList)));



    titleLabel = new QLabel(tr("Title") + ":");
    titleLabel->setSizePolicy(QSizePolicy::Minimum,
                              QSizePolicy::Maximum);

    titleLineEdit = new QLineEdit();
    titleLineEdit->setSizePolicy(QSizePolicy::MinimumExpanding,
                                 QSizePolicy::Maximum);
    titleLineEdit->setPlaceholderText(tr("Optional title for the post"));
    titleLineEdit->setToolTip(tr("Title for the post. Setting a title "
                                 "helps make the Meanwhile feed "
                                 "more informative.") + "<b></b>");


    pictureLabel = new QLabel();
    pictureLabel->setAlignment(Qt::AlignCenter);
    //pictureLabel->setScaledContents(true);
    pictureLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
    pictureLabel->hide();


    pictureInfoLabel = new QLabel();
    pictureInfoLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);

    selectPictureButton = new QPushButton(QIcon::fromTheme("insert-image"),
                                          tr("Select Picture..."));
    selectPictureButton->setSizePolicy(QSizePolicy::MinimumExpanding,
                                       QSizePolicy::Maximum);
    selectPictureButton->setToolTip(tr("Find the picture in your folders"));
    connect(selectPictureButton, SIGNAL(clicked()),
            this, SLOT(findPictureFile()));
    selectPictureButton->hide();



    // Set default pixmap and "picture not set" message
    this->setEmptyPictureData();
    lastUsedDirectory = QDir::homePath();


    // Composer
    composerBox = new Composer(true); // forPublisher = true
    composerBox->setSizePolicy(QSizePolicy::Minimum,
                               QSizePolicy::Minimum);
    connect(composerBox, SIGNAL(focusReceived()),    this, SLOT(setFullMode()));
    connect(composerBox, SIGNAL(editingFinished()),  this, SLOT(sendPost()));
    connect(composerBox, SIGNAL(editingCancelled()), this, SLOT(setMinimumMode()));

    // Add formatting button exported from Composer
    this->toolsButton = composerBox->getToolsButton();


    // To... menu
    toPublicAction = new QAction(tr("Public"), this);
    toPublicAction->setCheckable(true);
    connect(toPublicAction, SIGNAL(toggled(bool)),
            this, SLOT(setToPublic(bool)));

    toFollowersAction = new QAction(tr("Followers"), this);
    toFollowersAction->setCheckable(true);
    connect(toFollowersAction, SIGNAL(toggled(bool)),
            this, SLOT(setToFollowers(bool)));

    toSelectorListsMenu = new QMenu(tr("Lists"));
    toSelectorListsMenu->setDisabled(true); // Disabled until lists are received, if any
    connect(toSelectorListsMenu, SIGNAL(triggered(QAction*)),
            this, SLOT(updateToListsFields(QAction*)));

    toSelectorMenu = new QMenu("to-menu");
    toSelectorMenu->addAction(toPublicAction);
    toSelectorMenu->addAction(toFollowersAction);
    toSelectorMenu->addMenu(toSelectorListsMenu);
    toSelectorMenu->addSeparator();
    toSelectorMenu->addAction(tr("People..."), audienceSelectorTo, SLOT(show()));

    toSelectorButton = new QPushButton(QIcon::fromTheme("system-users"),
                                     tr("To..."));
    toSelectorButton->setToolTip(tr("Select who will see this post"));
    toSelectorButton->setMenu(toSelectorMenu);



    // CC... menu
    ccPublicAction = new QAction(tr("Public"), this);
    ccPublicAction->setCheckable(true);
    connect(ccPublicAction, SIGNAL(toggled(bool)),
            this, SLOT(setCCPublic(bool)));

    ccFollowersAction = new QAction(tr("Followers"), this);
    ccFollowersAction->setCheckable(true);
    connect(ccFollowersAction, SIGNAL(toggled(bool)),
            this, SLOT(setCCFollowers(bool)));

    ccSelectorListsMenu = new QMenu(tr("Lists"));
    ccSelectorListsMenu->setDisabled(true); // Disabled until lists are received, if any
    connect(ccSelectorListsMenu, SIGNAL(triggered(QAction*)),
            this, SLOT(updateCcListsFields(QAction*)));

    ccSelectorMenu = new QMenu("cc-menu");
    ccSelectorMenu->addAction(ccPublicAction);
    ccSelectorMenu->addAction(ccFollowersAction);
    ccSelectorMenu->addMenu(ccSelectorListsMenu);
    ccSelectorMenu->addSeparator();
    ccSelectorMenu->addAction(tr("People..."), audienceSelectorCC, SLOT(show()));

    ccSelectorButton = new QPushButton(QIcon::fromTheme("system-users"),
                                     tr("CC..."));
    ccSelectorButton->setToolTip(tr("Select who will get a copy of this post"));
    ccSelectorButton->setMenu(ccSelectorMenu);


    QFont audienceLabelsFont; // "To" column will be normal, "CC" will be italic
    audienceLabelsFont.setPointSize(audienceLabelsFont.pointSize()-1);

    // These will hold the names of the *people* selected for the To or CC fields, if any
    toAudienceLabel = new QLabel();
    toAudienceLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    toAudienceLabel->setFont(audienceLabelsFont);  // Normal
    toAudienceLabel->setWordWrap(true);

    ccAudienceLabel = new QLabel();
    ccAudienceLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    audienceLabelsFont.setItalic(true);
    ccAudienceLabel->setFont(audienceLabelsFont);  // Italic
    ccAudienceLabel->setWordWrap(true);

    // And these will show "public" or "followers" current selection (in bold)
    toPublicFollowersLabel = new QLabel();
    toPublicFollowersLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    audienceLabelsFont.setBold(true);
    audienceLabelsFont.setItalic(false);
    toPublicFollowersLabel->setFont(audienceLabelsFont);  // Bold

    ccPublicFollowersLabel = new QLabel();
    ccPublicFollowersLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    audienceLabelsFont.setItalic(true);
    ccPublicFollowersLabel->setFont(audienceLabelsFont);  // Bold + italic


    // To set the 'picture mode'
    pictureButton = new QPushButton(QIcon::fromTheme("camera-photo"),
                                    tr("Ad&d Picture"));
    pictureButton->setToolTip(tr("Upload photo"));
    connect(pictureButton, SIGNAL(clicked()),
            this, SLOT(setPictureMode()));


    // Sending status info label
    statusInfoLabel = new QLabel();
    statusInfoLabel->setSizePolicy(QSizePolicy::MinimumExpanding,
                                   QSizePolicy::Maximum);
    statusInfoLabel->setAlignment(Qt::AlignCenter);
    statusInfoLabel->setWordWrap(true);
    audienceLabelsFont.setBold(false);
    audienceLabelsFont.setItalic(false);
    statusInfoLabel->setFont(audienceLabelsFont);


    // To send the post
    postButton = new QPushButton(QIcon::fromTheme("mail-send"),
                                  tr("Post"));
    postButton->setToolTip(tr("Hit Control+Enter to post with the keyboard"));
    connect(postButton, SIGNAL(clicked()),
            this, SLOT(sendPost()));

    cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel"),
                                   tr("Cancel"));
    cancelButton->setToolTip(tr("Cancel the post"));
    connect(cancelButton, SIGNAL(clicked()),
            composerBox, SLOT(cancelPost()));


    // Now the layout, starting with the Title field and "picture mode" stuff
    titleLayout = new QHBoxLayout();
    titleLayout->addWidget(titleLabel);
    titleLayout->addWidget(titleLineEdit);
    titleLayout->addSpacing(12);
    titleLayout->addWidget(toolsButton);

    mainLayout = new QGridLayout();
    mainLayout->setVerticalSpacing(1);
    mainLayout->setContentsMargins(1, 1, 1, 1);

    mainLayout->addLayout(titleLayout,         0, 0, 1, 8, Qt::AlignTop | Qt::AlignLeft);
    mainLayout->addWidget(pictureLabel,        1, 0, 2, 4);
    mainLayout->addWidget(pictureInfoLabel,    1, 4, 1, 4, Qt::AlignTop);
    mainLayout->addWidget(selectPictureButton, 2, 4, 1, 4, Qt::AlignBottom | Qt::AlignLeft);

    mainLayout->addWidget(composerBox,         3, 0, 3, 8); // 3 rows, 8 columns

    mainLayout->addWidget(toSelectorButton,    6, 0, 1, 1, Qt::AlignLeft);
    mainLayout->addWidget(ccSelectorButton,    6, 1, 1, 1, Qt::AlignLeft);
    mainLayout->addWidget(pictureButton,       6, 3, 1, 2, Qt::AlignLeft);
    mainLayout->addWidget(statusInfoLabel,     6, 5, 3, 2, Qt::AlignCenter);
    mainLayout->addWidget(postButton,          6, 7, 1, 1);


    // The 2 labels holding people's names, if any
    mainLayout->addWidget(toAudienceLabel, 7, 0, 1, 1);
    mainLayout->addWidget(ccAudienceLabel, 7, 1, 1, 1);

    // The 2 labels holding "public/followers", if selected
    mainLayout->addWidget(toPublicFollowersLabel, 8, 0, 1, 1);
    mainLayout->addWidget(ccPublicFollowersLabel, 8, 1, 1, 1);

    // The "Cancel post" button
    mainLayout->addWidget(cancelButton,        8, 7, 1, 1);

    this->setLayout(mainLayout);


    this->setMinimumMode();

    qDebug() << "Publisher created";
}


Publisher::~Publisher()
{
    qDebug() << "Publisher destroyed";
}


/*
 * Set if "public" option for audience is checked as default
 *
 */
void Publisher::setDefaultPublicPosting(bool defaultPublicPosts)
{
    this->defaultPublicPosting = defaultPublicPosts;

    // Ensure status is sync'ed
    this->toSelectorMenu->actions().at(0)->setChecked(this->defaultPublicPosting);
}


/*
 * Set default "no photo" pixmap and "picture not set" message
 *
 * Clear the filename and content type variables too
 */
void Publisher::setEmptyPictureData()
{
    pictureLabel->setPixmap(QIcon::fromTheme("image-x-generic")
                            .pixmap(200, 150)
                            .scaled(200, 150,
                                    Qt::IgnoreAspectRatio,
                                    Qt::SmoothTransformation));
    pictureLabel->setToolTip(tr("Picture not set"));

    pictureInfoLabel->clear();

    this->pictureFilename.clear();
    this->pictureContentType.clear();
}


void Publisher::updatePublicFollowersLabels()
{
    QString toString;
    foreach (QString listName, toListsNameStringList)
    {
        toString.append(QString::fromUtf8("\342\236\224 ") // arrow sign in front
                        + listName + "\n");
    }
    if (toPublicAction->isChecked())
    {
        toString.append("+" + tr("Public") + "\n");
    }
    if (toFollowersAction->isChecked())
    {
        toString.append("+" + tr("Followers") + "\n");
    }
    toPublicFollowersLabel->setText(toString);


    QString ccString;
    foreach (QString listName, ccListsNameStringList)
    {
        ccString.append(QString::fromUtf8("\342\236\224 ") // arrow sign
                        + listName + "\n");
    }
    if (ccPublicAction->isChecked())
    {
        ccString.append("+" + tr("Public") + "\n");
    }
    if (ccFollowersAction->isChecked())
    {
        ccString.append("+" + tr("Followers") + "\n");
    }
    ccPublicFollowersLabel->setText(ccString);
}



/*
 * Create a key:value map, listing who will receive a post, like:
 *
 * "collection" : "http://activityschema.org/collection/public"
 * "collection" : "https://pump.example/api/user/followers"
 * "person"     : "acct:somecontact@pumpserver.example"
 *
 */
QMap<QString,QString> Publisher::getAudienceMap()
{
    QMap<QString,QString> audienceMap;


    // To: Public is checked
    if (toPublicAction->isChecked())
    {
        audienceMap.insertMulti("to|collection", "http://activityschema.org/collection/public");
    }

    // To: List of individual people
    foreach (QString address, this->toAddressStringList)
    {
        audienceMap.insertMulti("to|person", "acct:" + address);
    }

    // To: Lists
    foreach (QString address, this->toListsIdStringList)
    {
        audienceMap.insertMulti("to|collection", address);
    }


    // To: Followers is checked
    if (toFollowersAction->isChecked())
    {
        audienceMap.insertMulti("to|collection", this->pController->currentFollowersUrl());
    }


    // CC: Public is checked
    if (ccPublicAction->isChecked())
    {
        audienceMap.insertMulti("cc|collection", "http://activityschema.org/collection/public");
    }

    // CC: List of individual people
    foreach (QString address, this->ccAddressStringList)
    {
        audienceMap.insertMulti("cc|person", "acct:" + address);
    }

    // CC: Lists
    foreach (QString address, this->ccListsIdStringList)
    {
        audienceMap.insertMulti("cc|collection", address);
    }



    // Last check: if CC: Followers is checked, or nothing is, add Followers
    if (ccFollowersAction->isChecked() || audienceMap.isEmpty())
    {
        audienceMap.insertMulti("cc|collection", this->pController->currentFollowersUrl());
    }


    return audienceMap;
}



/********************************************************************/
/***************************** SLOTS ********************************/
/********************************************************************/



void Publisher::setMinimumMode()
{
    qDebug() << "setting Publisher to minimum mode";

    this->postButton->setFocus(); // Give focus to button,
                                  // in case user shared with Ctrl+Enter

    // Disable possible scrollbars
    this->composerBox->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    this->composerBox->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    // ~1 row
    int composerHeight = composerBox->getMessageLabelHeight();
    this->composerBox->setMinimumHeight(composerHeight);
    this->composerBox->setMaximumHeight(composerHeight);

    this->setMinimumHeight(composerHeight + 2);
    this->setMaximumHeight(composerHeight + 2);


    this->audienceSelectorTo->resetLists();
    this->audienceSelectorCC->resetLists();

    this->toAudienceLabel->setText("...");
    toAudienceLabel->repaint(); // Avoid a flicker-like effect later
    toAudienceLabel->clear();
    toAudienceLabel->hide();

    this->toAddressStringList.clear();
    this->toListsNameStringList.clear();
    this->toListsIdStringList.clear();

    toPublicFollowersLabel->clear();
    toPublicFollowersLabel->hide();


    this->ccAudienceLabel->setText("...");
    ccAudienceLabel->repaint();
    ccAudienceLabel->clear();
    ccAudienceLabel->hide();

    this->ccAddressStringList.clear();
    this->ccListsNameStringList.clear();
    this->ccListsIdStringList.clear();

    ccPublicFollowersLabel->clear();
    ccPublicFollowersLabel->hide();


    this->toSelectorButton->hide();
    this->ccSelectorButton->hide();

    // Check "public" if "public posts" is set in the preferences
    toPublicAction->setChecked(this->defaultPublicPosting);
    toFollowersAction->setChecked(false);

    ccPublicAction->setChecked(false);
    ccFollowersAction->setChecked(true);  // CC: Followers by default

    // Uncheck the lists, very TMP / FIXME
    // This doesn't clear the To/CC labels sometimes
    foreach (QAction *action, toSelectorListsMenu->actions())
    {
        action->setChecked(false);
    }
    foreach (QAction *action, ccSelectorListsMenu->actions())
    {
        action->setChecked(false);
    }


    titleLabel->hide();
    this->titleLineEdit->clear();
    titleLineEdit->hide();

    toolsButton->hide();

    this->statusInfoLabel->clear();
    statusInfoLabel->hide();
    this->pictureButton->hide();
    this->postButton->hide();
    this->cancelButton->hide();


    // Clear formatting options like bold, italic and underline
    this->composerBox->setCurrentCharFormat(QTextCharFormat());

    // Hide "picture mode" controls
    this->pictureLabel->hide();
    this->pictureInfoLabel->hide();
    this->selectPictureButton->hide();

    this->setEmptyPictureData();


    // Clear "editing mode", restore stuff
    if (editingMode)
    {
        this->editingMode = false;
        this->editingPostId.clear();

        this->postButton->setText(tr("Post")); // Button text back to "Post" as usual
        this->toSelectorButton->setEnabled(true);
        this->ccSelectorButton->setEnabled(true);
        this->pictureButton->setEnabled(true);
    }
}




void Publisher::setFullMode()
{
    qDebug() << "setting Publisher to full mode";

    this->titleLabel->show();
    this->titleLineEdit->show();

    this->toolsButton->show();

    this->composerBox->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    this->composerBox->setHorizontalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    this->composerBox->setMaximumHeight(2048);
    this->setMaximumHeight(2048); // i.e. "unlimited"

    this->toSelectorButton->show();
    this->ccSelectorButton->show();

    this->toAudienceLabel->show();
    this->toPublicFollowersLabel->show();

    this->ccAudienceLabel->show();
    this->ccPublicFollowersLabel->show();

    updatePublicFollowersLabels();

    // Avoid re-enabling the picture button when re-focusing publisher, but still in picture mode...
    if (pictureButton->isHidden())
    {
        this->pictureButton->setEnabled(true); // If it wasn't hidden, don't re-enable
    }
    this->pictureButton->show();
    this->statusInfoLabel->show();
    this->postButton->show();
    this->cancelButton->show();

    this->composerBox->setFocus(); // In case user used menu or shortcut
                                   // instead of clicking on it
}




void Publisher::setPictureMode()
{
    this->pictureButton->setDisabled(true);

    this->pictureLabel->show();

    this->pictureInfoLabel->show();
    this->selectPictureButton->show();

    this->findPictureFile(); // Show the open file dialog directly
}


/*
 * Set Publiser to edit mode, after user clicks on "Edit" in a post
 *
 */
void Publisher::setEditingMode(QString postID, QString postText,
                               QString postTitle)
{
    if (postButton->isVisible())  // FIXME, use a proper state variable
    {
        // TMPFIX; temporary way of preventing the "Edit" option
        // from destroying a post currently being composed!
        QMessageBox::warning(this, tr("Error: Already composing"),
                             tr("You can't edit a post at this time, "
                                "because a post is already being composed."));
        return;
    }

    this->editingMode = true;
    this->editingPostId = postID;
    setFullMode();


    // Fill in the contents of the post
    this->titleLineEdit->setText(postTitle);
    this->composerBox->setText(postText);

    // Change/disable some controls
    this->postButton->setText(tr("Update"));
    this->toSelectorButton->setDisabled(true);
    this->ccSelectorButton->setDisabled(true);
    this->pictureButton->setDisabled(true);


    this->statusInfoLabel->setText(tr("Editing post"));
}


/*
 * After the post is confirmed to have been received by the server
 * re-enable publisher, clear text, etc.
 *
 */
void Publisher::onPublishingOk()
{
    this->statusInfoLabel->clear();
    this->setEnabled(true);

    this->composerBox->erase();

    // Done composing message, hide buttons until we get focus again
    setMinimumMode();
}

/*
 * If there was a HTTP error while posting...
 *
 */
void Publisher::onPublishingFailed()
{
    qDebug() << "Posting failed, re-enabling Publisher";
    this->statusInfoLabel->setText(tr("Posting failed.\n\nTry again."));
    this->setEnabled(true);
    this->composerBox->setFocus();
}


/*
 * These are called when selecting Public or Followers in the menus
 *
 * When selecting "CC: Followers", "To: Followers" gets unselected, etc.
 *
 */
void Publisher::setToPublic(bool activated)
{
    if (activated)
    {
        ccPublicAction->setChecked(false);     // Unselect "CC: Public"
    }
    updatePublicFollowersLabels();
}

void Publisher::setToFollowers(bool activated)
{
    if (activated)
    {
        ccFollowersAction->setChecked(false);  // Unselect "CC: Followers"
    }
    updatePublicFollowersLabels();
}

void Publisher::setCCPublic(bool activated)
{
    if (activated)
    {
        toPublicAction->setChecked(false);     // Unselect "To: Public"
    }
    updatePublicFollowersLabels();
}

void Publisher::setCCFollowers(bool activated)
{
    if (activated)
    {
        toFollowersAction->setChecked(false);  // Unselect "To: Followers"
    }
    updatePublicFollowersLabels();
}




/*
 * Receive a list of addresses for the To or CC fields
 *
 */
void Publisher::updateToCcFields(QString selectorType,
                                 QStringList contactsList)
{
    qDebug() << "Publisher::updateToCcFields()" << selectorType << contactsList;

    QRegExp contactRE("(.+)\\s+\\<(.+@.+)\\>", Qt::CaseInsensitive);
    contactRE.setMinimal(true);

    QString addressesString;
    QStringList addressesStringList;

    foreach (QString contactString, contactsList)
    {
        contactRE.indexIn(contactString);

        addressesString.append(contactRE.cap(1).trimmed());
        addressesString.append(", ");

        addressesStringList.append(contactRE.cap(2).trimmed());
    }
    addressesString.remove(-2, 2); // remove ", " at the end


    if (selectorType == "to")
    {
        this->toAudienceLabel->setText(addressesString);
        this->toAddressStringList = addressesStringList;
    }
    else   // "cc"
    {
        this->ccAudienceLabel->setText(addressesString);
        this->ccAddressStringList = addressesStringList;
    }

    qDebug() << "Names:" << addressesString;
    qDebug() << "Addresses:" << addressesStringList;
}



/*
 * Fill the "Lists" submenus with PumpController's lists info
 *
 *
 */
void Publisher::updateListsMenus(QVariantList listsList)
{
    // First, clear the menus
    toSelectorListsMenu->clear(); // clear() should delete the actions
    ccSelectorListsMenu->clear();

    if (listsList.length() > 0) // if there are some lists, enable the menu
    {
        toSelectorListsMenu->setEnabled(true);
        ccSelectorListsMenu->setEnabled(true);
    }

    QString listName;
    QVariant listId;
    foreach (QVariant list, listsList)
    {
        listName = list.toMap().value("displayName").toString();
        listId = list.toMap().value("id");

        QAction *toListAction = new QAction(listName, this);
        toListAction->setCheckable(true);
        toListAction->setData(listId);
        toSelectorListsMenu->addAction(toListAction);

        QAction *ccListAction = new QAction(listName, this);
        ccListAction->setCheckable(true);
        ccListAction->setData(listId);
        ccSelectorListsMenu->addAction(ccListAction);
    }
}



void Publisher::updateToListsFields(QAction *listAction)
{
    QString listName = listAction->text();
    QString listId = listAction->data().toString();

    if (listAction->isChecked())
    {
        qDebug() << "To list selected" << listName << listId;

        toListsNameStringList.append(listName);
        toListsIdStringList.append(listId);
    }
    else
    {
        qDebug() << "To list unselected" << listName;

        toListsNameStringList.removeAll(listName);
        toListsIdStringList.removeAll(listId);
    }

    this->updatePublicFollowersLabels();

    qDebug() << "Current To Lists:" << toListsNameStringList << toListsIdStringList;
}



void Publisher::updateCcListsFields(QAction *listAction)
{
    QString listName = listAction->text();
    QString listId = listAction->data().toString();

    if (listAction->isChecked())
    {
        qDebug() << "CC list selected" << listName << listId;

        ccListsNameStringList.append(listName);
        ccListsIdStringList.append(listId);
    }
    else
    {
        qDebug() << "CC list unselected" << listName;

        ccListsNameStringList.removeAll(listName);
        ccListsIdStringList.removeAll(listId);
    }

    this->updatePublicFollowersLabels();

    qDebug() << "Current CC Lists:" << ccListsNameStringList << ccListsIdStringList;
}




/*
 * Send the post (note, image...) to the server
 *
 */
void Publisher::sendPost()
{
    qDebug() << "Publisher character count:" << composerBox->textCursor().document()->characterCount();


    // If there's some text in the post, or attached picture, send it
    if (composerBox->textCursor().document()->characterCount() > 1     // kinda tmp
      || !pictureFilename.isEmpty())
    {
        QString postTitle = this->titleLineEdit->text();
        postTitle.replace("\n", " "); // Post title could have newlines if copy-pasted

        QString cleanHtmlString = MiscHelpers::cleanupHtml(composerBox->toHtml());


        QMap<QString,QString> audienceMap = this->getAudienceMap();


        // Don't erase just yet!! Just disable until we get "200 OK" from the server.
        this->setDisabled(true);

        if (!editingMode)
        {
            this->statusInfoLabel->setText(tr("Posting..."));

            if (this->pictureFilename.isEmpty())
            {
                this->pController->postNote(audienceMap,
                                            cleanHtmlString,
                                            postTitle);
            }
            else
            {
                this->pController->postImage(audienceMap,
                                             cleanHtmlString,
                                             postTitle,
                                             pictureFilename,
                                             pictureContentType);
            }
        }
        else
        {
            this->statusInfoLabel->setText(tr("Updating..."));
            this->pController->updatePost(this->editingPostId,
                                          cleanHtmlString,
                                          postTitle);
        }
    }
    else
    {
        this->statusInfoLabel->setText(tr("Post is empty."));
        qDebug() << "Can't send post: text is empty, and no picture";
    }
}



/*
 * Let the user find an image file in his/her folders, for picture upload
 *
 */
void Publisher::findPictureFile()
{
    QString filename;

    filename = QFileDialog::getOpenFileName(this, tr("Select one image"),
                                            this->lastUsedDirectory,
                                            tr("Image files") + "(*.png *.jpg *.jpeg *.gif);;"
                                            + tr("All files") + " (*)");

    if (!filename.isEmpty())
    {
        qDebug() << "Selected" << filename << "for upload";
        this->pictureLabel->setToolTip(filename);

        this->pictureContentType = MiscHelpers::getImageContentType(filename);
        if (!pictureContentType.isEmpty())
        {
            this->pictureFilename = filename;

            QPixmap imagePixmap = QPixmap(filename);
            // FIXME: check imagePixmap.isNull() ?
            this->pictureLabel->setPixmap(imagePixmap.scaled(320, 180,  // 16:9
                                                             Qt::KeepAspectRatio,
                                                             Qt::SmoothTransformation));
            QFileInfo fileInfo(filename);
            this->lastUsedDirectory = fileInfo.path();
            qDebug() << "last used directory:" << lastUsedDirectory;


            pictureInfoLabel->setText(QString("<b>" +tr("Resolution") + ":</b> %1x%2"
                                      "<br>"
                                      "<b>" + tr("Size") + ":</b> %3")
                                      .arg(imagePixmap.width())
                                      .arg(imagePixmap.height())
                                      .arg(MiscHelpers::fileSizeString(filename)));
        }
        else
        {
            // In the future, avoid this by using libmagic or similar
            qDebug() << "Unknown image format; Extension is probably wrong";
            QMessageBox::warning(this, tr("Invalid image"),
                                 tr("The image format cannot be detected.\n"
                                    "The extension might be wrong, like a GIF "
                                    "image renamed to image.jpg or similar."));

            this->pictureContentType.clear();

            this->pictureFilename.clear();
        }
    }
}
