/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * hnc2unicode.h
 *
 * Copyright (C) 2013 Hodong Kim <cogniti@gmail.com>
 *
 * This library is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include "hnc2unicode.h"
#include <stdio.h>

static gchar *_hnc_to_utf8 (guint16 c);

gchar *hnchar_to_utf8 (guint16 c)
{
    GString *string = g_string_new (NULL);
     /* ASCII printable characters */
    if (c >= 0x0020 && c <= 0x007e) {
        g_string_append_unichar (string, c);
        return g_string_free (string, FALSE);
    } else if (c >= 0x007f && c <= 0x3fff) {
        g_string_free (string, TRUE);
        return _hnc_to_utf8(c);
    /* 1수준 한자 4888자 */
    } else if (c >= 0x4000 && c <= 0x5317) {
        g_string_append_unichar (string, ksc5601_2uni_page4a[c-0x4000]);
        return g_string_free (string, FALSE);
    /* 2수준 한자 */
    } else if (c >= 0x5318 && c <= 0x7fff) {
        g_string_free (string, TRUE);
        return _hnc_to_utf8(c);
    /* 한글 영역 */
    } else if (c >= 0x8000 && c <= 0xffff) {
        guint8 l = (c & 0x7c00) >> 10; /* 초성 */
        guint8 v = (c & 0x03e0) >> 5;  /* 중성 */
        guint8 t = (c & 0x001f);       /* 종성 */

        /* 조합형 현대 한글 음절(11172)을 유니코드로 변환 */
        if (L_MAP[l] != NONE && V_MAP[v] != NONE && T_MAP[t] != NONE) {
            guint16 syllable = 0xac00 + (L_MAP[l] * 21 * 28) +
                                (V_MAP[v] * 28) + T_MAP[t];
            g_string_append_unichar (string, syllable);
            return g_string_free (string, FALSE);
        /* 초성만 존재하는 경우 유니코드 한글 호환 자모로 변환 */
        } else if ((HNC_L1[v] != FILL) &&
                   (HNC_V1[v] == FILL || HNC_V1[v] == NONE) &&
                   (HNC_T1[t] == FILL)) {
            g_string_append_unichar (string, HNC_L1[l]);
            return g_string_free (string, FALSE);
        /* 중성만 존재하는 경우 유니코드 한글 호환 자모로 변환 */
        } else if ((HNC_L1[l] == FILL) &&
                   (HNC_V1[v] != FILL || HNC_V1[v] != NONE) &&
                   (HNC_T1[t] == FILL)) {
            g_string_append_unichar (string, HNC_V1[v]);
            return g_string_free (string, FALSE);
        /* 종성만 존재하는 경우 유니코드 한글 호환 자모로 변환 */
        } else if ((HNC_L1[l] == FILL) &&
                   (HNC_V1[v] == FILL || HNC_V1[v] == NONE) &&
                   (HNC_T1[t] != FILL)) {
            g_string_append_unichar (string, HNC_T1[t]);
            return g_string_free (string, FALSE);
        /* 초성과 중성만 존재하는 조합형 옛한글의 경우 */
        } else if ((HNC_L1[l] != FILL) &&
                   (HNC_V1[v] != FILL || HNC_V1[v] != NONE) &&
                   (HNC_T1[t] == FILL)) {
            g_string_append_unichar (string, HNC_L2[l]);
            g_string_append_unichar (string, HNC_V2[v]);
            return g_string_free (string, FALSE);
        /* 초성, 중성, 종성 모두 존재하는 조합형 옛한글의 경우 */
        } else if ((HNC_L1[l] != FILL) &&
                   (HNC_V1[v] != FILL || HNC_V1[v] != NONE) &&
                   (HNC_T1[t] != FILL)) {
            g_string_append_unichar (string, HNC_L2[l]);
            g_string_append_unichar (string, HNC_V2[v]);
            g_string_append_unichar (string, HNC_T2[t]);
            return g_string_free (string, FALSE);
        /* 완성형 옛한글 */
        } else if (v == 0) {
            gchar *ch;
            ch = _hnc_to_utf8(c);
            if (ch != NULL) {
                g_string_free (string, TRUE);
                return ch;
            } else {
                g_warning ("HNC code: %04x", c);
                g_string_free (string, TRUE);
                return NULL;
            }
        } else {
            g_warning ("HNC code: %04x", c);
            g_string_free (string, TRUE);
            return NULL;
        }
    } else {
        g_warning ("%04x: out of hnc code range", c);
        g_string_free (string, TRUE);
        return NULL;
    }
}

static gchar *_hnc_to_utf8 (guint16 c)
{
    gunichar wc;
    GString *string = g_string_new (NULL);
    switch (c) {
        case 0x007f: wc = 0x00a1; break;
        case 0x0080: wc = 0x20ac; break;
        case 0x0081: wc = 0x201c; break;
        case 0x0082: wc = 0x201d; break;
        case 0x0083: wc = 0x2018; break;
        case 0x0084: wc = 0x2019; break;
        case 0x0085: wc = 0x00b7; break;
        case 0x0086: wc = 0x201e; break;
        case 0x0087: wc = 0x201f; break;
        case 0x00ae: wc = 0x00ae; break;
        case 0x00af: wc = 0x00af; break;
        case 0x00b0: wc = 0x00b0; break;
        case 0x00b1: wc = 0x00b1; break;
        case 0x00b2: wc = 0x00b2; break;
        case 0x00b3: wc = 0x00b3; break;
        case 0x00b4: wc = 0x00b4; break;
        case 0x00b5: wc = 0x00b5; break;
        case 0x00b6: wc = 0x00b6; break;
        case 0x00b7: wc = 0x00b7; break;
        case 0x00b8: wc = 0x00b8; break;
        case 0x00b9: wc = 0x00b9; break;
        case 0x00ba: wc = 0x00ba; break;
        case 0x00bb: wc = 0x00bb; break;
        case 0x00bc: wc = 0x00bc; break;
        case 0x00bd: wc = 0x00bd; break;
        case 0x00be: wc = 0x00be; break;
        case 0x00bf: wc = 0x00bf; break;
        case 0x00c0: wc = 0x00c0; break;
        case 0x00c1: wc = 0x00c1; break;
        case 0x00c2: wc = 0x00c2; break;
        case 0x00c3: wc = 0x00c3; break;
        case 0x00c4: wc = 0x00c4; break;
        case 0x00c5: wc = 0x00c5; break;
        case 0x00c6: wc = 0x00c6; break;
        case 0x00c7: wc = 0x00c7; break;
        case 0x00c8: wc = 0x00c8; break;
        case 0x00c9: wc = 0x00c9; break;
        case 0x00ca: wc = 0x00ca; break;
        case 0x00cb: wc = 0x00cb; break;
        case 0x00cc: wc = 0x00cc; break;
        case 0x00cd: wc = 0x00cd; break;
        case 0x00ce: wc = 0x00ce; break;
        case 0x00cf: wc = 0x00cf; break;
        case 0x00d0: wc = 0x00d0; break;
        case 0x00d1: wc = 0x00d1; break;
        case 0x00d2: wc = 0x00d2; break;
        case 0x00d3: wc = 0x00d3; break;
        case 0x00d4: wc = 0x00d4; break;
        case 0x00d5: wc = 0x00d5; break;
        case 0x00d6: wc = 0x00d6; break;
        case 0x00d7: wc = 0x00d7; break;
        case 0x00d8: wc = 0x00d8; break;
        case 0x00d9: wc = 0x00d9; break;
        case 0x00da: wc = 0x00da; break;
        case 0x00db: wc = 0x00db; break;
        case 0x00dc: wc = 0x00dc; break;
        case 0x00dd: wc = 0x00dd; break;
        case 0x00de: wc = 0x00de; break;
        case 0x00df: wc = 0x00df; break;
        case 0x00e0: wc = 0x00e0; break;
        case 0x00e1: wc = 0x00e1; break;
        case 0x00e2: wc = 0x00e2; break;
        case 0x00e3: wc = 0x00e3; break;
        case 0x00e4: wc = 0x00e4; break;
        case 0x00e5: wc = 0x00e5; break;
        case 0x00e6: wc = 0x00e6; break;
        case 0x00e7: wc = 0x00e7; break;
        case 0x00e8: wc = 0x00e8; break;
        case 0x00e9: wc = 0x00e9; break;
        case 0x00ea: wc = 0x00ea; break;
        case 0x00eb: wc = 0x00eb; break;
        case 0x00ec: wc = 0x00ec; break;
        case 0x00ed: wc = 0x00ed; break;
        case 0x00ee: wc = 0x00ee; break;
        case 0x00ef: wc = 0x00ef; break;
        case 0x00f0: wc = 0x00f0; break;
        case 0x00f1: wc = 0x00f1; break;
        case 0x00f2: wc = 0x00f2; break;
        case 0x00f3: wc = 0x00f3; break;
        case 0x00f4: wc = 0x00f4; break;
        case 0x00f5: wc = 0x00f5; break;
        case 0x00f6: wc = 0x00f6; break;
        case 0x00f7: wc = 0x00f7; break;
        case 0x00f8: wc = 0x00f8; break;
        case 0x00f9: wc = 0x00f9; break;
        case 0x00fa: wc = 0x00fa; break;
        case 0x00fb: wc = 0x00fb; break;
        case 0x00fc: wc = 0x00fc; break;
        case 0x00fd: wc = 0x00fd; break;
        case 0x00fe: wc = 0x00fe; break;
        case 0x00ff: wc = 0x00ff; break;
        case 0x0100: wc = 0x0100; break;
        case 0x0101: wc = 0x0101; break;
        case 0x0102: wc = 0x0102; break;
        case 0x0103: wc = 0x0103; break;
        case 0x0104: wc = 0x0104; break;
        case 0x0105: wc = 0x0105; break;
        case 0x0106: wc = 0x0106; break;
        case 0x0107: wc = 0x0107; break;
        case 0x0108: wc = 0x0108; break;
        case 0x0109: wc = 0x0109; break;
        case 0x010a: wc = 0x010a; break;
        case 0x010b: wc = 0x010b; break;
        case 0x010c: wc = 0x010c; break;
        case 0x010d: wc = 0x010d; break;
        case 0x010e: wc = 0x010e; break;
        case 0x010f: wc = 0x010f; break;
        case 0x0110: wc = 0x0110; break;
        case 0x0111: wc = 0x0111; break;
        case 0x0112: wc = 0x0112; break;
        case 0x0113: wc = 0x0113; break;
        case 0x0114: wc = 0x0114; break;
        case 0x0115: wc = 0x0115; break;
        case 0x0116: wc = 0x0116; break;
        case 0x0117: wc = 0x0117; break;
        case 0x0118: wc = 0x0118; break;
        case 0x0119: wc = 0x0119; break;
        case 0x011a: wc = 0x011a; break;
        case 0x011b: wc = 0x011b; break;
        case 0x011c: wc = 0x011c; break;
        case 0x011d: wc = 0x011d; break;
        case 0x011e: wc = 0x011e; break;
        case 0x011f: wc = 0x011f; break;
        case 0x0120: wc = 0x0120; break;
        case 0x0121: wc = 0x0121; break;
        case 0x0122: wc = 0x0122; break;
        case 0x0123: wc = 0x0123; break;
        case 0x0124: wc = 0x0124; break;
        case 0x0125: wc = 0x0125; break;
        case 0x0126: wc = 0x0126; break;
        case 0x0127: wc = 0x0127; break;
        case 0x0128: wc = 0x0128; break;
        case 0x0129: wc = 0x0129; break;
        case 0x012a: wc = 0x012a; break;
        case 0x012b: wc = 0x012b; break;
        case 0x012c: wc = 0x012c; break;
        case 0x012d: wc = 0x012d; break;
        case 0x012e: wc = 0x012e; break;
        case 0x012f: wc = 0x012f; break;
        case 0x0130: wc = 0x0130; break;
        case 0x0131: wc = 0x0131; break;
        case 0x0132: wc = 0x0132; break;
        case 0x0133: wc = 0x0133; break;
        case 0x0134: wc = 0x0134; break;
        case 0x0135: wc = 0x0135; break;
        case 0x0136: wc = 0x0136; break;
        case 0x0137: wc = 0x0137; break;
        case 0x0138: wc = 0x0138; break;
        case 0x0139: wc = 0x0139; break;
        case 0x013a: wc = 0x013a; break;
        case 0x013b: wc = 0x013b; break;
        case 0x013c: wc = 0x013c; break;
        case 0x013d: wc = 0x013d; break;
        case 0x013e: wc = 0x013e; break;
        case 0x013f: wc = 0x013f; break;
        case 0x0140: wc = 0x0140; break;
        case 0x0141: wc = 0x0141; break;
        case 0x0142: wc = 0x0142; break;
        case 0x0143: wc = 0x0143; break;
        case 0x0144: wc = 0x0144; break;
        case 0x0145: wc = 0x0145; break;
        case 0x0146: wc = 0x0146; break;
        case 0x0147: wc = 0x0147; break;
        case 0x0148: wc = 0x0148; break;
        case 0x0149: wc = 0x0149; break;
        case 0x014a: wc = 0x014a; break;
        case 0x014b: wc = 0x014b; break;
        case 0x014c: wc = 0x014c; break;
        case 0x014d: wc = 0x014d; break;
        case 0x014e: wc = 0x014e; break;
        case 0x014f: wc = 0x014f; break;
        case 0x0150: wc = 0x0150; break;
        case 0x0151: wc = 0x0151; break;
        case 0x0152: wc = 0x0152; break;
        case 0x0153: wc = 0x0153; break;
        case 0x0154: wc = 0x0154; break;
        case 0x0155: wc = 0x0155; break;
        case 0x0156: wc = 0x0156; break;
        case 0x0157: wc = 0x0157; break;
        case 0x0158: wc = 0x0158; break;
        case 0x0159: wc = 0x0159; break;
        case 0x015a: wc = 0x015a; break;
        case 0x015b: wc = 0x015b; break;
        case 0x015c: wc = 0x015c; break;
        case 0x015d: wc = 0x015d; break;
        case 0x015e: wc = 0x015e; break;
        case 0x015f: wc = 0x015f; break;
        case 0x0160: wc = 0x0160; break;
        case 0x0161: wc = 0x0161; break;
        case 0x0162: wc = 0x0162; break;
        case 0x0163: wc = 0x0163; break;
        case 0x0164: wc = 0x0164; break;
        case 0x0165: wc = 0x0165; break;
        case 0x0166: wc = 0x0166; break;
        case 0x0167: wc = 0x0167; break;
        case 0x0168: wc = 0x0168; break;
        case 0x0169: wc = 0x0169; break;
        case 0x016a: wc = 0x016a; break;
        case 0x016b: wc = 0x016b; break;
        case 0x016c: wc = 0x016c; break;
        case 0x016d: wc = 0x016d; break;
        case 0x016e: wc = 0x016e; break;
        case 0x016f: wc = 0x016f; break;
        case 0x0170: wc = 0x0170; break;
        case 0x0171: wc = 0x0171; break;
        case 0x0172: wc = 0x0172; break;
        case 0x0173: wc = 0x0173; break;
        case 0x0174: wc = 0x0174; break;
        case 0x0175: wc = 0x0175; break;
        case 0x0176: wc = 0x0176; break;
        case 0x0177: wc = 0x0177; break;
        case 0x0178: wc = 0x0178; break;
        case 0x0179: wc = 0x0179; break;
        case 0x017a: wc = 0x017a; break;
        case 0x017b: wc = 0x017b; break;
        case 0x017c: wc = 0x017c; break;
        case 0x017d: wc = 0x017d; break;
        case 0x017e: wc = 0x017e; break;
        case 0x017f: wc = 0x017f; break;
        case 0x0180: wc = 0x0180; break;
        case 0x0181: wc = 0x0181; break;
        case 0x0182: wc = 0x0182; break;
        case 0x0183: wc = 0x0183; break;
        case 0x0184: wc = 0x0184; break;
        case 0x0185: wc = 0x0185; break;
        case 0x0186: wc = 0x0186; break;
        case 0x0187: wc = 0x0187; break;
        case 0x0188: wc = 0x0188; break;
        case 0x0189: wc = 0x0189; break;
        case 0x018a: wc = 0x018a; break;
        case 0x018b: wc = 0x018b; break;
        case 0x018c: wc = 0x018c; break;
        case 0x018d: wc = 0x018d; break;
        case 0x018e: wc = 0x018e; break;
        case 0x018f: wc = 0x018f; break;
        case 0x0190: wc = 0x0190; break;
        case 0x0191: wc = 0x0191; break;
        case 0x0192: wc = 0x0192; break;
        case 0x0193: wc = 0x0193; break;
        case 0x0194: wc = 0x0194; break;
        case 0x0195: wc = 0x0195; break;
        case 0x0196: wc = 0x0196; break;
        case 0x0197: wc = 0x0197; break;
        case 0x0198: wc = 0x0198; break;
        case 0x0199: wc = 0x0199; break;
        case 0x019a: wc = 0x019a; break;
        case 0x019b: wc = 0x019b; break;
        case 0x019c: wc = 0x019c; break;
        case 0x019d: wc = 0x019d; break;
        case 0x019e: wc = 0x019e; break;
        case 0x019f: wc = 0x019f; break;
        case 0x01a0: wc = 0x01a0; break;
        case 0x01a1: wc = 0x01a1; break;
        case 0x01a2: wc = 0x01a2; break;
        case 0x01a3: wc = 0x01a3; break;
        case 0x01a4: wc = 0x01a4; break;
        case 0x01a5: wc = 0x01a5; break;
        case 0x01a6: wc = 0x01a6; break;
        case 0x01a7: wc = 0x01a7; break;
        case 0x01a8: wc = 0x01a8; break;
        case 0x01a9: wc = 0x01a9; break;
        case 0x01aa: wc = 0x01aa; break;
        case 0x01ab: wc = 0x01ab; break;
        case 0x01ac: wc = 0x01ac; break;
        case 0x01ad: wc = 0x01ad; break;
        case 0x01ae: wc = 0x01ae; break;
        case 0x01af: wc = 0x01af; break;
        case 0x01b0: wc = 0x01b0; break;
        case 0x01b1: wc = 0x01b1; break;
        case 0x01b2: wc = 0x01b2; break;
        case 0x01b3: wc = 0x01b3; break;
        case 0x01b4: wc = 0x01b4; break;
        case 0x01b5: wc = 0x01b5; break;
        case 0x01b6: wc = 0x01b6; break;
        case 0x01b7: wc = 0x01b7; break;
        case 0x01b8: wc = 0x01b8; break;
        case 0x01b9: wc = 0x01b9; break;
        case 0x01ba: wc = 0x01ba; break;
        case 0x01bb: wc = 0x01bb; break;
        case 0x01bc: wc = 0x01bc; break;
        case 0x01bd: wc = 0x01bd; break;
        case 0x01be: wc = 0x01be; break;
        case 0x01bf: wc = 0x01bf; break;
        case 0x01c0: wc = 0x01c0; break;
        case 0x01c1: wc = 0x01c1; break;
        case 0x01c2: wc = 0x01c2; break;
        case 0x01c3: wc = 0x01c3; break;
        case 0x01c4: wc = 0x01c4; break;
        case 0x01c5: wc = 0x01c5; break;
        case 0x01c6: wc = 0x01c6; break;
        case 0x01c7: wc = 0x01c7; break;
        case 0x01c8: wc = 0x01c8; break;
        case 0x01c9: wc = 0x01c9; break;
        case 0x01ca: wc = 0x01ca; break;
        case 0x01cb: wc = 0x01cb; break;
        case 0x01cc: wc = 0x01cc; break;
        case 0x01cd: wc = 0x01cd; break;
        case 0x01ce: wc = 0x01ce; break;
        case 0x01cf: wc = 0x01cf; break;
        case 0x01d0: wc = 0x01d0; break;
        case 0x01d1: wc = 0x01d1; break;
        case 0x01d2: wc = 0x01d2; break;
        case 0x01d3: wc = 0x01d3; break;
        case 0x01d4: wc = 0x01d4; break;
        case 0x01d5: wc = 0x01d5; break;
        case 0x01d6: wc = 0x01d6; break;
        case 0x01d7: wc = 0x01d7; break;
        case 0x01d8: wc = 0x01d8; break;
        case 0x01d9: wc = 0x01d9; break;
        case 0x01da: wc = 0x01da; break;
        case 0x01db: wc = 0x01db; break;
        case 0x01dc: wc = 0x01dc; break;
        case 0x01dd: wc = 0x01dd; break;
        case 0x01de: wc = 0x01de; break;
        case 0x01df: wc = 0x01df; break;
        case 0x01e0: wc = 0x01e0; break;
        case 0x01e1: wc = 0x01e1; break;
        case 0x01e2: wc = 0x01e2; break;
        case 0x01e3: wc = 0x01e3; break;
        case 0x01e4: wc = 0x01e4; break;
        case 0x01e5: wc = 0x01e5; break;
        case 0x01e6: wc = 0x01e6; break;
        case 0x01e7: wc = 0x01e7; break;
        case 0x01e8: wc = 0x01e8; break;
        case 0x01e9: wc = 0x01e9; break;
        case 0x01ea: wc = 0x01ea; break;
        case 0x01eb: wc = 0x01eb; break;
        case 0x01ec: wc = 0x01ec; break;
        case 0x01ed: wc = 0x01ed; break;
        case 0x01ee: wc = 0x01ee; break;
        case 0x01ef: wc = 0x01ef; break;
        case 0x01f0: wc = 0x01f0; break;
        case 0x01f1: wc = 0x01f1; break;
        case 0x01f2: wc = 0x01f2; break;
        case 0x01f3: wc = 0x01f3; break;
        case 0x01f4: wc = 0x01f4; break;
        case 0x01f5: wc = 0x01f5; break;
        case 0x01f6: wc = 0x01f6; break;
        case 0x01f7: wc = 0x01f7; break;
        case 0x01f8: wc = 0x01f8; break;
        case 0x01f9: wc = 0x01f9; break;
        case 0x01fa: wc = 0x01fa; break;
        case 0x01fb: wc = 0x01fb; break;
        case 0x01fc: wc = 0x01fc; break;
        case 0x01fd: wc = 0x01fd; break;
        case 0x01fe: wc = 0x01fe; break;
        case 0x01ff: wc = 0x01ff; break;
        case 0x0200: wc = 0x0200; break;
        case 0x0201: wc = 0x0201; break;
        case 0x0202: wc = 0x0202; break;
        case 0x0203: wc = 0x0203; break;
        case 0x0204: wc = 0x0204; break;
        case 0x0205: wc = 0x0205; break;
        case 0x0206: wc = 0x0206; break;
        case 0x0207: wc = 0x0207; break;
        case 0x0208: wc = 0x0208; break;
        case 0x0209: wc = 0x0209; break;
        case 0x020a: wc = 0x020a; break;
        case 0x020b: wc = 0x020b; break;
        case 0x020c: wc = 0x020c; break;
        case 0x020d: wc = 0x020d; break;
        case 0x020e: wc = 0x020e; break;
        case 0x020f: wc = 0x020f; break;
        case 0x0210: wc = 0x0210; break;
        case 0x0211: wc = 0x0211; break;
        case 0x0212: wc = 0x0212; break;
        case 0x0213: wc = 0x0213; break;
        case 0x0214: wc = 0x0214; break;
        case 0x0215: wc = 0x0215; break;
        case 0x0216: wc = 0x0216; break;
        case 0x0217: wc = 0x0217; break;
        case 0x2030: wc = 0x2030; break;
        case 0x2031: wc = 0x2031; break;
        case 0x2032: wc = 0x2032; break;
        case 0x2033: wc = 0x2033; break;
        case 0x2034: wc = 0x2034; break;
        case 0x2035: wc = 0x2035; break;
        case 0x2036: wc = 0x2036; break;
        case 0x2037: wc = 0x2037; break;
        case 0x2038: wc = 0x2038; break;
        case 0x2039: wc = 0x2039; break;
        case 0x203a: wc = 0x203a; break;
        case 0x203b: wc = 0x203b; break;
        case 0x203c: wc = 0x203c; break;
        case 0x203d: wc = 0x203d; break;
        case 0x203e: wc = 0x203e; break;
        case 0x203f: wc = 0x203f; break;
        case 0x2040: wc = 0x2040; break;
        case 0x2041: wc = 0x2041; break;
        case 0x2042: wc = 0x2042; break;
        case 0x2043: wc = 0x2043; break;
        case 0x2044: wc = 0x2044; break;
        case 0x2045: wc = 0x2045; break;
        case 0x2046: wc = 0x2046; break;
        case 0x205a: wc = 0x25cb; break;
        case 0x2126: wc = 0x2126; break;
        case 0x2190: wc = 0x2190; break;
        case 0x2191: wc = 0x2191; break;
        case 0x2192: wc = 0x2192; break;
        case 0x2193: wc = 0x2193; break;
        case 0x2194: wc = 0x2194; break;
        case 0x2195: wc = 0x2195; break;
        case 0x2196: wc = 0x2196; break;
        case 0x2197: wc = 0x2197; break;
        case 0x2198: wc = 0x2198; break;
        case 0x2199: wc = 0x2199; break;
        case 0x219a: wc = 0x219a; break;
        case 0x219b: wc = 0x219b; break;
        case 0x21cc: wc = 0x21cc; break;
        case 0x21e8: wc = 0x21e8; break;
        case 0x2218: wc = 0x2218; break;
        case 0x2219: wc = 0x2219; break;
        case 0x2592: wc = 0xf007e; break;
        case 0x25c7: wc = 0x2709; break;
        case 0x2b4e: wc = 0xf00e1; break;
        case 0x2bce: wc = 0xf012b; break;
        case 0x2e40: wc = 0xf02b0; break;
        case 0x2e41: wc = 0xf02b1; break;
        case 0x2e42: wc = 0xf02b2; break;
        case 0x2e43: wc = 0xf02b3; break;
        case 0x2e44: wc = 0xf02b4; break;
        case 0x2e45: wc = 0xf02b5; break;
        case 0x2e46: wc = 0xf02b6; break;
        case 0x2e47: wc = 0xf02b7; break;
        case 0x2e48: wc = 0xf02b8; break;
        case 0x2e49: wc = 0xf02b9; break;
        case 0x2e4a: wc = 0xf02ba; break;
        case 0x2e4b: wc = 0xf02bb; break;
        case 0x2e4c: wc = 0xf02bc; break;
        case 0x2e4d: wc = 0xf02bd; break;
        case 0x2e4e: wc = 0xf02be; break;
        case 0x2e4f: wc = 0xf02bf; break;
        case 0x2e50: wc = 0xf02c0; break;
        case 0x2e51: wc = 0xf02c1; break;
        case 0x2e52: wc = 0xf02c2; break;
        case 0x2f00: wc = 0x25a1; break;
        case 0x2f11: wc = 0x25e6; break;
        case 0x2f18: wc = 0x2219; break;
        case 0x2f66: wc = 0x25b6; break;
        case 0x2f67: wc = 0x25b8; break;
        case 0x301c: wc = 0x2500; break;
        case 0x312a: wc = 0x00d7; break;
        case 0x3401: wc = 0x3000; break;
        case 0x3402: wc = 0x3001; break;
        case 0x3403: wc = 0x3002; break;
        case 0x3404: wc = 0x2024; break;
        case 0x3405: wc = 0x2025; break;
        case 0x3406: wc = 0x2026; break;
        case 0x3407: wc = 0x00a8; break;
        case 0x3408: wc = 0x3003; break;
        case 0x3409: wc = 0xff0d; break;
        case 0x340a: wc = 0x2015; break;
        case 0x340b: wc = 0x2225; break;
        case 0x340c: wc = 0xff3c; break;
        case 0x340d: wc = 0xff5e; break;
        case 0x340e: wc = 0xf0850; break;
        case 0x340f: wc = 0xf0851; break;
        case 0x3410: wc = 0xf0852; break;
        case 0x3411: wc = 0xf0853; break;
        case 0x3412: wc = 0x3014; break;
        case 0x3413: wc = 0x3015; break;
        case 0x3414: wc = 0x3008; break;
        case 0x3415: wc = 0x3009; break;
        case 0x3416: wc = 0x300a; break;
        case 0x3417: wc = 0x300b; break;
        case 0x3418: wc = 0x300c; break;
        case 0x3419: wc = 0x300d; break;
        case 0x341a: wc = 0x300e; break;
        case 0x341b: wc = 0x300f; break;
        case 0x341c: wc = 0x3010; break;
        case 0x341d: wc = 0x3011; break;
        case 0x341e: wc = 0x00b1; break;
        case 0x341f: wc = 0x00d7; break;
        case 0x3420: wc = 0x00f7; break;
        case 0x3424: wc = 0x221e; break;
        case 0x3429: wc = 0x2103; break;
        case 0x3438: wc = 0x203b; break;
        case 0x3439: wc = 0x2606; break;
        case 0x343a: wc = 0x2605; break;
        case 0x343b: wc = 0x25cb; break;
        case 0x343c: wc = 0x25cf; break;
        case 0x3440: wc = 0x25a1; break;
        case 0x3441: wc = 0x25a0; break;
        case 0x3442: wc = 0x25b3; break;
        case 0x3446: wc = 0x2192; break;
        case 0x3447: wc = 0x2190; break;
        case 0x3450: wc = 0x221d; break;
        case 0x347f: wc = 0x2667; break;
        case 0x3491: wc = 0x261e; break;
        case 0x34c1: wc = 0xff01; break;
        case 0x34c2: wc = 0xff02; break;
        case 0x34c3: wc = 0xff03; break;
        case 0x34c4: wc = 0xff04; break;
        case 0x34c5: wc = 0xff05; break;
        case 0x34c6: wc = 0xff06; break;
        case 0x34c7: wc = 0xff07; break;
        case 0x34c8: wc = 0xff08; break;
        case 0x34c9: wc = 0xff09; break;
        case 0x34ca: wc = 0xff0a; break;
        case 0x34cb: wc = 0xff0b; break;
        case 0x34cc: wc = 0xff0c; break;
        case 0x34cd: wc = 0xff0d; break;
        case 0x34ce: wc = 0xff0e; break;
        case 0x34cf: wc = 0xff0f; break;
        case 0x34d0: wc = 0xff10; break;
        case 0x34d1: wc = 0xff11; break;
        case 0x34d2: wc = 0xff12; break;
        case 0x34d3: wc = 0xff13; break;
        case 0x34d4: wc = 0xff14; break;
        case 0x34d5: wc = 0xff15; break;
        case 0x34d6: wc = 0xff16; break;
        case 0x34d7: wc = 0xff17; break;
        case 0x34d8: wc = 0xff18; break;
        case 0x34d9: wc = 0xff19; break;
        case 0x34da: wc = 0xff1a; break;
        case 0x34db: wc = 0xff1b; break;
        case 0x34dc: wc = 0xff1c; break;
        case 0x34dd: wc = 0xff1d; break;
        case 0x34de: wc = 0xff1e; break;
        case 0x34df: wc = 0xff1f; break;
        case 0x34e0: wc = 0xff20; break;
        case 0x34e1: wc = 0xff21; break;
        case 0x34e2: wc = 0xff22; break;
        case 0x34e3: wc = 0xff23; break;
        case 0x34e4: wc = 0xff24; break;
        case 0x34e5: wc = 0xff25; break;
        case 0x34e6: wc = 0xff26; break;
        case 0x34e7: wc = 0xff27; break;
        case 0x34e8: wc = 0xff28; break;
        case 0x34e9: wc = 0xff29; break;
        case 0x34ea: wc = 0xff2a; break;
        case 0x34eb: wc = 0xff2b; break;
        case 0x34ec: wc = 0xff2c; break;
        case 0x34ed: wc = 0xff2d; break;
        case 0x34ee: wc = 0xff2e; break;
        case 0x34ef: wc = 0xff2f; break;
        case 0x34f0: wc = 0xff30; break;
        case 0x34f1: wc = 0xff31; break;
        case 0x34f2: wc = 0xff32; break;
        case 0x34f3: wc = 0xff33; break;
        case 0x34f4: wc = 0xff34; break;
        case 0x34f5: wc = 0xff35; break;
        case 0x34f6: wc = 0xff36; break;
        case 0x34f7: wc = 0xff37; break;
        case 0x34f8: wc = 0xff38; break;
        case 0x34f9: wc = 0xff39; break;
        case 0x34fa: wc = 0xff3a; break;
        case 0x34fb: wc = 0xff3b; break;
        case 0x34fc: wc = 0xff3c; break;
        case 0x34fd: wc = 0xff3d; break;
        case 0x34fe: wc = 0xff3e; break;
        case 0x34ff: wc = 0xff3f; break;
        case 0x3500: wc = 0xff40; break;
        case 0x3501: wc = 0xff41; break;
        case 0x3502: wc = 0xff42; break;
        case 0x3503: wc = 0xff43; break;
        case 0x3504: wc = 0xff44; break;
        case 0x3505: wc = 0xff45; break;
        case 0x3506: wc = 0xff46; break;
        case 0x3507: wc = 0xff47; break;
        case 0x3508: wc = 0xff48; break;
        case 0x3509: wc = 0xff49; break;
        case 0x350a: wc = 0xff4a; break;
        case 0x350b: wc = 0xff4b; break;
        case 0x350c: wc = 0xff4c; break;
        case 0x350d: wc = 0xff4d; break;
        case 0x350e: wc = 0xff4e; break;
        case 0x350f: wc = 0xff4f; break;
        case 0x3510: wc = 0xff50; break;
        case 0x3511: wc = 0xff51; break;
        case 0x3512: wc = 0xff52; break;
        case 0x3513: wc = 0xff53; break;
        case 0x3514: wc = 0xff54; break;
        case 0x3515: wc = 0xff55; break;
        case 0x3516: wc = 0xff56; break;
        case 0x3517: wc = 0xff57; break;
        case 0x3518: wc = 0xff58; break;
        case 0x3519: wc = 0xff59; break;
        case 0x351a: wc = 0xff5a; break;
        case 0x351b: wc = 0xff5b; break;
        case 0x351c: wc = 0xff5c; break;
        case 0x351d: wc = 0xff5d; break;
        case 0x3521: wc = 0x3131; break;
        case 0x3522: wc = 0x3132; break;
        case 0x3523: wc = 0x3133; break;
        case 0x3524: wc = 0x3134; break;
        case 0x3525: wc = 0x3135; break;
        case 0x3526: wc = 0x3136; break;
        case 0x3527: wc = 0x3137; break;
        case 0x3528: wc = 0x3138; break;
        case 0x3529: wc = 0x3139; break;
        case 0x352a: wc = 0x313a; break;
        case 0x352b: wc = 0x313b; break;
        case 0x352c: wc = 0x313c; break;
        case 0x352d: wc = 0x313d; break;
        case 0x352e: wc = 0x313e; break;
        case 0x352f: wc = 0x313f; break;
        case 0x3530: wc = 0x3140; break;
        case 0x3531: wc = 0x3141; break;
        case 0x3532: wc = 0x3142; break;
        case 0x3533: wc = 0x3143; break;
        case 0x3534: wc = 0x3144; break;
        case 0x3535: wc = 0x3145; break;
        case 0x3536: wc = 0x3146; break;
        case 0x3537: wc = 0x3147; break;
        case 0x3538: wc = 0x3148; break;
        case 0x3539: wc = 0x3149; break;
        case 0x353a: wc = 0x314a; break;
        case 0x353b: wc = 0x314b; break;
        case 0x353c: wc = 0x314c; break;
        case 0x353d: wc = 0x314d; break;
        case 0x353e: wc = 0x314e; break;
        case 0x353f: wc = 0x314f; break;
        case 0x3540: wc = 0x3150; break;
        case 0x3541: wc = 0x3151; break;
        case 0x3542: wc = 0x3152; break;
        case 0x3543: wc = 0x3153; break;
        case 0x3544: wc = 0x3154; break;
        case 0x3545: wc = 0x3155; break;
        case 0x3546: wc = 0x3156; break;
        case 0x3547: wc = 0x3157; break;
        case 0x3548: wc = 0x3158; break;
        case 0x3549: wc = 0x3159; break;
        case 0x354a: wc = 0x315a; break;
        case 0x354b: wc = 0x315b; break;
        case 0x354c: wc = 0x315c; break;
        case 0x354d: wc = 0x315d; break;
        case 0x354e: wc = 0x315e; break;
        case 0x354f: wc = 0x315f; break;
        case 0x3550: wc = 0x3160; break;
        case 0x3551: wc = 0x3161; break;
        case 0x3552: wc = 0x3162; break;
        case 0x3553: wc = 0x3163; break;
        case 0x3554: wc = 0x3164; break;
        case 0x3555: wc = 0x3165; break;
        case 0x3556: wc = 0x3166; break;
        case 0x3557: wc = 0x3167; break;
        case 0x3558: wc = 0x3168; break;
        case 0x3559: wc = 0x3169; break;
        case 0x355a: wc = 0x316a; break;
        case 0x355b: wc = 0x316b; break;
        case 0x355c: wc = 0x316c; break;
        case 0x355d: wc = 0x316d; break;
        case 0x355e: wc = 0x316e; break;
        case 0x355f: wc = 0x316f; break;
        case 0x3560: wc = 0x3170; break;
        case 0x3561: wc = 0x3171; break;
        case 0x3562: wc = 0x3172; break;
        case 0x3563: wc = 0x3173; break;
        case 0x3564: wc = 0x3174; break;
        case 0x3565: wc = 0x3175; break;
        case 0x3566: wc = 0x3176; break;
        case 0x3567: wc = 0x3177; break;
        case 0x3568: wc = 0x3178; break;
        case 0x3569: wc = 0x3179; break;
        case 0x356a: wc = 0x317a; break;
        case 0x356b: wc = 0x317b; break;
        case 0x356c: wc = 0x317c; break;
        case 0x356d: wc = 0x317d; break;
        case 0x356e: wc = 0x317e; break;
        case 0x356f: wc = 0x317f; break;
        case 0x3570: wc = 0x3180; break;
        case 0x3571: wc = 0x3181; break;
        case 0x3572: wc = 0x3182; break;
        case 0x3573: wc = 0x3183; break;
        case 0x3574: wc = 0x3184; break;
        case 0x3575: wc = 0x3185; break;
        case 0x3576: wc = 0x3186; break;
        case 0x3577: wc = 0x3187; break;
        case 0x3578: wc = 0x3188; break;
        case 0x3579: wc = 0x3189; break;
        case 0x357a: wc = 0x318a; break;
        case 0x357b: wc = 0x318b; break;
        case 0x357c: wc = 0x318c; break;
        case 0x357d: wc = 0x318d; break;
        case 0x357e: wc = 0x318e; break;
        case 0x3581: wc = 0x2170; break;
        case 0x3582: wc = 0x2171; break;
        case 0x3583: wc = 0x2172; break;
        case 0x3584: wc = 0x2173; break;
        case 0x3585: wc = 0x2174; break;
        case 0x3586: wc = 0x2175; break;
        case 0x3587: wc = 0x2176; break;
        case 0x3588: wc = 0x2177; break;
        case 0x3589: wc = 0x2178; break;
        case 0x358a: wc = 0x2179; break;
        case 0x358b: wc = 0x217a; break;
        case 0x358c: wc = 0x217b; break;
        case 0x3590: wc = 0x2160; break;
        case 0x3591: wc = 0x2161; break;
        case 0x3592: wc = 0x2162; break;
        case 0x3593: wc = 0x2163; break;
        case 0x3594: wc = 0x2164; break;
        case 0x3595: wc = 0x2165; break;
        case 0x3596: wc = 0x2166; break;
        case 0x3597: wc = 0x2167; break;
        case 0x3598: wc = 0x2168; break;
        case 0x3599: wc = 0x2169; break;
        case 0x359a: wc = 0x216a; break;
        case 0x359b: wc = 0x216b; break;
        case 0x35a1: wc = 0x0391; break;
        case 0x35a2: wc = 0x0392; break;
        case 0x35a3: wc = 0x0393; break;
        case 0x35a4: wc = 0x0394; break;
        case 0x35a5: wc = 0x0395; break;
        case 0x35a6: wc = 0x0396; break;
        case 0x35a7: wc = 0x0397; break;
        case 0x35a8: wc = 0x0398; break;
        case 0x35a9: wc = 0x0399; break;
        case 0x35aa: wc = 0x039a; break;
        case 0x35ab: wc = 0x039b; break;
        case 0x35ac: wc = 0x039c; break;
        case 0x35ad: wc = 0x039d; break;
        case 0x35ae: wc = 0x039e; break;
        case 0x35af: wc = 0x039f; break;
        case 0x35b0: wc = 0x03a0; break;
        case 0x35b1: wc = 0x03a1; break;
        case 0x35b2: wc = 0x03a3; break;
        case 0x35b3: wc = 0x03a4; break;
        case 0x35b4: wc = 0x03a5; break;
        case 0x35b5: wc = 0x03a6; break;
        case 0x35b6: wc = 0x03a7; break;
        case 0x35b7: wc = 0x03a8; break;
        case 0x35b8: wc = 0x03a9; break;
        case 0x35c1: wc = 0x03b1; break;
        case 0x35c2: wc = 0x03b2; break;
        case 0x35c3: wc = 0x03b3; break;
        case 0x35c4: wc = 0x03b4; break;
        case 0x35c5: wc = 0x03b5; break;
        case 0x35c6: wc = 0x03b6; break;
        case 0x35c7: wc = 0x03b7; break;
        case 0x35c8: wc = 0x03b8; break;
        case 0x35c9: wc = 0x03b9; break;
        case 0x35ca: wc = 0x03ba; break;
        case 0x35cb: wc = 0x03bb; break;
        case 0x35cc: wc = 0x03bc; break;
        case 0x35cd: wc = 0x03bd; break;
        case 0x35ce: wc = 0x03be; break;
        case 0x35cf: wc = 0x03bf; break;
        case 0x35d0: wc = 0x03c0; break;
        case 0x35d1: wc = 0x03c1; break;
        case 0x35d2: wc = 0x03c3; break;
        case 0x35d3: wc = 0x03c4; break;
        case 0x35d4: wc = 0x03c5; break;
        case 0x35d5: wc = 0x03c6; break;
        case 0x35d6: wc = 0x03c7; break;
        case 0x35d7: wc = 0x03c8; break;
        case 0x35d8: wc = 0x03c9; break;
        case 0x35e1: wc = 0x2500; break;
        case 0x35e2: wc = 0x2502; break;
        case 0x3647: wc = 0x33a3; break;
        case 0x3648: wc = 0x33a4; break;
        case 0x3649: wc = 0x33a5; break;
        case 0x364a: wc = 0x33a6; break;
        case 0x364b: wc = 0x3399; break;
        case 0x364c: wc = 0x339a; break;
        case 0x364d: wc = 0x339b; break;
        case 0x364e: wc = 0x339c; break;
        case 0x364f: wc = 0x339d; break;
        case 0x3650: wc = 0x339e; break;
        case 0x3651: wc = 0x339f; break;
        case 0x3652: wc = 0x33a0; break;
        case 0x3653: wc = 0x33a1; break;
        case 0x3654: wc = 0x33a2; break;
        case 0x3675: wc = 0x3391; break;
        case 0x36b1: wc = 0x3260; break;
        case 0x36b2: wc = 0x3261; break;
        case 0x36b3: wc = 0x3262; break;
        case 0x36b4: wc = 0x3263; break;
        case 0x36b5: wc = 0x3264; break;
        case 0x36b6: wc = 0x3265; break;
        case 0x36b7: wc = 0x3266; break;
        case 0x36b8: wc = 0x3267; break;
        case 0x36b9: wc = 0x3268; break;
        case 0x36ba: wc = 0x3269; break;
        case 0x36bb: wc = 0x326a; break;
        case 0x36bc: wc = 0x326b; break;
        case 0x36bd: wc = 0x326c; break;
        case 0x36be: wc = 0x326d; break;
        case 0x36bf: wc = 0x326e; break;
        case 0x36c0: wc = 0x326f; break;
        case 0x36c1: wc = 0x3270; break;
        case 0x36c2: wc = 0x3271; break;
        case 0x36c3: wc = 0x3272; break;
        case 0x36c4: wc = 0x3273; break;
        case 0x36c5: wc = 0x3274; break;
        case 0x36c6: wc = 0x3275; break;
        case 0x36c7: wc = 0x3276; break;
        case 0x36c8: wc = 0x3277; break;
        case 0x36c9: wc = 0x3278; break;
        case 0x36ca: wc = 0x3279; break;
        case 0x36cb: wc = 0x327a; break;
        case 0x36cc: wc = 0x327b; break;
        case 0x36cd: wc = 0x24d0; break;
        case 0x36ce: wc = 0x24d1; break;
        case 0x36cf: wc = 0x24d2; break;
        case 0x36d0: wc = 0x24d3; break;
        case 0x36d1: wc = 0x24d4; break;
        case 0x36d2: wc = 0x24d5; break;
        case 0x36d3: wc = 0x24d6; break;
        case 0x36d4: wc = 0x24d7; break;
        case 0x36d5: wc = 0x24d8; break;
        case 0x36d6: wc = 0x24d9; break;
        case 0x36d7: wc = 0x24da; break;
        case 0x36d8: wc = 0x24db; break;
        case 0x36d9: wc = 0x24dc; break;
        case 0x36da: wc = 0x24dd; break;
        case 0x36db: wc = 0x24de; break;
        case 0x36dc: wc = 0x24df; break;
        case 0x36dd: wc = 0x24e0; break;
        case 0x36de: wc = 0x24e1; break;
        case 0x36df: wc = 0x24e2; break;
        case 0x36e0: wc = 0x24e3; break;
        case 0x36e1: wc = 0x24e4; break;
        case 0x36e2: wc = 0x24e5; break;
        case 0x36e3: wc = 0x24e6; break;
        case 0x36e4: wc = 0x24e7; break;
        case 0x36e5: wc = 0x24e8; break;
        case 0x36e6: wc = 0x24e9; break;
        case 0x36e7: wc = 0x2460; break;
        case 0x36e8: wc = 0x2461; break;
        case 0x36e9: wc = 0x2462; break;
        case 0x36ea: wc = 0x2463; break;
        case 0x36eb: wc = 0x2464; break;
        case 0x36ec: wc = 0x2465; break;
        case 0x36ed: wc = 0x2466; break;
        case 0x36ee: wc = 0x2467; break;
        case 0x36ef: wc = 0x2468; break;
        case 0x36f0: wc = 0x2469; break;
        case 0x36f1: wc = 0x246a; break;
        case 0x36f2: wc = 0x246b; break;
        case 0x36f3: wc = 0x246c; break;
        case 0x36f4: wc = 0x246d; break;
        case 0x36f5: wc = 0x246e; break;
        case 0x3757: wc = 0x00b2; break;
        case 0x3758: wc = 0x00b3; break;
        case 0x375b: wc = 0x2081; break;
        case 0x375c: wc = 0x2082; break;
        case 0x375d: wc = 0x2083; break;
        case 0x375e: wc = 0x2084; break;

        case 0x5318: wc = 0x4eee; break;
        case 0x5319: wc = 0x4fa1; break;
        case 0x531a: wc = 0x50a2; break;
        case 0x531b: wc = 0x53da; break;
        case 0x531c: wc = 0x20850; break;
        case 0x531d: wc = 0x5496; break;
        case 0x531e: wc = 0x54ff; break;
        case 0x531f: wc = 0x5777; break;
        case 0x5320: wc = 0x5a7d; break;
        case 0x5321: wc = 0x5ca2; break;
        case 0x5322: wc = 0x5e4f; break;
        case 0x5323: wc = 0x5fa6; break;
        case 0x5324: wc = 0x6118; break;
        case 0x5325: wc = 0x659d; break;
        case 0x5326: wc = 0x6935; break;
        case 0x5327: wc = 0x698e; break;
        case 0x5328: wc = 0x6a16; break;
        case 0x5329: wc = 0x6a9f; break;
        case 0x532a: wc = 0x6be0; break;
        case 0x532b: wc = 0x6cc7; break;
        case 0x532c: wc = 0x6e2e; break;
        case 0x532d: wc = 0x6e8a; break;
        case 0x532e: wc = 0x6ed2; break;
        case 0x532f: wc = 0x70a3; break;
        case 0x5330: wc = 0x7241; break;
        case 0x5331: wc = 0x728c; break;
        case 0x5332: wc = 0x7333; break;
        case 0x5333: wc = 0x73c8; break;
        case 0x5334: wc = 0x7638; break;
        case 0x5335: wc = 0x767f; break;
        case 0x5336: wc = 0x7b34; break;
        case 0x5337: wc = 0x7b33; break;
        case 0x5338: wc = 0x801e; break;
        case 0x5339: wc = 0x8238; break;
        case 0x533a: wc = 0x846d; break;
        case 0x533b: wc = 0x86b5; break;
        case 0x533c: wc = 0x8b0c; break;
        case 0x533d: wc = 0x8c6d; break;
        case 0x533e: wc = 0x8c91; break;
        case 0x533f: wc = 0x9160; break;
        case 0x5340: wc = 0x926b; break;
        case 0x5341: wc = 0x93b5; break;
        case 0x5342: wc = 0x4bca; break;
        case 0x5343: wc = 0x9ac2; break;
        case 0x5344: wc = 0x9b7a; break;
        case 0x5345: wc = 0x9d10; break;
        case 0x5346: wc = 0x9e9a; break;
        case 0x5347: wc = 0x4d97; break;
        case 0x5348: wc = 0x4f6b; break;
        case 0x5349: wc = 0x5095; break;
        case 0x534a: wc = 0x537b; break;
        case 0x534b: wc = 0x54af; break;
        case 0x534c: wc = 0x57c6; break;
        case 0x534d: wc = 0x5859; break;
        case 0x534e: wc = 0x6354; break;
        case 0x534f: wc = 0x6409; break;
        case 0x5350: wc = 0x64f1; break;
        case 0x5351: wc = 0x65a0; break;
        case 0x5352: wc = 0x6877; break;
        case 0x5353: wc = 0x69b7; break;
        case 0x5354: wc = 0x6aca; break;
        case 0x5355: wc = 0x6bc3; break;
        case 0x5356: wc = 0x7910; break;
        case 0x5357: wc = 0x80f3; break;
        case 0x5358: wc = 0x8316; break;
        case 0x5359: wc = 0x88bc; break;
        case 0x535a: wc = 0x96ba; break;
        case 0x535b: wc = 0x9d45; break;
        case 0x535c: wc = 0x5a5c; break;
        case 0x535d: wc = 0x5fd3; break;
        case 0x535e: wc = 0x6173; break;
        case 0x535f: wc = 0x653c; break;
        case 0x5360: wc = 0x65f0; break;
        case 0x5361: wc = 0x6695; break;
        case 0x5362: wc = 0x681e; break;
        case 0x5363: wc = 0x69a6; break;
        case 0x5364: wc = 0x6a8a; break;
        case 0x5365: wc = 0x23c68; break;
        case 0x5366: wc = 0x6d86; break;
        case 0x5367: wc = 0x7395; break;
        case 0x5368: wc = 0x7647; break;
        case 0x5369: wc = 0x76af; break;
        case 0x536a: wc = 0x3ffb; break;
        case 0x536b: wc = 0x76f0; break;
        case 0x536c: wc = 0x77af; break;
        case 0x536d: wc = 0x784d; break;
        case 0x536e: wc = 0x79c6; break;
        case 0x536f: wc = 0x7b78; break;
        case 0x5370: wc = 0x7c33; break;
        case 0x5371: wc = 0x7caf; break;
        case 0x5372: wc = 0x7fa5; break;
        case 0x5373: wc = 0x8289; break;
        case 0x5374: wc = 0x831b; break;
        case 0x5375: wc = 0x844c; break;
        case 0x5376: wc = 0x8551; break;
        case 0x5377: wc = 0x8677; break;
        case 0x5378: wc = 0x884e; break;
        case 0x5379: wc = 0x88c9; break;
        case 0x537a: wc = 0x8949; break;
        case 0x537b: wc = 0x8c64; break;
        case 0x537c: wc = 0x8d76; break;
        case 0x537d: wc = 0x8d95; break;
        case 0x537e: wc = 0x8fc0; break;
        case 0x537f: wc = 0x9417; break;
        case 0x5380: wc = 0x495c; break;
        case 0x5381: wc = 0x976c; break;
        case 0x5382: wc = 0x9845; break;
        case 0x5383: wc = 0x99af; break;
        case 0x5384: wc = 0x9b1c; break;
        case 0x5385: wc = 0x9cf1; break;
        case 0x5386: wc = 0x4d5f; break;
        case 0x5387: wc = 0x9f66; break;
        case 0x5388: wc = 0x3413; break;
        case 0x5389: wc = 0x34e4; break;
        case 0x538a: wc = 0x5676; break;
        case 0x538b: wc = 0x573f; break;
        case 0x538c: wc = 0x5d51; break;
        case 0x538d: wc = 0x5db1; break;
        case 0x538e: wc = 0x6274; break;
        case 0x538f: wc = 0x64d6; break;
        case 0x5390: wc = 0x668d; break;
        case 0x5391: wc = 0x696c; break;
        case 0x5392: wc = 0x6bfc; break;
        case 0x5393: wc = 0x3d63; break;
        case 0x5394: wc = 0x7332; break;
        case 0x5395: wc = 0x7366; break;
        case 0x5396: wc = 0x24e0e; break;
        case 0x5397: wc = 0x79f8; break;
        case 0x5398: wc = 0x7faf; break;
        case 0x5399: wc = 0x8f35; break;
        case 0x539a: wc = 0x8f55; break;
        case 0x539b: wc = 0x937b; break;
        case 0x539c: wc = 0x9782; break;
        case 0x539d: wc = 0x9a14; break;
        case 0x539e: wc = 0x9ab1; break;
        case 0x539f: wc = 0x29a72; break;
        case 0x53a0: wc = 0x9da1; break;
        case 0x53a1: wc = 0x51cf; break;
        case 0x53a2: wc = 0x51f5; break;
        case 0x53a3: wc = 0x4aaa; break;
        case 0x53a4: wc = 0x5481; break;
        case 0x53a5: wc = 0x361a; break;
        case 0x53a6: wc = 0x5769; break;
        case 0x53a7: wc = 0x57f3; break;
        case 0x53a8: wc = 0x583f; break;
        case 0x53a9: wc = 0x5888; break;
        case 0x53aa: wc = 0x3673; break;
        case 0x53ab: wc = 0x5c32; break;
        case 0x53ac: wc = 0x5c36; break;
        case 0x53ad: wc = 0x5d01; break;
        case 0x53ae: wc = 0x5d45; break;
        case 0x53af: wc = 0x5d41; break;
        case 0x53b0: wc = 0x60c2; break;
        case 0x53b1: wc = 0x61a8; break;
        case 0x53b2: wc = 0x6496; break;
        case 0x53b3: wc = 0x64bc; break;
        case 0x53b4: wc = 0x6937; break;
        case 0x53b5: wc = 0x6b3f; break;
        case 0x53b6: wc = 0x6b41; break;
        case 0x53b7: wc = 0x6b5b; break;
        case 0x53b8: wc = 0x6c39; break;
        case 0x53b9: wc = 0x6cd4; break;
        case 0x53ba: wc = 0x6de6; break;
        case 0x53bb: wc = 0x6f89; break;
        case 0x53bc: wc = 0x73b5; break;
        case 0x53bd: wc = 0x3ee3; break;
        case 0x53be: wc = 0x744a; break;
        case 0x53bf: wc = 0x751d; break;
        case 0x53c0: wc = 0x77d9; break;
        case 0x53c1: wc = 0x780d; break;
        case 0x53c2: wc = 0x78e1; break;
        case 0x53c3: wc = 0x791b; break;
        case 0x53c4: wc = 0x7af7; break;
        case 0x53c5: wc = 0x7bcf; break;
        case 0x53c6: wc = 0x82f7; break;
        case 0x53c7: wc = 0x83b0; break;
        case 0x53c8: wc = 0x86b6; break;
        case 0x53c9: wc = 0x8a4c; break;
        case 0x53ca: wc = 0x8f21; break;
        case 0x53cb: wc = 0x8f57; break;
        case 0x53cc: wc = 0x9163; break;
        case 0x53cd: wc = 0x95de; break;
        case 0x53ce: wc = 0x9c64; break;
        case 0x53cf: wc = 0x9e7b; break;
        case 0x53d0: wc = 0x9e7c; break;
        case 0x53d1: wc = 0x5392; break;
        case 0x53d2: wc = 0x5e22; break;
        case 0x53d3: wc = 0x6415; break;
        case 0x53d4: wc = 0x6546; break;
        case 0x53d5: wc = 0x73be; break;
        case 0x53d6: wc = 0x778c; break;
        case 0x53d7: wc = 0x7b1a; break;
        case 0x53d8: wc = 0x823a; break;
        case 0x53d9: wc = 0x3599; break;
        case 0x53da: wc = 0x50f5; break;
        case 0x53db: wc = 0x531f; break;
        case 0x53dc: wc = 0x5534; break;
        case 0x53dd: wc = 0x554c; break;
        case 0x53de: wc = 0x55f4; break;
        case 0x53df: wc = 0x56e5; break;
        case 0x53e0: wc = 0x583d; break;
        case 0x53e1: wc = 0x58c3; break;
        case 0x53e2: wc = 0x5add; break;
        case 0x53e3: wc = 0x5d79; break;
        case 0x53e4: wc = 0x5f36; break;
        case 0x53e5: wc = 0x5ffc; break;
        case 0x53e6: wc = 0x625b; break;
        case 0x53e7: wc = 0x6386; break;
        case 0x53e8: wc = 0x6760; break;
        case 0x53e9: wc = 0x68e1; break;
        case 0x53ea: wc = 0x690c; break;
        case 0x53eb: wc = 0x6a7f; break;
        case 0x53ec: wc = 0x6bad; break;
        case 0x53ed: wc = 0x6f12; break;
        case 0x53ee: wc = 0x6f2e; break;
        case 0x53ef: wc = 0x7135; break;
        case 0x53f0: wc = 0x7285; break;
        case 0x53f1: wc = 0x729f; break;
        case 0x53f2: wc = 0x3e54; break;
        case 0x53f3: wc = 0x72ba; break;
        case 0x53f4: wc = 0x7347; break;
        case 0x53f5: wc = 0x7392; break;
        case 0x53f6: wc = 0x74e8; break;
        case 0x53f7: wc = 0x77fc; break;
        case 0x53f8: wc = 0x7899; break;
        case 0x53f9: wc = 0x254cc; break;
        case 0x53fa: wc = 0x40e8; break;
        case 0x53fb: wc = 0x7913; break;
        case 0x53fc: wc = 0x7a45; break;
        case 0x53fd: wc = 0x7ce8; break;
        case 0x53fe: wc = 0x7ce1; break;
        case 0x53ff: wc = 0x7e48; break;
        case 0x5400: wc = 0x7e6e; break;
        case 0x5401: wc = 0x7f61; break;
        case 0x5402: wc = 0x7f97; break;
        case 0x5403: wc = 0x7fde; break;
        case 0x5404: wc = 0x8029; break;
        case 0x5405: wc = 0x8199; break;
        case 0x5406: wc = 0x8333; break;
        case 0x5407: wc = 0x8723; break;
        case 0x5408: wc = 0x45e7; break;
        case 0x5409: wc = 0x88b6; break;
        case 0x540a: wc = 0x8b3d; break;
        case 0x540b: wc = 0x8c47; break;
        case 0x540c: wc = 0x91ed; break;
        case 0x540d: wc = 0x93ee; break;
        case 0x540e: wc = 0x93f9; break;
        case 0x540f: wc = 0x97c1; break;
        case 0x5410: wc = 0x985c; break;
        case 0x5411: wc = 0x4e10; break;
        case 0x5412: wc = 0x4e2a; break;
        case 0x5413: wc = 0x5274; break;
        case 0x5414: wc = 0x52d3; break;
        case 0x5415: wc = 0x5303; break;
        case 0x5416: wc = 0x5424; break;
        case 0x5417: wc = 0x5588; break;
        case 0x5418: wc = 0x5605; break;
        case 0x5419: wc = 0x374f; break;
        case 0x541a: wc = 0x5c2c; break;
        case 0x541b: wc = 0x5c95; break;
        case 0x541c: wc = 0x6224; break;
        case 0x541d: wc = 0x63e9; break;
        case 0x541e: wc = 0x6461; break;
        case 0x541f: wc = 0x669f; break;
        case 0x5420: wc = 0x69e9; break;
        case 0x5421: wc = 0x23665; break;
        case 0x5422: wc = 0x6e5d; break;
        case 0x5423: wc = 0x708c; break;
        case 0x5424: wc = 0x70d7; break;
        case 0x5425: wc = 0x7297; break;
        case 0x5426: wc = 0x73a0; break;
        case 0x5427: wc = 0x74c2; break;
        case 0x5428: wc = 0x780e; break;
        case 0x5429: wc = 0x78d5; break;
        case 0x542a: wc = 0x7944; break;
        case 0x542b: wc = 0x7974; break;
        case 0x542c: wc = 0x7a2d; break;
        case 0x542d: wc = 0x4578; break;
        case 0x542e: wc = 0x86a7; break;
        case 0x542f: wc = 0x8754; break;
        case 0x5430: wc = 0x8878; break;
        case 0x5431: wc = 0x256ba; break;
        case 0x5432: wc = 0x9223; break;
        case 0x5433: wc = 0x9347; break;
        case 0x5434: wc = 0x938e; break;
        case 0x5435: wc = 0x9426; break;
        case 0x5436: wc = 0x95d3; break;
        case 0x5437: wc = 0x9623; break;
        case 0x5438: wc = 0x98bd; break;
        case 0x5439: wc = 0x9d9b; break;
        case 0x543a: wc = 0x4614; break;
        case 0x543b: wc = 0x358b; break;
        case 0x543c: wc = 0x2227c; break;
        case 0x543d: wc = 0x39f6; break;
        case 0x543e: wc = 0x785c; break;
        case 0x543f: wc = 0x787b; break;
        case 0x5440: wc = 0x40d8; break;
        case 0x5441: wc = 0x79d4; break;
        case 0x5442: wc = 0x8ce1; break;
        case 0x5443: wc = 0x4856; break;
        case 0x5444: wc = 0x92b5; break;
        case 0x5445: wc = 0x935e; break;
        case 0x5446: wc = 0x93d7; break;
        case 0x5447: wc = 0x962c; break;
        case 0x5448: wc = 0x5671; break;
        case 0x5449: wc = 0x81c4; break;
        case 0x544a: wc = 0x8c3b; break;
        case 0x544b: wc = 0x4f62; break;
        case 0x544c: wc = 0x4fe5; break;
        case 0x544d: wc = 0x5337; break;
        case 0x544e: wc = 0x547f; break;
        case 0x544f: wc = 0x58c9; break;
        case 0x5450: wc = 0x59d6; break;
        case 0x5451: wc = 0x5ca0; break;
        case 0x5452: wc = 0x5d0c; break;
        case 0x5453: wc = 0x6007; break;
        case 0x5454: wc = 0x61c5; break;
        case 0x5455: wc = 0x661b; break;
        case 0x5456: wc = 0x67dc; break;
        case 0x5457: wc = 0x6910; break;
        case 0x5458: wc = 0x249b2; break;
        case 0x5459: wc = 0x6b05; break;
        case 0x545a: wc = 0x3c26; break;
        case 0x545b: wc = 0x6b6b; break;
        case 0x545c: wc = 0x6d30; break;
        case 0x545d: wc = 0x6dba; break;
        case 0x545e: wc = 0x6fbd; break;
        case 0x545f: wc = 0x741a; break;
        case 0x5460: wc = 0x74a9; break;
        case 0x5461: wc = 0x78f2; break;
        case 0x5462: wc = 0x79ec; break;
        case 0x5463: wc = 0x7b65; break;
        case 0x5464: wc = 0x7c34; break;
        case 0x5465: wc = 0x7c67; break;
        case 0x5466: wc = 0x7c94; break;
        case 0x5467: wc = 0x7d36; break;
        case 0x5468: wc = 0x28b54; break;
        case 0x5469: wc = 0x7fd7; break;
        case 0x546a: wc = 0x801f; break;
        case 0x546b: wc = 0x80e0; break;
        case 0x546c: wc = 0x8152; break;
        case 0x546d: wc = 0x824d; break;
        case 0x546e: wc = 0x82e3; break;
        case 0x546f: wc = 0x8392; break;
        case 0x5470: wc = 0x83c3; break;
        case 0x5471: wc = 0x8556; break;
        case 0x5472: wc = 0x8627; break;
        case 0x5473: wc = 0x8661; break;
        case 0x5474: wc = 0x86b7; break;
        case 0x5475: wc = 0x871b; break;
        case 0x5476: wc = 0x87b6; break;
        case 0x5477: wc = 0x87dd; break;
        case 0x5478: wc = 0x8850; break;
        case 0x5479: wc = 0x88aa; break;
        case 0x547a: wc = 0x88fe; break;
        case 0x547b: wc = 0x8977; break;
        case 0x547c: wc = 0x8a4e; break;
        case 0x547d: wc = 0x8c66; break;
        case 0x547e: wc = 0x8e86; break;
        case 0x547f: wc = 0x943b; break;
        case 0x5480: wc = 0x9639; break;
        case 0x5481: wc = 0x99cf; break;
        case 0x5482: wc = 0x9d8b; break;
        case 0x5483: wc = 0x9eae; break;
        case 0x5484: wc = 0x9f01; break;
        case 0x5485: wc = 0x5dea; break;
        case 0x5486: wc = 0x2043b; break;
        case 0x5487: wc = 0x34fa; break;
        case 0x5488: wc = 0x56dd; break;
        case 0x5489: wc = 0x5898; break;
        case 0x548a: wc = 0x5bcb; break;
        case 0x548b: wc = 0x63f5; break;
        case 0x548c: wc = 0x3a1c; break;
        case 0x548d: wc = 0x6434; break;
        case 0x548e: wc = 0x6510; break;
        case 0x548f: wc = 0x6513; break;
        case 0x5490: wc = 0x69a9; break;
        case 0x5491: wc = 0x6bfd; break;
        case 0x5492: wc = 0x6e55; break;
        case 0x5493: wc = 0x6f27; break;
        case 0x5494: wc = 0x703d; break;
        case 0x5495: wc = 0x728d; break;
        case 0x5496: wc = 0x7777; break;
        case 0x5497: wc = 0x85c6; break;
        case 0x5498: wc = 0x8930; break;
        case 0x5499: wc = 0x466d; break;
        case 0x549a: wc = 0x8ad0; break;
        case 0x549b: wc = 0x8b07; break;
        case 0x549c: wc = 0x9473; break;
        case 0x549d: wc = 0x97ac; break;
        case 0x549e: wc = 0x9a1d; break;
        case 0x549f: wc = 0x9c0e; break;
        case 0x54a0: wc = 0x9c2c; break;
        case 0x54a1: wc = 0x9db1; break;
        case 0x54a2: wc = 0x4e6c; break;
        case 0x54a3: wc = 0x5d65; break;
        case 0x54a4: wc = 0x6705; break;
        case 0x54a5: wc = 0x69a4; break;
        case 0x54a6: wc = 0x6aed; break;
        case 0x54a7: wc = 0x6ed0; break;
        case 0x54a8: wc = 0x25e6e; break;
        case 0x54a9: wc = 0x829e; break;
        case 0x54aa: wc = 0x85d2; break;
        case 0x54ab: wc = 0x4ef1; break;
        case 0x54ac: wc = 0x5ff4; break;
        case 0x54ad: wc = 0x64bf; break;
        case 0x54ae: wc = 0x81c9; break;
        case 0x54af: wc = 0x82a1; break;
        case 0x54b0: wc = 0x5226; break;
        case 0x54b1: wc = 0x393c; break;
        case 0x54b2: wc = 0x3958; break;
        case 0x54b3: wc = 0x62be; break;
        case 0x54b4: wc = 0x6605; break;
        case 0x54b5: wc = 0x6781; break;
        case 0x54b6: wc = 0x781d; break;
        case 0x54b7: wc = 0x8871; break;
        case 0x54b8: wc = 0x88b7; break;
        case 0x54b9: wc = 0x8df2; break;
        case 0x54ba: wc = 0x9263; break;
        case 0x54bb: wc = 0x9d56; break;
        case 0x54bc: wc = 0x551f; break;
        case 0x54bd: wc = 0x6112; break;
        case 0x54be: wc = 0x453e; break;
        case 0x54bf: wc = 0x55dd; break;
        case 0x54c0: wc = 0x58bc; break;
        case 0x54c1: wc = 0x6105; break;
        case 0x54c2: wc = 0x630c; break;
        case 0x54c3: wc = 0x6439; break;
        case 0x54c4: wc = 0x643f; break;
        case 0x54c5: wc = 0x6941; break;
        case 0x54c6: wc = 0x6bc4; break;
        case 0x54c7: wc = 0x6e68; break;
        case 0x54c8: wc = 0x6ec6; break;
        case 0x54c9: wc = 0x7291; break;
        case 0x54ca: wc = 0x72ca; break;
        case 0x54cb: wc = 0x7365; break;
        case 0x54cc: wc = 0x7d8c; break;
        case 0x54cd: wc = 0x7dd9; break;
        case 0x54ce: wc = 0x7e73; break;
        case 0x54cf: wc = 0x8582; break;
        case 0x54d0: wc = 0x86d2; break;
        case 0x54d1: wc = 0x89e1; break;
        case 0x54d2: wc = 0x8f5a; break;
        case 0x54d3: wc = 0x90f9; break;
        case 0x54d4: wc = 0x95c3; break;
        case 0x54d5: wc = 0x4a82; break;
        case 0x54d6: wc = 0x9abc; break;
        case 0x54d7: wc = 0x9b32; break;
        case 0x54d8: wc = 0x9d03; break;
        case 0x54d9: wc = 0x9d59; break;
        case 0x54da: wc = 0x9daa; break;
        case 0x54db: wc = 0x9e04; break;
        case 0x54dc: wc = 0x9f33; break;
        case 0x54dd: wc = 0x5c8d; break;
        case 0x54de: wc = 0x2f87a; break;
        case 0x54df: wc = 0x5e75; break;
        case 0x54e0: wc = 0x6338; break;
        case 0x54e1: wc = 0x63ae; break;
        case 0x54e2: wc = 0x6394; break;
        case 0x54e3: wc = 0x67a4; break;
        case 0x54e4: wc = 0x6898; break;
        case 0x54e5: wc = 0x6a2b; break;
        case 0x54e6: wc = 0x6c71; break;
        case 0x54e7: wc = 0x6c67; break;
        case 0x54e8: wc = 0x6d80; break;
        case 0x54e9: wc = 0x719e; break;
        case 0x54ea: wc = 0x72ad; break;
        case 0x54eb: wc = 0x72f7; break;
        case 0x54ec: wc = 0x730f; break;
        case 0x54ed: wc = 0x7367; break;
        case 0x54ee: wc = 0x753d; break;
        case 0x54ef: wc = 0x754e; break;
        case 0x54f0: wc = 0x774a; break;
        case 0x54f1: wc = 0x7b67; break;
        case 0x54f2: wc = 0x7e34; break;
        case 0x54f3: wc = 0x7e7e; break;
        case 0x54f4: wc = 0x7f65; break;
        case 0x54f5: wc = 0x7f82; break;
        case 0x54f6: wc = 0x83fa; break;
        case 0x54f7: wc = 0x8738; break;
        case 0x54f8: wc = 0x8832; break;
        case 0x54f9: wc = 0x88d0; break;
        case 0x54fa: wc = 0x897a; break;
        case 0x54fb: wc = 0x8a43; break;
        case 0x54fc: wc = 0x8c5c; break;
        case 0x54fd: wc = 0x8c63; break;
        case 0x54fe: wc = 0x8dbc; break;
        case 0x54ff: wc = 0x9104; break;
        case 0x5500: wc = 0x9203; break;
        case 0x5501: wc = 0x9292; break;
        case 0x5502: wc = 0x92fb; break;
        case 0x5503: wc = 0x93f2; break;
        case 0x5504: wc = 0x96c3; break;
        case 0x5505: wc = 0x4b7e; break;
        case 0x5506: wc = 0x9c39; break;
        case 0x5507: wc = 0x9cfd; break;
        case 0x5508: wc = 0x9d73; break;
        case 0x5509: wc = 0x9e89; break;
        case 0x550a: wc = 0x50ea; break;
        case 0x550b: wc = 0x6d2f; break;
        case 0x550c: wc = 0x7094; break;
        case 0x550d: wc = 0x7106; break;
        case 0x550e: wc = 0x73a6; break;
        case 0x550f: wc = 0x258aa; break;
        case 0x5510: wc = 0x82b5; break;
        case 0x5511: wc = 0x849b; break;
        case 0x5512: wc = 0x8697; break;
        case 0x5513: wc = 0x86ea; break;
        case 0x5514: wc = 0x88ba; break;
        case 0x5515: wc = 0x89d6; break;
        case 0x5516: wc = 0x89fc; break;
        case 0x5517: wc = 0x8d7d; break;
        case 0x5518: wc = 0x8db9; break;
        case 0x5519: wc = 0x8ffc; break;
        case 0x551a: wc = 0x91e8; break;
        case 0x551b: wc = 0x920c; break;
        case 0x551c: wc = 0x493f; break;
        case 0x551d: wc = 0x9411; break;
        case 0x551e: wc = 0x95cb; break;
        case 0x551f: wc = 0x99c3; break;
        case 0x5520: wc = 0x9d02; break;
        case 0x5521: wc = 0x5094; break;
        case 0x5522: wc = 0x55db; break;
        case 0x5523: wc = 0x5c92; break;
        case 0x5524: wc = 0x62d1; break;
        case 0x5525: wc = 0x69cf; break;
        case 0x5526: wc = 0x6b49; break;
        case 0x5527: wc = 0x3ee9; break;
        case 0x5528: wc = 0x7e11; break;
        case 0x5529: wc = 0x7fac; break;
        case 0x552a: wc = 0x8181; break;
        case 0x552b: wc = 0x84b9; break;
        case 0x552c: wc = 0x9c1c; break;
        case 0x552d: wc = 0x9dbc; break;
        case 0x552e: wc = 0x9eda; break;
        case 0x552f: wc = 0x9f38; break;
        case 0x5530: wc = 0x550a; break;
        case 0x5531: wc = 0x6390; break;
        case 0x5532: wc = 0x6b8e; break;
        case 0x5533: wc = 0x88cc; break;
        case 0x5534: wc = 0x90df; break;
        case 0x5535: wc = 0x97d0; break;
        case 0x5536: wc = 0x9d4a; break;
        case 0x5537: wc = 0x5182; break;
        case 0x5538: wc = 0x518f; break;
        case 0x5539: wc = 0x34cf; break;
        case 0x553a: wc = 0x5244; break;
        case 0x553b: wc = 0x5260; break;
        case 0x553c: wc = 0x54fd; break;
        case 0x553d: wc = 0x56e7; break;
        case 0x553e: wc = 0x57c2; break;
        case 0x553f: wc = 0x57db; break;
        case 0x5540: wc = 0x5a5b; break;
        case 0x5541: wc = 0x5de0; break;
        case 0x5542: wc = 0x5e5c; break;
        case 0x5543: wc = 0x22234; break;
        case 0x5544: wc = 0x5ece; break;
        case 0x5545: wc = 0x5f33; break;
        case 0x5546: wc = 0x60f8; break;
        case 0x5547: wc = 0x61bc; break;
        case 0x5548: wc = 0x6243; break;
        case 0x5549: wc = 0x22aad; break;
        case 0x554a: wc = 0x632d; break;
        case 0x554b: wc = 0x6333; break;
        case 0x554c: wc = 0x63b6; break;
        case 0x554d: wc = 0x64cf; break;
        case 0x554e: wc = 0x66d4; break;
        case 0x554f: wc = 0x6871; break;
        case 0x5550: wc = 0x6929; break;
        case 0x5551: wc = 0x6a08; break;
        case 0x5552: wc = 0x6a69; break;
        case 0x5553: wc = 0x6aa0; break;
        case 0x5554: wc = 0x6abe; break;
        case 0x5555: wc = 0x6ae6; break;
        case 0x5556: wc = 0x6b8c; break;
        case 0x5557: wc = 0x6c2b; break;
        case 0x5558: wc = 0x6d6d; break;
        case 0x5559: wc = 0x6ef0; break;
        case 0x555a: wc = 0x6f00; break;
        case 0x555b: wc = 0x70f4; break;
        case 0x555c: wc = 0x712d; break;
        case 0x555d: wc = 0x713f; break;
        case 0x555e: wc = 0x7162; break;
        case 0x555f: wc = 0x715a; break;
        case 0x5560: wc = 0x3de1; break;
        case 0x5561: wc = 0x71b2; break;
        case 0x5562: wc = 0x71db; break;
        case 0x5563: wc = 0x71dd; break;
        case 0x5564: wc = 0x727c; break;
        case 0x5565: wc = 0x72c5; break;
        case 0x5566: wc = 0x734d; break;
        case 0x5567: wc = 0x743c; break;
        case 0x5568: wc = 0x749a; break;
        case 0x5569: wc = 0x754a; break;
        case 0x556a: wc = 0x7758; break;
        case 0x556b: wc = 0x778f; break;
        case 0x556c: wc = 0x7d4c; break;
        case 0x556d: wc = 0x7d86; break;
        case 0x556e: wc = 0x9848; break;
        case 0x556f: wc = 0x7f44; break;
        case 0x5570: wc = 0x82d8; break;
        case 0x5571: wc = 0x8384; break;
        case 0x5572: wc = 0x83ee; break;
        case 0x5573: wc = 0x845d; break;
        case 0x5574: wc = 0x451b; break;
        case 0x5575: wc = 0x85d1; break;
        case 0x5576: wc = 0x87fc; break;
        case 0x5577: wc = 0x8927; break;
        case 0x5578: wc = 0x8a99; break;
        case 0x5579: wc = 0x8b26; break;
        case 0x557a: wc = 0x8d79; break;
        case 0x557b: wc = 0x8e01; break;
        case 0x557c: wc = 0x90e0; break;
        case 0x557d: wc = 0x9344; break;
        case 0x557e: wc = 0x944b; break;
        case 0x557f: wc = 0x99c9; break;
        case 0x5580: wc = 0x9abe; break;
        case 0x5581: wc = 0x9bc1; break;
        case 0x5582: wc = 0x4c94; break;
        case 0x5583: wc = 0x9d5b; break;
        case 0x5584: wc = 0x9d81; break;
        case 0x5585: wc = 0x9d8a; break;
        case 0x5586: wc = 0x9e96; break;
        case 0x5587: wc = 0x9ea0; break;
        case 0x5588: wc = 0x9ee5; break;
        case 0x5589: wc = 0x4e69; break;
        case 0x558a: wc = 0x535f; break;
        case 0x558b: wc = 0x5447; break;
        case 0x558c: wc = 0x5826; break;
        case 0x558d: wc = 0x213e8; break;
        case 0x558e: wc = 0x5c4a; break;
        case 0x558f: wc = 0x5cdc; break;
        case 0x5590: wc = 0x5f50; break;
        case 0x5591: wc = 0x2e95; break;
        case 0x5592: wc = 0x5f51; break;
        case 0x5593: wc = 0x6088; break;
        case 0x5594: wc = 0x660b; break;
        case 0x5595: wc = 0x66a9; break;
        case 0x5596: wc = 0x6785; break;
        case 0x5597: wc = 0x6814; break;
        case 0x5598: wc = 0x6a95; break;
        case 0x5599: wc = 0x6ab5; break;
        case 0x559a: wc = 0x6d0e; break;
        case 0x559b: wc = 0x7031; break;
        case 0x559c: wc = 0x9e02; break;
        case 0x559d: wc = 0x70d3; break;
        case 0x559e: wc = 0xf0700; break;
        case 0x559f: wc = 0x7508; break;
        case 0x55a0: wc = 0x75f5; break;
        case 0x55a1: wc = 0x7608; break;
        case 0x55a2: wc = 0x761b; break;
        case 0x55a3: wc = 0x78b6; break;
        case 0x55a4: wc = 0x798a; break;
        case 0x55a5: wc = 0x7a27; break;
        case 0x55a6: wc = 0x7b04; break;
        case 0x55a7: wc = 0x7b40; break;
        case 0x55a8: wc = 0x7d12; break;
        case 0x55a9: wc = 0x7d99; break;
        case 0x55aa: wc = 0x7dae; break;
        case 0x55ab: wc = 0x7f4a; break;
        case 0x55ac: wc = 0x7f7d; break;
        case 0x55ad: wc = 0x43ff; break;
        case 0x55ae: wc = 0x845c; break;
        case 0x55af: wc = 0x858a; break;
        case 0x55b0: wc = 0x862e; break;
        case 0x55b1: wc = 0x863b; break;
        case 0x55b2: wc = 0x87ff; break;
        case 0x55b3: wc = 0x9288; break;
        case 0x55b4: wc = 0x9365; break;
        case 0x55b5: wc = 0x96de; break;
        case 0x55b6: wc = 0x9afb; break;
        case 0x55b7: wc = 0x9bda; break;
        case 0x55b8: wc = 0x9e02; break;
        case 0x55b9: wc = 0x9f58; break;
        case 0x55ba: wc = 0x4f30; break;
        case 0x55bb: wc = 0x4fa4; break;
        case 0x55bc: wc = 0x346c; break;
        case 0x55bd: wc = 0x50f1; break;
        case 0x55be: wc = 0x51c5; break;
        case 0x55bf: wc = 0x5233; break;
        case 0x55c0: wc = 0x52c2; break;
        case 0x55c1: wc = 0x5495; break;
        case 0x55c2: wc = 0x2129c; break;
        case 0x55c3: wc = 0x580c; break;
        case 0x55c4: wc = 0x5903; break;
        case 0x55c5: wc = 0x5a5f; break;
        case 0x55c6: wc = 0x5af4; break;
        case 0x55c7: wc = 0x5d13; break;
        case 0x55c8: wc = 0x38e8; break;
        case 0x55c9: wc = 0x630e; break;
        case 0x55ca: wc = 0x641e; break;
        case 0x55cb: wc = 0x6772; break;
        case 0x55cc: wc = 0x67e7; break;
        case 0x55cd: wc = 0x6832; break;
        case 0x55ce: wc = 0x684d; break;
        case 0x55cf: wc = 0x68dd; break;
        case 0x55d0: wc = 0x695b; break;
        case 0x55d1: wc = 0x69c0; break;
        case 0x55d2: wc = 0x69f9; break;
        case 0x55d3: wc = 0x6a6d; break;
        case 0x55d4: wc = 0x6a70; break;
        case 0x55d5: wc = 0x6aba; break;
        case 0x55d6: wc = 0x6adc; break;
        case 0x55d7: wc = 0x6cd2; break;
        case 0x55d8: wc = 0x6d18; break;
        case 0x55d9: wc = 0x3d0c; break;
        case 0x55da: wc = 0x3d48; break;
        case 0x55db: wc = 0x23f4e; break;
        case 0x55dc: wc = 0x70e4; break;
        case 0x55dd: wc = 0x7138; break;
        case 0x55de: wc = 0x7170; break;
        case 0x55df: wc = 0x7187; break;
        case 0x55e0: wc = 0x726f; break;
        case 0x55e1: wc = 0x72dc; break;
        case 0x55e2: wc = 0x7614; break;
        case 0x55e3: wc = 0x3fb8; break;
        case 0x55e4: wc = 0x4007; break;
        case 0x55e5: wc = 0x76ec; break;
        case 0x55e6: wc = 0x4026; break;
        case 0x55e7: wc = 0x77bd; break;
        case 0x55e8: wc = 0x7970; break;
        case 0x55e9: wc = 0x7a12; break;
        case 0x55ea: wc = 0x7b1f; break;
        case 0x55eb: wc = 0x7b76; break;
        case 0x55ec: wc = 0x7b8d; break;
        case 0x55ed: wc = 0x7b9b; break;
        case 0x55ee: wc = 0x7bd9; break;
        case 0x55ef: wc = 0x4249; break;
        case 0x55f0: wc = 0x7cd5; break;
        case 0x55f1: wc = 0x7d5d; break;
        case 0x55f2: wc = 0x7f5f; break;
        case 0x55f3: wc = 0x7f5b; break;
        case 0x55f4: wc = 0x7f96; break;
        case 0x55f5: wc = 0x7ff6; break;
        case 0x55f6: wc = 0x80cd; break;
        case 0x55f7: wc = 0x81cc; break;
        case 0x55f8: wc = 0x81ef; break;
        case 0x55f9: wc = 0x83c7; break;
        case 0x55fa: wc = 0x2700e; break;
        case 0x55fb: wc = 0x85a7; break;
        case 0x55fc: wc = 0x85a3; break;
        case 0x55fd: wc = 0x86c4; break;
        case 0x55fe: wc = 0x86cc; break;
        case 0x55ff: wc = 0x8932; break;
        case 0x5600: wc = 0x89da; break;
        case 0x5601: wc = 0x8a41; break;
        case 0x5602: wc = 0x8dcd; break;
        case 0x5603: wc = 0x8ef1; break;
        case 0x5604: wc = 0x8ef2; break;
        case 0x5605: wc = 0x90dc; break;
        case 0x5606: wc = 0x9164; break;
        case 0x5607: wc = 0x9232; break;
        case 0x5608: wc = 0x9237; break;
        case 0x5609: wc = 0x92ac; break;
        case 0x560a: wc = 0x92ef; break;
        case 0x560b: wc = 0x9760; break;
        case 0x560c: wc = 0x993b; break;
        case 0x560d: wc = 0x9ab7; break;
        case 0x560e: wc = 0x4bfb; break;
        case 0x560f: wc = 0x9b95; break;
        case 0x5610: wc = 0x9bdd; break;
        case 0x5611: wc = 0x9d23; break;
        case 0x5612: wc = 0x9dce; break;
        case 0x5613: wc = 0x9f14; break;
        case 0x5614: wc = 0x9f1b; break;
        case 0x5615: wc = 0x5503; break;
        case 0x5616: wc = 0x561d; break;
        case 0x5617: wc = 0x56b3; break;
        case 0x5618: wc = 0x69f2; break;
        case 0x5619: wc = 0x236de; break;
        case 0x561a: wc = 0x6d40; break;
        case 0x561b: wc = 0x6ff2; break;
        case 0x561c: wc = 0x7014; break;
        case 0x561d: wc = 0x702b; break;
        case 0x561e: wc = 0x7105; break;
        case 0x561f: wc = 0x7100; break;
        case 0x5620: wc = 0x727f; break;
        case 0x5621: wc = 0x7474; break;
        case 0x5622: wc = 0x785e; break;
        case 0x5623: wc = 0xfa54; break;
        case 0x5624: wc = 0x7cac; break;
        case 0x5625: wc = 0x7e20; break;
        case 0x5626: wc = 0x851b; break;
        case 0x5627: wc = 0x86d0; break;
        case 0x5628: wc = 0x879c; break;
        case 0x5629: wc = 0x89f3; break;
        case 0x562a: wc = 0x8f42; break;
        case 0x562b: wc = 0x9af7; break;
        case 0x562c: wc = 0x2a360; break;
        case 0x562d: wc = 0x4e28; break;
        case 0x562e: wc = 0x5803; break;
        case 0x562f: wc = 0x5812; break;
        case 0x5630: wc = 0x58fc; break;
        case 0x5631: wc = 0x5d10; break;
        case 0x5632: wc = 0x6083; break;
        case 0x5633: wc = 0x60c3; break;
        case 0x5634: wc = 0x6346; break;
        case 0x5635: wc = 0x665c; break;
        case 0x5636: wc = 0x5c21; break;
        case 0x5637: wc = 0x68de; break;
        case 0x5638: wc = 0x6d83; break;
        case 0x5639: wc = 0x7111; break;
        case 0x563a: wc = 0x71b4; break;
        case 0x563b: wc = 0x7311; break;
        case 0x563c: wc = 0x24973; break;
        case 0x563d: wc = 0x74ad; break;
        case 0x563e: wc = 0x774f; break;
        case 0x563f: wc = 0x7754; break;
        case 0x5640: wc = 0x7871; break;
        case 0x5641: wc = 0x25566; break;
        case 0x5642: wc = 0x7a07; break;
        case 0x5643: wc = 0x7d91; break;
        case 0x5644: wc = 0x7dc4; break;
        case 0x5645: wc = 0x264fc; break;
        case 0x5646: wc = 0x81d7; break;
        case 0x5647: wc = 0x83ce; break;
        case 0x5648: wc = 0x8509; break;
        case 0x5649: wc = 0x88cd; break;
        case 0x564a: wc = 0x88e9; break;
        case 0x564b: wc = 0x890c; break;
        case 0x564c: wc = 0x8f25; break;
        case 0x564d: wc = 0x918c; break;
        case 0x564e: wc = 0x9315; break;
        case 0x564f: wc = 0x95ab; break;
        case 0x5650: wc = 0x9a09; break;
        case 0x5651: wc = 0x9ae1; break;
        case 0x5652: wc = 0x9b8c; break;
        case 0x5653: wc = 0x9bc0; break;
        case 0x5654: wc = 0x9d7e; break;
        case 0x5655: wc = 0x9da4; break;
        case 0x5656: wc = 0x9f6b; break;
        case 0x5657: wc = 0x3414; break;
        case 0x5658: wc = 0x3423; break;
        case 0x5659: wc = 0x50a6; break;
        case 0x565a: wc = 0x5723; break;
        case 0x565b: wc = 0x5c33; break;
        case 0x565c: wc = 0x6132; break;
        case 0x565d: wc = 0x6287; break;
        case 0x565e: wc = 0x6430; break;
        case 0x565f: wc = 0x675a; break;
        case 0x5660: wc = 0x69be; break;
        case 0x5661: wc = 0x77fb; break;
        case 0x5662: wc = 0x7e0e; break;
        case 0x5663: wc = 0x84c7; break;
        case 0x5664: wc = 0x985d; break;
        case 0x5665: wc = 0x9936; break;
        case 0x5666: wc = 0x9dbb; break;
        case 0x5667: wc = 0x4be9; break;
        case 0x5668: wc = 0x359c; break;
        case 0x5669: wc = 0x848a; break;
        case 0x566a: wc = 0x5ee4; break;
        case 0x566b: wc = 0x2012b; break;
        case 0x566c: wc = 0x5025; break;
        case 0x566d: wc = 0x5868; break;
        case 0x566e: wc = 0x5d06; break;
        case 0x566f: wc = 0x382b; break;
        case 0x5670: wc = 0x5e4a; break;
        case 0x5671: wc = 0x5efe; break;
        case 0x5672: wc = 0x60be; break;
        case 0x5673: wc = 0x6129; break;
        case 0x5674: wc = 0x62f2; break;
        case 0x5675: wc = 0x684f; break;
        case 0x5676: wc = 0x6831; break;
        case 0x5677: wc = 0x3b5f; break;
        case 0x5678: wc = 0x69d3; break;
        case 0x5679: wc = 0x6db3; break;
        case 0x567a: wc = 0x6ed8; break;
        case 0x567b: wc = 0x7068; break;
        case 0x567c: wc = 0x3eac; break;
        case 0x567d: wc = 0x7598; break;
        case 0x567e: wc = 0x40ac; break;
        case 0x567f: wc = 0x787f; break;
        case 0x5680: wc = 0x25524; break;
        case 0x5681: wc = 0x7b3b; break;
        case 0x5682: wc = 0x7b47; break;
        case 0x5683: wc = 0x7b9c; break;
        case 0x5684: wc = 0x7be2; break;
        case 0x5685: wc = 0x7fbe; break;
        case 0x5686: wc = 0x823c; break;
        case 0x5687: wc = 0x26de7; break;
        case 0x5688: wc = 0x4508; break;
        case 0x5689: wc = 0x86ec; break;
        case 0x568a: wc = 0x86e9; break;
        case 0x568b: wc = 0x8d1b; break;
        case 0x568c: wc = 0x8deb; break;
        case 0x568d: wc = 0x8f01; break;
        case 0x568e: wc = 0x909b; break;
        case 0x568f: wc = 0x928e; break;
        case 0x5690: wc = 0x979a; break;
        case 0x5691: wc = 0x4c4b; break;
        case 0x5692: wc = 0x9f94; break;
        case 0x5693: wc = 0x201d7; break;
        case 0x5694: wc = 0x4f89; break;
        case 0x5695: wc = 0x526e; break;
        case 0x5696: wc = 0x54b5; break;
        case 0x5697: wc = 0x576c; break;
        case 0x5698: wc = 0x57ae; break;
        case 0x5699: wc = 0x5801; break;
        case 0x569a: wc = 0x581d; break;
        case 0x569b: wc = 0x5925; break;
        case 0x569c: wc = 0x5938; break;
        case 0x569d: wc = 0x59f1; break;
        case 0x569e: wc = 0x5d59; break;
        case 0x569f: wc = 0x60c8; break;
        case 0x56a0: wc = 0x39d3; break;
        case 0x56a1: wc = 0x64be; break;
        case 0x56a2: wc = 0x6564; break;
        case 0x56a3: wc = 0x6647; break;
        case 0x56a4: wc = 0x68f5; break;
        case 0x56a5: wc = 0x6a9b; break;
        case 0x56a6: wc = 0x6dc9; break;
        case 0x56a7: wc = 0x6fc4; break;
        case 0x56a8: wc = 0x7290; break;
        case 0x56a9: wc = 0x7313; break;
        case 0x56aa: wc = 0x24b01; break;
        case 0x56ab: wc = 0x7611; break;
        case 0x56ac: wc = 0x7a1e; break;
        case 0x56ad: wc = 0x7aa0; break;
        case 0x56ae: wc = 0x7c3b; break;
        case 0x56af: wc = 0x7cbf; break;
        case 0x56b0: wc = 0x80ef; break;
        case 0x56b1: wc = 0x81bc; break;
        case 0x56b2: wc = 0x823f; break;
        case 0x56b3: wc = 0x8342; break;
        case 0x56b4: wc = 0x842a; break;
        case 0x56b5: wc = 0x8596; break;
        case 0x56b6: wc = 0x873e; break;
        case 0x56b7: wc = 0x874c; break;
        case 0x56b8: wc = 0x88f9; break;
        case 0x56b9: wc = 0x8e1d; break;
        case 0x56ba: wc = 0x8f20; break;
        case 0x56bb: wc = 0x9299; break;
        case 0x56bc: wc = 0x9301; break;
        case 0x56bd: wc = 0x9439; break;
        case 0x56be: wc = 0x991c; break;
        case 0x56bf: wc = 0x298d4; break;
        case 0x56c0: wc = 0x9a0d; break;
        case 0x56c1: wc = 0x9abb; break;
        case 0x56c2: wc = 0x9ac1; break;
        case 0x56c3: wc = 0x588e; break;
        case 0x56c4: wc = 0x5d1e; break;
        case 0x56c5: wc = 0x3c0c; break;
        case 0x56c6: wc = 0x6b14; break;
        case 0x56c7: wc = 0x6f37; break;
        case 0x56c8: wc = 0x7016; break;
        case 0x56c9: wc = 0x7668; break;
        case 0x56ca: wc = 0x7c57; break;
        case 0x56cb: wc = 0x8ea9; break;
        case 0x56cc: wc = 0x9481; break;
        case 0x56cd: wc = 0x970d; break;
        case 0x56ce: wc = 0x97b9; break;
        case 0x56cf: wc = 0x4e31; break;
        case 0x56d0: wc = 0x500c; break;
        case 0x56d1: wc = 0x535d; break;
        case 0x56d2: wc = 0x60ba; break;
        case 0x56d3: wc = 0x60b9; break;
        case 0x56d4: wc = 0x645c; break;
        case 0x56d5: wc = 0x6a0c; break;
        case 0x56d6: wc = 0x6bcc; break;
        case 0x56d7: wc = 0x6dab; break;
        case 0x56d8: wc = 0x70ea; break;
        case 0x56d9: wc = 0x721f; break;
        case 0x56da: wc = 0x75ef; break;
        case 0x56db: wc = 0x761d; break;
        case 0x56dc: wc = 0x76e5; break;
        case 0x56dd: wc = 0x77d4; break;
        case 0x56de: wc = 0x7936; break;
        case 0x56df: wc = 0x797c; break;
        case 0x56e0: wc = 0x7abe; break;
        case 0x56e1: wc = 0x7b66; break;
        case 0x56e2: wc = 0x7db0; break;
        case 0x56e3: wc = 0x7f46; break;
        case 0x56e4: wc = 0x43d3; break;
        case 0x56e5: wc = 0x8218; break;
        case 0x56e6: wc = 0x8484; break;
        case 0x56e7: wc = 0x4540; break;
        case 0x56e8: wc = 0x8cd7; break;
        case 0x56e9: wc = 0x8f28; break;
        case 0x56ea: wc = 0x9327; break;
        case 0x56eb: wc = 0x9467; break;
        case 0x56ec: wc = 0x9475; break;
        case 0x56ed: wc = 0x96da; break;
        case 0x56ee: wc = 0x9ad6; break;
        case 0x56ef: wc = 0x4c98; break;
        case 0x56f0: wc = 0x9c79; break;
        case 0x56f1: wc = 0x9e1b; break;
        case 0x56f2: wc = 0x4f78; break;
        case 0x56f3: wc = 0x5280; break;
        case 0x56f4: wc = 0x61d6; break;
        case 0x56f5: wc = 0x681d; break;
        case 0x56f6: wc = 0x7b48; break;
        case 0x56f7: wc = 0x8052; break;
        case 0x56f8: wc = 0x44b7; break;
        case 0x56f9: wc = 0x8440; break;
        case 0x56fa: wc = 0x98b3; break;
        case 0x56fb: wc = 0x4bcf; break;
        case 0x56fc: wc = 0x9afa; break;
        case 0x56fd: wc = 0x9b20; break;
        case 0x56fe: wc = 0x9d30; break;
        case 0x56ff: wc = 0x4fc7; break;
        case 0x5700: wc = 0x50d9; break;
        case 0x5701: wc = 0x52bb; break;
        case 0x5702: wc = 0x54d0; break;
        case 0x5703: wc = 0x5799; break;
        case 0x5704: wc = 0x6047; break;
        case 0x5705: wc = 0x61ed; break;
        case 0x5706: wc = 0x6282; break;
        case 0x5707: wc = 0x6497; break;
        case 0x5708: wc = 0x6846; break;
        case 0x5709: wc = 0x6844; break;
        case 0x570a: wc = 0x6d2d; break;
        case 0x570b: wc = 0x70e1; break;
        case 0x570c: wc = 0x720c; break;
        case 0x570d: wc = 0x3e70; break;
        case 0x570e: wc = 0x7377; break;
        case 0x570f: wc = 0x7736; break;
        case 0x5710: wc = 0x77cc; break;
        case 0x5711: wc = 0x7844; break;
        case 0x5712: wc = 0x78fa; break;
        case 0x5713: wc = 0x7926; break;
        case 0x5714: wc = 0x7a6c; break;
        case 0x5715: wc = 0x7d56; break;
        case 0x5716: wc = 0x7e8a; break;
        case 0x5717: wc = 0x81e6; break;
        case 0x5718: wc = 0x81e9; break;
        case 0x5719: wc = 0x44b0; break;
        case 0x571a: wc = 0x832a; break;
        case 0x571b: wc = 0x8a86; break;
        case 0x571c: wc = 0x8a91; break;
        case 0x571d: wc = 0x8fcb; break;
        case 0x571e: wc = 0x901b; break;
        case 0x571f: wc = 0x913a; break;
        case 0x5720: wc = 0x9271; break;
        case 0x5721: wc = 0x92a7; break;
        case 0x5722: wc = 0x92db; break;
        case 0x5723: wc = 0x9d5f; break;
        case 0x5724: wc = 0x9ec6; break;
        case 0x5725: wc = 0x54bc; break;
        case 0x5726: wc = 0x558e; break;
        case 0x5727: wc = 0x6302; break;
        case 0x5728: wc = 0x67b4; break;
        case 0x5729: wc = 0x7d53; break;
        case 0x572a: wc = 0x7f63; break;
        case 0x572b: wc = 0x8902; break;
        case 0x572c: wc = 0x8a7f; break;
        case 0x572d: wc = 0x5abf; break;
        case 0x572e: wc = 0x5ddc; break;
        case 0x572f: wc = 0x5ee5; break;
        case 0x5730: wc = 0x65dd; break;
        case 0x5731: wc = 0x6ac6; break;
        case 0x5732: wc = 0x6b33; break;
        case 0x5733: wc = 0x70e0; break;
        case 0x5734: wc = 0x7470; break;
        case 0x5735: wc = 0x74cc; break;
        case 0x5736: wc = 0x7650; break;
        case 0x5737: wc = 0x26bb1; break;
        case 0x5738: wc = 0x84af; break;
        case 0x5739: wc = 0x8562; break;
        case 0x573a: wc = 0x863e; break;
        case 0x573b: wc = 0x4661; break;
        case 0x573c: wc = 0x8958; break;
        case 0x573d: wc = 0x5268; break;
        case 0x573e: wc = 0x5613; break;
        case 0x573f: wc = 0x5e57; break;
        case 0x5740: wc = 0x6156; break;
        case 0x5741: wc = 0x6451; break;
        case 0x5742: wc = 0x6f0d; break;
        case 0x5743: wc = 0x3d81; break;
        case 0x5744: wc = 0x805d; break;
        case 0x5745: wc = 0x8195; break;
        case 0x5746: wc = 0x8662; break;
        case 0x5747: wc = 0x87c8; break;
        case 0x5748: wc = 0x9998; break;
        case 0x5749: wc = 0x54a3; break;
        case 0x574a: wc = 0x6c6f; break;
        case 0x574b: wc = 0x6d64; break;
        case 0x574c: wc = 0x6e39; break;
        case 0x574d: wc = 0x7861; break;
        case 0x574e: wc = 0x7ae4; break;
        case 0x574f: wc = 0x7fc3; break;
        case 0x5750: wc = 0x26483; break;
        case 0x5751: wc = 0x803e; break;
        case 0x5752: wc = 0x89e5; break;
        case 0x5753: wc = 0x8a07; break;
        case 0x5754: wc = 0x9313; break;
        case 0x5755: wc = 0x9360; break;
        case 0x5756: wc = 0x9367; break;
        case 0x5757: wc = 0x958e; break;
        case 0x5758: wc = 0x9783; break;
        case 0x5759: wc = 0x4f7c; break;
        case 0x575a: wc = 0x510c; break;
        case 0x575b: wc = 0x554b; break;
        case 0x575c: wc = 0x5602; break;
        case 0x575d: wc = 0x5610; break;
        case 0x575e: wc = 0x563a; break;
        case 0x575f: wc = 0x566d; break;
        case 0x5760: wc = 0x5699; break;
        case 0x5761: wc = 0x58bd; break;
        case 0x5762: wc = 0x59e3; break;
        case 0x5763: wc = 0x5b13; break;
        case 0x5764: wc = 0x5c69; break;
        case 0x5765: wc = 0x5ce7; break;
        case 0x5766: wc = 0x5d7a; break;
        case 0x5767: wc = 0x6054; break;
        case 0x5768: wc = 0x618d; break;
        case 0x5769: wc = 0x630d; break;
        case 0x576a: wc = 0x6341; break;
        case 0x576b: wc = 0x649f; break;
        case 0x576c: wc = 0x64bd; break;
        case 0x576d: wc = 0x656b; break;
        case 0x576e: wc = 0x657d; break;
        case 0x576f: wc = 0x6648; break;
        case 0x5770: wc = 0x669e; break;
        case 0x5771: wc = 0x66d2; break;
        case 0x5772: wc = 0x705a; break;
        case 0x5773: wc = 0x71c6; break;
        case 0x5774: wc = 0x729e; break;
        case 0x5775: wc = 0x735f; break;
        case 0x5776: wc = 0x73d3; break;
        case 0x5777: wc = 0x3f71; break;
        case 0x5778: wc = 0x76a6; break;
        case 0x5779: wc = 0x4009; break;
        case 0x577a: wc = 0x7904; break;
        case 0x577b: wc = 0x78fd; break;
        case 0x577c: wc = 0x7909; break;
        case 0x577d: wc = 0x7a5a; break;
        case 0x577e: wc = 0x7a8c; break;
        case 0x577f: wc = 0x7a96; break;
        case 0x5780: wc = 0x7c25; break;
        case 0x5781: wc = 0x7e51; break;
        case 0x5782: wc = 0x832d; break;
        case 0x5783: wc = 0x834d; break;
        case 0x5784: wc = 0x273b8; break;
        case 0x5785: wc = 0x87c2; break;
        case 0x5786: wc = 0x87dc; break;
        case 0x5787: wc = 0x8b51; break;
        case 0x5788: wc = 0x8b65; break;
        case 0x5789: wc = 0x8dac; break;
        case 0x578a: wc = 0x8dab; break;
        case 0x578b: wc = 0x8de4; break;
        case 0x578c: wc = 0x8e7a; break;
        case 0x578d: wc = 0x8e7b; break;
        case 0x578e: wc = 0x8e88; break;
        case 0x578f: wc = 0x8f47; break;
        case 0x5790: wc = 0x90fb; break;
        case 0x5791: wc = 0x9121; break;
        case 0x5792: wc = 0x9125; break;
        case 0x5793: wc = 0x9278; break;
        case 0x5794: wc = 0x97bd; break;
        case 0x5795: wc = 0x981d; break;
        case 0x5796: wc = 0x9ab9; break;
        case 0x5797: wc = 0x9adc; break;
        case 0x5798: wc = 0x9c4e; break;
        case 0x5799: wc = 0x9d41; break;
        case 0x579a: wc = 0x9dcd; break;
        case 0x579b: wc = 0x9dee; break;
        case 0x579c: wc = 0x9f69; break;
        case 0x579d: wc = 0x4f49; break;
        case 0x579e: wc = 0x4f5d; break;
        case 0x579f: wc = 0x4fc5; break;
        case 0x57a0: wc = 0x5003; break;
        case 0x57a1: wc = 0x508b; break;
        case 0x57a2: wc = 0x50b4; break;
        case 0x57a3: wc = 0x5193; break;
        case 0x57a4: wc = 0x52ac; break;
        case 0x57a5: wc = 0x52fc; break;
        case 0x57a6: wc = 0x53b9; break;
        case 0x57a7: wc = 0x53f4; break;
        case 0x57a8: wc = 0x5474; break;
        case 0x57a9: wc = 0x3617; break;
        case 0x57aa: wc = 0x57e7; break;
        case 0x57ab: wc = 0x591f; break;
        case 0x57ac: wc = 0x5920; break;
        case 0x57ad: wc = 0x59e4; break;
        case 0x57ae: wc = 0x5abe; break;
        case 0x57af: wc = 0x5ad7; break;
        case 0x57b0: wc = 0x5be0; break;
        case 0x57b1: wc = 0x3797; break;
        case 0x57b2: wc = 0x5c68; break;
        case 0x57b3: wc = 0x5ca3; break;
        case 0x57b4: wc = 0x3871; break;
        case 0x57b5: wc = 0x5ecf; break;
        case 0x57b6: wc = 0x5f40; break;
        case 0x57b7: wc = 0x5f44; break;
        case 0x57b8: wc = 0x6010; break;
        case 0x57b9: wc = 0x60e7; break;
        case 0x57ba: wc = 0x6166; break;
        case 0x57bb: wc = 0x97ef; break;
        case 0x57bc: wc = 0x6235; break;
        case 0x57bd: wc = 0x6263; break;
        case 0x57be: wc = 0x6344; break;
        case 0x57bf: wc = 0x6406; break;
        case 0x57c0: wc = 0x6473; break;
        case 0x57c1: wc = 0x6542; break;
        case 0x57c2: wc = 0x657a; break;
        case 0x57c3: wc = 0x3aba; break;
        case 0x57c4: wc = 0x65aa; break;
        case 0x57c5: wc = 0x65e7; break;
        case 0x57c6: wc = 0x662b; break;
        case 0x57c7: wc = 0x6710; break;
        case 0x57c8: wc = 0x6739; break;
        case 0x57c9: wc = 0x6784; break;
        case 0x57ca: wc = 0x6855; break;
        case 0x57cb: wc = 0x6882; break;
        case 0x57cc: wc = 0x6907; break;
        case 0x57cd: wc = 0x6998; break;
        case 0x57ce: wc = 0x6b0b; break;
        case 0x57cf: wc = 0x6b0d; break;
        case 0x57d0: wc = 0x6b28; break;
        case 0x57d1: wc = 0x6b8f; break;
        case 0x57d2: wc = 0x6c0d; break;
        case 0x57d3: wc = 0x23c84; break;
        case 0x57d4: wc = 0x6cc3; break;
        case 0x57d5: wc = 0x6ef1; break;
        case 0x57d6: wc = 0x6f1a; break;
        case 0x57d7: wc = 0x7048; break;
        case 0x57d8: wc = 0x71b0; break;
        case 0x57d9: wc = 0x725e; break;
        case 0x57da: wc = 0x728b; break;
        case 0x57db: wc = 0x72b0; break;
        case 0x57dc: wc = 0x73bd; break;
        case 0x57dd: wc = 0x7486; break;
        case 0x57de: wc = 0x750c; break;
        case 0x57df: wc = 0x759a; break;
        case 0x57e0: wc = 0x75c0; break;
        case 0x57e1: wc = 0x766f; break;
        case 0x57e2: wc = 0x76da; break;
        case 0x57e3: wc = 0x7789; break;
        case 0x57e4: wc = 0x7798; break;
        case 0x57e5: wc = 0x7ab6; break;
        case 0x57e6: wc = 0x7ad8; break;
        case 0x57e7: wc = 0x7b31; break;
        case 0x57e8: wc = 0x7b58; break;
        case 0x57e9: wc = 0x7bdd; break;
        case 0x57ea: wc = 0x7c06; break;
        case 0x57eb: wc = 0x7c3c; break;
        case 0x57ec: wc = 0x7c82; break;
        case 0x57ed: wc = 0x7cd7; break;
        case 0x57ee: wc = 0x7d0c; break;
        case 0x57ef: wc = 0x7d47; break;
        case 0x57f0: wc = 0x7df1; break;
        case 0x57f1: wc = 0x7fd1; break;
        case 0x57f2: wc = 0x80ca; break;
        case 0x57f3: wc = 0x8119; break;
        case 0x57f4: wc = 0x81de; break;
        case 0x57f5: wc = 0x827d; break;
        case 0x57f6: wc = 0x82b6; break;
        case 0x57f7: wc = 0x8329; break;
        case 0x57f8: wc = 0x844b; break;
        case 0x57f9: wc = 0x849f; break;
        case 0x57fa: wc = 0x26d77; break;
        case 0x57fb: wc = 0x84f2; break;
        case 0x57fc: wc = 0x8532; break;
        case 0x57fd: wc = 0x86af; break;
        case 0x57fe: wc = 0x86bc; break;
        case 0x57ff: wc = 0x86f7; break;
        case 0x5800: wc = 0x877a; break;
        case 0x5801: wc = 0x8837; break;
        case 0x5802: wc = 0x883c; break;
        case 0x5803: wc = 0x88a7; break;
        case 0x5804: wc = 0x88d8; break;
        case 0x5805: wc = 0x8920; break;
        case 0x5806: wc = 0x4654; break;
        case 0x5807: wc = 0x89af; break;
        case 0x5808: wc = 0x89d3; break;
        case 0x5809: wc = 0x89e9; break;
        case 0x580a: wc = 0x8a04; break;
        case 0x580b: wc = 0x8a3d; break;
        case 0x580c: wc = 0x46ee; break;
        case 0x580d: wc = 0x8cd5; break;
        case 0x580e: wc = 0x8dd4; break;
        case 0x580f: wc = 0x8ea3; break;
        case 0x5810: wc = 0x8ee5; break;
        case 0x5811: wc = 0x9058; break;
        case 0x5812: wc = 0x90ad; break;
        case 0x5813: wc = 0x91d3; break;
        case 0x5814: wc = 0x491b; break;
        case 0x5815: wc = 0x91da; break;
        case 0x5816: wc = 0x91e6; break;
        case 0x5817: wc = 0x28a23; break;
        case 0x5818: wc = 0x96ca; break;
        case 0x5819: wc = 0x97b2; break;
        case 0x581a: wc = 0x97dd; break;
        case 0x581b: wc = 0x97ed; break;
        case 0x581c: wc = 0x97ee; break;
        case 0x581d: wc = 0x98b6; break;
        case 0x581e: wc = 0x99c8; break;
        case 0x581f: wc = 0x9b2e; break;
        case 0x5820: wc = 0x9b88; break;
        case 0x5821: wc = 0x9bc4; break;
        case 0x5822: wc = 0x9be6; break;
        case 0x5823: wc = 0x9d1d; break;
        case 0x5824: wc = 0x9dc7; break;
        case 0x5825: wc = 0x9e1c; break;
        case 0x5826: wc = 0x9e94; break;
        case 0x5827: wc = 0x2a4df; break;
        case 0x5828: wc = 0x9f29; break;
        case 0x5829: wc = 0x9f3d; break;
        case 0x582a: wc = 0x2a6c3; break;
        case 0x582b: wc = 0x4fb7; break;
        case 0x582c: wc = 0x530a; break;
        case 0x582d: wc = 0x56d7; break;
        case 0x582e: wc = 0x6336; break;
        case 0x582f: wc = 0x63ac; break;
        case 0x5830: wc = 0x68ae; break;
        case 0x5831: wc = 0x6908; break;
        case 0x5832: wc = 0x6dd7; break;
        case 0x5833: wc = 0x7117; break;
        case 0x5834: wc = 0x861c; break;
        case 0x5835: wc = 0x8d9c; break;
        case 0x5836: wc = 0x8dfc; break;
        case 0x5837: wc = 0x8e18; break;
        case 0x5838: wc = 0x8eb9; break;
        case 0x5839: wc = 0x8f02; break;
        case 0x583a: wc = 0x92e6; break;
        case 0x583b: wc = 0x95b0; break;
        case 0x583c: wc = 0x9671; break;
        case 0x583d: wc = 0x99f6; break;
        case 0x583e: wc = 0x9a67; break;
        case 0x583f: wc = 0x29af4; break;
        case 0x5840: wc = 0x4c61; break;
        case 0x5841: wc = 0x9d74; break;
        case 0x5842: wc = 0x9eaf; break;
        case 0x5843: wc = 0x5bad; break;
        case 0x5844: wc = 0x6343; break;
        case 0x5845: wc = 0x6508; break;
        case 0x5846: wc = 0x687e; break;
        case 0x5847: wc = 0x6d92; break;
        case 0x5848: wc = 0x73fa; break;
        case 0x5849: wc = 0x76b8; break;
        case 0x584a: wc = 0x25c03; break;
        case 0x584b: wc = 0x7fa3; break;
        case 0x584c: wc = 0x4403; break;
        case 0x584d: wc = 0x8399; break;
        case 0x584e: wc = 0x8720; break;
        case 0x584f: wc = 0x8f11; break;
        case 0x5850: wc = 0x9355; break;
        case 0x5851: wc = 0x9835; break;
        case 0x5852: wc = 0x9bb6; break;
        case 0x5853: wc = 0x9d58; break;
        case 0x5854: wc = 0x3407; break;
        case 0x5855: wc = 0x341d; break;
        case 0x5856: wc = 0x5014; break;
        case 0x5857: wc = 0x5d1b; break;
        case 0x5858: wc = 0x3b7e; break;
        case 0x5859: wc = 0x6dc8; break;
        case 0x585a: wc = 0x3d60; break;
        case 0x585b: wc = 0x80d0; break;
        case 0x585c: wc = 0x8673; break;
        case 0x585d: wc = 0x27311; break;
        case 0x585e: wc = 0x8a58; break;
        case 0x585f: wc = 0x8d89; break;
        case 0x5860: wc = 0x957c; break;
        case 0x5861: wc = 0x9d8c; break;
        case 0x5862: wc = 0x358c; break;
        case 0x5863: wc = 0x5311; break;
        case 0x5864: wc = 0x38b1; break;
        case 0x5865: wc = 0x85ed; break;
        case 0x5866: wc = 0x8eb3; break;
        case 0x5867: wc = 0x9d0c; break;
        case 0x5868: wc = 0x5573; break;
        case 0x5869: wc = 0x57e2; break;
        case 0x586a: wc = 0x5a58; break;
        case 0x586b: wc = 0x5dcf; break;
        case 0x586c: wc = 0x5e23; break;
        case 0x586d: wc = 0x60d3; break;
        case 0x586e: wc = 0x684a; break;
        case 0x586f: wc = 0x68ec; break;
        case 0x5870: wc = 0x7276; break;
        case 0x5871: wc = 0x7288; break;
        case 0x5872: wc = 0x7760; break;
        case 0x5873: wc = 0x7d6d; break;
        case 0x5874: wc = 0x7da3; break;
        case 0x5875: wc = 0x83e4; break;
        case 0x5876: wc = 0x8528; break;
        case 0x5877: wc = 0x8647; break;
        case 0x5878: wc = 0x8737; break;
        case 0x5879: wc = 0x8838; break;
        case 0x587a: wc = 0x88f7; break;
        case 0x587b: wc = 0x89e0; break;
        case 0x587c: wc = 0x8e21; break;
        case 0x587d: wc = 0x9308; break;
        case 0x587e: wc = 0x97cf; break;
        case 0x587f: wc = 0x9874; break;
        case 0x5880: wc = 0x9b08; break;
        case 0x5881: wc = 0x9b33; break;
        case 0x5882: wc = 0x9f64; break;
        case 0x5883: wc = 0x4e85; break;
        case 0x5884: wc = 0x5282; break;
        case 0x5885: wc = 0x5658; break;
        case 0x5886: wc = 0x5b52; break;
        case 0x5887: wc = 0x5da1; break;
        case 0x5888: wc = 0x6a5b; break;
        case 0x5889: wc = 0x761a; break;
        case 0x588a: wc = 0x87e8; break;
        case 0x588b: wc = 0x9c56; break;
        case 0x588c: wc = 0x9de2; break;
        case 0x588d: wc = 0x4f79; break;
        case 0x588e: wc = 0x51e0; break;
        case 0x588f: wc = 0x51e7; break;
        case 0x5890: wc = 0x532d; break;
        case 0x5891: wc = 0x5331; break;
        case 0x5892: wc = 0x53ac; break;
        case 0x5893: wc = 0x571a; break;
        case 0x5894: wc = 0x579d; break;
        case 0x5895: wc = 0x367a; break;
        case 0x5896: wc = 0x59fd; break;
        case 0x5897: wc = 0x5b07; break;
        case 0x5898: wc = 0x6192; break;
        case 0x5899: wc = 0x39ea; break;
        case 0x589a: wc = 0x6485; break;
        case 0x589b: wc = 0x6a3b; break;
        case 0x589c: wc = 0x6c3f; break;
        case 0x589d: wc = 0x796a; break;
        case 0x589e: wc = 0x7c02; break;
        case 0x589f: wc = 0x7c0b; break;
        case 0x58a0: wc = 0x7c23; break;
        case 0x58a1: wc = 0x7c44; break;
        case 0x58a2: wc = 0x7e62; break;
        case 0x58a3: wc = 0x852e; break;
        case 0x58a4: wc = 0x86eb; break;
        case 0x58a5: wc = 0x89e4; break;
        case 0x58a6: wc = 0x8dea; break;
        case 0x58a7: wc = 0x9400; break;
        case 0x58a8: wc = 0x95e0; break;
        case 0x58a9: wc = 0x993d; break;
        case 0x58aa: wc = 0x9e82; break;
        case 0x58ab: wc = 0x528c; break;
        case 0x58ac: wc = 0x5633; break;
        case 0x58ad: wc = 0x5b84; break;
        case 0x58ae: wc = 0x5c2f; break;
        case 0x58af: wc = 0x648c; break;
        case 0x58b0: wc = 0x69f6; break;
        case 0x58b1: wc = 0x6af7; break;
        case 0x58b2: wc = 0x7688; break;
        case 0x58b3: wc = 0x77b6; break;
        case 0x58b4: wc = 0x8143; break;
        case 0x58b5: wc = 0x862c; break;
        case 0x58b6: wc = 0x9a29; break;
        case 0x58b7: wc = 0x4be3; break;
        case 0x58b8: wc = 0x4c0e; break;
        case 0x58b9: wc = 0x5232; break;
        case 0x58ba: wc = 0x544c; break;
        case 0x58bb: wc = 0x55b9; break;
        case 0x58bc: wc = 0x5604; break;
        case 0x58bd: wc = 0x5ae2; break;
        case 0x58be: wc = 0x5b00; break;
        case 0x58bf: wc = 0x5dcb; break;
        case 0x58c0: wc = 0x6058; break;
        case 0x58c1: wc = 0x6223; break;
        case 0x58c2: wc = 0x644e; break;
        case 0x58c3: wc = 0x646b; break;
        case 0x58c4: wc = 0x668c; break;
        case 0x58c5: wc = 0x673b; break;
        case 0x58c6: wc = 0x694f; break;
        case 0x58c7: wc = 0x6951; break;
        case 0x58c8: wc = 0x69fc; break;
        case 0x58c9: wc = 0x6a1b; break;
        case 0x58ca: wc = 0x6e40; break;
        case 0x58cb: wc = 0x7143; break;
        case 0x58cc: wc = 0x72aa; break;
        case 0x58cd: wc = 0x777d; break;
        case 0x58ce: wc = 0x77a1; break;
        case 0x58cf: wc = 0x7a90; break;
        case 0x58d0: wc = 0x7cfa; break;
        case 0x58d1: wc = 0x8067; break;
        case 0x58d2: wc = 0x26af6; break;
        case 0x58d3: wc = 0x82a4; break;
        case 0x58d4: wc = 0x8325; break;
        case 0x58d5: wc = 0x26c71; break;
        case 0x58d6: wc = 0x85c8; break;
        case 0x58d7: wc = 0x866f; break;
        case 0x58d8: wc = 0x8770; break;
        case 0x58d9: wc = 0x88bf; break;
        case 0x58da: wc = 0x8dec; break;
        case 0x58db: wc = 0x8e5e; break;
        case 0x58dc: wc = 0x90bd; break;
        case 0x58dd: wc = 0x9108; break;
        case 0x58de: wc = 0x9377; break;
        case 0x58df: wc = 0x95da; break;
        case 0x58e0: wc = 0x9804; break;
        case 0x58e1: wc = 0x980d; break;
        case 0x58e2: wc = 0x982f; break;
        case 0x58e3: wc = 0x9997; break;
        case 0x58e4: wc = 0x9a24; break;
        case 0x58e5: wc = 0x9b39; break;
        case 0x58e6: wc = 0x4c85; break;
        case 0x58e7: wc = 0x5441; break;
        case 0x58e8: wc = 0x56f7; break;
        case 0x58e9: wc = 0x5cee; break;
        case 0x58ea: wc = 0x62a3; break;
        case 0x58eb: wc = 0x6c6e; break;
        case 0x58ec: wc = 0x7885; break;
        case 0x58ed: wc = 0x7a1b; break;
        case 0x58ee: wc = 0x7b98; break;
        case 0x58ef: wc = 0x8690; break;
        case 0x58f0: wc = 0x8880; break;
        case 0x58f1: wc = 0x9e87; break;
        case 0x58f2: wc = 0x9e8f; break;
        case 0x58f3: wc = 0x9e95; break;
        case 0x58f4: wc = 0x4e2e; break;
        case 0x58f5: wc = 0x4e9f; break;
        case 0x58f6: wc = 0x52c0; break;
        case 0x58f7: wc = 0x5c50; break;
        case 0x58f8: wc = 0x395b; break;
        case 0x58f9: wc = 0x64a0; break;
        case 0x58fa: wc = 0x6a76; break;
        case 0x58fb: wc = 0x6b9b; break;
        case 0x58fc: wc = 0x6c2a; break;
        case 0x58fd: wc = 0x8540; break;
        case 0x58fe: wc = 0x894b; break;
        case 0x58ff: wc = 0x90c4; break;
        case 0x5900: wc = 0x90e4; break;
        case 0x5901: wc = 0x92f4; break;
        case 0x5902: wc = 0x4f12; break;
        case 0x5903: wc = 0x53aa; break;
        case 0x5904: wc = 0x54cf; break;
        case 0x5905: wc = 0x3626; break;
        case 0x5906: wc = 0x5807; break;
        case 0x5907: wc = 0x5890; break;
        case 0x5908: wc = 0x22037; break;
        case 0x5909: wc = 0x5df9; break;
        case 0x590a: wc = 0x5ed1; break;
        case 0x590b: wc = 0x616c; break;
        case 0x590c: wc = 0x39c6; break;
        case 0x590d: wc = 0x65b3; break;
        case 0x590e: wc = 0x6ba3; break;
        case 0x590f: wc = 0x6f0c; break;
        case 0x5910: wc = 0x763d; break;
        case 0x5911: wc = 0x423d; break;
        case 0x5912: wc = 0x80b5; break;
        case 0x5913: wc = 0x84f3; break;
        case 0x5914: wc = 0x87bc; break;
        case 0x5915: wc = 0x89d4; break;
        case 0x5916: wc = 0x8ddf; break;
        case 0x5917: wc = 0x91ff; break;
        case 0x5918: wc = 0x9773; break;
        case 0x5919: wc = 0x50f8; break;
        case 0x591a: wc = 0x51ce; break;
        case 0x591b: wc = 0x552b; break;
        case 0x591c: wc = 0x5659; break;
        case 0x591d: wc = 0x5664; break;
        case 0x591e: wc = 0x5745; break;
        case 0x591f: wc = 0x5d94; break;
        case 0x5920: wc = 0x5e88; break;
        case 0x5921: wc = 0x60cd; break;
        case 0x5922: wc = 0x6272; break;
        case 0x5923: wc = 0x6407; break;
        case 0x5924: wc = 0x6fbf; break;
        case 0x5925: wc = 0x7439; break;
        case 0x5926: wc = 0x743b; break;
        case 0x5927: wc = 0x3fa3; break;
        case 0x5928: wc = 0x781b; break;
        case 0x5929: wc = 0x40e2; break;
        case 0x592a: wc = 0x7b12; break;
        case 0x592b: wc = 0x7d1f; break;
        case 0x592c: wc = 0x8039; break;
        case 0x592d: wc = 0x80a3; break;
        case 0x592e: wc = 0x83f3; break;
        case 0x592f: wc = 0x8699; break;
        case 0x5930: wc = 0x8d7a; break;
        case 0x5931: wc = 0x8ee1; break;
        case 0x5932: wc = 0x9219; break;
        case 0x5933: wc = 0x9342; break;
        case 0x5934: wc = 0x9422; break;
        case 0x5935: wc = 0x96c2; break;
        case 0x5936: wc = 0x9772; break;
        case 0x5937: wc = 0x9849; break;
        case 0x5938: wc = 0x9cf9; break;
        case 0x5939: wc = 0x573e; break;
        case 0x593a: wc = 0x5b06; break;
        case 0x593b: wc = 0x5c8c; break;
        case 0x593c: wc = 0x5f76; break;
        case 0x593d: wc = 0x5fe3; break;
        case 0x593e: wc = 0x39c0; break;
        case 0x593f: wc = 0x7680; break;
        case 0x5940: wc = 0x7b08; break;
        case 0x5941: wc = 0x82a8; break;
        case 0x5942: wc = 0x89d9; break;
        case 0x5943: wc = 0x9dd1; break;
        case 0x5944: wc = 0x5829; break;
        case 0x5945: wc = 0x6046; break;
        case 0x5946: wc = 0x63af; break;
        case 0x5947: wc = 0x63ef; break;
        case 0x5948: wc = 0x6b91; break;
        case 0x5949: wc = 0x7d5a; break;
        case 0x594a: wc = 0x7dea; break;
        case 0x594b: wc = 0x8903; break;
        case 0x594c: wc = 0x9339; break;
        case 0x594d: wc = 0x4c4d; break;
        case 0x594e: wc = 0x4e0c; break;
        case 0x594f: wc = 0x4e93; break;
        case 0x5950: wc = 0x501b; break;
        case 0x5951: wc = 0x50db; break;
        case 0x5952: wc = 0x5209; break;
        case 0x5953: wc = 0x525e; break;
        case 0x5954: wc = 0x552d; break;
        case 0x5955: wc = 0x5630; break;
        case 0x5956: wc = 0x588d; break;
        case 0x5957: wc = 0x5980; break;
        case 0x5958: wc = 0x36bb; break;
        case 0x5959: wc = 0x5a38; break;
        case 0x595a: wc = 0x5a4d; break;
        case 0x595b: wc = 0x5c7a; break;
        case 0x595c: wc = 0x5c93; break;
        case 0x595d: wc = 0x22033; break;
        case 0x595e: wc = 0x5e3a; break;
        case 0x595f: wc = 0x5e8b; break;
        case 0x5960: wc = 0x5eaa; break;
        case 0x5961: wc = 0x5f03; break;
        case 0x5962: wc = 0x5f9b; break;
        case 0x5963: wc = 0x5fee; break;
        case 0x5964: wc = 0x603e; break;
        case 0x5965: wc = 0x60ce; break;
        case 0x5966: wc = 0x612d; break;
        case 0x5967: wc = 0x61fb; break;
        case 0x5968: wc = 0x638e; break;
        case 0x5969: wc = 0x6391; break;
        case 0x596a: wc = 0x3a7b; break;
        case 0x596b: wc = 0x6532; break;
        case 0x596c: wc = 0x6567; break;
        case 0x596d: wc = 0x65c2; break;
        case 0x596e: wc = 0x65e1; break;
        case 0x596f: wc = 0x66a3; break;
        case 0x5970: wc = 0x66c1; break;
        case 0x5971: wc = 0x68ca; break;
        case 0x5972: wc = 0x69bf; break;
        case 0x5973: wc = 0x6a2d; break;
        case 0x5974: wc = 0x6b67; break;
        case 0x5975: wc = 0x6c14; break;
        case 0x5976: wc = 0x6e0f; break;
        case 0x5977: wc = 0x6eb0; break;
        case 0x5978: wc = 0x6fdd; break;
        case 0x5979: wc = 0x7081; break;
        case 0x597a: wc = 0x7382; break;
        case 0x597b: wc = 0x76f5; break;
        case 0x597c: wc = 0x7895; break;
        case 0x597d: wc = 0x79a8; break;
        case 0x597e: wc = 0x7a18; break;
        case 0x597f: wc = 0x7a4a; break;
        case 0x5980: wc = 0x7a56; break;
        case 0x5981: wc = 0x7c0a; break;
        case 0x5982: wc = 0x7c2f; break;
        case 0x5983: wc = 0x7c4f; break;
        case 0x5984: wc = 0x7cb8; break;
        case 0x5985: wc = 0x7da6; break;
        case 0x5986: wc = 0x7f87; break;
        case 0x5987: wc = 0x81ee; break;
        case 0x5988: wc = 0x8291; break;
        case 0x5989: wc = 0x82aa; break;
        case 0x598a: wc = 0x82b0; break;
        case 0x598b: wc = 0x44bb; break;
        case 0x598c: wc = 0x8401; break;
        case 0x598d: wc = 0x8507; break;
        case 0x598e: wc = 0x85c4; break;
        case 0x598f: wc = 0x8637; break;
        case 0x5990: wc = 0x8604; break;
        case 0x5991: wc = 0x8641; break;
        case 0x5992: wc = 0x866e; break;
        case 0x5993: wc = 0x8691; break;
        case 0x5994: wc = 0x869a; break;
        case 0x5995: wc = 0x8694; break;
        case 0x5996: wc = 0x871e; break;
        case 0x5997: wc = 0x87e3; break;
        case 0x5998: wc = 0x8879; break;
        case 0x5999: wc = 0x8900; break;
        case 0x599a: wc = 0x898a; break;
        case 0x599b: wc = 0x89ac; break;
        case 0x599c: wc = 0x89ed; break;
        case 0x599d: wc = 0x8a8b; break;
        case 0x599e: wc = 0x8ac6; break;
        case 0x599f: wc = 0x8ac5; break;
        case 0x59a0: wc = 0x8dc2; break;
        case 0x59a1: wc = 0x8dfd; break;
        case 0x59a2: wc = 0x8e11; break;
        case 0x59a3: wc = 0x8e26; break;
        case 0x59a4: wc = 0x8ea8; break;
        case 0x59a5: wc = 0x8edd; break;
        case 0x59a6: wc = 0x9094; break;
        case 0x59a7: wc = 0x913f; break;
        case 0x59a8: wc = 0x9416; break;
        case 0x59a9: wc = 0x966d; break;
        case 0x59aa: wc = 0x9691; break;
        case 0x59ab: wc = 0x97bf; break;
        case 0x59ac: wc = 0x980e; break;
        case 0x59ad: wc = 0x9b10; break;
        case 0x59ae: wc = 0x9b3e; break;
        case 0x59af: wc = 0x9b3f; break;
        case 0x59b0: wc = 0x9b4c; break;
        case 0x59b1: wc = 0x9b55; break;
        case 0x59b2: wc = 0x29d57; break;
        case 0x59b3: wc = 0x9b62; break;
        case 0x59b4: wc = 0x9bd5; break;
        case 0x59b5: wc = 0x9c2d; break;
        case 0x59b6: wc = 0x9c40; break;
        case 0x59b7: wc = 0x9d4b; break;
        case 0x59b8: wc = 0x9d80; break;
        case 0x59b9: wc = 0x9d78; break;
        case 0x59ba: wc = 0x9ed6; break;
        case 0x59bb: wc = 0x9f6e; break;
        case 0x59bc: wc = 0x83e3; break;
        case 0x59bd: wc = 0x54ad; break;
        case 0x59be: wc = 0x59de; break;
        case 0x59bf: wc = 0x6d01; break;
        case 0x59c0: wc = 0x72e4; break;
        case 0x59c1: wc = 0x86e3; break;
        case 0x59c2: wc = 0x8d8c; break;
        case 0x59c3: wc = 0x92a1; break;
        case 0x59c4: wc = 0x978a; break;
        case 0x59c5: wc = 0x9b9a; break;
        case 0x59c6: wc = 0x54ea; break;
        case 0x59c7: wc = 0x35a0; break;
        case 0x59c8: wc = 0x61e7; break;
        case 0x59c9: wc = 0x6310; break;
        case 0x59ca: wc = 0x632a; break;
        case 0x59cb: wc = 0x689b; break;
        case 0x59cc: wc = 0x6a60; break;
        case 0x59cd: wc = 0x7a2c; break;
        case 0x59ce: wc = 0x7cef; break;
        case 0x59cf: wc = 0x8498; break;
        case 0x59d0: wc = 0x8abd; break;
        case 0x59d1: wc = 0x63bf; break;
        case 0x59d2: wc = 0x5044; break;
        case 0x59d3: wc = 0x597b; break;
        case 0x59d4: wc = 0x6201; break;
        case 0x59d5: wc = 0x6e1c; break;
        case 0x59d6: wc = 0x8d67; break;
        case 0x59d7: wc = 0x29021; break;
        case 0x59d8: wc = 0x992a; break;
        case 0x59d9: wc = 0x82f6; break;
        case 0x59da: wc = 0x9667; break;
        case 0x59db: wc = 0x3472; break;
        case 0x59dc: wc = 0x5583; break;
        case 0x59dd: wc = 0x5a1a; break;
        case 0x59de: wc = 0x5a7b; break;
        case 0x59df: wc = 0x38ee; break;
        case 0x59e0: wc = 0x63c7; break;
        case 0x59e1: wc = 0x6694; break;
        case 0x59e2: wc = 0x67df; break;
        case 0x59e3: wc = 0x3d5c; break;
        case 0x59e4: wc = 0x7175; break;
        case 0x59e5: wc = 0x7f71; break;
        case 0x59e6: wc = 0x8169; break;
        case 0x59e7: wc = 0x83ae; break;
        case 0x59e8: wc = 0x8433; break;
        case 0x59e9: wc = 0x877b; break;
        case 0x59ea: wc = 0x8af5; break;
        case 0x59eb: wc = 0x59a0; break;
        case 0x59ec: wc = 0x7b1d; break;
        case 0x59ed: wc = 0x84b3; break;
        case 0x59ee: wc = 0x8edc; break;
        case 0x59ef: wc = 0x9209; break;
        case 0x59f0: wc = 0x9b76; break;
        case 0x59f1: wc = 0x56d4; break;
        case 0x59f2: wc = 0x64c3; break;
        case 0x59f3: wc = 0x652e; break;
        case 0x59f4: wc = 0x66e9; break;
        case 0x59f5: wc = 0x7062; break;
        case 0x59f6: wc = 0x3fbf; break;
        case 0x59f7: wc = 0x9962; break;
        case 0x59f8: wc = 0x9f49; break;
        case 0x59f9: wc = 0x5302; break;
        case 0x59fa: wc = 0x5976; break;
        case 0x59fb: wc = 0x5b2d; break;
        case 0x59fc: wc = 0x5b7b; break;
        case 0x59fd: wc = 0x6468; break;
        case 0x59fe: wc = 0x3b88; break;
        case 0x59ff: wc = 0x6c16; break;
        case 0x5a00: wc = 0x6c1d; break;
        case 0x5a01: wc = 0x6e3f; break;
        case 0x5a02: wc = 0x800f; break;
        case 0x5a03: wc = 0x8418; break;
        case 0x5a04: wc = 0x879a; break;
        case 0x5a05: wc = 0x8926; break;
        case 0x5a06: wc = 0x8ffa; break;
        case 0x5a07: wc = 0x91e2; break;
        case 0x5a08: wc = 0x933c; break;
        case 0x5a09: wc = 0x9f10; break;
        case 0x5a0a: wc = 0x903d; break;
        case 0x5a0b: wc = 0x5e24; break;
        case 0x5a0c: wc = 0x88bd; break;
        case 0x5a0d: wc = 0x91f9; break;
        case 0x5a0e: wc = 0x60c4; break;
        case 0x5a0f: wc = 0x7592; break;
        case 0x5a10: wc = 0x54d6; break;
        case 0x5a11: wc = 0x59e9; break;
        case 0x5a12: wc = 0x6d8a; break;
        case 0x5a13: wc = 0x78be; break;
        case 0x5a14: wc = 0x4135; break;
        case 0x5a15: wc = 0x7c10; break;
        case 0x5a16: wc = 0x8e8e; break;
        case 0x5a17: wc = 0x9f30; break;
        case 0x5a18: wc = 0x7bde; break;
        case 0x5a19: wc = 0x57dd; break;
        case 0x5a1a: wc = 0x8e17; break;
        case 0x5a1b: wc = 0x49d4; break;
        case 0x5a1c: wc = 0x4b2f; break;
        case 0x5a1d: wc = 0x655c; break;
        case 0x5a1e: wc = 0x4f5e; break;
        case 0x5a1f: wc = 0x4fab; break;
        case 0x5a20: wc = 0x511c; break;
        case 0x5a21: wc = 0x5680; break;
        case 0x5a22: wc = 0x5b23; break;
        case 0x5a23: wc = 0x5bcd; break;
        case 0x5a24: wc = 0x64f0; break;
        case 0x5a25: wc = 0x6a63; break;
        case 0x5a26: wc = 0x6ab8; break;
        case 0x5a27: wc = 0x6f9d; break;
        case 0x5a28: wc = 0x6fd8; break;
        case 0x5a29: wc = 0x752f; break;
        case 0x5a2a: wc = 0x8079; break;
        case 0x5a2b: wc = 0x85b4; break;
        case 0x5a2c: wc = 0x45ff; break;
        case 0x5a2d: wc = 0x944f; break;
        case 0x5a2e: wc = 0x9e0b; break;
        case 0x5a2f: wc = 0x79b0; break;
        case 0x5a30: wc = 0x5476; break;
        case 0x5a31: wc = 0x5912; break;
        case 0x5a32: wc = 0x5b65; break;
        case 0x5a33: wc = 0xf0701; break;
        case 0x5a34: wc = 0x5cf1; break;
        case 0x5a35: wc = 0x6013; break;
        case 0x5a36: wc = 0x7331; break;
        case 0x5a37: wc = 0x737f; break;
        case 0x5a38: wc = 0x782e; break;
        case 0x5a39: wc = 0x7b2f; break;
        case 0x5a3a: wc = 0x7ca9; break;
        case 0x5a3b: wc = 0x80ec; break;
        case 0x5a3c: wc = 0x81d1; break;
        case 0x5a3d: wc = 0x86ef; break;
        case 0x5a3e: wc = 0x9465; break;
        case 0x5a3f: wc = 0x5089; break;
        case 0x5a40: wc = 0x3410; break;
        case 0x5a41: wc = 0x3417; break;
        case 0x5a42: wc = 0x439b; break;
        case 0x5a43: wc = 0x5102; break;
        case 0x5a44: wc = 0x5665; break;
        case 0x5a45: wc = 0x5da9; break;
        case 0x5a46: wc = 0x7651; break;
        case 0x5a47: wc = 0x7a60; break;
        case 0x5a48: wc = 0x7e77; break;
        case 0x5a49: wc = 0x857d; break;
        case 0x5a4a: wc = 0x895b; break;
        case 0x5a4b: wc = 0x91b2; break;
        case 0x5a4c: wc = 0x9b1e; break;
        case 0x5a4d: wc = 0x9f48; break;
        case 0x5a4e: wc = 0x96eb; break;
        case 0x5a4f: wc = 0x8c7d; break;
        case 0x5a50: wc = 0x8c80; break;
        case 0x5a51: wc = 0x209e7; break;
        case 0x5a52: wc = 0x5a1e; break;
        case 0x5a53: wc = 0x61b9; break;
        case 0x5a54: wc = 0x633c; break;
        case 0x5a55: wc = 0x637c; break;
        case 0x5a56: wc = 0x254f2; break;
        case 0x5a57: wc = 0x25525; break;
        case 0x5a58: wc = 0x8147; break;
        case 0x5a59: wc = 0x27d98; break;
        case 0x5a5a: wc = 0x9912; break;
        case 0x5a5b: wc = 0x9bbe; break;
        case 0x5a5c: wc = 0x5acb; break;
        case 0x5a5d: wc = 0x5b1d; break;
        case 0x5a5e: wc = 0x5b32; break;
        case 0x5a5f: wc = 0x6dd6; break;
        case 0x5a60: wc = 0x7847; break;
        case 0x5a61: wc = 0x78e0; break;
        case 0x5a62: wc = 0x8132; break;
        case 0x5a63: wc = 0x88ca; break;
        case 0x5a64: wc = 0x892d; break;
        case 0x5a65: wc = 0x8b4a; break;
        case 0x5a66: wc = 0x9403; break;
        case 0x5a67: wc = 0x6abd; break;
        case 0x5a68: wc = 0x7373; break;
        case 0x5a69: wc = 0x7fba; break;
        case 0x5a6a: wc = 0x8028; break;
        case 0x5a6b: wc = 0x8b68; break;
        case 0x5a6c: wc = 0x9392; break;
        case 0x5a6d: wc = 0x36cb; break;
        case 0x5a6e: wc = 0x5436; break;
        case 0x5a6f: wc = 0x6290; break;
        case 0x5a70: wc = 0x80ad; break;
        case 0x5a71: wc = 0x599e; break;
        case 0x5a72: wc = 0x5ff8; break;
        case 0x5a73: wc = 0x626d; break;
        case 0x5a74: wc = 0x7084; break;
        case 0x5a75: wc = 0x72c3; break;
        case 0x5a76: wc = 0x7c88; break;
        case 0x5a77: wc = 0x83a5; break;
        case 0x5a78: wc = 0x9215; break;
        case 0x5a79: wc = 0x6067; break;
        case 0x5a7a: wc = 0x6712; break;
        case 0x5a7b: wc = 0x45a1; break;
        case 0x5a7c: wc = 0x8844; break;
        case 0x5a7d: wc = 0x83bb; break;
        case 0x5a7e: wc = 0x4f60; break;
        case 0x5a7f: wc = 0x5037; break;
        case 0x5a80: wc = 0x5462; break;
        case 0x5a81: wc = 0x576d; break;
        case 0x5a82: wc = 0x57ff; break;
        case 0x5a83: wc = 0x59b3; break;
        case 0x5a84: wc = 0x59ae; break;
        case 0x5a85: wc = 0x5c54; break;
        case 0x5a86: wc = 0x6029; break;
        case 0x5a87: wc = 0x60a8; break;
        case 0x5a88: wc = 0x62b3; break;
        case 0x5a89: wc = 0x65ce; break;
        case 0x5a8a: wc = 0x67c5; break;
        case 0x5a8b: wc = 0x6ab7; break;
        case 0x5a8c: wc = 0x6fd4; break;
        case 0x5a8d: wc = 0x72d4; break;
        case 0x5a8e: wc = 0x7962; break;
        case 0x5a8f: wc = 0x79dc; break;
        case 0x5a90: wc = 0x81a9; break;
        case 0x5a91: wc = 0x81e1; break;
        case 0x5a92: wc = 0x82e8; break;
        case 0x5a93: wc = 0x8ddc; break;
        case 0x5a94: wc = 0x8fe1; break;
        case 0x5a95: wc = 0x922e; break;
        case 0x5a96: wc = 0x9448; break;
        case 0x5a97: wc = 0x96ac; break;
        case 0x5a98: wc = 0x999c; break;
        case 0x5a99: wc = 0x5adf; break;
        case 0x5a9a: wc = 0x6135; break;
        case 0x5a9b: wc = 0x6426; break;
        case 0x5a9c: wc = 0x643b; break;
        case 0x5a9d: wc = 0x7cd1; break;
        case 0x5a9e: wc = 0x460c; break;
        case 0x5a9f: wc = 0x6635; break;
        case 0x5aa0: wc = 0x66b1; break;
        case 0x5aa1: wc = 0x62f0; break;
        case 0x5aa2: wc = 0x8ee0; break;
        case 0x5aa3: wc = 0x56e1; break;
        case 0x5aa4: wc = 0x55f2; break;
        case 0x5aa5: wc = 0x7239; break;
        case 0x5aa6: wc = 0x8324; break;
        case 0x5aa7: wc = 0x89f0; break;
        case 0x5aa8: wc = 0x8de2; break;
        case 0x5aa9: wc = 0x5073; break;
        case 0x5aaa: wc = 0x5210; break;
        case 0x5aab: wc = 0x526c; break;
        case 0x5aac: wc = 0x52ef; break;
        case 0x5aad: wc = 0x5330; break;
        case 0x5aae: wc = 0x550c; break;
        case 0x5aaf: wc = 0x56e3; break;
        case 0x5ab0: wc = 0x5845; break;
        case 0x5ab1: wc = 0x5a8f; break;
        case 0x5ab2: wc = 0x6171; break;
        case 0x5ab3: wc = 0x62c5; break;
        case 0x5ab4: wc = 0x6476; break;
        case 0x5ab5: wc = 0x3adc; break;
        case 0x5ab6: wc = 0x67e6; break;
        case 0x5ab7: wc = 0x6934; break;
        case 0x5ab8: wc = 0x69eb; break;
        case 0x5ab9: wc = 0x6bc8; break;
        case 0x5aba: wc = 0x6f19; break;
        case 0x5abb: wc = 0x6f6c; break;
        case 0x5abc: wc = 0x6fb6; break;
        case 0x5abd: wc = 0x7153; break;
        case 0x5abe: wc = 0x7145; break;
        case 0x5abf: wc = 0x732f; break;
        case 0x5ac0: wc = 0x73ac; break;
        case 0x5ac1: wc = 0x7649; break;
        case 0x5ac2: wc = 0x7803; break;
        case 0x5ac3: wc = 0x78ab; break;
        case 0x5ac4: wc = 0x7c6a; break;
        case 0x5ac5: wc = 0x7cf0; break;
        case 0x5ac6: wc = 0x7e75; break;
        case 0x5ac7: wc = 0x8011; break;
        case 0x5ac8: wc = 0x80c6; break;
        case 0x5ac9: wc = 0x8176; break;
        case 0x5aca: wc = 0x846e; break;
        case 0x5acb: wc = 0x26d53; break;
        case 0x5acc: wc = 0x8711; break;
        case 0x5acd: wc = 0x8916; break;
        case 0x5ace: wc = 0x894c; break;
        case 0x5acf: wc = 0x8962; break;
        case 0x5ad0: wc = 0x89db; break;
        case 0x5ad1: wc = 0x8c92; break;
        case 0x5ad2: wc = 0x8c9a; break;
        case 0x5ad3: wc = 0x8e39; break;
        case 0x5ad4: wc = 0x482a; break;
        case 0x5ad5: wc = 0x926d; break;
        case 0x5ad6: wc = 0x9374; break;
        case 0x5ad7: wc = 0x93c4; break;
        case 0x5ad8: wc = 0x977c; break;
        case 0x5ad9: wc = 0x9a59; break;
        case 0x5ada: wc = 0x9d20; break;
        case 0x5adb: wc = 0x9da8; break;
        case 0x5adc: wc = 0x9dd2; break;
        case 0x5add: wc = 0x9de4; break;
        case 0x5ade: wc = 0x9dfb; break;
        case 0x5adf: wc = 0x3493; break;
        case 0x5ae0: wc = 0x547e; break;
        case 0x5ae1: wc = 0x5660; break;
        case 0x5ae2: wc = 0x58b6; break;
        case 0x5ae3: wc = 0x59b2; break;
        case 0x5ae4: wc = 0x601b; break;
        case 0x5ae5: wc = 0x6a7d; break;
        case 0x5ae6: wc = 0x709f; break;
        case 0x5ae7: wc = 0x71f5; break;
        case 0x5ae8: wc = 0x72da; break;
        case 0x5ae9: wc = 0x3e9a; break;
        case 0x5aea: wc = 0x7b2a; break;
        case 0x5aeb: wc = 0x7e68; break;
        case 0x5aec: wc = 0x7f8d; break;
        case 0x5aed: wc = 0x8598; break;
        case 0x5aee: wc = 0x8e82; break;
        case 0x5aef: wc = 0x943d; break;
        case 0x5af0: wc = 0x95e5; break;
        case 0x5af1: wc = 0x97c3; break;
        case 0x5af2: wc = 0x5013; break;
        case 0x5af3: wc = 0x5061; break;
        case 0x5af4: wc = 0x510b; break;
        case 0x5af5: wc = 0x5557; break;
        case 0x5af6: wc = 0x557f; break;
        case 0x5af7: wc = 0x563e; break;
        case 0x5af8: wc = 0x5649; break;
        case 0x5af9: wc = 0x56aa; break;
        case 0x5afa: wc = 0x57ee; break;
        case 0x5afb: wc = 0x58dc; break;
        case 0x5afc: wc = 0x59cf; break;
        case 0x5afd: wc = 0x5a52; break;
        case 0x5afe: wc = 0x5e0e; break;
        case 0x5aff: wc = 0x60d4; break;
        case 0x5b00: wc = 0x619b; break;
        case 0x5b01: wc = 0x66cb; break;
        case 0x5b02: wc = 0x6a5d; break;
        case 0x5b03: wc = 0x6bef; break;
        case 0x5b04: wc = 0x6c2e; break;
        case 0x5b05: wc = 0x3d02; break;
        case 0x5b06: wc = 0x6fb8; break;
        case 0x5b07: wc = 0x7514; break;
        case 0x5b08: wc = 0x765a; break;
        case 0x5b09: wc = 0x79ab; break;
        case 0x5b0a: wc = 0x7a9e; break;
        case 0x5b0b: wc = 0x7d1e; break;
        case 0x5b0c: wc = 0x7dc2; break;
        case 0x5b0d: wc = 0x7f48; break;
        case 0x5b0e: wc = 0x7f4e; break;
        case 0x5b0f: wc = 0x803c; break;
        case 0x5b10: wc = 0x8078; break;
        case 0x5b11: wc = 0x8211; break;
        case 0x5b12: wc = 0x8215; break;
        case 0x5b13: wc = 0x840f; break;
        case 0x5b14: wc = 0x83fc; break;
        case 0x5b15: wc = 0x859d; break;
        case 0x5b16: wc = 0x85eb; break;
        case 0x5b17: wc = 0x87eb; break;
        case 0x5b18: wc = 0x8874; break;
        case 0x5b19: wc = 0x8d09; break;
        case 0x5b1a: wc = 0x90ef; break;
        case 0x5b1b: wc = 0x91b0; break;
        case 0x5b1c: wc = 0x972e; break;
        case 0x5b1d: wc = 0x9843; break;
        case 0x5b1e: wc = 0x9924; break;
        case 0x5b1f: wc = 0x9a54; break;
        case 0x5b20: wc = 0x9ae7; break;
        case 0x5b21: wc = 0x9ed5; break;
        case 0x5b22: wc = 0x9eee; break;
        case 0x5b23: wc = 0x9ef5; break;
        case 0x5b24: wc = 0x5273; break;
        case 0x5b25: wc = 0x5312; break;
        case 0x5b26: wc = 0x3b7c; break;
        case 0x5b27: wc = 0x6999; break;
        case 0x5b28: wc = 0x3bda; break;
        case 0x5b29: wc = 0x6dbe; break;
        case 0x5b2a: wc = 0x6e9a; break;
        case 0x5b2b: wc = 0x6fcc; break;
        case 0x5b2c: wc = 0x7563; break;
        case 0x5b2d: wc = 0x441b; break;
        case 0x5b2e: wc = 0x8345; break;
        case 0x5b2f: wc = 0x8921; break;
        case 0x5b30: wc = 0x891f; break;
        case 0x5b31: wc = 0x8abb; break;
        case 0x5b32: wc = 0x8e4b; break;
        case 0x5b33: wc = 0x8e79; break;
        case 0x5b34: wc = 0x9389; break;
        case 0x5b35: wc = 0x4d6c; break;
        case 0x5b36: wc = 0x5018; break;
        case 0x5b37: wc = 0x5105; break;
        case 0x5b38: wc = 0x513b; break;
        case 0x5b39: wc = 0x515a; break;
        case 0x5b3a: wc = 0x5621; break;
        case 0x5b3b: wc = 0x5647; break;
        case 0x5b3c: wc = 0x5679; break;
        case 0x5b3d: wc = 0x58cb; break;
        case 0x5b3e: wc = 0x5f53; break;
        case 0x5b3f: wc = 0x5d63; break;
        case 0x5b40: wc = 0x6203; break;
        case 0x5b41: wc = 0x642a; break;
        case 0x5b42: wc = 0x64cb; break;
        case 0x5b43: wc = 0x6529; break;
        case 0x5b44: wc = 0x66ed; break;
        case 0x5b45: wc = 0x69b6; break;
        case 0x5b46: wc = 0x6a94; break;
        case 0x5b47: wc = 0x6b13; break;
        case 0x5b48: wc = 0x6e8f; break;
        case 0x5b49: wc = 0x6f1f; break;
        case 0x5b4a: wc = 0x6fa2; break;
        case 0x5b4b: wc = 0x7059; break;
        case 0x5b4c: wc = 0x717b; break;
        case 0x5b4d: wc = 0x7223; break;
        case 0x5b4e: wc = 0x746d; break;
        case 0x5b4f: wc = 0x74ab; break;
        case 0x5b50: wc = 0x74fd; break;
        case 0x5b51: wc = 0x77a0; break;
        case 0x5b52: wc = 0x77d8; break;
        case 0x5b53: wc = 0x78c4; break;
        case 0x5b54: wc = 0x7911; break;
        case 0x5b55: wc = 0x7c39; break;
        case 0x5b56: wc = 0x819b; break;
        case 0x5b57: wc = 0x8261; break;
        case 0x5b58: wc = 0x84ce; break;
        case 0x5b59: wc = 0x454b; break;
        case 0x5b5a: wc = 0x8797; break;
        case 0x5b5b: wc = 0x87f7; break;
        case 0x5b5c: wc = 0x8960; break;
        case 0x5b5d: wc = 0x8b9c; break;
        case 0x5b5e: wc = 0x8d6f; break;
        case 0x5b5f: wc = 0x8e5a; break;
        case 0x5b60: wc = 0x8eba; break;
        case 0x5b61: wc = 0x91a3; break;
        case 0x5b62: wc = 0x9395; break;
        case 0x5b63: wc = 0x93b2; break;
        case 0x5b64: wc = 0x93dc; break;
        case 0x5b65: wc = 0x943a; break;
        case 0x5b66: wc = 0x9482; break;
        case 0x5b67: wc = 0x95db; break;
        case 0x5b68: wc = 0x95e3; break;
        case 0x5b69: wc = 0x97ba; break;
        case 0x5b6a: wc = 0x9933; break;
        case 0x5b6b: wc = 0x9939; break;
        case 0x5b6c: wc = 0x9db6; break;
        case 0x5b6d: wc = 0x9f1e; break;
        case 0x5b6e: wc = 0x3432; break;
        case 0x5b6f: wc = 0x5113; break;
        case 0x5b70: wc = 0x5b2f; break;
        case 0x5b71: wc = 0x5e12; break;
        case 0x5b72: wc = 0x5ed7; break;
        case 0x5b73: wc = 0x38d5; break;
        case 0x5b74: wc = 0x619d; break;
        case 0x5b75: wc = 0x61df; break;
        case 0x5b76: wc = 0x65f2; break;
        case 0x5b77: wc = 0x67cb; break;
        case 0x5b78: wc = 0x6aaf; break;
        case 0x5b79: wc = 0x6c4f; break;
        case 0x5b7a: wc = 0x23d18; break;
        case 0x5b7b: wc = 0x6fe7; break;
        case 0x5b7c: wc = 0x7029; break;
        case 0x5b7d: wc = 0x703b; break;
        case 0x5b7e: wc = 0x7447; break;
        case 0x5b7f: wc = 0x3fc3; break;
        case 0x5b80: wc = 0x7893; break;
        case 0x5b81: wc = 0x794b; break;
        case 0x5b82: wc = 0x7c49; break;
        case 0x5b83: wc = 0x825c; break;
        case 0x5b84: wc = 0x85b9; break;
        case 0x5b85: wc = 0x85b1; break;
        case 0x5b86: wc = 0x8968; break;
        case 0x5b87: wc = 0x8976; break;
        case 0x5b88: wc = 0x8b48; break;
        case 0x5b89: wc = 0x8e5b; break;
        case 0x5b8a: wc = 0x8ed1; break;
        case 0x5b8b: wc = 0x8f5b; break;
        case 0x5b8c: wc = 0x9413; break;
        case 0x5b8d: wc = 0x9b98; break;
        case 0x5b8e: wc = 0x3516; break;
        case 0x5b8f: wc = 0x3941; break;
        case 0x5b90: wc = 0x3bd6; break;
        case 0x5b91: wc = 0x3830; break;
        case 0x5b92: wc = 0x3514; break;
        case 0x5b93: wc = 0x5202; break;
        case 0x5b94: wc = 0x530b; break;
        case 0x5b95: wc = 0x53e8; break;
        case 0x5b96: wc = 0x54b7; break;
        case 0x5b97: wc = 0x550b; break;
        case 0x5b98: wc = 0x5555; break;
        case 0x5b99: wc = 0x55a5; break;
        case 0x5b9a: wc = 0x561f; break;
        case 0x5b9b: wc = 0x5675; break;
        case 0x5b9c: wc = 0x56f3; break;
        case 0x5b9d: wc = 0x58d4; break;
        case 0x5b9e: wc = 0x5932; break;
        case 0x5b9f: wc = 0x5acd; break;
        case 0x5ba0: wc = 0x5cf9; break;
        case 0x5ba1: wc = 0x5d5e; break;
        case 0x5ba2: wc = 0x5d80; break;
        case 0x5ba3: wc = 0x5e3e; break;
        case 0x5ba4: wc = 0x5e4d; break;
        case 0x5ba5: wc = 0x5edc; break;
        case 0x5ba6: wc = 0x5f22; break;
        case 0x5ba7: wc = 0x5fc9; break;
        case 0x5ba8: wc = 0x6146; break;
        case 0x5ba9: wc = 0x6348; break;
        case 0x5baa: wc = 0x638f; break;
        case 0x5bab: wc = 0x636f; break;
        case 0x5bac: wc = 0x642f; break;
        case 0x5bad: wc = 0x64e3; break;
        case 0x5bae: wc = 0x688c; break;
        case 0x5baf: wc = 0x69c4; break;
        case 0x5bb0: wc = 0x6aa4; break;
        case 0x5bb1: wc = 0x6aae; break;
        case 0x5bb2: wc = 0x6c18; break;
        case 0x5bb3: wc = 0x6d82; break;
        case 0x5bb4: wc = 0x6f73; break;
        case 0x5bb5: wc = 0x7196; break;
        case 0x5bb6: wc = 0x74d9; break;
        case 0x5bb7: wc = 0x760f; break;
        case 0x5bb8: wc = 0x7979; break;
        case 0x5bb9: wc = 0x7982; break;
        case 0x5bba: wc = 0x413b; break;
        case 0x5bbb: wc = 0x7a0c; break;
        case 0x5bbc: wc = 0x4183; break;
        case 0x5bbd: wc = 0x7b61; break;
        case 0x5bbe: wc = 0x7daf; break;
        case 0x5bbf: wc = 0x7e1a; break;
        case 0x5bc0: wc = 0x7fff; break;
        case 0x5bc1: wc = 0x8220; break;
        case 0x5bc2: wc = 0x446c; break;
        case 0x5bc3: wc = 0x4472; break;
        case 0x5bc4: wc = 0x837c; break;
        case 0x5bc5: wc = 0x83ff; break;
        case 0x5bc6: wc = 0x84a4; break;
        case 0x5bc7: wc = 0x866d; break;
        case 0x5bc8: wc = 0x872a; break;
        case 0x5bc9: wc = 0x885c; break;
        case 0x5bca: wc = 0x8b1f; break;
        case 0x5bcb: wc = 0x8dff; break;
        case 0x5bcc: wc = 0x9174; break;
        case 0x5bcd: wc = 0x9184; break;
        case 0x5bce: wc = 0x918f; break;
        case 0x5bcf: wc = 0x92fe; break;
        case 0x5bd0: wc = 0x95cd; break;
        case 0x5bd1: wc = 0x9780; break;
        case 0x5bd2: wc = 0x9789; break;
        case 0x5bd3: wc = 0x97b1; break;
        case 0x5bd4: wc = 0x9940; break;
        case 0x5bd5: wc = 0x9955; break;
        case 0x5bd6: wc = 0x99fc; break;
        case 0x5bd7: wc = 0x9a0a; break;
        case 0x5bd8: wc = 0x9b5b; break;
        case 0x5bd9: wc = 0x9ced; break;
        case 0x5bda: wc = 0x9d4c; break;
        case 0x5bdb: wc = 0x9dcb; break;
        case 0x5bdc: wc = 0x9df5; break;
        case 0x5bdd: wc = 0x4d5a; break;
        case 0x5bde: wc = 0x9f17; break;
        case 0x5bdf: wc = 0x51df; break;
        case 0x5be0: wc = 0x5335; break;
        case 0x5be1: wc = 0x5b3b; break;
        case 0x5be2: wc = 0x6add; break;
        case 0x5be3: wc = 0x6bb0; break;
        case 0x5be4: wc = 0x6d8b; break;
        case 0x5be5: wc = 0x76be; break;
        case 0x5be6: wc = 0x78a1; break;
        case 0x5be7: wc = 0x8773; break;
        case 0x5be8: wc = 0x88fb; break;
        case 0x5be9: wc = 0x8b9f; break;
        case 0x5bea: wc = 0x8d15; break;
        case 0x5beb: wc = 0x92f5; break;
        case 0x5bec: wc = 0x9316; break;
        case 0x5bed: wc = 0x97c7; break;
        case 0x5bee: wc = 0x97e3; break;
        case 0x5bef: wc = 0x97e5; break;
        case 0x5bf0: wc = 0x9d5a; break;
        case 0x5bf1: wc = 0x9ef7; break;
        case 0x5bf2: wc = 0x4f05; break;
        case 0x5bf3: wc = 0x56e4; break;
        case 0x5bf4: wc = 0x5d38; break;
        case 0x5bf5: wc = 0x5e89; break;
        case 0x5bf6: wc = 0x5f34; break;
        case 0x5bf7: wc = 0x5ff3; break;
        case 0x5bf8: wc = 0x6265; break;
        case 0x5bf9: wc = 0x627d; break;
        case 0x5bfa: wc = 0x6489; break;
        case 0x5bfb: wc = 0x6a54; break;
        case 0x5bfc: wc = 0x6f61; break;
        case 0x5bfd: wc = 0x7096; break;
        case 0x5bfe: wc = 0x729c; break;
        case 0x5bff: wc = 0x3e60; break;
        case 0x5c00: wc = 0x7364; break;
        case 0x5c01: wc = 0x7818; break;
        case 0x5c02: wc = 0x40e6; break;
        case 0x5c03: wc = 0x8e7e; break;
        case 0x5c04: wc = 0x8e89; break;
        case 0x5c05: wc = 0x8ed8; break;
        case 0x5c06: wc = 0x98e9; break;
        case 0x5c07: wc = 0x9a50; break;
        case 0x5c08: wc = 0x9b68; break;
        case 0x5c09: wc = 0x5484; break;
        case 0x5c0a: wc = 0x5817; break;
        case 0x5c0b: wc = 0x6022; break;
        case 0x5c0c: wc = 0x67ee; break;
        case 0x5c0d: wc = 0x6e65; break;
        case 0x5c0e: wc = 0x816f; break;
        case 0x5c0f: wc = 0x8456; break;
        case 0x5c10: wc = 0x934e; break;
        case 0x5c11: wc = 0x9d9f; break;
        case 0x5c12: wc = 0x9f35; break;
        case 0x5c13: wc = 0x360f; break;
        case 0x5c14: wc = 0x4f5f; break;
        case 0x5c15: wc = 0x4f97; break;
        case 0x5c16: wc = 0x5032; break;
        case 0x5c17: wc = 0x5045; break;
        case 0x5c18: wc = 0x50cd; break;
        case 0x5c19: wc = 0x50ee; break;
        case 0x5c1a: wc = 0x549a; break;
        case 0x5c1b: wc = 0x54c3; break;
        case 0x5c1c: wc = 0x578c; break;
        case 0x5c1d: wc = 0x57ec; break;
        case 0x5c1e: wc = 0x58a5; break;
        case 0x5c1f: wc = 0x59db; break;
        case 0x5c20: wc = 0x5b1e; break;
        case 0x5c21: wc = 0x5cd2; break;
        case 0x5c22: wc = 0x5d20; break;
        case 0x5c23: wc = 0x3809; break;
        case 0x5c24: wc = 0x5f64; break;
        case 0x5c25: wc = 0x61c2; break;
        case 0x5c26: wc = 0x6219; break;
        case 0x5c27: wc = 0x630f; break;
        case 0x5c28: wc = 0x3a02; break;
        case 0x5c29: wc = 0x66c8; break;
        case 0x5c2a: wc = 0x6723; break;
        case 0x5c2b: wc = 0x6a66; break;
        case 0x5c2c: wc = 0x6c03; break;
        case 0x5c2d: wc = 0x6c21; break;
        case 0x5c2e: wc = 0x6d75; break;
        case 0x5c2f: wc = 0x6db7; break;
        case 0x5c30: wc = 0x6e69; break;
        case 0x5c31: wc = 0x70b5; break;
        case 0x5c32: wc = 0x70d4; break;
        case 0x5c33: wc = 0x71d1; break;
        case 0x5c34: wc = 0x729d; break;
        case 0x5c35: wc = 0x72ea; break;
        case 0x5c36: wc = 0x735e; break;
        case 0x5c37: wc = 0x783c; break;
        case 0x5c38: wc = 0x7850; break;
        case 0x5c39: wc = 0x79f1; break;
        case 0x5c3a: wc = 0x7a5c; break;
        case 0x5c3b: wc = 0x7ca1; break;
        case 0x5c3c: wc = 0x7d67; break;
        case 0x5c3d: wc = 0x7f7f; break;
        case 0x5c3e: wc = 0x81a7; break;
        case 0x5c3f: wc = 0x825f; break;
        case 0x5c40: wc = 0x82f3; break;
        case 0x5c41: wc = 0x833c; break;
        case 0x5c42: wc = 0x83c4; break;
        case 0x5c43: wc = 0x856b; break;
        case 0x5c44: wc = 0x8740; break;
        case 0x5c45: wc = 0x8855; break;
        case 0x5c46: wc = 0x8a77; break;
        case 0x5c47: wc = 0x8acc; break;
        case 0x5c48: wc = 0x8d68; break;
        case 0x5c49: wc = 0x8ff5; break;
        case 0x5c4a: wc = 0x916e; break;
        case 0x5c4b: wc = 0x9256; break;
        case 0x5c4c: wc = 0x932c; break;
        case 0x5c4d: wc = 0x9718; break;
        case 0x5c4e: wc = 0x9907; break;
        case 0x5c4f: wc = 0x99e7; break;
        case 0x5c50: wc = 0x9b97; break;
        case 0x5c51: wc = 0x9ba6; break;
        case 0x5c52: wc = 0x9bdf; break;
        case 0x5c53: wc = 0x9d87; break;
        case 0x5c54: wc = 0x9f15; break;
        case 0x5c55: wc = 0x4ea0; break;
        case 0x5c56: wc = 0x5245; break;
        case 0x5c57: wc = 0x5517; break;
        case 0x5c58: wc = 0x6296; break;
        case 0x5c59: wc = 0x6568; break;
        case 0x5c5a: wc = 0x6581; break;
        case 0x5c5b: wc = 0x68aa; break;
        case 0x5c5c: wc = 0x6d62; break;
        case 0x5c5d: wc = 0x3eb6; break;
        case 0x5c5e: wc = 0x7bfc; break;
        case 0x5c5f: wc = 0x7d0f; break;
        case 0x5c60: wc = 0x809a; break;
        case 0x5c61: wc = 0x8130; break;
        case 0x5c62: wc = 0x8370; break;
        case 0x5c63: wc = 0x8538; break;
        case 0x5c64: wc = 0x86aa; break;
        case 0x5c65: wc = 0x8839; break;
        case 0x5c66: wc = 0x488f; break;
        case 0x5c67: wc = 0x90d6; break;
        case 0x5c68: wc = 0x9158; break;
        case 0x5c69: wc = 0x9204; break;
        case 0x5c6a: wc = 0x92c0; break;
        case 0x5c6b: wc = 0x9597; break;
        case 0x5c6c: wc = 0x9627; break;
        case 0x5c6d: wc = 0x9661; break;
        case 0x5c6e: wc = 0x9916; break;
        case 0x5c6f: wc = 0x9b25; break;
        case 0x5c70: wc = 0x3ab2; break;
        case 0x5c71: wc = 0x5428; break;
        case 0x5c72: wc = 0x5749; break;
        case 0x5c73: wc = 0x7a80; break;
        case 0x5c74: wc = 0x4733; break;
        case 0x5c75: wc = 0x8fcd; break;
        case 0x5c76: wc = 0x3408; break;
        case 0x5c77: wc = 0x4e67; break;
        case 0x5c78: wc = 0x3419; break;
        case 0x5c79: wc = 0x358d; break;
        case 0x5c7a: wc = 0x3ab3; break;
        case 0x5c7b: wc = 0x561a; break;
        case 0x5c7c: wc = 0x6dc2; break;
        case 0x5c7d: wc = 0x9340; break;
        case 0x5c7e: wc = 0x51f3; break;
        case 0x5c7f: wc = 0x5654; break;
        case 0x5c80: wc = 0x58b1; break;
        case 0x5c81: wc = 0x5b01; break;
        case 0x5c82: wc = 0x6225; break;
        case 0x5c83: wc = 0x3a50; break;
        case 0x5c84: wc = 0x6ac8; break;
        case 0x5c85: wc = 0x3caa; break;
        case 0x5c86: wc = 0x23c06; break;
        case 0x5c87: wc = 0x6ed5; break;
        case 0x5c88: wc = 0x7492; break;
        case 0x5c89: wc = 0x78f4; break;
        case 0x5c8a: wc = 0x7c26; break;
        case 0x5c8b: wc = 0x7c50; break;
        case 0x5c8c: wc = 0x7e22; break;
        case 0x5c8d: wc = 0x81af; break;
        case 0x5c8e: wc = 0x8645; break;
        case 0x5c8f: wc = 0x87a3; break;
        case 0x5c90: wc = 0x8c4b; break;
        case 0x5c91: wc = 0x8e6c; break;
        case 0x5c92: wc = 0x9086; break;
        case 0x5c93: wc = 0x9419; break;
        case 0x5c94: wc = 0x972f; break;
        case 0x5c95: wc = 0x99e6; break;
        case 0x5c96: wc = 0x4cfe; break;
        case 0x5c97: wc = 0x35b0; break;
        case 0x5c98: wc = 0x35af; break;
        case 0x5c99: wc = 0x340d; break;
        case 0x5c9a: wc = 0x502e; break;
        case 0x5c9b: wc = 0x5138; break;
        case 0x5c9c: wc = 0x5246; break;
        case 0x5c9d: wc = 0x56c9; break;
        case 0x5c9e: wc = 0x645e; break;
        case 0x5c9f: wc = 0x652d; break;
        case 0x5ca0: wc = 0x65ef; break;
        case 0x5ca1: wc = 0x6b0f; break;
        case 0x5ca2: wc = 0x7055; break;
        case 0x5ca3: wc = 0x7380; break;
        case 0x5ca4: wc = 0x7630; break;
        case 0x5ca5: wc = 0x7822; break;
        case 0x5ca6: wc = 0x7c6e; break;
        case 0x5ca7: wc = 0x26181; break;
        case 0x5ca8: wc = 0x7e99; break;
        case 0x5ca9: wc = 0x8161; break;
        case 0x5caa: wc = 0x81dd; break;
        case 0x5cab: wc = 0x84cf; break;
        case 0x5cac: wc = 0x8803; break;
        case 0x5cad: wc = 0x89b6; break;
        case 0x5cae: wc = 0x8eb6; break;
        case 0x5caf: wc = 0x93af; break;
        case 0x5cb0: wc = 0x93cd; break;
        case 0x5cb1: wc = 0x947c; break;
        case 0x5cb2: wc = 0x9a3e; break;
        case 0x5cb3: wc = 0x9e01; break;
        case 0x5cb4: wc = 0x5cc8; break;
        case 0x5cb5: wc = 0x6ffc; break;
        case 0x5cb6: wc = 0x7296; break;
        case 0x5cb7: wc = 0x784c; break;
        case 0x5cb8: wc = 0x927b; break;
        case 0x5cb9: wc = 0x96d2; break;
        case 0x5cba: wc = 0x9ba5; break;
        case 0x5cbb: wc = 0x571d; break;
        case 0x5cbc: wc = 0x571e; break;
        case 0x5cbd: wc = 0x5b3e; break;
        case 0x5cbe: wc = 0x5e71; break;
        case 0x5cbf: wc = 0x6514; break;
        case 0x5cc0: wc = 0x6595; break;
        case 0x5cc1: wc = 0x66eb; break;
        case 0x5cc2: wc = 0x6b17; break;
        case 0x5cc3: wc = 0x7061; break;
        case 0x5cc4: wc = 0x7064; break;
        case 0x5cc5: wc = 0x74d3; break;
        case 0x5cc6: wc = 0x7675; break;
        case 0x5cc7: wc = 0x7c63; break;
        case 0x5cc8: wc = 0x7cf7; break;
        case 0x5cc9: wc = 0x7f89; break;
        case 0x5cca: wc = 0x8974; break;
        case 0x5ccb: wc = 0x8b4b; break;
        case 0x5ccc: wc = 0x8b95; break;
        case 0x5ccd: wc = 0x8e9d; break;
        case 0x5cce: wc = 0x946d; break;
        case 0x5ccf: wc = 0x947e; break;
        case 0x5cd0: wc = 0x95cc; break;
        case 0x5cd1: wc = 0x5783; break;
        case 0x5cd2: wc = 0x57d2; break;
        case 0x5cd3: wc = 0x634b; break;
        case 0x5cd4: wc = 0x63e6; break;
        case 0x5cd5: wc = 0x63e7; break;
        case 0x5cd6: wc = 0x694b; break;
        case 0x5cd7: wc = 0x760c; break;
        case 0x5cd8: wc = 0x9b0e; break;
        case 0x5cd9: wc = 0x9bfb; break;
        case 0x5cda: wc = 0x3463; break;
        case 0x5cdb: wc = 0x5549; break;
        case 0x5cdc: wc = 0x5682; break;
        case 0x5cdd: wc = 0x56d5; break;
        case 0x5cde: wc = 0x58c8; break;
        case 0x5cdf: wc = 0x5a6a; break;
        case 0x5ce0: wc = 0x372e; break;
        case 0x5ce1: wc = 0x6d68; break;
        case 0x5ce2: wc = 0x6f24; break;
        case 0x5ce3: wc = 0x7046; break;
        case 0x5ce4: wc = 0x7060; break;
        case 0x5ce5: wc = 0x7201; break;
        case 0x5ce6: wc = 0x7226; break;
        case 0x5ce7: wc = 0x862b; break;
        case 0x5ce8: wc = 0x9872; break;
        case 0x5ce9: wc = 0x5566; break;
        case 0x5cea: wc = 0x641a; break;
        case 0x5ceb: wc = 0x67c6; break;
        case 0x5cec: wc = 0x78d6; break;
        case 0x5ced: wc = 0x7fcb; break;
        case 0x5cee: wc = 0x83c8; break;
        case 0x5cef: wc = 0x908b; break;
        case 0x5cf0: wc = 0x945e; break;
        case 0x5cf1: wc = 0x97a1; break;
        case 0x5cf2: wc = 0x5577; break;
        case 0x5cf3: wc = 0x57cc; break;
        case 0x5cf4: wc = 0x5871; break;
        case 0x5cf5: wc = 0x5acf; break;
        case 0x5cf6: wc = 0x5d00; break;
        case 0x5cf7: wc = 0x3af0; break;
        case 0x5cf8: wc = 0x6879; break;
        case 0x5cf9: wc = 0x6994; break;
        case 0x5cfa: wc = 0x6a03; break;
        case 0x5cfb: wc = 0x6b34; break;
        case 0x5cfc: wc = 0x70fa; break;
        case 0x5cfd: wc = 0x7860; break;
        case 0x5cfe: wc = 0x7a02; break;
        case 0x5cff: wc = 0x41a1; break;
        case 0x5d00: wc = 0x7b64; break;
        case 0x5d01: wc = 0x8246; break;
        case 0x5d02: wc = 0x83a8; break;
        case 0x5d03: wc = 0x8497; break;
        case 0x5d04: wc = 0x870b; break;
        case 0x5d05: wc = 0x8a8f; break;
        case 0x5d06: wc = 0x92c3; break;
        case 0x5d07: wc = 0x95ac; break;
        case 0x5d08: wc = 0x99fa; break;
        case 0x5d09: wc = 0x5008; break;
        case 0x5d0a: wc = 0x513d; break;
        case 0x5d0b: wc = 0x52d1; break;
        case 0x5d0c: wc = 0x553b; break;
        case 0x5d0d: wc = 0x3693; break;
        case 0x5d0e: wc = 0x5eb2; break;
        case 0x5d0f: wc = 0x68f6; break;
        case 0x5d10: wc = 0x6df6; break;
        case 0x5d11: wc = 0x775e; break;
        case 0x5d12: wc = 0x7b82; break;
        case 0x5d13: wc = 0x90f2; break;
        case 0x5d14: wc = 0x9338; break;
        case 0x5d15: wc = 0x9a0b; break;
        case 0x5d16: wc = 0x9be0; break;
        case 0x5d17: wc = 0x9d86; break;
        case 0x5d18: wc = 0x6482; break;
        case 0x5d19: wc = 0x7567; break;
        case 0x5d1a: wc = 0x4fcd; break;
        case 0x5d1b: wc = 0x3500; break;
        case 0x5d1c: wc = 0x5562; break;
        case 0x5d1d: wc = 0x55a8; break;
        case 0x5d1e: wc = 0x589a; break;
        case 0x5d1f: wc = 0x60a2; break;
        case 0x5d20: wc = 0x60ca; break;
        case 0x5d21: wc = 0x667e; break;
        case 0x5d22: wc = 0x690b; break;
        case 0x5d23: wc = 0x6dbc; break;
        case 0x5d24: wc = 0x6e78; break;
        case 0x5d25: wc = 0x7177; break;
        case 0x5d26: wc = 0x3e41; break;
        case 0x5d27: wc = 0x3f97; break;
        case 0x5d28: wc = 0x7c17; break;
        case 0x5d29: wc = 0x7da1; break;
        case 0x5d2a: wc = 0x7dc9; break;
        case 0x5d2b: wc = 0x873d; break;
        case 0x5d2c: wc = 0x88f2; break;
        case 0x5d2d: wc = 0x8e09; break;
        case 0x5d2e: wc = 0x8f2c; break;
        case 0x5d2f: wc = 0x9b4e; break;
        case 0x5d30: wc = 0x5122; break;
        case 0x5d31: wc = 0x52f4; break;
        case 0x5d32: wc = 0x53b2; break;
        case 0x5d33: wc = 0x5533; break;
        case 0x5d34: wc = 0x5ef2; break;
        case 0x5d35: wc = 0x60b7; break;
        case 0x5d36: wc = 0x6314; break;
        case 0x5d37: wc = 0x6526; break;
        case 0x5d38: wc = 0x66de; break;
        case 0x5d39: wc = 0x68a0; break;
        case 0x5d3a: wc = 0x68d9; break;
        case 0x5d3b: wc = 0x6b10; break;
        case 0x5d3c: wc = 0x6cb4; break;
        case 0x5d3d: wc = 0x7112; break;
        case 0x5d3e: wc = 0x7208; break;
        case 0x5d3f: wc = 0x74c8; break;
        case 0x5d40: wc = 0x7658; break;
        case 0x5d41: wc = 0x76ed; break;
        case 0x5d42: wc = 0x7963; break;
        case 0x5d43: wc = 0x79b2; break;
        case 0x5d44: wc = 0x7a6d; break;
        case 0x5d45: wc = 0x7cf2; break;
        case 0x5d46: wc = 0x7d7d; break;
        case 0x5d47: wc = 0x7d9f; break;
        case 0x5d48: wc = 0x8182; break;
        case 0x5d49: wc = 0x81da; break;
        case 0x5d4a: wc = 0x8354; break;
        case 0x5d4b: wc = 0x85d8; break;
        case 0x5d4c: wc = 0x4561; break;
        case 0x5d4d: wc = 0x8727; break;
        case 0x5d4e: wc = 0x8821; break;
        case 0x5d4f: wc = 0x90d8; break;
        case 0x5d50: wc = 0x92c1; break;
        case 0x5d51: wc = 0x9462; break;
        case 0x5d52: wc = 0x9457; break;
        case 0x5d53: wc = 0x9a39; break;
        case 0x5d54: wc = 0x9c71; break;
        case 0x5d55: wc = 0x9c7a; break;
        case 0x5d56: wc = 0x9d79; break;
        case 0x5d57: wc = 0x9ee7; break;
        case 0x5d58: wc = 0x512e; break;
        case 0x5d59: wc = 0x53a4; break;
        case 0x5d5a: wc = 0x53fb; break;
        case 0x5d5b: wc = 0x56a6; break;
        case 0x5d5c: wc = 0x58e2; break;
        case 0x5d5d: wc = 0x5c74; break;
        case 0x5d5e: wc = 0x64fd; break;
        case 0x5d5f: wc = 0x6702; break;
        case 0x5d60: wc = 0x6738; break;
        case 0x5d61: wc = 0x6adf; break;
        case 0x5d62: wc = 0x6aea; break;
        case 0x5d63: wc = 0x74c5; break;
        case 0x5d64: wc = 0x7667; break;
        case 0x5d65: wc = 0x76aa; break;
        case 0x5d66: wc = 0x7833; break;
        case 0x5d67: wc = 0x78ff; break;
        case 0x5d68: wc = 0x7930; break;
        case 0x5d69: wc = 0x79dd; break;
        case 0x5d6a: wc = 0x849a; break;
        case 0x5d6b: wc = 0x85f6; break;
        case 0x5d6c: wc = 0x86b8; break;
        case 0x5d6d: wc = 0x8777; break;
        case 0x5d6e: wc = 0x89fb; break;
        case 0x5d6f: wc = 0x8b88; break;
        case 0x5d70: wc = 0x8d72; break;
        case 0x5d71: wc = 0x8e92; break;
        case 0x5d72: wc = 0x8f63; break;
        case 0x5d73: wc = 0x9148; break;
        case 0x5d74: wc = 0x9398; break;
        case 0x5d75: wc = 0x974b; break;
        case 0x5d76: wc = 0x9c73; break;
        case 0x5d77: wc = 0x50c6; break;
        case 0x5d78: wc = 0x55f9; break;
        case 0x5d79: wc = 0x581c; break;
        case 0x5d7a: wc = 0x5971; break;
        case 0x5d7b: wc = 0x5b4c; break;
        case 0x5d7c: wc = 0x5b7f; break;
        case 0x5d7d: wc = 0x5f9a; break;
        case 0x5d7e: wc = 0x6459; break;
        case 0x5d7f: wc = 0x6506; break;
        case 0x5d80: wc = 0x695d; break;
        case 0x5d81: wc = 0x69e4; break;
        case 0x5d82: wc = 0x6e45; break;
        case 0x5d83: wc = 0x7e3a; break;
        case 0x5d84: wc = 0x7ff4; break;
        case 0x5d85: wc = 0x81a6; break;
        case 0x5d86: wc = 0x81e0; break;
        case 0x5d87: wc = 0x8430; break;
        case 0x5d88: wc = 0x8933; break;
        case 0x5d89: wc = 0x8b30; break;
        case 0x5d8a: wc = 0x8e65; break;
        case 0x5d8b: wc = 0x913b; break;
        case 0x5d8c: wc = 0x93c8; break;
        case 0x5d8d: wc = 0x9c0a; break;
        case 0x5d8e: wc = 0x9c31; break;
        case 0x5d8f: wc = 0x54a7; break;
        case 0x5d90: wc = 0x54f7; break;
        case 0x5d91: wc = 0x59f4; break;
        case 0x5d92: wc = 0x6318; break;
        case 0x5d93: wc = 0x6369; break;
        case 0x5d94: wc = 0x6835; break;
        case 0x5d95: wc = 0x7759; break;
        case 0x5d96: wc = 0x811f; break;
        case 0x5d97: wc = 0x8322; break;
        case 0x5d98: wc = 0x86da; break;
        case 0x5d99: wc = 0x86f6; break;
        case 0x5d9a: wc = 0x8d94; break;
        case 0x5d9b: wc = 0x8fe3; break;
        case 0x5d9c: wc = 0x8ffe; break;
        case 0x5d9d: wc = 0x92dd; break;
        case 0x5d9e: wc = 0x98b2; break;
        case 0x5d9f: wc = 0x9ba4; break;
        case 0x5da0: wc = 0x9d37; break;
        case 0x5da1: wc = 0x5333; break;
        case 0x5da2: wc = 0x5969; break;
        case 0x5da3: wc = 0x5b1a; break;
        case 0x5da4: wc = 0x5e18; break;
        case 0x5da5: wc = 0x641b; break;
        case 0x5da6: wc = 0x6e93; break;
        case 0x5da7: wc = 0x6fb0; break;
        case 0x5da8: wc = 0x7032; break;
        case 0x5da9: wc = 0x71eb; break;
        case 0x5daa: wc = 0x78cf; break;
        case 0x5dab: wc = 0x2559d; break;
        case 0x5dac: wc = 0x7c62; break;
        case 0x5dad: wc = 0x7fb7; break;
        case 0x5dae: wc = 0x81c1; break;
        case 0x5daf: wc = 0x8595; break;
        case 0x5db0: wc = 0x859f; break;
        case 0x5db1: wc = 0x861e; break;
        case 0x5db2: wc = 0x880a; break;
        case 0x5db3: wc = 0x899d; break;
        case 0x5db4: wc = 0x942e; break;
        case 0x5db5: wc = 0x9b11; break;
        case 0x5db6: wc = 0x5120; break;
        case 0x5db7: wc = 0x5de4; break;
        case 0x5db8: wc = 0x64f8; break;
        case 0x5db9: wc = 0x72a3; break;
        case 0x5dba: wc = 0x8e90; break;
        case 0x5dbb: wc = 0x9b23; break;
        case 0x5dbc: wc = 0x9c72; break;
        case 0x5dbd: wc = 0x53e6; break;
        case 0x5dbe: wc = 0x5464; break;
        case 0x5dbf: wc = 0x577d; break;
        case 0x5dc0: wc = 0x59c8; break;
        case 0x5dc1: wc = 0x5cad; break;
        case 0x5dc2: wc = 0x5f7e; break;
        case 0x5dc3: wc = 0x62ce; break;
        case 0x5dc4: wc = 0x6624; break;
        case 0x5dc5: wc = 0x67c3; break;
        case 0x5dc6: wc = 0x6afa; break;
        case 0x5dc7: wc = 0x6b1e; break;
        case 0x5dc8: wc = 0x3c53; break;
        case 0x5dc9: wc = 0x6ce0; break;
        case 0x5dca: wc = 0x6faa; break;
        case 0x5dcb: wc = 0x7075; break;
        case 0x5dcc: wc = 0x70a9; break;
        case 0x5dcd: wc = 0x7227; break;
        case 0x5dce: wc = 0x72d1; break;
        case 0x5dcf: wc = 0x74f4; break;
        case 0x5dd0: wc = 0x768a; break;
        case 0x5dd1: wc = 0x7831; break;
        case 0x5dd2: wc = 0x79e2; break;
        case 0x5dd3: wc = 0x8232; break;
        case 0x5dd4: wc = 0x82d3; break;
        case 0x5dd5: wc = 0x8576; break;
        case 0x5dd6: wc = 0x8626; break;
        case 0x5dd7: wc = 0x86c9; break;
        case 0x5dd8: wc = 0x8a45; break;
        case 0x5dd9: wc = 0x8ee8; break;
        case 0x5dda: wc = 0x9143; break;
        case 0x5ddb: wc = 0x91bd; break;
        case 0x5ddc: wc = 0x971d; break;
        case 0x5ddd: wc = 0x99d6; break;
        case 0x5dde: wc = 0x9d12; break;
        case 0x5ddf: wc = 0x6b1a; break;
        case 0x5de0: wc = 0x6fff; break;
        case 0x5de1: wc = 0x72a1; break;
        case 0x5de2: wc = 0x3f5d; break;
        case 0x5de3: wc = 0x7805; break;
        case 0x5de4: wc = 0x793c; break;
        case 0x5de5: wc = 0x96b8; break;
        case 0x5de6: wc = 0x9c67; break;
        case 0x5de7: wc = 0x50d7; break;
        case 0x5de8: wc = 0x54be; break;
        case 0x5de9: wc = 0x54f0; break;
        case 0x5dea: wc = 0x562e; break;
        case 0x5deb: wc = 0x5695; break;
        case 0x5dec: wc = 0x56a7; break;
        case 0x5ded: wc = 0x58da; break;
        case 0x5dee: wc = 0x5aea; break;
        case 0x5def: wc = 0x5d97; break;
        case 0x5df0: wc = 0x6045; break;
        case 0x5df1: wc = 0x61a6; break;
        case 0x5df2: wc = 0x64fc; break;
        case 0x5df3: wc = 0x650e; break;
        case 0x5df4: wc = 0x6833; break;
        case 0x5df5: wc = 0x3bdd; break;
        case 0x5df6: wc = 0x6a6f; break;
        case 0x5df7: wc = 0x6ae8; break;
        case 0x5df8: wc = 0x6c0c; break;
        case 0x5df9: wc = 0x3ce3; break;
        case 0x5dfa: wc = 0x6d76; break;
        case 0x5dfb: wc = 0x6ef7; break;
        case 0x5dfc: wc = 0x6f87; break;
        case 0x5dfd: wc = 0x72eb; break;
        case 0x5dfe: wc = 0x7379; break;
        case 0x5dff: wc = 0x7388; break;
        case 0x5e00: wc = 0x73ef; break;
        case 0x5e01: wc = 0x7490; break;
        case 0x5e02: wc = 0x7646; break;
        case 0x5e03: wc = 0x77d1; break;
        case 0x5e04: wc = 0x7853; break;
        case 0x5e05: wc = 0x7a82; break;
        case 0x5e06: wc = 0x7c29; break;
        case 0x5e07: wc = 0x7c2c; break;
        case 0x5e08: wc = 0x7c35; break;
        case 0x5e09: wc = 0x7c5a; break;
        case 0x5e0a: wc = 0x7e91; break;
        case 0x5e0b: wc = 0x7f4f; break;
        case 0x5e0c: wc = 0x8002; break;
        case 0x5e0d: wc = 0x802e; break;
        case 0x5e0e: wc = 0x6725; break;
        case 0x5e0f: wc = 0x8263; break;
        case 0x5e10: wc = 0x826a; break;
        case 0x5e11: wc = 0x826b; break;
        case 0x5e12: wc = 0x84fe; break;
        case 0x5e13: wc = 0x8557; break;
        case 0x5e14: wc = 0x87e7; break;
        case 0x5e15: wc = 0x8826; break;
        case 0x5e16: wc = 0x27c09; break;
        case 0x5e17: wc = 0x8ec2; break;
        case 0x5e18: wc = 0x8f64; break;
        case 0x5e19: wc = 0x92a0; break;
        case 0x5e1a: wc = 0x93c0; break;
        case 0x5e1b: wc = 0x9412; break;
        case 0x5e1c: wc = 0x93f4; break;
        case 0x5e1d: wc = 0x942a; break;
        case 0x5e1e: wc = 0x946a; break;
        case 0x5e1f: wc = 0x9871; break;
        case 0x5e20: wc = 0x4c90; break;
        case 0x5e21: wc = 0x9c78; break;
        case 0x5e22: wc = 0x9e15; break;
        case 0x5e23: wc = 0x35a8; break;
        case 0x5e24: wc = 0x5876; break;
        case 0x5e25: wc = 0x5ed8; break;
        case 0x5e26: wc = 0x5f54; break;
        case 0x5e27: wc = 0x645d; break;
        case 0x5e28: wc = 0x6c2f; break;
        case 0x5e29: wc = 0x6de5; break;
        case 0x5e2a: wc = 0x6f09; break;
        case 0x5e2b: wc = 0x719d; break;
        case 0x5e2c: wc = 0x742d; break;
        case 0x5e2d: wc = 0x752a; break;
        case 0x5e2e: wc = 0x76dd; break;
        case 0x5e2f: wc = 0x7769; break;
        case 0x5e30: wc = 0x7c0f; break;
        case 0x5e31: wc = 0x7c59; break;
        case 0x5e32: wc = 0x850d; break;
        case 0x5e33: wc = 0x87b0; break;
        case 0x5e34: wc = 0x8e57; break;
        case 0x5e35: wc = 0x8f46; break;
        case 0x5e36: wc = 0x902f; break;
        case 0x5e37: wc = 0x9181; break;
        case 0x5e38: wc = 0x9334; break;
        case 0x5e39: wc = 0x9a04; break;
        case 0x5e3a: wc = 0x4c5a; break;
        case 0x5e3b: wc = 0x2a2de; break;
        case 0x5e3c: wc = 0x60c0; break;
        case 0x5e3d: wc = 0x6ea3; break;
        case 0x5e3e: wc = 0x7896; break;
        case 0x5e3f: wc = 0x83d5; break;
        case 0x5e40: wc = 0x5131; break;
        case 0x5e41: wc = 0x54e2; break;
        case 0x5e42: wc = 0x56a8; break;
        case 0x5e43: wc = 0x5dc3; break;
        case 0x5e44: wc = 0x650f; break;
        case 0x5e45: wc = 0x66e8; break;
        case 0x5e46: wc = 0x6887; break;
        case 0x5e47: wc = 0x6af3; break;
        case 0x5e48: wc = 0x7931; break;
        case 0x5e49: wc = 0x8622; break;
        case 0x5e4a: wc = 0x882a; break;
        case 0x5e4b: wc = 0x88c3; break;
        case 0x5e4c: wc = 0x8971; break;
        case 0x5e4d: wc = 0x8c45; break;
        case 0x5e4e: wc = 0x93e7; break;
        case 0x5e4f: wc = 0x9468; break;
        case 0x5e50: wc = 0x96b4; break;
        case 0x5e51: wc = 0x9747; break;
        case 0x5e52: wc = 0x9e17; break;
        case 0x5e53: wc = 0x9f93; break;
        case 0x5e54: wc = 0x58e8; break;
        case 0x5e55: wc = 0x64c2; break;
        case 0x5e56: wc = 0x6502; break;
        case 0x5e57: wc = 0x6a91; break;
        case 0x5e58: wc = 0x6ad1; break;
        case 0x5e59: wc = 0x74c3; break;
        case 0x5e5a: wc = 0x757e; break;
        case 0x5e5b: wc = 0x7657; break;
        case 0x5e5c: wc = 0x78e5; break;
        case 0x5e5d: wc = 0x790c; break;
        case 0x5e5e: wc = 0x7927; break;
        case 0x5e5f: wc = 0x7928; break;
        case 0x5e60: wc = 0x7b59; break;
        case 0x5e61: wc = 0x7c5f; break;
        case 0x5e62: wc = 0x7e87; break;
        case 0x5e63: wc = 0x7f4d; break;
        case 0x5e64: wc = 0x8012; break;
        case 0x5e65: wc = 0x857e; break;
        case 0x5e66: wc = 0x85fe; break;
        case 0x5e67: wc = 0x881d; break;
        case 0x5e68: wc = 0x8a84; break;
        case 0x5e69: wc = 0x8b84; break;
        case 0x5e6a: wc = 0x8f60; break;
        case 0x5e6b: wc = 0x9179; break;
        case 0x5e6c: wc = 0x9287; break;
        case 0x5e6d: wc = 0x9433; break;
        case 0x5e6e: wc = 0x9478; break;
        case 0x5e6f: wc = 0x981b; break;
        case 0x5e70: wc = 0x9842; break;
        case 0x5e71: wc = 0x4f6c; break;
        case 0x5e72: wc = 0x5639; break;
        case 0x5e73: wc = 0x5afd; break;
        case 0x5e74: wc = 0x5c25; break;
        case 0x5e75: wc = 0x5c6a; break;
        case 0x5e76: wc = 0x5d9a; break;
        case 0x5e77: wc = 0x6180; break;
        case 0x5e78: wc = 0x61ad; break;
        case 0x5e79: wc = 0x64a9; break;
        case 0x5e7a: wc = 0x6579; break;
        case 0x5e7b: wc = 0x66b8; break;
        case 0x5e7c: wc = 0x66e2; break;
        case 0x5e7d: wc = 0x6a51; break;
        case 0x5e7e: wc = 0x6f66; break;
        case 0x5e7f: wc = 0x7093; break;
        case 0x5e80: wc = 0x720e; break;
        case 0x5e81: wc = 0x7360; break;
        case 0x5e82: wc = 0x7ab2; break;
        case 0x5e83: wc = 0x7ab7; break;
        case 0x5e84: wc = 0x25cbf; break;
        case 0x5e85: wc = 0x7c1d; break;
        case 0x5e86: wc = 0x7e5a; break;
        case 0x5e87: wc = 0x7fcf; break;
        case 0x5e88: wc = 0x818b; break;
        case 0x5e89: wc = 0x81ab; break;
        case 0x5e8a: wc = 0x85d4; break;
        case 0x5e8b: wc = 0x87c9; break;
        case 0x5e8c: wc = 0x87df; break;
        case 0x5e8d: wc = 0x8c42; break;
        case 0x5e8e: wc = 0x8f51; break;
        case 0x5e8f: wc = 0x911d; break;
        case 0x5e90: wc = 0x91aa; break;
        case 0x5e91: wc = 0x9410; break;
        case 0x5e92: wc = 0x957d; break;
        case 0x5e93: wc = 0x985f; break;
        case 0x5e94: wc = 0x98c2; break;
        case 0x5e95: wc = 0x98c9; break;
        case 0x5e96: wc = 0x4b5c; break;
        case 0x5e97: wc = 0x9ace; break;
        case 0x5e98: wc = 0x9def; break;
        case 0x5e99: wc = 0x5fbf; break;
        case 0x5e9a: wc = 0x7216; break;
        case 0x5e9b: wc = 0x77d3; break;
        case 0x5e9c: wc = 0x882c; break;
        case 0x5e9d: wc = 0x8e98; break;
        case 0x5e9e: wc = 0x9f92; break;
        case 0x5e9f: wc = 0x50c2; break;
        case 0x5ea0: wc = 0x560d; break;
        case 0x5ea1: wc = 0x587f; break;
        case 0x5ea2: wc = 0x5ad8; break;
        case 0x5ea3: wc = 0x218ba; break;
        case 0x5ea4: wc = 0x5d81; break;
        case 0x5ea5: wc = 0x5ed4; break;
        case 0x5ea6: wc = 0x617a; break;
        case 0x5ea7: wc = 0x645f; break;
        case 0x5ea8: wc = 0x6ad0; break;
        case 0x5ea9: wc = 0x6c00; break;
        case 0x5eaa: wc = 0x6cea; break;
        case 0x5eab: wc = 0x3d03; break;
        case 0x5eac: wc = 0x6f0a; break;
        case 0x5ead: wc = 0x71a1; break;
        case 0x5eae: wc = 0x3e4e; break;
        case 0x5eaf: wc = 0x763a; break;
        case 0x5eb0: wc = 0x779c; break;
        case 0x5eb1: wc = 0x7c0d; break;
        case 0x5eb2: wc = 0x802c; break;
        case 0x5eb3: wc = 0x81a2; break;
        case 0x5eb4: wc = 0x825b; break;
        case 0x5eb5: wc = 0x87bb; break;
        case 0x5eb6: wc = 0x8b31; break;
        case 0x5eb7: wc = 0x8c97; break;
        case 0x5eb8: wc = 0x97bb; break;
        case 0x5eb9: wc = 0x4bab; break;
        case 0x5eba: wc = 0x9acf; break;
        case 0x5ebb: wc = 0x9ddc; break;
        case 0x5ebc: wc = 0x9f3a; break;
        case 0x5ebd: wc = 0x52e0; break;
        case 0x5ebe: wc = 0x56a0; break;
        case 0x5ebf: wc = 0x3667; break;
        case 0x5ec0: wc = 0x5ab9; break;
        case 0x5ec1: wc = 0x5b3c; break;
        case 0x5ec2: wc = 0x5d67; break;
        case 0x5ec3: wc = 0x5ec7; break;
        case 0x5ec4: wc = 0x61f0; break;
        case 0x5ec5: wc = 0x6a0f; break;
        case 0x5ec6: wc = 0x6b19; break;
        case 0x5ec7: wc = 0x6f3b; break;
        case 0x5ec8: wc = 0x7045; break;
        case 0x5ec9: wc = 0x7198; break;
        case 0x5eca: wc = 0x73cb; break;
        case 0x5ecb: wc = 0x746c; break;
        case 0x5ecc: wc = 0x79b7; break;
        case 0x5ecd: wc = 0x7d6b; break;
        case 0x5ece: wc = 0x7db9; break;
        case 0x5ecf: wc = 0x7e32; break;
        case 0x5ed0: wc = 0x7e8d; break;
        case 0x5ed1: wc = 0x7f76; break;
        case 0x5ed2: wc = 0x8248; break;
        case 0x5ed3: wc = 0x84c5; break;
        case 0x5ed4: wc = 0x8502; break;
        case 0x5ed5: wc = 0x85f0; break;
        case 0x5ed6: wc = 0x85df; break;
        case 0x5ed7: wc = 0x8632; break;
        case 0x5ed8: wc = 0x8631; break;
        case 0x5ed9: wc = 0x8646; break;
        case 0x5eda: wc = 0x88d7; break;
        case 0x5edb: wc = 0x905b; break;
        case 0x5edc: wc = 0x925a; break;
        case 0x5edd: wc = 0x938f; break;
        case 0x5ede: wc = 0x93a6; break;
        case 0x5edf: wc = 0x93d0; break;
        case 0x5ee0: wc = 0x96e1; break;
        case 0x5ee1: wc = 0x9724; break;
        case 0x5ee2: wc = 0x98c5; break;
        case 0x5ee3: wc = 0x98f9; break;
        case 0x5ee4: wc = 0x993e; break;
        case 0x5ee5: wc = 0x99f5; break;
        case 0x5ee6: wc = 0x9a2e; break;
        case 0x5ee7: wc = 0x9c21; break;
        case 0x5ee8: wc = 0x9db9; break;
        case 0x5ee9: wc = 0x9dda; break;
        case 0x5eea: wc = 0x9e13; break;
        case 0x5eeb: wc = 0x9e8d; break;
        case 0x5eec: wc = 0x50c7; break;
        case 0x5eed: wc = 0x5279; break;
        case 0x5eee: wc = 0x5774; break;
        case 0x5eef: wc = 0x6dd5; break;
        case 0x5ef0: wc = 0x78df; break;
        case 0x5ef1: wc = 0x7a11; break;
        case 0x5ef2: wc = 0x7a4b; break;
        case 0x5ef3: wc = 0x8e1b; break;
        case 0x5ef4: wc = 0x9be5; break;
        case 0x5ef5: wc = 0x9d71; break;
        case 0x5ef6: wc = 0x55e7; break;
        case 0x5ef7: wc = 0x5707; break;
        case 0x5ef8: wc = 0x6384; break;
        case 0x5ef9: wc = 0x68c6; break;
        case 0x5efa: wc = 0x78ee; break;
        case 0x5efb: wc = 0x7a10; break;
        case 0x5efc: wc = 0x8726; break;
        case 0x5efd: wc = 0x8e1a; break;
        case 0x5efe: wc = 0x9300; break;
        case 0x5eff: wc = 0x966f; break;
        case 0x5f00: wc = 0x5088; break;
        case 0x5f01: wc = 0x51d3; break;
        case 0x5f02: wc = 0x578f; break;
        case 0x5f03: wc = 0x585b; break;
        case 0x5f04: wc = 0x5d42; break;
        case 0x5f05: wc = 0x642e; break;
        case 0x5f06: wc = 0x6ea7; break;
        case 0x5f07: wc = 0x746e; break;
        case 0x5f08: wc = 0x7849; break;
        case 0x5f09: wc = 0x7be5; break;
        case 0x5f0a: wc = 0x7e42; break;
        case 0x5f0b: wc = 0x819f; break;
        case 0x5f0c: wc = 0x844e; break;
        case 0x5f0d: wc = 0x9dc5; break;
        case 0x5f0e: wc = 0x9e9c; break;
        case 0x5f0f: wc = 0x70ff; break;
        case 0x5f10: wc = 0x7643; break;
        case 0x5f11: wc = 0x7abf; break;
        case 0x5f12: wc = 0x856f; break;
        case 0x5f13: wc = 0x4ec2; break;
        case 0x5f14: wc = 0x561e; break;
        case 0x5f15: wc = 0x6250; break;
        case 0x5f16: wc = 0x691a; break;
        case 0x5f17: wc = 0x6c3b; break;
        case 0x5f18: wc = 0x6cd0; break;
        case 0x5f19: wc = 0x738f; break;
        case 0x5f1a: wc = 0x7afb; break;
        case 0x5f1b: wc = 0x7c15; break;
        case 0x5f1c: wc = 0x827b; break;
        case 0x5f1d: wc = 0x961e; break;
        case 0x5f1e: wc = 0x9c33; break;
        case 0x5f1f: wc = 0x5ee9; break;
        case 0x5f20: wc = 0x61cd; break;
        case 0x5f21: wc = 0x6a81; break;
        case 0x5f22: wc = 0x6f9f; break;
        case 0x5f23: wc = 0x83fb; break;
        case 0x5f24: wc = 0x4572; break;
        case 0x5f25: wc = 0x5030; break;
        case 0x5f26: wc = 0x3604; break;
        case 0x5f27: wc = 0x580e; break;
        case 0x5f28: wc = 0x590c; break;
        case 0x5f29: wc = 0x5a48; break;
        case 0x5f2a: wc = 0x5d1a; break;
        case 0x5f2b: wc = 0x5eb1; break;
        case 0x5f2c: wc = 0x6123; break;
        case 0x5f2d: wc = 0x68f1; break;
        case 0x5f2e: wc = 0x6de9; break;
        case 0x5f2f: wc = 0x740c; break;
        case 0x5f30: wc = 0x7756; break;
        case 0x5f31: wc = 0x8506; break;
        case 0x5f32: wc = 0x8590; break;
        case 0x5f33: wc = 0x88ec; break;
        case 0x5f34: wc = 0x8e1c; break;
        case 0x5f35: wc = 0x8f18; break;
        case 0x5f36: wc = 0x9302; break;
        case 0x5f37: wc = 0x9bea; break;
        case 0x5f38: wc = 0x4fec; break;
        case 0x5f39: wc = 0x4fd0; break;
        case 0x5f3a: wc = 0x527a; break;
        case 0x5f3b: wc = 0x5299; break;
        case 0x5f3c: wc = 0x54e9; break;
        case 0x5f3d: wc = 0x55b1; break;
        case 0x5f3e: wc = 0x5a33; break;
        case 0x5f3f: wc = 0x5a0c; break;
        case 0x5f40: wc = 0x5ae0; break;
        case 0x5f41: wc = 0x5b4b; break;
        case 0x5f42: wc = 0x5b77; break;
        case 0x5f43: wc = 0x5cdb; break;
        case 0x5f44: wc = 0x5f72; break;
        case 0x5f45: wc = 0x609d; break;
        case 0x5f46: wc = 0x645b; break;
        case 0x5f47: wc = 0x6521; break;
        case 0x5f48: wc = 0x68a9; break;
        case 0x5f49: wc = 0x6a06; break;
        case 0x5f4a: wc = 0x6c02; break;
        case 0x5f4b: wc = 0x6d96; break;
        case 0x5f4c: wc = 0x6d70; break;
        case 0x5f4d: wc = 0x6f13; break;
        case 0x5f4e: wc = 0x7282; break;
        case 0x5f4f: wc = 0x729b; break;
        case 0x5f50: wc = 0x7301; break;
        case 0x5f51: wc = 0x740d; break;
        case 0x5f52: wc = 0x3f16; break;
        case 0x5f53: wc = 0x7505; break;
        case 0x5f54: wc = 0x76e0; break;
        case 0x5f55: wc = 0x779d; break;
        case 0x5f56: wc = 0x79bb; break;
        case 0x5f57: wc = 0x7af0; break;
        case 0x5f58: wc = 0x7b63; break;
        case 0x5f59: wc = 0x7bf1; break;
        case 0x5f5a: wc = 0x7cb4; break;
        case 0x5f5b: wc = 0x7cce; break;
        case 0x5f5c: wc = 0x7e2d; break;
        case 0x5f5d: wc = 0x7e4c; break;
        case 0x5f5e: wc = 0x7e9a; break;
        case 0x5f5f: wc = 0x4366; break;
        case 0x5f60: wc = 0x8385; break;
        case 0x5f61: wc = 0x83de; break;
        case 0x5f62: wc = 0x849e; break;
        case 0x5f63: wc = 0x853e; break;
        case 0x5f64: wc = 0x863a; break;
        case 0x5f65: wc = 0x870a; break;
        case 0x5f66: wc = 0x87ad; break;
        case 0x5f67: wc = 0x882b; break;
        case 0x5f68: wc = 0x8935; break;
        case 0x5f69: wc = 0x8a48; break;
        case 0x5f6a: wc = 0x8b27; break;
        case 0x5f6b: wc = 0x8c8d; break;
        case 0x5f6c: wc = 0x284e6; break;
        case 0x5f6d: wc = 0x9090; break;
        case 0x5f6e: wc = 0x91a8; break;
        case 0x5f6f: wc = 0x92f0; break;
        case 0x5f70: wc = 0x92eb; break;
        case 0x5f71: wc = 0x9b01; break;
        case 0x5f72: wc = 0x9b51; break;
        case 0x5f73: wc = 0x9bec; break;
        case 0x5f74: wc = 0x9e1d; break;
        case 0x5f75: wc = 0x9ed0; break;
        case 0x5f76: wc = 0x4e83; break;
        case 0x5f77: wc = 0x50ef; break;
        case 0x5f78: wc = 0x58e3; break;
        case 0x5f79: wc = 0x5d99; break;
        case 0x5f7a: wc = 0x222f3; break;
        case 0x5f7b: wc = 0x6061; break;
        case 0x5f7c: wc = 0x608b; break;
        case 0x5f7d: wc = 0x66bd; break;
        case 0x5f7e: wc = 0x6a49; break;
        case 0x5f7f: wc = 0x711b; break;
        case 0x5f80: wc = 0x735c; break;
        case 0x5f81: wc = 0x7510; break;
        case 0x5f82: wc = 0x7584; break;
        case 0x5f83: wc = 0x77b5; break;
        case 0x5f84: wc = 0x78f7; break;
        case 0x5f85: wc = 0x7cbc; break;
        case 0x5f86: wc = 0x7e57; break;
        case 0x5f87: wc = 0x7ff7; break;
        case 0x5f88: wc = 0x8e78; break;
        case 0x5f89: wc = 0x8e99; break;
        case 0x5f8a: wc = 0x8f54; break;
        case 0x5f8b: wc = 0x9074; break;
        case 0x5f8c: wc = 0x9130; break;
        case 0x5f8d: wc = 0x93fb; break;
        case 0x5f8e: wc = 0x95b5; break;
        case 0x5f8f: wc = 0x9a4e; break;
        case 0x5f90: wc = 0x9e90; break;
        case 0x5f91: wc = 0x38e9; break;
        case 0x5f92: wc = 0x60cf; break;
        case 0x5f93: wc = 0x68fd; break;
        case 0x5f94: wc = 0x3d49; break;
        case 0x5f95: wc = 0x702e; break;
        case 0x5f96: wc = 0x75f3; break;
        case 0x5f97: wc = 0x7884; break;
        case 0x5f98: wc = 0x7b96; break;
        case 0x5f99: wc = 0x9182; break;
        case 0x5f9a: wc = 0x5ca6; break;
        case 0x5f9b: wc = 0x6e41; break;
        case 0x5f9c: wc = 0x82d9; break;
        case 0x5f9d: wc = 0x96f4; break;
        case 0x5f9e: wc = 0x9d17; break;
        case 0x5f9f: wc = 0x3403; break;
        case 0x5fa0: wc = 0x4e87; break;
        case 0x5fa1: wc = 0x508c; break;
        case 0x5fa2: wc = 0x5298; break;
        case 0x5fa3: wc = 0x54f6; break;
        case 0x5fa4: wc = 0x55ce; break;
        case 0x5fa5: wc = 0x561b; break;
        case 0x5fa6: wc = 0x561c; break;
        case 0x5fa7: wc = 0x5692; break;
        case 0x5fa8: wc = 0x56b0; break;
        case 0x5fa9: wc = 0x5abd; break;
        case 0x5faa: wc = 0x5b24; break;
        case 0x5fab: wc = 0x69aa; break;
        case 0x5fac: wc = 0x6ea4; break;
        case 0x5fad: wc = 0x79a1; break;
        case 0x5fae: wc = 0x8031; break;
        case 0x5faf: wc = 0x8534; break;
        case 0x5fb0: wc = 0x85e6; break;
        case 0x5fb1: wc = 0x8611; break;
        case 0x5fb2: wc = 0x879e; break;
        case 0x5fb3: wc = 0x87c6; break;
        case 0x5fb4: wc = 0x45eb; break;
        case 0x5fb5: wc = 0x222d9; break;
        case 0x5fb6: wc = 0x93b7; break;
        case 0x5fb7: wc = 0x995d; break;
        case 0x5fb8: wc = 0x9acd; break;
        case 0x5fb9: wc = 0x9b15; break;
        case 0x5fba: wc = 0x4c88; break;
        case 0x5fbb: wc = 0x9dcc; break;
        case 0x5fbc: wc = 0x9ebd; break;
        case 0x5fbd: wc = 0x9ebf; break;
        case 0x5fbe: wc = 0x55fc; break;
        case 0x5fbf: wc = 0x587b; break;
        case 0x5fc0: wc = 0x5e59; break;
        case 0x5fc1: wc = 0x66af; break;
        case 0x5fc2: wc = 0x763c; break;
        case 0x5fc3: wc = 0x7799; break;
        case 0x5fc4: wc = 0x7e38; break;
        case 0x5fc5: wc = 0x85d0; break;
        case 0x5fc6: wc = 0x911a; break;
        case 0x5fc7: wc = 0x93cc; break;
        case 0x5fc8: wc = 0x9943; break;
        case 0x5fc9: wc = 0x50c8; break;
        case 0x5fca: wc = 0x5881; break;
        case 0x5fcb: wc = 0x5ada; break;
        case 0x5fcc: wc = 0x5e54; break;
        case 0x5fcd: wc = 0x6097; break;
        case 0x5fce: wc = 0x6172; break;
        case 0x5fcf: wc = 0x61e3; break;
        case 0x5fd0: wc = 0x69fe; break;
        case 0x5fd1: wc = 0x6a20; break;
        case 0x5fd2: wc = 0x6e7e; break;
        case 0x5fd3: wc = 0x6f6b; break;
        case 0x5fd4: wc = 0x6fab; break;
        case 0x5fd5: wc = 0x6ff7; break;
        case 0x5fd6: wc = 0x71b3; break;
        case 0x5fd7: wc = 0x734c; break;
        case 0x5fd8: wc = 0x774c; break;
        case 0x5fd9: wc = 0x77d5; break;
        case 0x5fda: wc = 0x7e35; break;
        case 0x5fdb: wc = 0x8115; break;
        case 0x5fdc: wc = 0x86ee; break;
        case 0x5fdd: wc = 0x87c3; break;
        case 0x5fde: wc = 0x8954; break;
        case 0x5fdf: wc = 0x8b3e; break;
        case 0x5fe0: wc = 0x9124; break;
        case 0x5fe1: wc = 0x93cb; break;
        case 0x5fe2: wc = 0x93dd; break;
        case 0x5fe3: wc = 0x9794; break;
        case 0x5fe4: wc = 0x9862; break;
        case 0x5fe5: wc = 0x9b17; break;
        case 0x5fe6: wc = 0x9b18; break;
        case 0x5fe7: wc = 0x59ba; break;
        case 0x5fe8: wc = 0x6629; break;
        case 0x5fe9: wc = 0x3b46; break;
        case 0x5fea: wc = 0x3b50; break;
        case 0x5feb: wc = 0x67ba; break;
        case 0x5fec: wc = 0x700e; break;
        case 0x5fed: wc = 0x7923; break;
        case 0x5fee: wc = 0x79e3; break;
        case 0x5fef: wc = 0x7c96; break;
        case 0x5ff0: wc = 0x889c; break;
        case 0x5ff1: wc = 0x88b9; break;
        case 0x5ff2: wc = 0x4b34; break;
        case 0x5ff3: wc = 0x92c4; break;
        case 0x5ff4: wc = 0x58fe; break;
        case 0x5ff5: wc = 0x5940; break;
        case 0x5ff6: wc = 0x5a0f; break;
        case 0x5ff7: wc = 0x607e; break;
        case 0x5ff8: wc = 0x60d8; break;
        case 0x5ff9: wc = 0x6722; break;
        case 0x5ffa: wc = 0x6757; break;
        case 0x5ffb: wc = 0x6767; break;
        case 0x5ffc: wc = 0x6c52; break;
        case 0x5ffd: wc = 0x6f2d; break;
        case 0x5ffe: wc = 0x7139; break;
        case 0x5fff: wc = 0x7264; break;
        case 0x6000: wc = 0x75dd; break;
        case 0x6001: wc = 0x76f3; break;
        case 0x6002: wc = 0x7865; break;
        case 0x6003: wc = 0x786d; break;
        case 0x6004: wc = 0x40c3; break;
        case 0x6005: wc = 0x7b00; break;
        case 0x6006: wc = 0x7f53; break;
        case 0x6007: wc = 0x34c1; break;
        case 0x6008: wc = 0x2626a; break;
        case 0x6009: wc = 0x7f52; break;
        case 0x600a: wc = 0x7f51; break;
        case 0x600b: wc = 0x83a3; break;
        case 0x600c: wc = 0x833b; break;
        case 0x600d: wc = 0x83f5; break;
        case 0x600e: wc = 0x8609; break;
        case 0x600f: wc = 0x8744; break;
        case 0x6010: wc = 0x87d2; break;
        case 0x6011: wc = 0x92e9; break;
        case 0x6012: wc = 0x9b4d; break;
        case 0x6013: wc = 0x4c36; break;
        case 0x6014: wc = 0x4c69; break;
        case 0x6015: wc = 0x52f1; break;
        case 0x6016: wc = 0x5445; break;
        case 0x6017: wc = 0x562a; break;
        case 0x6018: wc = 0x5746; break;
        case 0x6019: wc = 0x587a; break;
        case 0x601a: wc = 0x6334; break;
        case 0x601b: wc = 0x6973; break;
        case 0x601c: wc = 0x69d1; break;
        case 0x601d: wc = 0x6cac; break;
        case 0x601e: wc = 0x6d7c; break;
        case 0x601f: wc = 0x73ab; break;
        case 0x6020: wc = 0x75d7; break;
        case 0x6021: wc = 0x771b; break;
        case 0x6022: wc = 0x2531a; break;
        case 0x6023: wc = 0x7996; break;
        case 0x6024: wc = 0x259a4; break;
        case 0x6025: wc = 0x8104; break;
        case 0x6026: wc = 0x8122; break;
        case 0x6027: wc = 0x815c; break;
        case 0x6028: wc = 0x82fa; break;
        case 0x6029: wc = 0x8393; break;
        case 0x602a: wc = 0x85b6; break;
        case 0x602b: wc = 0x9176; break;
        case 0x602c: wc = 0x92c2; break;
        case 0x602d: wc = 0x9709; break;
        case 0x602e: wc = 0x973e; break;
        case 0x602f: wc = 0x97ce; break;
        case 0x6030: wc = 0x9b3d; break;
        case 0x6031: wc = 0x9df6; break;
        case 0x6032: wc = 0x9ee3; break;
        case 0x6033: wc = 0x773d; break;
        case 0x6034: wc = 0x8109; break;
        case 0x6035: wc = 0x86e8; break;
        case 0x6036: wc = 0x899b; break;
        case 0x6037: wc = 0x8c89; break;
        case 0x6038: wc = 0x8c98; break;
        case 0x6039: wc = 0x9722; break;
        case 0x603a: wc = 0x511a; break;
        case 0x603b: wc = 0x52d0; break;
        case 0x603c: wc = 0x74fe; break;
        case 0x603d: wc = 0x750d; break;
        case 0x603e: wc = 0x753f; break;
        case 0x603f: wc = 0x824b; break;
        case 0x6040: wc = 0x8394; break;
        case 0x6041: wc = 0x8544; break;
        case 0x6042: wc = 0x867b; break;
        case 0x6043: wc = 0x8722; break;
        case 0x6044: wc = 0x8771; break;
        case 0x6045: wc = 0x9133; break;
        case 0x6046: wc = 0x9333; break;
        case 0x6047: wc = 0x9f06; break;
        case 0x6048: wc = 0x4e5c; break;
        case 0x6049: wc = 0x3ac6; break;
        case 0x604a: wc = 0x65c0; break;
        case 0x604b: wc = 0x5196; break;
        case 0x604c: wc = 0x5853; break;
        case 0x604d: wc = 0x5e4e; break;
        case 0x604e: wc = 0x5e66; break;
        case 0x604f: wc = 0x6f1e; break;
        case 0x6050: wc = 0x7c1a; break;
        case 0x6051: wc = 0x7f83; break;
        case 0x6052: wc = 0x9f0f; break;
        case 0x6053: wc = 0x4e0f; break;
        case 0x6054: wc = 0x4fdb; break;
        case 0x6055: wc = 0x506d; break;
        case 0x6056: wc = 0x52d4; break;
        case 0x6057: wc = 0x5595; break;
        case 0x6058: wc = 0x5a42; break;
        case 0x6059: wc = 0x5a94; break;
        case 0x605a: wc = 0x5b80; break;
        case 0x605b: wc = 0x6110; break;
        case 0x605c: wc = 0x22c4d; break;
        case 0x605d: wc = 0x3a20; break;
        case 0x605e: wc = 0x6ab0; break;
        case 0x605f: wc = 0x6acb; break;
        case 0x6060: wc = 0x3d10; break;
        case 0x6061: wc = 0x6e4e; break;
        case 0x6062: wc = 0x77cf; break;
        case 0x6063: wc = 0x77ca; break;
        case 0x6064: wc = 0x7cc6; break;
        case 0x6065: wc = 0x7ddc; break;
        case 0x6066: wc = 0x817c; break;
        case 0x6067: wc = 0x8442; break;
        case 0x6068: wc = 0x8752; break;
        case 0x6069: wc = 0x9bb8; break;
        case 0x606a: wc = 0x9eaa; break;
        case 0x606b: wc = 0x5e6d; break;
        case 0x606c: wc = 0x61f1; break;
        case 0x606d: wc = 0x6423; break;
        case 0x606e: wc = 0x7bfe; break;
        case 0x606f: wc = 0x858e; break;
        case 0x6070: wc = 0x881b; break;
        case 0x6071: wc = 0x884a; break;
        case 0x6072: wc = 0x9c74; break;
        case 0x6073: wc = 0x59f3; break;
        case 0x6074: wc = 0x5ac7; break;
        case 0x6075: wc = 0x614f; break;
        case 0x6076: wc = 0x69a0; break;
        case 0x6077: wc = 0x6d3a; break;
        case 0x6078: wc = 0x7190; break;
        case 0x6079: wc = 0x7700; break;
        case 0x607a: wc = 0x7733; break;
        case 0x607b: wc = 0x89ad; break;
        case 0x607c: wc = 0x8a7a; break;
        case 0x607d: wc = 0x910d; break;
        case 0x607e: wc = 0x4cdf; break;
        case 0x607f: wc = 0x4f94; break;
        case 0x6080: wc = 0x54de; break;
        case 0x6081: wc = 0x59e5; break;
        case 0x6082: wc = 0x5aa2; break;
        case 0x6083: wc = 0x5aeb; break;
        case 0x6084: wc = 0x5cd4; break;
        case 0x6085: wc = 0x6048; break;
        case 0x6086: wc = 0x2274c; break;
        case 0x6087: wc = 0x6154; break;
        case 0x6088: wc = 0x65c4; break;
        case 0x6089: wc = 0x6786; break;
        case 0x608a: wc = 0x6802; break;
        case 0x608b: wc = 0x6bdf; break;
        case 0x608c: wc = 0x6bea; break;
        case 0x608d: wc = 0x6bf7; break;
        case 0x608e: wc = 0x6c01; break;
        case 0x608f: wc = 0x6d20; break;
        case 0x6090: wc = 0x7266; break;
        case 0x6091: wc = 0x73a5; break;
        case 0x6092: wc = 0x74f1; break;
        case 0x6093: wc = 0x7683; break;
        case 0x6094: wc = 0x770a; break;
        case 0x6095: wc = 0x782a; break;
        case 0x6096: wc = 0x79cf; break;
        case 0x6097: wc = 0x7ad3; break;
        case 0x6098: wc = 0x7c8d; break;
        case 0x6099: wc = 0x7ce2; break;
        case 0x609a: wc = 0x7f5e; break;
        case 0x609b: wc = 0x8004; break;
        case 0x609c: wc = 0x8252; break;
        case 0x609d: wc = 0x843a; break;
        case 0x609e: wc = 0x84e9; break;
        case 0x609f: wc = 0x86d1; break;
        case 0x60a0: wc = 0x8750; break;
        case 0x60a1: wc = 0x8765; break;
        case 0x60a2: wc = 0x87ca; break;
        case 0x60a3: wc = 0x87f1; break;
        case 0x60a4: wc = 0x8ede; break;
        case 0x60a5: wc = 0x9155; break;
        case 0x60a6: wc = 0x927e; break;
        case 0x60a7: wc = 0x296e8; break;
        case 0x60a8: wc = 0x9ae6; break;
        case 0x60a9: wc = 0x9d3e; break;
        case 0x60aa: wc = 0x9d9c; break;
        case 0x60ab: wc = 0x9eb0; break;
        case 0x60ac: wc = 0x51e9; break;
        case 0x60ad: wc = 0x5776; break;
        case 0x60ae: wc = 0x233cf; break;
        case 0x60af: wc = 0x6762; break;
        case 0x60b0: wc = 0x6958; break;
        case 0x60b1: wc = 0x6be3; break;
        case 0x60b2: wc = 0x7091; break;
        case 0x60b3: wc = 0x4032; break;
        case 0x60b4: wc = 0x82dc; break;
        case 0x60b5: wc = 0x83af; break;
        case 0x60b6: wc = 0x869e; break;
        case 0x60b7: wc = 0x926c; break;
        case 0x60b8: wc = 0x9702; break;
        case 0x60b9: wc = 0x6b7e; break;
        case 0x60ba: wc = 0x8388; break;
        case 0x60bb: wc = 0x51a1; break;
        case 0x60bc: wc = 0x5e6a; break;
        case 0x60bd: wc = 0x61de; break;
        case 0x60be: wc = 0x61dc; break;
        case 0x60bf: wc = 0x61f5; break;
        case 0x60c0: wc = 0x66da; break;
        case 0x60c1: wc = 0x6aac; break;
        case 0x60c2: wc = 0x6c0b; break;
        case 0x60c3: wc = 0x6fdb; break;
        case 0x60c4: wc = 0x77a2; break;
        case 0x60c5: wc = 0x77c7; break;
        case 0x60c6: wc = 0x791e; break;
        case 0x60c7: wc = 0x8268; break;
        case 0x60c8: wc = 0x8813; break;
        case 0x60c9: wc = 0x9138; break;
        case 0x60ca: wc = 0x96fa; break;
        case 0x60cb: wc = 0x9725; break;
        case 0x60cc: wc = 0x973f; break;
        case 0x60cd: wc = 0x995b; break;
        case 0x60ce: wc = 0x9e0f; break;
        case 0x60cf: wc = 0x55b5; break;
        case 0x60d0: wc = 0x5a8c; break;
        case 0x60d1: wc = 0x5af9; break;
        case 0x60d2: wc = 0x5cc1; break;
        case 0x60d3: wc = 0x6cd6; break;
        case 0x60d4: wc = 0x6dfc; break;
        case 0x60d5: wc = 0x6e35; break;
        case 0x60d6: wc = 0x7385; break;
        case 0x60d7: wc = 0x7707; break;
        case 0x60d8: wc = 0x7784; break;
        case 0x60d9: wc = 0x7b37; break;
        case 0x60da: wc = 0x7bce; break;
        case 0x60db: wc = 0x7df2; break;
        case 0x60dc: wc = 0x43da; break;
        case 0x60dd: wc = 0x8306; break;
        case 0x60de: wc = 0x8c93; break;
        case 0x60df: wc = 0x9c59; break;
        case 0x60e0: wc = 0x4e6e; break;
        case 0x60e1: wc = 0x5035; break;
        case 0x60e2: wc = 0x3487; break;
        case 0x60e3: wc = 0x511b; break;
        case 0x60e4: wc = 0x5463; break;
        case 0x60e5: wc = 0x5638; break;
        case 0x60e6: wc = 0x5825; break;
        case 0x60e7: wc = 0x5a2c; break;
        case 0x60e8: wc = 0x5a7a; break;
        case 0x60e9: wc = 0x5af5; break;
        case 0x60ea: wc = 0x5e60; break;
        case 0x60eb: wc = 0x5ee1; break;
        case 0x60ec: wc = 0x38f3; break;
        case 0x60ed: wc = 0x53af; break;
        case 0x60ee: wc = 0x6693; break;
        case 0x60ef: wc = 0x6a46; break;
        case 0x60f0: wc = 0x3c4f; break;
        case 0x60f1: wc = 0x6f55; break;
        case 0x60f2: wc = 0x3d83; break;
        case 0x60f3: wc = 0x7273; break;
        case 0x60f4: wc = 0x7512; break;
        case 0x60f5: wc = 0x7780; break;
        case 0x60f6: wc = 0x77b4; break;
        case 0x60f7: wc = 0x7894; break;
        case 0x60f8: wc = 0x80df; break;
        case 0x60f9: wc = 0x81b4; break;
        case 0x60fa: wc = 0x268b2; break;
        case 0x60fb: wc = 0x8381; break;
        case 0x60fc: wc = 0x88a4; break;
        case 0x60fd: wc = 0x8b55; break;
        case 0x60fe: wc = 0x8e07; break;
        case 0x60ff: wc = 0x8e8c; break;
        case 0x6100: wc = 0x912e; break;
        case 0x6101: wc = 0x9267; break;
        case 0x6102: wc = 0x936a; break;
        case 0x6103: wc = 0x4950; break;
        case 0x6104: wc = 0x97aa; break;
        case 0x6105: wc = 0x9a16; break;
        case 0x6106: wc = 0x9af3; break;
        case 0x6107: wc = 0x9d50; break;
        case 0x6108: wc = 0x9de1; break;
        case 0x6109: wc = 0x563f; break;
        case 0x610a: wc = 0x569c; break;
        case 0x610b: wc = 0x7205; break;
        case 0x610c: wc = 0x7e86; break;
        case 0x610d: wc = 0x87d4; break;
        case 0x610e: wc = 0x5301; break;
        case 0x610f: wc = 0x5461; break;
        case 0x6110: wc = 0x598f; break;
        case 0x6111: wc = 0x5f63; break;
        case 0x6112: wc = 0x6286; break;
        case 0x6113: wc = 0x636b; break;
        case 0x6114: wc = 0x66aa; break;
        case 0x6115: wc = 0x7086; break;
        case 0x6116: wc = 0x748a; break;
        case 0x6117: wc = 0x74ba; break;
        case 0x6118: wc = 0x7d7b; break;
        case 0x6119: wc = 0x8117; break;
        case 0x611a: wc = 0x82a0; break;
        case 0x611b: wc = 0x83db; break;
        case 0x611c: wc = 0x864b; break;
        case 0x611d: wc = 0x922b; break;
        case 0x611e: wc = 0x9346; break;
        case 0x611f: wc = 0x95bf; break;
        case 0x6120: wc = 0x95ba; break;
        case 0x6121: wc = 0x95c5; break;
        case 0x6122: wc = 0x99c7; break;
        case 0x6123: wc = 0x99bc; break;
        case 0x6124: wc = 0x9b70; break;
        case 0x6125: wc = 0x9cfc; break;
        case 0x6126: wc = 0x9f24; break;
        case 0x6127: wc = 0x5c89; break;
        case 0x6128: wc = 0x6612; break;
        case 0x6129: wc = 0x66f6; break;
        case 0x612a: wc = 0x6706; break;
        case 0x612b: wc = 0x7c85; break;
        case 0x612c: wc = 0x8985; break;
        case 0x612d: wc = 0x4eb9; break;
        case 0x612e: wc = 0x4f8e; break;
        case 0x612f: wc = 0x519e; break;
        case 0x6130: wc = 0x54aa; break;
        case 0x6131: wc = 0x54a9; break;
        case 0x6132: wc = 0x5833; break;
        case 0x6133: wc = 0x5a13; break;
        case 0x6134: wc = 0x5a84; break;
        case 0x6135: wc = 0x5aba; break;
        case 0x6136: wc = 0x5db6; break;
        case 0x6137: wc = 0x222cb; break;
        case 0x6138: wc = 0x5f25; break;
        case 0x6139: wc = 0x5f2d; break;
        case 0x613a: wc = 0x224ba; break;
        case 0x613b: wc = 0x224db; break;
        case 0x613c: wc = 0x5fbe; break;
        case 0x613d: wc = 0x6364; break;
        case 0x613e: wc = 0x6517; break;
        case 0x613f: wc = 0x6520; break;
        case 0x6140: wc = 0x6549; break;
        case 0x6141: wc = 0x6596; break;
        case 0x6142: wc = 0x6d23; break;
        case 0x6143: wc = 0x6d58; break;
        case 0x6144: wc = 0x6e33; break;
        case 0x6145: wc = 0x6ea6; break;
        case 0x6146: wc = 0x3d5f; break;
        case 0x6147: wc = 0x7030; break;
        case 0x6148: wc = 0x7056; break;
        case 0x6149: wc = 0x715d; break;
        case 0x614a: wc = 0x7222; break;
        case 0x614b: wc = 0x7338; break;
        case 0x614c: wc = 0x737c; break;
        case 0x614d: wc = 0x7442; break;
        case 0x614e: wc = 0x7653; break;
        case 0x614f: wc = 0x772f; break;
        case 0x6150: wc = 0x7787; break;
        case 0x6151: wc = 0x7a48; break;
        case 0x6152: wc = 0x418a; break;
        case 0x6153: wc = 0x7cdc; break;
        case 0x6154: wc = 0x7e3b; break;
        case 0x6155: wc = 0x7f8b; break;
        case 0x6156: wc = 0x8249; break;
        case 0x6157: wc = 0x83cb; break;
        case 0x6158: wc = 0x845e; break;
        case 0x6159: wc = 0x851d; break;
        case 0x615a: wc = 0x862a; break;
        case 0x615b: wc = 0x863c; break;
        case 0x615c: wc = 0x875e; break;
        case 0x615d: wc = 0x89b9; break;
        case 0x615e: wc = 0x8ebe; break;
        case 0x615f: wc = 0x90ff; break;
        case 0x6160: wc = 0x919a; break;
        case 0x6161: wc = 0x91be; break;
        case 0x6162: wc = 0x92a4; break;
        case 0x6163: wc = 0x9382; break;
        case 0x6164: wc = 0x973a; break;
        case 0x6165: wc = 0x9b87; break;
        case 0x6166: wc = 0x9da5; break;
        case 0x6167: wc = 0x9e8b; break;
        case 0x6168: wc = 0x9e8a; break;
        case 0x6169: wc = 0x9e9b; break;
        case 0x616a: wc = 0x50f6; break;
        case 0x616b: wc = 0x5221; break;
        case 0x616c: wc = 0x578a; break;
        case 0x616d: wc = 0x5d0f; break;
        case 0x616e: wc = 0x5fdf; break;
        case 0x616f: wc = 0x5fde; break;
        case 0x6170: wc = 0x600b; break;
        case 0x6171: wc = 0x615c; break;
        case 0x6172: wc = 0x62bf; break;
        case 0x6173: wc = 0x6543; break;
        case 0x6174: wc = 0x656f; break;
        case 0x6175: wc = 0x668b; break;
        case 0x6176: wc = 0x6f63; break;
        case 0x6177: wc = 0x6fa0; break;
        case 0x6178: wc = 0x71dc; break;
        case 0x6179: wc = 0x71d8; break;
        case 0x617a: wc = 0x741d; break;
        case 0x617b: wc = 0x7449; break;
        case 0x617c: wc = 0x75fb; break;
        case 0x617d: wc = 0x7b22; break;
        case 0x617e: wc = 0x7c22; break;
        case 0x617f: wc = 0x7f60; break;
        case 0x6180: wc = 0x82e0; break;
        case 0x6181: wc = 0x9231; break;
        case 0x6182: wc = 0x9309; break;
        case 0x6183: wc = 0x9372; break;
        case 0x6184: wc = 0x95a9; break;
        case 0x6185: wc = 0x4ab8; break;
        case 0x6186: wc = 0x9c35; break;
        case 0x6187: wc = 0x9efd; break;
        case 0x6188: wc = 0x5627; break;
        case 0x6189: wc = 0x6a12; break;
        case 0x618a: wc = 0x6ac1; break;
        case 0x618b: wc = 0x6ef5; break;
        case 0x618c: wc = 0x8524; break;
        case 0x618d: wc = 0x8820; break;
        case 0x618e: wc = 0x4eb3; break;
        case 0x618f: wc = 0x556a; break;
        case 0x6190: wc = 0x5657; break;
        case 0x6191: wc = 0x5697; break;
        case 0x6192: wc = 0x5724; break;
        case 0x6193: wc = 0x5f74; break;
        case 0x6194: wc = 0x61ea; break;
        case 0x6195: wc = 0x6b02; break;
        case 0x6196: wc = 0x70de; break;
        case 0x6197: wc = 0x717f; break;
        case 0x6198: wc = 0x7254; break;
        case 0x6199: wc = 0x72a6; break;
        case 0x619a: wc = 0x72db; break;
        case 0x619b: wc = 0x733c; break;
        case 0x619c: wc = 0x74df; break;
        case 0x619d: wc = 0x77a8; break;
        case 0x619e: wc = 0x7921; break;
        case 0x619f: wc = 0x7934; break;
        case 0x61a0: wc = 0x7a87; break;
        case 0x61a1: wc = 0x420f; break;
        case 0x61a2: wc = 0x7c19; break;
        case 0x61a3: wc = 0x260b8; break;
        case 0x61a4: wc = 0x80c9; break;
        case 0x61a5: wc = 0x8421; break;
        case 0x61a6: wc = 0x84aa; break;
        case 0x61a7: wc = 0x45da; break;
        case 0x61a8: wc = 0x896e; break;
        case 0x61a9: wc = 0x8db5; break;
        case 0x61aa: wc = 0x91d9; break;
        case 0x61ab: wc = 0x9251; break;
        case 0x61ac: wc = 0x939b; break;
        case 0x61ad: wc = 0x93f7; break;
        case 0x61ae: wc = 0x946e; break;
        case 0x61af: wc = 0x98ae; break;
        case 0x61b0: wc = 0x993a; break;
        case 0x61b1: wc = 0x99ee; break;
        case 0x61b2: wc = 0x9ab2; break;
        case 0x61b3: wc = 0x9ac6; break;
        case 0x61b4: wc = 0x51b8; break;
        case 0x61b5: wc = 0x5762; break;
        case 0x61b6: wc = 0x59c5; break;
        case 0x61b7: wc = 0x5abb; break;
        case 0x61b8: wc = 0x5b14; break;
        case 0x61b9: wc = 0x5e4b; break;
        case 0x61ba: wc = 0x6011; break;
        case 0x61bb: wc = 0x6273; break;
        case 0x61bc: wc = 0x642b; break;
        case 0x61bd: wc = 0x653d; break;
        case 0x61be: wc = 0x6592; break;
        case 0x61bf: wc = 0x67c8; break;
        case 0x61c0: wc = 0x3bbd; break;
        case 0x61c1: wc = 0x6c9c; break;
        case 0x61c2: wc = 0x6ebf; break;
        case 0x61c3: wc = 0x700a; break;
        case 0x61c4: wc = 0x7249; break;
        case 0x61c5: wc = 0x764d; break;
        case 0x61c6: wc = 0x24e8f; break;
        case 0x61c7: wc = 0x7e0f; break;
        case 0x61c8: wc = 0x80d6; break;
        case 0x61c9: wc = 0x84b0; break;
        case 0x61ca: wc = 0x8668; break;
        case 0x61cb: wc = 0x878c; break;
        case 0x61cc: wc = 0x8929; break;
        case 0x61cd: wc = 0x897b; break;
        case 0x61ce: wc = 0x8a4a; break;
        case 0x61cf: wc = 0x8dd8; break;
        case 0x61d0: wc = 0x8e63; break;
        case 0x61d1: wc = 0x9261; break;
        case 0x61d2: wc = 0x95c6; break;
        case 0x61d3: wc = 0x977d; break;
        case 0x61d4: wc = 0x97b6; break;
        case 0x61d5: wc = 0x9816; break;
        case 0x61d6: wc = 0x9b6c; break;
        case 0x61d7: wc = 0x9cfb; break;
        case 0x61d8: wc = 0x54f1; break;
        case 0x61d9: wc = 0x39ca; break;
        case 0x61da: wc = 0x632c; break;
        case 0x61db: wc = 0x3a8d; break;
        case 0x61dc: wc = 0x67ed; break;
        case 0x61dd: wc = 0x6872; break;
        case 0x61de: wc = 0x23b77; break;
        case 0x61df: wc = 0x6cb7; break;
        case 0x61e0: wc = 0x6d61; break;
        case 0x61e1: wc = 0x3db1; break;
        case 0x61e2: wc = 0x72ae; break;
        case 0x61e3: wc = 0x7676; break;
        case 0x61e4: wc = 0x7679; break;
        case 0x61e5: wc = 0x76cb; break;
        case 0x61e6: wc = 0x253b1; break;
        case 0x61e7: wc = 0x7835; break;
        case 0x61e8: wc = 0x79e1; break;
        case 0x61e9: wc = 0x7d8d; break;
        case 0x61ea: wc = 0x7f3d; break;
        case 0x61eb: wc = 0x80c8; break;
        case 0x61ec: wc = 0x8116; break;
        case 0x61ed: wc = 0x8307; break;
        case 0x61ee: wc = 0x8378; break;
        case 0x61ef: wc = 0x83dd; break;
        case 0x61f0: wc = 0x8467; break;
        case 0x61f1: wc = 0x889a; break;
        case 0x61f2: wc = 0x894f; break;
        case 0x61f3: wc = 0x8a59; break;
        case 0x61f4: wc = 0x8e73; break;
        case 0x61f5: wc = 0x8ef7; break;
        case 0x61f6: wc = 0x90e3; break;
        case 0x61f7: wc = 0x9238; break;
        case 0x61f8: wc = 0x92cd; break;
        case 0x61f9: wc = 0x93fa; break;
        case 0x61fa: wc = 0x9911; break;
        case 0x61fb: wc = 0x999e; break;
        case 0x61fc: wc = 0x4bcb; break;
        case 0x61fd: wc = 0x9b81; break;
        case 0x61fe: wc = 0x9c4d; break;
        case 0x61ff: wc = 0x9d53; break;
        case 0x6200: wc = 0x9f25; break;
        case 0x6201: wc = 0x6e74; break;
        case 0x6202: wc = 0x8e2b; break;
        case 0x6203: wc = 0x4eff; break;
        case 0x6204: wc = 0x531a; break;
        case 0x6205: wc = 0x5396; break;
        case 0x6206: wc = 0x54e4; break;
        case 0x6207: wc = 0x55d9; break;
        case 0x6208: wc = 0x57c5; break;
        case 0x6209: wc = 0x57b9; break;
        case 0x620a: wc = 0x580f; break;
        case 0x620b: wc = 0x5844; break;
        case 0x620c: wc = 0x585d; break;
        case 0x620d: wc = 0x5e2e; break;
        case 0x620e: wc = 0x5e6b; break;
        case 0x620f: wc = 0x5eac; break;
        case 0x6210: wc = 0x5fac; break;
        case 0x6211: wc = 0x396c; break;
        case 0x6212: wc = 0x6412; break;
        case 0x6213: wc = 0x23045; break;
        case 0x6214: wc = 0x65ca; break;
        case 0x6215: wc = 0x6886; break;
        case 0x6216: wc = 0x6c06; break;
        case 0x6217: wc = 0x6c78; break;
        case 0x6218: wc = 0x6dd3; break;
        case 0x6219: wc = 0x245d0; break;
        case 0x621a: wc = 0x7253; break;
        case 0x621b: wc = 0x7265; break;
        case 0x621c: wc = 0x727b; break;
        case 0x621d: wc = 0x73a4; break;
        case 0x621e: wc = 0x74ec; break;
        case 0x621f: wc = 0x7be3; break;
        case 0x6220: wc = 0x7d81; break;
        case 0x6221: wc = 0x7e0d; break;
        case 0x6222: wc = 0x802a; break;
        case 0x6223: wc = 0x80a8; break;
        case 0x6224: wc = 0x8196; break;
        case 0x6225: wc = 0x823d; break;
        case 0x6226: wc = 0x8255; break;
        case 0x6227: wc = 0x8684; break;
        case 0x6228: wc = 0x86d6; break;
        case 0x6229: wc = 0x8783; break;
        case 0x622a: wc = 0x9004; break;
        case 0x622b: wc = 0x90a1; break;
        case 0x622c: wc = 0x9201; break;
        case 0x622d: wc = 0x933a; break;
        case 0x622e: wc = 0x938a; break;
        case 0x622f: wc = 0x96f1; break;
        case 0x6230: wc = 0x4a26; break;
        case 0x6231: wc = 0x97a4; break;
        case 0x6232: wc = 0x99f9; break;
        case 0x6233: wc = 0x9ac8; break;
        case 0x6234: wc = 0x9ae3; break;
        case 0x6235: wc = 0x9b74; break;
        case 0x6236: wc = 0x9dad; break;
        case 0x6237: wc = 0x4f13; break;
        case 0x6238: wc = 0x4fd6; break;
        case 0x6239: wc = 0x505d; break;
        case 0x623a: wc = 0x5561; break;
        case 0x623b: wc = 0x574f; break;
        case 0x623c: wc = 0x576f; break;
        case 0x623d: wc = 0x6252; break;
        case 0x623e: wc = 0x62dd; break;
        case 0x623f: wc = 0x63b0; break;
        case 0x6240: wc = 0x63f9; break;
        case 0x6241: wc = 0x67f8; break;
        case 0x6242: wc = 0x686e; break;
        case 0x6243: wc = 0x6bf0; break;
        case 0x6244: wc = 0x73af; break;
        case 0x6245: wc = 0x7423; break;
        case 0x6246: wc = 0x7432; break;
        case 0x6247: wc = 0x3fa6; break;
        case 0x6248: wc = 0x789a; break;
        case 0x6249: wc = 0x7999; break;
        case 0x624a: wc = 0x25d54; break;
        case 0x624b: wc = 0x80a7; break;
        case 0x624c: wc = 0x8356; break;
        case 0x624d: wc = 0x84d3; break;
        case 0x624e: wc = 0x4512; break;
        case 0x624f: wc = 0x8843; break;
        case 0x6250: wc = 0x9185; break;
        case 0x6251: wc = 0x5161; break;
        case 0x6252: wc = 0x34e6; break;
        case 0x6253: wc = 0x5cb6; break;
        case 0x6254: wc = 0x6300; break;
        case 0x6255: wc = 0x6d26; break;
        case 0x6256: wc = 0x74f8; break;
        case 0x6257: wc = 0x7ae1; break;
        case 0x6258: wc = 0x7ca8; break;
        case 0x6259: wc = 0x7d54; break;
        case 0x625a: wc = 0x82e9; break;
        case 0x625b: wc = 0x9286; break;
        case 0x625c: wc = 0x50e0; break;
        case 0x625d: wc = 0x52eb; break;
        case 0x625e: wc = 0x58a6; break;
        case 0x625f: wc = 0x597f; break;
        case 0x6260: wc = 0x5b0f; break;
        case 0x6261: wc = 0x65db; break;
        case 0x6262: wc = 0x6a4e; break;
        case 0x6263: wc = 0x702a; break;
        case 0x6264: wc = 0x703f; break;
        case 0x6265: wc = 0x3e0b; break;
        case 0x6266: wc = 0x74a0; break;
        case 0x6267: wc = 0x7c53; break;
        case 0x6268: wc = 0x7e59; break;
        case 0x6269: wc = 0x7fb3; break;
        case 0x626a: wc = 0x7ffb; break;
        case 0x626b: wc = 0x81b0; break;
        case 0x626c: wc = 0x85a0; break;
        case 0x626d: wc = 0x8629; break;
        case 0x626e: wc = 0x881c; break;
        case 0x626f: wc = 0x88a2; break;
        case 0x6270: wc = 0x8e6f; break;
        case 0x6271: wc = 0x8f53; break;
        case 0x6272: wc = 0x9407; break;
        case 0x6273: wc = 0x9c55; break;
        case 0x6274: wc = 0x9ded; break;
        case 0x6275: wc = 0x57a1; break;
        case 0x6276: wc = 0x6a43; break;
        case 0x6277: wc = 0x6d4c; break;
        case 0x6278: wc = 0x7782; break;
        case 0x6279: wc = 0x85c5; break;
        case 0x627a: wc = 0x201a9; break;
        case 0x627b: wc = 0x674b; break;
        case 0x627c: wc = 0x67c9; break;
        case 0x627d: wc = 0x6efc; break;
        case 0x627e: wc = 0x7b35; break;
        case 0x627f: wc = 0x8ed3; break;
        case 0x6280: wc = 0x91e9; break;
        case 0x6281: wc = 0x98bf; break;
        case 0x6282: wc = 0x29afc; break;
        case 0x6283: wc = 0x75ba; break;
        case 0x6284: wc = 0x567c; break;
        case 0x6285: wc = 0x3620; break;
        case 0x6286: wc = 0x581b; break;
        case 0x6287: wc = 0x5ee6; break;
        case 0x6288: wc = 0x61b5; break;
        case 0x6289: wc = 0x63ca; break;
        case 0x628a: wc = 0x64d7; break;
        case 0x628b: wc = 0x6e62; break;
        case 0x628c: wc = 0x6fbc; break;
        case 0x628d: wc = 0x7513; break;
        case 0x628e: wc = 0x7695; break;
        case 0x628f: wc = 0x7914; break;
        case 0x6290: wc = 0x7a2b; break;
        case 0x6291: wc = 0x7cea; break;
        case 0x6292: wc = 0x7dbc; break;
        case 0x6293: wc = 0x7e74; break;
        case 0x6294: wc = 0x859c; break;
        case 0x6295: wc = 0x895e; break;
        case 0x6296: wc = 0x8e84; break;
        case 0x6297: wc = 0x8f9f; break;
        case 0x6298: wc = 0x91fd; break;
        case 0x6299: wc = 0x92a2; break;
        case 0x629a: wc = 0x9434; break;
        case 0x629b: wc = 0x9dff; break;
        case 0x629c: wc = 0x9f0a; break;
        case 0x629d: wc = 0x533e; break;
        case 0x629e: wc = 0x5909; break;
        case 0x629f: wc = 0x5fed; break;
        case 0x62a0: wc = 0x6283; break;
        case 0x62a1: wc = 0x62da; break;
        case 0x62a2: wc = 0x662a; break;
        case 0x62a3: wc = 0x6c74; break;
        case 0x62a4: wc = 0x6c73; break;
        case 0x62a5: wc = 0x72bf; break;
        case 0x62a6: wc = 0x7502; break;
        case 0x62a7: wc = 0x7a28; break;
        case 0x62a8: wc = 0x7b32; break;
        case 0x62a9: wc = 0x7c69; break;
        case 0x62aa: wc = 0x8fae; break;
        case 0x62ab: wc = 0x80fc; break;
        case 0x62ac: wc = 0x85ca; break;
        case 0x62ad: wc = 0x8cc6; break;
        case 0x62ae: wc = 0x8df0; break;
        case 0x62af: wc = 0x8fba; break;
        case 0x62b0: wc = 0x91c6; break;
        case 0x62b1: wc = 0x959e; break;
        case 0x62b2: wc = 0x9abf; break;
        case 0x62b3: wc = 0x9d18; break;
        case 0x62b4: wc = 0x4e3f; break;
        case 0x62b5: wc = 0x5af3; break;
        case 0x62b6: wc = 0x5f46; break;
        case 0x62b7: wc = 0x5fb6; break;
        case 0x62b8: wc = 0x618b; break;
        case 0x62b9: wc = 0x6487; break;
        case 0x62ba: wc = 0x6f4e; break;
        case 0x62bb: wc = 0x765f; break;
        case 0x62bc: wc = 0x8382; break;
        case 0x62bd: wc = 0x864c; break;
        case 0x62be: wc = 0x86c2; break;
        case 0x62bf: wc = 0x87de; break;
        case 0x62c0: wc = 0x8952; break;
        case 0x62c1: wc = 0x8995; break;
        case 0x62c2: wc = 0x8e69; break;
        case 0x62c3: wc = 0x9de9; break;
        case 0x62c4: wc = 0x4e26; break;
        case 0x62c5: wc = 0x4e53; break;
        case 0x62c6: wc = 0x4e52; break;
        case 0x62c7: wc = 0x504b; break;
        case 0x62c8: wc = 0x5840; break;
        case 0x62c9: wc = 0x5bce; break;
        case 0x62ca: wc = 0x5e32; break;
        case 0x62cb: wc = 0x5eb0; break;
        case 0x62cc: wc = 0x6032; break;
        case 0x62cd: wc = 0x6072; break;
        case 0x62ce: wc = 0x62a6; break;
        case 0x62cf: wc = 0x6360; break;
        case 0x62d0: wc = 0x62fc; break;
        case 0x62d1: wc = 0x6452; break;
        case 0x62d2: wc = 0x681f; break;
        case 0x62d3: wc = 0x692a; break;
        case 0x62d4: wc = 0x6d34; break;
        case 0x62d5: wc = 0x7415; break;
        case 0x62d6: wc = 0x7539; break;
        case 0x62d7: wc = 0x7a89; break;
        case 0x62d8: wc = 0x7aee; break;
        case 0x62d9: wc = 0x7d63; break;
        case 0x62da: wc = 0x7f3e; break;
        case 0x62db: wc = 0xfab1; break;
        case 0x62dc: wc = 0x8060; break;
        case 0x62dd: wc = 0x8275; break;
        case 0x62de: wc = 0x82ea; break;
        case 0x62df: wc = 0x8353; break;
        case 0x62e0: wc = 0x86c3; break;
        case 0x62e1: wc = 0x86e2; break;
        case 0x62e2: wc = 0x8a81; break;
        case 0x62e3: wc = 0x8eff; break;
        case 0x62e4: wc = 0x8ff8; break;
        case 0x62e5: wc = 0x90b4; break;
        case 0x62e6: wc = 0x90f1; break;
        case 0x62e7: wc = 0x9235; break;
        case 0x62e8: wc = 0x927c; break;
        case 0x62e9: wc = 0x92f2; break;
        case 0x62ea: wc = 0x2f9ea; break;
        case 0x62eb: wc = 0x9643; break;
        case 0x62ec: wc = 0x9829; break;
        case 0x62ed: wc = 0x9905; break;
        case 0x62ee: wc = 0x9d67; break;
        case 0x62ef: wc = 0x4fcc; break;
        case 0x62f0: wc = 0x5aac; break;
        case 0x62f1: wc = 0x5b9d; break;
        case 0x62f2: wc = 0x6357; break;
        case 0x62f3: wc = 0x693a; break;
        case 0x62f4: wc = 0x70f3; break;
        case 0x62f5: wc = 0x7172; break;
        case 0x62f6: wc = 0x402f; break;
        case 0x62f7: wc = 0x7c20; break;
        case 0x62f8: wc = 0x7de5; break;
        case 0x62f9: wc = 0x8386; break;
        case 0x62fa: wc = 0x8446; break;
        case 0x62fb: wc = 0x8554; break;
        case 0x62fc: wc = 0x85f5; break;
        case 0x62fd: wc = 0x86a5; break;
        case 0x62fe: wc = 0x8705; break;
        case 0x62ff: wc = 0x8e04; break;
        case 0x6300: wc = 0x9420; break;
        case 0x6301: wc = 0x4a49; break;
        case 0x6302: wc = 0x99c2; break;
        case 0x6303: wc = 0x9cf5; break;
        case 0x6304: wc = 0x9d07; break;
        case 0x6305: wc = 0x9efc; break;
        case 0x6306: wc = 0x5798; break;
        case 0x6307: wc = 0x58a3; break;
        case 0x6308: wc = 0x590d; break;
        case 0x6309: wc = 0x5e5e; break;
        case 0x630a: wc = 0x3864; break;
        case 0x630b: wc = 0x6251; break;
        case 0x630c: wc = 0x6534; break;
        case 0x630d: wc = 0x6535; break;
        case 0x630e: wc = 0x683f; break;
        case 0x630f: wc = 0x68f4; break;
        case 0x6310: wc = 0x6945; break;
        case 0x6311: wc = 0x6f93; break;
        case 0x6312: wc = 0x23fb4; break;
        case 0x6313: wc = 0x6fee; break;
        case 0x6314: wc = 0x735b; break;
        case 0x6315: wc = 0x250e8; break;
        case 0x6316: wc = 0x7b99; break;
        case 0x6317: wc = 0x7e80; break;
        case 0x6318: wc = 0x83d4; break;
        case 0x6319: wc = 0x844d; break;
        case 0x631a: wc = 0x8567; break;
        case 0x631b: wc = 0x8659; break;
        case 0x631c: wc = 0x8672; break;
        case 0x631d: wc = 0x8760; break;
        case 0x631e: wc = 0x876e; break;
        case 0x631f: wc = 0x88b1; break;
        case 0x6320: wc = 0x8946; break;
        case 0x6321: wc = 0x8984; break;
        case 0x6322: wc = 0x8ae8; break;
        case 0x6323: wc = 0x8d0c; break;
        case 0x6324: wc = 0x8e3e; break;
        case 0x6325: wc = 0x8e7c; break;
        case 0x6326: wc = 0x8f50; break;
        case 0x6327: wc = 0x91ad; break;
        case 0x6328: wc = 0x9351; break;
        case 0x6329: wc = 0x4a31; break;
        case 0x632a: wc = 0x9cea; break;
        case 0x632b: wc = 0x9d69; break;
        case 0x632c: wc = 0x9d9d; break;
        case 0x632d: wc = 0x9e14; break;
        case 0x632e: wc = 0x2a588; break;
        case 0x632f: wc = 0x5460; break;
        case 0x6330: wc = 0x7d4a; break;
        case 0x6331: wc = 0x20006; break;
        case 0x6332: wc = 0x4e30; break;
        case 0x6333: wc = 0x552a; break;
        case 0x6334: wc = 0x57c4; break;
        case 0x6335: wc = 0x57f2; break;
        case 0x6336: wc = 0x585c; break;
        case 0x6337: wc = 0x5906; break;
        case 0x6338: wc = 0x59a6; break;
        case 0x6339: wc = 0x5d36; break;
        case 0x633a: wc = 0x6340; break;
        case 0x633b: wc = 0x6453; break;
        case 0x633c: wc = 0x687b; break;
        case 0x633d: wc = 0x6a25; break;
        case 0x633e: wc = 0x69f0; break;
        case 0x633f: wc = 0x6e84; break;
        case 0x6340: wc = 0x6f28; break;
        case 0x6341: wc = 0x7128; break;
        case 0x6342: wc = 0x728e; break;
        case 0x6343: wc = 0x7412; break;
        case 0x6344: wc = 0x7bc8; break;
        case 0x6345: wc = 0x7bf7; break;
        case 0x6346: wc = 0x8242; break;
        case 0x6347: wc = 0x8283; break;
        case 0x6348: wc = 0x83f6; break;
        case 0x6349: wc = 0x8451; break;
        case 0x634a: wc = 0x8982; break;
        case 0x634b: wc = 0x8cf5; break;
        case 0x634c: wc = 0x93bd; break;
        case 0x634d: wc = 0x945d; break;
        case 0x634e: wc = 0x979b; break;
        case 0x634f: wc = 0x97f8; break;
        case 0x6350: wc = 0x4b70; break;
        case 0x6351: wc = 0x9afc; break;
        case 0x6352: wc = 0x4ec6; break;
        case 0x6353: wc = 0x4f15; break;
        case 0x6354: wc = 0x4fd8; break;
        case 0x6355: wc = 0x5069; break;
        case 0x6356: wc = 0x527b; break;
        case 0x6357: wc = 0x5ded; break;
        case 0x6358: wc = 0x5425; break;
        case 0x6359: wc = 0x356e; break;
        case 0x635a: wc = 0x56a9; break;
        case 0x635b: wc = 0x577f; break;
        case 0x635c: wc = 0x57ba; break;
        case 0x635d: wc = 0x59c7; break;
        case 0x635e: wc = 0x5a44; break;
        case 0x635f: wc = 0x5c03; break;
        case 0x6360: wc = 0x5cca; break;
        case 0x6361: wc = 0x5ecd; break;
        case 0x6362: wc = 0x5f23; break;
        case 0x6363: wc = 0x6000; break;
        case 0x6364: wc = 0x6024; break;
        case 0x6365: wc = 0x6294; break;
        case 0x6366: wc = 0x62ca; break;
        case 0x6367: wc = 0x634a; break;
        case 0x6368: wc = 0x638a; break;
        case 0x6369: wc = 0x636c; break;
        case 0x636a: wc = 0x668a; break;
        case 0x636b: wc = 0x678e; break;
        case 0x636c: wc = 0x67ce; break;
        case 0x636d: wc = 0x6874; break;
        case 0x636e: wc = 0x68d3; break;
        case 0x636f: wc = 0x6991; break;
        case 0x6370: wc = 0x6b95; break;
        case 0x6371: wc = 0x6ced; break;
        case 0x6372: wc = 0x6daa; break;
        case 0x6373: wc = 0x6ecf; break;
        case 0x6374: wc = 0x7f39; break;
        case 0x6375: wc = 0x70f0; break;
        case 0x6376: wc = 0x739e; break;
        case 0x6377: wc = 0x73b8; break;
        case 0x6378: wc = 0x7408; break;
        case 0x6379: wc = 0x74b7; break;
        case 0x637a: wc = 0x74ff; break;
        case 0x637b: wc = 0x75e1; break;
        case 0x637c: wc = 0x7806; break;
        case 0x637d: wc = 0x7954; break;
        case 0x637e: wc = 0x79a3; break;
        case 0x637f: wc = 0x7a03; break;
        case 0x6380: wc = 0x7bf0; break;
        case 0x6381: wc = 0x7cb0; break;
        case 0x6382: wc = 0x7cd0; break;
        case 0x6383: wc = 0x7d11; break;
        case 0x6384: wc = 0x7d28; break;
        case 0x6385: wc = 0x7d92; break;
        case 0x6386: wc = 0x7dee; break;
        case 0x6387: wc = 0x7f58; break;
        case 0x6388: wc = 0x7f66; break;
        case 0x6389: wc = 0x80d5; break;
        case 0x638a: wc = 0x82a3; break;
        case 0x638b: wc = 0x82fb; break;
        case 0x638c: wc = 0x8374; break;
        case 0x638d: wc = 0x842f; break;
        case 0x638e: wc = 0x8500; break;
        case 0x638f: wc = 0x86a8; break;
        case 0x6390: wc = 0x86b9; break;
        case 0x6391: wc = 0x86d7; break;
        case 0x6392: wc = 0x8709; break;
        case 0x6393: wc = 0x875c; break;
        case 0x6394: wc = 0x886d; break;
        case 0x6395: wc = 0x88d2; break;
        case 0x6396: wc = 0x8914; break;
        case 0x6397: wc = 0x8dd7; break;
        case 0x6398: wc = 0x909e; break;
        case 0x6399: wc = 0x90d9; break;
        case 0x639a: wc = 0x90db; break;
        case 0x639b: wc = 0x911c; break;
        case 0x639c: wc = 0x915c; break;
        case 0x639d: wc = 0x9207; break;
        case 0x639e: wc = 0x9307; break;
        case 0x639f: wc = 0x961d; break;
        case 0x63a0: wc = 0x4a96; break;
        case 0x63a1: wc = 0x982b; break;
        case 0x63a2: wc = 0x9b34; break;
        case 0x63a3: wc = 0x9b92; break;
        case 0x63a4: wc = 0x9d00; break;
        case 0x63a5: wc = 0x9cfa; break;
        case 0x63a6: wc = 0x9ea9; break;
        case 0x63a7: wc = 0x50f0; break;
        case 0x63a8: wc = 0x8e23; break;
        case 0x63a9: wc = 0x50e8; break;
        case 0x63aa: wc = 0x55af; break;
        case 0x63ab: wc = 0x574c; break;
        case 0x63ac: wc = 0x574b; break;
        case 0x63ad: wc = 0x59a2; break;
        case 0x63ae: wc = 0x5e69; break;
        case 0x63af: wc = 0x5f05; break;
        case 0x63b0: wc = 0x678c; break;
        case 0x63b1: wc = 0x68a4; break;
        case 0x63b2: wc = 0x68fb; break;
        case 0x63b3: wc = 0x68fc; break;
        case 0x63b4: wc = 0x6a68; break;
        case 0x63b5: wc = 0x6b55; break;
        case 0x63b6: wc = 0x6c1b; break;
        case 0x63b7: wc = 0x6ccd; break;
        case 0x63b8: wc = 0x6e00; break;
        case 0x63b9: wc = 0x6e53; break;
        case 0x63ba: wc = 0x6fc6; break;
        case 0x63bb: wc = 0x7035; break;
        case 0x63bc: wc = 0x7083; break;
        case 0x63bd: wc = 0x7287; break;
        case 0x63be: wc = 0x7356; break;
        case 0x63bf: wc = 0x73a2; break;
        case 0x63c0: wc = 0x74f0; break;
        case 0x63c1: wc = 0x755a; break;
        case 0x63c2: wc = 0x780f; break;
        case 0x63c3: wc = 0x79ce; break;
        case 0x63c4: wc = 0x7ad5; break;
        case 0x63c5: wc = 0x7b28; break;
        case 0x63c6: wc = 0x7f92; break;
        case 0x63c7: wc = 0x7fb5; break;
        case 0x63c8: wc = 0x7fc2; break;
        case 0x63c9: wc = 0x7fc9; break;
        case 0x63ca: wc = 0x7ff8; break;
        case 0x63cb: wc = 0x80a6; break;
        case 0x63cc: wc = 0x81b9; break;
        case 0x63cd: wc = 0x82ef; break;
        case 0x63ce: wc = 0x8450; break;
        case 0x63cf: wc = 0x84b6; break;
        case 0x63d0: wc = 0x8561; break;
        case 0x63d1: wc = 0x86a1; break;
        case 0x63d2: wc = 0x86a0; break;
        case 0x63d3: wc = 0x886f; break;
        case 0x63d4: wc = 0x8c76; break;
        case 0x63d5: wc = 0x8eec; break;
        case 0x63d6: wc = 0x8f52; break;
        case 0x63d7: wc = 0x915a; break;
        case 0x63d8: wc = 0x9216; break;
        case 0x63d9: wc = 0x931b; break;
        case 0x63da: wc = 0x943c; break;
        case 0x63db: wc = 0x9959; break;
        case 0x63dc: wc = 0x99a9; break;
        case 0x63dd: wc = 0x9b75; break;
        case 0x63de: wc = 0x9c5d; break;
        case 0x63df: wc = 0x9ec2; break;
        case 0x63e0: wc = 0x9efa; break;
        case 0x63e1: wc = 0x9f16; break;
        case 0x63e2: wc = 0x9f22; break;
        case 0x63e3: wc = 0x4e40; break;
        case 0x63e4: wc = 0x4ecf; break;
        case 0x63e5: wc = 0x51b9; break;
        case 0x63e6: wc = 0x521c; break;
        case 0x63e7: wc = 0x5488; break;
        case 0x63e8: wc = 0x5772; break;
        case 0x63e9: wc = 0x5caa; break;
        case 0x63ea: wc = 0x5dff; break;
        case 0x63eb: wc = 0x5e17; break;
        case 0x63ec: wc = 0x602b; break;
        case 0x63ed: wc = 0x6622; break;
        case 0x63ee: wc = 0x67eb; break;
        case 0x63ef: wc = 0x6c1f; break;
        case 0x63f0: wc = 0x7158; break;
        case 0x63f1: wc = 0x7953; break;
        case 0x63f2: wc = 0x7b30; break;
        case 0x63f3: wc = 0x7d3c; break;
        case 0x63f4: wc = 0x7d31; break;
        case 0x63f5: wc = 0x7fc7; break;
        case 0x63f6: wc = 0x8274; break;
        case 0x63f7: wc = 0x82be; break;
        case 0x63f8: wc = 0x8300; break;
        case 0x63f9: wc = 0x97cd; break;
        case 0x63fa: wc = 0x9af4; break;
        case 0x63fb: wc = 0x9efb; break;
        case 0x63fc: wc = 0x5017; break;
        case 0x63fd: wc = 0x50b0; break;
        case 0x63fe: wc = 0x5623; break;
        case 0x63ff: wc = 0x580b; break;
        case 0x6400: wc = 0x5874; break;
        case 0x6401: wc = 0x5f38; break;
        case 0x6402: wc = 0x63a4; break;
        case 0x6403: wc = 0x6f30; break;
        case 0x6404: wc = 0x7129; break;
        case 0x6405: wc = 0x75ed; break;
        case 0x6406: wc = 0x8e66; break;
        case 0x6407: wc = 0x93f0; break;
        case 0x6408: wc = 0x9b05; break;
        case 0x6409: wc = 0x4ef3; break;
        case 0x640a: wc = 0x4f3e; break;
        case 0x640b: wc = 0x4ffe; break;
        case 0x640c: wc = 0x4ff7; break;
        case 0x640d: wc = 0x5255; break;
        case 0x640e: wc = 0x539e; break;
        case 0x640f: wc = 0x5478; break;
        case 0x6410: wc = 0x5564; break;
        case 0x6411: wc = 0x555a; break;
        case 0x6412: wc = 0x568a; break;
        case 0x6413: wc = 0x56ad; break;
        case 0x6414: wc = 0x572e; break;
        case 0x6415: wc = 0x5752; break;
        case 0x6416: wc = 0x57e4; break;
        case 0x6417: wc = 0x595c; break;
        case 0x6418: wc = 0x5970; break;
        case 0x6419: wc = 0x59a3; break;
        case 0x641a: wc = 0x5a53; break;
        case 0x641b: wc = 0x5ab2; break;
        case 0x641c: wc = 0x5b36; break;
        case 0x641d: wc = 0x5c41; break;
        case 0x641e: wc = 0x5c44; break;
        case 0x641f: wc = 0x5c5d; break;
        case 0x6420: wc = 0x5caf; break;
        case 0x6421: wc = 0x5d25; break;
        case 0x6422: wc = 0x5e80; break;
        case 0x6423: wc = 0x5eb3; break;
        case 0x6424: wc = 0x600c; break;
        case 0x6425: wc = 0x6082; break;
        case 0x6426: wc = 0x60b1; break;
        case 0x6427: wc = 0x3a4c; break;
        case 0x6428: wc = 0x6632; break;
        case 0x6429: wc = 0x66ca; break;
        case 0x642a: wc = 0x670f; break;
        case 0x642b: wc = 0x673c; break;
        case 0x642c: wc = 0x67f2; break;
        case 0x642d: wc = 0x68d0; break;
        case 0x642e: wc = 0x6911; break;
        case 0x642f: wc = 0x698c; break;
        case 0x6430: wc = 0x6bde; break;
        case 0x6431: wc = 0x6c98; break;
        case 0x6432: wc = 0x6de0; break;
        case 0x6433: wc = 0x6ddd; break;
        case 0x6434: wc = 0x6e12; break;
        case 0x6435: wc = 0x6fde; break;
        case 0x6436: wc = 0x7295; break;
        case 0x6437: wc = 0x72c9; break;
        case 0x6438: wc = 0x72d2; break;
        case 0x6439: wc = 0x7540; break;
        case 0x643a: wc = 0x7595; break;
        case 0x643b: wc = 0x75aa; break;
        case 0x643c: wc = 0x75bf; break;
        case 0x643d: wc = 0x75de; break;
        case 0x643e: wc = 0x75f1; break;
        case 0x643f: wc = 0x75f9; break;
        case 0x6440: wc = 0x401d; break;
        case 0x6441: wc = 0x7765; break;
        case 0x6442: wc = 0x7955; break;
        case 0x6443: wc = 0x79e0; break;
        case 0x6444: wc = 0x7b13; break;
        case 0x6445: wc = 0x7b84; break;
        case 0x6446: wc = 0x7be6; break;
        case 0x6447: wc = 0x7bda; break;
        case 0x6448: wc = 0x7c8a; break;
        case 0x6449: wc = 0x7cd2; break;
        case 0x644a: wc = 0x7d15; break;
        case 0x644b: wc = 0x260b9; break;
        case 0x644c: wc = 0x7f86; break;
        case 0x644d: wc = 0x80b6; break;
        case 0x644e: wc = 0x8153; break;
        case 0x644f: wc = 0x818d; break;
        case 0x6450: wc = 0x4444; break;
        case 0x6451: wc = 0x822d; break;
        case 0x6452: wc = 0x8298; break;
        case 0x6453: wc = 0x82e4; break;
        case 0x6454: wc = 0x8409; break;
        case 0x6455: wc = 0x8406; break;
        case 0x6456: wc = 0x841e; break;
        case 0x6457: wc = 0x84d6; break;
        case 0x6458: wc = 0x868d; break;
        case 0x6459: wc = 0x86bd; break;
        case 0x645a: wc = 0x8731; break;
        case 0x645b: wc = 0x8730; break;
        case 0x645c: wc = 0x8795; break;
        case 0x645d: wc = 0x87b7; break;
        case 0x645e: wc = 0x87e6; break;
        case 0x645f: wc = 0x882f; break;
        case 0x6460: wc = 0x88f6; break;
        case 0x6461: wc = 0x8963; break;
        case 0x6462: wc = 0x8ac0; break;
        case 0x6463: wc = 0x8c4d; break;
        case 0x6464: wc = 0x8c7c; break;
        case 0x6465: wc = 0x8c94; break;
        case 0x6466: wc = 0x8d14; break;
        case 0x6467: wc = 0x8f61; break;
        case 0x6468: wc = 0x90b3; break;
        case 0x6469: wc = 0x90eb; break;
        case 0x646a: wc = 0x912a; break;
        case 0x646b: wc = 0x921a; break;
        case 0x646c: wc = 0x9208; break;
        case 0x646d: wc = 0x930d; break;
        case 0x646e: wc = 0x939e; break;
        case 0x646f: wc = 0x9428; break;
        case 0x6470: wc = 0x959f; break;
        case 0x6471: wc = 0x9630; break;
        case 0x6472: wc = 0x966b; break;
        case 0x6473: wc = 0x9674; break;
        case 0x6474: wc = 0x970f; break;
        case 0x6475: wc = 0x975f; break;
        case 0x6476: wc = 0x979e; break;
        case 0x6477: wc = 0x97b4; break;
        case 0x6478: wc = 0x9925; break;
        case 0x6479: wc = 0x99a1; break;
        case 0x647a: wc = 0x99d3; break;
        case 0x647b: wc = 0x9a11; break;
        case 0x647c: wc = 0x9a1b; break;
        case 0x647d: wc = 0x9ac0; break;
        case 0x647e: wc = 0x9aec; break;
        case 0x647f: wc = 0x9b7e; break;
        case 0x6480: wc = 0x9be1; break;
        case 0x6481: wc = 0x9f19; break;
        case 0x6482: wc = 0x4efd; break;
        case 0x6483: wc = 0x5110; break;
        case 0x6484: wc = 0x64ef; break;
        case 0x6485: wc = 0x6915; break;
        case 0x6486: wc = 0x74b8; break;
        case 0x6487: wc = 0x77c9; break;
        case 0x6488: wc = 0x7917; break;
        case 0x6489: wc = 0x7e7d; break;
        case 0x648a: wc = 0x81cf; break;
        case 0x648b: wc = 0x860b; break;
        case 0x648c: wc = 0x8819; break;
        case 0x648d: wc = 0x8c69; break;
        case 0x648e: wc = 0x8c73; break;
        case 0x648f: wc = 0x90a0; break;
        case 0x6490: wc = 0x944c; break;
        case 0x6491: wc = 0x9726; break;
        case 0x6492: wc = 0x9870; break;
        case 0x6493: wc = 0x99aa; break;
        case 0x6494: wc = 0x9a5e; break;
        case 0x6495: wc = 0x9ad5; break;
        case 0x6496: wc = 0x9b02; break;
        case 0x6497: wc = 0x9b22; break;
        case 0x6498: wc = 0x4fdc; break;
        case 0x6499: wc = 0x51ab; break;
        case 0x649a: wc = 0x51b0; break;
        case 0x649b: wc = 0x51ed; break;
        case 0x649c: wc = 0x5a09; break;
        case 0x649d: wc = 0x6cf5; break;
        case 0x649e: wc = 0x6d84; break;
        case 0x649f: wc = 0x6ddc; break;
        case 0x64a0: wc = 0x782f; break;
        case 0x64a1: wc = 0x517a; break;
        case 0x64a2: wc = 0x54db; break;
        case 0x64a3: wc = 0x4f58; break;
        case 0x64a4: wc = 0x201fb; break;
        case 0x64a5: wc = 0x5033; break;
        case 0x64a6: wc = 0x503d; break;
        case 0x64a7: wc = 0x509e; break;
        case 0x64a8: wc = 0x50bb; break;
        case 0x64a9: wc = 0x5129; break;
        case 0x64aa: wc = 0x525a; break;
        case 0x64ab: wc = 0x5378; break;
        case 0x64ac: wc = 0x5399; break;
        case 0x64ad: wc = 0x53b6; break;
        case 0x64ae: wc = 0x548b; break;
        case 0x64af: wc = 0x5526; break;
        case 0x64b0: wc = 0x5565; break;
        case 0x64b1: wc = 0x55b3; break;
        case 0x64b2: wc = 0x55c4; break;
        case 0x64b3: wc = 0x565d; break;
        case 0x64b4: wc = 0x586e; break;
        case 0x64b5: wc = 0x59d2; break;
        case 0x64b6: wc = 0x5c63; break;
        case 0x64b7: wc = 0x5ceb; break;
        case 0x64b8: wc = 0x623a; break;
        case 0x64b9: wc = 0x6331; break;
        case 0x64ba: wc = 0x6332; break;
        case 0x64bb: wc = 0x63f8; break;
        case 0x64bc: wc = 0x6463; break;
        case 0x64bd: wc = 0x67e4; break;
        case 0x64be: wc = 0x67cc; break;
        case 0x64bf: wc = 0x686b; break;
        case 0x64c0: wc = 0x6942; break;
        case 0x64c1: wc = 0x6952; break;
        case 0x64c2: wc = 0x69ad; break;
        case 0x64c3: wc = 0x69b9; break;
        case 0x64c4: wc = 0x69ce; break;
        case 0x64c5: wc = 0x6a1d; break;
        case 0x64c6: wc = 0x8082; break;
        case 0x64c7: wc = 0x6c5c; break;
        case 0x64c8: wc = 0x6cc0; break;
        case 0x64c9: wc = 0x6d0d; break;
        case 0x64ca: wc = 0x6d98; break;
        case 0x64cb: wc = 0x6dbb; break;
        case 0x64cc: wc = 0x6eae; break;
        case 0x64cd: wc = 0x6f07; break;
        case 0x64ce: wc = 0x707a; break;
        case 0x64cf: wc = 0x3e3a; break;
        case 0x64d0: wc = 0x7300; break;
        case 0x64d1: wc = 0x731e; break;
        case 0x64d2: wc = 0x7339; break;
        case 0x64d3: wc = 0x75e7; break;
        case 0x64d4: wc = 0x3fb4; break;
        case 0x64d5: wc = 0x76bb; break;
        case 0x64d6: wc = 0x781f; break;
        case 0x64d7: wc = 0x7870; break;
        case 0x64d8: wc = 0x78b4; break;
        case 0x64d9: wc = 0x78c3; break;
        case 0x64da: wc = 0x7997; break;
        case 0x64db: wc = 0x79a0; break;
        case 0x64dc: wc = 0x256d6; break;
        case 0x64dd: wc = 0x7ae2; break;
        case 0x64de: wc = 0x7b25; break;
        case 0x64df: wc = 0x7c11; break;
        case 0x64e0: wc = 0x7c01; break;
        case 0x64e1: wc = 0x7cf8; break;
        case 0x64e2: wc = 0x800d; break;
        case 0x64e3: wc = 0x801c; break;
        case 0x64e4: wc = 0x26b0a; break;
        case 0x64e5: wc = 0x838f; break;
        case 0x64e6: wc = 0x8478; break;
        case 0x64e7: wc = 0x8492; break;
        case 0x64e8: wc = 0x84f0; break;
        case 0x64e9: wc = 0x856c; break;
        case 0x64ea: wc = 0x857c; break;
        case 0x64eb: wc = 0x8652; break;
        case 0x64ec: wc = 0x8724; break;
        case 0x64ed: wc = 0x8721; break;
        case 0x64ee: wc = 0x8784; break;
        case 0x64ef: wc = 0x87f4; break;
        case 0x64f0: wc = 0x887a; break;
        case 0x64f1: wc = 0x8997; break;
        case 0x64f2: wc = 0x8c84; break;
        case 0x64f3: wc = 0x8cd6; break;
        case 0x64f4: wc = 0x8e45; break;
        case 0x64f5: wc = 0x8e5d; break;
        case 0x64f6: wc = 0x8ea7; break;
        case 0x64f7: wc = 0x9024; break;
        case 0x64f8: wc = 0x9248; break;
        case 0x64f9: wc = 0x9270; break;
        case 0x64fa: wc = 0x9236; break;
        case 0x64fb: wc = 0x492c; break;
        case 0x64fc: wc = 0x9366; break;
        case 0x64fd: wc = 0x4a7e; break;
        case 0x64fe: wc = 0x98e4; break;
        case 0x64ff: wc = 0x99db; break;
        case 0x6500: wc = 0x9a07; break;
        case 0x6501: wc = 0x9b73; break;
        case 0x6502: wc = 0x9b66; break;
        case 0x6503: wc = 0x9bbb; break;
        case 0x6504: wc = 0x9bca; break;
        case 0x6505: wc = 0x9c24; break;
        case 0x6506: wc = 0x9db3; break;
        case 0x6507: wc = 0x9de5; break;
        case 0x6508: wc = 0x9f36; break;
        case 0x6509: wc = 0x9f44; break;
        case 0x650a: wc = 0x55cd; break;
        case 0x650b: wc = 0x63f1; break;
        case 0x650c: wc = 0x6420; break;
        case 0x650d: wc = 0x69ca; break;
        case 0x650e: wc = 0x720d; break;
        case 0x650f: wc = 0x7361; break;
        case 0x6510: wc = 0x77df; break;
        case 0x6511: wc = 0x84b4; break;
        case 0x6512: wc = 0x9399; break;
        case 0x6513: wc = 0x9460; break;
        case 0x6514: wc = 0x5277; break;
        case 0x6515: wc = 0x5334; break;
        case 0x6516: wc = 0x5381; break;
        case 0x6517: wc = 0x5738; break;
        case 0x6518: wc = 0x59cd; break;
        case 0x6519: wc = 0x5d7c; break;
        case 0x651a: wc = 0x3983; break;
        case 0x651b: wc = 0x3a94; break;
        case 0x651c: wc = 0x6763; break;
        case 0x651d: wc = 0x3bc6; break;
        case 0x651e: wc = 0x6a75; break;
        case 0x651f: wc = 0x6efb; break;
        case 0x6520: wc = 0x6f78; break;
        case 0x6521: wc = 0x72e6; break;
        case 0x6522: wc = 0x72fb; break;
        case 0x6523: wc = 0x75e0; break;
        case 0x6524: wc = 0x7958; break;
        case 0x6525: wc = 0x7b6d; break;
        case 0x6526: wc = 0x7c05; break;
        case 0x6527: wc = 0x7e56; break;
        case 0x6528: wc = 0x8222; break;
        case 0x6529: wc = 0x4509; break;
        case 0x652a: wc = 0x8a15; break;
        case 0x652b: wc = 0x8d78; break;
        case 0x652c: wc = 0x8dda; break;
        case 0x652d: wc = 0x8ed5; break;
        case 0x652e: wc = 0x93df; break;
        case 0x652f: wc = 0x93fe; break;
        case 0x6530: wc = 0x9582; break;
        case 0x6531: wc = 0x958a; break;
        case 0x6532: wc = 0x340a; break;
        case 0x6533: wc = 0x644b; break;
        case 0x6534: wc = 0x6a27; break;
        case 0x6535: wc = 0x3b20; break;
        case 0x6536: wc = 0x8531; break;
        case 0x6537: wc = 0x8644; break;
        case 0x6538: wc = 0x95b7; break;
        case 0x6539: wc = 0x4ee8; break;
        case 0x653a: wc = 0x527c; break;
        case 0x653b: wc = 0x5e53; break;
        case 0x653c: wc = 0x5f61; break;
        case 0x653d: wc = 0x66d1; break;
        case 0x653e: wc = 0x69ee; break;
        case 0x653f: wc = 0x6bdd; break;
        case 0x6540: wc = 0x6bff; break;
        case 0x6541: wc = 0x6d81; break;
        case 0x6542: wc = 0x7a47; break;
        case 0x6543: wc = 0x7cc2; break;
        case 0x6544: wc = 0x7cdd; break;
        case 0x6545: wc = 0x7e3f; break;
        case 0x6546: wc = 0x7f67; break;
        case 0x6547: wc = 0x8460; break;
        case 0x6548: wc = 0x8942; break;
        case 0x6549: wc = 0x8d02; break;
        case 0x654a: wc = 0x91e4; break;
        case 0x654b: wc = 0x93d2; break;
        case 0x654c: wc = 0x9590; break;
        case 0x654d: wc = 0x29780; break;
        case 0x654e: wc = 0x9b16; break;
        case 0x654f: wc = 0x29df2; break;
        case 0x6550: wc = 0x505b; break;
        case 0x6551: wc = 0x5345; break;
        case 0x6552: wc = 0x553c; break;
        case 0x6553: wc = 0x55a2; break;
        case 0x6554: wc = 0x633f; break;
        case 0x6555: wc = 0x6b43; break;
        case 0x6556: wc = 0x6f80; break;
        case 0x6557: wc = 0x7250; break;
        case 0x6558: wc = 0x7b91; break;
        case 0x6559: wc = 0x7fdc; break;
        case 0x655a: wc = 0x7fe3; break;
        case 0x655b: wc = 0x81ff; break;
        case 0x655c: wc = 0x8410; break;
        case 0x655d: wc = 0x8b45; break;
        case 0x655e: wc = 0x8dbf; break;
        case 0x655f: wc = 0x48c9; break;
        case 0x6560: wc = 0x9364; break;
        case 0x6561: wc = 0x939d; break;
        case 0x6562: wc = 0x9705; break;
        case 0x6563: wc = 0x970e; break;
        case 0x6564: wc = 0x9778; break;
        case 0x6565: wc = 0x99ba; break;
        case 0x6566: wc = 0x4ee9; break;
        case 0x6567: wc = 0x50b8; break;
        case 0x6568: wc = 0x55d3; break;
        case 0x6569: wc = 0x5690; break;
        case 0x656a: wc = 0x57a7; break;
        case 0x656b: wc = 0x587d; break;
        case 0x656c: wc = 0x5892; break;
        case 0x656d: wc = 0x5f9c; break;
        case 0x656e: wc = 0x6066; break;
        case 0x656f: wc = 0x6113; break;
        case 0x6570: wc = 0x6161; break;
        case 0x6571: wc = 0x616f; break;
        case 0x6572: wc = 0x6421; break;
        case 0x6573: wc = 0x664c; break;
        case 0x6574: wc = 0x6a09; break;
        case 0x6575: wc = 0x6ba4; break;
        case 0x6576: wc = 0x6f3a; break;
        case 0x6577: wc = 0x6f21; break;
        case 0x6578: wc = 0x6f52; break;
        case 0x6579: wc = 0x7040; break;
        case 0x657a: wc = 0x747a; break;
        case 0x657b: wc = 0x78c9; break;
        case 0x657c: wc = 0x7935; break;
        case 0x657d: wc = 0x7dd7; break;
        case 0x657e: wc = 0x7e14; break;
        case 0x657f: wc = 0x8025; break;
        case 0x6580: wc = 0x8459; break;
        case 0x6581: wc = 0x850f; break;
        case 0x6582: wc = 0x87aa; break;
        case 0x6583: wc = 0x87d3; break;
        case 0x6584: wc = 0x8830; break;
        case 0x6585: wc = 0x892c; break;
        case 0x6586: wc = 0x8950; break;
        case 0x6587: wc = 0x8b2a; break;
        case 0x6588: wc = 0x8d18; break;
        case 0x6589: wc = 0x92ff; break;
        case 0x658a: wc = 0x939f; break;
        case 0x658b: wc = 0x93db; break;
        case 0x658c: wc = 0x93ef; break;
        case 0x658d: wc = 0x940c; break;
        case 0x658e: wc = 0x9859; break;
        case 0x658f: wc = 0x9a3b; break;
        case 0x6590: wc = 0x9a66; break;
        case 0x6591: wc = 0x9b3a; break;
        case 0x6592: wc = 0x9bd7; break;
        case 0x6593: wc = 0x9c4c; break;
        case 0x6594: wc = 0x9c68; break;
        case 0x6595: wc = 0x9dde; break;
        case 0x6596: wc = 0x4d02; break;
        case 0x6597: wc = 0x5d3d; break;
        case 0x6598: wc = 0x74d5; break;
        case 0x6599: wc = 0x46a1; break;
        case 0x659a: wc = 0x9c13; break;
        case 0x659b: wc = 0x55e6; break;
        case 0x659c: wc = 0x6475; break;
        case 0x659d: wc = 0x681c; break;
        case 0x659e: wc = 0x3c47; break;
        case 0x659f: wc = 0x6b6e; break;
        case 0x65a0: wc = 0x6cce; break;
        case 0x65a1: wc = 0x6fc7; break;
        case 0x65a2: wc = 0x77e0; break;
        case 0x65a3: wc = 0x8cfe; break;
        case 0x65a4: wc = 0x8f56; break;
        case 0x65a5: wc = 0x92ab; break;
        case 0x65a6: wc = 0x938d; break;
        case 0x65a7: wc = 0x9f5a; break;
        case 0x65a8: wc = 0x9f70; break;
        case 0x65a9: wc = 0x680d; break;
        case 0x65aa: wc = 0x6ce9; break;
        case 0x65ab: wc = 0x6e66; break;
        case 0x65ac: wc = 0x771a; break;
        case 0x65ad: wc = 0x82fc; break;
        case 0x65ae: wc = 0x8cb9; break;
        case 0x65af: wc = 0x924e; break;
        case 0x65b0: wc = 0x9f2a; break;
        case 0x65b1: wc = 0x4fc6; break;
        case 0x65b2: wc = 0x53d9; break;
        case 0x65b3: wc = 0x5511; break;
        case 0x65b4: wc = 0x566c; break;
        case 0x65b5: wc = 0x5715; break;
        case 0x65b6: wc = 0x57bf; break;
        case 0x65b7: wc = 0x5a7f; break;
        case 0x65b8: wc = 0x5aec; break;
        case 0x65b9: wc = 0xf0705; break;
        case 0x65ba: wc = 0x5c56; break;
        case 0x65bb: wc = 0x5f86; break;
        case 0x65bc: wc = 0x6053; break;
        case 0x65bd: wc = 0x6087; break;
        case 0x65be: wc = 0x3960; break;
        case 0x65bf: wc = 0x63df; break;
        case 0x65c0: wc = 0x6558; break;
        case 0x65c1: wc = 0x6948; break;
        case 0x65c2: wc = 0x6a28; break;
        case 0x65c3: wc = 0x6c25; break;
        case 0x65c4: wc = 0x6c80; break;
        case 0x65c5: wc = 0x6e51; break;
        case 0x65c6: wc = 0x6f35; break;
        case 0x65c7: wc = 0x6f7b; break;
        case 0x65c8: wc = 0x6fa8; break;
        case 0x65c9: wc = 0x7479; break;
        case 0x65ca: wc = 0x7659; break;
        case 0x65cb: wc = 0x7852; break;
        case 0x65cc: wc = 0x7a30; break;
        case 0x65cd: wc = 0x7c2d; break;
        case 0x65ce: wc = 0x7c9e; break;
        case 0x65cf: wc = 0x7cc8; break;
        case 0x65d0: wc = 0x7cec; break;
        case 0x65d1: wc = 0x7d13; break;
        case 0x65d2: wc = 0x7e03; break;
        case 0x65d3: wc = 0x8021; break;
        case 0x65d4: wc = 0x82a7; break;
        case 0x65d5: wc = 0x85c7; break;
        case 0x65d6: wc = 0x870d; break;
        case 0x65d7: wc = 0x8751; break;
        case 0x65d8: wc = 0x89e2; break;
        case 0x65d9: wc = 0x8add; break;
        case 0x65da: wc = 0x27d81; break;
        case 0x65db: wc = 0x907e; break;
        case 0x65dc: wc = 0x9191; break;
        case 0x65dd: wc = 0x924f; break;
        case 0x65de: wc = 0x93e3; break;
        case 0x65df: wc = 0x95aa; break;
        case 0x65e0: wc = 0x9c6e; break;
        case 0x65e1: wc = 0x9d68; break;
        case 0x65e2: wc = 0x4f66; break;
        case 0x65e3: wc = 0x5307; break;
        case 0x65e4: wc = 0x375c; break;
        case 0x65e5: wc = 0x60c1; break;
        case 0x65e6: wc = 0x6670; break;
        case 0x65e7: wc = 0x685e; break;
        case 0x65e8: wc = 0x68e4; break;
        case 0x65e9: wc = 0x6aa1; break;
        case 0x65ea: wc = 0x6caf; break;
        case 0x65eb: wc = 0x3cfb; break;
        case 0x65ec: wc = 0x711f; break;
        case 0x65ed: wc = 0x7699; break;
        case 0x65ee: wc = 0x77fd; break;
        case 0x65ef: wc = 0x794f; break;
        case 0x65f0: wc = 0x4137; break;
        case 0x65f1: wc = 0x7a78; break;
        case 0x65f2: wc = 0x7dc6; break;
        case 0x65f3: wc = 0x814a; break;
        case 0x65f4: wc = 0x8203; break;
        case 0x65f5: wc = 0x8312; break;
        case 0x65f6: wc = 0x83e5; break;
        case 0x65f7: wc = 0x856e; break;
        case 0x65f8: wc = 0x8725; break;
        case 0x65f9: wc = 0x87ab; break;
        case 0x65fa: wc = 0x88fc; break;
        case 0x65fb: wc = 0x892f; break;
        case 0x65fc: wc = 0x896b; break;
        case 0x65fd: wc = 0x909c; break;
        case 0x65fe: wc = 0x91f8; break;
        case 0x65ff: wc = 0x9250; break;
        case 0x6600: wc = 0x9f2b; break;
        case 0x6601: wc = 0x4e99; break;
        case 0x6602: wc = 0x50ce; break;
        case 0x6603: wc = 0x50d0; break;
        case 0x6604: wc = 0x5847; break;
        case 0x6605: wc = 0x58a0; break;
        case 0x6606: wc = 0x58a1; break;
        case 0x6607: wc = 0x597e; break;
        case 0x6608: wc = 0x59fa; break;
        case 0x6609: wc = 0x5ad9; break;
        case 0x660a: wc = 0x5af8; break;
        case 0x660b: wc = 0x5b17; break;
        case 0x660c: wc = 0x5c1f; break;
        case 0x660d: wc = 0x5c20; break;
        case 0x660e: wc = 0x5c73; break;
        case 0x660f: wc = 0x5eef; break;
        case 0x6610: wc = 0x6103; break;
        case 0x6611: wc = 0x63ce; break;
        case 0x6612: wc = 0x6427; break;
        case 0x6613: wc = 0x6b5a; break;
        case 0x6614: wc = 0x6be8; break;
        case 0x6615: wc = 0x6c19; break;
        case 0x6616: wc = 0x6f29; break;
        case 0x6617: wc = 0x3d5b; break;
        case 0x6618: wc = 0x7057; break;
        case 0x6619: wc = 0x70cd; break;
        case 0x661a: wc = 0x3dfd; break;
        case 0x661b: wc = 0x71f9; break;
        case 0x661c: wc = 0x736e; break;
        case 0x661d: wc = 0x73d7; break;
        case 0x661e: wc = 0x4062; break;
        case 0x661f: wc = 0x78b9; break;
        case 0x6620: wc = 0x7992; break;
        case 0x6621: wc = 0x79c8; break;
        case 0x6622: wc = 0x7b45; break;
        case 0x6623: wc = 0x7bb2; break;
        case 0x6624: wc = 0x7c7c; break;
        case 0x6625: wc = 0x7e07; break;
        case 0x6626: wc = 0x7e3c; break;
        case 0x6627: wc = 0x8519; break;
        case 0x6628: wc = 0x8701; break;
        case 0x6629: wc = 0x87ee; break;
        case 0x662a: wc = 0x87fa; break;
        case 0x662b: wc = 0x893c; break;
        case 0x662c: wc = 0x8948; break;
        case 0x662d: wc = 0x8b06; break;
        case 0x662e: wc = 0x8b54; break;
        case 0x662f: wc = 0x8d7b; break;
        case 0x6630: wc = 0x8df4; break;
        case 0x6631: wc = 0x8e9a; break;
        case 0x6632: wc = 0x912f; break;
        case 0x6633: wc = 0x493c; break;
        case 0x6634: wc = 0x9379; break;
        case 0x6635: wc = 0x934c; break;
        case 0x6636: wc = 0x93c7; break;
        case 0x6637: wc = 0x98b4; break;
        case 0x6638: wc = 0x9a38; break;
        case 0x6639: wc = 0x9c53; break;
        case 0x663a: wc = 0x9c54; break;
        case 0x663b: wc = 0x9c7b; break;
        case 0x663c: wc = 0x3425; break;
        case 0x663d: wc = 0x4f33; break;
        case 0x663e: wc = 0x5070; break;
        case 0x663f: wc = 0x50c1; break;
        case 0x6640: wc = 0x56d3; break;
        case 0x6641: wc = 0x5a9f; break;
        case 0x6642: wc = 0x62b4; break;
        case 0x6643: wc = 0x6308; break;
        case 0x6644: wc = 0x63f3; break;
        case 0x6645: wc = 0x63f2; break;
        case 0x6646: wc = 0x66ac; break;
        case 0x6647: wc = 0x67bb; break;
        case 0x6648: wc = 0x698d; break;
        case 0x6649: wc = 0x712b; break;
        case 0x664a: wc = 0x7207; break;
        case 0x664b: wc = 0x75b6; break;
        case 0x664c: wc = 0x789f; break;
        case 0x664d: wc = 0x7d32; break;
        case 0x664e: wc = 0x7d4f; break;
        case 0x664f: wc = 0x7de4; break;
        case 0x6650: wc = 0x850e; break;
        case 0x6651: wc = 0x272cd; break;
        case 0x6652: wc = 0x8909; break;
        case 0x6653: wc = 0x8ea0; break;
        case 0x6654: wc = 0x8fa5; break;
        case 0x6655: wc = 0x4979; break;
        case 0x6656: wc = 0x9c48; break;
        case 0x6657: wc = 0x5b10; break;
        case 0x6658: wc = 0x5b45; break;
        case 0x6659: wc = 0x2f89b; break;
        case 0x665a: wc = 0x61b8; break;
        case 0x665b: wc = 0x639e; break;
        case 0x665c: wc = 0x647b; break;
        case 0x665d: wc = 0x6515; break;
        case 0x665e: wc = 0x6671; break;
        case 0x665f: wc = 0x68ce; break;
        case 0x6660: wc = 0x71c5; break;
        case 0x6661: wc = 0x7213; break;
        case 0x6662: wc = 0x7752; break;
        case 0x6663: wc = 0x4061; break;
        case 0x6664: wc = 0x7e4a; break;
        case 0x6665: wc = 0x456d; break;
        case 0x6666: wc = 0x8973; break;
        case 0x6667: wc = 0x89a2; break;
        case 0x6668: wc = 0x8b6b; break;
        case 0x6669: wc = 0x929b; break;
        case 0x666a: wc = 0x97f1; break;
        case 0x666b: wc = 0x34a4; break;
        case 0x666c: wc = 0x56c1; break;
        case 0x666d: wc = 0x5c67; break;
        case 0x666e: wc = 0x5f3d; break;
        case 0x666f: wc = 0x5fa2; break;
        case 0x6670: wc = 0x61fe; break;
        case 0x6671: wc = 0x3c14; break;
        case 0x6672: wc = 0x6b07; break;
        case 0x6673: wc = 0x7044; break;
        case 0x6674: wc = 0x7215; break;
        case 0x6675: wc = 0x758c; break;
        case 0x6676: wc = 0x7c4b; break;
        case 0x6677: wc = 0x8076; break;
        case 0x6678: wc = 0x8802; break;
        case 0x6679: wc = 0x8b8b; break;
        case 0x667a: wc = 0x8b98; break;
        case 0x667b: wc = 0x8e02; break;
        case 0x667c: wc = 0x8e19; break;
        case 0x667d: wc = 0x8e9e; break;
        case 0x667e: wc = 0x8ea1; break;
        case 0x667f: wc = 0x9371; break;
        case 0x6680: wc = 0x9477; break;
        case 0x6681: wc = 0x97a2; break;
        case 0x6682: wc = 0x97d8; break;
        case 0x6683: wc = 0x9873; break;
        case 0x6684: wc = 0x5057; break;
        case 0x6685: wc = 0x57b6; break;
        case 0x6686: wc = 0x58ad; break;
        case 0x6687: wc = 0x5a0d; break;
        case 0x6688: wc = 0x5cf8; break;
        case 0x6689: wc = 0x6e3b; break;
        case 0x668a: wc = 0x24051; break;
        case 0x668b: wc = 0x714b; break;
        case 0x668c: wc = 0x72cc; break;
        case 0x668d: wc = 0x7446; break;
        case 0x668e: wc = 0x7bb5; break;
        case 0x668f: wc = 0x80dc; break;
        case 0x6690: wc = 0x837f; break;
        case 0x6691: wc = 0x89f2; break;
        case 0x6692: wc = 0x90d5; break;
        case 0x6693: wc = 0x92ee; break;
        case 0x6694: wc = 0x935f; break;
        case 0x6695: wc = 0x9a02; break;
        case 0x6696: wc = 0x5e28; break;
        case 0x6697: wc = 0x6329; break;
        case 0x6698: wc = 0x6a85; break;
        case 0x6699: wc = 0x6d97; break;
        case 0x669a: wc = 0x7e50; break;
        case 0x669b: wc = 0x6711; break;
        case 0x669c: wc = 0x8527; break;
        case 0x669d: wc = 0x88de; break;
        case 0x669e: wc = 0x8de9; break;
        case 0x669f: wc = 0x8ece; break;
        case 0x66a0: wc = 0x8f4a; break;
        case 0x66a1: wc = 0x92b4; break;
        case 0x66a2: wc = 0x9f5b; break;
        case 0x66a3: wc = 0x9f65; break;
        case 0x66a4: wc = 0x4f4b; break;
        case 0x66a5: wc = 0x5083; break;
        case 0x66a6: wc = 0x52ad; break;
        case 0x66a7: wc = 0x5372; break;
        case 0x66a8: wc = 0x54b2; break;
        case 0x66a9: wc = 0x55c9; break;
        case 0x66aa: wc = 0x55bf; break;
        case 0x66ab: wc = 0x56cc; break;
        case 0x66ac: wc = 0x57fd; break;
        case 0x66ad: wc = 0x5aca; break;
        case 0x66ae: wc = 0x5c15; break;
        case 0x66af: wc = 0x53bc; break;
        case 0x66b0: wc = 0x5f30; break;
        case 0x66b1: wc = 0x612b; break;
        case 0x66b2: wc = 0x6145; break;
        case 0x66b3: wc = 0x612c; break;
        case 0x66b4: wc = 0x634e; break;
        case 0x66b5: wc = 0x65d3; break;
        case 0x66b6: wc = 0x67d6; break;
        case 0x66b7: wc = 0x69a1; break;
        case 0x66b8: wc = 0x6a14; break;
        case 0x66b9: wc = 0x2372e; break;
        case 0x66ba: wc = 0x6aef; break;
        case 0x66bb: wc = 0x6af9; break;
        case 0x66bc: wc = 0x6bdc; break;
        case 0x66bd: wc = 0x6c09; break;
        case 0x66be: wc = 0x6cdd; break;
        case 0x66bf: wc = 0x6e9e; break;
        case 0x66c0: wc = 0x6eb8; break;
        case 0x66c1: wc = 0x6f05; break;
        case 0x66c2: wc = 0x6f72; break;
        case 0x66c3: wc = 0x7107; break;
        case 0x66c4: wc = 0x3dcc; break;
        case 0x66c5: wc = 0x71bd; break;
        case 0x66c6: wc = 0x724a; break;
        case 0x66c7: wc = 0x73bf; break;
        case 0x66c8: wc = 0x7411; break;
        case 0x66c9: wc = 0x7485; break;
        case 0x66ca: wc = 0x75df; break;
        case 0x66cb: wc = 0x3e34; break;
        case 0x66cc: wc = 0x7a4c; break;
        case 0x66cd: wc = 0x7b24; break;
        case 0x66ce: wc = 0x7b72; break;
        case 0x66cf: wc = 0x7b71; break;
        case 0x66d0: wc = 0x7bbe; break;
        case 0x66d1: wc = 0x7d80; break;
        case 0x66d2: wc = 0x7e24; break;
        case 0x66d3: wc = 0x7e45; break;
        case 0x66d4: wc = 0x7fdb; break;
        case 0x66d5: wc = 0x8186; break;
        case 0x66d6: wc = 0x268a9; break;
        case 0x66d7: wc = 0x8244; break;
        case 0x66d8: wc = 0x8258; break;
        case 0x66d9: wc = 0x83a6; break;
        case 0x66da: wc = 0x8437; break;
        case 0x66db: wc = 0x86f8; break;
        case 0x66dc: wc = 0x45bb; break;
        case 0x66dd: wc = 0x87a6; break;
        case 0x66de: wc = 0x87f0; break;
        case 0x66df: wc = 0x8828; break;
        case 0x66e0: wc = 0x8891; break;
        case 0x66e1: wc = 0x8b0f; break;
        case 0x66e2: wc = 0x8e03; break;
        case 0x66e3: wc = 0x8e08; break;
        case 0x66e4: wc = 0x47fd; break;
        case 0x66e5: wc = 0x8f0e; break;
        case 0x66e6: wc = 0x911b; break;
        case 0x66e7: wc = 0x9165; break;
        case 0x66e8: wc = 0x9704; break;
        case 0x66e9: wc = 0x98be; break;
        case 0x66ea: wc = 0x9afe; break;
        case 0x66eb: wc = 0x9b48; break;
        case 0x66ec: wc = 0x9bb9; break;
        case 0x66ed: wc = 0x9bc2; break;
        case 0x66ee: wc = 0x9c62; break;
        case 0x66ef: wc = 0x50f3; break;
        case 0x66f0: wc = 0x619f; break;
        case 0x66f1: wc = 0x6880; break;
        case 0x66f2: wc = 0x6a15; break;
        case 0x66f3: wc = 0x6d2c; break;
        case 0x66f4: wc = 0x7c0c; break;
        case 0x66f5: wc = 0x850c; break;
        case 0x66f6: wc = 0x85da; break;
        case 0x66f7: wc = 0x85d7; break;
        case 0x66f8: wc = 0x89eb; break;
        case 0x66f9: wc = 0x92c9; break;
        case 0x66fa: wc = 0x9917; break;
        case 0x66fb: wc = 0x5640; break;
        case 0x66fc: wc = 0x613b; break;
        case 0x66fd: wc = 0x640e; break;
        case 0x66fe: wc = 0x69c2; break;
        case 0x66ff: wc = 0x6f60; break;
        case 0x6700: wc = 0x733b; break;
        case 0x6701: wc = 0x859e; break;
        case 0x6702: wc = 0x9868; break;
        case 0x6703: wc = 0x98e7; break;
        case 0x6704: wc = 0x98f1; break;
        case 0x6705: wc = 0x3412; break;
        case 0x6706: wc = 0x4e7a; break;
        case 0x6707: wc = 0x6454; break;
        case 0x6708: wc = 0x7529; break;
        case 0x6709: wc = 0x7aa3; break;
        case 0x670a: wc = 0x87c0; break;
        case 0x670b: wc = 0x357e; break;
        case 0x670c: wc = 0x502f; break;
        case 0x670d: wc = 0x51c7; break;
        case 0x670e: wc = 0x5f78; break;
        case 0x670f: wc = 0x612f; break;
        case 0x6710: wc = 0x3a73; break;
        case 0x6711: wc = 0x6964; break;
        case 0x6712: wc = 0x7ae6; break;
        case 0x6713: wc = 0x8719; break;
        case 0x6714: wc = 0x9376; break;
        case 0x6715: wc = 0x93b9; break;
        case 0x6716: wc = 0x99f7; break;
        case 0x6717: wc = 0x9b06; break;
        case 0x6718: wc = 0x5530; break;
        case 0x6719: wc = 0x55e9; break;
        case 0x671a: wc = 0x57e3; break;
        case 0x671b: wc = 0x6652; break;
        case 0x671c: wc = 0x66ec; break;
        case 0x671d: wc = 0x6d12; break;
        case 0x671e: wc = 0x6dae; break;
        case 0x671f: wc = 0x24a0f; break;
        case 0x6720: wc = 0x7463; break;
        case 0x6721: wc = 0x7e30; break;
        case 0x6722: wc = 0x7e40; break;
        case 0x6723: wc = 0x93a9; break;
        case 0x6724: wc = 0x93c1; break;
        case 0x6725: wc = 0x54f8; break;
        case 0x6726: wc = 0x590a; break;
        case 0x6727: wc = 0x5015; break;
        case 0x6728: wc = 0x51c1; break;
        case 0x6729: wc = 0x539c; break;
        case 0x672a: wc = 0x53df; break;
        case 0x672b: wc = 0x552e; break;
        case 0x672c: wc = 0x55d6; break;
        case 0x672d: wc = 0x2101f; break;
        case 0x672e: wc = 0x5733; break;
        case 0x672f: wc = 0x5a37; break;
        case 0x6730: wc = 0x36ee; break;
        case 0x6731: wc = 0x5b03; break;
        case 0x6732: wc = 0x5b26; break;
        case 0x6733: wc = 0x5db2; break;
        case 0x6734: wc = 0x5dbf; break;
        case 0x6735: wc = 0x5e41; break;
        case 0x6736: wc = 0x2213c; break;
        case 0x6737: wc = 0x5ec0; break;
        case 0x6738: wc = 0x5ecb; break;
        case 0x6739: wc = 0x60fe; break;
        case 0x673a: wc = 0x60e5; break;
        case 0x673b: wc = 0x624c; break;
        case 0x673c: wc = 0x63b1; break;
        case 0x673d: wc = 0x6457; break;
        case 0x673e: wc = 0x64fb; break;
        case 0x673f: wc = 0x65de; break;
        case 0x6740: wc = 0x666c; break;
        case 0x6741: wc = 0x6778; break;
        case 0x6742: wc = 0x677d; break;
        case 0x6743: wc = 0x6826; break;
        case 0x6744: wc = 0x6a07; break;
        case 0x6745: wc = 0x6a7e; break;
        case 0x6746: wc = 0x6a96; break;
        case 0x6747: wc = 0x6b36; break;
        case 0x6748: wc = 0x6bb3; break;
        case 0x6749: wc = 0x23bdc; break;
        case 0x674a: wc = 0x6c35; break;
        case 0x674b: wc = 0x6c3a; break;
        case 0x674c: wc = 0x6cc5; break;
        case 0x674d: wc = 0x6d7d; break;
        case 0x674e: wc = 0x6ed6; break;
        case 0x674f: wc = 0x6eb2; break;
        case 0x6750: wc = 0x6f44; break;
        case 0x6751: wc = 0x6eeb; break;
        case 0x6752: wc = 0x6fc9; break;
        case 0x6753: wc = 0x7021; break;
        case 0x6754: wc = 0x702d; break;
        case 0x6755: wc = 0x7340; break;
        case 0x6756: wc = 0x74cd; break;
        case 0x6757: wc = 0x3f77; break;
        case 0x6758: wc = 0x76e8; break;
        case 0x6759: wc = 0x775f; break;
        case 0x675a: wc = 0x778d; break;
        case 0x675b: wc = 0x25343; break;
        case 0x675c: wc = 0x795f; break;
        case 0x675d: wc = 0x79ad; break;
        case 0x675e: wc = 0x7a24; break;
        case 0x675f: wc = 0x7a5f; break;
        case 0x6760: wc = 0x41d3; break;
        case 0x6761: wc = 0x41d5; break;
        case 0x6762: wc = 0x7bf2; break;
        case 0x6763: wc = 0x7c54; break;
        case 0x6764: wc = 0x7cd4; break;
        case 0x6765: wc = 0x7d52; break;
        case 0x6766: wc = 0x7d89; break;
        case 0x6767: wc = 0x7e78; break;
        case 0x6768: wc = 0x7e7b; break;
        case 0x6769: wc = 0x7e97; break;
        case 0x676a: wc = 0x43c2; break;
        case 0x676b: wc = 0x813d; break;
        case 0x676c: wc = 0x813a; break;
        case 0x676d: wc = 0x8144; break;
        case 0x676e: wc = 0x8167; break;
        case 0x676f: wc = 0x8184; break;
        case 0x6770: wc = 0x81b8; break;
        case 0x6771: wc = 0x81f9; break;
        case 0x6772: wc = 0x824f; break;
        case 0x6773: wc = 0x82ec; break;
        case 0x6774: wc = 0x837e; break;
        case 0x6775: wc = 0x83d9; break;
        case 0x6776: wc = 0x84e8; break;
        case 0x6777: wc = 0x8566; break;
        case 0x6778: wc = 0x45cf; break;
        case 0x6779: wc = 0x878b; break;
        case 0x677a: wc = 0x88cb; break;
        case 0x677b: wc = 0x895a; break;
        case 0x677c: wc = 0x8a39; break;
        case 0x677d: wc = 0x8a76; break;
        case 0x677e: wc = 0x46f5; break;
        case 0x677f: wc = 0x8ab6; break;
        case 0x6780: wc = 0x4714; break;
        case 0x6781: wc = 0x8b8e; break;
        case 0x6782: wc = 0x8c4e; break;
        case 0x6783: wc = 0x8ce5; break;
        case 0x6784: wc = 0x8ed7; break;
        case 0x6785: wc = 0x4898; break;
        case 0x6786: wc = 0x910b; break;
        case 0x6787: wc = 0x9199; break;
        case 0x6788: wc = 0x9380; break;
        case 0x6789: wc = 0x93aa; break;
        case 0x678a: wc = 0x93fd; break;
        case 0x678b: wc = 0x9406; break;
        case 0x678c: wc = 0x9429; break;
        case 0x678d: wc = 0x964e; break;
        case 0x678e: wc = 0x9672; break;
        case 0x678f: wc = 0x96d4; break;
        case 0x6790: wc = 0x98bc; break;
        case 0x6791: wc = 0x993f; break;
        case 0x6792: wc = 0x9948; break;
        case 0x6793: wc = 0x9a2a; break;
        case 0x6794: wc = 0x9b57; break;
        case 0x6795: wc = 0x9b82; break;
        case 0x6796: wc = 0x9c3d; break;
        case 0x6797: wc = 0x4ff6; break;
        case 0x6798: wc = 0x500f; break;
        case 0x6799: wc = 0x5135; break;
        case 0x679a: wc = 0x57f1; break;
        case 0x679b: wc = 0x5a4c; break;
        case 0x679c: wc = 0x644d; break;
        case 0x679d: wc = 0x6a5a; break;
        case 0x679e: wc = 0x7102; break;
        case 0x679f: wc = 0x738a; break;
        case 0x67a0: wc = 0x4398; break;
        case 0x67a1: wc = 0x83a4; break;
        case 0x67a2: wc = 0x84ff; break;
        case 0x67a3: wc = 0x866a; break;
        case 0x67a4: wc = 0x8ad4; break;
        case 0x67a5: wc = 0x8dfe; break;
        case 0x67a6: wc = 0x9a4c; break;
        case 0x67a7: wc = 0x9b9b; break;
        case 0x67a8: wc = 0x9c50; break;
        case 0x67a9: wc = 0x9deb; break;
        case 0x67aa: wc = 0x4f9a; break;
        case 0x67ab: wc = 0x54b0; break;
        case 0x67ac: wc = 0x5ccb; break;
        case 0x67ad: wc = 0x618c; break;
        case 0x67ae: wc = 0x63d7; break;
        case 0x67af: wc = 0x674a; break;
        case 0x67b0: wc = 0x69ab; break;
        case 0x67b1: wc = 0x69c6; break;
        case 0x67b2: wc = 0x6ee3; break;
        case 0x67b3: wc = 0x6f18; break;
        case 0x67b4: wc = 0x7289; break;
        case 0x67b5: wc = 0x72e5; break;
        case 0x67b6: wc = 0x76f9; break;
        case 0x67b7: wc = 0x4022; break;
        case 0x67b8: wc = 0x779a; break;
        case 0x67b9: wc = 0x77a4; break;
        case 0x67ba: wc = 0x78b7; break;
        case 0x67bb: wc = 0x78ed; break;
        case 0x67bc: wc = 0x7b0b; break;
        case 0x67bd: wc = 0x7c28; break;
        case 0x67be: wc = 0x7d03; break;
        case 0x67bf: wc = 0x80ab; break;
        case 0x67c0: wc = 0x829b; break;
        case 0x67c1: wc = 0x8493; break;
        case 0x67c2: wc = 0x8f34; break;
        case 0x67c3: wc = 0x90c7; break;
        case 0x67c4: wc = 0x99e8; break;
        case 0x67c5: wc = 0x9b0a; break;
        case 0x67c6: wc = 0x9bd9; break;
        case 0x67c7: wc = 0x9d89; break;
        case 0x67c8: wc = 0x5379; break;
        case 0x67c9: wc = 0x6cad; break;
        case 0x67ca: wc = 0x73ec; break;
        case 0x67cb: wc = 0x8357; break;
        case 0x67cc: wc = 0x8895; break;
        case 0x67cd: wc = 0x928a; break;
        case 0x67ce: wc = 0x9d90; break;
        case 0x67cf: wc = 0x7879; break;
        case 0x67d0: wc = 0x83d8; break;
        case 0x67d1: wc = 0x2a01a; break;
        case 0x67d2: wc = 0x6dec; break;
        case 0x67d3: wc = 0x7120; break;
        case 0x67d4: wc = 0x74b1; break;
        case 0x67d5: wc = 0x8671; break;
        case 0x67d6: wc = 0x5543; break;
        case 0x67d7: wc = 0x5d8d; break;
        case 0x67d8: wc = 0x6174; break;
        case 0x67d9: wc = 0x69e2; break;
        case 0x67da: wc = 0x6ebc; break;
        case 0x67db: wc = 0x71a0; break;
        case 0x67dc: wc = 0x8b35; break;
        case 0x67dd: wc = 0x96b0; break;
        case 0x67de: wc = 0x972b; break;
        case 0x67df: wc = 0x98c1; break;
        case 0x67e0: wc = 0x9a3d; break;
        case 0x67e1: wc = 0x9c3c; break;
        case 0x67e2: wc = 0x51bc; break;
        case 0x67e3: wc = 0x544f; break;
        case 0x67e4: wc = 0x584d; break;
        case 0x67e5: wc = 0x5d4a; break;
        case 0x67e6: wc = 0x61b4; break;
        case 0x67e7: wc = 0x628d; break;
        case 0x67e8: wc = 0x67a1; break;
        case 0x67e9: wc = 0x685d; break;
        case 0x67ea: wc = 0x6c36; break;
        case 0x67eb: wc = 0x3d0d; break;
        case 0x67ec: wc = 0x6e97; break;
        case 0x67ed: wc = 0x713a; break;
        case 0x67ee: wc = 0x7ad4; break;
        case 0x67ef: wc = 0x7e04; break;
        case 0x67f0: wc = 0x8100; break;
        case 0x67f1: wc = 0x8542; break;
        case 0x67f2: wc = 0x8b5d; break;
        case 0x67f3: wc = 0x8cf8; break;
        case 0x67f4: wc = 0x9a2c; break;
        case 0x67f5: wc = 0x9b19; break;
        case 0x67f6: wc = 0x9c66; break;
        case 0x67f7: wc = 0x9d7f; break;
        case 0x67f8: wc = 0x5072; break;
        case 0x67f9: wc = 0x5155; break;
        case 0x67fa: wc = 0x51d8; break;
        case 0x67fb: wc = 0x53ae; break;
        case 0x67fc: wc = 0x54b6; break;
        case 0x67fd: wc = 0x557b; break;
        case 0x67fe: wc = 0x5852; break;
        case 0x67ff: wc = 0x5cd5; break;
        case 0x6800: wc = 0x5d3c; break;
        case 0x6801: wc = 0x5d75; break;
        case 0x6802: wc = 0x5edd; break;
        case 0x6803: wc = 0x5fa5; break;
        case 0x6804: wc = 0x6220; break;
        case 0x6805: wc = 0x63d3; break;
        case 0x6806: wc = 0x63cc; break;
        case 0x6807: wc = 0x6495; break;
        case 0x6808: wc = 0x233da; break;
        case 0x6809: wc = 0x67be; break;
        case 0x680a: wc = 0x67b2; break;
        case 0x680b: wc = 0x67f9; break;
        case 0x680c: wc = 0x69af; break;
        case 0x680d: wc = 0x6be2; break;
        case 0x680e: wc = 0x6bf8; break;
        case 0x680f: wc = 0x6cb6; break;
        case 0x6810: wc = 0x6e64; break;
        case 0x6811: wc = 0x6ea1; break;
        case 0x6812: wc = 0x6f26; break;
        case 0x6813: wc = 0x6f8c; break;
        case 0x6814: wc = 0x2472e; break;
        case 0x6815: wc = 0x793b; break;
        case 0x6816: wc = 0x7961; break;
        case 0x6817: wc = 0x7c1b; break;
        case 0x6818: wc = 0x7d41; break;
        case 0x6819: wc = 0x7de6; break;
        case 0x681a: wc = 0x7f73; break;
        case 0x681b: wc = 0x26461; break;
        case 0x681c: wc = 0x7fe8; break;
        case 0x681d: wc = 0x816e; break;
        case 0x681e: wc = 0x81aa; break;
        case 0x681f: wc = 0x8479; break;
        case 0x6820: wc = 0x8448; break;
        case 0x6821: wc = 0x8937; break;
        case 0x6822: wc = 0x8979; break;
        case 0x6823: wc = 0x8adf; break;
        case 0x6824: wc = 0x8af0; break;
        case 0x6825: wc = 0x8c49; break;
        case 0x6826: wc = 0x90bf; break;
        case 0x6827: wc = 0x91c3; break;
        case 0x6828: wc = 0x9230; break;
        case 0x6829: wc = 0x9349; break;
        case 0x682a: wc = 0x4999; break;
        case 0x682b: wc = 0x984b; break;
        case 0x682c: wc = 0x98b8; break;
        case 0x682d: wc = 0x9c23; break;
        case 0x682e: wc = 0x9cf2; break;
        case 0x682f: wc = 0x4cad; break;
        case 0x6830: wc = 0x9e0d; break;
        case 0x6831: wc = 0x9f2d; break;
        case 0x6832: wc = 0x55b0; break;
        case 0x6833: wc = 0x5ab3; break;
        case 0x6834: wc = 0x683b; break;
        case 0x6835: wc = 0x70d2; break;
        case 0x6836: wc = 0x7176; break;
        case 0x6837: wc = 0x761c; break;
        case 0x6838: wc = 0x7c42; break;
        case 0x6839: wc = 0x84a0; break;
        case 0x683a: wc = 0x27139; break;
        case 0x683b: wc = 0x910e; break;
        case 0x683c: wc = 0x927d; break;
        case 0x683d: wc = 0x93b4; break;
        case 0x683e: wc = 0x515f; break;
        case 0x683f: wc = 0x5432; break;
        case 0x6840: wc = 0x54c2; break;
        case 0x6841: wc = 0x567a; break;
        case 0x6842: wc = 0x56df; break;
        case 0x6843: wc = 0x59bd; break;
        case 0x6844: wc = 0x372a; break;
        case 0x6845: wc = 0x5c7e; break;
        case 0x6846: wc = 0x5cf7; break;
        case 0x6847: wc = 0x62bb; break;
        case 0x6848: wc = 0x231db; break;
        case 0x6849: wc = 0x67db; break;
        case 0x684a: wc = 0x698a; break;
        case 0x684b: wc = 0x6c20; break;
        case 0x684c: wc = 0x6c5b; break;
        case 0x684d: wc = 0x70e5; break;
        case 0x684e: wc = 0x71ca; break;
        case 0x684f: wc = 0x73c5; break;
        case 0x6850: wc = 0x74b6; break;
        case 0x6851: wc = 0x7521; break;
        case 0x6852: wc = 0x7712; break;
        case 0x6853: wc = 0x77e7; break;
        case 0x6854: wc = 0x7837; break;
        case 0x6855: wc = 0x7973; break;
        case 0x6856: wc = 0x7b09; break;
        case 0x6857: wc = 0x7c78; break;
        case 0x6858: wc = 0x80c2; break;
        case 0x6859: wc = 0x8124; break;
        case 0x685a: wc = 0x8cee; break;
        case 0x685b: wc = 0x8d10; break;
        case 0x685c: wc = 0x48bb; break;
        case 0x685d: wc = 0x92e0; break;
        case 0x685e: wc = 0x9620; break;
        case 0x685f: wc = 0x9823; break;
        case 0x6860: wc = 0x9856; break;
        case 0x6861: wc = 0x99ea; break;
        case 0x6862: wc = 0x9bd3; break;
        case 0x6863: wc = 0x9dd0; break;
        case 0x6864: wc = 0x9e8e; break;
        case 0x6865: wc = 0x7ab8; break;
        case 0x6866: wc = 0x8785; break;
        case 0x6867: wc = 0x87cb; break;
        case 0x6868: wc = 0x98cb; break;
        case 0x6869: wc = 0x4f08; break;
        case 0x686a: wc = 0x4fba; break;
        case 0x686b: wc = 0x5423; break;
        case 0x686c: wc = 0x565a; break;
        case 0x686d: wc = 0x5814; break;
        case 0x686e: wc = 0x5b38; break;
        case 0x686f: wc = 0x5bb7; break;
        case 0x6870: wc = 0x5fc4; break;
        case 0x6871: wc = 0x38fa; break;
        case 0x6872: wc = 0x6116; break;
        case 0x6873: wc = 0x61b3; break;
        case 0x6874: wc = 0x6533; break;
        case 0x6875: wc = 0x677a; break;
        case 0x6876: wc = 0x6794; break;
        case 0x6877: wc = 0x6939; break;
        case 0x6878: wc = 0x6a33; break;
        case 0x6879: wc = 0x6df0; break;
        case 0x687a: wc = 0x6f6f; break;
        case 0x687b: wc = 0x7141; break;
        case 0x687c: wc = 0x71d6; break;
        case 0x687d: wc = 0x7495; break;
        case 0x687e: wc = 0x3f95; break;
        case 0x687f: wc = 0x77ab; break;
        case 0x6880: wc = 0x845a; break;
        case 0x6881: wc = 0x8548; break;
        case 0x6882: wc = 0x87f3; break;
        case 0x6883: wc = 0x8951; break;
        case 0x6884: wc = 0x89be; break;
        case 0x6885: wc = 0x8ad7; break;
        case 0x6886: wc = 0x8b85; break;
        case 0x6887: wc = 0x90a5; break;
        case 0x6888: wc = 0x9129; break;
        case 0x6889: wc = 0x920a; break;
        case 0x688a: wc = 0x9414; break;
        case 0x688b: wc = 0x9b35; break;
        case 0x688c: wc = 0x9c58; break;
        case 0x688d: wc = 0x9c4f; break;
        case 0x688e: wc = 0x534c; break;
        case 0x688f: wc = 0x74e7; break;
        case 0x6890: wc = 0x7acd; break;
        case 0x6891: wc = 0x7c75; break;
        case 0x6892: wc = 0x8fbb; break;
        case 0x6893: wc = 0x3418; break;
        case 0x6894: wc = 0x53cc; break;
        case 0x6895: wc = 0x5b47; break;
        case 0x6896: wc = 0x22960; break;
        case 0x6897: wc = 0x4276; break;
        case 0x6898: wc = 0x826d; break;
        case 0x6899: wc = 0x3612; break;
        case 0x689a: wc = 0x4e2b; break;
        case 0x689b: wc = 0x4f22; break;
        case 0x689c: wc = 0x4ff9; break;
        case 0x689d: wc = 0x538a; break;
        case 0x689e: wc = 0x54e6; break;
        case 0x689f: wc = 0x554a; break;
        case 0x68a0: wc = 0x59b8; break;
        case 0x68a1: wc = 0x5a3f; break;
        case 0x68a2: wc = 0x5a6d; break;
        case 0x68a3: wc = 0x5a40; break;
        case 0x68a4: wc = 0x5b72; break;
        case 0x68a5: wc = 0x5c59; break;
        case 0x68a6: wc = 0x5ce9; break;
        case 0x68a7: wc = 0x5e8c; break;
        case 0x68a8: wc = 0x6397; break;
        case 0x68a9: wc = 0x690f; break;
        case 0x68aa: wc = 0x6c2c; break;
        case 0x68ab: wc = 0x6d90; break;
        case 0x68ac: wc = 0x72bd; break;
        case 0x68ad: wc = 0x73a1; break;
        case 0x68ae: wc = 0x73f4; break;
        case 0x68af: wc = 0x75b4; break;
        case 0x68b0: wc = 0x7602; break;
        case 0x68b1: wc = 0x75fe; break;
        case 0x68b2: wc = 0x7692; break;
        case 0x68b3: wc = 0x774b; break;
        case 0x68b4: wc = 0x7811; break;
        case 0x68b5: wc = 0x786a; break;
        case 0x68b6: wc = 0x8063; break;
        case 0x68b7: wc = 0x8565; break;
        case 0x68b8: wc = 0x869c; break;
        case 0x68b9: wc = 0x897e; break;
        case 0x68ba: wc = 0x8fd3; break;
        case 0x68bb: wc = 0x9233; break;
        case 0x68bc: wc = 0x9312; break;
        case 0x68bd: wc = 0x930f; break;
        case 0x68be: wc = 0x9d5e; break;
        case 0x68bf: wc = 0x9f56; break;
        case 0x68c0: wc = 0x5053; break;
        case 0x68c1: wc = 0x537e; break;
        case 0x68c2: wc = 0x54a2; break;
        case 0x68c3: wc = 0x5594; break;
        case 0x68c4: wc = 0x35c1; break;
        case 0x68c5: wc = 0x5669; break;
        case 0x68c6: wc = 0x582e; break;
        case 0x68c7: wc = 0x5d3f; break;
        case 0x68c8: wc = 0x6943; break;
        case 0x68c9: wc = 0x6aee; break;
        case 0x68ca: wc = 0x815b; break;
        case 0x68cb: wc = 0x843c; break;
        case 0x68cc: wc = 0x8741; break;
        case 0x68cd: wc = 0x8ae4; break;
        case 0x68ce: wc = 0x9469; break;
        case 0x68cf: wc = 0x9c77; break;
        case 0x68d0: wc = 0x9d9a; break;
        case 0x68d1: wc = 0x9e11; break;
        case 0x68d2: wc = 0x9f76; break;
        case 0x68d3: wc = 0x4f92; break;
        case 0x68d4: wc = 0x5050; break;
        case 0x68d5: wc = 0x57b5; break;
        case 0x68d6: wc = 0x5813; break;
        case 0x68d7: wc = 0x59f2; break;
        case 0x68d8: wc = 0x5a69; break;
        case 0x68d9: wc = 0x5cd6; break;
        case 0x68da: wc = 0x6849; break;
        case 0x68db: wc = 0x694c; break;
        case 0x68dc: wc = 0x6c28; break;
        case 0x68dd: wc = 0x6d1d; break;
        case 0x68de: wc = 0x72b4; break;
        case 0x68df: wc = 0x77f8; break;
        case 0x68e0: wc = 0x834c; break;
        case 0x68e1: wc = 0x8d0b; break;
        case 0x68e2: wc = 0x8d17; break;
        case 0x68e3: wc = 0x92a8; break;
        case 0x68e4: wc = 0x930c; break;
        case 0x68e5: wc = 0x9807; break;
        case 0x68e6: wc = 0x9a34; break;
        case 0x68e7: wc = 0x9d08; break;
        case 0x68e8: wc = 0x9d33; break;
        case 0x68e9: wc = 0x9dc3; break;
        case 0x68ea: wc = 0x9eeb; break;
        case 0x68eb: wc = 0x560e; break;
        case 0x68ec: wc = 0x5720; break;
        case 0x68ed: wc = 0x5828; break;
        case 0x68ee: wc = 0x5dad; break;
        case 0x68ef: wc = 0x621b; break;
        case 0x68f0: wc = 0x621e; break;
        case 0x68f1: wc = 0x6316; break;
        case 0x68f2: wc = 0x63e0; break;
        case 0x68f3: wc = 0x6b79; break;
        case 0x68f4: wc = 0x6b7a; break;
        case 0x68f5: wc = 0x6fe3; break;
        case 0x68f6: wc = 0x7125; break;
        case 0x68f7: wc = 0x7330; break;
        case 0x68f8: wc = 0x7a75; break;
        case 0x68f9: wc = 0x7aab; break;
        case 0x68fa: wc = 0x8050; break;
        case 0x68fb: wc = 0x80fa; break;
        case 0x68fc: wc = 0x8a10; break;
        case 0x68fd: wc = 0x904f; break;
        case 0x68fe: wc = 0x9480; break;
        case 0x68ff: wc = 0x981e; break;
        case 0x6900: wc = 0x4c32; break;
        case 0x6901: wc = 0x9d36; break;
        case 0x6902: wc = 0x4d5d; break;
        case 0x6903: wc = 0x9f43; break;
        case 0x6904: wc = 0x9f7e; break;
        case 0x6905: wc = 0x533c; break;
        case 0x6906: wc = 0x5591; break;
        case 0x6907: wc = 0x557d; break;
        case 0x6908: wc = 0x3619; break;
        case 0x6909: wc = 0x57ef; break;
        case 0x690a: wc = 0x5837; break;
        case 0x690b: wc = 0x58e7; break;
        case 0x690c: wc = 0x5a95; break;
        case 0x690d: wc = 0x5d53; break;
        case 0x690e: wc = 0x5d52; break;
        case 0x690f: wc = 0x63de; break;
        case 0x6910: wc = 0x247df; break;
        case 0x6911: wc = 0x75f7; break;
        case 0x6912: wc = 0x76e6; break;
        case 0x6913: wc = 0x789e; break;
        case 0x6914: wc = 0x7939; break;
        case 0x6915: wc = 0x7f6f; break;
        case 0x6916: wc = 0x8164; break;
        case 0x6917: wc = 0x843b; break;
        case 0x6918: wc = 0x844a; break;
        case 0x6919: wc = 0x84ed; break;
        case 0x691a: wc = 0x88fa; break;
        case 0x691b: wc = 0x8af3; break;
        case 0x691c: wc = 0x96f8; break;
        case 0x691d: wc = 0x97fd; break;
        case 0x691e: wc = 0x99a3; break;
        case 0x691f: wc = 0x9d6a; break;
        case 0x6920: wc = 0x9e99; break;
        case 0x6921: wc = 0x9ee4; break;
        case 0x6922: wc = 0x9eef; break;
        case 0x6923: wc = 0x9eed; break;
        case 0x6924: wc = 0x5111; break;
        case 0x6925: wc = 0x530e; break;
        case 0x6926: wc = 0x5714; break;
        case 0x6927: wc = 0x59f6; break;
        case 0x6928: wc = 0x5c8b; break;
        case 0x6929: wc = 0x7810; break;
        case 0x692a: wc = 0x97a5; break;
        case 0x692b: wc = 0x4f52; break;
        case 0x692c: wc = 0x536c; break;
        case 0x692d: wc = 0x5771; break;
        case 0x692e: wc = 0x59ce; break;
        case 0x692f: wc = 0x5c9f; break;
        case 0x6930: wc = 0x62b0; break;
        case 0x6931: wc = 0x6602; break;
        case 0x6932: wc = 0x678a; break;
        case 0x6933: wc = 0x6cf1; break;
        case 0x6934: wc = 0x70b4; break;
        case 0x6935: wc = 0x76ce; break;
        case 0x6936: wc = 0x7d3b; break;
        case 0x6937: wc = 0x7fbb; break;
        case 0x6938: wc = 0x80e6; break;
        case 0x6939: wc = 0x8eee; break;
        case 0x693a: wc = 0x91a0; break;
        case 0x693b: wc = 0x9260; break;
        case 0x693c: wc = 0x96f5; break;
        case 0x693d: wc = 0x9785; break;
        case 0x693e: wc = 0x99da; break;
        case 0x693f: wc = 0x50fe; break;
        case 0x6940: wc = 0x5443; break;
        case 0x6941: wc = 0x54ce; break;
        case 0x6942: wc = 0x5509; break;
        case 0x6943: wc = 0x5532; break;
        case 0x6944: wc = 0x5540; break;
        case 0x6945: wc = 0x560a; break;
        case 0x6946: wc = 0x566f; break;
        case 0x6947: wc = 0x5810; break;
        case 0x6948: wc = 0x58d2; break;
        case 0x6949: wc = 0x5a2d; break;
        case 0x694a: wc = 0x5a3e; break;
        case 0x694b: wc = 0x5b21; break;
        case 0x694c: wc = 0x3775; break;
        case 0x694d: wc = 0x6328; break;
        case 0x694e: wc = 0x6371; break;
        case 0x694f: wc = 0x6571; break;
        case 0x6950: wc = 0x6639; break;
        case 0x6951: wc = 0x6b38; break;
        case 0x6952: wc = 0x6bd0; break;
        case 0x6953: wc = 0x6fed; break;
        case 0x6954: wc = 0x3dbc; break;
        case 0x6955: wc = 0x71f0; break;
        case 0x6956: wc = 0x7343; break;
        case 0x6957: wc = 0x74a6; break;
        case 0x6958: wc = 0x24d97; break;
        case 0x6959: wc = 0x769a; break;
        case 0x695a: wc = 0x775a; break;
        case 0x695b: wc = 0x77b9; break;
        case 0x695c: wc = 0x78d1; break;
        case 0x695d: wc = 0x7919; break;
        case 0x695e: wc = 0x25bc5; break;
        case 0x695f: wc = 0x8586; break;
        case 0x6960: wc = 0x85f9; break;
        case 0x6961: wc = 0x8b6a; break;
        case 0x6962: wc = 0x8cf9; break;
        case 0x6963: wc = 0x9440; break;
        case 0x6964: wc = 0x95a1; break;
        case 0x6965: wc = 0x9749; break;
        case 0x6966: wc = 0x9932; break;
        case 0x6967: wc = 0x9a03; break;
        case 0x6968: wc = 0x9d31; break;
        case 0x6969: wc = 0x5cc9; break;
        case 0x696a: wc = 0x6239; break;
        case 0x696b: wc = 0x6424; break;
        case 0x696c: wc = 0x7732; break;
        case 0x696d: wc = 0x7828; break;
        case 0x696e: wc = 0x8685; break;
        case 0x696f: wc = 0x4638; break;
        case 0x6970: wc = 0x8a7b; break;
        case 0x6971: wc = 0x8c5f; break;
        case 0x6972: wc = 0x8edb; break;
        case 0x6973: wc = 0x8ef6; break;
        case 0x6974: wc = 0x9628; break;
        case 0x6975: wc = 0x97b7; break;
        case 0x6976: wc = 0x981f; break;
        case 0x6977: wc = 0x296ac; break;
        case 0x6978: wc = 0x56b6; break;
        case 0x6979: wc = 0x5ac8; break;
        case 0x697a: wc = 0x7507; break;
        case 0x697b: wc = 0x7516; break;
        case 0x697c: wc = 0x7f43; break;
        case 0x697d: wc = 0x8b7b; break;
        case 0x697e: wc = 0x9e0e; break;
        case 0x697f: wc = 0x504c; break;
        case 0x6980: wc = 0x5414; break;
        case 0x6981: wc = 0x558f; break;
        case 0x6982: wc = 0x5622; break;
        case 0x6983: wc = 0x57dc; break;
        case 0x6984: wc = 0x5a7c; break;
        case 0x6985: wc = 0x6353; break;
        case 0x6986: wc = 0x64e8; break;
        case 0x6987: wc = 0x6792; break;
        case 0x6988: wc = 0x6f1c; break;
        case 0x6989: wc = 0x7132; break;
        case 0x698a: wc = 0x740a; break;
        case 0x698b: wc = 0x7942; break;
        case 0x698c: wc = 0x91fe; break;
        case 0x698d: wc = 0x9381; break;
        case 0x698e: wc = 0x9d7a; break;
        case 0x698f: wc = 0x55b2; break;
        case 0x6990: wc = 0x6949; break;
        case 0x6991: wc = 0x6e03; break;
        case 0x6992: wc = 0x7039; break;
        case 0x6993: wc = 0x721a; break;
        case 0x6994: wc = 0x793f; break;
        case 0x6995: wc = 0x79b4; break;
        case 0x6996: wc = 0x7bac; break;
        case 0x6997: wc = 0x7bb9; break;
        case 0x6998: wc = 0x7bdb; break;
        case 0x6999: wc = 0x7c65; break;
        case 0x699a: wc = 0x85de; break;
        case 0x699b: wc = 0x8625; break;
        case 0x699c: wc = 0x9100; break;
        case 0x699d: wc = 0x9470; break;
        case 0x699e: wc = 0x29969; break;
        case 0x699f: wc = 0x9c2f; break;
        case 0x69a0: wc = 0x9db8; break;
        case 0x69a1: wc = 0x9e19; break;
        case 0x69a2: wc = 0x9fa0; break;
        case 0x69a3: wc = 0x35b1; break;
        case 0x69a4: wc = 0x5134; break;
        case 0x69a5: wc = 0x52f7; break;
        case 0x69a6: wc = 0x56b7; break;
        case 0x69a7: wc = 0x579f; break;
        case 0x69a8: wc = 0x5d35; break;
        case 0x69a9: wc = 0x5f89; break;
        case 0x69aa: wc = 0x5fc0; break;
        case 0x69ab: wc = 0x61e9; break;
        case 0x69ac: wc = 0x61f9; break;
        case 0x69ad: wc = 0x6501; break;
        case 0x69ae: wc = 0x661c; break;
        case 0x69af: wc = 0x6aa8; break;
        case 0x69b0: wc = 0x6b00; break;
        case 0x69b1: wc = 0x6c27; break;
        case 0x69b2: wc = 0x6c26; break;
        case 0x69b3: wc = 0x6f3e; break;
        case 0x69b4: wc = 0x703c; break;
        case 0x69b5: wc = 0x70ca; break;
        case 0x69b6: wc = 0x7219; break;
        case 0x69b7: wc = 0x737d; break;
        case 0x69b8: wc = 0x73dc; break;
        case 0x69b9: wc = 0x74d6; break;
        case 0x69ba: wc = 0x74e4; break;
        case 0x69bb: wc = 0x7662; break;
        case 0x69bc: wc = 0x773b; break;
        case 0x69bd: wc = 0x7922; break;
        case 0x69be: wc = 0x7993; break;
        case 0x69bf: wc = 0x7e95; break;
        case 0x69c0: wc = 0x7f99; break;
        case 0x69c1: wc = 0x7f95; break;
        case 0x69c2: wc = 0x444b; break;
        case 0x69c3: wc = 0x8618; break;
        case 0x69c4: wc = 0x86d8; break;
        case 0x69c5: wc = 0x45b9; break;
        case 0x69c6: wc = 0x277c4; break;
        case 0x69c7: wc = 0x27950; break;
        case 0x69c8: wc = 0x8e9f; break;
        case 0x69c9: wc = 0x8f30; break;
        case 0x69ca: wc = 0x284ab; break;
        case 0x69cb: wc = 0x935a; break;
        case 0x69cc: wc = 0x940a; break;
        case 0x69cd: wc = 0x9472; break;
        case 0x69ce: wc = 0x98ba; break;
        case 0x69cf: wc = 0x995f; break;
        case 0x69d0: wc = 0x9a64; break;
        case 0x69d1: wc = 0x9b24; break;
        case 0x69d2: wc = 0x9e09; break;
        case 0x69d3: wc = 0x20202; break;
        case 0x69d4: wc = 0x5539; break;
        case 0x69d5: wc = 0x5709; break;
        case 0x69d6: wc = 0x5cff; break;
        case 0x69d7: wc = 0x6554; break;
        case 0x69d8: wc = 0x23297; break;
        case 0x69d9: wc = 0x68dc; break;
        case 0x69da: wc = 0x6de4; break;
        case 0x69db: wc = 0x4069; break;
        case 0x69dc: wc = 0x7b8a; break;
        case 0x69dd: wc = 0x7bfd; break;
        case 0x69de: wc = 0x7c5e; break;
        case 0x69df: wc = 0x83f8; break;
        case 0x69e0: wc = 0x84f9; break;
        case 0x69e1: wc = 0x860c; break;
        case 0x69e2: wc = 0x91a7; break;
        case 0x69e3: wc = 0x92d9; break;
        case 0x69e4: wc = 0x98eb; break;
        case 0x69e5: wc = 0x9947; break;
        case 0x69e6: wc = 0x4ebf; break;
        case 0x69e7: wc = 0x5db7; break;
        case 0x69e8: wc = 0x3ac7; break;
        case 0x69e9: wc = 0x6fba; break;
        case 0x69ea: wc = 0x7654; break;
        case 0x69eb: wc = 0x7e76; break;
        case 0x69ec: wc = 0x808a; break;
        case 0x69ed: wc = 0x9929; break;
        case 0x69ee: wc = 0x50bf; break;
        case 0x69ef: wc = 0x533d; break;
        case 0x69f0: wc = 0x5501; break;
        case 0x69f1: wc = 0x55ad; break;
        case 0x69f2: wc = 0x5615; break;
        case 0x69f3: wc = 0x5a2e; break;
        case 0x69f4: wc = 0x5ae3; break;
        case 0x69f5: wc = 0x5d43; break;
        case 0x69f6: wc = 0x611d; break;
        case 0x69f7: wc = 0x693b; break;
        case 0x69f8: wc = 0x6f39; break;
        case 0x69f9: wc = 0x726a; break;
        case 0x69fa: wc = 0x7517; break;
        case 0x69fb: wc = 0x8412; break;
        case 0x69fc: wc = 0x852b; break;
        case 0x69fd: wc = 0x8758; break;
        case 0x69fe: wc = 0x8917; break;
        case 0x69ff: wc = 0x8ebd; break;
        case 0x6a00: wc = 0x90fe; break;
        case 0x6a01: wc = 0x9122; break;
        case 0x6a02: wc = 0x9c0b; break;
        case 0x6a03: wc = 0x9da0; break;
        case 0x6a04: wc = 0x9f34; break;
        case 0x6a05: wc = 0x9f5e; break;
        case 0x6a06: wc = 0x9f74; break;
        case 0x6a07: wc = 0x4e7b; break;
        case 0x6a08: wc = 0x5b7d; break;
        case 0x6a09: wc = 0x5d72; break;
        case 0x6a0a: wc = 0x69f7; break;
        case 0x6a0b: wc = 0x6af1; break;
        case 0x6a0c: wc = 0x7cf1; break;
        case 0x6a0d: wc = 0x7cf5; break;
        case 0x6a0e: wc = 0x81ec; break;
        case 0x6a0f: wc = 0x81f2; break;
        case 0x6a10: wc = 0x8825; break;
        case 0x6a11: wc = 0x8b9e; break;
        case 0x6a12: wc = 0x93b3; break;
        case 0x6a13: wc = 0x95d1; break;
        case 0x6a14: wc = 0x9689; break;
        case 0x6a15: wc = 0x9667; break;
        case 0x6a16: wc = 0x5382; break;
        case 0x6a17: wc = 0x565e; break;
        case 0x6a18: wc = 0x36ea; break;
        case 0x6a19: wc = 0x5b4d; break;
        case 0x6a1a: wc = 0x5d26; break;
        case 0x6a1b: wc = 0x5e7f; break;
        case 0x6a1c: wc = 0x5f07; break;
        case 0x6a1d: wc = 0x63dc; break;
        case 0x6a1e: wc = 0x667b; break;
        case 0x6a1f: wc = 0x66ee; break;
        case 0x6a20: wc = 0x6b15; break;
        case 0x6a21: wc = 0x6e30; break;
        case 0x6a22: wc = 0x7f68; break;
        case 0x6a23: wc = 0x814c; break;
        case 0x6a24: wc = 0x277d3; break;
        case 0x6a25: wc = 0x9183; break;
        case 0x6a26: wc = 0x91c5; break;
        case 0x6a27: wc = 0x95b9; break;
        case 0x6a28: wc = 0x9692; break;
        case 0x6a29: wc = 0x9869; break;
        case 0x6a2a: wc = 0x9f91; break;
        case 0x6a2b: wc = 0x6b97; break;
        case 0x6a2c: wc = 0x6fb2; break;
        case 0x6a2d: wc = 0x9134; break;
        case 0x6a2e: wc = 0x65d5; break;
        case 0x6a2f: wc = 0x3ac8; break;
        case 0x6a30: wc = 0x605a; break;
        case 0x6a31: wc = 0x66c0; break;
        case 0x6a32: wc = 0x6baa; break;
        case 0x6a33: wc = 0x4e0e; break;
        case 0x6a34: wc = 0x4f03; break;
        case 0x6a35: wc = 0x4f9e; break;
        case 0x6a36: wc = 0x59a4; break;
        case 0x6a37: wc = 0x381f; break;
        case 0x6a38: wc = 0x6086; break;
        case 0x6a39: wc = 0x65df; break;
        case 0x6a3a: wc = 0x6d33; break;
        case 0x6a3b: wc = 0x6fa6; break;
        case 0x6a3c: wc = 0x72f3; break;
        case 0x6a3d: wc = 0x756c; break;
        case 0x6a3e: wc = 0x7862; break;
        case 0x6a3f: wc = 0x791c; break;
        case 0x6a40: wc = 0x7b4e; break;
        case 0x6a41: wc = 0x7c45; break;
        case 0x6a42: wc = 0x7c79; break;
        case 0x6a43: wc = 0x8201; break;
        case 0x6a44: wc = 0x8560; break;
        case 0x6a45: wc = 0x8577; break;
        case 0x6a46: wc = 0x283d0; break;
        case 0x6a47: wc = 0x92a3; break;
        case 0x6a48: wc = 0x96d3; break;
        case 0x6a49: wc = 0x9bbd; break;
        case 0x6a4a: wc = 0x9d3d; break;
        case 0x6a4b: wc = 0x9e12; break;
        case 0x6a4c: wc = 0x344a; break;
        case 0x6a4d: wc = 0x571b; break;
        case 0x6a4e: wc = 0x57bc; break;
        case 0x6a4f: wc = 0x57f8; break;
        case 0x6a50: wc = 0x58bf; break;
        case 0x6a51: wc = 0x5da7; break;
        case 0x6a52: wc = 0x5e1f; break;
        case 0x6a53: wc = 0x61cc; break;
        case 0x6a54: wc = 0x622b; break;
        case 0x6a55: wc = 0x6679; break;
        case 0x6a56: wc = 0x68eb; break;
        case 0x6a57: wc = 0x6c31; break;
        case 0x6a58: wc = 0x6de2; break;
        case 0x6a59: wc = 0x7088; break;
        case 0x6a5a: wc = 0x71e1; break;
        case 0x6a5b: wc = 0x3edb; break;
        case 0x6a5c: wc = 0x776a; break;
        case 0x6a5d: wc = 0x7dce; break;
        case 0x6a5e: wc = 0x7e0c; break;
        case 0x6a5f: wc = 0x7f6d; break;
        case 0x6a60: wc = 0x44c8; break;
        case 0x6a61: wc = 0x27312; break;
        case 0x6a62: wc = 0x8c5b; break;
        case 0x6a63: wc = 0x91b3; break;
        case 0x6a64: wc = 0x9220; break;
        case 0x6a65: wc = 0x95be; break;
        case 0x6a66: wc = 0x9b4a; break;
        case 0x6a67: wc = 0x9d82; break;
        case 0x6a68: wc = 0x9dca; break;
        case 0x6a69: wc = 0x5157; break;
        case 0x6a6a: wc = 0x542e; break;
        case 0x6a6b: wc = 0x562b; break;
        case 0x6a6c: wc = 0x566e; break;
        case 0x6a6d: wc = 0x57cf; break;
        case 0x6a6e: wc = 0x58d6; break;
        case 0x6a6f: wc = 0x5a2b; break;
        case 0x6a70: wc = 0x5b3d; break;
        case 0x6a71: wc = 0x5b3f; break;
        case 0x6a72: wc = 0x6081; break;
        case 0x6a73: wc = 0x611e; break;
        case 0x6a74: wc = 0x6281; break;
        case 0x6a75: wc = 0x63be; break;
        case 0x6a76: wc = 0x63c5; break;
        case 0x6a77: wc = 0x648b; break;
        case 0x6a78: wc = 0x6616; break;
        case 0x6a79: wc = 0x3adf; break;
        case 0x6a7a: wc = 0x66e3; break;
        case 0x6a7b: wc = 0x68e9; break;
        case 0x6a7c: wc = 0x693c; break;
        case 0x6a7d: wc = 0x6a6a; break;
        case 0x6a7e: wc = 0x6ade; break;
        case 0x6a7f: wc = 0x6e15; break;
        case 0x6a80: wc = 0x6e37; break;
        case 0x6a81: wc = 0x6e8e; break;
        case 0x6a82: wc = 0x7041; break;
        case 0x6a83: wc = 0x70fb; break;
        case 0x6a84: wc = 0x72ff; break;
        case 0x6a85: wc = 0x73da; break;
        case 0x6a86: wc = 0x74c0; break;
        case 0x6a87: wc = 0x791d; break;
        case 0x6a88: wc = 0x7d96; break;
        case 0x6a89: wc = 0x7ddb; break;
        case 0x6a8a: wc = 0x800e; break;
        case 0x6a8b: wc = 0x8099; break;
        case 0x6a8c: wc = 0x80b0; break;
        case 0x6a8d: wc = 0x80ed; break;
        case 0x6a8e: wc = 0x81d9; break;
        case 0x6a8f: wc = 0x839a; break;
        case 0x6a90: wc = 0x870e; break;
        case 0x6a91: wc = 0x8712; break;
        case 0x6a92: wc = 0x8735; break;
        case 0x6a93: wc = 0x875d; break;
        case 0x6a94: wc = 0x8815; break;
        case 0x6a95: wc = 0x88eb; break;
        case 0x6a96: wc = 0x89fe; break;
        case 0x6a97: wc = 0x8b8c; break;
        case 0x6a98: wc = 0x8f2d; break;
        case 0x6a99: wc = 0x8f4b; break;
        case 0x6a9a: wc = 0x90d4; break;
        case 0x6a9b: wc = 0x9140; break;
        case 0x6a9c: wc = 0x91bc; break;
        case 0x6a9d: wc = 0x9206; break;
        case 0x6a9e: wc = 0x92cb; break;
        case 0x6a9f: wc = 0x9a60; break;
        case 0x6aa0: wc = 0x9df0; break;
        case 0x6aa1: wc = 0x9f18; break;
        case 0x6aa2: wc = 0x63d1; break;
        case 0x6aa3: wc = 0x882e; break;
        case 0x6aa4: wc = 0x4b47; break;
        case 0x6aa5: wc = 0x5063; break;
        case 0x6aa6: wc = 0x5189; break;
        case 0x6aa7: wc = 0x53b4; break;
        case 0x6aa8: wc = 0x5465; break;
        case 0x6aa9: wc = 0x5869; break;
        case 0x6aaa: wc = 0x58db; break;
        case 0x6aab: wc = 0x59cc; break;
        case 0x6aac: wc = 0x5b2e; break;
        case 0x6aad: wc = 0x61d5; break;
        case 0x6aae: wc = 0x624a; break;
        case 0x6aaf: wc = 0x62a9; break;
        case 0x6ab0: wc = 0x6565; break;
        case 0x6ab1: wc = 0x68ea; break;
        case 0x6ab2: wc = 0x6ab6; break;
        case 0x6ab3: wc = 0x6abf; break;
        case 0x6ab4: wc = 0x6ae9; break;
        case 0x6ab5: wc = 0x7069; break;
        case 0x6ab6: wc = 0x7131; break;
        case 0x6ab7: wc = 0x71c4; break;
        case 0x6ab8: wc = 0x7312; break;
        case 0x6ab9: wc = 0x73c3; break;
        case 0x6aba: wc = 0x2604f; break;
        case 0x6abb: wc = 0x86a6; break;
        case 0x6abc: wc = 0x86ba; break;
        case 0x6abd: wc = 0x88a1; break;
        case 0x6abe: wc = 0x8c54; break;
        case 0x6abf: wc = 0x9153; break;
        case 0x6ac0: wc = 0x91b6; break;
        case 0x6ac1: wc = 0x9586; break;
        case 0x6ac2: wc = 0x995c; break;
        case 0x6ac3: wc = 0x9b58; break;
        case 0x6ac4: wc = 0x9bf0; break;
        case 0x6ac5: wc = 0x9ef6; break;
        case 0x6ac6: wc = 0x505e; break;
        case 0x6ac7: wc = 0x55a6; break;
        case 0x6ac8: wc = 0x64db; break;
        case 0x6ac9: wc = 0x64eb; break;
        case 0x6aca: wc = 0x6b9c; break;
        case 0x6acb: wc = 0x7180; break;
        case 0x6acc: wc = 0x7217; break;
        case 0x6acd: wc = 0x77b1; break;
        case 0x6ace: wc = 0x9437; break;
        case 0x6acf: wc = 0x9768; break;
        case 0x6ad0: wc = 0x9941; break;
        case 0x6ad1: wc = 0x35b3; break;
        case 0x6ad2: wc = 0x5040; break;
        case 0x6ad3: wc = 0x548f; break;
        case 0x6ad4: wc = 0x5a96; break;
        case 0x6ad5: wc = 0x5b34; break;
        case 0x6ad6: wc = 0x5b30; break;
        case 0x6ad7: wc = 0x5b7e; break;
        case 0x6ad8: wc = 0x5dc6; break;
        case 0x6ad9: wc = 0x5eee; break;
        case 0x6ada: wc = 0x6125; break;
        case 0x6adb: wc = 0x646c; break;
        case 0x6adc: wc = 0x650d; break;
        case 0x6add: wc = 0x6516; break;
        case 0x6ade: wc = 0x651a; break;
        case 0x6adf: wc = 0x6720; break;
        case 0x6ae0: wc = 0x67cd; break;
        case 0x6ae1: wc = 0x6810; break;
        case 0x6ae2: wc = 0x68ac; break;
        case 0x6ae3: wc = 0x235d0; break;
        case 0x6ae4: wc = 0x6d67; break;
        case 0x6ae5: wc = 0x6e8b; break;
        case 0x6ae6: wc = 0x6ff4; break;
        case 0x6ae7: wc = 0x6fd9; break;
        case 0x6ae8: wc = 0x7034; break;
        case 0x6ae9: wc = 0x7203; break;
        case 0x6aea: wc = 0x2492f; break;
        case 0x6aeb: wc = 0x7484; break;
        case 0x6aec: wc = 0x766d; break;
        case 0x6aed: wc = 0x4050; break;
        case 0x6aee: wc = 0x77e8; break;
        case 0x6aef: wc = 0x78a4; break;
        case 0x6af0: wc = 0x792f; break;
        case 0x6af1: wc = 0x799c; break;
        case 0x6af2: wc = 0x7c5d; break;
        case 0x6af3: wc = 0x7c6f; break;
        case 0x6af4: wc = 0x7e08; break;
        case 0x6af5: wc = 0x843e; break;
        case 0x6af6: wc = 0x4566; break;
        case 0x6af7: wc = 0x8621; break;
        case 0x6af8: wc = 0x8767; break;
        case 0x6af9: wc = 0x8811; break;
        case 0x6afa: wc = 0x8833; break;
        case 0x6afb: wc = 0x8b0d; break;
        case 0x6afc: wc = 0x8ccf; break;
        case 0x6afd: wc = 0x8d0f; break;
        case 0x6afe: wc = 0x90e2; break;
        case 0x6aff: wc = 0x919f; break;
        case 0x6b00: wc = 0x941b; break;
        case 0x6b01: wc = 0x97fa; break;
        case 0x6b02: wc = 0x9da7; break;
        case 0x6b03: wc = 0x52e9; break;
        case 0x6b04: wc = 0x546d; break;
        case 0x6b05: wc = 0x3611; break;
        case 0x6b06: wc = 0x56c8; break;
        case 0x6b07: wc = 0x57f6; break;
        case 0x6b08: wc = 0x5804; break;
        case 0x6b09: wc = 0x5a57; break;
        case 0x6b0a: wc = 0x5ad5; break;
        case 0x6b0b: wc = 0x5adb; break;
        case 0x6b0c: wc = 0x5bf1; break;
        case 0x6b0d: wc = 0x5e20; break;
        case 0x6b0e: wc = 0x6008; break;
        case 0x6b0f: wc = 0x2263d; break;
        case 0x6b10: wc = 0x60e2; break;
        case 0x6b11: wc = 0x62fd; break;
        case 0x6b12: wc = 0x6359; break;
        case 0x6b13: wc = 0x639c; break;
        case 0x6b14: wc = 0x6672; break;
        case 0x6b15: wc = 0x6798; break;
        case 0x6b16: wc = 0x678d; break;
        case 0x6b17: wc = 0x6827; break;
        case 0x6b18: wc = 0x69f8; break;
        case 0x6b19: wc = 0x6a64; break;
        case 0x6b1a: wc = 0x6de3; break;
        case 0x6b1b: wc = 0x7369; break;
        case 0x6b1c: wc = 0x73b4; break;
        case 0x6b1d: wc = 0x747f; break;
        case 0x6b1e: wc = 0x761e; break;
        case 0x6b1f: wc = 0x7631; break;
        case 0x6b20: wc = 0x7768; break;
        case 0x6b21: wc = 0x7796; break;
        case 0x6b22: wc = 0x40dc; break;
        case 0x6b23: wc = 0x7e44; break;
        case 0x6b24: wc = 0x7e60; break;
        case 0x6b25: wc = 0x7fbf; break;
        case 0x6b26: wc = 0x7ff3; break;
        case 0x6b27: wc = 0x266c8; break;
        case 0x6b28: wc = 0x26772; break;
        case 0x6b29: wc = 0x84fa; break;
        case 0x6b2a: wc = 0x854a; break;
        case 0x6b2b: wc = 0x8589; break;
        case 0x6b2c: wc = 0x8619; break;
        case 0x6b2d: wc = 0x868b; break;
        case 0x6b2e: wc = 0x8739; break;
        case 0x6b2f: wc = 0x873a; break;
        case 0x6b30: wc = 0x897c; break;
        case 0x6b31: wc = 0x89ec; break;
        case 0x6b32: wc = 0x8a4d; break;
        case 0x6b33: wc = 0x8c8e; break;
        case 0x6b34: wc = 0x27d16; break;
        case 0x6b35: wc = 0x8dc7; break;
        case 0x6b36: wc = 0x8e05; break;
        case 0x6b37: wc = 0x8f17; break;
        case 0x6b38: wc = 0x90f3; break;
        case 0x6b39: wc = 0x93cf; break;
        case 0x6b3a: wc = 0x977e; break;
        case 0x6b3b: wc = 0x9956; break;
        case 0x6b3c: wc = 0x9be2; break;
        case 0x6b3d: wc = 0x9dd6; break;
        case 0x6b3e: wc = 0x9e91; break;
        case 0x6b3f: wc = 0x9ef3; break;
        case 0x6b40: wc = 0x9f6f; break;
        case 0x6b41: wc = 0x4ef5; break;
        case 0x6b42: wc = 0x4fe3; break;
        case 0x6b43: wc = 0x53ab; break;
        case 0x6b44: wc = 0x5514; break;
        case 0x6b45: wc = 0x554e; break;
        case 0x6b46: wc = 0x55f7; break;
        case 0x6b47: wc = 0x55f8; break;
        case 0x6b48: wc = 0x5641; break;
        case 0x6b49: wc = 0x5662; break;
        case 0x6b4a: wc = 0x572c; break;
        case 0x6b4b: wc = 0x57e1; break;
        case 0x6b4c: wc = 0x5961; break;
        case 0x6b4d: wc = 0x5a2a; break;
        case 0x6b4e: wc = 0x5d68; break;
        case 0x6b4f: wc = 0x5db4; break;
        case 0x6b50: wc = 0x5ed2; break;
        case 0x6b51: wc = 0x5fe4; break;
        case 0x6b52: wc = 0x609e; break;
        case 0x6b53: wc = 0x22780; break;
        case 0x6b54: wc = 0x6160; break;
        case 0x6b55: wc = 0x2f8b4; break;
        case 0x6b56: wc = 0x6277; break;
        case 0x6b57: wc = 0x6342; break;
        case 0x6b58: wc = 0x6440; break;
        case 0x6b59: wc = 0x646e; break;
        case 0x6b5a: wc = 0x64d9; break;
        case 0x6b5b: wc = 0x6747; break;
        case 0x6b5c: wc = 0x3ba7; break;
        case 0x6b5d: wc = 0x6b4d; break;
        case 0x6b5e: wc = 0x6c59; break;
        case 0x6b5f: wc = 0x6c7b; break;
        case 0x6b60: wc = 0x6d3f; break;
        case 0x6b61: wc = 0x6d6f; break;
        case 0x6b62: wc = 0x6d16; break;
        case 0x6b63: wc = 0x6ef6; break;
        case 0x6b64: wc = 0x7110; break;
        case 0x6b65: wc = 0x720a; break;
        case 0x6b66: wc = 0x727e; break;
        case 0x6b67: wc = 0x739d; break;
        case 0x6b68: wc = 0x73f8; break;
        case 0x6b69: wc = 0x3ecd; break;
        case 0x6b6a: wc = 0x7488; break;
        case 0x6b6b: wc = 0x75e6; break;
        case 0x6b6c: wc = 0x78dd; break;
        case 0x6b6d: wc = 0x7a8f; break;
        case 0x6b6e: wc = 0x7ab9; break;
        case 0x6b6f: wc = 0x7e7a; break;
        case 0x6b70: wc = 0x43b8; break;
        case 0x6b71: wc = 0x8071; break;
        case 0x6b72: wc = 0x8323; break;
        case 0x6b73: wc = 0x851c; break;
        case 0x6b74: wc = 0x8601; break;
        case 0x6b75: wc = 0x8790; break;
        case 0x6b76: wc = 0x87af; break;
        case 0x6b77: wc = 0x8956; break;
        case 0x6b78: wc = 0x8b37; break;
        case 0x6b79: wc = 0x8db6; break;
        case 0x6b7a: wc = 0x8fd5; break;
        case 0x6b7b: wc = 0x901c; break;
        case 0x6b7c: wc = 0x9068; break;
        case 0x6b7d: wc = 0x907b; break;
        case 0x6b7e: wc = 0x90da; break;
        case 0x6b7f: wc = 0x9114; break;
        case 0x6b80: wc = 0x92d8; break;
        case 0x6b81: wc = 0x93a2; break;
        case 0x6b82: wc = 0x93d6; break;
        case 0x6b83: wc = 0x93ca; break;
        case 0x6b84: wc = 0x93d5; break;
        case 0x6b85: wc = 0x942d; break;
        case 0x6b86: wc = 0x957a; break;
        case 0x6b87: wc = 0x9696; break;
        case 0x6b88: wc = 0x969e; break;
        case 0x6b89: wc = 0x96a9; break;
        case 0x6b8a: wc = 0x4ae8; break;
        case 0x6b8b: wc = 0x9a41; break;
        case 0x6b8c: wc = 0x4be0; break;
        case 0x6b8d: wc = 0x9d2e; break;
        case 0x6b8e: wc = 0x9dd4; break;
        case 0x6b8f: wc = 0x9f2f; break;
        case 0x6b90: wc = 0x526d; break;
        case 0x6b91: wc = 0x7821; break;
        case 0x6b92: wc = 0x83b9; break;
        case 0x6b93: wc = 0x8ec9; break;
        case 0x6b94: wc = 0x92c8; break;
        case 0x6b95: wc = 0x9cff; break;
        case 0x6b96: wc = 0x586d; break;
        case 0x6b97: wc = 0x5abc; break;
        case 0x6b98: wc = 0x614d; break;
        case 0x6b99: wc = 0x6435; break;
        case 0x6b9a: wc = 0x6637; break;
        case 0x6b9b: wc = 0x687d; break;
        case 0x6b9c: wc = 0x69b2; break;
        case 0x6b9d: wc = 0x6b9f; break;
        case 0x6b9e: wc = 0x6c33; break;
        case 0x6b9f: wc = 0x7185; break;
        case 0x6ba0: wc = 0x8495; break;
        case 0x6ba1: wc = 0x8580; break;
        case 0x6ba2: wc = 0x891e; break;
        case 0x6ba3: wc = 0x8c71; break;
        case 0x6ba4: wc = 0x8f40; break;
        case 0x6ba5: wc = 0x919e; break;
        case 0x6ba6: wc = 0x93be; break;
        case 0x6ba7: wc = 0x97b0; break;
        case 0x6ba8: wc = 0x97de; break;
        case 0x6ba9: wc = 0x9942; break;
        case 0x6baa: wc = 0x99a7; break;
        case 0x6bab: wc = 0x9c2e; break;
        case 0x6bac: wc = 0x341a; break;
        case 0x6bad: wc = 0x537c; break;
        case 0x6bae: wc = 0x55e2; break;
        case 0x6baf: wc = 0x5c7c; break;
        case 0x6bb0: wc = 0x6264; break;
        case 0x6bb1: wc = 0x674c; break;
        case 0x6bb2: wc = 0x6df4; break;
        case 0x6bb3: wc = 0x7783; break;
        case 0x6bb4: wc = 0x77f9; break;
        case 0x6bb5: wc = 0x8183; break;
        case 0x6bb6: wc = 0x9622; break;
        case 0x6bb7: wc = 0x9770; break;
        case 0x6bb8: wc = 0x4ab2; break;
        case 0x6bb9: wc = 0x55c8; break;
        case 0x6bba: wc = 0x55e1; break;
        case 0x6bbb: wc = 0x5670; break;
        case 0x6bbc: wc = 0x5855; break;
        case 0x6bbd: wc = 0x5ef1; break;
        case 0x6bbe: wc = 0x66a1; break;
        case 0x6bbf: wc = 0x6ec3; break;
        case 0x6bc0: wc = 0x6fad; break;
        case 0x6bc1: wc = 0x7049; break;
        case 0x6bc2: wc = 0x7788; break;
        case 0x6bc3: wc = 0x7f4b; break;
        case 0x6bc4: wc = 0x806c; break;
        case 0x6bc5: wc = 0x81c3; break;
        case 0x6bc6: wc = 0x84ca; break;
        case 0x6bc7: wc = 0x8579; break;
        case 0x6bc8: wc = 0x8789; break;
        case 0x6bc9: wc = 0x90fa; break;
        case 0x6bca: wc = 0x96dd; break;
        case 0x6bcb: wc = 0x9852; break;
        case 0x6bcc: wc = 0x297c0; break;
        case 0x6bcd: wc = 0x9c05; break;
        case 0x6bce: wc = 0x9db2; break;
        case 0x6bcf: wc = 0x9f46; break;
        case 0x6bd0: wc = 0x4f64; break;
        case 0x6bd1: wc = 0x542a; break;
        case 0x6bd2: wc = 0x56ee; break;
        case 0x6bd3: wc = 0x36c2; break;
        case 0x6bd4: wc = 0x5a50; break;
        case 0x6bd5: wc = 0x22af7; break;
        case 0x6bd6: wc = 0x6370; break;
        case 0x6bd7: wc = 0x6799; break;
        case 0x6bd8: wc = 0x6d3c; break;
        case 0x6bd9: wc = 0x6e9b; break;
        case 0x6bda: wc = 0x6f25; break;
        case 0x6bdb: wc = 0x7171; break;
        case 0x6bdc: wc = 0x7327; break;
        case 0x6bdd: wc = 0x7819; break;
        case 0x6bde: wc = 0x7a8a; break;
        case 0x6bdf: wc = 0x8435; break;
        case 0x6be0: wc = 0x8b4c; break;
        case 0x6be1: wc = 0x8fd7; break;
        case 0x6be2: wc = 0x4d77; break;
        case 0x6be3: wc = 0x9f40; break;
        case 0x6be4: wc = 0x5213; break;
        case 0x6be5: wc = 0x525c; break;
        case 0x6be6: wc = 0x5558; break;
        case 0x6be7: wc = 0x57b8; break;
        case 0x6be8: wc = 0x57e6; break;
        case 0x6be9: wc = 0x59a7; break;
        case 0x6bea: wc = 0x5a60; break;
        case 0x6beb: wc = 0x5c8f; break;
        case 0x6bec: wc = 0x5fe8; break;
        case 0x6bed: wc = 0x60cb; break;
        case 0x6bee: wc = 0x628f; break;
        case 0x6bef: wc = 0x6356; break;
        case 0x6bf0: wc = 0x6365; break;
        case 0x6bf1: wc = 0x22bf2; break;
        case 0x6bf2: wc = 0x6db4; break;
        case 0x6bf3: wc = 0x70f7; break;
        case 0x6bf4: wc = 0x76cc; break;
        case 0x6bf5: wc = 0x7755; break;
        case 0x6bf6: wc = 0x7fb1; break;
        case 0x6bf7: wc = 0x858d; break;
        case 0x6bf8: wc = 0x873f; break;
        case 0x6bf9: wc = 0x8f10; break;
        case 0x6bfa: wc = 0x344c; break;
        case 0x6bfb: wc = 0x5c22; break;
        case 0x6bfc: wc = 0xfa0c; break;
        case 0x6bfd: wc = 0x5c23; break;
        case 0x6bfe: wc = 0x5c2b; break;
        case 0x6bff: wc = 0x6680; break;
        case 0x6c00: wc = 0x7007; break;
        case 0x6c01: wc = 0x738d; break;
        case 0x6c02: wc = 0x8fec; break;
        case 0x6c03: wc = 0x54c7; break;
        case 0x6c04: wc = 0x35cf; break;
        case 0x6c05: wc = 0x5aa7; break;
        case 0x6c06: wc = 0x5b6c; break;
        case 0x6c07: wc = 0x7dfa; break;
        case 0x6c08: wc = 0x9a27; break;
        case 0x6c09: wc = 0x504e; break;
        case 0x6c0a: wc = 0x20c3b; break;
        case 0x6c0b: wc = 0x5cde; break;
        case 0x6c0c: wc = 0x5d54; break;
        case 0x6c0d: wc = 0x5d34; break;
        case 0x6c0e: wc = 0x5ec6; break;
        case 0x6c0f: wc = 0x5fbb; break;
        case 0x6c10: wc = 0x6104; break;
        case 0x6c11: wc = 0x63cb; break;
        case 0x6c12: wc = 0x6845; break;
        case 0x6c13: wc = 0x6933; break;
        case 0x6c14: wc = 0x6e28; break;
        case 0x6c15: wc = 0x6ebe; break;
        case 0x6c16: wc = 0x7168; break;
        case 0x6c17: wc = 0x7623; break;
        case 0x6c18: wc = 0x78a8; break;
        case 0x6c19: wc = 0x78c8; break;
        case 0x6c1a: wc = 0x8075; break;
        case 0x6c1b: wc = 0x8172; break;
        case 0x6c1c: wc = 0x8468; break;
        case 0x6c1d: wc = 0x8a74; break;
        case 0x6c1e: wc = 0x9361; break;
        case 0x6c1f: wc = 0x9688; break;
        case 0x6c20: wc = 0x9697; break;
        case 0x6c21: wc = 0x9ba0; break;
        case 0x6c22: wc = 0x591e; break;
        case 0x6c23: wc = 0x4e48; break;
        case 0x6c24: wc = 0x5060; break;
        case 0x6c25: wc = 0x509c; break;
        case 0x6c26: wc = 0x5406; break;
        case 0x6c27: wc = 0x5593; break;
        case 0x6c28: wc = 0x55c2; break;
        case 0x6c29: wc = 0x5773; break;
        case 0x6c2a: wc = 0x579a; break;
        case 0x6c2b: wc = 0x589d; break;
        case 0x6c2c: wc = 0x5ab1; break;
        case 0x6c2d: wc = 0x5ad1; break;
        case 0x6c2e: wc = 0x5b08; break;
        case 0x6c2f: wc = 0x5b8e; break;
        case 0x6c30: wc = 0x3754; break;
        case 0x6c31: wc = 0x5c86; break;
        case 0x6c32: wc = 0x5d3e; break;
        case 0x6c33: wc = 0x5e7a; break;
        case 0x6c34: wc = 0x5fad; break;
        case 0x6c35: wc = 0x5fbc; break;
        case 0x6c36: wc = 0x612e; break;
        case 0x6c37: wc = 0x61bf; break;
        case 0x6c38: wc = 0x67fc; break;
        case 0x6c39: wc = 0x67ea; break;
        case 0x6c3a: wc = 0x688e; break;
        case 0x6c3b: wc = 0x69a3; break;
        case 0x6c3c: wc = 0x699a; break;
        case 0x6c3d: wc = 0x6b80; break;
        case 0x6c3e: wc = 0x3d2d; break;
        case 0x6c3f: wc = 0x6e94; break;
        case 0x6c40: wc = 0x6f86; break;
        case 0x6c41: wc = 0x70d1; break;
        case 0x6c42: wc = 0x733a; break;
        case 0x6c43: wc = 0x73e7; break;
        case 0x6c44: wc = 0x78d8; break;
        case 0x6c45: wc = 0x7945; break;
        case 0x6c46: wc = 0x7a7e; break;
        case 0x6c47: wc = 0x7a85; break;
        case 0x6c48: wc = 0x7a94; break;
        case 0x6c49: wc = 0x7b44; break;
        case 0x6c4a: wc = 0x7df0; break;
        case 0x6c4b: wc = 0x430a; break;
        case 0x6c4c: wc = 0x26537; break;
        case 0x6c4d: wc = 0x8200; break;
        case 0x6c4e: wc = 0x825e; break;
        case 0x6c4f: wc = 0x82ba; break;
        case 0x6c50: wc = 0x82ed; break;
        case 0x6c51: wc = 0x847d; break;
        case 0x6c52: wc = 0x84d4; break;
        case 0x6c53: wc = 0x8558; break;
        case 0x6c54: wc = 0x888e; break;
        case 0x6c55: wc = 0x4645; break;
        case 0x6c56: wc = 0x8953; break;
        case 0x6c57: wc = 0x899e; break;
        case 0x6c58: wc = 0x8a1e; break;
        case 0x6c59: wc = 0x8a4f; break;
        case 0x6c5a: wc = 0x9076; break;
        case 0x6c5b: wc = 0x929a; break;
        case 0x6c5c: wc = 0x96a2; break;
        case 0x6c5d: wc = 0x977f; break;
        case 0x6c5e: wc = 0x98bb; break;
        case 0x6c5f: wc = 0x9a15; break;
        case 0x6c60: wc = 0x9c29; break;
        case 0x6c61: wc = 0x9d22; break;
        case 0x6c62: wc = 0x9dc2; break;
        case 0x6c63: wc = 0x9dc2; break;
        case 0x6c64: wc = 0x9dd5; break;
        case 0x6c65: wc = 0x55d5; break;
        case 0x6c66: wc = 0x5ab7; break;
        case 0x6c67: wc = 0x5cea; break;
        case 0x6c68: wc = 0x6ebd; break;
        case 0x6c69: wc = 0x84d0; break;
        case 0x6c6a: wc = 0x8f0d; break;
        case 0x6c6b: wc = 0x910f; break;
        case 0x6c6c: wc = 0x92ca; break;
        case 0x6c6d: wc = 0x9d52; break;
        case 0x6c6e: wc = 0x4f63; break;
        case 0x6c6f: wc = 0x509b; break;
        case 0x6c70: wc = 0x5087; break;
        case 0x6c71: wc = 0x584e; break;
        case 0x6c72: wc = 0x5ac6; break;
        case 0x6c73: wc = 0x5ab6; break;
        case 0x6c74: wc = 0x5ade; break;
        case 0x6c75: wc = 0x5b82; break;
        case 0x6c76: wc = 0x5d71; break;
        case 0x6c77: wc = 0x6080; break;
        case 0x6c78: wc = 0x607f; break;
        case 0x6c79: wc = 0x6175; break;
        case 0x6c7a: wc = 0x644f; break;
        case 0x6c7b: wc = 0x69b5; break;
        case 0x6c7c: wc = 0x69e6; break;
        case 0x6c7d: wc = 0x6c04; break;
        case 0x6c7e: wc = 0x6efd; break;
        case 0x6c7f: wc = 0x752d; break;
        case 0x6c80: wc = 0x752e; break;
        case 0x6c81: wc = 0x783d; break;
        case 0x6c82: wc = 0x7b69; break;
        case 0x6c83: wc = 0x8202; break;
        case 0x6c84: wc = 0x86f9; break;
        case 0x6c85: wc = 0x8923; break;
        case 0x6c86: wc = 0x8e34; break;
        case 0x6c87: wc = 0x8e56; break;
        case 0x6c88: wc = 0x8ef5; break;
        case 0x6c89: wc = 0x9118; break;
        case 0x6c8a: wc = 0x9af6; break;
        case 0x6c8b: wc = 0x9c2b; break;
        case 0x6c8c: wc = 0x9c45; break;
        case 0x6c8d: wc = 0x9ddb; break;
        case 0x6c8e: wc = 0x4e8f; break;
        case 0x6c8f: wc = 0x4eb4; break;
        case 0x6c90: wc = 0x4f18; break;
        case 0x6c91: wc = 0x4fc1; break;
        case 0x6c92: wc = 0x504a; break;
        case 0x6c93: wc = 0x5401; break;
        case 0x6c94: wc = 0x5418; break;
        case 0x6c95: wc = 0x543d; break;
        case 0x6c96: wc = 0x5581; break;
        case 0x6c97: wc = 0x5673; break;
        case 0x6c98: wc = 0x5698; break;
        case 0x6c99: wc = 0x5729; break;
        case 0x6c9a: wc = 0x5823; break;
        case 0x6c9b: wc = 0x3762; break;
        case 0x6c9c: wc = 0x5cdf; break;
        case 0x6c9d: wc = 0x5d4e; break;
        case 0x6c9e: wc = 0x616a; break;
        case 0x6c9f: wc = 0x61ee; break;
        case 0x6ca0: wc = 0x625c; break;
        case 0x6ca1: wc = 0x6270; break;
        case 0x6ca2: wc = 0x3b02; break;
        case 0x6ca3: wc = 0x6745; break;
        case 0x6ca4: wc = 0x6940; break;
        case 0x6ca5: wc = 0x6acc; break;
        case 0x6ca6: wc = 0x6ad9; break;
        case 0x6ca7: wc = 0x6c8b; break;
        case 0x6ca8: wc = 0x3cd3; break;
        case 0x6ca9: wc = 0x6e61; break;
        case 0x6caa: wc = 0x6f9e; break;
        case 0x6cab: wc = 0x7000; break;
        case 0x6cac: wc = 0x725c; break;
        case 0x6cad: wc = 0x7298; break;
        case 0x6cae: wc = 0x7376; break;
        case 0x6caf: wc = 0x75a3; break;
        case 0x6cb0: wc = 0x76d3; break;
        case 0x6cb1: wc = 0x76f1; break;
        case 0x6cb2: wc = 0x7964; break;
        case 0x6cb3: wc = 0x79ba; break;
        case 0x6cb4: wc = 0x6b76; break;
        case 0x6cb5: wc = 0x7a7b; break;
        case 0x6cb6: wc = 0x7afd; break;
        case 0x6cb7: wc = 0x42c5; break;
        case 0x6cb8: wc = 0x7e8b; break;
        case 0x6cb9: wc = 0x7fe2; break;
        case 0x6cba: wc = 0x8026; break;
        case 0x6cbb: wc = 0x2650a; break;
        case 0x6cbc: wc = 0x8030; break;
        case 0x6cbd: wc = 0x43cf; break;
        case 0x6cbe: wc = 0x80ac; break;
        case 0x6cbf: wc = 0x8162; break;
        case 0x6cc0: wc = 0x82c3; break;
        case 0x6cc1: wc = 0x842d; break;
        case 0x6cc2: wc = 0x8545; break;
        case 0x6cc3: wc = 0x8867; break;
        case 0x6cc4: wc = 0x8a0f; break;
        case 0x6cc5: wc = 0x8a27; break;
        case 0x6cc6: wc = 0x8b23; break;
        case 0x6cc7: wc = 0x8e3d; break;
        case 0x6cc8: wc = 0x9098; break;
        case 0x6cc9: wc = 0x9105; break;
        case 0x6cca: wc = 0x913e; break;
        case 0x6ccb: wc = 0x935d; break;
        case 0x6ccc: wc = 0x93c2; break;
        case 0x6ccd: wc = 0x9653; break;
        case 0x6cce: wc = 0x9aac; break;
        case 0x6ccf: wc = 0x9ac3; break;
        case 0x6cd0: wc = 0x9b77; break;
        case 0x6cd1: wc = 0x9bc3; break;
        case 0x6cd2: wc = 0x9e06; break;
        case 0x6cd3: wc = 0x9e80; break;
        case 0x6cd4: wc = 0x9e8c; break;
        case 0x6cd5: wc = 0x9f72; break;
        case 0x6cd6: wc = 0x9f75; break;
        case 0x6cd7: wc = 0x52d7; break;
        case 0x6cd8: wc = 0x65ee; break;
        case 0x6cd9: wc = 0x71e0; break;
        case 0x6cda: wc = 0x7a22; break;
        case 0x6cdb: wc = 0x8581; break;
        case 0x6cdc: wc = 0x490b; break;
        case 0x6cdd: wc = 0x593d; break;
        case 0x6cde: wc = 0x5998; break;
        case 0x6cdf: wc = 0x60f2; break;
        case 0x6ce0: wc = 0x612a; break;
        case 0x6ce1: wc = 0x628e; break;
        case 0x6ce2: wc = 0x679f; break;
        case 0x6ce3: wc = 0x6c84; break;
        case 0x6ce4: wc = 0x6eb3; break;
        case 0x6ce5: wc = 0x7703; break;
        case 0x6ce6: wc = 0x7bd4; break;
        case 0x6ce7: wc = 0x7d1c; break;
        case 0x6ce8: wc = 0x7e1c; break;
        case 0x6ce9: wc = 0x7e67; break;
        case 0x6cea: wc = 0x816a; break;
        case 0x6ceb: wc = 0x8779; break;
        case 0x6cec: wc = 0x8cf1; break;
        case 0x6ced: wc = 0x9106; break;
        case 0x6cee: wc = 0x9116; break;
        case 0x6cef: wc = 0x9723; break;
        case 0x6cf0: wc = 0x4a75; break;
        case 0x6cf1: wc = 0x97d7; break;
        case 0x6cf2: wc = 0x992b; break;
        case 0x6cf3: wc = 0x4cb0; break;
        case 0x6cf4: wc = 0x341b; break;
        case 0x6cf5: wc = 0x3b57; break;
        case 0x6cf6: wc = 0x706a; break;
        case 0x6cf7: wc = 0x7229; break;
        case 0x6cf8: wc = 0x83c0; break;
        case 0x6cf9: wc = 0x9ee6; break;
        case 0x6cfa: wc = 0x5007; break;
        case 0x6cfb: wc = 0x51a4; break;
        case 0x6cfc: wc = 0x586c; break;
        case 0x6cfd: wc = 0x5917; break;
        case 0x6cfe: wc = 0x59b4; break;
        case 0x6cff: wc = 0x5ab4; break;
        case 0x6d00: wc = 0x3768; break;
        case 0x6d01: wc = 0x60cc; break;
        case 0x6d02: wc = 0x3973; break;
        case 0x6d03: wc = 0x667c; break;
        case 0x6d04: wc = 0x670a; break;
        case 0x6d05: wc = 0x676c; break;
        case 0x6d06: wc = 0x6965; break;
        case 0x6d07: wc = 0x69ac; break;
        case 0x6d08: wc = 0x699e; break;
        case 0x6d09: wc = 0x6e92; break;
        case 0x6d0a: wc = 0x7328; break;
        case 0x6d0b: wc = 0x7342; break;
        case 0x6d0c: wc = 0x7579; break;
        case 0x6d0d: wc = 0x7722; break;
        case 0x6d0e: wc = 0x7990; break;
        case 0x6d0f: wc = 0x7b0e; break;
        case 0x6d10: wc = 0x7da9; break;
        case 0x6d11: wc = 0x82ab; break;
        case 0x6d12: wc = 0x847e; break;
        case 0x6d13: wc = 0x849d; break;
        case 0x6d14: wc = 0x84ac; break;
        case 0x6d15: wc = 0x8597; break;
        case 0x6d16: wc = 0x85b3; break;
        case 0x6d17: wc = 0x8696; break;
        case 0x6d18: wc = 0x45a4; break;
        case 0x6d19: wc = 0x876f; break;
        case 0x6d1a: wc = 0x8788; break;
        case 0x6d1b: wc = 0x8911; break;
        case 0x6d1c: wc = 0x8ae2; break;
        case 0x6d1d: wc = 0x8b1c; break;
        case 0x6d1e: wc = 0x8c72; break;
        case 0x6d1f: wc = 0x8e20; break;
        case 0x6d20: wc = 0x908d; break;
        case 0x6d21: wc = 0x90a7; break;
        case 0x6d22: wc = 0x9228; break;
        case 0x6d23: wc = 0x92fa; break;
        case 0x6d24: wc = 0x93b1; break;
        case 0x6d25: wc = 0x95e7; break;
        case 0x6d26: wc = 0x9850; break;
        case 0x6d27: wc = 0x99cc; break;
        case 0x6d28: wc = 0x9a35; break;
        case 0x6d29: wc = 0x9d77; break;
        case 0x6d2a: wc = 0x9da2; break;
        case 0x6d2b: wc = 0x9eff; break;
        case 0x6d2c: wc = 0x4ef4; break;
        case 0x6d2d: wc = 0x5216; break;
        case 0x6d2e: wc = 0x6034; break;
        case 0x6d2f: wc = 0x6209; break;
        case 0x6d30: wc = 0x6288; break;
        case 0x6d31: wc = 0x6a3e; break;
        case 0x6d32: wc = 0x6ce7; break;
        case 0x6d33: wc = 0x72d8; break;
        case 0x6d34: wc = 0x7ca4; break;
        case 0x6d35: wc = 0x7cb5; break;
        case 0x6d36: wc = 0x868e; break;
        case 0x6d37: wc = 0x8ecf; break;
        case 0x6d38: wc = 0x9205; break;
        case 0x6d39: wc = 0x2988b; break;
        case 0x6d3a: wc = 0x5130; break;
        case 0x6d3b: wc = 0x5582; break;
        case 0x6d3c: wc = 0x559f; break;
        case 0x6d3d: wc = 0x55a1; break;
        case 0x6d3e: wc = 0x55b4; break;
        case 0x6d3f: wc = 0x5645; break;
        case 0x6d40: wc = 0x56f2; break;
        case 0x6d41: wc = 0x589b; break;
        case 0x6d42: wc = 0x5aa6; break;
        case 0x6d43: wc = 0x5a81; break;
        case 0x6d44: wc = 0x5bea; break;
        case 0x6d45: wc = 0x5cd7; break;
        case 0x6d46: wc = 0x5d23; break;
        case 0x6d47: wc = 0x5d8e; break;
        case 0x6d48: wc = 0x5e43; break;
        case 0x6d49: wc = 0x5fab; break;
        case 0x6d4a: wc = 0x6d08; break;
        case 0x6d4b: wc = 0x6e4b; break;
        case 0x6d4c: wc = 0x6f7f; break;
        case 0x6d4d: wc = 0x6f59; break;
        case 0x6d4e: wc = 0x7152; break;
        case 0x6d4f: wc = 0x715f; break;
        case 0x6d50: wc = 0x71a8; break;
        case 0x6d51: wc = 0x729a; break;
        case 0x6d52: wc = 0x72a9; break;
        case 0x6d53: wc = 0x732c; break;
        case 0x6d54: wc = 0x75ff; break;
        case 0x6d55: wc = 0x24e06; break;
        case 0x6d56: wc = 0x784a; break;
        case 0x6d57: wc = 0x7e05; break;
        case 0x6d58: wc = 0x7f7b; break;
        case 0x6d59: wc = 0x8473; break;
        case 0x6d5a: wc = 0x85ef; break;
        case 0x6d5b: wc = 0x8624; break;
        case 0x6d5c: wc = 0x8636; break;
        case 0x6d5d: wc = 0x8732; break;
        case 0x6d5e: wc = 0x875b; break;
        case 0x6d5f: wc = 0x2736b; break;
        case 0x6d60: wc = 0x87b1; break;
        case 0x6d61: wc = 0x87e1; break;
        case 0x6d62: wc = 0x893d; break;
        case 0x6d63: wc = 0x89a3; break;
        case 0x6d64: wc = 0x8ac9; break;
        case 0x6d65: wc = 0x8b8f; break;
        case 0x6d66: wc = 0x8d00; break;
        case 0x6d67: wc = 0x8e12; break;
        case 0x6d68: wc = 0x8e9b; break;
        case 0x6d69: wc = 0x9036; break;
        case 0x6d6a: wc = 0x912c; break;
        case 0x6d6b: wc = 0x934f; break;
        case 0x6d6c: wc = 0x95c8; break;
        case 0x6d6d: wc = 0xf0708; break;
        case 0x6d6e: wc = 0x9687; break;
        case 0x6d6f: wc = 0x9728; break;
        case 0x6d70: wc = 0x97d9; break;
        case 0x6d71: wc = 0x97e1; break;
        case 0x6d72: wc = 0x9820; break;
        case 0x6d73: wc = 0x9927; break;
        case 0x6d74: wc = 0x9935; break;
        case 0x6d75: wc = 0x9aaa; break;
        case 0x6d76: wc = 0x9aab; break;
        case 0x6d77: wc = 0x9c04; break;
        case 0x6d78: wc = 0x5187; break;
        case 0x6d79: wc = 0x5064; break;
        case 0x6d7a: wc = 0x3471; break;
        case 0x6d7b: wc = 0x2042d; break;
        case 0x6d7c: wc = 0x5198; break;
        case 0x6d7d: wc = 0x5363; break;
        case 0x6d7e: wc = 0x5466; break;
        case 0x6d7f: wc = 0x5685; break;
        case 0x6d80: wc = 0x56ff; break;
        case 0x6d81: wc = 0x582c; break;
        case 0x6d82: wc = 0x58dd; break;
        case 0x6d83: wc = 0x59f7; break;
        case 0x6d84: wc = 0x5a51; break;
        case 0x6d85: wc = 0x5a7e; break;
        case 0x6d86: wc = 0x5a83; break;
        case 0x6d87: wc = 0x5b2c; break;
        case 0x6d88: wc = 0x5b67; break;
        case 0x6d89: wc = 0x5bd9; break;
        case 0x6d8a: wc = 0x5cb0; break;
        case 0x6d8b: wc = 0x5d33; break;
        case 0x6d8c: wc = 0x5e37; break;
        case 0x6d8d: wc = 0x5eae; break;
        case 0x6d8e: wc = 0x38f8; break;
        case 0x6d8f: wc = 0x602e; break;
        case 0x6d90: wc = 0x62ad; break;
        case 0x6d91: wc = 0x63c9; break;
        case 0x6d92: wc = 0x64e9; break;
        case 0x6d93: wc = 0x6594; break;
        case 0x6d94: wc = 0x659e; break;
        case 0x6d95: wc = 0x65bf; break;
        case 0x6d96: wc = 0x66d8; break;
        case 0x6d97: wc = 0x6875; break;
        case 0x6d98: wc = 0x6970; break;
        case 0x6d99: wc = 0x697a; break;
        case 0x6d9a: wc = 0x69f1; break;
        case 0x6d9b: wc = 0x6b48; break;
        case 0x6d9c: wc = 0x6bf9; break;
        case 0x6d9d: wc = 0x6cd1; break;
        case 0x6d9e: wc = 0x6d5f; break;
        case 0x6d9f: wc = 0x6e2a; break;
        case 0x6da0: wc = 0x6efa; break;
        case 0x6da1: wc = 0x6ff0; break;
        case 0x6da2: wc = 0x6ffb; break;
        case 0x6da3: wc = 0x3d8b; break;
        case 0x6da4: wc = 0x7022; break;
        case 0x6da5: wc = 0x3d99; break;
        case 0x6da6: wc = 0x7163; break;
        case 0x6da7: wc = 0x71f8; break;
        case 0x6da8: wc = 0x7256; break;
        case 0x6da9: wc = 0x7270; break;
        case 0x6daa: wc = 0x72d6; break;
        case 0x6dab: wc = 0x7524; break;
        case 0x6dac: wc = 0x3f54; break;
        case 0x6dad: wc = 0x75cf; break;
        case 0x6dae: wc = 0x7610; break;
        case 0x6daf: wc = 0x7609; break;
        case 0x6db0: wc = 0x7711; break;
        case 0x6db1: wc = 0x776e; break;
        case 0x6db2: wc = 0x79b8; break;
        case 0x6db3: wc = 0x79de; break;
        case 0x6db4: wc = 0x259a0; break;
        case 0x6db5: wc = 0x7aac; break;
        case 0x6db6: wc = 0x7ab3; break;
        case 0x6db7: wc = 0x7c72; break;
        case 0x6db8: wc = 0x7cc5; break;
        case 0x6db9: wc = 0x7dcc; break;
        case 0x6dba: wc = 0x42f4; break;
        case 0x6dbb: wc = 0x7f91; break;
        case 0x6dbc: wc = 0x7fad; break;
        case 0x6dbd: wc = 0x8048; break;
        case 0x6dbe: wc = 0x8174; break;
        case 0x6dbf: wc = 0x837d; break;
        case 0x6dc0: wc = 0x83a0; break;
        case 0x6dc1: wc = 0x848f; break;
        case 0x6dc2: wc = 0x8447; break;
        case 0x6dc3: wc = 0x84f6; break;
        case 0x6dc4: wc = 0x854d; break;
        case 0x6dc5: wc = 0x8564; break;
        case 0x6dc6: wc = 0x8555; break;
        case 0x6dc7: wc = 0x85b7; break;
        case 0x6dc8: wc = 0x86b4; break;
        case 0x6dc9: wc = 0x86b0; break;
        case 0x6dca: wc = 0x873c; break;
        case 0x6dcb: wc = 0x8753; break;
        case 0x6dcc: wc = 0x8763; break;
        case 0x6dcd: wc = 0x875a; break;
        case 0x6dce: wc = 0x87a4; break;
        case 0x6dcf: wc = 0x87b8; break;
        case 0x6dd0: wc = 0x8915; break;
        case 0x6dd1: wc = 0x890e; break;
        case 0x6dd2: wc = 0x8966; break;
        case 0x6dd3: wc = 0x89a6; break;
        case 0x6dd4: wc = 0x8c81; break;
        case 0x6dd5: wc = 0x8c90; break;
        case 0x6dd6: wc = 0x8da1; break;
        case 0x6dd7: wc = 0x8e13; break;
        case 0x6dd8: wc = 0x8f2e; break;
        case 0x6dd9: wc = 0x8f36; break;
        case 0x6dda: wc = 0x8ff6; break;
        case 0x6ddb: wc = 0x900c; break;
        case 0x6ddc: wc = 0x9103; break;
        case 0x6ddd: wc = 0x9180; break;
        case 0x6dde: wc = 0x91b9; break;
        case 0x6ddf: wc = 0x923e; break;
        case 0x6de0: wc = 0x92aa; break;
        case 0x6de1: wc = 0x9352; break;
        case 0x6de2: wc = 0x9450; break;
        case 0x6de3: wc = 0x9683; break;
        case 0x6de4: wc = 0x97a3; break;
        case 0x6de5: wc = 0x97d6; break;
        case 0x6de6: wc = 0x986c; break;
        case 0x6de7: wc = 0x9a25; break;
        case 0x6de8: wc = 0x9a1f; break;
        case 0x6de9: wc = 0x9baa; break;
        case 0x6dea: wc = 0x9c07; break;
        case 0x6deb: wc = 0x9d94; break;
        case 0x6dec: wc = 0x9edd; break;
        case 0x6ded: wc = 0x9f2c; break;
        case 0x6dee: wc = 0x9fa5; break;
        case 0x6def: wc = 0x5125; break;
        case 0x6df0: wc = 0x5537; break;
        case 0x6df1: wc = 0x6def; break;
        case 0x6df2: wc = 0x7134; break;
        case 0x6df3: wc = 0x2ebc; break;
        case 0x6df4: wc = 0x84ae; break;
        case 0x6df5: wc = 0x861b; break;
        case 0x6df6: wc = 0x9325; break;
        case 0x6df7: wc = 0x38e7; break;
        case 0x6df8: wc = 0x6600; break;
        case 0x6df9: wc = 0x6a4d; break;
        case 0x6dfa: wc = 0x72c1; break;
        case 0x6dfb: wc = 0x43d6; break;
        case 0x6dfc: wc = 0x837a; break;
        case 0x6dfd: wc = 0x8733; break;
        case 0x6dfe: wc = 0x8761; break;
        case 0x6dff: wc = 0x9173; break;
        case 0x6e00: wc = 0x92c6; break;
        case 0x6e01: wc = 0x95a0; break;
        case 0x6e02: wc = 0x962d; break;
        case 0x6e03: wc = 0x99bb; break;
        case 0x6e04: wc = 0x3415; break;
        case 0x6e05: wc = 0x564a; break;
        case 0x6e06: wc = 0x6c69; break;
        case 0x6e07: wc = 0x71cf; break;
        case 0x6e08: wc = 0x77de; break;
        case 0x6e09: wc = 0x7e58; break;
        case 0x6e0a: wc = 0x9a48; break;
        case 0x6e0b: wc = 0x9c4a; break;
        case 0x6e0c: wc = 0x9d25; break;
        case 0x6e0d: wc = 0x5a00; break;
        case 0x6e0e: wc = 0x6be7; break;
        case 0x6e0f: wc = 0x6f0b; break;
        case 0x6e10: wc = 0x72e8; break;
        case 0x6e11: wc = 0x24dec; break;
        case 0x6e12: wc = 0x7fa2; break;
        case 0x6e13: wc = 0x809c; break;
        case 0x6e14: wc = 0x8319; break;
        case 0x6e15: wc = 0x99e5; break;
        case 0x6e16: wc = 0x542c; break;
        case 0x6e17: wc = 0x55ef; break;
        case 0x6e18: wc = 0x569a; break;
        case 0x6e19: wc = 0x5701; break;
        case 0x6e1a: wc = 0x57bd; break;
        case 0x6e1b: wc = 0x5cce; break;
        case 0x6e1c: wc = 0x5dbe; break;
        case 0x6e1d: wc = 0x6196; break;
        case 0x6e1e: wc = 0x6441; break;
        case 0x6e1f: wc = 0x65a6; break;
        case 0x6e20: wc = 0x6a83; break;
        case 0x6e21: wc = 0x6abc; break;
        case 0x6e22: wc = 0x6aad; break;
        case 0x6e23: wc = 0x6cff; break;
        case 0x6e24: wc = 0x6eb5; break;
        case 0x6e25: wc = 0x6fe6; break;
        case 0x6e26: wc = 0x72fa; break;
        case 0x6e27: wc = 0x730c; break;
        case 0x6e28: wc = 0x73e2; break;
        case 0x6e29: wc = 0x766e; break;
        case 0x6e2a: wc = 0x4325; break;
        case 0x6e2b: wc = 0x8491; break;
        case 0x6e2c: wc = 0x84bd; break;
        case 0x6e2d: wc = 0x861f; break;
        case 0x6e2e: wc = 0x4590; break;
        case 0x6e2f: wc = 0x273ed; break;
        case 0x6e30: wc = 0x8a14; break;
        case 0x6e31: wc = 0x8b94; break;
        case 0x6e32: wc = 0x911e; break;
        case 0x6e33: wc = 0x9f57; break;
        case 0x6e34: wc = 0x572a; break;
        case 0x6e35: wc = 0x91d4; break;
        case 0x6e36: wc = 0x9ce6; break;
        case 0x6e37: wc = 0x567e; break;
        case 0x6e38: wc = 0x5a6c; break;
        case 0x6e39: wc = 0x5d1f; break;
        case 0x6e3a: wc = 0x5ed5; break;
        case 0x6e3b: wc = 0x6114; break;
        case 0x6e3c: wc = 0x647f; break;
        case 0x6e3d: wc = 0x70ce; break;
        case 0x6e3e: wc = 0x7616; break;
        case 0x6e3f: wc = 0x7892; break;
        case 0x6e40: wc = 0x7aa8; break;
        case 0x6e41: wc = 0x82c2; break;
        case 0x6e42: wc = 0x8376; break;
        case 0x6e43: wc = 0x8d9b; break;
        case 0x6e44: wc = 0x9712; break;
        case 0x6e45: wc = 0x9720; break;
        case 0x6e46: wc = 0x972a; break;
        case 0x6e47: wc = 0x97fe; break;
        case 0x6e48: wc = 0x9de3; break;
        case 0x6e49: wc = 0x4fcb; break;
        case 0x6e4a: wc = 0x5508; break;
        case 0x6e4b: wc = 0x6092; break;
        case 0x6e4c: wc = 0x6339; break;
        case 0x6e4d: wc = 0x6b5e; break;
        case 0x6e4e: wc = 0x6d65; break;
        case 0x6e4f: wc = 0x6e47; break;
        case 0x6e50: wc = 0x6e46; break;
        case 0x6e51: wc = 0x88db; break;
        case 0x6e52: wc = 0x2ecf; break;
        case 0x6e53: wc = 0x4c52; break;
        case 0x6e54: wc = 0x34a3; break;
        case 0x6e55: wc = 0x565f; break;
        case 0x6e56: wc = 0x38f9; break;
        case 0x6e57: wc = 0x5fdc; break;
        case 0x6e58: wc = 0x24052; break;
        case 0x6e59: wc = 0x3d90; break;
        case 0x6e5a: wc = 0x3d9d; break;
        case 0x6e5b: wc = 0x252b9; break;
        case 0x6e5c: wc = 0x8b4d; break;
        case 0x6e5d: wc = 0x8ec8; break;
        case 0x6e5e: wc = 0x4e84; break;
        case 0x6e5f: wc = 0x506f; break;
        case 0x6e60: wc = 0x5117; break;
        case 0x6e61: wc = 0x5293; break;
        case 0x6e62: wc = 0x533b; break;
        case 0x6e63: wc = 0x541a; break;
        case 0x6e64: wc = 0x5bf2; break;
        case 0x6e65: wc = 0x5dac; break;
        case 0x6e66: wc = 0x6246; break;
        case 0x6e67: wc = 0x65d6; break;
        case 0x6e68: wc = 0x692c; break;
        case 0x6e69: wc = 0x6aa5; break;
        case 0x6e6a: wc = 0x6ab9; break;
        case 0x6e6b: wc = 0x6b39; break;
        case 0x6e6c: wc = 0x6bc9; break;
        case 0x6e6d: wc = 0x6f2a; break;
        case 0x6e6e: wc = 0x71f1; break;
        case 0x6e6f: wc = 0x7284; break;
        case 0x6e70: wc = 0x72cb; break;
        case 0x6e71: wc = 0x7317; break;
        case 0x6e72: wc = 0x3fc4; break;
        case 0x6e73: wc = 0x7912; break;
        case 0x6e74: wc = 0x7995; break;
        case 0x6e75: wc = 0x7a26; break;
        case 0x6e76: wc = 0x25a2f; break;
        case 0x6e77: wc = 0x7ae9; break;
        case 0x6e78: wc = 0x7c4e; break;
        case 0x6e79: wc = 0x7f9b; break;
        case 0x6e7a: wc = 0x8413; break;
        case 0x6e7b: wc = 0x85bf; break;
        case 0x6e7c: wc = 0x85d9; break;
        case 0x6e7d: wc = 0x8798; break;
        case 0x6e7e: wc = 0x87a0; break;
        case 0x6e7f: wc = 0x8864; break;
        case 0x6e80: wc = 0x89fa; break;
        case 0x6e81: wc = 0x8b69; break;
        case 0x6e82: wc = 0x8b7a; break;
        case 0x6e83: wc = 0x8f59; break;
        case 0x6e84: wc = 0x90fc; break;
        case 0x6e85: wc = 0x91b7; break;
        case 0x6e86: wc = 0x9218; break;
        case 0x6e87: wc = 0x92a5; break;
        case 0x6e88: wc = 0x943f; break;
        case 0x6e89: wc = 0x9857; break;
        case 0x6e8a: wc = 0x9950; break;
        case 0x6e8b: wc = 0x4c2e; break;
        case 0x6e8c: wc = 0x4c2f; break;
        case 0x6e8d: wc = 0x4c91; break;
        case 0x6e8e: wc = 0x9dfe; break;
        case 0x6e8f: wc = 0x9e03; break;
        case 0x6e90: wc = 0x340c; break;
        case 0x6e91: wc = 0x4f41; break;
        case 0x6e92: wc = 0x4f3f; break;
        case 0x6e93: wc = 0x4f95; break;
        case 0x6e94: wc = 0x4f74; break;
        case 0x6e95: wc = 0x4f87; break;
        case 0x6e96: wc = 0x3483; break;
        case 0x6e97: wc = 0x511e; break;
        case 0x6e98: wc = 0x5235; break;
        case 0x6e99: wc = 0x531c; break;
        case 0x6e9a: wc = 0x54a6; break;
        case 0x6e9b: wc = 0x54bf; break;
        case 0x6e9c: wc = 0x54a1; break;
        case 0x6e9d: wc = 0x572f; break;
        case 0x6e9e: wc = 0x5a90; break;
        case 0x6e9f: wc = 0x5ba7; break;
        case 0x6ea0: wc = 0x5c12; break;
        case 0x6ea1: wc = 0x5cd3; break;
        case 0x6ea2: wc = 0x5ccf; break;
        case 0x6ea3: wc = 0x5d3a; break;
        case 0x6ea4: wc = 0x5ed9; break;
        case 0x6ea5: wc = 0x5f02; break;
        case 0x6ea6: wc = 0x5f5d; break;
        case 0x6ea7: wc = 0x224e1; break;
        case 0x6ea8: wc = 0x605e; break;
        case 0x6ea9: wc = 0x6245; break;
        case 0x6eaa: wc = 0x6561; break;
        case 0x6eab: wc = 0x6686; break;
        case 0x6eac: wc = 0x675d; break;
        case 0x6ead: wc = 0x67c2; break;
        case 0x6eae: wc = 0x684b; break;
        case 0x6eaf: wc = 0x682d; break;
        case 0x6eb0: wc = 0x682e; break;
        case 0x6eb1: wc = 0x6938; break;
        case 0x6eb2: wc = 0x3b95; break;
        case 0x6eb3: wc = 0x6a32; break;
        case 0x6eb4: wc = 0x6b4b; break;
        case 0x6eb5: wc = 0x6b94; break;
        case 0x6eb6: wc = 0x6be6; break;
        case 0x6eb7: wc = 0x6d1f; break;
        case 0x6eb8: wc = 0x6d22; break;
        case 0x6eb9: wc = 0x6d31; break;
        case 0x6eba: wc = 0x6d0f; break;
        case 0x6ebb: wc = 0x3d23; break;
        case 0x6ebc: wc = 0x6f69; break;
        case 0x6ebd: wc = 0x3d8a; break;
        case 0x6ebe: wc = 0x71aa; break;
        case 0x6ebf: wc = 0x3ebf; break;
        case 0x6ec0: wc = 0x74f5; break;
        case 0x6ec1: wc = 0x7719; break;
        case 0x6ec2: wc = 0x7724; break;
        case 0x6ec3: wc = 0x7731; break;
        case 0x6ec4: wc = 0x7bb7; break;
        case 0x6ec5: wc = 0x7c03; break;
        case 0x6ec6: wc = 0x7d9b; break;
        case 0x6ec7: wc = 0x7fa0; break;
        case 0x6ec8: wc = 0x7fa1; break;
        case 0x6ec9: wc = 0x43b6; break;
        case 0x6eca: wc = 0x804f; break;
        case 0x6ecb: wc = 0x265ae; break;
        case 0x6ecc: wc = 0x8094; break;
        case 0x6ecd: wc = 0x80e3; break;
        case 0x6ece: wc = 0x80f0; break;
        case 0x6ecf: wc = 0x80f9; break;
        case 0x6ed0: wc = 0x2669f; break;
        case 0x6ed1: wc = 0x82e2; break;
        case 0x6ed2: wc = 0x834b; break;
        case 0x6ed3: wc = 0x85be; break;
        case 0x6ed4: wc = 0x27245; break;
        case 0x6ed5: wc = 0x86e6; break;
        case 0x6ed6: wc = 0x86dc; break;
        case 0x6ed7: wc = 0x8794; break;
        case 0x6ed8: wc = 0x8848; break;
        case 0x6ed9: wc = 0x886a; break;
        case 0x6eda: wc = 0x8898; break;
        case 0x6edb: wc = 0x88b2; break;
        case 0x6edc: wc = 0x8a11; break;
        case 0x6edd: wc = 0x8a51; break;
        case 0x6ede: wc = 0x8a52; break;
        case 0x6edf: wc = 0x8b3b; break;
        case 0x6ee0: wc = 0x8ca4; break;
        case 0x6ee1: wc = 0x8db0; break;
        case 0x6ee2: wc = 0x8de0; break;
        case 0x6ee3: wc = 0x8f00; break;
        case 0x6ee4: wc = 0x8f5c; break;
        case 0x6ee5: wc = 0x8fc6; break;
        case 0x6ee6: wc = 0x8fe4; break;
        case 0x6ee7: wc = 0x8ffb; break;
        case 0x6ee8: wc = 0x914f; break;
        case 0x6ee9: wc = 0x48f5; break;
        case 0x6eea: wc = 0x926f; break;
        case 0x6eeb: wc = 0x927a; break;
        case 0x6eec: wc = 0x9651; break;
        case 0x6eed: wc = 0x96b6; break;
        case 0x6eee: wc = 0x9809; break;
        case 0x6eef: wc = 0x9824; break;
        case 0x6ef0: wc = 0x984a; break;
        case 0x6ef1: wc = 0x99ec; break;
        case 0x6ef2: wc = 0x9ba7; break;
        case 0x6ef3: wc = 0x9b9e; break;
        case 0x6ef4: wc = 0x9d2f; break;
        case 0x6ef5: wc = 0x9edf; break;
        case 0x6ef6: wc = 0x55cc; break;
        case 0x6ef7: wc = 0x5f0b; break;
        case 0x6ef8: wc = 0x6759; break;
        case 0x6ef9: wc = 0x71a4; break;
        case 0x6efa: wc = 0x71bc; break;
        case 0x6efb: wc = 0x7348; break;
        case 0x6efc: wc = 0x8189; break;
        case 0x6efd: wc = 0x8257; break;
        case 0x6efe: wc = 0x8285; break;
        case 0x6eff: wc = 0x86e1; break;
        case 0x6f00: wc = 0x27fb0; break;
        case 0x6f01: wc = 0x91f4; break;
        case 0x6f02: wc = 0x96bf; break;
        case 0x6f03: wc = 0x972c; break;
        case 0x6f04: wc = 0x9dc1; break;
        case 0x6f05: wc = 0x4d2c; break;
        case 0x6f06: wc = 0x9ed3; break;
        case 0x6f07: wc = 0x9f78; break;
        case 0x6f08: wc = 0x4ebb; break;
        case 0x6f09: wc = 0x4ede; break;
        case 0x6f0a: wc = 0x513f; break;
        case 0x6f0b: wc = 0x51d0; break;
        case 0x6f0c: wc = 0x5794; break;
        case 0x6f0d: wc = 0x5819; break;
        case 0x6f0e: wc = 0x5924; break;
        case 0x6f0f: wc = 0x5a63; break;
        case 0x6f10: wc = 0x5c7b; break;
        case 0x6f11: wc = 0x5ef4; break;
        case 0x6f12: wc = 0x622d; break;
        case 0x6f13: wc = 0x6268; break;
        case 0x6f14: wc = 0x6704; break;
        case 0x6f15: wc = 0x6b2d; break;
        case 0x6f16: wc = 0x6b45; break;
        case 0x6f17: wc = 0x6ba5; break;
        case 0x6f18: wc = 0x6c24; break;
        case 0x6f19: wc = 0x3cbd; break;
        case 0x6f1a: wc = 0x6d07; break;
        case 0x6f1b: wc = 0x6e5a; break;
        case 0x6f1c: wc = 0x7263; break;
        case 0x6f1d: wc = 0x798b; break;
        case 0x6f1e: wc = 0x7c7e; break;
        case 0x6f1f: wc = 0x7c8c; break;
        case 0x6f20: wc = 0x7d09; break;
        case 0x6f21: wc = 0x7df8; break;
        case 0x6f22: wc = 0x8095; break;
        case 0x6f23: wc = 0x82a2; break;
        case 0x6f24: wc = 0x831a; break;
        case 0x6f25: wc = 0x8375; break;
        case 0x6f26: wc = 0x8529; break;
        case 0x6f27: wc = 0x87be; break;
        case 0x6f28: wc = 0x88c0; break;
        case 0x6f29: wc = 0x8a12; break;
        case 0x6f2a: wc = 0x8af2; break;
        case 0x6f2b: wc = 0x8ed4; break;
        case 0x6f2c: wc = 0x920f; break;
        case 0x6f2d: wc = 0x92a6; break;
        case 0x6f2e: wc = 0x93d4; break;
        case 0x6f2f: wc = 0x95c9; break;
        case 0x6f30: wc = 0x967b; break;
        case 0x6f31: wc = 0x97cc; break;
        case 0x6f32: wc = 0x99f0; break;
        case 0x6f33: wc = 0x52ae; break;
        case 0x6f34: wc = 0x564e; break;
        case 0x6f35: wc = 0x6b25; break;
        case 0x6f36: wc = 0x6cc6; break;
        case 0x6f37: wc = 0x6d42; break;
        case 0x6f38: wc = 0x71da; break;
        case 0x6f39: wc = 0x8875; break;
        case 0x6f3a: wc = 0x8fb7; break;
        case 0x6f3b: wc = 0x91f0; break;
        case 0x6f3c: wc = 0x9224; break;
        case 0x6f3d: wc = 0x4b7f; break;
        case 0x6f3e: wc = 0x9d29; break;
        case 0x6f3f: wc = 0x6820; break;
        case 0x6f40: wc = 0x68ef; break;
        case 0x6f41: wc = 0x7d1d; break;
        case 0x6f42: wc = 0x43d5; break;
        case 0x6f43: wc = 0x814d; break;
        case 0x6f44: wc = 0x887d; break;
        case 0x6f45: wc = 0x88b5; break;
        case 0x6f46: wc = 0x928b; break;
        case 0x6f47: wc = 0x98ea; break;
        case 0x6f48: wc = 0x9d40; break;
        case 0x6f49: wc = 0x53fa; break;
        case 0x6f4a: wc = 0x8fbc; break;
        case 0x6f4b: wc = 0x9cf0; break;
        case 0x6f4c: wc = 0x5ab5; break;
        case 0x6f4d: wc = 0x6254; break;
        case 0x6f4e: wc = 0x793d; break;
        case 0x6f4f: wc = 0x827f; break;
        case 0x6f50: wc = 0x8fb8; break;
        case 0x6f51: wc = 0x967e; break;
        case 0x6f52: wc = 0x5407; break;
        case 0x6f53: wc = 0x5470; break;
        case 0x6f54: wc = 0x5472; break;
        case 0x6f55: wc = 0x5559; break;
        case 0x6f56: wc = 0x556b; break;
        case 0x6f57: wc = 0x55de; break;
        case 0x6f58: wc = 0x5790; break;
        case 0x6f59: wc = 0x59ca; break;
        case 0x6f5a: wc = 0x5b28; break;
        case 0x6f5b: wc = 0x5b56; break;
        case 0x6f5c: wc = 0x5b76; break;
        case 0x6f5d: wc = 0x5c58; break;
        case 0x6f5e: wc = 0x5d6b; break;
        case 0x6f5f: wc = 0x5e9b; break;
        case 0x6f60: wc = 0x627b; break;
        case 0x6f61: wc = 0x673f; break;
        case 0x6f62: wc = 0x674d; break;
        case 0x6f63: wc = 0x67d8; break;
        case 0x6f64: wc = 0x6825; break;
        case 0x6f65: wc = 0x69a8; break;
        case 0x6f66: wc = 0x6a74; break;
        case 0x6f67: wc = 0x6ea0; break;
        case 0x6f68: wc = 0x6fac; break;
        case 0x6f69: wc = 0x6fe8; break;
        case 0x6f6a: wc = 0x7278; break;
        case 0x6f6b: wc = 0x73bc; break;
        case 0x6f6c: wc = 0x75c4; break;
        case 0x6f6d: wc = 0x7726; break;
        case 0x6f6e: wc = 0x7725; break;
        case 0x6f6f: wc = 0x77f7; break;
        case 0x6f70: wc = 0x40a3; break;
        case 0x6f71: wc = 0x410d; break;
        case 0x6f72: wc = 0x79c4; break;
        case 0x6f73: wc = 0x79ed; break;
        case 0x6f74: wc = 0x79f6; break;
        case 0x6f75: wc = 0x7b2b; break;
        case 0x6f76: wc = 0x7c7d; break;
        case 0x6f77: wc = 0x7ca2; break;
        case 0x6f78: wc = 0x8014; break;
        case 0x6f79: wc = 0x80cf; break;
        case 0x6f7a: wc = 0x43d1; break;
        case 0x6f7b: wc = 0x80d4; break;
        case 0x6f7c: wc = 0x80fe; break;
        case 0x6f7d: wc = 0x4409; break;
        case 0x6f7e: wc = 0x8293; break;
        case 0x6f7f: wc = 0x8308; break;
        case 0x6f80: wc = 0x8332; break;
        case 0x6f81: wc = 0x83bf; break;
        case 0x6f82: wc = 0x858b; break;
        case 0x6f83: wc = 0x869d; break;
        case 0x6f84: wc = 0x86d3; break;
        case 0x6f85: wc = 0x87c5; break;
        case 0x6f86: wc = 0x8800; break;
        case 0x6f87: wc = 0x89dc; break;
        case 0x6f88: wc = 0x8a3f; break;
        case 0x6f89: wc = 0x8a3e; break;
        case 0x6f8a: wc = 0x8cb2; break;
        case 0x6f8b: wc = 0x8d6d; break;
        case 0x6f8c: wc = 0x8d80; break;
        case 0x6f8d: wc = 0x8d91; break;
        case 0x6f8e: wc = 0x9111; break;
        case 0x6f8f: wc = 0x91a1; break;
        case 0x6f90: wc = 0x922d; break;
        case 0x6f91: wc = 0x92c5; break;
        case 0x6f92: wc = 0x93a1; break;
        case 0x6f93: wc = 0x28ac3; break;
        case 0x6f94: wc = 0x983f; break;
        case 0x6f95: wc = 0x98fa; break;
        case 0x6f96: wc = 0x9908; break;
        case 0x6f97: wc = 0x9ab4; break;
        case 0x6f98: wc = 0x9aca; break;
        case 0x6f99: wc = 0x9aed; break;
        case 0x6f9a: wc = 0x4c07; break;
        case 0x6f9b: wc = 0x9b93; break;
        case 0x6f9c: wc = 0x9dbf; break;
        case 0x6f9d: wc = 0x9dd3; break;
        case 0x6f9e: wc = 0x9f12; break;
        case 0x6f9f: wc = 0x9f4d; break;
        case 0x6fa0: wc = 0x5734; break;
        case 0x6fa1: wc = 0x5981; break;
        case 0x6fa2: wc = 0x5a65; break;
        case 0x6fa3: wc = 0x5c9d; break;
        case 0x6fa4: wc = 0x600d; break;
        case 0x6fa5: wc = 0x625a; break;
        case 0x6fa6: wc = 0x65b1; break;
        case 0x6fa7: wc = 0x67de; break;
        case 0x6fa8: wc = 0x6c4b; break;
        case 0x6fa9: wc = 0x7042; break;
        case 0x6faa: wc = 0x712f; break;
        case 0x6fab: wc = 0x721d; break;
        case 0x6fac: wc = 0x72b3; break;
        case 0x6fad: wc = 0x76ad; break;
        case 0x6fae: wc = 0x76b5; break;
        case 0x6faf: wc = 0x788f; break;
        case 0x6fb0: wc = 0x799a; break;
        case 0x6fb1: wc = 0x7a13; break;
        case 0x6fb2: wc = 0x7b70; break;
        case 0x6fb3: wc = 0x439e; break;
        case 0x6fb4: wc = 0x82f2; break;
        case 0x6fb5: wc = 0x838b; break;
        case 0x6fb6: wc = 0x8b36; break;
        case 0x6fb7: wc = 0x942f; break;
        case 0x6fb8: wc = 0x4c5c; break;
        case 0x6fb9: wc = 0x9ddf; break;
        case 0x6fba: wc = 0x50dd; break;
        case 0x6fbb: wc = 0x5257; break;
        case 0x6fbc: wc = 0x62c3; break;
        case 0x6fbd: wc = 0x7416; break;
        case 0x6fbe: wc = 0x8665; break;
        case 0x6fbf: wc = 0x8f1a; break;
        case 0x6fc0: wc = 0x8f4f; break;
        case 0x6fc1: wc = 0x9186; break;
        case 0x6fc2: wc = 0x9a4f; break;
        case 0x6fc3: wc = 0x4d3c; break;
        case 0x6fc4: wc = 0x4e7d; break;
        case 0x6fc5: wc = 0x5592; break;
        case 0x6fc6: wc = 0x57c1; break;
        case 0x6fc7: wc = 0x5bc1; break;
        case 0x6fc8: wc = 0x63dd; break;
        case 0x6fc9: wc = 0x648f; break;
        case 0x6fca: wc = 0x661d; break;
        case 0x6fcb: wc = 0x6d94; break;
        case 0x6fcc: wc = 0x71b8; break;
        case 0x6fcd: wc = 0x7876; break;
        case 0x6fce: wc = 0x7bf8; break;
        case 0x6fcf: wc = 0x81e2; break;
        case 0x6fd0: wc = 0x8638; break;
        case 0x6fd1: wc = 0x8b67; break;
        case 0x6fd2: wc = 0x8cfa; break;
        case 0x6fd3: wc = 0x8e54; break;
        case 0x6fd4: wc = 0x9415; break;
        case 0x6fd5: wc = 0x4ce5; break;
        case 0x6fd6: wc = 0x531d; break;
        case 0x6fd7: wc = 0x5361; break;
        case 0x6fd8: wc = 0x5494; break;
        case 0x6fd9: wc = 0x5482; break;
        case 0x6fda: wc = 0x5551; break;
        case 0x6fdb: wc = 0x5601; break;
        case 0x6fdc: wc = 0x56c3; break;
        case 0x6fdd: wc = 0x5e00; break;
        case 0x6fde: wc = 0x62b8; break;
        case 0x6fdf: wc = 0x62e4; break;
        case 0x6fe0: wc = 0x7160; break;
        case 0x6fe1: wc = 0x7580; break;
        case 0x6fe2: wc = 0x7728; break;
        case 0x6fe3: wc = 0x7838; break;
        case 0x6fe4: wc = 0x78fc; break;
        case 0x6fe5: wc = 0x7bd0; break;
        case 0x6fe6: wc = 0x894d; break;
        case 0x6fe7: wc = 0x8b57; break;
        case 0x6fe8: wc = 0x8fca; break;
        case 0x6fe9: wc = 0x9254; break;
        case 0x6fea: wc = 0x96e5; break;
        case 0x6feb: wc = 0x35ef; break;
        case 0x6fec: wc = 0x4ec9; break;
        case 0x6fed: wc = 0x50bd; break;
        case 0x6fee: wc = 0x364a; break;
        case 0x6fef: wc = 0x5887; break;
        case 0x6ff0: wc = 0x588f; break;
        case 0x6ff1: wc = 0x5958; break;
        case 0x6ff2: wc = 0x599d; break;
        case 0x6ff3: wc = 0x5adc; break;
        case 0x6ff4: wc = 0x5b19; break;
        case 0x6ff5: wc = 0x5d82; break;
        case 0x6ff6: wc = 0x5d88; break;
        case 0x6ff7: wc = 0x5e5b; break;
        case 0x6ff8: wc = 0x22248; break;
        case 0x6ff9: wc = 0x5ee7; break;
        case 0x6ffa: wc = 0x615e; break;
        case 0x6ffb: wc = 0x6215; break;
        case 0x6ffc: wc = 0x6259; break;
        case 0x6ffd: wc = 0x65a8; break;
        case 0x6ffe: wc = 0x6889; break;
        case 0x6fff: wc = 0x6a01; break;
        case 0x7000: wc = 0x69f3; break;
        case 0x7001: wc = 0x6f33; break;
        case 0x7002: wc = 0x710b; break;
        case 0x7003: wc = 0x723f; break;
        case 0x7004: wc = 0x7242; break;
        case 0x7005: wc = 0x7244; break;
        case 0x7006: wc = 0x7634; break;
        case 0x7007: wc = 0x7795; break;
        case 0x7008: wc = 0x7c80; break;
        case 0x7009: wc = 0x7cbb; break;
        case 0x700a: wc = 0x7cda; break;
        case 0x700b: wc = 0x7f98; break;
        case 0x700c: wc = 0x8193; break;
        case 0x700d: wc = 0x8262; break;
        case 0x700e: wc = 0x8407; break;
        case 0x700f: wc = 0x8620; break;
        case 0x7010: wc = 0x87d1; break;
        case 0x7011: wc = 0x87bf; break;
        case 0x7012: wc = 0x8ccd; break;
        case 0x7013: wc = 0x8cec; break;
        case 0x7014: wc = 0x8e61; break;
        case 0x7015: wc = 0x8ebc; break;
        case 0x7016: wc = 0x9067; break;
        case 0x7017: wc = 0x9123; break;
        case 0x7018: wc = 0xf0709; break;
        case 0x7019: wc = 0x93d8; break;
        case 0x701a: wc = 0x93f1; break;
        case 0x701b: wc = 0x9423; break;
        case 0x701c: wc = 0x9476; break;
        case 0x701d: wc = 0x9578; break;
        case 0x701e: wc = 0x979d; break;
        case 0x701f: wc = 0x9926; break;
        case 0x7020: wc = 0x99d4; break;
        case 0x7021: wc = 0x9a3f; break;
        case 0x7022: wc = 0x9ad2; break;
        case 0x7023: wc = 0x9c46; break;
        case 0x7024: wc = 0x9c42; break;
        case 0x7025: wc = 0x9e9e; break;
        case 0x7026: wc = 0x4fa2; break;
        case 0x7027: wc = 0x510e; break;
        case 0x7028: wc = 0x20960; break;
        case 0x7029: wc = 0x558d; break;
        case 0x702a: wc = 0x635a; break;
        case 0x702b: wc = 0x699f; break;
        case 0x702c: wc = 0x6ea8; break;
        case 0x702d: wc = 0x707e; break;
        case 0x702e: wc = 0x7e94; break;
        case 0x702f: wc = 0x4774; break;
        case 0x7030: wc = 0x8ceb; break;
        case 0x7031: wc = 0x8cf3; break;
        case 0x7032: wc = 0x9168; break;
        case 0x7033: wc = 0x91ee; break;
        case 0x7034: wc = 0x9f5c; break;
        case 0x7035: wc = 0x564c; break;
        case 0x7036: wc = 0x5d22; break;
        case 0x7037: wc = 0x5d1d; break;
        case 0x7038: wc = 0x6399; break;
        case 0x7039: wc = 0x6381; break;
        case 0x703a: wc = 0x68e6; break;
        case 0x703b: wc = 0x7319; break;
        case 0x703c: wc = 0x7424; break;
        case 0x703d: wc = 0x7880; break;
        case 0x703e: wc = 0x8d9f; break;
        case 0x703f: wc = 0x9397; break;
        case 0x7040: wc = 0x93ff; break;
        case 0x7041: wc = 0x93f3; break;
        case 0x7042: wc = 0x9953; break;
        case 0x7043: wc = 0x4f39; break;
        case 0x7044: wc = 0x5467; break;
        case 0x7045: wc = 0x5979; break;
        case 0x7046: wc = 0x5b81; break;
        case 0x7047: wc = 0x5ca8; break;
        case 0x7048: wc = 0x5f24; break;
        case 0x7049: wc = 0x2252a; break;
        case 0x704a: wc = 0x601a; break;
        case 0x704b: wc = 0x62af; break;
        case 0x704c: wc = 0x6474; break;
        case 0x704d: wc = 0x3add; break;
        case 0x704e: wc = 0x677c; break;
        case 0x704f: wc = 0x67e2; break;
        case 0x7050: wc = 0x6ae1; break;
        case 0x7051: wc = 0x6ae7; break;
        case 0x7052: wc = 0x6aeb; break;
        case 0x7053: wc = 0x6c10; break;
        case 0x7054: wc = 0x6cde; break;
        case 0x7055: wc = 0x6ec1; break;
        case 0x7056: wc = 0x7026; break;
        case 0x7057: wc = 0x3d86; break;
        case 0x7058: wc = 0x7274; break;
        case 0x7059: wc = 0x75a7; break;
        case 0x705a: wc = 0x75b7; break;
        case 0x705b: wc = 0x771d; break;
        case 0x705c: wc = 0x7820; break;
        case 0x705d: wc = 0x7ada; break;
        case 0x705e: wc = 0x7b6f; break;
        case 0x705f: wc = 0x7be8; break;
        case 0x7060: wc = 0x7f5d; break;
        case 0x7061: wc = 0x7f9c; break;
        case 0x7062: wc = 0x7f9d; break;
        case 0x7063: wc = 0x7fe5; break;
        case 0x7064: wc = 0x805c; break;
        case 0x7065: wc = 0x82f4; break;
        case 0x7066: wc = 0x830b; break;
        case 0x7067: wc = 0x83e7; break;
        case 0x7068: wc = 0x8445; break;
        case 0x7069: wc = 0x85f8; break;
        case 0x706a: wc = 0x86c6; break;
        case 0x706b: wc = 0x876b; break;
        case 0x706c: wc = 0x8829; break;
        case 0x706d: wc = 0x889b; break;
        case 0x706e: wc = 0x8893; break;
        case 0x706f: wc = 0x891a; break;
        case 0x7070: wc = 0x89dd; break;
        case 0x7071: wc = 0x8a5d; break;
        case 0x7072: wc = 0x8a46; break;
        case 0x7073: wc = 0x8c6c; break;
        case 0x7074: wc = 0x8d84; break;
        case 0x7075: wc = 0x8dd9; break;
        case 0x7076: wc = 0x8ee7; break;
        case 0x7077: wc = 0x963a; break;
        case 0x7078: wc = 0x967c; break;
        case 0x7079: wc = 0x9ab6; break;
        case 0x707a: wc = 0x9d21; break;
        case 0x707b: wc = 0x5547; break;
        case 0x707c: wc = 0x5600; break;
        case 0x707d: wc = 0x5681; break;
        case 0x707e: wc = 0x5891; break;
        case 0x707f: wc = 0x5eb4; break;
        case 0x7080: wc = 0x5eed; break;
        case 0x7081: wc = 0x5ef8; break;
        case 0x7082: wc = 0x64ff; break;
        case 0x7083: wc = 0x65f3; break;
        case 0x7084: wc = 0x6891; break;
        case 0x7085: wc = 0x6a00; break;
        case 0x7086: wc = 0x6a0d; break;
        case 0x7087: wc = 0x6d7e; break;
        case 0x7088: wc = 0x6f03; break;
        case 0x7089: wc = 0x71b5; break;
        case 0x708a: wc = 0x7393; break;
        case 0x708b: wc = 0x74cb; break;
        case 0x708c: wc = 0x750b; break;
        case 0x708d: wc = 0x766a; break;
        case 0x708e: wc = 0x7873; break;
        case 0x708f: wc = 0x78e7; break;
        case 0x7090: wc = 0x7bf4; break;
        case 0x7091: wc = 0x7c4a; break;
        case 0x7092: wc = 0x7cf4; break;
        case 0x7093: wc = 0x8024; break;
        case 0x7094: wc = 0x807b; break;
        case 0x7095: wc = 0x82d6; break;
        case 0x7096: wc = 0x83c2; break;
        case 0x7097: wc = 0x8443; break;
        case 0x7098: wc = 0x850b; break;
        case 0x7099: wc = 0x85e1; break;
        case 0x709a: wc = 0x8808; break;
        case 0x709b: wc = 0x8940; break;
        case 0x709c: wc = 0x89bf; break;
        case 0x709d: wc = 0x8b81; break;
        case 0x709e: wc = 0x8c74; break;
        case 0x709f: wc = 0x8daf; break;
        case 0x70a0: wc = 0x8e16; break;
        case 0x70a1: wc = 0x8ea4; break;
        case 0x70a2: wc = 0x9016; break;
        case 0x70a3: wc = 0x976e; break;
        case 0x70a4: wc = 0x9814; break;
        case 0x70a5: wc = 0x99b0; break;
        case 0x70a6: wc = 0x9c3f; break;
        case 0x70a7: wc = 0x2fa0e; break;
        case 0x70a8: wc = 0x9e10; break;
        case 0x70a9: wc = 0x4f1d; break;
        case 0x70aa: wc = 0x500e; break;
        case 0x70ab: wc = 0x508e; break;
        case 0x70ac: wc = 0x5278; break;
        case 0x70ad: wc = 0x53a7; break;
        case 0x70ae: wc = 0x20fcf; break;
        case 0x70af: wc = 0x56c0; break;
        case 0x70b0: wc = 0x212c4; break;
        case 0x70b1: wc = 0x3649; break;
        case 0x70b2: wc = 0x58c2; break;
        case 0x70b3: wc = 0x59fe; break;
        case 0x70b4: wc = 0x5a5d; break;
        case 0x70b5: wc = 0x5a70; break;
        case 0x70b6: wc = 0x5a8a; break;
        case 0x70b7: wc = 0x5ae5; break;
        case 0x70b8: wc = 0x5c47; break;
        case 0x70b9: wc = 0x5dd3; break;
        case 0x70ba: wc = 0x5e34; break;
        case 0x70bb: wc = 0x606e; break;
        case 0x70bc: wc = 0x394f; break;
        case 0x70bd: wc = 0x6214; break;
        case 0x70be: wc = 0x6229; break;
        case 0x70bf: wc = 0x62f4; break;
        case 0x70c0: wc = 0x6375; break;
        case 0x70c1: wc = 0x63c3; break;
        case 0x70c2: wc = 0x6437; break;
        case 0x70c3: wc = 0x640c; break;
        case 0x70c4: wc = 0x6527; break;
        case 0x70c5: wc = 0x65c3; break;
        case 0x70c6: wc = 0x65dc; break;
        case 0x70c7: wc = 0x666a; break;
        case 0x70c8: wc = 0x66b7; break;
        case 0x70c9: wc = 0x6718; break;
        case 0x70ca: wc = 0x6834; break;
        case 0x70cb: wc = 0x69c7; break;
        case 0x70cc: wc = 0x6990; break;
        case 0x70cd: wc = 0x6a4f; break;
        case 0x70ce: wc = 0x6a3f; break;
        case 0x70cf: wc = 0x3c1c; break;
        case 0x70d0: wc = 0x6cba; break;
        case 0x70d1: wc = 0x6ddf; break;
        case 0x70d2: wc = 0x6e79; break;
        case 0x70d3: wc = 0x6e54; break;
        case 0x70d4: wc = 0x6ec7; break;
        case 0x70d5: wc = 0x700d; break;
        case 0x70d6: wc = 0x70c7; break;
        case 0x70d7: wc = 0x724b; break;
        case 0x70d8: wc = 0x7277; break;
        case 0x70d9: wc = 0x7320; break;
        case 0x70da: wc = 0x7451; break;
        case 0x70db: wc = 0x7454; break;
        case 0x70dc: wc = 0x7450; break;
        case 0x70dd: wc = 0x747c; break;
        case 0x70de: wc = 0x750e; break;
        case 0x70df: wc = 0x754b; break;
        case 0x70e0: wc = 0x7560; break;
        case 0x70e1: wc = 0x75ca; break;
        case 0x70e2: wc = 0x75f6; break;
        case 0x70e3: wc = 0x7628; break;
        case 0x70e4: wc = 0x765c; break;
        case 0x70e5: wc = 0x76bd; break;
        case 0x70e6: wc = 0x76f7; break;
        case 0x70e7: wc = 0x7898; break;
        case 0x70e8: wc = 0x78cc; break;
        case 0x70e9: wc = 0x78da; break;
        case 0x70ea: wc = 0x7ab4; break;
        case 0x70eb: wc = 0x7bff; break;
        case 0x70ec: wc = 0x7c5b; break;
        case 0x70ed: wc = 0x7ccb; break;
        case 0x70ee: wc = 0x7d5f; break;
        case 0x70ef: wc = 0x7e13; break;
        case 0x70f0: wc = 0x7e33; break;
        case 0x70f1: wc = 0x7fb4; break;
        case 0x70f2: wc = 0x7fb6; break;
        case 0x70f3: wc = 0x7fe6; break;
        case 0x70f4: wc = 0x8146; break;
        case 0x70f5: wc = 0x815e; break;
        case 0x70f6: wc = 0x819e; break;
        case 0x70f7: wc = 0x81bb; break;
        case 0x70f8: wc = 0x81c7; break;
        case 0x70f9: wc = 0x8343; break;
        case 0x70fa: wc = 0x8465; break;
        case 0x70fb: wc = 0x8547; break;
        case 0x70fc: wc = 0x26f44; break;
        case 0x70fd: wc = 0x87e4; break;
        case 0x70fe: wc = 0x8ad3; break;
        case 0x70ff: wc = 0x8aef; break;
        case 0x7100: wc = 0x8b7e; break;
        case 0x7101: wc = 0x8cdf; break;
        case 0x7102: wc = 0x8dc8; break;
        case 0x7103: wc = 0x8de7; break;
        case 0x7104: wc = 0x8e4e; break;
        case 0x7105: wc = 0x8e4d; break;
        case 0x7106: wc = 0x8e94; break;
        case 0x7107: wc = 0x8f07; break;
        case 0x7108: wc = 0x9085; break;
        case 0x7109: wc = 0x911f; break;
        case 0x710a: wc = 0x92d1; break;
        case 0x710b: wc = 0x932a; break;
        case 0x710c: wc = 0x9386; break;
        case 0x710d: wc = 0x9409; break;
        case 0x710e: wc = 0x95d0; break;
        case 0x710f: wc = 0x975b; break;
        case 0x7110: wc = 0x9766; break;
        case 0x7111: wc = 0x9853; break;
        case 0x7112: wc = 0x98e6; break;
        case 0x7113: wc = 0x9930; break;
        case 0x7114: wc = 0x9958; break;
        case 0x7115: wc = 0x99e9; break;
        case 0x7116: wc = 0x9a1a; break;
        case 0x7117: wc = 0x9b0b; break;
        case 0x7118: wc = 0x9c44; break;
        case 0x7119: wc = 0x9c63; break;
        case 0x711a: wc = 0x9d2b; break;
        case 0x711b: wc = 0x9dc6; break;
        case 0x711c: wc = 0x9e07; break;
        case 0x711d: wc = 0x9f7b; break;
        case 0x711e: wc = 0x3409; break;
        case 0x711f: wc = 0x5369; break;
        case 0x7120: wc = 0x353e; break;
        case 0x7121: wc = 0x5c10; break;
        case 0x7122: wc = 0x5c8a; break;
        case 0x7123: wc = 0x5e6f; break;
        case 0x7124: wc = 0x6662; break;
        case 0x7125: wc = 0x6663; break;
        case 0x7126: wc = 0x68f3; break;
        case 0x7127: wc = 0x6976; break;
        case 0x7128: wc = 0x6c8f; break;
        case 0x7129: wc = 0x7a83; break;
        case 0x712a: wc = 0x44c6; break;
        case 0x712b: wc = 0x855d; break;
        case 0x712c: wc = 0x8818; break;
        case 0x712d: wc = 0x883d; break;
        case 0x712e: wc = 0x957b; break;
        case 0x712f: wc = 0x4c99; break;
        case 0x7130: wc = 0x4f54; break;
        case 0x7131: wc = 0x5538; break;
        case 0x7132: wc = 0x576b; break;
        case 0x7133: wc = 0x588a; break;
        case 0x7134: wc = 0x60e6; break;
        case 0x7135: wc = 0x6242; break;
        case 0x7136: wc = 0x6382; break;
        case 0x7137: wc = 0x7154; break;
        case 0x7138: wc = 0x73b7; break;
        case 0x7139: wc = 0x75c1; break;
        case 0x713a: wc = 0x78f9; break;
        case 0x713b: wc = 0x79e5; break;
        case 0x713c: wc = 0x7b18; break;
        case 0x713d: wc = 0x7c1f; break;
        case 0x713e: wc = 0x82eb; break;
        case 0x713f: wc = 0x84a7; break;
        case 0x7140: wc = 0x852a; break;
        case 0x7141: wc = 0x86c5; break;
        case 0x7142: wc = 0x87b9; break;
        case 0x7143: wc = 0x460b; break;
        case 0x7144: wc = 0x8e2e; break;
        case 0x7145: wc = 0x963d; break;
        case 0x7146: wc = 0x98ad; break;
        case 0x7147: wc = 0x9ecf; break;
        case 0x7148: wc = 0x6315; break;
        case 0x7149: wc = 0x6904; break;
        case 0x714a: wc = 0x696a; break;
        case 0x714b: wc = 0x8051; break;
        case 0x714c: wc = 0x8253; break;
        case 0x714d: wc = 0x83e8; break;
        case 0x714e: wc = 0x8728; break;
        case 0x714f: wc = 0x890b; break;
        case 0x7150: wc = 0x8dd5; break;
        case 0x7151: wc = 0x8e40; break;
        case 0x7152: wc = 0x9c08; break;
        case 0x7153: wc = 0x4e3c; break;
        case 0x7154: wc = 0x4ec3; break;
        case 0x7155: wc = 0x4fb9; break;
        case 0x7156: wc = 0x51c8; break;
        case 0x7157: wc = 0x53ee; break;
        case 0x7158: wc = 0x5576; break;
        case 0x7159: wc = 0x5722; break;
        case 0x715a: wc = 0x57d5; break;
        case 0x715b: wc = 0x57e5; break;
        case 0x715c: wc = 0x57e9; break;
        case 0x715d: wc = 0x5863; break;
        case 0x715e: wc = 0x5975; break;
        case 0x715f: wc = 0x598c; break;
        case 0x7160: wc = 0x5a17; break;
        case 0x7161: wc = 0x5a67; break;
        case 0x7162: wc = 0x5a77; break;
        case 0x7163: wc = 0x5a9c; break;
        case 0x7164: wc = 0x5bca; break;
        case 0x7165: wc = 0x5d49; break;
        case 0x7166: wc = 0x5d7f; break;
        case 0x7167: wc = 0x5e04; break;
        case 0x7168: wc = 0x6014; break;
        case 0x7169: wc = 0x609c; break;
        case 0x716a: wc = 0x639f; break;
        case 0x716b: wc = 0x63e8; break;
        case 0x716c: wc = 0x63c1; break;
        case 0x716d: wc = 0x22fc4; break;
        case 0x716e: wc = 0x65cd; break;
        case 0x716f: wc = 0x3acc; break;
        case 0x7170: wc = 0x673e; break;
        case 0x7171: wc = 0x6883; break;
        case 0x7172: wc = 0x686f; break;
        case 0x7173: wc = 0x68d6; break;
        case 0x7174: wc = 0x6917; break;
        case 0x7175: wc = 0x695f; break;
        case 0x7176: wc = 0x69b3; break;
        case 0x7177: wc = 0x6a99; break;
        case 0x7178: wc = 0x6c6b; break;
        case 0x7179: wc = 0x6d8f; break;
        case 0x717a: wc = 0x6fce; break;
        case 0x717b: wc = 0x706f; break;
        case 0x717c: wc = 0x70f6; break;
        case 0x717d: wc = 0x73f5; break;
        case 0x717e: wc = 0x7414; break;
        case 0x717f: wc = 0x7594; break;
        case 0x7180: wc = 0x76ef; break;
        case 0x7181: wc = 0x7710; break;
        case 0x7182: wc = 0x775c; break;
        case 0x7183: wc = 0x77f4; break;
        case 0x7184: wc = 0x78a0; break;
        case 0x7185: wc = 0x7aeb; break;
        case 0x7186: wc = 0x7b73; break;
        case 0x7187: wc = 0x7b90; break;
        case 0x7188: wc = 0x25e27; break;
        case 0x7189: wc = 0x7cfd; break;
        case 0x718a: wc = 0x8035; break;
        case 0x718b: wc = 0x8047; break;
        case 0x718c: wc = 0x8062; break;
        case 0x718d: wc = 0x8059; break;
        case 0x718e: wc = 0x8064; break;
        case 0x718f: wc = 0x812d; break;
        case 0x7190: wc = 0x8121; break;
        case 0x7191: wc = 0x8148; break;
        case 0x7192: wc = 0x815a; break;
        case 0x7193: wc = 0x827c; break;
        case 0x7194: wc = 0x839b; break;
        case 0x7195: wc = 0x8423; break;
        case 0x7196: wc = 0x8476; break;
        case 0x7197: wc = 0x85a1; break;
        case 0x7198: wc = 0x8670; break;
        case 0x7199: wc = 0x8713; break;
        case 0x719a: wc = 0x874a; break;
        case 0x719b: wc = 0x874f; break;
        case 0x719c: wc = 0x87f6; break;
        case 0x719d: wc = 0x88ce; break;
        case 0x719e: wc = 0x4793; break;
        case 0x719f: wc = 0x8d6c; break;
        case 0x71a0: wc = 0x9049; break;
        case 0x71a1: wc = 0x9172; break;
        case 0x71a2: wc = 0x92e5; break;
        case 0x71a3: wc = 0x9424; break;
        case 0x71a4: wc = 0x95ae; break;
        case 0x71a5: wc = 0x9631; break;
        case 0x71a6: wc = 0x9637; break;
        case 0x71a7: wc = 0x9758; break;
        case 0x71a8: wc = 0x975a; break;
        case 0x71a9: wc = 0x976a; break;
        case 0x71aa: wc = 0x9793; break;
        case 0x71ab: wc = 0x9832; break;
        case 0x71ac: wc = 0x9833; break;
        case 0x71ad: wc = 0x9841; break;
        case 0x71ae: wc = 0x98e3; break;
        case 0x71af: wc = 0x4c33; break;
        case 0x71b0: wc = 0x9f2e; break;
        case 0x71b1: wc = 0x9f31; break;
        case 0x71b2: wc = 0x4fe4; break;
        case 0x71b3: wc = 0x504d; break;
        case 0x71b4: wc = 0x50ba; break;
        case 0x71b5: wc = 0x5115; break;
        case 0x71b6: wc = 0x5397; break;
        case 0x71b7: wc = 0x55c1; break;
        case 0x71b8: wc = 0x568c; break;
        case 0x71b9: wc = 0x59fc; break;
        case 0x71ba: wc = 0x5a23; break;
        case 0x71bb: wc = 0x5a9e; break;
        case 0x71bc: wc = 0x5fb2; break;
        case 0x71bd: wc = 0x60ff; break;
        case 0x71be: wc = 0x61e0; break;
        case 0x71bf: wc = 0x64e0; break;
        case 0x71c0: wc = 0x682c; break;
        case 0x71c1: wc = 0x6ac5; break;
        case 0x71c2: wc = 0x6cf2; break;
        case 0x71c3: wc = 0x6ddb; break;
        case 0x71c4: wc = 0x6e27; break;
        case 0x71c5: wc = 0x6edb; break;
        case 0x71c6: wc = 0x6f08; break;
        case 0x71c7: wc = 0x710d; break;
        case 0x71c8: wc = 0x72fe; break;
        case 0x71c9: wc = 0x7318; break;
        case 0x71ca: wc = 0x73f6; break;
        case 0x71cb: wc = 0x7445; break;
        case 0x71cc: wc = 0x74be; break;
        case 0x71cd: wc = 0x7660; break;
        case 0x71ce: wc = 0x7747; break;
        case 0x71cf: wc = 0x777c; break;
        case 0x71d0: wc = 0x78b2; break;
        case 0x71d1: wc = 0x78fe; break;
        case 0x71d2: wc = 0x7994; break;
        case 0x71d3: wc = 0x7a0a; break;
        case 0x71d4: wc = 0x7a44; break;
        case 0x71d5: wc = 0x7a67; break;
        case 0x71d6: wc = 0x7d88; break;
        case 0x71d7: wc = 0x7df9; break;
        case 0x71d8: wc = 0x8269; break;
        case 0x71d9: wc = 0x82d0; break;
        case 0x71da: wc = 0x84a2; break;
        case 0x71db: wc = 0x855b; break;
        case 0x71dc: wc = 0x8640; break;
        case 0x71dd: wc = 0x876d; break;
        case 0x71de: wc = 0x8810; break;
        case 0x71df: wc = 0x8883; break;
        case 0x71e0: wc = 0x88da; break;
        case 0x71e1: wc = 0x8906; break;
        case 0x71e2: wc = 0x8b15; break;
        case 0x71e3: wc = 0x8da7; break;
        case 0x71e4: wc = 0x8e36; break;
        case 0x71e5: wc = 0x8e4f; break;
        case 0x71e6: wc = 0x8e8b; break;
        case 0x71e7: wc = 0x92bb; break;
        case 0x71e8: wc = 0x9357; break;
        case 0x71e9: wc = 0x9447; break;
        case 0x71ea: wc = 0x9684; break;
        case 0x71eb: wc = 0x96ae; break;
        case 0x71ec: wc = 0x97ae; break;
        case 0x71ed: wc = 0x9a20; break;
        case 0x71ee: wc = 0x9b86; break;
        case 0x71ef: wc = 0x4c71; break;
        case 0x71f0: wc = 0x9bf7; break;
        case 0x71f1: wc = 0x9c36; break;
        case 0x71f2: wc = 0x9c6d; break;
        case 0x71f3: wc = 0x9d5c; break;
        case 0x71f4: wc = 0x9d99; break;
        case 0x71f5: wc = 0x9d97; break;
        case 0x71f6: wc = 0x9ea1; break;
        case 0x71f7: wc = 0x2a5c6; break;
        case 0x71f8: wc = 0x9f4c; break;
        case 0x71f9: wc = 0x9f4f; break;
        case 0x71fa: wc = 0x2010f; break;
        case 0x71fb: wc = 0x4f04; break;
        case 0x71fc: wc = 0x4f7b; break;
        case 0x71fd: wc = 0x50ae; break;
        case 0x71fe: wc = 0x5201; break;
        case 0x71ff: wc = 0x209ea; break;
        case 0x7200: wc = 0x539d; break;
        case 0x7201: wc = 0x53fc; break;
        case 0x7202: wc = 0x5515; break;
        case 0x7203: wc = 0x5541; break;
        case 0x7204: wc = 0x5608; break;
        case 0x7205: wc = 0x566a; break;
        case 0x7206: wc = 0x5797; break;
        case 0x7207: wc = 0x5b25; break;
        case 0x7208: wc = 0x5ba8; break;
        case 0x7209: wc = 0x5d86; break;
        case 0x720a: wc = 0x5e67; break;
        case 0x720b: wc = 0x5ea3; break;
        case 0x720c: wc = 0x5f82; break;
        case 0x720d: wc = 0x604c; break;
        case 0x720e: wc = 0x6165; break;
        case 0x720f: wc = 0x61c6; break;
        case 0x7210: wc = 0x627e; break;
        case 0x7211: wc = 0x6293; break;
        case 0x7212: wc = 0x3a04; break;
        case 0x7213: wc = 0x65d0; break;
        case 0x7214: wc = 0x66cc; break;
        case 0x7215: wc = 0x6713; break;
        case 0x7216: wc = 0x6b82; break;
        case 0x7217: wc = 0x6ba6; break;
        case 0x7218: wc = 0x6ca0; break;
        case 0x7219: wc = 0x6d2e; break;
        case 0x721a: wc = 0x6da4; break;
        case 0x721b: wc = 0x6fa1; break;
        case 0x721c: wc = 0x7076; break;
        case 0x721d: wc = 0x3dee; break;
        case 0x721e: wc = 0x722b; break;
        case 0x721f: wc = 0x72e3; break;
        case 0x7220: wc = 0x73c7; break;
        case 0x7221: wc = 0x7431; break;
        case 0x7222: wc = 0x7475; break;
        case 0x7223: wc = 0x7682; break;
        case 0x7224: wc = 0x7681; break;
        case 0x7225: wc = 0x76c4; break;
        case 0x7226: wc = 0x7797; break;
        case 0x7227: wc = 0x7889; break;
        case 0x7228: wc = 0x7967; break;
        case 0x7229: wc = 0x4186; break;
        case 0x722a: wc = 0x7ab1; break;
        case 0x722b: wc = 0x7ab5; break;
        case 0x722c: wc = 0x7ac8; break;
        case 0x722d: wc = 0x7b0a; break;
        case 0x722e: wc = 0x7b8c; break;
        case 0x722f: wc = 0x7cd9; break;
        case 0x7230: wc = 0x7cf6; break;
        case 0x7231: wc = 0x7d69; break;
        case 0x7232: wc = 0x7d5b; break;
        case 0x7233: wc = 0x7f69; break;
        case 0x7234: wc = 0x7f7a; break;
        case 0x7235: wc = 0x808f; break;
        case 0x7236: wc = 0x80d9; break;
        case 0x7237: wc = 0x8101; break;
        case 0x7238: wc = 0x81ca; break;
        case 0x7239: wc = 0x8241; break;
        case 0x723a: wc = 0x825a; break;
        case 0x723b: wc = 0x839c; break;
        case 0x723c: wc = 0x8387; break;
        case 0x723d: wc = 0x8444; break;
        case 0x723e: wc = 0x84a9; break;
        case 0x723f: wc = 0x8526; break;
        case 0x7240: wc = 0x84e7; break;
        case 0x7241: wc = 0x85cb; break;
        case 0x7242: wc = 0x86c1; break;
        case 0x7243: wc = 0x8729; break;
        case 0x7244: wc = 0x87ac; break;
        case 0x7245: wc = 0x8959; break;
        case 0x7246: wc = 0x899c; break;
        case 0x7247: wc = 0x8a82; break;
        case 0x7248: wc = 0x8b2f; break;
        case 0x7249: wc = 0x8b5f; break;
        case 0x724a: wc = 0x8d92; break;
        case 0x724b: wc = 0x8dae; break;
        case 0x724c: wc = 0x8e67; break;
        case 0x724d: wc = 0x9135; break;
        case 0x724e: wc = 0x91c2; break;
        case 0x724f: wc = 0x91d5; break;
        case 0x7250: wc = 0x932d; break;
        case 0x7251: wc = 0x92fd; break;
        case 0x7252: wc = 0x93a5; break;
        case 0x7253: wc = 0x93ea; break;
        case 0x7254: wc = 0x9430; break;
        case 0x7255: wc = 0x963c; break;
        case 0x7256: wc = 0x96ff; break;
        case 0x7257: wc = 0x977b; break;
        case 0x7258: wc = 0x9797; break;
        case 0x7259: wc = 0x99cb; break;
        case 0x725a: wc = 0x99e3; break;
        case 0x725b: wc = 0x9b61; break;
        case 0x725c: wc = 0x9ba1; break;
        case 0x725d: wc = 0x9bc8; break;
        case 0x725e: wc = 0x9bdb; break;
        case 0x725f: wc = 0x9c37; break;
        case 0x7260: wc = 0x9d70; break;
        case 0x7261: wc = 0x2a163; break;
        case 0x7262: wc = 0x9e86; break;
        case 0x7263: wc = 0x9f02; break;
        case 0x7264: wc = 0x54eb; break;
        case 0x7265: wc = 0x762f; break;
        case 0x7266: wc = 0x851f; break;
        case 0x7267: wc = 0x8e3f; break;
        case 0x7268: wc = 0x62f5; break;
        case 0x7269: wc = 0x71c7; break;
        case 0x726a: wc = 0x88b8; break;
        case 0x726b: wc = 0x928c; break;
        case 0x726c: wc = 0x5005; break;
        case 0x726d: wc = 0x637d; break;
        case 0x726e: wc = 0x690a; break;
        case 0x726f: wc = 0x3c9e; break;
        case 0x7270: wc = 0x70aa; break;
        case 0x7271: wc = 0x7ba4; break;
        case 0x7272: wc = 0x8e24; break;
        case 0x7273: wc = 0x4abc; break;
        case 0x7274: wc = 0x4f00; break;
        case 0x7275: wc = 0x50b1; break;
        case 0x7276: wc = 0x5839; break;
        case 0x7277: wc = 0x5990; break;
        case 0x7278: wc = 0x5a43; break;
        case 0x7279: wc = 0x5b6e; break;
        case 0x727a: wc = 0x5c30; break;
        case 0x727b: wc = 0x5d55; break;
        case 0x727c: wc = 0x5d77; break;
        case 0x727d: wc = 0x220d3; break;
        case 0x727e: wc = 0x5f96; break;
        case 0x727f: wc = 0x5fea; break;
        case 0x7280: wc = 0x6152; break;
        case 0x7281: wc = 0x662e; break;
        case 0x7282: wc = 0x66b0; break;
        case 0x7283: wc = 0x67ca; break;
        case 0x7284: wc = 0x6936; break;
        case 0x7285: wc = 0x6a05; break;
        case 0x7286: wc = 0x6f40; break;
        case 0x7287: wc = 0x7082; break;
        case 0x7288: wc = 0x7144; break;
        case 0x7289: wc = 0x71a7; break;
        case 0x728a: wc = 0x7323; break;
        case 0x728b: wc = 0x747d; break;
        case 0x728c: wc = 0x7607; break;
        case 0x728d: wc = 0x7632; break;
        case 0x728e: wc = 0x779b; break;
        case 0x728f: wc = 0x7a2f; break;
        case 0x7290: wc = 0x7c66; break;
        case 0x7291: wc = 0x7cbd; break;
        case 0x7292: wc = 0x7cc9; break;
        case 0x7293: wc = 0x7ddf; break;
        case 0x7294: wc = 0x7df5; break;
        case 0x7295: wc = 0x7fea; break;
        case 0x7296: wc = 0x6721; break;
        case 0x7297: wc = 0x4478; break;
        case 0x7298: wc = 0x8250; break;
        case 0x7299: wc = 0x84d7; break;
        case 0x729a: wc = 0x8520; break;
        case 0x729b: wc = 0x876c; break;
        case 0x729c: wc = 0x87bd; break;
        case 0x729d: wc = 0x474b; break;
        case 0x729e: wc = 0x8c75; break;
        case 0x729f: wc = 0x8ce8; break;
        case 0x72a0: wc = 0x8e64; break;
        case 0x72a1: wc = 0x8e71; break;
        case 0x72a2: wc = 0x931d; break;
        case 0x72a3: wc = 0x9350; break;
        case 0x72a4: wc = 0x9a23; break;
        case 0x72a5: wc = 0x9b03; break;
        case 0x72a6: wc = 0x9b09; break;
        case 0x72a7: wc = 0x9b37; break;
        case 0x72a8: wc = 0x9bfc; break;
        case 0x72a9: wc = 0x9f28; break;
        case 0x72aa: wc = 0x4fb3; break;
        case 0x72ab: wc = 0x5249; break;
        case 0x72ac: wc = 0x590e; break;
        case 0x72ad: wc = 0x5c6e; break;
        case 0x72ae: wc = 0x75e4; break;
        case 0x72af: wc = 0x77ec; break;
        case 0x72b0: wc = 0x811e; break;
        case 0x72b1: wc = 0x839d; break;
        case 0x72b2: wc = 0x84cc; break;
        case 0x72b3: wc = 0x888f; break;
        case 0x72b4: wc = 0x8d96; break;
        case 0x72b5: wc = 0x9073; break;
        case 0x72b6: wc = 0x92bc; break;
        case 0x72b7: wc = 0x9afd; break;
        case 0x72b8: wc = 0x5db5; break;
        case 0x72b9: wc = 0x6a8c; break;
        case 0x72ba: wc = 0x4e36; break;
        case 0x72bb: wc = 0x4e22; break;
        case 0x72bc: wc = 0x4f37; break;
        case 0x72bd: wc = 0x4f9c; break;
        case 0x72be: wc = 0x5114; break;
        case 0x72bf: wc = 0x5492; break;
        case 0x72c0: wc = 0x54ae; break;
        case 0x72c1: wc = 0x5512; break;
        case 0x72c2: wc = 0x558c; break;
        case 0x72c3: wc = 0x20e08; break;
        case 0x72c4: wc = 0x5663; break;
        case 0x72c5: wc = 0x58f4; break;
        case 0x72c6: wc = 0x59b5; break;
        case 0x72c7: wc = 0x5a64; break;
        case 0x72c8: wc = 0x5c0c; break;
        case 0x72c9: wc = 0x5d40; break;
        case 0x72ca: wc = 0x5e6c; break;
        case 0x72cb: wc = 0x5e6e; break;
        case 0x72cc: wc = 0x5f9f; break;
        case 0x72cd: wc = 0x601e; break;
        case 0x72ce: wc = 0x61e4; break;
        case 0x72cf: wc = 0x62c4; break;
        case 0x72d0: wc = 0x63cd; break;
        case 0x72d1: wc = 0x65a2; break;
        case 0x72d2: wc = 0x666d; break;
        case 0x72d3: wc = 0x6906; break;
        case 0x72d4: wc = 0x6971; break;
        case 0x72d5: wc = 0x6a26; break;
        case 0x72d6: wc = 0x6ae5; break;
        case 0x72d7: wc = 0x6bb6; break;
        case 0x72d8: wc = 0x6d00; break;
        case 0x72d9: wc = 0x6dcd; break;
        case 0x72da: wc = 0x3d24; break;
        case 0x72db: wc = 0x70d0; break;
        case 0x72dc: wc = 0x71fd; break;
        case 0x72dd: wc = 0x72a8; break;
        case 0x72de: wc = 0x73d8; break;
        case 0x72df: wc = 0x75b0; break;
        case 0x72e0: wc = 0x76e9; break;
        case 0x72e1: wc = 0x782b; break;
        case 0x72e2: wc = 0x7843; break;
        case 0x72e3: wc = 0x7969; break;
        case 0x72e4: wc = 0x7c52; break;
        case 0x72e5: wc = 0x7c99; break;
        case 0x72e6: wc = 0x7d38; break;
        case 0x72e7: wc = 0x7d51; break;
        case 0x72e8: wc = 0x7f5c; break;
        case 0x72e9: wc = 0x43b7; break;
        case 0x72ea: wc = 0x8098; break;
        case 0x72eb: wc = 0x80c4; break;
        case 0x72ec: wc = 0x8160; break;
        case 0x72ed: wc = 0x83d7; break;
        case 0x72ee: wc = 0x8464; break;
        case 0x72ef: wc = 0x85b5; break;
        case 0x72f0: wc = 0x86c0; break;
        case 0x72f1: wc = 0x87d5; break;
        case 0x72f2: wc = 0x88be; break;
        case 0x72f3: wc = 0x88ef; break;
        case 0x72f4: wc = 0x8a4b; break;
        case 0x72f5: wc = 0x8b78; break;
        case 0x72f6: wc = 0x8cd9; break;
        case 0x72f7: wc = 0x8d8e; break;
        case 0x72f8: wc = 0x8dd3; break;
        case 0x72f9: wc = 0x8de6; break;
        case 0x72fa: wc = 0x8e95; break;
        case 0x72fb: wc = 0x8ef4; break;
        case 0x72fc: wc = 0x8f08; break;
        case 0x72fd: wc = 0x8f16; break;
        case 0x72fe: wc = 0x9052; break;
        case 0x72ff: wc = 0x90be; break;
        case 0x7300: wc = 0x9252; break;
        case 0x7301: wc = 0x9282; break;
        case 0x7302: wc = 0x92a9; break;
        case 0x7303: wc = 0x9714; break;
        case 0x7304: wc = 0x99b5; break;
        case 0x7305: wc = 0x9a06; break;
        case 0x7306: wc = 0x9ba2; break;
        case 0x7307: wc = 0x9d38; break;
        case 0x7308: wc = 0x9d43; break;
        case 0x7309: wc = 0x9e88; break;
        case 0x730a: wc = 0x9ec8; break;
        case 0x730b: wc = 0x9f04; break;
        case 0x730c: wc = 0x9b3b; break;
        case 0x730d: wc = 0x5046; break;
        case 0x730e: wc = 0x50d4; break;
        case 0x730f: wc = 0x20b0c; break;
        case 0x7310: wc = 0x3559; break;
        case 0x7311: wc = 0x5642; break;
        case 0x7312: wc = 0x57fb; break;
        case 0x7313: wc = 0x58ab; break;
        case 0x7314: wc = 0x58ff; break;
        case 0x7315: wc = 0x5b92; break;
        case 0x7316: wc = 0x5d9f; break;
        case 0x7317: wc = 0x60f7; break;
        case 0x7318: wc = 0x6358; break;
        case 0x7319: wc = 0x6499; break;
        case 0x731a: wc = 0x76b4; break;
        case 0x731b: wc = 0x7743; break;
        case 0x731c: wc = 0x7a15; break;
        case 0x731d: wc = 0x7da7; break;
        case 0x731e: wc = 0x7e5c; break;
        case 0x731f: wc = 0x7f47; break;
        case 0x7320: wc = 0x8470; break;
        case 0x7321: wc = 0x8a30; break;
        case 0x7322: wc = 0x8b50; break;
        case 0x7323: wc = 0x47ad; break;
        case 0x7324: wc = 0x8e06; break;
        case 0x7325: wc = 0x8e33; break;
        case 0x7326: wc = 0x8e72; break;
        case 0x7327: wc = 0x8fff; break;
        case 0x7328: wc = 0x93a8; break;
        case 0x7329: wc = 0x940f; break;
        case 0x732a: wc = 0x9656; break;
        case 0x732b: wc = 0x96bc; break;
        case 0x732c: wc = 0x9915; break;
        case 0x732d: wc = 0x9c52; break;
        case 0x732e: wc = 0x9d54; break;
        case 0x732f: wc = 0x9dbd; break;
        case 0x7330: wc = 0x9df7; break;
        case 0x7331: wc = 0x4e7c; break;
        case 0x7332: wc = 0x5d12; break;
        case 0x7333: wc = 0x7a8b; break;
        case 0x7334: wc = 0x7b1c; break;
        case 0x7335: wc = 0x35df; break;
        case 0x7336: wc = 0x3456; break;
        case 0x7337: wc = 0x38e1; break;
        case 0x7338: wc = 0x3cde; break;
        case 0x7339: wc = 0x3d22; break;
        case 0x733a: wc = 0x72c6; break;
        case 0x733b: wc = 0x822f; break;
        case 0x733c: wc = 0x869b; break;
        case 0x733d: wc = 0x8769; break;
        case 0x733e: wc = 0x8876; break;
        case 0x733f: wc = 0x8fda; break;
        case 0x7340: wc = 0x9680; break;
        case 0x7341: wc = 0x5d31; break;
        case 0x7342: wc = 0x63e4; break;
        case 0x7343: wc = 0x83ad; break;
        case 0x7344: wc = 0x8434; break;
        case 0x7345: wc = 0x874d; break;
        case 0x7346: wc = 0x9c02; break;
        case 0x7347: wc = 0x9bfd; break;
        case 0x7348: wc = 0x559e; break;
        case 0x7349: wc = 0x5832; break;
        case 0x734a: wc = 0x64f3; break;
        case 0x734b: wc = 0x6956; break;
        case 0x734c: wc = 0x7004; break;
        case 0x734d: wc = 0x9a2d; break;
        case 0x734e: wc = 0x600e; break;
        case 0x734f: wc = 0x6a9d; break;
        case 0x7350: wc = 0x6e52; break;
        case 0x7351: wc = 0x6fc8; break;
        case 0x7352: wc = 0x857a; break;
        case 0x7353: wc = 0x9735; break;
        case 0x7354: wc = 0x5d92; break;
        case 0x7355: wc = 0x649c; break;
        case 0x7356: wc = 0x6a67; break;
        case 0x7357: wc = 0x6f67; break;
        case 0x7358: wc = 0x71b7; break;
        case 0x7359: wc = 0x7494; break;
        case 0x735a: wc = 0x77f0; break;
        case 0x735b: wc = 0x78f3; break;
        case 0x735c: wc = 0x7bdc; break;
        case 0x735d: wc = 0x7f7e; break;
        case 0x735e: wc = 0x8a3c; break;
        case 0x735f: wc = 0x8b44; break;
        case 0x7360: wc = 0x912b; break;
        case 0x7361: wc = 0x9a53; break;
        case 0x7362: wc = 0x4d73; break;
        case 0x7363: wc = 0x4fe7; break;
        case 0x7364: wc = 0x51ea; break;
        case 0x7365: wc = 0x538e; break;
        case 0x7366: wc = 0x5431; break;
        case 0x7367: wc = 0x5741; break;
        case 0x7368: wc = 0x577b; break;
        case 0x7369: wc = 0x5880; break;
        case 0x736a: wc = 0x58ac; break;
        case 0x736b: wc = 0x5f7d; break;
        case 0x736c: wc = 0x5f94; break;
        case 0x736d: wc = 0x5fef; break;
        case 0x736e: wc = 0x6049; break;
        case 0x736f: wc = 0x627a; break;
        case 0x7370: wc = 0x6418; break;
        case 0x7371: wc = 0x683a; break;
        case 0x7372: wc = 0x6925; break;
        case 0x7373: wc = 0x69b0; break;
        case 0x7374: wc = 0x6c66; break;
        case 0x7375: wc = 0x6c65; break;
        case 0x7376: wc = 0x6cdc; break;
        case 0x7377: wc = 0x6d14; break;
        case 0x7378: wc = 0x6dfd; break;
        case 0x7379: wc = 0x6f6a; break;
        case 0x737a: wc = 0x75bb; break;
        case 0x737b: wc = 0x75e3; break;
        case 0x737c: wc = 0x79d6; break;
        case 0x737d: wc = 0x79ea; break;
        case 0x737e: wc = 0x7afe; break;
        case 0x737f: wc = 0x7b88; break;
        case 0x7380: wc = 0x7bea; break;
        case 0x7381: wc = 0x80dd; break;
        case 0x7382: wc = 0x80d1; break;
        case 0x7383: wc = 0x8210; break;
        case 0x7384: wc = 0x44cb; break;
        case 0x7385: wc = 0x86b3; break;
        case 0x7386: wc = 0x887c; break;
        case 0x7387: wc = 0x899f; break;
        case 0x7388: wc = 0x89d7; break;
        case 0x7389: wc = 0x8b18; break;
        case 0x738a: wc = 0x8cbe; break;
        case 0x738b: wc = 0x8e1f; break;
        case 0x738c: wc = 0x8e93; break;
        case 0x738d: wc = 0x8ef9; break;
        case 0x738e: wc = 0x8f0a; break;
        case 0x738f: wc = 0x916f; break;
        case 0x7390: wc = 0x92d5; break;
        case 0x7391: wc = 0x962f; break;
        case 0x7392: wc = 0x9a47; break;
        case 0x7393: wc = 0x9ba8; break;
        case 0x7394: wc = 0x9cf7; break;
        case 0x7395: wc = 0x9dd9; break;
        case 0x7396: wc = 0x5b02; break;
        case 0x7397: wc = 0x3ba8; break;
        case 0x7398: wc = 0x6a34; break;
        case 0x7399: wc = 0x6ead; break;
        case 0x739a: wc = 0x7286; break;
        case 0x739b: wc = 0x799d; break;
        case 0x739c: wc = 0x4408; break;
        case 0x739d: wc = 0x81b1; break;
        case 0x739e: wc = 0x8635; break;
        case 0x739f: wc = 0x87d9; break;
        case 0x73a0: wc = 0x4fb2; break;
        case 0x73a1: wc = 0x5118; break;
        case 0x73a2: wc = 0x568d; break;
        case 0x73a3: wc = 0x583b; break;
        case 0x73a4: wc = 0x5ac0; break;
        case 0x73a5: wc = 0x5b27; break;
        case 0x73a6: wc = 0x62ae; break;
        case 0x73a7: wc = 0x630b; break;
        case 0x73a8: wc = 0x6783; break;
        case 0x73a9: wc = 0x6997; break;
        case 0x73aa: wc = 0x6a04; break;
        case 0x73ab: wc = 0x6a3c; break;
        case 0x73ac: wc = 0x6e8d; break;
        case 0x73ad: wc = 0x6fdc; break;
        case 0x73ae: wc = 0x7349; break;
        case 0x73af: wc = 0x73ce; break;
        case 0x73b0: wc = 0x73d2; break;
        case 0x73b1: wc = 0x7467; break;
        case 0x73b2: wc = 0x7471; break;
        case 0x73b3: wc = 0x75a2; break;
        case 0x73b4: wc = 0x7715; break;
        case 0x73b5: wc = 0x7739; break;
        case 0x73b6: wc = 0x403c; break;
        case 0x73b7: wc = 0x2525c; break;
        case 0x73b8: wc = 0x40a6; break;
        case 0x73b9: wc = 0x799b; break;
        case 0x73ba: wc = 0x7a39; break;
        case 0x73bb: wc = 0x7c48; break;
        case 0x73bc: wc = 0x7d16; break;
        case 0x73bd: wc = 0x7d3e; break;
        case 0x73be: wc = 0x7d7c; break;
        case 0x73bf: wc = 0x7e25; break;
        case 0x73c0: wc = 0x80d7; break;
        case 0x73c1: wc = 0x441c; break;
        case 0x73c2: wc = 0x81f8; break;
        case 0x73c3: wc = 0x84c1; break;
        case 0x73c4: wc = 0x85bd; break;
        case 0x73c5: wc = 0x8704; break;
        case 0x73c6: wc = 0x8793; break;
        case 0x73c7: wc = 0x87b4; break;
        case 0x73c8: wc = 0x88d6; break;
        case 0x73c9: wc = 0x8aab; break;
        case 0x73ca: wc = 0x8b13; break;
        case 0x73cb: wc = 0x8d81; break;
        case 0x73cc: wc = 0x8d82; break;
        case 0x73cd: wc = 0x8f43; break;
        case 0x73ce: wc = 0x8fb4; break;
        case 0x73cf: wc = 0x9241; break;
        case 0x73d0: wc = 0x6576; break;
        case 0x73d1: wc = 0x99d7; break;
        case 0x73d2: wc = 0x9b12; break;
        case 0x73d3: wc = 0x9ef0; break;
        case 0x73d4: wc = 0x5394; break;
        case 0x73d5: wc = 0x54a5; break;
        case 0x73d6: wc = 0x35cc; break;
        case 0x73d7: wc = 0x57a4; break;
        case 0x73d8: wc = 0x59b7; break;
        case 0x73d9: wc = 0x5ccc; break;
        case 0x73da: wc = 0x5ea2; break;
        case 0x73db: wc = 0x62b6; break;
        case 0x73dc: wc = 0x6303; break;
        case 0x73dd: wc = 0x6633; break;
        case 0x73de: wc = 0x664a; break;
        case 0x73df: wc = 0x67e3; break;
        case 0x73e0: wc = 0x69c9; break;
        case 0x73e1: wc = 0x6acd; break;
        case 0x73e2: wc = 0x6d37; break;
        case 0x73e3: wc = 0x74de; break;
        case 0x73e4: wc = 0x7723; break;
        case 0x73e5: wc = 0x7929; break;
        case 0x73e6: wc = 0x7951; break;
        case 0x73e7: wc = 0x79f7; break;
        case 0x73e8: wc = 0x7d29; break;
        case 0x73e9: wc = 0x7d70; break;
        case 0x73ea: wc = 0x7fd0; break;
        case 0x73eb: wc = 0x800b; break;
        case 0x73ec: wc = 0x80c5; break;
        case 0x73ed: wc = 0x815f; break;
        case 0x73ee: wc = 0x81f7; break;
        case 0x73ef: wc = 0x82f5; break;
        case 0x73f0: wc = 0x84ba; break;
        case 0x73f1: wc = 0x878f; break;
        case 0x73f2: wc = 0x87b2; break;
        case 0x73f3: wc = 0x889f; break;
        case 0x73f4: wc = 0x8a44; break;
        case 0x73f5: wc = 0x8efc; break;
        case 0x73f6: wc = 0x90c5; break;
        case 0x73f7: wc = 0x928d; break;
        case 0x73f8: wc = 0x9455; break;
        case 0x73f9: wc = 0x681a; break;
        case 0x73fa: wc = 0x9156; break;
        case 0x73fb: wc = 0x9159; break;
        case 0x73fc: wc = 0x9d06; break;
        case 0x73fd: wc = 0x4ebc; break;
        case 0x73fe: wc = 0x506e; break;
        case 0x73ff: wc = 0x54a0; break;
        case 0x7400: wc = 0x5daf; break;
        case 0x7401: wc = 0x6179; break;
        case 0x7402: wc = 0x6222; break;
        case 0x7403: wc = 0x74e1; break;
        case 0x7404: wc = 0x7e36; break;
        case 0x7405: wc = 0x84fb; break;
        case 0x7406: wc = 0x8aff; break;
        case 0x7407: wc = 0x8b3a; break;
        case 0x7408: wc = 0x55ed; break;
        case 0x7409: wc = 0x60e9; break;
        case 0x740a: wc = 0x6f82; break;
        case 0x740b: wc = 0x7665; break;
        case 0x740c: wc = 0x77aa; break;
        case 0x740d: wc = 0x4f4c; break;
        case 0x740e: wc = 0x4f7d; break;
        case 0x740f: wc = 0x5056; break;
        case 0x7410: wc = 0x5068; break;
        case 0x7411: wc = 0x538f; break;
        case 0x7412: wc = 0x5513; break;
        case 0x7413: wc = 0x5536; break;
        case 0x7414: wc = 0x55cf; break;
        case 0x7415: wc = 0x55fb; break;
        case 0x7416: wc = 0x5953; break;
        case 0x7417: wc = 0x5972; break;
        case 0x7418: wc = 0x59f9; break;
        case 0x7419: wc = 0x5c94; break;
        case 0x741a: wc = 0x5fa3; break;
        case 0x741b: wc = 0x6260; break;
        case 0x741c: wc = 0x626f; break;
        case 0x741d: wc = 0x6313; break;
        case 0x741e: wc = 0x6413; break;
        case 0x741f: wc = 0x643d; break;
        case 0x7420: wc = 0x64a6; break;
        case 0x7421: wc = 0x6748; break;
        case 0x7422: wc = 0x6828; break;
        case 0x7423: wc = 0x6c4a; break;
        case 0x7424: wc = 0x6cda; break;
        case 0x7425: wc = 0x7473; break;
        case 0x7426: wc = 0x7868; break;
        case 0x7427: wc = 0x7d01; break;
        case 0x7428: wc = 0x7d58; break;
        case 0x7429: wc = 0x8256; break;
        case 0x742a: wc = 0x8397; break;
        case 0x742b: wc = 0x84ab; break;
        case 0x742c: wc = 0x8516; break;
        case 0x742d: wc = 0x8869; break;
        case 0x742e: wc = 0x8a0d; break;
        case 0x742f: wc = 0x8b47; break;
        case 0x7430: wc = 0x8dd0; break;
        case 0x7431: wc = 0x48dc; break;
        case 0x7432: wc = 0x919d; break;
        case 0x7433: wc = 0x28941; break;
        case 0x7434: wc = 0x9388; break;
        case 0x7435: wc = 0x9e7a; break;
        case 0x7436: wc = 0x9f47; break;
        case 0x7437: wc = 0x9f79; break;
        case 0x7438: wc = 0x5252; break;
        case 0x7439: wc = 0x5a16; break;
        case 0x743a: wc = 0x5a15; break;
        case 0x743b: wc = 0x6233; break;
        case 0x743c: wc = 0x64c9; break;
        case 0x743d: wc = 0x65ae; break;
        case 0x743e: wc = 0x65b2; break;
        case 0x743f: wc = 0x6d5e; break;
        case 0x7440: wc = 0x247ed; break;
        case 0x7441: wc = 0x73ff; break;
        case 0x7442: wc = 0x7a5b; break;
        case 0x7443: wc = 0x7a71; break;
        case 0x7444: wc = 0x7b2e; break;
        case 0x7445: wc = 0x7be7; break;
        case 0x7446: wc = 0x7c0e; break;
        case 0x7447: wc = 0x7e12; break;
        case 0x7448: wc = 0x8ad1; break;
        case 0x7449: wc = 0x8fb6; break;
        case 0x744a: wc = 0x8fb5; break;
        case 0x744b: wc = 0x92dc; break;
        case 0x744c: wc = 0x9461; break;
        case 0x744d: wc = 0x4e33; break;
        case 0x744e: wc = 0x510f; break;
        case 0x744f: wc = 0x5127; break;
        case 0x7450: wc = 0x5139; break;
        case 0x7451: wc = 0x5297; break;
        case 0x7452: wc = 0x56cb; break;
        case 0x7453: wc = 0x5dd1; break;
        case 0x7454: wc = 0x651b; break;
        case 0x7455: wc = 0x6522; break;
        case 0x7456: wc = 0x6b11; break;
        case 0x7457: wc = 0x6e4c; break;
        case 0x7458: wc = 0x7052; break;
        case 0x7459: wc = 0x7218; break;
        case 0x745a: wc = 0x7228; break;
        case 0x745b: wc = 0x7bf9; break;
        case 0x745c: wc = 0x7be1; break;
        case 0x745d: wc = 0x7c6b; break;
        case 0x745e: wc = 0x7fbc; break;
        case 0x745f: wc = 0x8978; break;
        case 0x7460: wc = 0x8db2; break;
        case 0x7461: wc = 0x8ea5; break;
        case 0x7462: wc = 0x8ea6; break;
        case 0x7463: wc = 0x9147; break;
        case 0x7464: wc = 0x9961; break;
        case 0x7465: wc = 0x4c16; break;
        case 0x7466: wc = 0x4c57; break;
        case 0x7467: wc = 0x507a; break;
        case 0x7468: wc = 0x54b1; break;
        case 0x7469: wc = 0x54f3; break;
        case 0x746a: wc = 0x5693; break;
        case 0x746b: wc = 0x5dc0; break;
        case 0x746c: wc = 0x624e; break;
        case 0x746d: wc = 0x62f6; break;
        case 0x746e: wc = 0x6503; break;
        case 0x746f: wc = 0x6aab; break;
        case 0x7470: wc = 0x3cd0; break;
        case 0x7471: wc = 0x7924; break;
        case 0x7472: wc = 0x7938; break;
        case 0x7473: wc = 0x7ccc; break;
        case 0x7474: wc = 0x86bb; break;
        case 0x7475: wc = 0x883f; break;
        case 0x7476: wc = 0x9358; break;
        case 0x7477: wc = 0x9454; break;
        case 0x7478: wc = 0x50e3; break;
        case 0x7479: wc = 0x5133; break;
        case 0x747a: wc = 0x5296; break;
        case 0x747b: wc = 0x53c5; break;
        case 0x747c: wc = 0x5646; break;
        case 0x747d: wc = 0x56b5; break;
        case 0x747e: wc = 0x588b; break;
        case 0x747f: wc = 0x5b20; break;
        case 0x7480: wc = 0x5d84; break;
        case 0x7481: wc = 0x5d7e; break;
        case 0x7482: wc = 0x5dc9; break;
        case 0x7483: wc = 0x615a; break;
        case 0x7484: wc = 0x61af; break;
        case 0x7485: wc = 0x6472; break;
        case 0x7486: wc = 0x6519; break;
        case 0x7487: wc = 0x65f5; break;
        case 0x7488: wc = 0x6701; break;
        case 0x7489: wc = 0x69e7; break;
        case 0x748a: wc = 0x6b03; break;
        case 0x748b: wc = 0x6bda; break;
        case 0x748c: wc = 0x6bf6; break;
        case 0x748d: wc = 0x703a; break;
        case 0x748e: wc = 0x7351; break;
        case 0x748f: wc = 0x762e; break;
        case 0x7490: wc = 0x406a; break;
        case 0x7491: wc = 0x78e3; break;
        case 0x7492: wc = 0x78db; break;
        case 0x7493: wc = 0x826c; break;
        case 0x7494: wc = 0x8a40; break;
        case 0x7495: wc = 0x8b56; break;
        case 0x7496: wc = 0x8dbb; break;
        case 0x7497: wc = 0x9141; break;
        case 0x7498: wc = 0x93e8; break;
        case 0x7499: wc = 0x9471; break;
        case 0x749a: wc = 0x995e; break;
        case 0x749b: wc = 0x9a42; break;
        case 0x749c: wc = 0x9b59; break;
        case 0x749d: wc = 0x9ef2; break;
        case 0x749e: wc = 0x5000; break;
        case 0x749f: wc = 0x5096; break;
        case 0x74a0: wc = 0x51d4; break;
        case 0x74a1: wc = 0x5231; break;
        case 0x74a2: wc = 0x55c6; break;
        case 0x74a3: wc = 0x56f1; break;
        case 0x74a4: wc = 0x5d62; break;
        case 0x74a5: wc = 0x60b5; break;
        case 0x74a6: wc = 0x60dd; break;
        case 0x74a7: wc = 0x6183; break;
        case 0x74a8: wc = 0x22835; break;
        case 0x74a9: wc = 0x6227; break;
        case 0x74aa: wc = 0x6436; break;
        case 0x74ab: wc = 0x6450; break;
        case 0x74ac: wc = 0x6464; break;
        case 0x74ad: wc = 0x6919; break;
        case 0x74ae: wc = 0x6c05; break;
        case 0x74af: wc = 0x6dd0; break;
        case 0x74b0: wc = 0x6dcc; break;
        case 0x74b1: wc = 0x6ff8; break;
        case 0x74b2: wc = 0x713b; break;
        case 0x74b3: wc = 0x7197; break;
        case 0x74b4: wc = 0x7255; break;
        case 0x74b5: wc = 0x734a; break;
        case 0x74b6: wc = 0x7429; break;
        case 0x74b7: wc = 0x7452; break;
        case 0x74b8: wc = 0x7472; break;
        case 0x74b9: wc = 0x78e2; break;
        case 0x74ba: wc = 0x7a97; break;
        case 0x74bb: wc = 0x88ee; break;
        case 0x74bc: wc = 0x8b12; break;
        case 0x74bd: wc = 0x8e4c; break;
        case 0x74be: wc = 0x92f9; break;
        case 0x74bf: wc = 0x9329; break;
        case 0x74c0: wc = 0x9306; break;
        case 0x74c1: wc = 0x95b6; break;
        case 0x74c2: wc = 0x97d4; break;
        case 0x74c3: wc = 0x9b2f; break;
        case 0x74c4: wc = 0x9be7; break;
        case 0x74c5: wc = 0x9dac; break;
        case 0x74c6: wc = 0x9f1a; break;
        case 0x74c7: wc = 0x56c6; break;
        case 0x74c8: wc = 0x5a47; break;
        case 0x74c9: wc = 0x3a5f; break;
        case 0x74ca: wc = 0x68cc; break;
        case 0x74cb: wc = 0x7625; break;
        case 0x74cc: wc = 0x7635; break;
        case 0x74cd: wc = 0x776c; break;
        case 0x74ce: wc = 0x7e29; break;
        case 0x74cf: wc = 0x831d; break;
        case 0x74d0: wc = 0x8806; break;
        case 0x74d1: wc = 0x8e29; break;
        case 0x74d2: wc = 0x976b; break;
        case 0x74d3: wc = 0x5616; break;
        case 0x74d4: wc = 0x5e58; break;
        case 0x74d5: wc = 0x61a1; break;
        case 0x74d6: wc = 0x62fa; break;
        case 0x74d7: wc = 0x78d4; break;
        case 0x74d8: wc = 0x7c00; break;
        case 0x74d9: wc = 0x7ca3; break;
        case 0x74da: wc = 0x8234; break;
        case 0x74db: wc = 0x8326; break;
        case 0x74dc: wc = 0x84db; break;
        case 0x74dd: wc = 0x8674; break;
        case 0x74de: wc = 0x86b1; break;
        case 0x74df: wc = 0x8b2e; break;
        case 0x74e0: wc = 0x8fee; break;
        case 0x74e1: wc = 0x93fc; break;
        case 0x74e2: wc = 0x9819; break;
        case 0x74e3: wc = 0x29460; break;
        case 0x74e4: wc = 0x4da6; break;
        case 0x74e5: wc = 0x521e; break;
        case 0x74e6: wc = 0x6dd2; break;
        case 0x74e7: wc = 0x7dc0; break;
        case 0x74e8: wc = 0x801d; break;
        case 0x74e9: wc = 0x840b; break;
        case 0x74ea: wc = 0x8904; break;
        case 0x74eb: wc = 0x89b0; break;
        case 0x74ec: wc = 0x90ea; break;
        case 0x74ed: wc = 0x970b; break;
        case 0x74ee: wc = 0x9d88; break;
        case 0x74ef: wc = 0x50b6; break;
        case 0x74f0: wc = 0x544e; break;
        case 0x74f1: wc = 0x5767; break;
        case 0x74f2: wc = 0x5849; break;
        case 0x74f3: wc = 0x5d74; break;
        case 0x74f4: wc = 0x5f73; break;
        case 0x74f5: wc = 0x6090; break;
        case 0x74f6: wc = 0x60d5; break;
        case 0x74f7: wc = 0x617c; break;
        case 0x74f8: wc = 0x646d; break;
        case 0x74f9: wc = 0x7869; break;
        case 0x74fa: wc = 0x78e9; break;
        case 0x74fb: wc = 0x7c8e; break;
        case 0x74fc: wc = 0x818c; break;
        case 0x74fd: wc = 0x8687; break;
        case 0x74fe: wc = 0x8734; break;
        case 0x74ff: wc = 0x274f8; break;
        case 0x7500: wc = 0x8901; break;
        case 0x7501: wc = 0x8dd6; break;
        case 0x7502: wc = 0x8e22; break;
        case 0x7503: wc = 0x8e50; break;
        case 0x7504: wc = 0x8e62; break;
        case 0x7505: wc = 0x8e91; break;
        case 0x7506: wc = 0x93da; break;
        case 0x7507: wc = 0x96b2; break;
        case 0x7508: wc = 0x9dba; break;
        case 0x7509: wc = 0x9f1c; break;
        case 0x750a: wc = 0x9f63; break;
        case 0x750b: wc = 0x5029; break;
        case 0x750c: wc = 0x4ff4; break;
        case 0x750d: wc = 0x5103; break;
        case 0x750e: wc = 0x520b; break;
        case 0x750f: wc = 0x56c5; break;
        case 0x7510: wc = 0x570c; break;
        case 0x7511: wc = 0x593c; break;
        case 0x7512: wc = 0x5977; break;
        case 0x7513: wc = 0x5ddb; break;
        case 0x7514: wc = 0x5e5d; break;
        case 0x7515: wc = 0x5fcf; break;
        case 0x7516: wc = 0x6266; break;
        case 0x7517: wc = 0x62ea; break;
        case 0x7518: wc = 0x682b; break;
        case 0x7519: wc = 0x68b4; break;
        case 0x751a: wc = 0x68c8; break;
        case 0x751b: wc = 0x6acf; break;
        case 0x751c: wc = 0x6b42; break;
        case 0x751d: wc = 0x6c1a; break;
        case 0x751e: wc = 0x6c4c; break;
        case 0x751f: wc = 0x6d0a; break;
        case 0x7520: wc = 0x6ffa; break;
        case 0x7521: wc = 0x7033; break;
        case 0x7522: wc = 0x705b; break;
        case 0x7523: wc = 0x7065; break;
        case 0x7524: wc = 0x71c0; break;
        case 0x7525: wc = 0x726e; break;
        case 0x7526: wc = 0x732d; break;
        case 0x7527: wc = 0x73d4; break;
        case 0x7528: wc = 0x744f; break;
        case 0x7529: wc = 0x74e9; break;
        case 0x752a: wc = 0x7946; break;
        case 0x752b: wc = 0x7acf; break;
        case 0x752c: wc = 0x7bdf; break;
        case 0x752d: wc = 0x7c81; break;
        case 0x752e: wc = 0x7daa; break;
        case 0x752f: wc = 0x7e5f; break;
        case 0x7530: wc = 0x8168; break;
        case 0x7531: wc = 0x81f6; break;
        case 0x7532: wc = 0x828a; break;
        case 0x7533: wc = 0x829c; break;
        case 0x7534: wc = 0x8350; break;
        case 0x7535: wc = 0x831c; break;
        case 0x7536: wc = 0x8348; break;
        case 0x7537: wc = 0x84a8; break;
        case 0x7538: wc = 0x8546; break;
        case 0x7539: wc = 0x8695; break;
        case 0x753a: wc = 0x8b42; break;
        case 0x753b: wc = 0x8f24; break;
        case 0x753c: wc = 0x8f32; break;
        case 0x753d: wc = 0x8fbf; break;
        case 0x753e: wc = 0x9044; break;
        case 0x753f: wc = 0x91fa; break;
        case 0x7540: wc = 0x95ce; break;
        case 0x7541: wc = 0x97c0; break;
        case 0x7542: wc = 0x97c9; break;
        case 0x7543: wc = 0x29390; break;
        case 0x7544: wc = 0x9c01; break;
        case 0x7545: wc = 0x525f; break;
        case 0x7546: wc = 0x555c; break;
        case 0x7547: wc = 0x56bd; break;
        case 0x7548: wc = 0xfa3c; break;
        case 0x7549: wc = 0x608a; break;
        case 0x754a: wc = 0x60d9; break;
        case 0x754b: wc = 0x6387; break;
        case 0x754c: wc = 0x6560; break;
        case 0x754d: wc = 0x3bd9; break;
        case 0x754e: wc = 0x6b60; break;
        case 0x754f: wc = 0x710e; break;
        case 0x7550: wc = 0x7577; break;
        case 0x7551: wc = 0x77ae; break;
        case 0x7552: wc = 0x7813; break;
        case 0x7553: wc = 0x7f6c; break;
        case 0x7554: wc = 0x814f; break;
        case 0x7555: wc = 0x86c8; break;
        case 0x7556: wc = 0x8707; break;
        case 0x7557: wc = 0x88f0; break;
        case 0x7558: wc = 0x918a; break;
        case 0x7559: wc = 0x919b; break;
        case 0x755a: wc = 0x9295; break;
        case 0x755b: wc = 0x92e8; break;
        case 0x755c: wc = 0x9323; break;
        case 0x755d: wc = 0x992e; break;
        case 0x755e: wc = 0x4b95; break;
        case 0x755f: wc = 0x9a56; break;
        case 0x7560: wc = 0x5661; break;
        case 0x7561: wc = 0x5a56; break;
        case 0x7562: wc = 0x5e68; break;
        case 0x7563: wc = 0x3868; break;
        case 0x7564: wc = 0x5fdd; break;
        case 0x7565: wc = 0x60bf; break;
        case 0x7566: wc = 0x60c9; break;
        case 0x7567: wc = 0x63ad; break;
        case 0x7568: wc = 0x6541; break;
        case 0x7569: wc = 0x6a90; break;
        case 0x756a: wc = 0x6afc; break;
        case 0x756b: wc = 0x6e49; break;
        case 0x756c: wc = 0x7038; break;
        case 0x756d: wc = 0x704a; break;
        case 0x756e: wc = 0x71c2; break;
        case 0x756f: wc = 0x7c37; break;
        case 0x7570: wc = 0x8214; break;
        case 0x7571: wc = 0x83fe; break;
        case 0x7572: wc = 0x8643; break;
        case 0x7573: wc = 0x88e7; break;
        case 0x7574: wc = 0x895c; break;
        case 0x7575: wc = 0x895d; break;
        case 0x7576: wc = 0x8998; break;
        case 0x7577: wc = 0x8b87; break;
        case 0x7578: wc = 0x915f; break;
        case 0x7579: wc = 0x92d3; break;
        case 0x757a: wc = 0x9369; break;
        case 0x757b: wc = 0x9431; break;
        case 0x757c: wc = 0x946f; break;
        case 0x757d: wc = 0x97c2; break;
        case 0x757e: wc = 0x9902; break;
        case 0x757f: wc = 0x5022; break;
        case 0x7580: wc = 0x503f; break;
        case 0x7581: wc = 0x546b; break;
        case 0x7582: wc = 0x558b; break;
        case 0x7583: wc = 0x5a55; break;
        case 0x7584: wc = 0x5d28; break;
        case 0x7585: wc = 0x6017; break;
        case 0x7586: wc = 0x60f5; break;
        case 0x7587: wc = 0x236fb; break;
        case 0x7588: wc = 0x6c0e; break;
        case 0x7589: wc = 0x7dc1; break;
        case 0x758a: wc = 0x8034; break;
        case 0x758b: wc = 0x8e25; break;
        case 0x758c: wc = 0x8e15; break;
        case 0x758d: wc = 0x8fe0; break;
        case 0x758e: wc = 0x9246; break;
        case 0x758f: wc = 0x4a5e; break;
        case 0x7590: wc = 0x9bbf; break;
        case 0x7591: wc = 0x9bdc; break;
        case 0x7592: wc = 0x4d34; break;
        case 0x7593: wc = 0x51ca; break;
        case 0x7594: wc = 0x570a; break;
        case 0x7595: wc = 0x6385; break;
        case 0x7596: wc = 0x6c30; break;
        case 0x7597: wc = 0x7883; break;
        case 0x7598: wc = 0x873b; break;
        case 0x7599: wc = 0x477c; break;
        case 0x759a: wc = 0x90ec; break;
        case 0x759b: wc = 0x9d84; break;
        case 0x759c: wc = 0x4f53; break;
        case 0x759d: wc = 0x5694; break;
        case 0x759e: wc = 0x5886; break;
        case 0x759f: wc = 0x5c49; break;
        case 0x75a0: wc = 0x5c5c; break;
        case 0x75a1: wc = 0x5d39; break;
        case 0x75a2: wc = 0x5d7d; break;
        case 0x75a3: wc = 0x5f58; break;
        case 0x75a4: wc = 0x61d8; break;
        case 0x75a5: wc = 0x632e; break;
        case 0x75a6: wc = 0x63a3; break;
        case 0x75a7: wc = 0x63e5; break;
        case 0x75a8: wc = 0x640b; break;
        case 0x75a9: wc = 0x6470; break;
        case 0x75aa: wc = 0x6755; break;
        case 0x75ab: wc = 0x6818; break;
        case 0x75ac: wc = 0x68e3; break;
        case 0x75ad: wc = 0x6974; break;
        case 0x75ae: wc = 0x6ba2; break;
        case 0x75af: wc = 0x7590; break;
        case 0x75b0: wc = 0x75f8; break;
        case 0x75b1: wc = 0x7689; break;
        case 0x75b2: wc = 0x780c; break;
        case 0x75b3: wc = 0x7998; break;
        case 0x75b4: wc = 0x8482; break;
        case 0x75b5: wc = 0x8515; break;
        case 0x75b6: wc = 0x8743; break;
        case 0x75b7: wc = 0x87ae; break;
        case 0x75b8: wc = 0x8905; break;
        case 0x75b9: wc = 0x8ec6; break;
        case 0x75ba: wc = 0x8fcf; break;
        case 0x75bb: wc = 0x9046; break;
        case 0x75bc: wc = 0x9070; break;
        case 0x75bd: wc = 0x91f1; break;
        case 0x75be: wc = 0x9290; break;
        case 0x75bf: wc = 0x9746; break;
        case 0x75c0: wc = 0x4a86; break;
        case 0x75c1: wc = 0x991f; break;
        case 0x75c2: wc = 0x9ae2; break;
        case 0x75c3: wc = 0x9b04; break;
        case 0x75c4: wc = 0x9dc8; break;
        case 0x75c5: wc = 0x4fcf; break;
        case 0x75c6: wc = 0x5062; break;
        case 0x75c7: wc = 0x50ec; break;
        case 0x75c8: wc = 0x510a; break;
        case 0x75c9: wc = 0x5281; break;
        case 0x75ca: wc = 0x528b; break;
        case 0x75cb: wc = 0x52e6; break;
        case 0x75cc: wc = 0x5435; break;
        case 0x75cd: wc = 0x564d; break;
        case 0x75ce: wc = 0x59b1; break;
        case 0x75cf: wc = 0x5af6; break;
        case 0x75d0: wc = 0x5c4c; break;
        case 0x75d1: wc = 0x5ca7; break;
        case 0x75d2: wc = 0x5ced; break;
        case 0x75d3: wc = 0x5d95; break;
        case 0x75d4: wc = 0x5e29; break;
        case 0x75d5: wc = 0x5f28; break;
        case 0x75d6: wc = 0x600a; break;
        case 0x75d7: wc = 0x6084; break;
        case 0x75d8: wc = 0x6100; break;
        case 0x75d9: wc = 0x61b7; break;
        case 0x75da: wc = 0x6477; break;
        case 0x75db: wc = 0x676a; break;
        case 0x75dc: wc = 0x6a9a; break;
        case 0x75dd: wc = 0x6af5; break;
        case 0x75de: wc = 0x6f50; break;
        case 0x75df: wc = 0x6fcb; break;
        case 0x75e0: wc = 0x714d; break;
        case 0x75e1: wc = 0x71cb; break;
        case 0x75e2: wc = 0x74b4; break;
        case 0x75e3: wc = 0x7727; break;
        case 0x75e4: wc = 0x7744; break;
        case 0x75e5: wc = 0x77a7; break;
        case 0x75e6: wc = 0x7952; break;
        case 0x75e7: wc = 0x7d83; break;
        case 0x75e8: wc = 0x8016; break;
        case 0x75e9: wc = 0x81b2; break;
        case 0x75ea: wc = 0x8279; break;
        case 0x75eb: wc = 0xfa5e; break;
        case 0x75ec: wc = 0x8280; break;
        case 0x75ed: wc = 0x8379; break;
        case 0x75ee: wc = 0x83ec; break;
        case 0x75ef: wc = 0x87ed; break;
        case 0x75f0: wc = 0x8a2c; break;
        case 0x75f1: wc = 0x8a9a; break;
        case 0x75f2: wc = 0x8b05; break;
        case 0x75f3: wc = 0x8b59; break;
        case 0x75f4: wc = 0x8da0; break;
        case 0x75f5: wc = 0x8dad; break;
        case 0x75f6: wc = 0x8efa; break;
        case 0x75f7: wc = 0x8f48; break;
        case 0x75f8: wc = 0x8fe2; break;
        case 0x75f9: wc = 0x9214; break;
        case 0x75fa: wc = 0x924a; break;
        case 0x75fb: wc = 0x936c; break;
        case 0x75fc: wc = 0x9363; break;
        case 0x75fd: wc = 0x940e; break;
        case 0x75fe: wc = 0x9798; break;
        case 0x75ff: wc = 0x9866; break;
        case 0x7600: wc = 0x9a32; break;
        case 0x7601: wc = 0x9aeb; break;
        case 0x7602: wc = 0x9de6; break;
        case 0x7603: wc = 0x9ea8; break;
        case 0x7604: wc = 0x9f60; break;
        case 0x7605: wc = 0x9f7c; break;
        case 0x7606: wc = 0x4e8d; break;
        case 0x7607: wc = 0x529a; break;
        case 0x7608: wc = 0x5b4e; break;
        case 0x7609: wc = 0x65b6; break;
        case 0x760a: wc = 0x65b8; break;
        case 0x760b: wc = 0x6b18; break;
        case 0x760c: wc = 0x6b5c; break;
        case 0x760d: wc = 0x705f; break;
        case 0x760e: wc = 0x7225; break;
        case 0x760f: wc = 0x7603; break;
        case 0x7610: wc = 0x77da; break;
        case 0x7611: wc = 0x81c5; break;
        case 0x7612: wc = 0x85a5; break;
        case 0x7613: wc = 0x880b; break;
        case 0x7614: wc = 0x883e; break;
        case 0x7615: wc = 0x8961; break;
        case 0x7616: wc = 0x8969; break;
        case 0x7617: wc = 0x89e6; break;
        case 0x7618: wc = 0x8e00; break;
        case 0x7619: wc = 0x8e85; break;
        case 0x761a: wc = 0x9483; break;
        case 0x761b: wc = 0x9ad1; break;
        case 0x761c: wc = 0x9e00; break;
        case 0x761d: wc = 0x520c; break;
        case 0x761e: wc = 0x540b; break;
        case 0x761f: wc = 0x5c57; break;
        case 0x7620: wc = 0x3e11; break;
        case 0x7621: wc = 0x7c7f; break;
        case 0x7622: wc = 0x50af; break;
        case 0x7623: wc = 0x51a2; break;
        case 0x7624: wc = 0x5306; break;
        case 0x7625: wc = 0x6031; break;
        case 0x7626: wc = 0x60e3; break;
        case 0x7627: wc = 0x6121; break;
        case 0x7628: wc = 0x68c7; break;
        case 0x7629: wc = 0x6b09; break;
        case 0x762a: wc = 0x6f0e; break;
        case 0x762b: wc = 0x6f68; break;
        case 0x762c: wc = 0x6fcd; break;
        case 0x762d: wc = 0x7047; break;
        case 0x762e: wc = 0x719c; break;
        case 0x762f: wc = 0x7481; break;
        case 0x7630: wc = 0x25865; break;
        case 0x7631: wc = 0x7deb; break;
        case 0x7632: wc = 0x8310; break;
        case 0x7633: wc = 0x8471; break;
        case 0x7634: wc = 0x84ef; break;
        case 0x7635: wc = 0x85c2; break;
        case 0x7636: wc = 0x457a; break;
        case 0x7637: wc = 0x87cc; break;
        case 0x7638: wc = 0x8b25; break;
        case 0x7639: wc = 0x93e6; break;
        case 0x763a: wc = 0x93d3; break;
        case 0x763b: wc = 0x9a44; break;
        case 0x763c: wc = 0x6525; break;
        case 0x763d: wc = 0x3b6e; break;
        case 0x763e: wc = 0x7e53; break;
        case 0x763f: wc = 0x5550; break;
        case 0x7640: wc = 0x797d; break;
        case 0x7641: wc = 0x51d7; break;
        case 0x7642: wc = 0x55fa; break;
        case 0x7643: wc = 0x562c; break;
        case 0x7644: wc = 0x5894; break;
        case 0x7645: wc = 0x3761; break;
        case 0x7646: wc = 0x5d8a; break;
        case 0x7647: wc = 0x615b; break;
        case 0x7648: wc = 0x6467; break;
        case 0x7649: wc = 0x69b1; break;
        case 0x764a: wc = 0x69ef; break;
        case 0x764b: wc = 0x6a36; break;
        case 0x764c: wc = 0x6b3c; break;
        case 0x764d: wc = 0x6f3c; break;
        case 0x764e: wc = 0x7480; break;
        case 0x764f: wc = 0x78ea; break;
        case 0x7650: wc = 0x7db7; break;
        case 0x7651: wc = 0x7e17; break;
        case 0x7652: wc = 0x7fa7; break;
        case 0x7653: wc = 0x8127; break;
        case 0x7654: wc = 0x8197; break;
        case 0x7655: wc = 0x855e; break;
        case 0x7656: wc = 0x92f7; break;
        case 0x7657: wc = 0x93d9; break;
        case 0x7658: wc = 0x3473; break;
        case 0x7659: wc = 0x50e6; break;
        case 0x765a: wc = 0x50fd; break;
        case 0x765b: wc = 0x541c; break;
        case 0x765c: wc = 0x35a9; break;
        case 0x765d: wc = 0x557e; break;
        case 0x765e: wc = 0x5a35; break;
        case 0x765f: wc = 0x5a9d; break;
        case 0x7660: wc = 0x5ab0; break;
        case 0x7661: wc = 0x5d37; break;
        case 0x7662: wc = 0x5e1a; break;
        case 0x7663: wc = 0x60c6; break;
        case 0x7664: wc = 0x396e; break;
        case 0x7665: wc = 0x61b1; break;
        case 0x7666: wc = 0x6376; break;
        case 0x7667: wc = 0x63ab; break;
        case 0x7668: wc = 0x63ea; break;
        case 0x7669: wc = 0x63c2; break;
        case 0x766a: wc = 0x63eb; break;
        case 0x766b: wc = 0x640a; break;
        case 0x766c: wc = 0x6425; break;
        case 0x766d: wc = 0x3aff; break;
        case 0x766e: wc = 0x68f7; break;
        case 0x766f: wc = 0x68f8; break;
        case 0x7670: wc = 0x68f0; break;
        case 0x7671: wc = 0x6a7b; break;
        case 0x7672: wc = 0x6ba0; break;
        case 0x7673: wc = 0x6c9d; break;
        case 0x7674: wc = 0x6eb4; break;
        case 0x7675: wc = 0x79cc; break;
        case 0x7676: wc = 0x716a; break;
        case 0x7677: wc = 0x717c; break;
        case 0x7678: wc = 0x71de; break;
        case 0x7679: wc = 0x7293; break;
        case 0x767a: wc = 0x7500; break;
        case 0x767b: wc = 0x7503; break;
        case 0x767c: wc = 0x7633; break;
        case 0x767d: wc = 0x776d; break;
        case 0x767e: wc = 0x7785; break;
        case 0x767f: wc = 0x787e; break;
        case 0x7680: wc = 0x7ba0; break;
        case 0x7681: wc = 0x7b92; break;
        case 0x7682: wc = 0x7bd8; break;
        case 0x7683: wc = 0x7c09; break;
        case 0x7684: wc = 0x7dc5; break;
        case 0x7685: wc = 0x7de7; break;
        case 0x7686: wc = 0x7e10; break;
        case 0x7687: wc = 0x7e0b; break;
        case 0x7688: wc = 0x4410; break;
        case 0x7689: wc = 0x8187; break;
        case 0x768a: wc = 0x4422; break;
        case 0x768b: wc = 0x83c6; break;
        case 0x768c: wc = 0x8411; break;
        case 0x768d: wc = 0x84ad; break;
        case 0x768e: wc = 0x8764; break;
        case 0x768f: wc = 0x8775; break;
        case 0x7690: wc = 0x87d7; break;
        case 0x7691: wc = 0x89d5; break;
        case 0x7692: wc = 0x8ac8; break;
        case 0x7693: wc = 0x8c99; break;
        case 0x7694: wc = 0x8d8d; break;
        case 0x7695: wc = 0x8da5; break;
        case 0x7696: wc = 0x90f0; break;
        case 0x7697: wc = 0x9139; break;
        case 0x7698: wc = 0x9446; break;
        case 0x7699: wc = 0x966c; break;
        case 0x769a: wc = 0x96b9; break;
        case 0x769b: wc = 0x97a6; break;
        case 0x769c: wc = 0x99ce; break;
        case 0x769d: wc = 0x9a05; break;
        case 0x769e: wc = 0x9b0c; break;
        case 0x769f: wc = 0x9bde; break;
        case 0x76a0: wc = 0x9beb; break;
        case 0x76a1: wc = 0x9c0c; break;
        case 0x76a2: wc = 0x9c43; break;
        case 0x76a3: wc = 0x9d7b; break;
        case 0x76a4: wc = 0x9d96; break;
        case 0x76a5: wc = 0x9db5; break;
        case 0x76a6: wc = 0x9ea4; break;
        case 0x76a7: wc = 0x9ec0; break;
        case 0x76a8: wc = 0x9f71; break;
        case 0x76a9: wc = 0x9f7a; break;
        case 0x76aa: wc = 0x35e4; break;
        case 0x76ab: wc = 0x5648; break;
        case 0x76ac: wc = 0x213ba; break;
        case 0x76ad: wc = 0x59af; break;
        case 0x76ae: wc = 0x654a; break;
        case 0x76af: wc = 0x67f7; break;
        case 0x76b0: wc = 0x69ed; break;
        case 0x76b1: wc = 0x6ba7; break;
        case 0x76b2: wc = 0x6ec0; break;
        case 0x76b3: wc = 0x7a38; break;
        case 0x76b4: wc = 0x7b01; break;
        case 0x76b5: wc = 0x7beb; break;
        case 0x76b6: wc = 0x8233; break;
        case 0x76b7: wc = 0x833f; break;
        case 0x76b8: wc = 0x84eb; break;
        case 0x76b9: wc = 0x8c56; break;
        case 0x76ba: wc = 0x8e27; break;
        case 0x76bb: wc = 0x8e5c; break;
        case 0x76bc: wc = 0x282c0; break;
        case 0x76bd: wc = 0x9110; break;
        case 0x76be: wc = 0x95a6; break;
        case 0x76bf: wc = 0x9863; break;
        case 0x76c0: wc = 0x9c41; break;
        case 0x76c1: wc = 0x9f00; break;
        case 0x76c2: wc = 0x5a8b; break;
        case 0x76c3: wc = 0x6699; break;
        case 0x76c4: wc = 0x6776; break;
        case 0x76c5: wc = 0x6a41; break;
        case 0x76c6: wc = 0x6e77; break;
        case 0x76c7: wc = 0x8436; break;
        case 0x76c8: wc = 0x877d; break;
        case 0x76c9: wc = 0x8cf0; break;
        case 0x76ca: wc = 0x9c06; break;
        case 0x76cb: wc = 0x9d9e; break;
        case 0x76cc: wc = 0x6035; break;
        case 0x76cd: wc = 0x6037; break;
        case 0x76ce: wc = 0x6ccf; break;
        case 0x76cf: wc = 0x7840; break;
        case 0x76d0: wc = 0x79eb; break;
        case 0x76d1: wc = 0x7d40; break;
        case 0x76d2: wc = 0x838d; break;
        case 0x76d3: wc = 0x51b2; break;
        case 0x76d4: wc = 0x5fe1; break;
        case 0x76d5: wc = 0x3959; break;
        case 0x76d6: wc = 0x2295e; break;
        case 0x76d7: wc = 0x63f0; break;
        case 0x76d8: wc = 0x3cd8; break;
        case 0x76d9: wc = 0x6d7a; break;
        case 0x76da: wc = 0x6f34; break;
        case 0x76db: wc = 0x721e; break;
        case 0x76dc: wc = 0x73eb; break;
        case 0x76dd: wc = 0x75cb; break;
        case 0x76de: wc = 0x76c5; break;
        case 0x76df: wc = 0x794c; break;
        case 0x76e0: wc = 0x79cd; break;
        case 0x76e1: wc = 0x7fc0; break;
        case 0x76e2: wc = 0x833a; break;
        case 0x76e3: wc = 0x8327; break;
        case 0x76e4: wc = 0x866b; break;
        case 0x76e5: wc = 0x60f4; break;
        case 0x76e6: wc = 0x63e3; break;
        case 0x76e7: wc = 0x7601; break;
        case 0x76e8: wc = 0x9847; break;
        case 0x76e9: wc = 0x51a3; break;
        case 0x76ea: wc = 0x567f; break;
        case 0x76eb: wc = 0x5d89; break;
        case 0x76ec: wc = 0x6a47; break;
        case 0x76ed: wc = 0x6a87; break;
        case 0x76ee: wc = 0x6b08; break;
        case 0x76ef: wc = 0x6bf3; break;
        case 0x76f0: wc = 0x6fe2; break;
        case 0x76f1: wc = 0x7123; break;
        case 0x76f2: wc = 0x74bb; break;
        case 0x76f3: wc = 0x7ac1; break;
        case 0x76f4: wc = 0x81ac; break;
        case 0x76f5: wc = 0x81ce; break;
        case 0x76f6: wc = 0x28d91; break;
        case 0x76f7: wc = 0x5e82; break;
        case 0x76f8: wc = 0x5ec1; break;
        case 0x76f9: wc = 0x6351; break;
        case 0x76fa: wc = 0x6603; break;
        case 0x76fb: wc = 0x755f; break;
        case 0x76fc: wc = 0x7a04; break;
        case 0x76fd: wc = 0x512d; break;
        case 0x76fe: wc = 0x6aec; break;
        case 0x76ff: wc = 0x9f54; break;
        case 0x7700: wc = 0x8e6d; break;
        case 0x7701: wc = 0x4e7f; break;
        case 0x7702: wc = 0x506b; break;
        case 0x7703: wc = 0x5082; break;
        case 0x7704: wc = 0x536e; break;
        case 0x7705: wc = 0x54c6; break;
        case 0x7706: wc = 0x5902; break;
        case 0x7707: wc = 0x5ab8; break;
        case 0x7708: wc = 0x5bd8; break;
        case 0x7709: wc = 0x5d30; break;
        case 0x770a: wc = 0x5df5; break;
        case 0x770b: wc = 0x5ea4; break;
        case 0x770c: wc = 0x5ecc; break;
        case 0x770d: wc = 0x6040; break;
        case 0x770e: wc = 0x61e5; break;
        case 0x770f: wc = 0x61eb; break;
        case 0x7710: wc = 0x62f8; break;
        case 0x7711: wc = 0x22dd0; break;
        case 0x7712: wc = 0x6914; break;
        case 0x7713: wc = 0x6b6d; break;
        case 0x7714: wc = 0x6ecd; break;
        case 0x7715: wc = 0x74fb; break;
        case 0x7716: wc = 0x753e; break;
        case 0x7717: wc = 0x2f936; break;
        case 0x7718: wc = 0x7564; break;
        case 0x7719: wc = 0x75d3; break;
        case 0x771a: wc = 0x7735; break;
        case 0x771b: wc = 0x7d0e; break;
        case 0x771c: wc = 0x7d7a; break;
        case 0x771d: wc = 0x801b; break;
        case 0x771e: wc = 0x803b; break;
        case 0x771f: wc = 0x80f5; break;
        case 0x7720: wc = 0x830c; break;
        case 0x7721: wc = 0x834e; break;
        case 0x7722: wc = 0x832c; break;
        case 0x7723: wc = 0x83d1; break;
        case 0x7724: wc = 0x8599; break;
        case 0x7725: wc = 0x85e2; break;
        case 0x7726: wc = 0x892b; break;
        case 0x7727: wc = 0x89f6; break;
        case 0x7728: wc = 0x8a83; break;
        case 0x7729: wc = 0x8aba; break;
        case 0x772a: wc = 0x8c78; break;
        case 0x772b: wc = 0x8df1; break;
        case 0x772c: wc = 0x8dee; break;
        case 0x772d: wc = 0x90d7; break;
        case 0x772e: wc = 0x9279; break;
        case 0x772f: wc = 0x9319; break;
        case 0x7730: wc = 0x964a; break;
        case 0x7731: wc = 0x99e4; break;
        case 0x7732: wc = 0x9bd4; break;
        case 0x7733: wc = 0x9d19; break;
        case 0x7734: wc = 0x9d1f; break;
        case 0x7735: wc = 0x9d44; break;
        case 0x7736: wc = 0x9d85; break;
        case 0x7737: wc = 0x9ef9; break;
        case 0x7738: wc = 0x9f5d; break;
        case 0x7739: wc = 0x4f99; break;
        case 0x773a: wc = 0x605c; break;
        case 0x773b: wc = 0x6555; break;
        case 0x773c: wc = 0x6dd4; break;
        case 0x773d: wc = 0x906b; break;
        case 0x773e: wc = 0x9253; break;
        case 0x773f: wc = 0x9d92; break;
        case 0x7740: wc = 0x9dd8; break;
        case 0x7741: wc = 0x56ab; break;
        case 0x7742: wc = 0x7019; break;
        case 0x7743: wc = 0x85fd; break;
        case 0x7744: wc = 0x896f; break;
        case 0x7745: wc = 0x478b; break;
        case 0x7746: wc = 0x4833; break;
        case 0x7747: wc = 0x34fc; break;
        case 0x7748: wc = 0x687c; break;
        case 0x7749: wc = 0x4f14; break;
        case 0x774a: wc = 0x551a; break;
        case 0x774b: wc = 0x57d0; break;
        case 0x774c: wc = 0x5bd1; break;
        case 0x774d: wc = 0x5bd6; break;
        case 0x774e: wc = 0x5d9c; break;
        case 0x774f: wc = 0x5ff1; break;
        case 0x7750: wc = 0x628c; break;
        case 0x7751: wc = 0x63d5; break;
        case 0x7752: wc = 0x67ae; break;
        case 0x7753: wc = 0x68ab; break;
        case 0x7754: wc = 0x68a3; break;
        case 0x7755: wc = 0x6c89; break;
        case 0x7756: wc = 0x78aa; break;
        case 0x7757: wc = 0x7972; break;
        case 0x7758: wc = 0x7d85; break;
        case 0x7759: wc = 0x7d9d; break;
        case 0x775a: wc = 0x8390; break;
        case 0x775b: wc = 0x8474; break;
        case 0x775c: wc = 0x8ac3; break;
        case 0x775d: wc = 0x8cdd; break;
        case 0x775e: wc = 0x8e38; break;
        case 0x775f: wc = 0x90f4; break;
        case 0x7760: wc = 0x9202; break;
        case 0x7761: wc = 0x92df; break;
        case 0x7762: wc = 0x9356; break;
        case 0x7763: wc = 0x99f8; break;
        case 0x7764: wc = 0x9c75; break;
        case 0x7765: wc = 0x6f10; break;
        case 0x7766: wc = 0x99bd; break;
        case 0x7767: wc = 0x5041; break;
        case 0x7768: wc = 0x79f0; break;
        case 0x7769: wc = 0x4f67; break;
        case 0x776a: wc = 0x5108; break;
        case 0x776b: wc = 0x5672; break;
        case 0x776c: wc = 0x592c; break;
        case 0x776d: wc = 0x7b77; break;
        case 0x776e: wc = 0x4f57; break;
        case 0x776f: wc = 0x5234; break;
        case 0x7770: wc = 0x5412; break;
        case 0x7771: wc = 0x5491; break;
        case 0x7772: wc = 0x56b2; break;
        case 0x7773: wc = 0x5768; break;
        case 0x7774: wc = 0x579e; break;
        case 0x7775: wc = 0x579c; break;
        case 0x7776: wc = 0x57f5; break;
        case 0x7777: wc = 0x5836; break;
        case 0x7778: wc = 0x58af; break;
        case 0x7779: wc = 0x597c; break;
        case 0x777a: wc = 0x5aa0; break;
        case 0x777b: wc = 0x5af7; break;
        case 0x777c: wc = 0x5b83; break;
        case 0x777d: wc = 0x5cae; break;
        case 0x777e: wc = 0x5d9e; break;
        case 0x777f: wc = 0x6261; break;
        case 0x7780: wc = 0x62d5; break;
        case 0x7781: wc = 0x64b1; break;
        case 0x7782: wc = 0x6735; break;
        case 0x7783: wc = 0x67c1; break;
        case 0x7784: wc = 0x692f; break;
        case 0x7785: wc = 0x6a62; break;
        case 0x7786: wc = 0x6be4; break;
        case 0x7787: wc = 0x6bfb; break;
        case 0x7788: wc = 0x6cb1; break;
        case 0x7789: wc = 0x6cb2; break;
        case 0x778a: wc = 0x7260; break;
        case 0x778b: wc = 0x75d1; break;
        case 0x778c: wc = 0x7823; break;
        case 0x778d: wc = 0x78a2; break;
        case 0x778e: wc = 0x79c5; break;
        case 0x778f: wc = 0x7d3d; break;
        case 0x7790: wc = 0x7d9e; break;
        case 0x7791: wc = 0x2658b; break;
        case 0x7792: wc = 0x8889; break;
        case 0x7793: wc = 0x8a6b; break;
        case 0x7794: wc = 0x8d93; break;
        case 0x7795: wc = 0x8dce; break;
        case 0x7796: wc = 0x8de5; break;
        case 0x7797: wc = 0x8eb1; break;
        case 0x7798: wc = 0x8ec3; break;
        case 0x7799: wc = 0x9161; break;
        case 0x779a: wc = 0x937a; break;
        case 0x779b: wc = 0x9624; break;
        case 0x779c: wc = 0x9641; break;
        case 0x779d: wc = 0x964f; break;
        case 0x779e: wc = 0x9b80; break;
        case 0x779f: wc = 0x9d15; break;
        case 0x77a0: wc = 0x9f09; break;
        case 0x77a1: wc = 0x9f27; break;
        case 0x77a2: wc = 0x4e47; break;
        case 0x77a3: wc = 0x4f32; break;
        case 0x77a4: wc = 0x4f82; break;
        case 0x77a5: wc = 0x51d9; break;
        case 0x77a6: wc = 0x5262; break;
        case 0x77a7: wc = 0x526b; break;
        case 0x77a8: wc = 0x20e0c; break;
        case 0x77a9: wc = 0x5545; break;
        case 0x77aa: wc = 0x5eb9; break;
        case 0x77ab: wc = 0x391e; break;
        case 0x77ac: wc = 0x62c6; break;
        case 0x77ad: wc = 0x6580; break;
        case 0x77ae: wc = 0x684c; break;
        case 0x77af: wc = 0x6913; break;
        case 0x77b0: wc = 0x69d6; break;
        case 0x77b1: wc = 0x69d5; break;
        case 0x77b2: wc = 0x6a50; break;
        case 0x77b3: wc = 0x6cb0; break;
        case 0x77b4: wc = 0x23d60; break;
        case 0x77b5: wc = 0x6dbf; break;
        case 0x77b6: wc = 0x77fa; break;
        case 0x77b7: wc = 0x7aa7; break;
        case 0x77b8: wc = 0x7c5c; break;
        case 0x77b9: wc = 0x8600; break;
        case 0x77ba: wc = 0x8817; break;
        case 0x77bb: wc = 0x8957; break;
        case 0x77bc: wc = 0x8dc5; break;
        case 0x77bd: wc = 0x8e14; break;
        case 0x77be: wc = 0x8e31; break;
        case 0x77bf: wc = 0x9034; break;
        case 0x77c0: wc = 0x9432; break;
        case 0x77c1: wc = 0x98e5; break;
        case 0x77c2: wc = 0x99b2; break;
        case 0x77c3: wc = 0x298b5; break;
        case 0x77c4: wc = 0x9b60; break;
        case 0x77c5: wc = 0x9d6b; break;
        case 0x77c6: wc = 0x50e4; break;
        case 0x77c7: wc = 0x563d; break;
        case 0x77c8: wc = 0x36f6; break;
        case 0x77c9: wc = 0x6039; break;
        case 0x77ca: wc = 0x64a3; break;
        case 0x77cb: wc = 0x6524; break;
        case 0x77cc: wc = 0x66ba; break;
        case 0x77cd: wc = 0x6bab; break;
        case 0x77ce: wc = 0x6c3d; break;
        case 0x77cf: wc = 0x6e60; break;
        case 0x77d0: wc = 0x7583; break;
        case 0x77d1: wc = 0x7613; break;
        case 0x77d2: wc = 0x7671; break;
        case 0x77d3: wc = 0x78b3; break;
        case 0x77d4: wc = 0x7fb0; break;
        case 0x77d5: wc = 0x8b60; break;
        case 0x77d6: wc = 0x9a52; break;
        case 0x77d7: wc = 0x4fbb; break;
        case 0x77d8: wc = 0x6553; break;
        case 0x77d9: wc = 0x68b2; break;
        case 0x77da: wc = 0x6bf2; break;
        case 0x77db: wc = 0x838c; break;
        case 0x77dc: wc = 0x9bb5; break;
        case 0x77dd: wc = 0x9d7d; break;
        case 0x77de: wc = 0x50cb; break;
        case 0x77df: wc = 0x55ff; break;
        case 0x77e0: wc = 0x5fd0; break;
        case 0x77e1: wc = 0x393e; break;
        case 0x77e2: wc = 0x64a2; break;
        case 0x77e3: wc = 0x6c8a; break;
        case 0x77e4: wc = 0x43d9; break;
        case 0x77e5: wc = 0x8ce7; break;
        case 0x77e6: wc = 0x9193; break;
        case 0x77e7: wc = 0x509d; break;
        case 0x77e8: wc = 0x55d2; break;
        case 0x77e9: wc = 0x5683; break;
        case 0x77ea: wc = 0x584c; break;
        case 0x77eb: wc = 0x366e; break;
        case 0x77ec: wc = 0x6428; break;
        case 0x77ed: wc = 0x6498; break;
        case 0x77ee: wc = 0x6bfe; break;
        case 0x77ef: wc = 0x3cae; break;
        case 0x77f0: wc = 0x6ebb; break;
        case 0x77f1: wc = 0x6f2f; break;
        case 0x77f2: wc = 0x7629; break;
        case 0x77f3: wc = 0x79a2; break;
        case 0x77f4: wc = 0x4308; break;
        case 0x77f5: wc = 0x8037; break;
        case 0x77f6: wc = 0x447d; break;
        case 0x77f7: wc = 0x27131; break;
        case 0x77f8: wc = 0x9062; break;
        case 0x77f9: wc = 0x9314; break;
        case 0x77fa: wc = 0x9391; break;
        case 0x77fb: wc = 0x9449; break;
        case 0x77fc: wc = 0x95d2; break;
        case 0x77fd: wc = 0x979c; break;
        case 0x77fe: wc = 0x97b3; break;
        case 0x77ff: wc = 0x9c28; break;
        case 0x7800: wc = 0x5a78; break;
        case 0x7801: wc = 0x6465; break;
        case 0x7802: wc = 0x71d9; break;
        case 0x7803: wc = 0x7497; break;
        case 0x7804: wc = 0x76ea; break;
        case 0x7805: wc = 0x78ad; break;
        case 0x7806: wc = 0x7c1c; break;
        case 0x7807: wc = 0x83ea; break;
        case 0x7808: wc = 0x26cdd; break;
        case 0x7809: wc = 0x859a; break;
        case 0x780a: wc = 0x876a; break;
        case 0x780b: wc = 0x8e3c; break;
        case 0x780c: wc = 0x903f; break;
        case 0x780d: wc = 0x940b; break;
        case 0x780e: wc = 0x3440; break;
        case 0x780f: wc = 0x50a3; break;
        case 0x7810: wc = 0x5454; break;
        case 0x7811: wc = 0x5446; break;
        case 0x7812: wc = 0x57ed; break;
        case 0x7813: wc = 0x5a27; break;
        case 0x7814: wc = 0x38d6; break;
        case 0x7815: wc = 0x5fd5; break;
        case 0x7816: wc = 0x5ff2; break;
        case 0x7817: wc = 0x623b; break;
        case 0x7818: wc = 0x62ac; break;
        case 0x7819: wc = 0x6378; break;
        case 0x781a: wc = 0x6584; break;
        case 0x781b: wc = 0x3cf2; break;
        case 0x781c: wc = 0x6e99; break;
        case 0x781d: wc = 0x70b1; break;
        case 0x781e: wc = 0x71e4; break;
        case 0x781f: wc = 0x73c6; break;
        case 0x7820: wc = 0x7d3f; break;
        case 0x7821: wc = 0x86fb; break;
        case 0x7822: wc = 0x469f; break;
        case 0x7823: wc = 0x8ee9; break;
        case 0x7824: wc = 0x8fe8; break;
        case 0x7825: wc = 0x9226; break;
        case 0x7826: wc = 0x294ac; break;
        case 0x7827: wc = 0x99c4; break;
        case 0x7828: wc = 0x99d8; break;
        case 0x7829: wc = 0x99fe; break;
        case 0x782a: wc = 0x9b90; break;
        case 0x782b: wc = 0x3b66; break;
        case 0x782c: wc = 0x70e2; break;
        case 0x782d: wc = 0x880c; break;
        case 0x782e: wc = 0x9e05; break;
        case 0x782f: wc = 0x6490; break;
        case 0x7830: wc = 0x6a18; break;
        case 0x7831: wc = 0x6a56; break;
        case 0x7832: wc = 0x6a55; break;
        case 0x7833: wc = 0x725a; break;
        case 0x7834: wc = 0x7ac0; break;
        case 0x7835: wc = 0xfa32; break;
        case 0x7836: wc = 0x580d; break;
        case 0x7837: wc = 0x6c62; break;
        case 0x7838: wc = 0x828f; break;
        case 0x7839: wc = 0x8314; break;
        case 0x783a: wc = 0x83df; break;
        case 0x783b: wc = 0x91f7; break;
        case 0x783c: wc = 0x29876; break;
        case 0x783d: wc = 0x9d75; break;
        case 0x783e: wc = 0x554d; break;
        case 0x783f: wc = 0x564b; break;
        case 0x7840: wc = 0x5678; break;
        case 0x7841: wc = 0x74f2; break;
        case 0x7842: wc = 0x757d; break;
        case 0x7843: wc = 0x340b; break;
        case 0x7844: wc = 0x606b; break;
        case 0x7845: wc = 0x6345; break;
        case 0x7846: wc = 0x2333e; break;
        case 0x7847: wc = 0x6a0b; break;
        case 0x7848: wc = 0x3dc1; break;
        case 0x7849: wc = 0x71a5; break;
        case 0x784a: wc = 0x75cc; break;
        case 0x784b: wc = 0x84ea; break;
        case 0x784c: wc = 0x9bd2; break;
        case 0x784d: wc = 0x4fc0; break;
        case 0x784e: wc = 0x50d3; break;
        case 0x784f: wc = 0x5c35; break;
        case 0x7850: wc = 0x5f1a; break;
        case 0x7851: wc = 0x75fd; break;
        case 0x7852: wc = 0x3fc9; break;
        case 0x7853: wc = 0x3fd7; break;
        case 0x7854: wc = 0x7a68; break;
        case 0x7855: wc = 0x84f7; break;
        case 0x7856: wc = 0x85ec; break;
        case 0x7857: wc = 0x8e6a; break;
        case 0x7858: wc = 0x96a4; break;
        case 0x7859: wc = 0x9827; break;
        case 0x785a: wc = 0x9b4b; break;
        case 0x785b: wc = 0x9d2d; break;
        case 0x785c: wc = 0x5992; break;
        case 0x785d: wc = 0x6e1d; break;
        case 0x785e: wc = 0x724f; break;
        case 0x785f: wc = 0x24d31; break;
        case 0x7860: wc = 0x79fa; break;
        case 0x7861: wc = 0x8623; break;
        case 0x7862: wc = 0x9ab0; break;
        case 0x7863: wc = 0x5fd1; break;
        case 0x7864: wc = 0x5fd2; break;
        case 0x7865: wc = 0x87d8; break;
        case 0x7866: wc = 0x8ca3; break;
        case 0x7867: wc = 0x92f1; break;
        case 0x7868: wc = 0x3bd1; break;
        case 0x7869: wc = 0x53f5; break;
        case 0x786a: wc = 0x5427; break;
        case 0x786b: wc = 0x20d38; break;
        case 0x786c: wc = 0x5c00; break;
        case 0x786d: wc = 0x5ca5; break;
        case 0x786e: wc = 0x5d93; break;
        case 0x786f: wc = 0x5e0a; break;
        case 0x7870: wc = 0x5e15; break;
        case 0x7871: wc = 0x5f1d; break;
        case 0x7872: wc = 0x6015; break;
        case 0x7873: wc = 0x6ac7; break;
        case 0x7874: wc = 0x705e; break;
        case 0x7875: wc = 0x7238; break;
        case 0x7876: wc = 0x73bb; break;
        case 0x7877: wc = 0x75a4; break;
        case 0x7878: wc = 0x76a4; break;
        case 0x7879: wc = 0x7886; break;
        case 0x787a: wc = 0x7b06; break;
        case 0x787b: wc = 0x7b38; break;
        case 0x787c: wc = 0x7b62; break;
        case 0x787d: wc = 0x7c38; break;
        case 0x787e: wc = 0x7c91; break;
        case 0x787f: wc = 0x7f93; break;
        case 0x7880: wc = 0x8019; break;
        case 0x7881: wc = 0x8225; break;
        case 0x7882: wc = 0x83e0; break;
        case 0x7883: wc = 0x848e; break;
        case 0x7884: wc = 0x8469; break;
        case 0x7885: wc = 0x8686; break;
        case 0x7886: wc = 0x86be; break;
        case 0x7887: wc = 0x8899; break;
        case 0x7888: wc = 0x8b52; break;
        case 0x7889: wc = 0x8c5d; break;
        case 0x788a: wc = 0x27ca7; break;
        case 0x788b: wc = 0x8db4; break;
        case 0x788c: wc = 0x8dc1; break;
        case 0x788d: wc = 0x9131; break;
        case 0x788e: wc = 0x9200; break;
        case 0x788f: wc = 0x9303; break;
        case 0x7890: wc = 0x9383; break;
        case 0x7891: wc = 0x9776; break;
        case 0x7892: wc = 0x4d95; break;
        case 0x7893: wc = 0x6604; break;
        case 0x7894: wc = 0x7548; break;
        case 0x7895: wc = 0x7705; break;
        case 0x7896: wc = 0x7c84; break;
        case 0x7897: wc = 0x8228; break;
        case 0x7898: wc = 0x8742; break;
        case 0x7899: wc = 0x4ec8; break;
        case 0x789a: wc = 0x54f5; break;
        case 0x789b: wc = 0x6733; break;
        case 0x789c: wc = 0x6c43; break;
        case 0x789d: wc = 0x5dfc; break;
        case 0x789e: wc = 0x4f02; break;
        case 0x789f: wc = 0x58e9; break;
        case 0x78a0: wc = 0x59f5; break;
        case 0x78a1: wc = 0x5b5b; break;
        case 0x78a2: wc = 0x636d; break;
        case 0x78a3: wc = 0x65c6; break;
        case 0x78a4: wc = 0x6896; break;
        case 0x78a5: wc = 0x68d1; break;
        case 0x78a6: wc = 0x726c; break;
        case 0x78a7: wc = 0x3e3d; break;
        case 0x78a8: wc = 0x7308; break;
        case 0x78a9: wc = 0x73ee; break;
        case 0x78aa: wc = 0x73fc; break;
        case 0x78ab: wc = 0x77f2; break;
        case 0x78ac: wc = 0x7c30; break;
        case 0x78ad: wc = 0x7cba; break;
        case 0x78ae: wc = 0x8337; break;
        case 0x78af: wc = 0x8a96; break;
        case 0x78b0: wc = 0x90b6; break;
        case 0x78b1: wc = 0x90e5; break;
        case 0x78b2: wc = 0x9101; break;
        case 0x78b3: wc = 0x92c7; break;
        case 0x78b4: wc = 0x9708; break;
        case 0x78b5: wc = 0x9738; break;
        case 0x78b6: wc = 0x4f3b; break;
        case 0x78b7: wc = 0x562d; break;
        case 0x78b8: wc = 0x6189; break;
        case 0x78b9: wc = 0x63bd; break;
        case 0x78ba: wc = 0x750f; break;
        case 0x78bb: wc = 0x768f; break;
        case 0x78bc: wc = 0x7830; break;
        case 0x78bd: wc = 0x78b0; break;
        case 0x78be: wc = 0x78de; break;
        case 0x78bf: wc = 0x794a; break;
        case 0x78c0: wc = 0x87db; break;
        case 0x78c1: wc = 0x8eef; break;
        case 0x78c2: wc = 0x8f23; break;
        case 0x78c3: wc = 0x930b; break;
        case 0x78c4: wc = 0x958d; break;
        case 0x78c5: wc = 0x95cf; break;
        case 0x78c6: wc = 0x9a2f; break;
        case 0x78c7: wc = 0x5aa5; break;
        case 0x78c8: wc = 0x5fa7; break;
        case 0x78c9: wc = 0x60fc; break;
        case 0x78ca: wc = 0x63d9; break;
        case 0x78cb: wc = 0x6969; break;
        case 0x78cc: wc = 0x6944; break;
        case 0x78cd: wc = 0x7178; break;
        case 0x78ce: wc = 0x728f; break;
        case 0x78cf: wc = 0x7335; break;
        case 0x78d0: wc = 0x78a5; break;
        case 0x78d1: wc = 0x7baf; break;
        case 0x78d2: wc = 0x7df6; break;
        case 0x78d3: wc = 0x2611f; break;
        case 0x78d4: wc = 0x8251; break;
        case 0x78d5: wc = 0x8439; break;
        case 0x78d6: wc = 0x8759; break;
        case 0x78d7: wc = 0x890a; break;
        case 0x78d8: wc = 0x8991; break;
        case 0x78d9: wc = 0x8ade; break;
        case 0x78da: wc = 0x8cb5; break;
        case 0x78db: wc = 0x8e41; break;
        case 0x78dc: wc = 0x9828; break;
        case 0x78dd: wc = 0x4b4f; break;
        case 0x78de: wc = 0x9bff; break;
        case 0x78df: wc = 0x9da3; break;
        case 0x78e0: wc = 0x782d; break;
        case 0x78e1: wc = 0x7a86; break;
        case 0x78e2: wc = 0x5309; break;
        case 0x78e3: wc = 0x546f; break;
        case 0x78e4: wc = 0x59d8; break;
        case 0x78e5: wc = 0x6026; break;
        case 0x78e6: wc = 0x62a8; break;
        case 0x78e7: wc = 0x2f8ba; break;
        case 0x78e8: wc = 0x6cd9; break;
        case 0x78e9: wc = 0x73b6; break;
        case 0x78ea: wc = 0x82f9; break;
        case 0x78eb: wc = 0x84f1; break;
        case 0x78ec: wc = 0x959b; break;
        case 0x78ed: wc = 0x99cd; break;
        case 0x78ee: wc = 0x9b83; break;
        case 0x78ef: wc = 0x655d; break;
        case 0x78f0: wc = 0x6890; break;
        case 0x78f1: wc = 0x6ae0; break;
        case 0x78f2: wc = 0x72f4; break;
        case 0x78f3: wc = 0x7359; break;
        case 0x78f4: wc = 0x7358; break;
        case 0x78f5: wc = 0x7648; break;
        case 0x78f6: wc = 0x7829; break;
        case 0x78f7: wc = 0x7b85; break;
        case 0x78f8: wc = 0x426c; break;
        case 0x78f9: wc = 0x43f6; break;
        case 0x78fa: wc = 0x870c; break;
        case 0x78fb: wc = 0x9128; break;
        case 0x78fc: wc = 0x9f23; break;
        case 0x78fd: wc = 0x4f68; break;
        case 0x78fe: wc = 0x5124; break;
        case 0x78ff: wc = 0x5228; break;
        case 0x7900: wc = 0x52f9; break;
        case 0x7901: wc = 0x5498; break;
        case 0x7902: wc = 0x5789; break;
        case 0x7903: wc = 0x57d4; break;
        case 0x7904: wc = 0x5945; break;
        case 0x7905: wc = 0x5b62; break;
        case 0x7906: wc = 0x5bb2; break;
        case 0x7907: wc = 0x5cec; break;
        case 0x7908: wc = 0x5e96; break;
        case 0x7909: wc = 0x5eaf; break;
        case 0x790a: wc = 0x5fc1; break;
        case 0x790b: wc = 0x6009; break;
        case 0x790c: wc = 0x62aa; break;
        case 0x790d: wc = 0x6661; break;
        case 0x790e: wc = 0x67b9; break;
        case 0x790f: wc = 0x70b0; break;
        case 0x7910: wc = 0x70ae; break;
        case 0x7911: wc = 0x72cd; break;
        case 0x7912: wc = 0x74dd; break;
        case 0x7913: wc = 0x76b0; break;
        case 0x7914: wc = 0x792e; break;
        case 0x7915: wc = 0x7b23; break;
        case 0x7916: wc = 0x812c; break;
        case 0x7917: wc = 0x8216; break;
        case 0x7918: wc = 0x83e2; break;
        case 0x7919: wc = 0x84b1; break;
        case 0x791a: wc = 0x8663; break;
        case 0x791b: wc = 0x888c; break;
        case 0x791c: wc = 0x8943; break;
        case 0x791d: wc = 0x8aa7; break;
        case 0x791e: wc = 0x8b08; break;
        case 0x791f: wc = 0x8dd1; break;
        case 0x7920: wc = 0x917a; break;
        case 0x7921: wc = 0x923d; break;
        case 0x7922: wc = 0x924b; break;
        case 0x7923: wc = 0x9464; break;
        case 0x7924: wc = 0x9764; break;
        case 0x7925: wc = 0x9784; break;
        case 0x7926: wc = 0x9914; break;
        case 0x7927: wc = 0x9af1; break;
        case 0x7928: wc = 0x9bc6; break;
        case 0x7929: wc = 0x9d4f; break;
        case 0x792a: wc = 0x9e83; break;
        case 0x792b: wc = 0x9e85; break;
        case 0x792c: wc = 0x9ead; break;
        case 0x792d: wc = 0x4d8c; break;
        case 0x792e: wc = 0x9f59; break;
        case 0x792f: wc = 0x2b0b5; break;
        case 0x7930: wc = 0x55b8; break;
        case 0x7931: wc = 0x50c4; break;
        case 0x7932: wc = 0x5126; break;
        case 0x7933: wc = 0x52e1; break;
        case 0x7934: wc = 0x560c; break;
        case 0x7935: wc = 0x5882; break;
        case 0x7936: wc = 0x5a4a; break;
        case 0x7937: wc = 0x5ad6; break;
        case 0x7938: wc = 0x5e56; break;
        case 0x7939: wc = 0x5f6f; break;
        case 0x793a: wc = 0x647d; break;
        case 0x793b: wc = 0x65da; break;
        case 0x793c: wc = 0x6aa6; break;
        case 0x793d: wc = 0x6b8d; break;
        case 0x793e: wc = 0x6df2; break;
        case 0x793f: wc = 0x6eee; break;
        case 0x7940: wc = 0x700c; break;
        case 0x7941: wc = 0x719b; break;
        case 0x7942: wc = 0x71e2; break;
        case 0x7943: wc = 0x7202; break;
        case 0x7944: wc = 0x72a5; break;
        case 0x7945: wc = 0x730b; break;
        case 0x7946: wc = 0x762d; break;
        case 0x7947: wc = 0x76ab; break;
        case 0x7948: wc = 0x779f; break;
        case 0x7949: wc = 0x7a6e; break;
        case 0x794a: wc = 0x7bfb; break;
        case 0x794b: wc = 0x7e39; break;
        case 0x794c: wc = 0x7ff2; break;
        case 0x794d: wc = 0x8198; break;
        case 0x794e: wc = 0x81d5; break;
        case 0x794f: wc = 0x8508; break;
        case 0x7950: wc = 0x85b8; break;
        case 0x7951: wc = 0x85e8; break;
        case 0x7952: wc = 0x87b5; break;
        case 0x7953: wc = 0x88f1; break;
        case 0x7954: wc = 0x893e; break;
        case 0x7955: wc = 0x8ad8; break;
        case 0x7956: wc = 0x8b24; break;
        case 0x7957: wc = 0x8d06; break;
        case 0x7958: wc = 0x91a5; break;
        case 0x7959: wc = 0x9336; break;
        case 0x795a: wc = 0x93e2; break;
        case 0x795b: wc = 0x9463; break;
        case 0x795c: wc = 0x9860; break;
        case 0x795d: wc = 0x98b7; break;
        case 0x795e: wc = 0x98c6; break;
        case 0x795f: wc = 0x9a6b; break;
        case 0x7960: wc = 0x9adf; break;
        case 0x7961: wc = 0x9b52; break;
        case 0x7962: wc = 0x9c3e; break;
        case 0x7963: wc = 0x6980; break;
        case 0x7964: wc = 0x7980; break;
        case 0x7965: wc = 0x5051; break;
        case 0x7966: wc = 0x50fc; break;
        case 0x7967: wc = 0x5838; break;
        case 0x7968: wc = 0x6a92; break;
        case 0x7969: wc = 0x6e22; break;
        case 0x796a: wc = 0x7043; break;
        case 0x796b: wc = 0x7326; break;
        case 0x796c: wc = 0x760b; break;
        case 0x796d: wc = 0x8634; break;
        case 0x796e: wc = 0x8c50; break;
        case 0x796f: wc = 0x9146; break;
        case 0x7970: wc = 0x98cc; break;
        case 0x7971: wc = 0x9eb7; break;
        case 0x7972: wc = 0x4f4a; break;
        case 0x7973: wc = 0x5e14; break;
        case 0x7974: wc = 0x67c0; break;
        case 0x7975: wc = 0x3c5f; break;
        case 0x7976: wc = 0x6c15; break;
        case 0x7977: wc = 0x72a4; break;
        case 0x7978: wc = 0x7fcd; break;
        case 0x7979: wc = 0xf070b; break;
        case 0x797a: wc = 0x85e3; break;
        case 0x797b: wc = 0x896c; break;
        case 0x797c: wc = 0x8a56; break;
        case 0x797d: wc = 0x8c8f; break;
        case 0x797e: wc = 0x9239; break;
        case 0x797f: wc = 0x9294; break;
        case 0x7980: wc = 0x9781; break;
        case 0x7981: wc = 0x9ab3; break;
        case 0x7982: wc = 0x9af2; break;
        case 0x7983: wc = 0x714f; break;
        case 0x7984: wc = 0x8177; break;
        case 0x7985: wc = 0x4f56; break;
        case 0x7986: wc = 0x5421; break;
        case 0x7987: wc = 0x5487; break;
        case 0x7988: wc = 0x55f6; break;
        case 0x7989: wc = 0x59bc; break;
        case 0x798a: wc = 0x5f43; break;
        case 0x798b: wc = 0x602d; break;
        case 0x798c: wc = 0x6993; break;
        case 0x798d: wc = 0x6eed; break;
        case 0x798e: wc = 0x6f77; break;
        case 0x798f: wc = 0x719a; break;
        case 0x7990: wc = 0x3eeb; break;
        case 0x7991: wc = 0x7bf3; break;
        case 0x7992: wc = 0x7e2a; break;
        case 0x7993: wc = 0x7f7c; break;
        case 0x7994: wc = 0x80c7; break;
        case 0x7995: wc = 0x84fd; break;
        case 0x7996: wc = 0x8890; break;
        case 0x7997: wc = 0x89f1; break;
        case 0x7998: wc = 0x8e55; break;
        case 0x7999: wc = 0x90b2; break;
        case 0x799a: wc = 0x924d; break;
        case 0x799b: wc = 0x93ce; break;
        case 0x799c: wc = 0x97b8; break;
        case 0x799d: wc = 0x97e0; break;
        case 0x799e: wc = 0x98f6; break;
        case 0x799f: wc = 0x99dc; break;
        case 0x79a0: wc = 0x9a46; break;
        case 0x79a1: wc = 0x9b85; break;
        case 0x79a2: wc = 0x9d04; break;
        case 0x79a3: wc = 0x9d6f; break;
        case 0x79a4: wc = 0x9ddd; break;
        case 0x79a5: wc = 0x506a; break;
        case 0x79a6: wc = 0x610a; break;
        case 0x79a7: wc = 0x9d14; break;
        case 0x79a8: wc = 0x5440; break;
        case 0x79a9: wc = 0x54e7; break;
        case 0x79aa: wc = 0x560f; break;
        case 0x79ab: wc = 0x55ec; break;
        case 0x79ac: wc = 0x5c88; break;
        case 0x79ad: wc = 0x7146; break;
        case 0x79ae: wc = 0x7615; break;
        case 0x79af: wc = 0x7771; break;
        case 0x79b0: wc = 0x78ac; break;
        case 0x79b1: wc = 0x25b86; break;
        case 0x79b2: wc = 0x7f45; break;
        case 0x79b3: wc = 0x8290; break;
        case 0x79b4: wc = 0x83cf; break;
        case 0x79b5: wc = 0x8578; break;
        case 0x79b6: wc = 0x8ad5; break;
        case 0x79b7: wc = 0x8c3a; break;
        case 0x79b8: wc = 0x8d6e; break;
        case 0x79b9: wc = 0x935c; break;
        case 0x79ba: wc = 0x93ec; break;
        case 0x79bb: wc = 0x959c; break;
        case 0x79bc: wc = 0x98ac; break;
        case 0x79bd: wc = 0x9a22; break;
        case 0x79be: wc = 0x55c0; break;
        case 0x79bf: wc = 0x55c3; break;
        case 0x79c0: wc = 0x56af; break;
        case 0x79c1: wc = 0x58c6; break;
        case 0x79c2: wc = 0x5da8; break;
        case 0x79c3: wc = 0x6db8; break;
        case 0x79c4: wc = 0x6fa9; break;
        case 0x79c5: wc = 0x3d85; break;
        case 0x79c6: wc = 0x72e2; break;
        case 0x79c7: wc = 0x7627; break;
        case 0x79c8: wc = 0x77d0; break;
        case 0x79c9: wc = 0x786e; break;
        case 0x79ca: wc = 0x7878; break;
        case 0x79cb: wc = 0x7fef; break;
        case 0x79cc: wc = 0x89f7; break;
        case 0x79cd: wc = 0x8b1e; break;
        case 0x79ce: wc = 0x8c88; break;
        case 0x79cf: wc = 0x90dd; break;
        case 0x79d0: wc = 0x9dfd; break;
        case 0x79d1: wc = 0x4f77; break;
        case 0x79d2: wc = 0x50e9; break;
        case 0x79d3: wc = 0x54fb; break;
        case 0x79d4: wc = 0x57be; break;
        case 0x79d5: wc = 0x5ae8; break;
        case 0x79d6: wc = 0x5afa; break;
        case 0x79d7: wc = 0x5afb; break;
        case 0x79d8: wc = 0x61aa; break;
        case 0x79d9: wc = 0x625e; break;
        case 0x79da: wc = 0x634d; break;
        case 0x79db: wc = 0x648a; break;
        case 0x79dc: wc = 0x66b5; break;
        case 0x79dd: wc = 0x6a4c; break;
        case 0x79de: wc = 0x6f96; break;
        case 0x79df: wc = 0x710a; break;
        case 0x79e0: wc = 0x71af; break;
        case 0x79e1: wc = 0x7233; break;
        case 0x79e2: wc = 0x72e0; break;
        case 0x79e3: wc = 0x7302; break;
        case 0x79e4: wc = 0x84d2; break;
        case 0x79e5: wc = 0x850a; break;
        case 0x79e6: wc = 0x86dd; break;
        case 0x79e7: wc = 0x8792; break;
        case 0x79e8: wc = 0x89b5; break;
        case 0x79e9: wc = 0x8c7b; break;
        case 0x79ea: wc = 0x9097; break;
        case 0x79eb: wc = 0x91ec; break;
        case 0x79ec: wc = 0x92b2; break;
        case 0x79ed: wc = 0x9588; break;
        case 0x79ee: wc = 0x96d7; break;
        case 0x79ef: wc = 0x99fb; break;
        case 0x79f0: wc = 0x9aad; break;
        case 0x79f1: wc = 0x29d5f; break;
        case 0x79f2: wc = 0x9dbe; break;
        case 0x79f3: wc = 0x9df4; break;
        case 0x79f4: wc = 0x9df3; break;
        case 0x79f5: wc = 0x9f3e; break;
        case 0x79f6: wc = 0x4e64; break;
        case 0x79f7: wc = 0x52bc; break;
        case 0x79f8: wc = 0x55d0; break;
        case 0x79f9: wc = 0x6433; break;
        case 0x79fa: wc = 0x778e; break;
        case 0x79fb: wc = 0x7848; break;
        case 0x79fc: wc = 0x78cd; break;
        case 0x79fd: wc = 0x7e16; break;
        case 0x79fe: wc = 0x821d; break;
        case 0x79ff: wc = 0x879b; break;
        case 0x7a00: wc = 0x938b; break;
        case 0x7a01: wc = 0x4c7b; break;
        case 0x7a02: wc = 0x9db7; break;
        case 0x7a03: wc = 0x5505; break;
        case 0x7a04: wc = 0x35f8; break;
        case 0x7a05: wc = 0x5705; break;
        case 0x7a06: wc = 0x58cf; break;
        case 0x7a07: wc = 0x5d21; break;
        case 0x7a08: wc = 0x6657; break;
        case 0x7a09: wc = 0x6892; break;
        case 0x7a0a: wc = 0x6d5b; break;
        case 0x7a0b: wc = 0x6dca; break;
        case 0x7a0c: wc = 0x7113; break;
        case 0x7a0d: wc = 0x7400; break;
        case 0x7a0e: wc = 0x7b68; break;
        case 0x7a0f: wc = 0x7cee; break;
        case 0x7a10: wc = 0x839f; break;
        case 0x7a11: wc = 0x83e1; break;
        case 0x7a12: wc = 0x44ff; break;
        case 0x7a13: wc = 0x872c; break;
        case 0x7a14: wc = 0x872d; break;
        case 0x7a15: wc = 0x8af4; break;
        case 0x7a16: wc = 0x8b40; break;
        case 0x7a17: wc = 0x8c3d; break;
        case 0x7a18: wc = 0x8c43; break;
        case 0x7a19: wc = 0x8c4f; break;
        case 0x7a1a: wc = 0x8f31; break;
        case 0x7a1b: wc = 0x8f5e; break;
        case 0x7a1c: wc = 0x918e; break;
        case 0x7a1d: wc = 0x92e1; break;
        case 0x7a1e: wc = 0x930e; break;
        case 0x7a1f: wc = 0x9837; break;
        case 0x7a20: wc = 0x9844; break;
        case 0x7a21: wc = 0x9851; break;
        case 0x7a22: wc = 0x9921; break;
        case 0x7a23: wc = 0x9b2b; break;
        case 0x7a24: wc = 0x9b7d; break;
        case 0x7a25: wc = 0x9c14; break;
        case 0x7a26: wc = 0x4f6e; break;
        case 0x7a27: wc = 0x530c; break;
        case 0x7a28: wc = 0x5477; break;
        case 0x7a29: wc = 0x55d1; break;
        case 0x7a2a: wc = 0x5cc6; break;
        case 0x7a2b: wc = 0x5cc7; break;
        case 0x7a2c: wc = 0x5ec5; break;
        case 0x7a2d: wc = 0x67d9; break;
        case 0x7a2e: wc = 0x69bc; break;
        case 0x7a2f: wc = 0x6b31; break;
        case 0x7a30: wc = 0x6e98; break;
        case 0x7a31: wc = 0x70da; break;
        case 0x7a32: wc = 0x7186; break;
        case 0x7a33: wc = 0x73e8; break;
        case 0x7a34: wc = 0x76cd; break;
        case 0x7a35: wc = 0x7bd5; break;
        case 0x7a36: wc = 0x8020; break;
        case 0x7a37: wc = 0x90c3; break;
        case 0x7a38: wc = 0x981c; break;
        case 0x7a39: wc = 0x9b7b; break;
        case 0x7a3a: wc = 0x9baf; break;
        case 0x7a3b: wc = 0x9d3f; break;
        case 0x7a3c: wc = 0x4d8e; break;
        case 0x7a3d: wc = 0x542d; break;
        case 0x7a3e: wc = 0x592f; break;
        case 0x7a3f: wc = 0x65bb; break;
        case 0x7a40: wc = 0x7095; break;
        case 0x7a41: wc = 0x7b10; break;
        case 0x7a42: wc = 0x7f3f; break;
        case 0x7a43: wc = 0x80ae; break;
        case 0x7a44: wc = 0x82c0; break;
        case 0x7a45: wc = 0x86a2; break;
        case 0x7a46: wc = 0x4615; break;
        case 0x7a47: wc = 0x8856; break;
        case 0x7a48: wc = 0x8ca5; break;
        case 0x7a49: wc = 0x8fd2; break;
        case 0x7a4a: wc = 0x909f; break;
        case 0x7a4b: wc = 0x9150; break;
        case 0x7a4c: wc = 0x9227; break;
        case 0x7a4d: wc = 0x958c; break;
        case 0x7a4e: wc = 0x95c0; break;
        case 0x7a4f: wc = 0x980f; break;
        case 0x7a50: wc = 0x9aaf; break;
        case 0x7a51: wc = 0x9b67; break;
        case 0x7a52: wc = 0x4f85; break;
        case 0x7a53: wc = 0x548d; break;
        case 0x7a54: wc = 0x55e8; break;
        case 0x7a55: wc = 0x598e; break;
        case 0x7a56: wc = 0x5cd0; break;
        case 0x7a57: wc = 0x5db0; break;
        case 0x7a58: wc = 0x5ee8; break;
        case 0x7a59: wc = 0x6140; break;
        case 0x7a5a: wc = 0x39e1; break;
        case 0x7a5b: wc = 0x6650; break;
        case 0x7a5c: wc = 0x6a9e; break;
        case 0x7a5d: wc = 0x6b2c; break;
        case 0x7a5e: wc = 0x6fa5; break;
        case 0x7a5f: wc = 0x736c; break;
        case 0x7a60: wc = 0x744e; break;
        case 0x7a61: wc = 0x75ce; break;
        case 0x7a62: wc = 0x3fac; break;
        case 0x7a63: wc = 0x7d6f; break;
        case 0x7a64: wc = 0x7e72; break;
        case 0x7a65: wc = 0x80f2; break;
        case 0x7a66: wc = 0x8344; break;
        case 0x7a67: wc = 0x85a4; break;
        case 0x7a68: wc = 0x85a2; break;
        case 0x7a69: wc = 0x89df; break;
        case 0x7a6a: wc = 0x8c65; break;
        case 0x7a6b: wc = 0x8c95; break;
        case 0x7a6c: wc = 0x8ccc; break;
        case 0x7a6d: wc = 0x8cc5; break;
        case 0x7a6e: wc = 0x8f06; break;
        case 0x7a6f: wc = 0x90cb; break;
        case 0x7a70: wc = 0x91a2; break;
        case 0x7a71: wc = 0x9654; break;
        case 0x7a72: wc = 0x97f0; break;
        case 0x7a73: wc = 0x9826; break;
        case 0x7a74: wc = 0x9900; break;
        case 0x7a75: wc = 0x99f4; break;
        case 0x7a76: wc = 0x9bad; break;
        case 0x7a77: wc = 0x9fa4; break;
        case 0x7a78: wc = 0x7fee; break;
        case 0x7a79: wc = 0x8988; break;
        case 0x7a7a: wc = 0x34d1; break;
        case 0x7a7b: wc = 0x5548; break;
        case 0x7a7c: wc = 0x5a5e; break;
        case 0x7a7d: wc = 0x2251b; break;
        case 0x7a7e: wc = 0x60bb; break;
        case 0x7a7f: wc = 0x23211; break;
        case 0x7a80: wc = 0x6d10; break;
        case 0x7a81: wc = 0x6dac; break;
        case 0x7a82: wc = 0x70c6; break;
        case 0x7a83: wc = 0x3f2c; break;
        case 0x7a84: wc = 0x7d4e; break;
        case 0x7a85: wc = 0x80fb; break;
        case 0x7a86: wc = 0x8395; break;
        case 0x7a87: wc = 0x88c4; break;
        case 0x7a88: wc = 0x4bd2; break;
        case 0x7a89: wc = 0x9d34; break;
        case 0x7a8a: wc = 0x59e0; break;
        case 0x7a8b: wc = 0x6651; break;
        case 0x7a8c: wc = 0x66cf; break;
        case 0x7a8d: wc = 0x81b7; break;
        case 0x7a8e: wc = 0x842b; break;
        case 0x7a8f: wc = 0x858c; break;
        case 0x7a90: wc = 0x45ae; break;
        case 0x7a91: wc = 0x8801; break;
        case 0x7a92: wc = 0x27b30; break;
        case 0x7a93: wc = 0x9e98; break;
        case 0x7a94: wc = 0x9ec1; break;
        case 0x7a95: wc = 0x380a; break;
        case 0x7a96: wc = 0x6829; break;
        case 0x7a97: wc = 0x6b54; break;
        case 0x7a98: wc = 0x9126; break;
        case 0x7a99: wc = 0x9a49; break;
        case 0x7a9a: wc = 0x9b56; break;
        case 0x7a9b: wc = 0x9b7c; break;
        case 0x7a9c: wc = 0x9c4b; break;
        case 0x7a9d: wc = 0x4eda; break;
        case 0x7a9e: wc = 0x4f61; break;
        case 0x7a9f: wc = 0x5dd8; break;
        case 0x7aa0: wc = 0x5e70; break;
        case 0x7aa1: wc = 0x641f; break;
        case 0x7aa2: wc = 0x6507; break;
        case 0x7aa3: wc = 0x7017; break;
        case 0x7aa4: wc = 0x283e5; break;
        case 0x7aa5: wc = 0x6eca; break;
        case 0x7aa6: wc = 0x880d; break;
        case 0x7aa7: wc = 0x59ed; break;
        case 0x7aa8: wc = 0x5dae; break;
        case 0x7aa9: wc = 0x5ffa; break;
        case 0x7aaa: wc = 0x6774; break;
        case 0x7aab: wc = 0x736b; break;
        case 0x7aac: wc = 0x7381; break;
        case 0x7aad: wc = 0x8b63; break;
        case 0x7aae: wc = 0x4f90; break;
        case 0x7aaf: wc = 0x9797; break;
        case 0x7ab0: wc = 0x5687; break;
        case 0x7ab1: wc = 0x5865; break;
        case 0x7ab2: wc = 0x5f08; break;
        case 0x7ab3: wc = 0x6347; break;
        case 0x7ab4: wc = 0x6b88; break;
        case 0x7ab5: wc = 0x6d2b; break;
        case 0x7ab6: wc = 0x7103; break;
        case 0x7ab7: wc = 0x7142; break;
        case 0x7ab8: wc = 0x7781; break;
        case 0x7ab9: wc = 0x8669; break;
        case 0x7aba: wc = 0x884b; break;
        case 0x7abb: wc = 0x89a4; break;
        case 0x7abc: wc = 0x8d69; break;
        case 0x7abd: wc = 0x49a7; break;
        case 0x7abe: wc = 0x9b29; break;
        case 0x7abf: wc = 0x4f2d; break;
        case 0x7ac0: wc = 0x5107; break;
        case 0x7ac1: wc = 0x545f; break;
        case 0x7ac2: wc = 0x54ef; break;
        case 0x7ac3: wc = 0x57cd; break;
        case 0x7ac4: wc = 0x59b6; break;
        case 0x7ac5: wc = 0x5a0a; break;
        case 0x7ac6: wc = 0x5a39; break;
        case 0x7ac7: wc = 0x5b1b; break;
        case 0x7ac8: wc = 0x60e4; break;
        case 0x7ac9: wc = 0x39cb; break;
        case 0x7aca: wc = 0x6621; break;
        case 0x7acb: wc = 0x688b; break;
        case 0x7acc: wc = 0x7066; break;
        case 0x7acd: wc = 0x7404; break;
        case 0x7ace: wc = 0x75c3; break;
        case 0x7acf: wc = 0x7734; break;
        case 0x7ad0: wc = 0x77ce; break;
        case 0x7ad1: wc = 0x7925; break;
        case 0x7ad2: wc = 0x7ffe; break;
        case 0x7ad3: wc = 0x80d8; break;
        case 0x7ad4: wc = 0x83a7; break;
        case 0x7ad5: wc = 0x8664; break;
        case 0x7ad6: wc = 0x86bf; break;
        case 0x7ad7: wc = 0x8706; break;
        case 0x7ad8: wc = 0x8809; break;
        case 0x7ad9: wc = 0x88a8; break;
        case 0x7ada: wc = 0x8ab8; break;
        case 0x7adb: wc = 0x8b5e; break;
        case 0x7adc: wc = 0x8b82; break;
        case 0x7add: wc = 0x476e; break;
        case 0x7ade: wc = 0x8d19; break;
        case 0x7adf: wc = 0x92d7; break;
        case 0x7ae0: wc = 0x92e7; break;
        case 0x7ae1: wc = 0x9466; break;
        case 0x7ae2: wc = 0x9799; break;
        case 0x7ae3: wc = 0x97c5; break;
        case 0x7ae4: wc = 0x99fd; break;
        case 0x7ae5: wc = 0x5437; break;
        case 0x7ae6: wc = 0x594a; break;
        case 0x7ae7: wc = 0x5a0e; break;
        case 0x7ae8: wc = 0x5ca4; break;
        case 0x7ae9: wc = 0x6cec; break;
        case 0x7aea: wc = 0x25134; break;
        case 0x7aeb: wc = 0x7d5c; break;
        case 0x7aec: wc = 0x7df3; break;
        case 0x7aed: wc = 0x8762; break;
        case 0x7aee: wc = 0x8d90; break;
        case 0x7aef: wc = 0x3fbe; break;
        case 0x7af0: wc = 0x99a6; break;
        case 0x7af1: wc = 0x51be; break;
        case 0x7af2: wc = 0x52a6; break;
        case 0x7af3: wc = 0x52f0; break;
        case 0x7af4: wc = 0x5327; break;
        case 0x7af5: wc = 0x53f6; break;
        case 0x7af6: wc = 0x55cb; break;
        case 0x7af7: wc = 0x57a5; break;
        case 0x7af8: wc = 0x57c9; break;
        case 0x7af9: wc = 0x38e3; break;
        case 0x7afa: wc = 0x604a; break;
        case 0x7afb: wc = 0x608f; break;
        case 0x7afc: wc = 0x611c; break;
        case 0x7afd: wc = 0x6136; break;
        case 0x7afe: wc = 0x62f9; break;
        case 0x7aff: wc = 0x689c; break;
        case 0x7b00: wc = 0x3d3a; break;
        case 0x7b01: wc = 0x7181; break;
        case 0x7b02: wc = 0x71f2; break;
        case 0x7b03: wc = 0x7864; break;
        case 0x7b04: wc = 0x796b; break;
        case 0x7b05: wc = 0x7b74; break;
        case 0x7b06: wc = 0x7bcb; break;
        case 0x7b07: wc = 0x261e6; break;
        case 0x7b08: wc = 0x8125; break;
        case 0x7b09: wc = 0x86fa; break;
        case 0x7b0a: wc = 0x46df; break;
        case 0x7b0b: wc = 0x4771; break;
        case 0x7b0c: wc = 0x927f; break;
        case 0x7b0d: wc = 0x967f; break;
        case 0x7b0e: wc = 0x4a10; break;
        case 0x7b0f: wc = 0x9788; break;
        case 0x7b10: wc = 0x9904; break;
        case 0x7b11: wc = 0x4f80; break;
        case 0x7b12: wc = 0x54fc; break;
        case 0x7b13: wc = 0x5910; break;
        case 0x7b14: wc = 0x5a19; break;
        case 0x7b15: wc = 0x64e4; break;
        case 0x7b16: wc = 0x657b; break;
        case 0x7b17: wc = 0x6d7b; break;
        case 0x7b18: wc = 0x6da5; break;
        case 0x7b19: wc = 0x7020; break;
        case 0x7b1a: wc = 0x784e; break;
        case 0x7b1b: wc = 0x811d; break;
        case 0x7b1c: wc = 0x85c0; break;
        case 0x7b1d: wc = 0x8605; break;
        case 0x7b1e: wc = 0x86f5; break;
        case 0x7b1f: wc = 0x892e; break;
        case 0x7b20: wc = 0x8a57; break;
        case 0x7b21: wc = 0x2897d; break;
        case 0x7b22: wc = 0x9276; break;
        case 0x7b23: wc = 0x92de; break;
        case 0x7b24: wc = 0x9658; break;
        case 0x7b25: wc = 0x5092; break;
        case 0x7b26: wc = 0x5338; break;
        case 0x7b27: wc = 0x5612; break;
        case 0x7b28: wc = 0x5d46; break;
        case 0x7b29: wc = 0x5d47; break;
        case 0x7b2a: wc = 0x5faf; break;
        case 0x7b2b: wc = 0x6193; break;
        case 0x7b2c: wc = 0x61f3; break;
        case 0x7b2d: wc = 0x2348e; break;
        case 0x7b2e: wc = 0x69bd; break;
        case 0x7b2f: wc = 0x69e5; break;
        case 0x7b30: wc = 0x6a5e; break;
        case 0x7b31: wc = 0x6a40; break;
        case 0x7b32: wc = 0x6f53; break;
        case 0x7b33: wc = 0x74a4; break;
        case 0x7b34: wc = 0x76fb; break;
        case 0x7b35: wc = 0x2654e; break;
        case 0x7b36: wc = 0x8787; break;
        case 0x7b37: wc = 0x87ea; break;
        case 0x7b38: wc = 0x8b11; break;
        case 0x7b39: wc = 0x8b53; break;
        case 0x7b3a: wc = 0x8b7f; break;
        case 0x7b3b: wc = 0x8c40; break;
        case 0x7b3c: wc = 0x8c6f; break;
        case 0x7b3d: wc = 0x93f8; break;
        case 0x7b3e: wc = 0x97b5; break;
        case 0x7b3f: wc = 0x9a31; break;
        case 0x7b40: wc = 0x9f37; break;
        case 0x7b41: wc = 0x512b; break;
        case 0x7b42: wc = 0x51b1; break;
        case 0x7b43: wc = 0x53f7; break;
        case 0x7b44: wc = 0x54e0; break;
        case 0x7b45: wc = 0x552c; break;
        case 0x7b46: wc = 0x55e5; break;
        case 0x7b47: wc = 0x5611; break;
        case 0x7b48: wc = 0x5637; break;
        case 0x7b49: wc = 0x568e; break;
        case 0x7b4a: wc = 0x5780; break;
        case 0x7b4b: wc = 0x36bc; break;
        case 0x7b4c: wc = 0x5aa9; break;
        case 0x7b4d: wc = 0x5aed; break;
        case 0x7b4e: wc = 0x5aee; break;
        case 0x7b4f: wc = 0x5e0d; break;
        case 0x7b50: wc = 0x6019; break;
        case 0x7b51: wc = 0x608e; break;
        case 0x7b52: wc = 0x623d; break;
        case 0x7b53: wc = 0x3a2d; break;
        case 0x7b54: wc = 0x6608; break;
        case 0x7b55: wc = 0x6626; break;
        case 0x7b56: wc = 0x6791; break;
        case 0x7b57: wc = 0x695c; break;
        case 0x7b58: wc = 0x69f4; break;
        case 0x7b59: wc = 0x6b51; break;
        case 0x7b5a: wc = 0x6c8d; break;
        case 0x7b5b: wc = 0x6cd8; break;
        case 0x7b5c: wc = 0x6ec8; break;
        case 0x7b5d: wc = 0x6ef9; break;
        case 0x7b5e: wc = 0x6eec; break;
        case 0x7b5f: wc = 0x70c0; break;
        case 0x7b60: wc = 0x7173; break;
        case 0x7b61: wc = 0x71a9; break;
        case 0x7b62: wc = 0x7292; break;
        case 0x7b63: wc = 0x7322; break;
        case 0x7b64: wc = 0x734b; break;
        case 0x7b65: wc = 0x769c; break;
        case 0x7b66: wc = 0x769e; break;
        case 0x7b67: wc = 0x76a5; break;
        case 0x7b68: wc = 0x7b8e; break;
        case 0x7b69: wc = 0x7bb6; break;
        case 0x7b6a: wc = 0x7c04; break;
        case 0x7b6b: wc = 0x7c47; break;
        case 0x7b6c: wc = 0x427f; break;
        case 0x7b6d: wc = 0x7d94; break;
        case 0x7b6e: wc = 0x8530; break;
        case 0x7b6f: wc = 0x8583; break;
        case 0x7b70: wc = 0x8585; break;
        case 0x7b71: wc = 0x85c3; break;
        case 0x7b72: wc = 0x864d; break;
        case 0x7b73: wc = 0x8656; break;
        case 0x7b74: wc = 0x8814; break;
        case 0x7b75: wc = 0x885a; break;
        case 0x7b76: wc = 0x8b3c; break;
        case 0x7b77: wc = 0x8b79; break;
        case 0x7b78: wc = 0x9117; break;
        case 0x7b79: wc = 0x9120; break;
        case 0x7b7a: wc = 0x9190; break;
        case 0x7b7b: wc = 0x933f; break;
        case 0x7b7c: wc = 0x49da; break;
        case 0x7b7d: wc = 0x96fd; break;
        case 0x7b7e: wc = 0x992c; break;
        case 0x7b7f: wc = 0x9b0d; break;
        case 0x7b80: wc = 0x9b71; break;
        case 0x7b81: wc = 0x9bf1; break;
        case 0x7b82: wc = 0x9c17; break;
        case 0x7b83: wc = 0x9c1d; break;
        case 0x7b84: wc = 0x9cf8; break;
        case 0x7b85: wc = 0x9d98; break;
        case 0x7b86: wc = 0x9da6; break;
        case 0x7b87: wc = 0x639d; break;
        case 0x7b88: wc = 0x8c70; break;
        case 0x7b89: wc = 0x9359; break;
        case 0x7b8a: wc = 0x9584; break;
        case 0x7b8b: wc = 0x5031; break;
        case 0x7b8c: wc = 0x60db; break;
        case 0x7b8d: wc = 0x60fd; break;
        case 0x7b8e: wc = 0x638d; break;
        case 0x7b8f: wc = 0x68d4; break;
        case 0x7b90: wc = 0x6b99; break;
        case 0x7b91: wc = 0x6dbd; break;
        case 0x7b92: wc = 0x6e63; break;
        case 0x7b93: wc = 0x6eb7; break;
        case 0x7b94: wc = 0x711c; break;
        case 0x7b95: wc = 0x7767; break;
        case 0x7b96: wc = 0x776f; break;
        case 0x7b97: wc = 0x7df7; break;
        case 0x7b98: wc = 0x7e49; break;
        case 0x7b99: wc = 0x89e8; break;
        case 0x7b9a: wc = 0x95bd; break;
        case 0x7b9b: wc = 0x991b; break;
        case 0x7b9c: wc = 0x4d37; break;
        case 0x7b9d: wc = 0x9f32; break;
        case 0x7b9e: wc = 0x5322; break;
        case 0x7b9f: wc = 0x5552; break;
        case 0x7ba0: wc = 0x553f; break;
        case 0x7ba1: wc = 0x56eb; break;
        case 0x7ba2: wc = 0x5be3; break;
        case 0x7ba3: wc = 0x39fe; break;
        case 0x7ba4: wc = 0x3e80; break;
        case 0x7ba5: wc = 0x82b4; break;
        case 0x7ba6: wc = 0x4edc; break;
        case 0x7ba7: wc = 0x55ca; break;
        case 0x7ba8: wc = 0x5985; break;
        case 0x7ba9: wc = 0x664e; break;
        case 0x7baa: wc = 0x6d1a; break;
        case 0x7bab: wc = 0x6e31; break;
        case 0x7bac: wc = 0x6f92; break;
        case 0x7bad: wc = 0x7074; break;
        case 0x7bae: wc = 0x7122; break;
        case 0x7baf: wc = 0x7bca; break;
        case 0x7bb0: wc = 0x82f0; break;
        case 0x7bb1: wc = 0x8452; break;
        case 0x7bb2: wc = 0x857b; break;
        case 0x7bb3: wc = 0x8c3c; break;
        case 0x7bb4: wc = 0x8c3e; break;
        case 0x7bb5: wc = 0x9277; break;
        case 0x7bb6: wc = 0x92be; break;
        case 0x7bb7: wc = 0x9710; break;
        case 0x7bb8: wc = 0x971f; break;
        case 0x7bb9: wc = 0x9b28; break;
        case 0x7bba: wc = 0x9b5f; break;
        case 0x7bbb: wc = 0x4f19; break;
        case 0x7bbc: wc = 0x4ff0; break;
        case 0x7bbd: wc = 0x5212; break;
        case 0x7bbe: wc = 0x3566; break;
        case 0x7bbf: wc = 0x5419; break;
        case 0x7bc0: wc = 0x5629; break;
        case 0x7bc1: wc = 0x58b7; break;
        case 0x7bc2: wc = 0x593b; break;
        case 0x7bc3: wc = 0x59c0; break;
        case 0x7bc4: wc = 0x5d0b; break;
        case 0x7bc5: wc = 0x398a; break;
        case 0x7bc6: wc = 0x6466; break;
        case 0x7bc7: wc = 0x64b6; break;
        case 0x7bc8: wc = 0x6779; break;
        case 0x7bc9: wc = 0x67a0; break;
        case 0x7bca: wc = 0x691b; break;
        case 0x7bcb: wc = 0x69ec; break;
        case 0x7bcc: wc = 0x6c8e; break;
        case 0x7bcd: wc = 0x6f95; break;
        case 0x7bce: wc = 0x706c; break;
        case 0x7bcf: wc = 0x748d; break;
        case 0x7bd0: wc = 0x756b; break;
        case 0x7bd1: wc = 0x76c9; break;
        case 0x7bd2: wc = 0x7af5; break;
        case 0x7bd3: wc = 0x7cc0; break;
        case 0x7bd4: wc = 0x8a38; break;
        case 0x7bd5: wc = 0x91eb; break;
        case 0x7bd6: wc = 0x9225; break;
        case 0x7bd7: wc = 0x924c; break;
        case 0x7bd8: wc = 0x9335; break;
        case 0x7bd9: wc = 0x93f5; break;
        case 0x7bda: wc = 0x97be; break;
        case 0x7bdb: wc = 0x9a4a; break;
        case 0x7bdc: wc = 0x9b64; break;
        case 0x7bdd: wc = 0x9c6f; break;
        case 0x7bde: wc = 0x9de8; break;
        case 0x7bdf: wc = 0x9fa2; break;
        case 0x7be0: wc = 0x5290; break;
        case 0x7be1: wc = 0x5f49; break;
        case 0x7be2: wc = 0x5f4f; break;
        case 0x7be3: wc = 0x5f60; break;
        case 0x7be4: wc = 0x6204; break;
        case 0x7be5: wc = 0x6509; break;
        case 0x7be6: wc = 0x66e4; break;
        case 0x7be7: wc = 0x6ab4; break;
        case 0x7be8: wc = 0x24430; break;
        case 0x7be9: wc = 0x7383; break;
        case 0x7bea: wc = 0x74c1; break;
        case 0x7beb: wc = 0x77cd; break;
        case 0x7bec: wc = 0x4084; break;
        case 0x7bed: wc = 0x77f1; break;
        case 0x7bee: wc = 0x792d; break;
        case 0x7bef: wc = 0x81d2; break;
        case 0x7bf0: wc = 0x81db; break;
        case 0x7bf1: wc = 0x8816; break;
        case 0x7bf2: wc = 0x8c9c; break;
        case 0x7bf3: wc = 0x944a; break;
        case 0x7bf4: wc = 0x96d8; break;
        case 0x7bf5: wc = 0x9729; break;
        case 0x7bf6: wc = 0x9743; break;
        case 0x7bf7: wc = 0x9e0c; break;
        case 0x7bf8: wc = 0x5702; break;
        case 0x7bf9: wc = 0x571c; break;
        case 0x7bfa: wc = 0x5bcf; break;
        case 0x7bfb: wc = 0x5bf0; break;
        case 0x7bfc: wc = 0x5cd8; break;
        case 0x7bfd: wc = 0x5f2e; break;
        case 0x7bfe: wc = 0x61c1; break;
        case 0x7bff: wc = 0x61fd; break;
        case 0x7c00: wc = 0x64d0; break;
        case 0x7c01: wc = 0x650c; break;
        case 0x7c02: wc = 0x69f5; break;
        case 0x7c03: wc = 0x6c4d; break;
        case 0x7c04: wc = 0x6f36; break;
        case 0x7c05: wc = 0x6fb4; break;
        case 0x7c06: wc = 0x72b1; break;
        case 0x7c07: wc = 0x737e; break;
        case 0x7c08: wc = 0x744d; break;
        case 0x7c09: wc = 0x74db; break;
        case 0x7c0a: wc = 0x764f; break;
        case 0x7c0b: wc = 0x7696; break;
        case 0x7c0c: wc = 0x7745; break;
        case 0x7c0d: wc = 0x7746; break;
        case 0x7c0e: wc = 0x7d59; break;
        case 0x7c0f: wc = 0x7d84; break;
        case 0x7c10: wc = 0x7e6f; break;
        case 0x7c11: wc = 0x7fa6; break;
        case 0x7c12: wc = 0x8092; break;
        case 0x7c13: wc = 0x8284; break;
        case 0x7c14: wc = 0x8341; break;
        case 0x7c15: wc = 0x841f; break;
        case 0x7c16: wc = 0x8b99; break;
        case 0x7c17: wc = 0x8c62; break;
        case 0x7c18: wc = 0x8c9b; break;
        case 0x7c19: wc = 0x8f58; break;
        case 0x7c1a: wc = 0x902d; break;
        case 0x7c1b: wc = 0x9144; break;
        case 0x7c1c: wc = 0x92ce; break;
        case 0x7c1d: wc = 0x9370; break;
        case 0x7c1e: wc = 0x9436; break;
        case 0x7c1f: wc = 0x95e4; break;
        case 0x7c20: wc = 0x9b1f; break;
        case 0x7c21: wc = 0x9bc7; break;
        case 0x7c22: wc = 0x9c5e; break;
        case 0x7c23: wc = 0x9d05; break;
        case 0x7c24: wc = 0x4d09; break;
        case 0x7c25: wc = 0x34c9; break;
        case 0x7c26: wc = 0x59e1; break;
        case 0x7c27: wc = 0x6ff6; break;
        case 0x7c28: wc = 0x7713; break;
        case 0x7c29: wc = 0x79ee; break;
        case 0x7c2a: wc = 0x843f; break;
        case 0x7c2b: wc = 0x86de; break;
        case 0x7c2c: wc = 0x8796; break;
        case 0x7c2d: wc = 0x8d8f; break;
        case 0x7c2e: wc = 0x505f; break;
        case 0x7c2f: wc = 0x51b5; break;
        case 0x7c30: wc = 0x55a4; break;
        case 0x7c31: wc = 0x582d; break;
        case 0x7c32: wc = 0x595b; break;
        case 0x7c33: wc = 0x5a93; break;
        case 0x7c34: wc = 0x5d32; break;
        case 0x7c35: wc = 0x5ddf; break;
        case 0x7c36: wc = 0x38f4; break;
        case 0x7c37: wc = 0x6033; break;
        case 0x7c38: wc = 0x63d8; break;
        case 0x7c39: wc = 0x697b; break;
        case 0x7c3a: wc = 0x3d82; break;
        case 0x7c3b: wc = 0x71bf; break;
        case 0x7c3c: wc = 0x735a; break;
        case 0x7c3d: wc = 0x745d; break;
        case 0x7c3e: wc = 0x3fae; break;
        case 0x7c3f: wc = 0x769d; break;
        case 0x7c40: wc = 0x76a9; break;
        case 0x7c41: wc = 0x402e; break;
        case 0x7c42: wc = 0x7a54; break;
        case 0x7c43: wc = 0x4319; break;
        case 0x7c44: wc = 0x8093; break;
        case 0x7c45: wc = 0x824e; break;
        case 0x7c46: wc = 0x845f; break;
        case 0x7c47: wc = 0x87e5; break;
        case 0x7c48: wc = 0x8841; break;
        case 0x7c49: wc = 0x8a64; break;
        case 0x7c4a: wc = 0x8afb; break;
        case 0x7c4b: wc = 0x8b0a; break;
        case 0x7c4c: wc = 0x8cba; break;
        case 0x7c4d: wc = 0x8daa; break;
        case 0x7c4e: wc = 0x8ee6; break;
        case 0x7c4f: wc = 0x93a4; break;
        case 0x7c50: wc = 0x97f9; break;
        case 0x7c51: wc = 0x992d; break;
        case 0x7c52: wc = 0x9a1c; break;
        case 0x7c53: wc = 0x9c09; break;
        case 0x7c54: wc = 0x9c51; break;
        case 0x7c55: wc = 0x9dec; break;
        case 0x7c56: wc = 0x5666; break;
        case 0x7c57: wc = 0x6f85; break;
        case 0x7c58: wc = 0x7ffd; break;
        case 0x7c59: wc = 0x4f6a; break;
        case 0x7c5a: wc = 0x528a; break;
        case 0x7c5b: wc = 0x54b4; break;
        case 0x7c5c: wc = 0x37a7; break;
        case 0x7c5d: wc = 0x605b; break;
        case 0x7c5e: wc = 0x62fb; break;
        case 0x7c5f: wc = 0x64d3; break;
        case 0x7c60: wc = 0x657c; break;
        case 0x7c61: wc = 0x6803; break;
        case 0x7c62: wc = 0x6af0; break;
        case 0x7c63: wc = 0x6d04; break;
        case 0x7c64: wc = 0x6d03; break;
        case 0x7c65: wc = 0x6e4f; break;
        case 0x7c66: wc = 0x6ed9; break;
        case 0x7c67: wc = 0x7024; break;
        case 0x7c68: wc = 0x70e3; break;
        case 0x7c69: wc = 0x70f8; break;
        case 0x7c6a: wc = 0x71f4; break;
        case 0x7c6b: wc = 0x74af; break;
        case 0x7c6c: wc = 0x75d0; break;
        case 0x7c6d: wc = 0x76d4; break;
        case 0x7c6e: wc = 0x77ba; break;
        case 0x7c6f: wc = 0x7858; break;
        case 0x7c70: wc = 0x79ac; break;
        case 0x7c71: wc = 0x8032; break;
        case 0x7c72: wc = 0x8588; break;
        case 0x7c73: wc = 0x85f1; break;
        case 0x7c74: wc = 0x8639; break;
        case 0x7c75: wc = 0x8698; break;
        case 0x7c76: wc = 0x86d5; break;
        case 0x7c77: wc = 0x879d; break;
        case 0x7c78: wc = 0x8931; break;
        case 0x7c79: wc = 0x8922; break;
        case 0x7c7a: wc = 0x8a7c; break;
        case 0x7c7b: wc = 0x8ad9; break;
        case 0x7c7c: wc = 0x470b; break;
        case 0x7c7d: wc = 0x8c57; break;
        case 0x7c7e: wc = 0x8ff4; break;
        case 0x7c7f: wc = 0x9136; break;
        case 0x7c80: wc = 0x942c; break;
        case 0x7c81: wc = 0x9767; break;
        case 0x7c82: wc = 0x982e; break;
        case 0x7c83: wc = 0x9bb0; break;
        case 0x7c84: wc = 0x9c60; break;
        case 0x7c85: wc = 0x5684; break;
        case 0x7c86: wc = 0x5aff; break;
        case 0x7c87: wc = 0x64ed; break;
        case 0x7c88: wc = 0x6e71; break;
        case 0x7c89: wc = 0x7809; break;
        case 0x7c8a: wc = 0x7aa2; break;
        case 0x7c8b: wc = 0x7e63; break;
        case 0x7c8c: wc = 0x8b0b; break;
        case 0x7c8d: wc = 0x8b22; break;
        case 0x7c8e: wc = 0x97c4; break;
        case 0x7c8f: wc = 0x9a1e; break;
        case 0x7c90: wc = 0x5430; break;
        case 0x7c91: wc = 0x569d; break;
        case 0x7c92: wc = 0x5f4b; break;
        case 0x7c93: wc = 0x63c8; break;
        case 0x7c94: wc = 0x6f8b; break;
        case 0x7c95: wc = 0x7ad1; break;
        case 0x7c96: wc = 0x8c39; break;
        case 0x7c97: wc = 0x8f37; break;
        case 0x7c98: wc = 0x921c; break;
        case 0x7c99: wc = 0x92d0; break;
        case 0x7c9a: wc = 0x9445; break;
        case 0x7c9b: wc = 0x9ecc; break;
        case 0x7c9c: wc = 0x509a; break;
        case 0x7c9d: wc = 0x52b9; break;
        case 0x7c9e: wc = 0x547a; break;
        case 0x7c9f: wc = 0x5635; break;
        case 0x7ca0: wc = 0x56c2; break;
        case 0x7ca1: wc = 0x5d24; break;
        case 0x7ca2: wc = 0x5ea8; break;
        case 0x7ca3: wc = 0x61a2; break;
        case 0x7ca4: wc = 0x64ac; break;
        case 0x7ca5: wc = 0x67b5; break;
        case 0x7ca6: wc = 0x6b4a; break;
        case 0x7ca7: wc = 0x6bbd; break;
        case 0x7ca8: wc = 0x6bca; break;
        case 0x7ca9: wc = 0x6d28; break;
        case 0x7caa: wc = 0x7071; break;
        case 0x7cab: wc = 0x7307; break;
        case 0x7cac: wc = 0x7362; break;
        case 0x7cad: wc = 0x75da; break;
        case 0x7cae: wc = 0x769b; break;
        case 0x7caf: wc = 0x7a58; break;
        case 0x7cb0: wc = 0x7a99; break;
        case 0x7cb1: wc = 0x7b05; break;
        case 0x7cb2: wc = 0x7b4a; break;
        case 0x7cb3: wc = 0x81ae; break;
        case 0x7cb4: wc = 0x85e0; break;
        case 0x7cb5: wc = 0x8648; break;
        case 0x7cb6: wc = 0x8653; break;
        case 0x7cb7: wc = 0x8a68; break;
        case 0x7cb8: wc = 0x90e9; break;
        case 0x7cb9: wc = 0x9864; break;
        case 0x7cba: wc = 0x991a; break;
        case 0x7cbb: wc = 0x9ac7; break;
        case 0x7cbc: wc = 0x9ad0; break;
        case 0x7cbd: wc = 0x9d1e; break;
        case 0x7cbe: wc = 0x5194; break;
        case 0x7cbf: wc = 0x55a3; break;
        case 0x7cc0: wc = 0x5795; break;
        case 0x7cc1: wc = 0x5820; break;
        case 0x7cc2: wc = 0x59c1; break;
        case 0x7cc3: wc = 0x6d09; break;
        case 0x7cc4: wc = 0x72bc; break;
        case 0x7cc5: wc = 0x7334; break;
        case 0x7cc6: wc = 0x73db; break;
        case 0x7cc7: wc = 0x24b08; break;
        case 0x7cc8: wc = 0x760a; break;
        case 0x7cc9: wc = 0x777a; break;
        case 0x7cca: wc = 0x7bcc; break;
        case 0x7ccb: wc = 0x25c4c; break;
        case 0x7ccc: wc = 0x7cc7; break;
        case 0x7ccd: wc = 0x7fed; break;
        case 0x7cce: wc = 0x8454; break;
        case 0x7ccf: wc = 0x8791; break;
        case 0x7cd0: wc = 0x8a61; break;
        case 0x7cd1: wc = 0x8a6c; break;
        case 0x7cd2: wc = 0x8b43; break;
        case 0x7cd3: wc = 0x90c8; break;
        case 0x7cd4: wc = 0x9107; break;
        case 0x7cd5: wc = 0x9157; break;
        case 0x7cd6: wc = 0x9297; break;
        case 0x7cd7: wc = 0x936d; break;
        case 0x7cd8: wc = 0x9931; break;
        case 0x7cd9: wc = 0x9aba; break;
        case 0x7cda: wc = 0x9b9c; break;
        case 0x7cdb: wc = 0x9bf8; break;
        case 0x7cdc: wc = 0x9c5f; break;
        case 0x7cdd: wc = 0x7ff5; break;
        case 0x7cde: wc = 0x9f41; break;
        case 0x7cdf: wc = 0x9f45; break;
        case 0x7ce0: wc = 0x5691; break;
        case 0x7ce1: wc = 0x66db; break;
        case 0x7ce2: wc = 0x720b; break;
        case 0x7ce3: wc = 0x736f; break;
        case 0x7ce4: wc = 0x77c4; break;
        case 0x7ce5: wc = 0x7e81; break;
        case 0x7ce6: wc = 0x26636; break;
        case 0x7ce7: wc = 0x81d0; break;
        case 0x7ce8: wc = 0x8477; break;
        case 0x7ce9: wc = 0x8512; break;
        case 0x7cea: wc = 0x91ba; break;
        case 0x7ceb: wc = 0x9442; break;
        case 0x7cec: wc = 0x6b3b; break;
        case 0x7ced: wc = 0x70fc; break;
        case 0x7cee: wc = 0x9b46; break;
        case 0x7cef: wc = 0x54ba; break;
        case 0x7cf0: wc = 0x56be; break;
        case 0x7cf1: wc = 0x5a97; break;
        case 0x7cf2: wc = 0x610b; break;
        case 0x7cf3: wc = 0x660d; break;
        case 0x7cf4: wc = 0x6645; break;
        case 0x7cf5: wc = 0x6966; break;
        case 0x7cf6: wc = 0x70dc; break;
        case 0x7cf7: wc = 0x72df; break;
        case 0x7cf8: wc = 0x79a4; break;
        case 0x7cf9: wc = 0x8432; break;
        case 0x7cfa: wc = 0x857f; break;
        case 0x7cfb: wc = 0x8610; break;
        case 0x7cfc: wc = 0x85fc; break;
        case 0x7cfd: wc = 0x8756; break;
        case 0x7cfe: wc = 0x8ae0; break;
        case 0x7cff: wc = 0x8afc; break;
        case 0x7d00: wc = 0x8c86; break;
        case 0x7d01: wc = 0x6a93; break;
        case 0x7d02: wc = 0x6bc7; break;
        case 0x7d03: wc = 0x71ec; break;
        case 0x7d04: wc = 0x8294; break;
        case 0x7d05: wc = 0x867a; break;
        case 0x7d06: wc = 0x8b6d; break;
        case 0x7d07: wc = 0x986a; break;
        case 0x7d08: wc = 0x992f; break;
        case 0x7d09: wc = 0x649d; break;
        case 0x7d0a: wc = 0x694e; break;
        case 0x7d0b: wc = 0x6932; break;
        case 0x7d0c: wc = 0x7988; break;
        case 0x7d0d: wc = 0x7fec; break;
        case 0x7d0e: wc = 0x49e6; break;
        case 0x7d0f: wc = 0x9c34; break;
        case 0x7d10: wc = 0x9064; break;
        case 0x7d11: wc = 0x5020; break;
        case 0x7d12: wc = 0x54bb; break;
        case 0x7d13: wc = 0x5b48; break;
        case 0x7d14: wc = 0x5ea5; break;
        case 0x7d15: wc = 0x6077; break;
        case 0x7d16: wc = 0x651c; break;
        case 0x7d17: wc = 0x74d7; break;
        case 0x7d18: wc = 0x772d; break;
        case 0x7d19: wc = 0x7762; break;
        case 0x7d1a: wc = 0x8320; break;
        case 0x7d1b: wc = 0x8633; break;
        case 0x7d1c: wc = 0x8835; break;
        case 0x7d1d: wc = 0x89ff; break;
        case 0x7d1e: wc = 0x8c85; break;
        case 0x7d1f: wc = 0x9145; break;
        case 0x7d20: wc = 0x929d; break;
        case 0x7d21: wc = 0x9474; break;
        case 0x7d22: wc = 0x9693; break;
        case 0x7d23: wc = 0x96b3; break;
        case 0x7d24: wc = 0x96df; break;
        case 0x7d25: wc = 0x9a68; break;
        case 0x7d26: wc = 0x9af9; break;
        case 0x7d27: wc = 0x9d42; break;
        case 0x7d28: wc = 0x6149; break;
        case 0x7d29: wc = 0x6410; break;
        case 0x7d2a: wc = 0x20e58; break;
        case 0x7d2b: wc = 0x391c; break;
        case 0x7d2c: wc = 0x61b0; break;
        case 0x7d2d: wc = 0x6f4f; break;
        case 0x7d2e: wc = 0x70c5; break;
        case 0x7d2f: wc = 0x735d; break;
        case 0x7d30: wc = 0x77b2; break;
        case 0x7d31: wc = 0x4389; break;
        case 0x7d32: wc = 0x9079; break;
        case 0x7d33: wc = 0x940d; break;
        case 0x7d34: wc = 0x9731; break;
        case 0x7d35: wc = 0x54c5; break;
        case 0x7d36: wc = 0x5ff7; break;
        case 0x7d37: wc = 0x605f; break;
        case 0x7d38: wc = 0x8a29; break;
        case 0x7d39: wc = 0x8a7e; break;
        case 0x7d3a: wc = 0x6f76; break;
        case 0x7d3b: wc = 0x91db; break;
        case 0x7d3c: wc = 0x4fd2; break;
        case 0x7d3d: wc = 0x59a1; break;
        case 0x7d3e: wc = 0x5e8d; break;
        case 0x7d3f: wc = 0x5f88; break;
        case 0x7d40: wc = 0x5ffb; break;
        case 0x7d41: wc = 0x6141; break;
        case 0x7d42: wc = 0x62eb; break;
        case 0x7d43: wc = 0x6380; break;
        case 0x7d44: wc = 0x712e; break;
        case 0x7d45: wc = 0x7161; break;
        case 0x7d46: wc = 0x76fa; break;
        case 0x7d47: wc = 0x266a3; break;
        case 0x7d48: wc = 0x812a; break;
        case 0x7d49: wc = 0x820b; break;
        case 0x7d4a: wc = 0x8a22; break;
        case 0x7d4b: wc = 0x91c1; break;
        case 0x7d4c: wc = 0x9341; break;
        case 0x7d4d: wc = 0x978e; break;
        case 0x7d4e: wc = 0x4ee1; break;
        case 0x7d4f: wc = 0x5fd4; break;
        case 0x7d50: wc = 0x6262; break;
        case 0x7d51: wc = 0x6c54; break;
        case 0x7d52: wc = 0x7599; break;
        case 0x7d53: wc = 0x7c7a; break;
        case 0x7d54: wc = 0x8090; break;
        case 0x7d55: wc = 0x867c; break;
        case 0x7d56: wc = 0x8db7; break;
        case 0x7d57: wc = 0x8fc4; break;
        case 0x7d58: wc = 0x91f3; break;
        case 0x7d59: wc = 0x4b23; break;
        case 0x7d5a: wc = 0x9ea7; break;
        case 0x7d5b: wc = 0x9f55; break;
        case 0x7d5c: wc = 0x5677; break;
        case 0x7d5d: wc = 0x5ede; break;
        case 0x7d5e: wc = 0x64b3; break;
        case 0x7d5f: wc = 0x946b; break;
        case 0x7d60: wc = 0x564f; break;
        case 0x7d61: wc = 0x6b59; break;
        case 0x7d62: wc = 0x3cb8; break;
        case 0x7d63: wc = 0x6f5d; break;
        case 0x7d64: wc = 0x95df; break;
        case 0x7d65: wc = 0x99ab; break;
        case 0x7d66: wc = 0x4fd9; break;
        case 0x7d67: wc = 0x546c; break;
        case 0x7d68: wc = 0x550f; break;
        case 0x7d69: wc = 0x563b; break;
        case 0x7d6a: wc = 0x56cd; break;
        case 0x7d6b: wc = 0x3746; break;
        case 0x7d6c: wc = 0x5c53; break;
        case 0x7d6d: wc = 0x5c6d; break;
        case 0x7d6e: wc = 0x5dc7; break;
        case 0x7d6f: wc = 0x5fe5; break;
        case 0x7d70: wc = 0x602c; break;
        case 0x7d71: wc = 0x6095; break;
        case 0x7d72: wc = 0x6232; break;
        case 0x7d73: wc = 0x66bf; break;
        case 0x7d74: wc = 0x6878; break;
        case 0x7d75: wc = 0x6a72; break;
        case 0x7d76: wc = 0x6b37; break;
        case 0x7d77: wc = 0x6b56; break;
        case 0x7d78: wc = 0x6d60; break;
        case 0x7d79: wc = 0x70ef; break;
        case 0x7d7a: wc = 0x7182; break;
        case 0x7d7b: wc = 0x71e8; break;
        case 0x7d7c: wc = 0x7214; break;
        case 0x7d7d: wc = 0x72a0; break;
        case 0x7d7e: wc = 0x72f6; break;
        case 0x7d7f: wc = 0x740b; break;
        case 0x7d80: wc = 0x774e; break;
        case 0x7d81: wc = 0x8383; break;
        case 0x7d82: wc = 0x4592; break;
        case 0x7d83: wc = 0x45b7; break;
        case 0x7d84: wc = 0x87e2; break;
        case 0x7d85: wc = 0x8a92; break;
        case 0x7d86: wc = 0x8b46; break;
        case 0x7d87: wc = 0x8c68; break;
        case 0x7d88: wc = 0x8c77; break;
        case 0x7d89: wc = 0x9919; break;
        case 0x7d8a: wc = 0x993c; break;
        case 0x7d8b: wc = 0x994e; break;
        case 0x7d8c: wc = 0x9c5a; break;
        case 0x7d8d: wc = 0x9d57; break;
        case 0x7d8e: wc = 0x9f42; break;
        case 0x7d8f: wc = 0x6044; break;
        case 0x7d90: wc = 0x64f7; break;
        case 0x7d91: wc = 0x6b2f; break;
        case 0x7d92: wc = 0x72b5; break;
        case 0x7d93: wc = 0x7e88; break;
        case 0x7d94: wc = 0x80b9; break;
        case 0x7d95: wc = 0x896d; break;
        case 0x7d96: wc = 0x9821; break;
        case 0x7d97: wc = 0x9ee0; break;
        case 0x7e00: wc = 0x7a50; break;
        case 0x7e01: wc = 0x9bf5; break;
        case 0x7e02: wc = 0x82c5; break;
        case 0x7e03: wc = 0x51b4; break;
        case 0x7e04: wc = 0x5b8d; break;
        case 0x7e05: wc = 0x5eff; break;
        case 0x7e06: wc = 0x7872; break;
        case 0x7e07: wc = 0x5f25; break;
        case 0x7e08: wc = 0x9453; break;
        case 0x7e09: wc = 0x5f0c; break;
        case 0x7e0a: wc = 0x5050; break;
        case 0x7e0b: wc = 0x5116; break;
        case 0x7e0c: wc = 0x5214; break;
        case 0x7e0d: wc = 0x5918; break;
        case 0x7e0e: wc = 0x54d8; break;
        case 0x7e0f: wc = 0x554c; break;
        case 0x7e10: wc = 0x555d; break;
        case 0x7e11: wc = 0x56ce; break;
        case 0x7e12: wc = 0x5726; break;
        case 0x7e13: wc = 0x5737; break;
        case 0x7e14: wc = 0x577f; break;
        case 0x7e15: wc = 0x57b3; break;
        case 0x7e16: wc = 0x57aa; break;
        case 0x7e17: wc = 0x57b0; break;
        case 0x7e18: wc = 0x57d6; break;
        case 0x7e19: wc = 0x57e3; break;
        case 0x7e1a: wc = 0x5870; break;
        case 0x7e1b: wc = 0x583d; break;
        case 0x7e1c: wc = 0x58b9; break;
        case 0x7e1d: wc = 0x58b8; break;
        case 0x7e1e: wc = 0x58d7; break;
        case 0x7e1f: wc = 0x58e5; break;
        case 0x7e20: wc = 0x5ad0; break;
        case 0x7e21: wc = 0x5bc9; break;
        case 0x7e22: wc = 0x4e62; break;
        case 0x7e23: wc = 0x5c76; break;
        case 0x7e24: wc = 0x599b; break;
        case 0x7e25: wc = 0x5cbb; break;
        case 0x7e26: wc = 0x5cbc; break;
        case 0x7e27: wc = 0x5cc5; break;
        case 0x7e28: wc = 0x5cfa; break;
        case 0x7e29: wc = 0x5d76; break;
        case 0x7e2a: wc = 0x5d90; break;
        case 0x7e2b: wc = 0x5f09; break;
        case 0x7e2c: wc = 0x5f41; break;
        case 0x7e2d: wc = 0x603a; break;
        case 0x7e2e: wc = 0x613d; break;
        case 0x7e2f: wc = 0x6327; break;
        case 0x7e30: wc = 0x63b5; break;
        case 0x7e31: wc = 0x64f6; break;
        case 0x7e32: wc = 0x6683; break;
        case 0x7e33: wc = 0x66bc; break;
        case 0x7e34: wc = 0x6741; break;
        case 0x7e35: wc = 0x6737; break;
        case 0x7e36: wc = 0x6764; break;
        case 0x7e37: wc = 0x68b9; break;
        case 0x7e38: wc = 0x68ba; break;
        case 0x7e39: wc = 0x6901; break;
        case 0x7e3a: wc = 0x6926; break;
        case 0x7e3b: wc = 0x6928; break;
        case 0x7e3c: wc = 0x6923; break;
        case 0x7e3d: wc = 0x6921; break;
        case 0x7e3e: wc = 0x697e; break;
        case 0x7e3f: wc = 0x6981; break;
        case 0x7e40: wc = 0x69dd; break;
        case 0x7e41: wc = 0x6a2e; break;
        case 0x7e42: wc = 0x6a36; break;
        case 0x7e43: wc = 0x6a78; break;
        case 0x7e44: wc = 0x6a22; break;
        case 0x7e45: wc = 0x6b1f; break;
        case 0x7e46: wc = 0x6c62; break;
        case 0x7e47: wc = 0x6e76; break;
        case 0x7e48: wc = 0x6e82; break;
        case 0x7e49: wc = 0x7195; break;
        case 0x7e4a: wc = 0x71d7; break;
        case 0x7e4b: wc = 0x72b9; break;
        case 0x7e4c: wc = 0x72b2; break;
        case 0x7e4d: wc = 0x732f; break;
        case 0x7e4e: wc = 0x7549; break;
        case 0x7e4f: wc = 0x7569; break;
        case 0x7e50: wc = 0x862f; break;
        case 0x7e51: wc = 0x7874; break;
        case 0x7e52: wc = 0x78b5; break;
        case 0x7e53: wc = 0x78c6; break;
        case 0x7e54: wc = 0x7907; break;
        case 0x7e55: wc = 0x7a43; break;
        case 0x7e56: wc = 0x9f9d; break;
        case 0x7e57: wc = 0x7b02; break;
        case 0x7e58: wc = 0x7b36; break;
        case 0x7e59: wc = 0x7c13; break;
        case 0x7e5a: wc = 0x7bf6; break;
        case 0x7e5b: wc = 0x7c58; break;
        case 0x7e5c: wc = 0x7c90; break;
        case 0x7e5d: wc = 0x7cad; break;
        case 0x7e5e: wc = 0x7cab; break;
        case 0x7e5f: wc = 0x7cc2; break;
        case 0x7e60: wc = 0x7cd8; break;
        case 0x7e61: wc = 0x7e67; break;
        case 0x7e62: wc = 0x7e5d; break;
        case 0x7e63: wc = 0x7e83; break;
        case 0x7e64: wc = 0x7dd5; break;
        case 0x7e65: wc = 0x7e7f; break;
        case 0x7e66: wc = 0x7e90; break;
        case 0x7e67: wc = 0x805f; break;
        case 0x7e68: wc = 0x5190; break;
        case 0x7e69: wc = 0x81a4; break;
        case 0x7e6a: wc = 0x825d; break;
        case 0x7e6b: wc = 0x83f7; break;
        case 0x7e6c: wc = 0x8422; break;
        case 0x7e6d: wc = 0x8420; break;
        case 0x7e6e: wc = 0x84d9; break;
        case 0x7e6f: wc = 0x8630; break;
        case 0x7e70: wc = 0x86ab; break;
        case 0x7e71: wc = 0x87d0; break;
        case 0x7e72: wc = 0x8842; break;
        case 0x7e73: wc = 0x88ae; break;
        case 0x7e74: wc = 0x88b0; break;
        case 0x7e75: wc = 0x8ada; break;
        case 0x7e76: wc = 0x621d; break;
        case 0x7e77: wc = 0x8ec5; break;
        case 0x7e78: wc = 0x8f4c; break;
        case 0x7e79: wc = 0x9027; break;
        case 0x7e7a: wc = 0x900e; break;
        case 0x7e7b: wc = 0x9056; break;
        case 0x7e7c: wc = 0x91d6; break;
        case 0x7e7d: wc = 0x91df; break;
        case 0x7e7e: wc = 0x91f6; break;
        case 0x7e7f: wc = 0x941a; break;
        case 0x7e80: wc = 0x9344; break;
        case 0x7e81: wc = 0x933b; break;
        case 0x7e82: wc = 0x93e5; break;
        case 0x7e83: wc = 0x9441; break;
        case 0x7e84: wc = 0x9587; break;
        case 0x7e85: wc = 0x9596; break;
        case 0x7e86: wc = 0x9666; break;
        case 0x7e87: wc = 0x894d; break;
        case 0x7e88: wc = 0x9764; break;
        case 0x7e89: wc = 0x9779; break;
        case 0x7e8a: wc = 0x9786; break;
        case 0x7e8b: wc = 0x9790; break;
        case 0x7e8c: wc = 0x97c8; break;
        case 0x7e8d: wc = 0x98aa; break;
        case 0x7e8e: wc = 0x99f2; break;
        case 0x7e8f: wc = 0x9ade; break;
        case 0x7e90: wc = 0x9b96; break;
        case 0x7e91: wc = 0x9bb4; break;
        case 0x7e92: wc = 0x9bcf; break;
        case 0x7e93: wc = 0x9bd1; break;
        case 0x7e94: wc = 0x9bd2; break;
        case 0x7e95: wc = 0x9be3; break;
        case 0x7e96: wc = 0x9c3a; break;
        case 0x7e97: wc = 0x9bf2; break;
        case 0x7e98: wc = 0x9c04; break;
        case 0x7e99: wc = 0x9c30; break;
        case 0x7e9a: wc = 0x9d48; break;
        case 0x7e9b: wc = 0x9d64; break;
        case 0x7e9c: wc = 0x9dab; break;
        case 0x7e9d: wc = 0x4e99; break;
        case 0x7e9e: wc = 0x4eb0; break;
        case 0x7e9f: wc = 0x4e17; break;
        case 0x7ea0: wc = 0x51d6; break;
        case 0x7ea1: wc = 0x53b0; break;
        case 0x7ea2: wc = 0x5f16; break;
        case 0x7ea3: wc = 0x66f5; break;
        case 0x7ea4: wc = 0x6e0a; break;
        case 0x7ea5: wc = 0x7188; break;
        case 0x7ea6: wc = 0x753c; break;
        case 0x7ea7: wc = 0x76b9; break;
        case 0x7ea8: wc = 0x89e7; break;
        case 0x7ea9: wc = 0x8d71; break;
        case 0x7eaa: wc = 0x98c3; break;
        case 0x7eab: wc = 0x76b7; break;
        case 0x7eac: wc = 0x591b; break;
        case 0x7ead: wc = 0x4e8a; break;
        case 0x7eae: wc = 0x9f98; break;
        case 0x7eaf: wc = 0x77e4; break;
        case 0x7eb0: wc = 0x7e5d; break;
        case 0x7eb1: wc = 0x833d; break;
        case 0x7eb2: wc = 0x83ed; break;
        case 0x7eb3: wc = 0x847c; break;
        case 0x7eb4: wc = 0x8716; break;
        case 0x7eb5: wc = 0x8a80; break;
        case 0x7eb6: wc = 0x8ae3; break;
        case 0x7eb7: wc = 0x9018; break;
        case 0x7eb8: wc = 0x90c0; break;
        case 0x7eb9: wc = 0x91e5; break;
        case 0x7eba: wc = 0x936b; break;
        case 0x7ebb: wc = 0x986e; break;
        case 0x7ebc: wc = 0x470c; break;
        case 0x7ebd: wc = 0x544b; break;
        case 0x7ebe: wc = 0x87ce; break;
        case 0x7ebf: wc = 0x80bc; break;
        case 0x7ec0: wc = 0x8ba0; break;
        case 0x7ec1: wc = 0x9963; break;
        case 0x7ec2: wc = 0x4e2c; break;
        case 0x7ec3: wc = 0x7e9f; break;
        case 0x7ec4: wc = 0x8080; break;
        case 0x7ec5: wc = 0x9485; break;
        case 0x7ec6: wc = 0x8012; break;
        case 0x7ec7: wc = 0x915e; break;
        case 0x7ec8: wc = 0x4eeb; break;
        case 0x7ec9: wc = 0x5def; break;
        case 0x7eca: wc = 0x57b4; break;
        case 0x7ecb: wc = 0x82ca; break;
        case 0x7ecc: wc = 0x82c4; break;
        case 0x7ecd: wc = 0x8488; break;
        case 0x7ece: wc = 0x64c0; break;
        case 0x7ecf: wc = 0x7519; break;
        case 0x7ed0: wc = 0x5416; break;
        case 0x7ed1: wc = 0x544b; break;
        case 0x7ed2: wc = 0x54cc; break;
        case 0x7ed3: wc = 0x54da; break;
        case 0x7ed4: wc = 0x5575; break;
        case 0x7ed5: wc = 0x55ea; break;
        case 0x7ed6: wc = 0x55f5; break;
        case 0x7ed7: wc = 0x567b; break;
        case 0x7ed8: wc = 0x5c9c; break;
        case 0x7ed9: wc = 0x72b8; break;
        case 0x7eda: wc = 0x7338; break;
        case 0x7edb: wc = 0x9993; break;
        case 0x7edc: wc = 0x5c1c; break;
        case 0x7edd: wc = 0x8f77; break;
        case 0x7ede: wc = 0x809f; break;
        case 0x7edf: wc = 0x80bd; break;
        case 0x7ee0: wc = 0x80e8; break;
        case 0x7ee1: wc = 0x80e9; break;
        case 0x7ee2: wc = 0x810e; break;
        case 0x7ee3: wc = 0x8112; break;
        case 0x7ee4: wc = 0x8159; break;
        case 0x7ee5: wc = 0x70bb; break;
        case 0x7ee6: wc = 0x717a; break;
        case 0x7ee7: wc = 0x781c; break;
        case 0x7ee8: wc = 0x7839; break;
        case 0x7ee9: wc = 0x94d8; break;
        case 0x7eea: wc = 0x94de; break;
        case 0x7eeb: wc = 0x952a; break;
        case 0x7eec: wc = 0x953f; break;
        case 0x7eed: wc = 0x9545; break;
        case 0x7eee: wc = 0x954e; break;
        case 0x7eef: wc = 0x9562; break;
        case 0x7ef0: wc = 0x9569; break;
        case 0x7ef1: wc = 0x9e4b; break;
        case 0x7ef2: wc = 0x7640; break;
        case 0x7ef3: wc = 0x7b7b; break;
        case 0x7ef4: wc = 0x7ba2; break;
        case 0x7ef5: wc = 0x823e; break;
        case 0x7ef6: wc = 0x7ccd; break;
        case 0x7ef7: wc = 0x9170; break;
        case 0x7ef8: wc = 0x8e3a; break;
        case 0x7ef9: wc = 0x9cba; break;
        case 0x7efa: wc = 0x9ccb; break;
        case 0x7efb: wc = 0x9b0f; break;
        case 0x7efc: wc = 0x9ee2; break;
        case 0x7efd: wc = 0x841c; break;
        case 0x7efe: wc = 0x6c46; break;
        case 0x7eff: wc = 0x9987; break;
        case 0x7f00: wc = 0x950d; break;
        case 0x7f01: wc = 0x94b7; break;
        case 0x7f02: wc = 0x813f; break;
        case 0xbc1f:	/* 르ᇝ */
            g_string_append_unichar (string, 0x1105);
            g_string_append_unichar (string, 0x1173);
            g_string_append_unichar (string, 0x11dd);
            return g_string_free (string, FALSE);
        case 0xd802:	/* 아ᇇ */
            g_string_append_unichar (string, 0x110b);
            g_string_append_unichar (string, 0x1161);
            g_string_append_unichar (string, 0x11c7);
            return g_string_free (string, FALSE);
        default:
            g_string_free (string, TRUE);
            return NULL;
    }
    g_string_append_unichar (string, wc);
    return g_string_free (string, FALSE);
}

#undef FILL
#undef NONE
