/*
FatRat download manager
http://fatrat.dolezel.info

Copyright (C) 2006-2008 Lubos Dolezel <lubos a dolezel.info>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
version 2 as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include "CzshareDownload.h"
#include "SettingsCzshare.h"
#include <fatrat/Settings.h>
#include <fatrat/RuntimeException.h>
#include <QRegExp>
#include <QBuffer>
#include <QHttp>
#include <QMenu>

const char* CZSHARE_REGEXP = "http://(www\\.)?czshare\\.com/\\d+/.+";
const char* CZSHARE_REGEXP_OLD = "http://(www\\.)?czshare\\.com/download_file\\.php\\?id=(\\d+)&file=(.+)";

CzshareDownload::CzshareDownload()
{
}

void CzshareDownload::globalInit()
{
	SettingsItem si;
	
	si.icon = DelayedIcon(":/czshare/czshare.png");
	si.title = tr("CZshare.com");
	si.lpfnCreate = SettingsCzshare::create;
	
	addSettingsPage(si);
}

void CzshareDownload::init(QString source, QString target)
{
	QRegExp re(CZSHARE_REGEXP), re_old(CZSHARE_REGEXP_OLD);
	
	if(re_old.exactMatch(source))
		source = QString("http://czshare.com/%1/%2").arg(re_old.cap(2)).arg(re_old.cap(3));
	else if(!re.exactMatch(source))
		throw RuntimeException(tr("This class doesn't support such URLs"));
	if(!target.startsWith("/"))
		throw RuntimeException(tr("The destination directory is invalid"));
	
	m_strOriginal = source;
	m_strTarget = target;
}

QString CzshareDownload::name() const
{
	if(m_strConverted.isEmpty())
		return tr("URL not converted yet");
	else
		return CurlDownload::name();
}

void CzshareDownload::load(const QDomNode& map)
{
	m_strOriginal = getXMLProperty(map, "czshare_original");
	m_strConverted = getXMLProperty(map, "czshare_converted");
	m_strTarget = getXMLProperty(map, "czshare_target");
	
	CurlDownload::load(map);
}

void CzshareDownload::save(QDomDocument& doc, QDomNode& map) const
{
	CurlDownload::save(doc, map);
	
	setXMLProperty(doc, map, "czshare_original", m_strOriginal);
	setXMLProperty(doc, map, "czshare_converted", m_strConverted);
	setXMLProperty(doc, map, "czshare_target", m_strTarget);
}

void CzshareDownload::setObject(QString newdir)
{
	if(!m_strConverted.isEmpty())
		CurlDownload::setObject(newdir);
	m_strTarget = newdir;
}

void CzshareDownload::changeActive(bool bActive)
{
	if(!m_strConverted.isEmpty())
	{
		CurlDownload::changeActive(bActive);
	}
	else if(bActive)
	{
		m_step = 0;
		if(m_strUser.isEmpty() || m_strPassword.isEmpty())
		{
			m_strUser = getSettingsValue("czshare/username").toString();
			m_strPassword = getSettingsValue("czshare/password").toString();
		}
		
		if(m_strUser.isEmpty() || m_strPassword.isEmpty())
		{
			m_strMessage = tr("Account information empty");
			setState(Failed);
		}
		else
		{
			if(QThread::currentThread() != QApplication::instance()->thread())
				moveToThread(QApplication::instance()->thread());
			QMetaObject::invokeMethod(this, "convertLink", Qt::QueuedConnection);
		}
	}
}

int CzshareDownload::acceptable(QString uri, bool)
{
	QRegExp re(CZSHARE_REGEXP), re_old(CZSHARE_REGEXP_OLD);
	if(re.exactMatch(uri) || re_old.exactMatch(uri))
		return 3;
	else
		return 0;
}

void CzshareDownload::convertLink()
{
	qDebug() << "convertLink()";
	m_http = new QHttp("czshare.com", 80);
	connect(m_http, SIGNAL(responseHeaderReceived(const QHttpResponseHeader&)), this, SLOT(authHeaders(const QHttpResponseHeader&)));
	m_http->get("/prihlasit.php");
	
	m_strSessID.clear();
}

void CzshareDownload::headersStep1(const QHttpResponseHeader& hdr)
{
	QList<QPair<QString, QString> > values = hdr.values();
	
	for(int i=0;i<values.size();i++)
	{
		if(values[i].first.compare("set-cookie", Qt::CaseInsensitive))
			continue;
		
		QRegExp re("PHPSESSID=([^;]+)");
		if(re.indexIn(values[i].second) >= 0)
		{
			m_strSessID = re.cap(1);
			break;
		}
	}
	
	if(m_strSessID.isEmpty())
	{
		m_strMessage = tr("Cannot get a session");
		setState(Failed);
		return;
	}
	
	qDebug() << "Session:" << m_strSessID;
	
	QHttpRequestHeader rhdr("POST", "/prihlasit.php");
	rhdr.addValue("Content-Type", "application/x-www-form-urlencoded");
	rhdr.addValue("Host", "czshare.com");
	rhdr.addValue("Cookie", "PHPSESSID="+m_strSessID);
	
	m_http->deleteLater();
	
	m_http = new QHttp("czshare.com", 80);
	connect(m_http, SIGNAL(responseHeaderReceived(const QHttpResponseHeader&)), this, SLOT(authHeaders(const QHttpResponseHeader&)));
	int id = m_http->request(rhdr, QByteArray("id=&file=&prihlasit=P%C5%99ihl%C3%A1sit&step=1&jmeno=") + m_strUser.toUtf8() + "&heslo=" + m_strPassword.toUtf8());
	qDebug() << "request id" << id;
}

void CzshareDownload::headersStep2(const QHttpResponseHeader& hdr)
{
	qDebug() << "CzshareDownload::headersStep2()";
	QList<QPair<QString, QString> > values = hdr.values();
	bool bLoginOK = false;
	
	for(int i=0;i<values.size();i++)
	{
		if(values[i].first.compare("location", Qt::CaseInsensitive) == 0)
		{
			if(values[i].second.startsWith("http://czshare.com/profi"))
				bLoginOK = true;
			break;
		}
	}
	
	m_http->deleteLater();
	
	if(!bLoginOK)
	{
		m_strMessage = tr("Login failed");
		setState(Failed);
		return;
	}
	else
		qDebug() << "Logged in";
	
	m_http = new QHttp("czshare.com", 80);
	m_buffer = new QBuffer;
	
	QHttpRequestHeader rhdr("POST", "/profi/graber.php");
	rhdr.addValue("Content-Type", "application/x-www-form-urlencoded");
	rhdr.addValue("Host", "czshare.com");
	rhdr.addValue("Cookie", "PHPSESSID="+m_strSessID);
	
	connect(m_http, SIGNAL(done(bool)), this, SLOT(requestDone(bool)));
	int id = m_http->request(rhdr, QByteArray("stahovat=stahovat&linky=") + QUrl::toPercentEncoding(m_strOriginal), m_buffer);
	
	qDebug() << "Request ID" << id;
}

void CzshareDownload::authHeaders(const QHttpResponseHeader& hdr)
{
	if(m_step == 0)
	{
		headersStep1(hdr);
		m_step++;
	}
	else
	{
		m_step++;
		headersStep2(hdr);
	}
}

void CzshareDownload::requestDone(bool error)
{
	m_http->deleteLater();
	m_buffer->deleteLater();
	
	if(error)
	{
		m_strMessage = tr("HTTP request failed");
		setState(Failed);
	}
	else
	{
		/*
		<td class="table2-black"><a href="http://www12.czshare.com/250879/ruP_pjC5PL7tpEmT_AM/VPhHr8Qg0B2L_7j_jJXogQeE/">rank.c</a></td><td><a href='smazat.php?kod=ruP_pjC5PL7tpEmT_AM'><u>smazat</u></a></td>
		*/
		
		//QRegExp re("http://www(\\d+)\\.czshare.com/profi.php\\?([^\"]+)");
		QRegExp re("<a href=\"(http://www\\d+.czshare.com/\\d+/[^\"]+/)\">");
		if(re.indexIn(m_buffer->data()) < 0)
		{
			m_strMessage = tr("Failed to convert the link");
			setState(Failed);
		}
		else
		{
			m_strConverted = re.cap(1);
			enterLogMessage(m_strConverted);
			
			qDebug() << m_strConverted << " -> " << m_strTarget;
			try
			{
				CurlDownload::init(m_strConverted, m_strTarget);
			}
			catch(const RuntimeException& e)
			{
				m_strMessage = e.what();
				setState(Failed);
				m_strConverted.clear();
				return;
			}
			
			if(isActive())
				changeActive(true);
		}
	}
}

WidgetHostChild* CzshareDownload::createOptionsWidget(QWidget* w)
{
	return new CzshareDownloadForm(w, this);
}

void CzshareDownload::fillContextMenu(QMenu& menu)
{
	QAction* a;
	
	a = menu.addAction(tr("Compute hash..."));
	connect(a, SIGNAL(triggered()), this, SLOT(computeHash()));
}

////////////////////////////////////////////////

CzshareDownloadForm::CzshareDownloadForm(QWidget* me, CzshareDownload* myobj)
        : m_myobj(myobj)
{
    setupUi(me);
}

void CzshareDownloadForm::load()
{
    lineUsername->setText(m_myobj->m_strUser);
    linePassword->setText(m_myobj->m_strPassword);
}

void CzshareDownloadForm::accepted()
{
    m_myobj->m_strUser = lineUsername->text();
    m_myobj->m_strPassword = linePassword->text();
}


