/* LIBGIMP - The GIMP Library
 * Copyright (C) 1995-1997 Peter Mattis and Spencer Kimball
 *
 * gimpmath.h
 *
 * This library is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#ifndef __GIMP_MATH_H__
#define __GIMP_MATH_H__


#include <math.h>

#ifdef HAVE_IEEEFP_H
#include <ieeefp.h>
#endif

#ifdef G_OS_WIN32
#include <float.h>
#endif

#define __GIMP_MATH_H_INSIDE__

/* A change for Glom:
#include <libgimpmath/gimpmathtypes.h>

#include <libgimpmath/gimpmatrix.h>
#include <libgimpmath/gimpmd5.h>
#include <libgimpmath/gimpvector.h>
*/

#undef __GIMP_MATH_H_INSIDE__


G_BEGIN_DECLS


/**
 * SECTION: gimpmath
 * @title: GimpMath
 * @short_description: Mathematical definitions and macros.
 *
 * Mathematical definitions and macros for use both by the GIMP
 * application and plug-ins. These macros should be used rather than
 * the ones from &lt;math.h&gt; for enhanced portability.
 **/


/**
 * RINT:
 * @x: the value to be rounded
 *
 * This macro rounds its argument @x to an integer value in floating
 * point format. Use RINT() instead of rint().
 **/
#ifdef HAVE_RINT
#define RINT(x) rint(x)
#else
#define RINT(x) floor ((x) + 0.5)
#endif

/**
 * ROUND:
 * @x: the value to be rounded.
 *
 * This macro rounds its argument @x to the nearest integer.
 **/
#define ROUND(x) ((int) ((x) + 0.5))

/**
 * SQR:
 * @x: the value to be squared.
 *
 * This macro squares its argument @x.
 **/
#define SQR(x) ((x) * (x))

/**
 * MAX255:
 * @a: the value to be limited.
 *
 * This macro limits it argument @a, an (0-511) int, to 255.
 **/
#define MAX255(a)  ((a) | (((a) & 256) - (((a) & 256) >> 8)))

/**
 * CLAMP0255:
 * @a: the value to be clamped.
 *
 * This macro clamps its argument @a, an int32-range int, between 0
 * and 255 inclusive.
 **/
#define CLAMP0255(a)  CLAMP(a,0,255)

/**
 * gimp_deg_to_rad:
 * @angle: the angle to be converted.
 *
 * This macro converts its argument @angle from degree to radian.
 **/
#define gimp_deg_to_rad(angle) ((angle) * (2.0 * G_PI) / 360.0)

/**
 * gimp_rad_to_deg:
 * @angle: the angle to be converted.
 *
 * This macro converts its argument @angle from radian to degree.
 **/
#define gimp_rad_to_deg(angle) ((angle) * 360.0 / (2.0 * G_PI))


G_END_DECLS

#endif /* __GIMP_MATH_H__ */
