------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-lobs.adb,v 1.21 2008/07/03 06:22:48 vagul Exp $

with OCI.Thread;
with System;

package body OCI.Thick.Lobs is

   use Ada.Strings.Unbounded;

   Type4Alloc : constant array (Lob_Type) of Ub4
     := (File  => OCI_DTYPE_FILE,
         Bin   => OCI_DTYPE_LOB,
         NChar => OCI_DTYPE_LOB,
         Char  => OCI_DTYPE_LOB);

   Type4Create_Temp : constant array (Lob_Type) of Ub1
     := (File  => 0,
         Bin   => OCI_TEMP_BLOB,
         NChar => OCI_TEMP_NCLOB,
         Char  => OCI_TEMP_CLOB);

   procedure Read
     (Loc    : in     Locator;
      Amount : in out Count;
      Offset : in     Positive_Count;
      Buffer : in     System.Address);

   procedure Write
     (Loc    : in out Locator;
      Offset : in     Positive_Count;
      Buffer : in     System.Address;
      Length : in     Ub4);

   ------------
   -- Create --
   ------------

   function Create
     (Connect : in Connection; Kind : in Lob_Type := Char) return Locator
   is
      Object : Locator;
   begin
      Object.Connect := Connect;
      Object.Kind    := Kind;
      Object.Handle
        := Alloc_Descriptor (Thread.Environment, Type4Alloc (Kind));
      return Object;
   end Create;

   ----------------------
   -- Create_Temporary --
   ----------------------

   function Create_Temporary
     (Connect : in Connection;
      Kind    : in Lob_Type := Char) return Locator
   is
      Loc : Locator := Create (Connect, Kind);
   begin
      Create_Temporary (Loc);
      return Loc;
   end Create_Temporary;

   procedure Create_Temporary
     (Loc   : in out Locator;
      Cache : in     Boolean := True)
   is
      Rc : SWord;
      To_OCI_Boolean : constant array (Boolean) of C.int :=
         (True  => Lib.TRUE,
          False => Lib.FALSE);
   begin
      Rc := OCILobCreateTemporary (Svchp  => OCISvcCtx (Handle (Loc.Connect)),
                                   Errhp   => Thread.Error,
                                   Locp    => OCILobLocator (Loc.Handle),
                                   Csid    => OCI_DEFAULT,
                                   Csfrm   => SQLCS_IMPLICIT,
                                   Lobtype => Type4Create_Temp (Loc.Kind),
                                   Cache    => To_OCI_Boolean (Cache),
                                   Duration => OCI_DURATION_SESSION);
      Check_Error (Rc);
   end Create_Temporary;

   procedure Create_Temporary
       (Stream  : in out Lob_Stream;
        Connect : in     Connection;
        Kind    : in     Lob_Type := Char) is
   begin
      Stream.Loc := Create (Connect, Kind);
      Create_Temporary (Stream.Loc);
   end Create_Temporary;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Object : in out Locator) is
      use type System.Address;
   begin
      if Object.Handle /= Empty_Handle then
         Check_Error
           (OCIDescriptorFree
              (Descp => Object.Handle, Dtype => Type4Alloc (Object.Kind)));
      end if;
   end Destroy;

   -----------------
   -- End_Of_File --
   -----------------

   function End_Of_File (Stream : in Lob_Stream) return Boolean is
   begin
      return Stream.Position > Length (Stream.Loc);
   end End_Of_File;

   --------------------
   -- Free_Temporary --
   --------------------

   procedure Free_Temporary (Loc : in out Locator) is
   begin
      Check_Error (OCILobFreeTemporary
                     (Svchp => OCISvcCtx (Handle (Loc.Connect)),
                      Errhp => Thread.Error,
                      Locp  => OCILobLocator (Loc.Handle)));
   end Free_Temporary;

   procedure Free_Temporary (Stream : in out Lob_Stream) is
   begin
      Free_Temporary (Stream.Loc);
   end Free_Temporary;

   ------------------
   -- Get_Lob_Type --
   ------------------

   function Get_Lob_Type (Loc : Locator) return Lob_Type is
   begin
      return Loc.Kind;
   end Get_Lob_Type;

   ----------------
   -- Is_Created --
   ----------------

   function Is_Created (Value : in Locator) return Boolean is
      use type OCIHandle;
   begin
      return Value.Handle /= Empty_Handle;
   end Is_Created;

   -------------
   -- Is_Init --
   -------------

   function Is_Init (Value : in Locator) return Boolean is
      Result : aliased C.int;
      Rc     : SWord;
      use type DVoid;
   begin
      if Value.Handle = Empty_Handle then
         return False;
      end if;

      Rc := OCILobLocatorIsInit
              (Envhp          => Thread.Environment,
               Errhp          => Thread.Error,
               Locp           => OCILobLocator (Value.Handle),
               Is_Initialized => Result'Unchecked_Access);

      Check_Error (Rc);

      return Boolean'Val (Result);
   end Is_Init;

   -------------
   -- Is_Open --
   -------------

   function Is_Open (Value : in Locator) return Boolean is
      Result : aliased C.int;
      Rc : SWord;
      use type DVoid;
   begin
      if Value.Handle = Empty_Handle then
         return False;
      end if;

      Rc := OCILobIsOpen (Svchp => OCISvcCtx (Handle (Value.Connect)),
                          Errhp => Thread.Error,
                          Locp  => OCILobLocator (Value.Handle),
                          Flag  => Result'Unchecked_Access);

      Check_Error (Rc);
      return Boolean'Val (Result);
   end Is_Open;

   ------------------
   -- Is_Temporary --
   ------------------

   function Is_Temporary (Value : in Locator) return Boolean is
      Result : aliased C.int;
   begin
      Check_Error (OCILobIsTemporary
                     (Envhp        => Thread.Environment,
                      Errhp        => Thread.Error,
                      Locp         => OCILobLocator (Value.Handle),
                      Is_Temporary => Result'Unchecked_Access));
      return Boolean'Val (Result);
   end Is_Temporary;

   ------------
   -- Length --
   ------------

   function Length (Loc : in Locator) return Count is
      Result : aliased Ub4;
   begin
      Check_Error (OCILobGetLength
                     (Svchp => OCISvcCtx (Handle (Loc.Connect)),
                      Errhp => Thread.Error,
                      Locp  => OCILobLocator (Loc.Handle),
                      Lenp  => Result'Unchecked_Access));

      return Count (Result);
   end Length;

   ----------
   -- Read --
   ----------

   procedure Read
     (Loc    : in     Locator;
      Amount : in out Count;
      Offset : in     Positive_Count;
      Buffer : in     System.Address)
   is
      Amt : aliased Ub4 := Ub4 (Amount);
      use System;
   begin
      Check_Error (OCILobRead
                     (Svchp  => OCISvcCtx (Handle (Loc.Connect)),
                      Errhp  => Thread.Error,
                      Locp   => OCILobLocator (Loc.Handle),
                      Amtp   => Amt'Unchecked_Access,
                      Offset => Ub4 (Offset),
                      Bufp   => Buffer,
                      Bufl   => Ub4 (Amount),
                      Ctxp   => Empty_Handle,
                      Cbfp   => Empty_Handle,
                      Csid   => 0));
      Amount := Count (Amt);
   end Read;

   procedure Read
     (Loc    :  in     Locator;
      Offset :  in     Positive_Count;
      Buffer :     out Raw;
      Last   :     out Raw_Offset)
   is
      Amt : Count := Buffer'Length;
   begin
      Read (Loc, Amt, Offset, Buffer'Address);
      Last := Buffer'Last - Raw_Offset (Buffer'Length - Amt);
   end Read;

   procedure Read
     (Loc    : in     Locator;
      Offset : in     Positive_Count;
      Buffer :    out String;
      Last   :    out Integer)
   is
      Amt : Count := Buffer'Length;
   begin
      Read (Loc, Amt, Offset, Buffer'Address);
      Last := Buffer'Last - Integer (Buffer'Length - Amt);
   end Read;

   procedure Read
     (Stream : in out Lob_Stream;
      Item   : out Stream_Element_Array;
      Last   : out Stream_Element_Offset) is
   begin
      Read
        (Stream.Loc,
         Offset => Stream.Position,
         Buffer => Item,
         Last   => Last);
      Stream.Position := Stream.Position + Count (Last - Item'First + 1);
   end Read;

   -----------
   -- Reset --
   -----------

   procedure Reset (Stream : in out Lob_Stream) is
   begin
      Stream.Position := Positive_Count'First;
   end Reset;

   procedure Set_Index
     (Stream : in out Lob_Stream; Offset : in Positive_Count) is
   begin
      Stream.Position := Offset;
   end Set_Index;

   ------------
   -- Stream --
   ------------

   procedure Stream (Stream : in out Lob_Stream'Class; Loc : in Locator) is
   begin
      Stream.Loc := Loc;
      Stream.Position   := Positive_Count'First;
   end Stream;

   ---------------
   -- To_String --
   ---------------

   function To_String (Loc : in Locator) return String is
      Len : Count := Length (Loc);
      Result : C.char_array (0 .. C.size_t (Len - 1));
   begin
      Read (Loc, Len, 1, Result'Address);
      return C.To_Ada (Result, False);
   end To_String;

   function To_String (Loc : in Lob_Stream) return String is
   begin
      return To_String (Loc.Loc);
   end To_String;

   -------------------------
   -- To_Unbounded_String --
   -------------------------

   function To_Unbounded_String (Loc : in Locator) return Unbounded_String is
      Position : Count := 1;
      Buffer   : String (1 .. 16#4000#);
      Result   : Unbounded_String;
      Last     : Natural;
   begin
      loop
         Read (Loc    => Loc,
               Offset => Position,
               Buffer => Buffer,
               Last   => Last);

         Append (Result, Buffer (1 .. Last));

         exit when Last < Buffer'Last;

         Position := Position + Count (Last);
      end loop;

      return Result;
   end To_Unbounded_String;

   ----------
   -- Trim --
   ----------

   procedure Trim (Loc : in out Locator; Length : in Count := 0) is
   begin
      Check_Error (OCILobTrim
                      (Svchp  => OCISvcCtx (Handle (Loc.Connect)),
                       Errhp  => Thread.Error,
                       Locp   => OCILobLocator (Loc.Handle),
                       Newlen => Ub4 (Length)));
   end Trim;

   -----------
   -- Write --
   -----------

   procedure Write
     (Loc    : in out Locator;
      Offset : in     Positive_Count;
      Buffer : in     System.Address;
      Length : in     Ub4)
   is
      use type Ub4;
      Amt : aliased Ub4 := Length;
   begin
      if Length > 0 then
         Check_Error (OCILobWrite
                        (Svchp  => OCISvcCtx (Handle (Loc.Connect)),
                         Errhp  => Thread.Error,
                         Locp   => OCILobLocator (Loc.Handle),
                         Amtp   => Amt'Unchecked_Access,
                         Offset => Ub4 (Offset),
                         Bufp   => Buffer,
                         Buflen => Length,
                         Piece  => OCI_ONE_PIECE,
                         Ctxp   => Empty_Handle,
                         Cbfp   => Empty_Handle,
                         Csid   => 0));
      end if;
   end Write;

   procedure Write
     (Loc    : in out Locator;
      Offset : in     Positive_Count;
      Buffer : in     Raw) is
   begin
      Write (Loc, Offset, Buffer'Address, Buffer'Length);
   end Write;

   procedure Write
     (Loc    : in out Locator;
      Offset : in     Positive_Count;
      Buffer : in     String) is
   begin
      Write (Loc, Offset, Buffer'Address, Buffer'Length);
   end Write;

   -----------
   -- Write --
   -----------

   procedure Write
     (Stream : in out Lob_Stream;
      Item   : in     Stream_Element_Array) is
   begin
      Write
        (Loc    => Stream.Loc,
         Buffer => Item,
         Offset => Stream.Position);

      Stream.Position := Stream.Position + Item'Length;
   end Write;

end OCI.Thick.Lobs;
