import os
import pipes
import subprocess
import sys
import tempfile

from twisted.internet.defer import inlineCallbacks, returnValue
from twisted.internet.threads import deferToThread

from juju.errors import JujuError

DATA_PATH = os.path.abspath(
    os.path.join(os.path.dirname(__file__), "data"))

CONTAINER_OPTIONS_DOC = """
The following options are expected.

JUJU_CONTAINER_NAME: Applied as the hostname of the machine.

JUJU_ORIGIN: Where to obtain the containers version of juju from.
             (ppa, distro or branch). When 'branch' JUJU_SOURCE should
             be set to the location of a bzr(1) accessible branch.

JUJU_PUBLIC_KEY: An SSH public key used by the ubuntu account for
             interaction with the container.

"""

DEVTMPFS_LINE = """devtmpfs dev devtmpfs mode=0755,nosuid 0 0"""

# Used to specify the name of the default LXC template used
# for container creation
DEFAULT_TEMPLATE = "ubuntu-cloud"


class LXCError(JujuError):
    """Indicates a low level error with an LXC container"""


def _cmd(args):
    p = subprocess.Popen(
        args, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
    stdout_data, _ = p.communicate()
    r = p.returncode
    if  r != 0:
        # read the stdout/err streams and show the user
        print >>sys.stderr, stdout_data
        raise LXCError(stdout_data)
    return (r, stdout_data)


# Wrapped lxc cli primitives
def _lxc_create(container_name,
                template,
                release,
                cloud_init_file=None,
                auth_key=None,
                release_stream=None):
    # the -- argument indicates the last parameters are passed
    # to the template and not handled by lxc-create
    args = ["sudo", "lxc-create",
            "-n", container_name,
            "-t", template,
            "--",
            "--debug",  # Debug erors / set -x
            "--hostid", container_name,
            "-r", release]
    if cloud_init_file:
        args.extend(("--userdata", cloud_init_file))
    if auth_key:
        args.extend(("-S", auth_key))
    if release_stream:
        args.extend(("-s", release_stream))
    return _cmd(args)


def _lxc_start(container_name, debug_log=None, console_log=None):
    args = ["sudo", "lxc-start", "--daemon", "-n", container_name]
    if console_log:
        args.extend(["-c", console_log])
    if debug_log:
        args.extend(["-l", "DEBUG", "-o", debug_log])
    return _cmd(args)


def _lxc_stop(container_name):
    _cmd(["sudo", "lxc-stop", "-n", container_name])


def _lxc_destroy(container_name):
    return _cmd(["sudo", "lxc-destroy", "-n", container_name])


def _lxc_ls():
    _, output = _cmd(["lxc-ls"])
    output = output.replace("\n", " ")
    return set([c for c in output.split(" ") if c])


def _lxc_wait(container_name, state="RUNNING"):
    """Wait for container to be in a given state RUNNING|STOPPED."""

    def wait(container_name):
        rc, _ = _cmd(["sudo", "lxc-wait",
                   "-n", container_name,
                   "-s", state])
        return rc == 0

    return deferToThread(wait, container_name)


def _lxc_clone(existing_container_name, new_container_name):
    return _cmd(["sudo", "lxc-clone", "-o", existing_container_name, "-n",
                 new_container_name])


def _customize_container(customize_script, container_root):
    if not os.path.isdir(container_root):
        raise LXCError("Expect container root directory: %s" %
                       container_root)

    # write the container scripts into the container
    fd, in_path = tempfile.mkstemp(prefix=os.path.basename(customize_script),
                                   dir=os.path.join(container_root, "tmp"))

    os.write(fd, open(customize_script, "r").read())
    os.close(fd)
    os.chmod(in_path, 0755)

    args = ["sudo", "chroot", container_root,
            os.path.join("/tmp", os.path.basename(in_path))]
    return _cmd(args)


def validate_path(pathname):
    if not os.access(pathname, os.R_OK):
        raise LXCError("Invalid or unreadable file: %s" % pathname)


@inlineCallbacks
def get_containers(prefix):
    """Return a dictionary of containers key names to runtime boolean value.

    :param prefix: Optionally specify a prefix that the container should
    match any returned containers.
    """
    _, output = yield deferToThread(_cmd, ["lxc-ls"])

    containers = {}
    for i in filter(None, output.split("\n")):
        if i in containers:
            containers[i] = True
        else:
            containers[i] = False

    if prefix:
        remove = [k for k in containers.keys() if not
                  k.startswith(prefix)]
        map(containers.pop, remove)

    returnValue(containers)


def ensure_devtmpfs_fstab(container_home):
    """ Workaround for bug in older LXC - We need to force mounting devtmpfs
        if it is not already in the rootfs, before starting.
    """
    rootfs = os.path.join(container_home, 'rootfs')
    devpts = os.path.join(rootfs, 'dev', 'pts')
    if not os.path.exists(devpts):
        fstab_path = os.path.join(container_home, 'fstab')
        if os.path.exists(fstab_path):
            with open(fstab_path) as fstab:
                for line in fstab:
                    if line.startswith('devtmpfs'):
                        # Line already there, we are done
                        return
            mode = 'a'
        else:
            mode = 'w'
        with open(fstab_path, mode) as fstab:
            print >>fstab, DEVTMPFS_LINE


class LXCContainer(object):
    def __init__(self,
                 container_name,
                 series,
                 cloud_init=None,
                 debug_log=None,
                 console_log=None,
                 release_stream="released"):
        """Create an LXCContainer

        :param container_name: should be unique within the system

        :param series: distro release series (oneiric, precise, etc)

        :param cloud_init: full string of cloud-init userdata

        See :data CONFIG_OPTIONS_DOC: explain how these values map
        into the container in more detail.
        """
        self.container_name = container_name
        self.debug_log = debug_log
        self.console_log = console_log
        self.cloud_init = cloud_init
        self.series = series
        self.release_stream = release_stream

    @property
    def container_home(self):
        return "/var/lib/lxc/%s" % self.container_name

    @property
    def rootfs(self):
        return "%s/rootfs/" % self.container_home

    def _p(self, path):
        if path[0] == "/":
            path = path[1:]
        return os.path.join(self.rootfs, path)

    def is_constructed(self):
        """Does the lxc image exist """
        return os.path.exists(self.rootfs)

    @inlineCallbacks
    def is_running(self):
        """Is the lxc image running."""
        state = yield get_containers(None)
        returnValue(state.get(self.container_name))

    def execute(self, args):
        if not isinstance(args, (list, tuple)):
            args = [args, ]

        args = ["sudo", "chroot", self.rootfs] + args
        return _cmd(args)

    def _create_wait(self):
        """Create the container synchronously."""
        if self.is_constructed():
            return

        with tempfile.NamedTemporaryFile() as fh:
            if self.cloud_init:
                fh.write(self.cloud_init.render())
                cloud_init_file = fh.name
            else:
                cloud_init_file = None
            fh.flush()
            _lxc_create(self.container_name,
                        template=DEFAULT_TEMPLATE,
                        cloud_init_file=cloud_init_file,
                        release=self.series)

        ensure_devtmpfs_fstab(self.container_home)

    @inlineCallbacks
    def create(self):
        # open the template file and create a new temp processed
        yield deferToThread(self._create_wait)

    def _clone_wait(self, container_name):
        """Return a cloned LXCContainer with a the new container name.

        This method is synchronous and will provision the new image
        blocking till done.
        """
        if not self.is_constructed():
            raise LXCError("Attempted to clone container "
                           "that hasn't been had create() called")

        container = LXCContainer(container_name,
                 series=self.series,
                 cloud_init=self.cloud_init,
                 debug_log=self.debug_log,
                 console_log=self.console_log,
                 release_stream=self.release_stream)

        if not container.is_constructed():
            _lxc_clone(self.container_name, container_name)
        return container

    def clone(self, container_name):
        return deferToThread(self._clone_wait, container_name)

    @inlineCallbacks
    def run(self):
        if not self.is_constructed():
            raise LXCError("Attempting to run a container that "
                           "hasn't been created or cloned.")

        yield deferToThread(
            _lxc_start, self.container_name,
            debug_log=self.debug_log, console_log=self.console_log)
        yield _lxc_wait(self.container_name, "RUNNING")

    @inlineCallbacks
    def stop(self):
        yield deferToThread(_lxc_stop, self.container_name)
        yield _lxc_wait(self.container_name, "STOPPED")

    @inlineCallbacks
    def destroy(self):
        yield self.stop()
        yield deferToThread(_lxc_destroy, self.container_name)
