module Context where

import AbsSyntax

import List
import Portability

-- Contexts containing names and type information

-- a context is represented as a map from the name to a list of all the
-- definitions of that name in all modules.

type NameList a = [(Name, a)]

newtype NameContext a = Ctxt (Dict Name (NameList a))

rootname (NS _ n) = n
rootname n = n

possibles (Ctxt ctxt) n = case dictLookup (rootname n) ctxt of
                             Nothing -> []
                             Just xs -> xs

update (Ctxt ctxt) n a = Ctxt (dictInsert n a ctxt)

empty :: NameContext a
empty = Ctxt dictEmpty

type Context = NameContext (Type, [FOpt])
type Types = NameContext TypeInfo

type Locals = NameList (Type, [FOpt])

type EContext = NameContext [Type]

getpos :: Name -> Locals -> Int
getpos n xs = getpos' n 0 (-1) xs
getpos' n _ last [] = last
getpos' n i last ((x,(t,_)):xs) | n==x = getpos' n (i+1) i xs
	  			| otherwise = getpos' n (i+1) last xs

-- ====== Primitives, relying on definition of Context ======

ctxtdump :: Context -> String
ctxtdump (Ctxt cs) = cd' (concat (dictElems cs))
   where cd' [] = ""
         cd' ((n,(ty,_)):xs) = showuser n ++ " :: " ++ show ty ++ "\n" ++ cd' xs

-- Lookup a name, bearing in mind namespaces.
-- Returns all possibilities, in the current module and others.
lookupname :: Name -> -- current module
              Name -> -- name to lookup (possibly decorated)
	      NameContext a -> [(Name,a)]
lookupname mod n gam = checkCurrent $ lu n (decorated n) 
                                           (possibles gam n) [] where
    lu n _ [] acc = acc
    lu n True ((x,a):xs) acc | n==x = lu n True xs ((x,a):acc)
			     | otherwise = lu n True xs acc
    lu n False ((x,a):xs) acc | nameMatches n x = lu n False xs ((x,a):acc)
			      | otherwise = lu n False xs acc

    -- in theory, returns names in the current module if they exist, or
    -- all names if the name is not in the current module. But in the
    -- presence of ad-hoc overloading, I don't think this makes sense.
    checkCurrent xs = xs {- cc xs xs
    cc [] xs = xs
    cc ((NS m n,x):xs) _ | m == mod = (NS m n,x):(cc xs [])
    cc (_:ys) xs = cc ys xs -}

    decorated (NS _ _) = True
    decorated _ = False
    nameMatches n (NS _ a) = nameMatches n a
    nameMatches n x = n == x

addToCtxt :: NameContext a -> Name -> a -> NameContext a
addToCtxt ctxt n d = 
   let gam = possibles ctxt n in
       update ctxt (rootname n) ((n,d):gam)

-- If name, type is already in the context, combine the options
mergeIntoCtxt :: Name -> Type -> [FOpt] -> Context -> (Context, [FOpt])
mergeIntoCtxt x xt fopt ctxt
    = let (newc, opts) = mctxt' [] (possibles ctxt x) in
          (update ctxt x newc, opts)
  where mctxt' acc [] = (acc,fopt)
	mctxt' acc ((n,(nt,nopt)):xs)
	     | x == n && xt == nt = let newopt = combine fopt nopt in
			                (((n,(nt,newopt)):(acc++xs)), newopt)
	     | otherwise = mctxt' ((n,(nt,nopt)):acc) xs

        combine xs ys = nub $! xs++ys

getNames :: NameContext a -> [Name]
getNames (Ctxt cs) = map fst (concat (dictElems cs))

-- ====== Derived functions, using above definitions only ======

defined :: Context -> Name -> Bool
defined ctxt n = (lookupname None n ctxt) /= []

addName :: Context -> Name -> Type -> [FOpt] -> Context
addName ctxt n ty fopts = addToCtxt ctxt n (ty,fopts)

addNames :: Context -> Locals -> Context
addNames ctxt [] = ctxt
addNames ctxt ((n,(t,f)):xs) = addNames (addName ctxt n t f) xs

-- Resolve a type synonym; does nothing on ambiguity or if nothing is found.
-- (The ambiguity thing is probably an error)

resolve :: Types -> Name -> Name
resolve ctxt n = case lookupname None n ctxt of
                    [(cn, ti)] -> cn
                    _ -> n