#ifndef LAUNCHTOOL_CFG_H
#define LAUNCHTOOL_CFG_H

#include <string>
#include <vector>
#include <set>
#include <sys/types.h>

// Singleton configuration data for this launchtool
class Cfg
{
public:
	// Identification tag
	std::string tag;

	// Visible tag, used for PID identification and logfile tagging
	std::string visible_tag;

	// Root directory
	std::string root_dir;

	// Starting directory (relative to root_dir, if root_dir has been specified)
	std::string start_dir;

	// Umask
	mode_t umask;

	// If true, wait_times behaves like the last value was repeated infinitely,
	// else stop execution after having passed the last value
	bool infinite_runs;

	// Wait times (in seconds) between a failure and a re-run
	std::vector<unsigned int> wait_times;

	// Minimum running time (in seconds) requested to return to the first
	// wait_times value
	unsigned int good_running_time;

	// Signal handling.  Signals can be:
	//  1) Caught by launchtool, that exits on response (default)
	//  2) Forwarded to the child
	//  3) Blocked
	std::vector<unsigned int> forwarded_signals;
	std::vector<unsigned int> blocked_signals;

	// If true, launchtool detaches from the terminal like a daemon
	bool detach;

	// If true, launchtool creates and manages a pidfile
	bool pidfile;

	// Directory where pidfiles will be created and managed
	std::string piddir;

	// Full commandline (passed to sh -c) of the command to be executed
	std::string command;

	// Resource limits to be enforced in the child
	int limit_cpu;
	int limit_file_size;
	int limit_data_memory;
	int limit_process_count;
	int limit_open_files;
	int limit_core_size;
	
	// User and group privileges for the child (unchanged if not specified)
	std::string user;
	std::string group;

	// Child environment configuration
	// All environment variables are passed to the child, except if
	// restrict_environment is true; in that case, only the variables listed in
	// allowed_env_vars will be passed to the child and the environment will be
	// passed empty and initialized by sh with its defaults.
	bool restrict_environment;
	std::set<std::string> allowed_env_vars;

	// Logging details
	// Format: method:<parms>
	// Methods: stdout, stderr, file, syslog
	// Parms:
	// 		stdout
	// 			none
	// 		stderr
	// 			none
	// 		file
	// 			filename
	// 		syslog
	// 			identity,facility,level
	// 			
	// 			identity: identity tag
	// 			facility: one of the syslog facilities described in syslog(3)
	// 			level: one of the syslog levels described in syslog(3)
	std::string log_launchtool_out;
	std::string log_launchtool_err;
	std::string log_child_stdout;
	std::string log_child_stderr;

	// When enabled, silently restart the child if it exists with exit status
	// `silent_restart_status', after a wait or `silent_restart_time' seconds.
	bool enable_silent_restart;
	int silent_restart_status;
	int silent_restart_time;

	// True if should execute verbosely
	bool verbose;

	// True if should enable debugging output
	bool debug;

	// True if should print an execution report after command completion
	bool stats;

	Cfg(const std::string& tag) throw ();

	void print() throw ();

protected:
	static Cfg* instance;

public:
	static void init(const std::string& tag) throw ()
	{
		if (instance)
			delete instance;
		instance = new Cfg(tag);
	}

	static Cfg& get()
	{
		return *instance;
	}

	static bool initialized()
	{
		return instance != 0;
	}
};

// vim:set ts=4 sw=4:
#endif
