/*
 * @(#)ModifiedMethodUTest.java
 *
 * Copyright (C) 2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.compiler;

import junit.framework.Test;
import junit.framework.TestSuite;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.BCELCreatorUtil;
import net.sourceforge.groboutils.codecoverage.v2.CCCreatorUtil;
import net.sourceforge.groboutils.junit.v1.SubTestTestCase;

import org.apache.bcel.classfile.JavaClass;
import org.apache.bcel.classfile.Method;
import org.apache.bcel.generic.MethodGen;


/**
 * Tests the ModifiedMethod class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:28 $
 * @since     January 13, 2003
 */
public class ModifiedMethodUTest extends SubTestTestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = ModifiedMethodUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public ModifiedMethodUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    public void testConstructor1() throws Exception
    {
        try
        {
            new ModifiedMethod( (short)0, 0, 0, null, null, null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException e)
        {
            // test exception
        }
    }
    
    
    public void testConstructor2() throws Exception
    {
        try
        {
            new ModifiedMethod( (short)0, 0, 0,
                createJavaClass( THIS_CLASS ), null, null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException e)
        {
            // test exception
        }
    }
    
    
    public void testConstructor3() throws Exception
    {
        try
        {
            new ModifiedMethod( (short)0, 0, 0,
                null, getMethod( createJavaClass( THIS_CLASS ), 0 ), null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException e)
        {
            // test exception
        }
    }
    
    
    public void testConstructor4() throws Exception
    {
        try
        {
            new ModifiedMethod( (short)0, 0, 0,
                null, null,
                createMethodGen( createJavaClass( THIS_CLASS ), 0 ) );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException e)
        {
            // test exception
        }
    }
    
    
    public void testConstructor5() throws Exception
    {
        JavaClass jc = createJavaClass( THIS_CLASS );
        try
        {
            new ModifiedMethod( (short)0, 0, 0,
                null, getMethod( jc, 0 ), createMethodGen( jc, 0 ) );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException e)
        {
            // test exception
        }
    }
    
    
    public void testConstructor6() throws Exception
    {
        JavaClass jc = createJavaClass( THIS_CLASS );
        try
        {
            new ModifiedMethod( (short)0, 0, 0,
                jc, null, createMethodGen( jc, 0 ) );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException e)
        {
            // test exception
        }
    }
    
    
    public void testConstructor7() throws Exception
    {
        JavaClass jc = createJavaClass( THIS_CLASS );
        try
        {
            new ModifiedMethod( (short)0, 0, 0,
                jc, getMethod( jc, 0 ), null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException e)
        {
            // test exception
        }
    }
    
    
    public void testGetMethodName1() throws Exception
    {
        JavaClass jc = createJavaClass( THIS_CLASS );
        Method m = getMethod( jc, 2 );
        MethodGen mg = createMethodGen( jc, m );
        ModifiedMethod mm = createModifiedMethod( jc, 2, m, mg );
        assertEquals(
            "Did not return correct method name.",
            m.getName() + m.getSignature(),
            mm.getMethodName() );
    }
    
    public void testGetInstructionList1() throws Exception
    {
        JavaClass jc = createJavaClass( THIS_CLASS );
        Method m = getMethod( jc, 4 );
        MethodGen mg = createMethodGen( jc, m );
        ModifiedMethod mm = createModifiedMethod( jc, 4, m, mg );
        ModifiedInstructionList mil = mm.getInstructionList();
        assertNotNull(
            "Returned null instruction list.",
            mil );
        assertEquals(
            "Instruction list size not correct.",
            mg.getInstructionList().size(),
            mil.getInstructionCount() );
    }
    
    
    public void testGetOriginalClass1() throws Exception
    {
        JavaClass jc = createJavaClass( THIS_CLASS );
        Method m = getMethod( jc, 4 );
        MethodGen mg = createMethodGen( jc, m );
        ModifiedMethod mm = createModifiedMethod( jc, 4, m, mg );
        assertSame(
            "Did not return the original java class.",
            jc,
            mm.getOriginalClass() );
    }
    
    
    public void testGetOriginalMethod1() throws Exception
    {
        JavaClass jc = createJavaClass( THIS_CLASS );
        Method m = getMethod( jc, 4 );
        MethodGen mg = createMethodGen( jc, m );
        ModifiedMethod mm = createModifiedMethod( jc, 4, m, mg );
        assertSame(
            "Did not return the original method.",
            m,
            mm.getOriginalMethod() );
    }
    
    
    public void testCanAddMarks1() throws Exception
    {
        ModifiedMethod mm = createModifiedMethod( THIS_CLASS, 14 );
        // every method in the test class should be able to be marked.
        assertTrue( mm.canAddMarks() );
    }
    
    
    public void testGetModifiedMethodGen1() throws Exception
    {
        JavaClass jc = createJavaClass( THIS_CLASS );
        Method m = getMethod( jc, 4 );
        MethodGen mg = createMethodGen( jc, m );
        ModifiedMethod mm = createModifiedMethod( jc, 4, m, mg );
        assertSame(
            "Did not return the original MethodGen.",
            mg,
            mm.getModifiedMethodGen() );
    }
    
    
    public void testGetNewMethod1() throws Exception
    {
        ModifiedMethod mm = createModifiedMethod( THIS_CLASS, 7 );
        try
        {
            mm.getNewMethod();
            fail( "Did not throw IllegalStateException." );
        }
        catch (IllegalStateException e)
        {
            // test exception
        }
    }
    
    
    public void testGetNewMethod2() throws Exception
    {
        JavaClass jc = createJavaClass( THIS_CLASS );
        Method m = getMethod( jc, 7 );
        MethodGen mg = createMethodGen( jc, m );
        ModifiedMethod mm = createModifiedMethod( jc, 7, m, mg );
        mm.close();
        Method m2 = mm.getNewMethod();
        assertNotSame(
            "Returned the original object.",
            m, m2 );
        assertEquals(
            "Method name was changed.",
            m.getName(), m2.getName() );
        assertEquals(
            "Method signature was changed.",
            m.getSignature(), m2.getSignature() );
    }
    
    
    public void testClose1() throws Exception
    {
        ModifiedMethod mm = createModifiedMethod( THIS_CLASS, 1 );
        mm.close();
        try
        {
            mm.close();
            fail( "Did not throw IllegalStateException." );
        }
        catch (IllegalStateException e)
        {
            // test exception
        }
    }
    
    
    public void testClose2() throws Exception
    {
        ModifiedMethod mm = createModifiedMethod( THIS_CLASS, 1 );
        mm.close();
        try
        {
            mm.getInstructionList();
            fail( "Did not throw IllegalStateException." );
        }
        catch (IllegalStateException e)
        {
            // test exception
        }
    }
    
    
    public void testClose3() throws Exception
    {
        ModifiedMethod mm = createModifiedMethod( THIS_CLASS, 1 );
        mm.close();
        try
        {
            mm.getModifiedMethodGen();
            fail( "Did not throw IllegalStateException." );
        }
        catch (IllegalStateException e)
        {
            // test exception
        }
    }
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    protected ModifiedMethod createModifiedMethod( Class c, int methodIndex )
            throws Exception
    {
        return CCCreatorUtil.createModifiedMethod( c, methodIndex );
    }
    
    protected ModifiedMethod createModifiedMethod( JavaClass jc,
            int methodIndex, Method m, MethodGen mg )
    {
        return CCCreatorUtil.createModifiedMethod( jc, methodIndex, m, mg );
    }
    
    
    protected JavaClass createJavaClass( Class c ) throws Exception
    {
        return BCELCreatorUtil.createJavaClass( c );
    }
    
    
    protected Method getMethod( JavaClass jc, int methodIndex )
    {
        return BCELCreatorUtil.getMethod( jc, methodIndex );
    }
    
    
    protected MethodGen createMethodGen( JavaClass jc, Method m )
    {
        return BCELCreatorUtil.createMethodGen( jc, m );
    }
    
    
    protected MethodGen createMethodGen( JavaClass jc, int methodIndex )
    {
        return BCELCreatorUtil.createMethodGen( jc, methodIndex );
    }
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

