/*
 *  @(#)Assembler.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.mbtf.v1.assembler;


import net.sourceforge.groboutils.mbtf.v1.IState;
import net.sourceforge.groboutils.mbtf.v1.IAction;
import net.sourceforge.groboutils.mbtf.v1.IValidate;
import net.sourceforge.groboutils.mbtf.v1.ITransition;

import net.sourceforge.groboutils.mbtf.v1.engine.StateImpl;
import net.sourceforge.groboutils.mbtf.v1.engine.TransitionImpl;

import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;


/**
 * Helper class which transforms the AsmblState and AsmblTransition instances
 * into a coherent IState machine.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:52:25 $
 * @since      June 13, 2002
 */
public class Assembler
{
    private Hashtable nameToIState;
    private Vector startStates;
    private Vector finalStates;

    
    /**
     * 
     */
    public Assembler( AsmblTransitionSet trans, AsmblStateSet states )
    {
        if (trans == null || states == null)
        {
            throw new IllegalArgumentException("no null args");
        }
        parseStates( trans, states );
    }
    
    
    /**
     * Returns a list of all States passed in to the constructor.
     */
    public IState[] getStates()
    {
        IState s[] = new IState[ this.nameToIState.size() ];
        Enumeration enum = this.nameToIState.elements();
        for (int i = 0; enum.hasMoreElements(); ++i)
        {
            s[i] = (IState)enum.nextElement();
        }
        return s;
    }
    
    
    /**
     * 
     */
    public IState[] getStartStates()
    {
        IState s[] = new IState[ this.startStates.size() ];
        this.startStates.copyInto( s );
        return s;
    }
    
    
    /**
     * 
     */
    public IState[] getFinalStates()
    {
        IState s[] = new IState[ this.finalStates.size() ];
        this.finalStates.copyInto( s );
        return s;
    }
    
    
    //-------------------------------------------------------------------------
    
    
    protected void parseStates( AsmblTransitionSet trans,
            AsmblStateSet stateSet )
    {
        populateIStates( trans, stateSet.getStates() );
        
        // Now load all the transitions destination states correctly
        trans.setDestinationStates( this.nameToIState );
    }
    
    
    protected void populateIStates( AsmblTransitionSet trans, AsmblState[] as )
    {
        this.nameToIState = new Hashtable();
        this.startStates = new Vector();
        this.finalStates = new Vector();
        
        for (int i = as.length; --i >= 0;)
        {
            addAsmblState( as[i], as[i].createState( trans ) );
        }
    }
    
    
    protected void addAsmblState( AsmblState as, IState s )
    {
        this.nameToIState.put( s.getName(), s );
        
        if (as.isStartState())
        {
            this.startStates.addElement( s );
        }
        
        if (as.isFinalState())
        {
            this.finalStates.addElement( s );
        }
    }
    
    
    //-------------------------------------------------------------------------
    
    
    protected IState getState( String name )
    {
        return (IState)this.nameToIState.get( name );
    }
    
    
}

