// This file is part of libigl, a simple c++ geometry processing library.
// 
// Copyright (C) 2020 Alec Jacobson <alecjacobson@gmail.com>
// 
// This Source Code Form is subject to the terms of the Mozilla Public License 
// v. 2.0. If a copy of the MPL was not distributed with this file, You can 
// obtain one at http://mozilla.org/MPL/2.0/.
#include "exploded_view.h"
#include "barycenter.h"
#include "volume.h"

template <
  typename DerivedV,
  typename DerivedT,
  typename DerivedEV,
  typename DerivedEF,
  typename DerivedI,
  typename DerivedJ>
IGL_INLINE void igl::exploded_view(
  const Eigen::MatrixBase<DerivedV> & V,
  const Eigen::MatrixBase<DerivedT> & T,
  const typename DerivedV::Scalar s,
  const typename DerivedV::Scalar t,
  Eigen::PlainObjectBase<DerivedEV> & EV,
  Eigen::PlainObjectBase<DerivedEF> & EF,
  Eigen::PlainObjectBase<DerivedI> & I,
  Eigen::PlainObjectBase<DerivedJ> & J)
{
  assert(T.cols() == 4 && "T should be a tet mesh");
  EV.resize(4*T.rows(),3);
  EF.resize(4*T.rows(),3);
  I.resize(EV.rows());
  J.resize(EF.rows());
  Eigen::MatrixXd BC;
  igl::barycenter(V,T,BC);
  Eigen::VectorXd vol;
  igl::volume(V,T,vol);
  const Eigen::RowVectorXd c = vol.transpose()*BC/vol.array().sum();
  for(int i = 0;i<T.rows();i++)
  {
    // scale the barycenters outward
    const auto tbc = ((BC.row(i)-c)*t+c).eval();
    for(int j = 0;j<4;j++)
    {
      // vector to barycenter
      const auto v = V.row(T(i,j))-BC.row(i);
      // scale vector and add to scaled barycenter
      EV.row(i*4+j) = v*s+tbc;
      I(i*4+j) = T(i,j);
      J(i*4+j) = i;
      if(j%2)
      {
        EF(i*4+j,0) = i*4+((j+3)%4);
        EF(i*4+j,1) = i*4+((j+2)%4);
        EF(i*4+j,2) = i*4+((j+1)%4);
      }else
      {
        EF(i*4+j,0) = i*4+((j+1)%4);
        EF(i*4+j,1) = i*4+((j+2)%4);
        EF(i*4+j,2) = i*4+((j+3)%4);
      }
    }
  }
}

#ifdef IGL_STATIC_LIBRARY
// Explicit template instantiation
// generated by autoexplicit.sh
template void igl::exploded_view<Eigen::Matrix<double, -1, -1, 0, -1, -1>, Eigen::Matrix<int, -1, -1, 0, -1, -1>, Eigen::Matrix<double, -1, -1, 0, -1, -1>, Eigen::Matrix<int, -1, -1, 0, -1, -1>, Eigen::Matrix<int, -1, 1, 0, -1, 1>, Eigen::Matrix<int, -1, 1, 0, -1, 1> >(Eigen::MatrixBase<Eigen::Matrix<double, -1, -1, 0, -1, -1> > const&, Eigen::MatrixBase<Eigen::Matrix<int, -1, -1, 0, -1, -1> > const&, Eigen::Matrix<double, -1, -1, 0, -1, -1>::Scalar, Eigen::Matrix<double, -1, -1, 0, -1, -1>::Scalar, Eigen::PlainObjectBase<Eigen::Matrix<double, -1, -1, 0, -1, -1> >&, Eigen::PlainObjectBase<Eigen::Matrix<int, -1, -1, 0, -1, -1> >&, Eigen::PlainObjectBase<Eigen::Matrix<int, -1, 1, 0, -1, 1> >&, Eigen::PlainObjectBase<Eigen::Matrix<int, -1, 1, 0, -1, 1> >&);
#endif
