/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "XPalette.h"

#ifndef ColourEditor_H
#include "ColourEditor.h"
#endif

#ifndef EditorFactory_H
#include "EditorFactory.h"
#endif

#ifndef Request_H
#include "Request.h"
#endif

#ifndef ColorName_H
#include "ColorName.h"
#endif


#include <Xm/Scale.h>

#ifndef RGBMaker_H
#include "RGBMaker.h"
#endif

#ifndef HSLMaker_H
#include "HSLMaker.h"
#endif

#ifndef IconClass_H
#include "IconClass.h"
#endif


#include <Tab.h>

//------------------------------------------------------------------

ColourEditor::ColourEditor(const IconClass& name,const string& kind):
	XEditor<UIColourEditor_c>(name,kind),
	rainbow_r_(rr_),
	rainbow_g_(gg_),
	rainbow_b_(bb_),
	rainbow_h_(hh_),
	rainbow_s_(ss_),
	rainbow_l_(ll_)
{
	XPalette::scan(*this);
}

ColourEditor::~ColourEditor()
{
}

void ColourEditor::apply()
{
	const char* mode = XtName(TabGetCurrent(tab_));

	if(strcmp(mode,"MAGICS") == 0)
		if(!XPalette::convert(rgb_,name_))
			mode = "RGB";

	Request r("COLOUR");

	
	r("MODE")       = mode;

	r("RED")        = rgb_.red;
	r("GREEN")      = rgb_.green;
	r("BLUE")       = rgb_.blue;

	r("HUE")        = hsl_.hue;
	r("SATURATION") = hsl_.saturation;
	r("LIGHTNESS")  = hsl_.lightness;

	r("NAME")       = name_.c_str();


	current_->request(r);
}

void ColourEditor::reset()
{
	replace(current_);

	XtVaSetValues(
		old_,XmNbackground,
		XPalette::pixel(rgb_),
		0);
}

void ColourEditor::close()
{
}

void ColourEditor::merge(IconObject* o)
{
	replace(o);
}

void ColourEditor::replace(IconObject* o)
{
	if(!o->iconClass().isSubClassOf(class_))
		return;

	Request r  = o->request();
	const char* mode = r("MODE");
	if(mode == 0) mode = "RGB";


	rgb_.red        = r("RED");
	rgb_.green      = r("GREEN");
	rgb_.blue       = r("BLUE");

	hsl_.hue        = r("HUE");
	hsl_.saturation = r("SATURATION");
	hsl_.lightness  = r("LIGHTNESS");

	const char* name = r("NAME");
	name_       = name ? name : "";

	switch(*mode)
	{
		case 'R':
			XPalette::convert(rgb_,hsl_);
			break;

		case 'H':
			XPalette::convert(hsl_,rgb_);
			break;

		case 'M':
			XPalette::convert(name_,rgb_);
			XPalette::convert(rgb_,hsl_);
			break;
	}

	Widget w = XtNameToWidget(tab_,mode);
	if(w) TabSetCurrent(tab_,w,False);

	refresh();
}

void ColourEditor::hslCB(Widget,XtPointer)
{
	int h = 0,s = 0,l = 0;

	XmScaleGetValue(h_,&h);
	XmScaleGetValue(s_,&s);
	XmScaleGetValue(l_,&l);

	hsl_.hue        = h;
	hsl_.saturation = s/100.0;
	hsl_.lightness  = l/100.0;

	XPalette::convert(hsl_,rgb_);

	refresh();
}

void ColourEditor::tabCB(Widget,XtPointer)
{
}

void ColourEditor::rgbCB(Widget,XtPointer)
{ 	

	int r = 0,g = 0,b = 0;

	XmScaleGetValue(r_,&r);
	XmScaleGetValue(g_,&g);
	XmScaleGetValue(b_,&b);

	rgb_.red   = r/100.0;
	rgb_.green = g/100.0;
	rgb_.blue  = b/100.0;

	XPalette::convert(rgb_,hsl_);

	refresh();

}

void ColourEditor::refresh()
{
	XtVaSetValues(
		new_,XmNbackground,
		XPalette::pixel(rgb_),
		0);

	XmScaleSetValue(r_,rgb_.red   * 100);
	XmScaleSetValue(g_,rgb_.green * 100);
	XmScaleSetValue(b_,rgb_.blue  * 100);
	XmScaleSetValue(h_,hsl_.hue);
	XmScaleSetValue(s_,hsl_.saturation * 100);
	XmScaleSetValue(l_,hsl_.lightness  * 100);

	rainbow_r_.fill(RGBMaker<makeR>(rgb_));
	rainbow_g_.fill(RGBMaker<makeG>(rgb_));
	rainbow_b_.fill(RGBMaker<makeB>(rgb_));

	rainbow_h_.fill(HSLMaker<makeH>(hsl_));
	rainbow_s_.fill(HSLMaker<makeS>(hsl_));
	rainbow_l_.fill(HSLMaker<makeL>(hsl_));

}

void ColourEditor::exposeCB(Widget,XtPointer)
{
	rainbow_r_.redraw();
	rainbow_g_.redraw();
	rainbow_b_.redraw();
	rainbow_h_.redraw();
	rainbow_s_.redraw();
	rainbow_l_.redraw();
}

void ColourEditor::next(const string& name)
{
	magics_[name] = new ColorName(*this,name);
}

Widget ColourEditor::where()
{
	return names_;
}

void ColourEditor::magics(const string& name)
{
	name_ = name;
	XPalette::convert(name_,rgb_);
	XPalette::convert(rgb_,hsl_);
	refresh();
}

static EditorMaker<ColourEditor> editorMaker("ColourEditor");
