/* readdvi.c -- read and interpret a DVI file.
   
   Copyright (C) 1994 Ralph Schleicher  */
   
/* This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.
      
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
      
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */


#include <config.h>
#include <errno.h>
#include "extern.h"


char *dvi_name = 0;
FILE *dvi_file = 0;
int dvi_format = 0;
int dvi_pos = 0;

static int dispatch PROTO ((int));


int
DEFUN (read_dvi, (name),
char *name)
{
  int op;

  if (name)
    {
      dvi_file = fopen (name, "r");
      if (!dvi_file)
	die ("%s:%s: %s\n", prog_name, name, strerror (errno));

      dvi_name = name;
    }
  else
    {
      dvi_file = stdin;
      dvi_name = "(stdin)";
    }

  dvi_pos = 0;

  while (1)
    {
      op = get_char (RELAX);
      if (op == EOF)
	break;

      dispatch (op);
    }

  if (name)
    fclose (dvi_file);
  else
    clearerr (stdin);

  return (0);
}


static int
DEFUN (dispatch, (op),
int op)
{
  if (op >= 0 && op <= 127)		/* set_char_0 .. set_char_127 */
    return (0);

  if (op >= 171 && op <= 234)		/* fnt_num_0 .. fnt_num_63 */
    return (0);

  switch (op)
    {
    case 128:				/* set1 */
      return skip (1);
    case 129:				/* set2 */
      return skip (2);
    case 130:				/* set3 */
      return skip (3);
    case 131:				/* set4 */
      return skip (4);

    case 132:				/* set_rule */
      return skip (8);

    case 133:				/* put1 */
      return skip (1);
    case 134:				/* put2 */
      return skip (2);
    case 135:				/* put3 */
      return skip (3);
    case 136:				/* put4 */
      return skip (4);

    case 137:				/* put_rule */
      return skip (8);

    case 138:				/* nop */
      return skip (0);

    case 139:				/* bop */
      return skip (44);
    case 140:				/* eop */
      return skip (0);

    case 141:				/* push */
      return skip (0);
    case 142:				/* pop */
      return skip (0);

    case 143:				/* right1 */
      return skip (1);
    case 144:				/* right2 */
      return skip (2);
    case 145:				/* right3 */
      return skip (3);
    case 146:				/* right4 */
      return skip (4);

    case 147:				/* w0 */
      return skip (0);
    case 148:				/* w1 */
      return skip (1);
    case 149:				/* w2 */
      return skip (2);
    case 150:				/* w3 */
      return skip (3);
    case 151:				/* w4 */
      return skip (4);

    case 152:				/* x0 */
      return skip (0);
    case 153:				/* x1 */
      return skip (1);
    case 154:				/* x2 */
      return skip (2);
    case 155:				/* x3 */
      return skip (3);
    case 156:				/* x4 */
      return skip (4);

    case 157:				/* down1 */
      return skip (1);
    case 158:				/* down2 */
      return skip (2);
    case 159:				/* down3 */
      return skip (3);
    case 160:				/* down4 */
      return skip (4);

    case 161:				/* y0 */
      return skip (0);
    case 162:				/* y1 */
      return skip (1);
    case 163:				/* y2 */
      return skip (2);
    case 164:				/* y3 */
      return skip (3);
    case 165:				/* y4 */
      return skip (4);

    case 166:				/* z0 */
      return skip (0);
    case 167:				/* z1 */
      return skip (1);
    case 168:				/* z2 */
      return skip (2);
    case 169:				/* z3 */
      return skip (3);
    case 170:				/* z4 */
      return skip (4);

    case 235:				/* fnt1 */
      return skip (1);
    case 236:				/* fnt2 */
      return skip (2);
    case 237:				/* fnt3 */
      return skip (3);
    case 238:				/* fnt4 */
      return skip (4);

    case 239:				/* xxx1 */
      return special (1);
    case 240:				/* xxx2 */
      return special (2);
    case 241:				/* xxx3 */
      return special (3);
    case 242:				/* xxx4 */
      return special (4);

    case 243:				/* fnt_def1 */
      return font_def (1);
    case 244:				/* fnt_def2 */
      return font_def (2);
    case 245:				/* fnt_def3 */
      return font_def (3);
    case 246:				/* fnt_def4 */
      return font_def (4);

    case 247:				/* pre */
      return preamble ();
    case 248:				/* post */
      return skip (28);
    case 249:				/* post_post */
      return post_postamble ();
    }

  die ("%s:%s: Unknown DVI command %d\n", prog_name, dvi_name, op);
}
