# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2013 Canonical
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.

"""Tests for Notes app"""

from __future__ import absolute_import

from testtools.matchers import Equals
from autopilot.matchers import Eventually
from autopilot.introspection import dbus

from notes_app.tests import NotesAppTestCase

import os


class TestCreate(NotesAppTestCase):
    """Tests creating notes"""

    """ This is needed to wait for the application to start.
        In the testfarm, the application may take some time to show up."""
    def setUp(self):
        # Clean up the database before starting up the app, or it will read
        # the old one
        self.clean_db()
        super(TestCreate, self).setUp()

    def clean_db(self):
        path = self.ensure_db()
        try:
            os.remove(path)
        except OSError:
            pass

    def focus_note_for_editing(self, index):
        notes = self.main_window.get_notes()
        note = notes[index]

        self.pointing_device.click_object(note)
        self.assert_note_eventually_expanded(note)
        self.pointing_device.click()

        return note

    def ensure_one_collapsed_note(self):
        note = self.focus_note_for_editing(0)
        self.keyboard.type("Hello", delay=self.TYPING_DELAY)
        self._collapse_note(note)

    def _collapse_note(self, note):
        """Collapse a note by clicking outside of it."""
        x, y, width, height = note.globalRect
        self.pointing_device.move(x + width // 2, y + height + 20)
        self.pointing_device.click()
        try:
            self.assert_note_eventually_collapsed(note)
        except dbus.StateNotFoundError:
            # The note doesn't have text so it's deleted.
            pass

    def test_note_expand_and_unexpand(self):
        notes = self.main_window.get_notes()

        # When starting with no database there should be one empty note there
        self.assertThat(len(notes), Equals(1))
        note = notes[0]

        # Clicking on a note should expand it
        self.pointing_device.click_object(note)
        self.assert_note_eventually_expanded(note)

        # Clicking outside of the note should unexpand it, and this should
        # cause it to be deleted
        self._collapse_note(note)
        self.assertThat(
            lambda: len(self.main_window.get_notes()), Eventually(Equals(0)))

    def test_note_focus_on_second_click_outside(self):
        notes = self.main_window.get_notes()

        self.ensure_one_collapsed_note()
        self.assertThat(len(notes), Equals(1))
        note = notes[0]

        # Clicking on a note should expand it but it should not be focused
        self.pointing_device.click_object(note)
        self.assert_note_eventually_expanded(note)

        content = self.main_window.get_note_content(note)
        self.assertThat(content.activeFocus, Equals(False))

        # Now click on it the note outside of the content area and verify
        # that the content area still gets focus
        content_x, content_y, content_w, content_h = content.globalRect
        note_x, note_y, note_w, note_h = note.globalRect

        self.pointing_device.move(
            note_x + note_w / 2,
            note_y + content_h + ((note_h - content_h) / 2))
        self.pointing_device.click()

        self.assertThat(content.activeFocus, Eventually(Equals(True)))

    def test_note_focus_on_second_click_inside(self):
        notes = self.main_window.get_notes()

        self.ensure_one_collapsed_note()
        self.assertThat(len(notes), Equals(1))
        note = notes[0]

        # Clicking on a note should expand it but it should not be focused
        self.pointing_device.click_object(note)
        self.assert_note_eventually_expanded(note)

        content = self.main_window.get_note_content(note)
        self.assertThat(content.activeFocus, Equals(False))

        # Now click on it the note inside the content area and verify it
        # focuses
        content_x, content_y, content_w, content_h = content.globalRect

        self.pointing_device.move(
            content_x + content_w / 2, content_y + content_h / 2)
        self.pointing_device.click()

        self.assertThat(content.activeFocus, Eventually(Equals(True)))

    def test_note_unfocused_when_collapsed(self):
        note = self.focus_note_for_editing(0)

        # Type something so that the note doesn't get deleted then collapse it.
        self.keyboard.type("Hello", delay=self.TYPING_DELAY)
        self._collapse_note(note)

        content = self.main_window.get_note_content(note)
        self.assertThat(content.activeFocus, Equals(False))

    def test_note_focused_when_empty(self):
        notes = self.main_window.get_notes()
        note = notes[0]

        self.pointing_device.click_object(note)

        content = self.main_window.get_note_content(note)
        self.assertThat(content.activeFocus, Eventually(Equals(True)))
