/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "fakeimagecapturecontrol.h"
#include "fakecameracontrol.h"
#include "fakecameraservice.h"
#include "storagemanager.h"

#include <QDir>
#include <QElapsedTimer>
#include <QFileInfo>
#include <QImage>
#include <QTemporaryFile>
#include <QThread>
#include <QTimer>
#include <QUrl>

void SaveJpegWorker::saveJpeg(QImage image, const QString &fileName)
{
    QTemporaryFile tempFile;
    tempFile.open();

    image.save(&tempFile, "JPEG");

    QFile file(tempFile.fileName());
    tempFile.close();
    file.rename(fileName);

    Q_EMIT saveDone();
}


FakeImageCaptureControl::FakeImageCaptureControl(FakeCameraService *service, QObject *parent)
   : QCameraImageCaptureControl(parent),
    m_service(service),
    m_cameraControl(service->cameraControl()),
    m_lastRequestId(0),
    m_pendingCaptureFile(),
    m_ready(false)
{
    QObject::connect(m_cameraControl, SIGNAL(stateChanged(QCamera::State)),
                     this, SLOT(updateReady()));
    m_saveThread = new QThread(this);
    m_saveWorker = new SaveJpegWorker;
    m_saveWorker->moveToThread(m_saveThread);
    m_saveThread->start();
    QObject::connect(m_saveWorker, SIGNAL(saveDone()), this, SLOT(saveDone()));
}

FakeImageCaptureControl::~FakeImageCaptureControl()
{
    m_saveThread->exit();
    m_saveThread->wait();
}

bool FakeImageCaptureControl::isReadyForCapture() const
{
    return m_ready;
}

int FakeImageCaptureControl::capture(const QString &fileName)
{
    m_lastRequestId++;

    QFileInfo fi(fileName);
    if (fileName.isEmpty() || fi.isDir()) {
        m_pendingCaptureFile = m_storageManager.nextPhotoFileName(fileName);
    } else {
        m_pendingCaptureFile = fileName;
    }
    m_storageManager.checkDirectory(m_pendingCaptureFile);

    m_service->fakeSensor().stopMove();
    QTimer::singleShot(100, this, SLOT(performPreview()));
    QTimer::singleShot(500, this, SLOT(performCapture()));
    Q_EMIT imageExposed(m_lastRequestId);
    updateReady();
    return m_lastRequestId;
}

void FakeImageCaptureControl::cancelCapture()
{
}

void FakeImageCaptureControl::updateReady()
{
    bool ready = m_cameraControl->state() == QCamera::ActiveState && m_pendingCaptureFile.isEmpty();
    if (m_ready != ready) {
        m_ready = ready;
        Q_EMIT readyForCaptureChanged(m_ready);
    }
}

void FakeImageCaptureControl::performPreview()
{
    QImage preview = m_service->fakeSensor().previewFrame();
    m_service->fakeSensor().startMove();
    Q_EMIT imageCaptured(m_lastRequestId, preview);
}

void FakeImageCaptureControl::performCapture()
{
    QMetaObject::invokeMethod(m_saveWorker, "saveJpeg", Qt::QueuedConnection,
                              Q_ARG(QImage, m_service->fakeSensor().capturedImage()),
                              Q_ARG(QString, m_pendingCaptureFile));
}

void FakeImageCaptureControl::saveDone()
{
    Q_EMIT imageSaved(m_lastRequestId, m_pendingCaptureFile);
    m_pendingCaptureFile.clear();
    updateReady();
}
